import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Link } from "wouter";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber, formatDate } from "@/lib/formatters";
import { 
  XCircle, 
  Eye, 
  Calendar,
  User,
  RefreshCw,
  ArrowRight,
  CreditCard,
  CheckCircle,
} from "lucide-react";

interface Employee {
  id: number;
  nameAr: string;
  code: string | null;
  role: string | null;
  isActive: boolean;
}

interface Customer {
  id: number;
  nameAr: string;
  phone: string | null;
}

interface Service {
  id: number;
  nameAr: string;
  code: string | null;
  category?: {
    id: number;
    nameAr: string;
  };
}

interface OrderService {
  id: number;
  orderId: number;
  serviceId: number;
  quantity: number;
  basePrice: number;
  vatType: string;
  vatRate: number;
  lineSubtotal: number;
  vatAmount: number;
  lineTotal: number;
  scheduledAt: string | null;
  executingEmployeeId: number | null;
  service?: Service;
  executingEmployee?: Employee;
}

interface Order {
  id: number;
  orderNumber: string;
  orderDate: string;
  source: string;
  orderType: string;
  clientId: number | null;
  createdByEmployeeId: number;
  branchId: number | null;
  couponCode: string | null;
  couponDiscountAmount: number;
  status: string;
  notes: string | null;
  subtotalAmount: number;
  vatAmount: number;
  totalAmount: number;
  paidAmount?: number;
  paymentStatus?: string;
  createdAt: string;
  client?: Customer;
  createdByEmployee?: Employee;
  services: OrderService[];
}

interface PaymentMethod {
  id: number;
  nameAr: string;
  code: string | null;
  type: string | null;
  isActive: boolean;
}

const sourceLabels: Record<string, string> = {
  pos: "نقطة البيع",
  app: "التطبيق",
};

const paymentStatusLabels: Record<string, string> = {
  unpaid: "غير مسدد",
  partially_paid: "مسدد جزئياً",
  paid: "مسدد",
};

export default function OrdersCanceledPage() {
  const [paymentOrderId, setPaymentOrderId] = useState<number | null>(null);
  const [paymentOrder, setPaymentOrder] = useState<Order | null>(null);
  const [paymentAmount, setPaymentAmount] = useState<string>("");
  const [paymentMethodId, setPaymentMethodId] = useState<string>("");
  const [paymentNotes, setPaymentNotes] = useState<string>("");
  const { toast } = useToast();

  const { data: orders, isLoading, refetch } = useQuery<Order[]>({
    queryKey: ["/api/orders", "canceled"],
    queryFn: async () => {
      const response = await fetch("/api/orders?status=canceled");
      if (!response.ok) throw new Error("Failed to fetch canceled orders");
      return response.json();
    },
  });

  const { data: paymentMethods } = useQuery<PaymentMethod[]>({
    queryKey: ["/api/payment-methods"],
  });

  const payCancelledMutation = useMutation({
    mutationFn: ({ id, amount, paymentMethodId, notes }: { id: number; amount: number; paymentMethodId: number; notes?: string }) =>
      apiRequest("POST", `/api/orders/${id}/pay-cancelled`, { amount, paymentMethodId, notes }),
    onSuccess: () => {
      refetch();
      toast({ 
        title: "تم السداد بنجاح",
        description: "تم سداد الطلب الملغي وتسجيل القيد المحاسبي",
      });
      closePaymentDialog();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في سداد الطلب الملغي",
        variant: "destructive",
      });
    },
  });

  const openPaymentDialog = (order: Order) => {
    setPaymentOrderId(order.id);
    setPaymentOrder(order);
    const remainingAmount = order.totalAmount - (order.paidAmount || 0);
    setPaymentAmount(remainingAmount.toFixed(2));
    setPaymentMethodId("");
    setPaymentNotes("");
  };

  const closePaymentDialog = () => {
    setPaymentOrderId(null);
    setPaymentOrder(null);
    setPaymentAmount("");
    setPaymentMethodId("");
    setPaymentNotes("");
  };

  const handlePayment = () => {
    if (!paymentOrderId || !paymentMethodId || !paymentAmount) {
      toast({
        title: "خطأ",
        description: "الرجاء إدخال جميع البيانات المطلوبة",
        variant: "destructive",
      });
      return;
    }

    const amount = parseFloat(paymentAmount);
    if (isNaN(amount) || amount <= 0) {
      toast({
        title: "خطأ",
        description: "مبلغ السداد غير صحيح",
        variant: "destructive",
      });
      return;
    }

    if (paymentOrder) {
      const remainingBalance = paymentOrder.totalAmount - (paymentOrder.paidAmount || 0);
      if (amount > remainingBalance) {
        toast({
          title: "خطأ",
          description: `مبلغ السداد (${formatNumber(amount)}) أكبر من المبلغ المتبقي (${formatNumber(remainingBalance)})`,
          variant: "destructive",
        });
        return;
      }
    }

    payCancelledMutation.mutate({
      id: paymentOrderId,
      amount,
      paymentMethodId: parseInt(paymentMethodId),
      notes: paymentNotes || undefined,
    });
  };

  const activePaymentMethods = paymentMethods?.filter(pm => pm.isActive) || [];

  const stats = {
    total: orders?.length || 0,
    totalLostRevenue: orders?.reduce((sum, o) => sum + o.totalAmount, 0) || 0,
    totalPaid: orders?.reduce((sum, o) => sum + (o.paidAmount || 0), 0) || 0,
  };

  const getPaymentStatusBadge = (order: Order) => {
    const status = order.paymentStatus || "unpaid";
    if (status === "paid") {
      return <Badge className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">مسدد</Badge>;
    } else if (status === "partially_paid") {
      return <Badge className="bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200">مسدد جزئياً</Badge>;
    }
    return <Badge variant="destructive">غير مسدد</Badge>;
  };

  return (
    <div className="container mx-auto p-6 space-y-6" dir="rtl">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div className="flex items-center gap-4">
          <Link href="/orders">
            <Button variant="ghost" size="icon" data-testid="button-back">
              <ArrowRight className="h-5 w-5" />
            </Button>
          </Link>
          <div className="flex items-center gap-2">
            <XCircle className="h-6 w-6 text-red-600" />
            <h1 className="text-2xl font-bold">الطلبات الملغاة</h1>
          </div>
        </div>
        <Button onClick={() => refetch()} variant="outline" size="icon" data-testid="button-refresh">
          <RefreshCw className="h-4 w-4" />
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-red-600" data-testid="stat-total">{stats.total}</div>
            <div className="text-sm text-muted-foreground">عدد الطلبات الملغاة</div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-red-600" data-testid="stat-lost-revenue">{formatNumber(stats.totalLostRevenue)}</div>
            <div className="text-sm text-muted-foreground">إجمالي المبالغ المفقودة (ر.س)</div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-green-600" data-testid="stat-paid">{formatNumber(stats.totalPaid)}</div>
            <div className="text-sm text-muted-foreground">إجمالي المبالغ المسددة (ر.س)</div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardContent className="p-0">
          {isLoading ? (
            <div className="p-4 space-y-2">
              {[...Array(5)].map((_, i) => (
                <Skeleton key={i} className="h-16 w-full" />
              ))}
            </div>
          ) : orders && orders.length > 0 ? (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">رقم الطلب</TableHead>
                  <TableHead className="text-right">التاريخ</TableHead>
                  <TableHead className="text-right">العميل</TableHead>
                  <TableHead className="text-right">المصدر</TableHead>
                  <TableHead className="text-right">الخدمات</TableHead>
                  <TableHead className="text-right">الإجمالي</TableHead>
                  <TableHead className="text-right">حالة السداد</TableHead>
                  <TableHead className="text-right">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {orders.map((order) => {
                  const isPaid = order.paymentStatus === "paid";
                  const remainingAmount = order.totalAmount - (order.paidAmount || 0);
                  
                  return (
                    <TableRow key={order.id} data-testid={`row-order-${order.id}`}>
                      <TableCell className="font-medium" data-testid={`text-order-number-${order.id}`}>
                        {order.orderNumber}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1">
                          <Calendar className="h-4 w-4 text-muted-foreground" />
                          <span className="text-sm">{formatDate(order.orderDate)}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        {order.client ? (
                          <div className="flex items-center gap-1">
                            <User className="h-4 w-4 text-muted-foreground" />
                            <span>{order.client.nameAr}</span>
                          </div>
                        ) : (
                          <span className="text-muted-foreground">زائر</span>
                        )}
                      </TableCell>
                      <TableCell>
                        <Badge variant="outline">
                          {sourceLabels[order.source] || order.source}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div className="flex flex-wrap gap-1">
                          {order.services.slice(0, 2).map((svc) => (
                            <Badge key={svc.id} variant="secondary" className="text-xs">
                              {svc.service?.nameAr || `خدمة ${svc.serviceId}`}
                            </Badge>
                          ))}
                          {order.services.length > 2 && (
                            <Badge variant="outline" className="text-xs">
                              +{order.services.length - 2}
                            </Badge>
                          )}
                        </div>
                      </TableCell>
                      <TableCell className="font-semibold text-red-600" data-testid={`text-order-total-${order.id}`}>
                        {formatNumber(order.totalAmount)} ر.س
                      </TableCell>
                      <TableCell>
                        {getPaymentStatusBadge(order)}
                      </TableCell>
                      <TableCell>
                        <div className="flex gap-2">
                          {!isPaid && (
                            <Button 
                              variant="default"
                              size="sm"
                              onClick={() => openPaymentDialog(order)}
                              data-testid={`button-pay-cancelled-${order.id}`}
                            >
                              <CreditCard className="h-4 w-4 ml-1" />
                              سداد الطلب الملغي
                            </Button>
                          )}
                          {isPaid && (
                            <Badge className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200 flex items-center gap-1">
                              <CheckCircle className="h-3 w-3" />
                              تم السداد
                            </Badge>
                          )}
                          <Link href={`/orders/${order.id}`}>
                            <Button variant="ghost" size="icon" data-testid={`button-view-order-${order.id}`}>
                              <Eye className="h-4 w-4" />
                            </Button>
                          </Link>
                        </div>
                      </TableCell>
                    </TableRow>
                  );
                })}
              </TableBody>
            </Table>
          ) : (
            <div className="p-8 text-center text-muted-foreground">
              <XCircle className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p>لا توجد طلبات ملغاة</p>
            </div>
          )}
        </CardContent>
      </Card>

      <Dialog open={paymentOrderId !== null} onOpenChange={() => closePaymentDialog()}>
        <DialogContent dir="rtl" className="max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <CreditCard className="h-5 w-5" />
              سداد الطلب الملغي
            </DialogTitle>
            <DialogDescription>
              {paymentOrder && (
                <span>
                  طلب رقم: <strong>{paymentOrder.orderNumber}</strong> - 
                  إجمالي الطلب: <strong className="text-red-600">{formatNumber(paymentOrder.totalAmount)} ر.س</strong>
                </span>
              )}
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="paymentAmount">مبلغ السداد (ر.س) <span className="text-red-500">*</span></Label>
              <Input
                id="paymentAmount"
                type="number"
                step="0.01"
                min="0.01"
                max={paymentOrder ? paymentOrder.totalAmount - (paymentOrder.paidAmount || 0) : undefined}
                value={paymentAmount}
                onChange={(e) => setPaymentAmount(e.target.value)}
                placeholder="أدخل مبلغ السداد"
                data-testid="input-payment-amount"
              />
              {paymentOrder && (
                <p className="text-xs text-muted-foreground">
                  الحد الأقصى: {formatNumber(paymentOrder.totalAmount - (paymentOrder.paidAmount || 0))} ر.س
                </p>
              )}
            </div>

            <div className="space-y-2">
              <Label htmlFor="paymentMethod">طريقة الدفع <span className="text-red-500">*</span></Label>
              <Select 
                value={paymentMethodId} 
                onValueChange={setPaymentMethodId}
              >
                <SelectTrigger data-testid="select-payment-method">
                  <SelectValue placeholder="اختر طريقة الدفع" />
                </SelectTrigger>
                <SelectContent>
                  {activePaymentMethods.map((pm) => (
                    <SelectItem key={pm.id} value={pm.id.toString()}>
                      {pm.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="paymentNotes">ملاحظات (اختياري)</Label>
              <Textarea
                id="paymentNotes"
                value={paymentNotes}
                onChange={(e) => setPaymentNotes(e.target.value)}
                placeholder="أدخل أي ملاحظات إضافية..."
                rows={3}
                data-testid="input-payment-notes"
              />
            </div>
          </div>

          <DialogFooter className="gap-2">
            <Button 
              variant="outline" 
              onClick={closePaymentDialog}
              data-testid="button-cancel-payment"
            >
              إلغاء
            </Button>
            <Button 
              onClick={handlePayment} 
              disabled={payCancelledMutation.isPending || !paymentMethodId || !paymentAmount}
              data-testid="button-confirm-payment"
            >
              {payCancelledMutation.isPending ? "جاري السداد..." : "تأكيد السداد"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
