import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Link } from "wouter";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber, formatDate } from "@/lib/formatters";
import { 
  CalendarClock, 
  Eye, 
  Calendar,
  User,
  RefreshCw,
  ArrowRight,
  Clock,
} from "lucide-react";

interface Employee {
  id: number;
  nameAr: string;
  code: string | null;
  role: string | null;
  isActive: boolean;
}

interface Customer {
  id: number;
  nameAr: string;
  phone: string | null;
}

interface Service {
  id: number;
  nameAr: string;
  code: string | null;
  category?: {
    id: number;
    nameAr: string;
  };
}

interface OrderService {
  id: number;
  orderId: number;
  serviceId: number;
  quantity: number;
  basePrice: number;
  vatType: string;
  vatRate: number;
  lineSubtotal: number;
  vatAmount: number;
  lineTotal: number;
  scheduledAt: string | null;
  executingEmployeeId: number | null;
  service?: Service;
  executingEmployee?: Employee;
}

interface Order {
  id: number;
  orderNumber: string;
  orderDate: string;
  source: string;
  orderType: string;
  clientId: number | null;
  createdByEmployeeId: number;
  branchId: number | null;
  couponCode: string | null;
  couponDiscountAmount: number;
  status: string;
  notes: string | null;
  subtotalAmount: number;
  vatAmount: number;
  totalAmount: number;
  createdAt: string;
  scheduledAt?: string | null;
  client?: Customer;
  createdByEmployee?: Employee;
  services: OrderService[];
}

const sourceLabels: Record<string, string> = {
  pos: "نقطة البيع",
  app: "التطبيق",
};

const statusLabels: Record<string, string> = {
  scheduled: "مجدول",
  postponed: "مؤجل",
  deferred: "مؤجل",
};

const statusColors: Record<string, string> = {
  scheduled: "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-300",
  postponed: "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-300",
  deferred: "bg-amber-100 text-amber-800 dark:bg-amber-900 dark:text-amber-300",
};

export default function OrdersPostponedPage() {
  const [statusChangeOrderId, setStatusChangeOrderId] = useState<number | null>(null);
  const [newStatus, setNewStatus] = useState<string>("");
  const { toast } = useToast();

  const { data: scheduledOrders, isLoading: isLoadingScheduled } = useQuery<Order[]>({
    queryKey: ["/api/orders", "scheduled"],
    queryFn: async () => {
      const response = await fetch("/api/orders?status=scheduled");
      if (!response.ok) throw new Error("Failed to fetch scheduled orders");
      return response.json();
    },
  });

  const { data: postponedOrders, isLoading: isLoadingPostponed } = useQuery<Order[]>({
    queryKey: ["/api/orders", "postponed"],
    queryFn: async () => {
      const response = await fetch("/api/orders?status=postponed");
      if (!response.ok) throw new Error("Failed to fetch postponed orders");
      return response.json();
    },
  });

  const { data: deferredOrders, isLoading: isLoadingDeferred } = useQuery<Order[]>({
    queryKey: ["/api/orders", "deferred"],
    queryFn: async () => {
      const response = await fetch("/api/orders?status=deferred");
      if (!response.ok) throw new Error("Failed to fetch deferred orders");
      return response.json();
    },
  });

  const isLoading = isLoadingScheduled || isLoadingPostponed || isLoadingDeferred;
  
  // Merge orders from all three status queries
  // Deduplicate by order ID using Set to prevent duplicate rows if backend ever returns overlapping records
  // Sort by date (newest first) for consistent display
  const allOrders = [...(scheduledOrders || []), ...(postponedOrders || []), ...(deferredOrders || [])];
  const seenIds = new Set<number>();
  const orders = allOrders
    .filter(order => {
      if (seenIds.has(order.id)) return false;
      seenIds.add(order.id);
      return true;
    })
    .sort((a, b) => new Date(b.orderDate).getTime() - new Date(a.orderDate).getTime());

  // Use partial key matching to invalidate all order-related queries at once
  // TanStack Query v5 defaults to partial matching on queryKey arrays
  const refetch = () => {
    queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
  };

  const updateStatusMutation = useMutation({
    mutationFn: ({ id, status }: { id: number; status: string }) =>
      apiRequest("PUT", `/api/orders/${id}`, { status }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
      toast({ title: "تم تحديث حالة الطلب بنجاح" });
      setStatusChangeOrderId(null);
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث حالة الطلب",
        variant: "destructive",
      });
    },
  });

  const handleStatusChange = () => {
    if (statusChangeOrderId && newStatus) {
      updateStatusMutation.mutate({ id: statusChangeOrderId, status: newStatus });
    }
  };

  const openStatusChangeDialog = (order: Order) => {
    setStatusChangeOrderId(order.id);
    setNewStatus(order.status);
  };

  const stats = {
    total: orders.length,
    scheduled: scheduledOrders?.length || 0,
    postponed: postponedOrders?.length || 0,
    deferred: deferredOrders?.length || 0,
    totalAmount: orders.reduce((sum, o) => sum + o.totalAmount, 0),
  };

  return (
    <div className="container mx-auto p-6 space-y-6" dir="rtl">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div className="flex items-center gap-4">
          <Link href="/orders">
            <Button variant="ghost" size="icon" data-testid="button-back">
              <ArrowRight className="h-5 w-5" />
            </Button>
          </Link>
          <div className="flex items-center gap-2">
            <CalendarClock className="h-6 w-6 text-purple-600" />
            <h1 className="text-2xl font-bold">الطلبات المؤجلة</h1>
          </div>
        </div>
        <Button onClick={() => refetch()} variant="outline" size="icon" data-testid="button-refresh">
          <RefreshCw className="h-4 w-4" />
        </Button>
      </div>

      <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-primary" data-testid="stat-total">{stats.total}</div>
            <div className="text-sm text-muted-foreground">إجمالي الطلبات</div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-purple-600" data-testid="stat-scheduled">{stats.scheduled}</div>
            <div className="text-sm text-muted-foreground">مجدول</div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-orange-600" data-testid="stat-postponed">{stats.postponed}</div>
            <div className="text-sm text-muted-foreground">مؤجل</div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-amber-600" data-testid="stat-deferred">{stats.deferred}</div>
            <div className="text-sm text-muted-foreground">مؤجل (تأجيل)</div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardContent className="p-0">
          {isLoading ? (
            <div className="p-4 space-y-2">
              {[...Array(5)].map((_, i) => (
                <Skeleton key={i} className="h-16 w-full" />
              ))}
            </div>
          ) : orders && orders.length > 0 ? (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">رقم الطلب</TableHead>
                  <TableHead className="text-right">التاريخ</TableHead>
                  <TableHead className="text-right">العميل</TableHead>
                  <TableHead className="text-right">المصدر</TableHead>
                  <TableHead className="text-right">الخدمات</TableHead>
                  <TableHead className="text-right">الإجمالي</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-right">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {orders.map((order) => (
                  <TableRow key={order.id} data-testid={`row-order-${order.id}`}>
                    <TableCell className="font-medium" data-testid={`text-order-number-${order.id}`}>
                      {order.orderNumber}
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-1">
                        <Calendar className="h-4 w-4 text-muted-foreground" />
                        <span className="text-sm">{formatDate(order.orderDate)}</span>
                      </div>
                      {order.scheduledAt && (
                        <div className="flex items-center gap-1 mt-1 text-xs text-purple-600">
                          <Clock className="h-3 w-3" />
                          <span>مجدول: {formatDate(order.scheduledAt)}</span>
                        </div>
                      )}
                    </TableCell>
                    <TableCell>
                      {order.client ? (
                        <div className="flex items-center gap-1">
                          <User className="h-4 w-4 text-muted-foreground" />
                          <span>{order.client.nameAr}</span>
                        </div>
                      ) : (
                        <span className="text-muted-foreground">زائر</span>
                      )}
                    </TableCell>
                    <TableCell>
                      <Badge variant="outline">
                        {sourceLabels[order.source] || order.source}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <div className="flex flex-wrap gap-1">
                        {order.services.slice(0, 2).map((svc) => (
                          <Badge key={svc.id} variant="secondary" className="text-xs">
                            {svc.service?.nameAr || `خدمة ${svc.serviceId}`}
                          </Badge>
                        ))}
                        {order.services.length > 2 && (
                          <Badge variant="outline" className="text-xs">
                            +{order.services.length - 2}
                          </Badge>
                        )}
                      </div>
                    </TableCell>
                    <TableCell className="font-semibold" data-testid={`text-order-total-${order.id}`}>
                      {formatNumber(order.totalAmount)} ر.س
                    </TableCell>
                    <TableCell>
                      <Badge 
                        className={`${statusColors[order.status]} cursor-pointer`}
                        onClick={() => openStatusChangeDialog(order)}
                        data-testid={`badge-status-${order.id}`}
                      >
                        {statusLabels[order.status] || order.status}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <div className="flex gap-2">
                        <Button 
                          variant="outline" 
                          size="sm"
                          onClick={() => openStatusChangeDialog(order)}
                          data-testid={`button-change-status-${order.id}`}
                        >
                          تغيير الحالة
                        </Button>
                        <Link href={`/orders/${order.id}`}>
                          <Button variant="ghost" size="icon" data-testid={`button-view-order-${order.id}`}>
                            <Eye className="h-4 w-4" />
                          </Button>
                        </Link>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          ) : (
            <div className="p-8 text-center text-muted-foreground">
              <CalendarClock className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p>لا توجد طلبات مجدولة أو مؤجلة</p>
            </div>
          )}
        </CardContent>
      </Card>

      <Dialog open={statusChangeOrderId !== null} onOpenChange={() => setStatusChangeOrderId(null)}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle>تغيير حالة الطلب</DialogTitle>
            <DialogDescription>اختر الحالة الجديدة للطلب</DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label>الحالة الجديدة</Label>
              <Select value={newStatus} onValueChange={setNewStatus} data-testid="select-new-status">
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="new">جديد</SelectItem>
                  <SelectItem value="in_progress">قيد التنفيذ</SelectItem>
                  <SelectItem value="completed">مكتمل</SelectItem>
                  <SelectItem value="canceled">ملغي</SelectItem>
                  <SelectItem value="scheduled">مجدول</SelectItem>
                  <SelectItem value="postponed">مؤجل</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setStatusChangeOrderId(null)}>
              إلغاء
            </Button>
            <Button 
              onClick={handleStatusChange} 
              disabled={updateStatusMutation.isPending}
              data-testid="button-confirm-status"
            >
              {updateStatusMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
