import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Link } from "wouter";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber, formatDate } from "@/lib/formatters";
import { 
  CalendarCheck, 
  Eye, 
  Calendar,
  User,
  RefreshCw,
  ArrowRight,
  Clock,
  Filter,
} from "lucide-react";

interface Employee {
  id: number;
  nameAr: string;
  code: string | null;
  role: string | null;
  isActive: boolean;
}

interface Customer {
  id: number;
  nameAr: string;
  phone: string | null;
}

interface Service {
  id: number;
  nameAr: string;
  code: string | null;
  category?: {
    id: number;
    nameAr: string;
  };
}

interface OrderService {
  id: number;
  orderId: number;
  serviceId: number;
  quantity: number;
  basePrice: number;
  vatType: string;
  vatRate: number;
  lineSubtotal: number;
  vatAmount: number;
  lineTotal: number;
  scheduledAt: string | null;
  executingEmployeeId: number | null;
  service?: Service;
  executingEmployee?: Employee;
}

interface Order {
  id: number;
  orderNumber: string;
  orderDate: string;
  source: string;
  orderType: string;
  clientId: number | null;
  createdByEmployeeId: number;
  branchId: number | null;
  couponCode: string | null;
  couponDiscountAmount: number;
  status: string;
  notes: string | null;
  subtotalAmount: number;
  vatAmount: number;
  totalAmount: number;
  createdAt: string;
  scheduledAt?: string | null;
  client?: Customer;
  createdByEmployee?: Employee;
  services: OrderService[];
}

const sourceLabels: Record<string, string> = {
  pos: "نقطة البيع",
  app: "التطبيق",
};

const statusLabels: Record<string, string> = {
  pending: "قيد الانتظار",
  confirmed: "مؤكد",
  in_progress: "جاري التنفيذ",
  completed: "مكتمل",
  cancelled: "ملغي",
  scheduled: "مجدول",
};

const statusColors: Record<string, string> = {
  pending: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-300",
  confirmed: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300",
  in_progress: "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-300",
  completed: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300",
  cancelled: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300",
  scheduled: "bg-indigo-100 text-indigo-800 dark:bg-indigo-900 dark:text-indigo-300",
};

const getTodayDate = (): string => {
  const today = new Date();
  return today.toISOString().split('T')[0];
};

export default function OrdersScheduledPage() {
  const { toast } = useToast();
  const [dateFilter, setDateFilter] = useState<string>("");
  const [statusFilter, setStatusFilter] = useState<string>("all");

  const buildQueryString = () => {
    const params = new URLSearchParams();
    params.append("hasScheduledAt", "true");
    if (dateFilter) params.append("scheduledDate", dateFilter);
    if (statusFilter && statusFilter !== "all") params.append("status", statusFilter);
    return params.toString();
  };

  const { data: orders = [], isLoading, refetch } = useQuery<Order[]>({
    queryKey: ["/api/orders", "scheduled-date", dateFilter, statusFilter],
    queryFn: async () => {
      const qs = buildQueryString();
      const response = await fetch(`/api/orders?${qs}`);
      if (!response.ok) throw new Error("Failed to fetch scheduled orders");
      return response.json();
    },
  });

  const filteredOrders = orders.filter(order => order.scheduledAt);

  const todayOrders = filteredOrders.filter(order => {
    if (!order.scheduledAt) return false;
    const scheduledDate = new Date(order.scheduledAt).toISOString().split('T')[0];
    return scheduledDate === getTodayDate();
  });

  const upcomingOrders = filteredOrders.filter(order => {
    if (!order.scheduledAt) return false;
    const scheduledDate = new Date(order.scheduledAt);
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    return scheduledDate > today;
  });

  const stats = {
    total: filteredOrders.length,
    today: todayOrders.length,
    upcoming: upcomingOrders.length,
    totalAmount: filteredOrders.reduce((sum, o) => sum + o.totalAmount, 0),
  };

  return (
    <div className="container mx-auto p-6 space-y-6" dir="rtl">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div className="flex items-center gap-4">
          <Link href="/orders">
            <Button variant="ghost" size="icon" data-testid="button-back">
              <ArrowRight className="h-5 w-5" />
            </Button>
          </Link>
          <div className="flex items-center gap-2">
            <CalendarCheck className="h-6 w-6 text-indigo-600" />
            <h1 className="text-2xl font-bold">الطلبات المجدولة</h1>
          </div>
        </div>
        <Button onClick={() => refetch()} variant="outline" size="icon" data-testid="button-refresh">
          <RefreshCw className="h-4 w-4" />
        </Button>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-primary" data-testid="stat-total">{stats.total}</div>
            <div className="text-sm text-muted-foreground">إجمالي المجدول</div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-indigo-600" data-testid="stat-today">{stats.today}</div>
            <div className="text-sm text-muted-foreground">اليوم</div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-blue-600" data-testid="stat-upcoming">{stats.upcoming}</div>
            <div className="text-sm text-muted-foreground">القادم</div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-green-600" data-testid="stat-amount">
              {formatNumber(stats.totalAmount)}
            </div>
            <div className="text-sm text-muted-foreground">إجمالي المبلغ (ر.س)</div>
          </CardContent>
        </Card>
      </div>

      {/* Filters */}
      <Card>
        <CardContent className="p-4">
          <div className="flex flex-wrap gap-4 items-end">
            <div className="space-y-2">
              <Label className="flex items-center gap-1">
                <Calendar className="h-4 w-4" />
                التاريخ المجدول
              </Label>
              <Input
                type="date"
                value={dateFilter}
                onChange={(e) => setDateFilter(e.target.value)}
                className="w-44"
                data-testid="input-date-filter"
              />
            </div>
            <div className="space-y-2">
              <Label className="flex items-center gap-1">
                <Filter className="h-4 w-4" />
                الحالة
              </Label>
              <Select value={statusFilter} onValueChange={setStatusFilter} data-testid="select-status-filter">
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="جميع الحالات" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">جميع الحالات</SelectItem>
                  <SelectItem value="pending">قيد الانتظار</SelectItem>
                  <SelectItem value="confirmed">مؤكد</SelectItem>
                  <SelectItem value="scheduled">مجدول</SelectItem>
                  <SelectItem value="in_progress">جاري التنفيذ</SelectItem>
                  <SelectItem value="completed">مكتمل</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <Button
              variant="outline"
              onClick={() => setDateFilter(getTodayDate())}
              data-testid="button-today-filter"
            >
              <Clock className="h-4 w-4 ml-1" />
              اليوم
            </Button>
            <Button
              variant="ghost"
              onClick={() => {
                setDateFilter("");
                setStatusFilter("all");
              }}
              data-testid="button-clear-filters"
            >
              مسح الفلاتر
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* Orders Table */}
      <Card>
        <CardContent className="p-0">
          {isLoading ? (
            <div className="p-4 space-y-2">
              {[...Array(5)].map((_, i) => (
                <Skeleton key={i} className="h-16 w-full" />
              ))}
            </div>
          ) : filteredOrders.length > 0 ? (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">رقم الطلب</TableHead>
                  <TableHead className="text-right">تاريخ الإنشاء</TableHead>
                  <TableHead className="text-right">التاريخ المجدول</TableHead>
                  <TableHead className="text-right">العميل</TableHead>
                  <TableHead className="text-right">المصدر</TableHead>
                  <TableHead className="text-right">الخدمات</TableHead>
                  <TableHead className="text-right">الإجمالي</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-right">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {filteredOrders.map((order) => (
                  <TableRow key={order.id} data-testid={`row-order-${order.id}`}>
                    <TableCell className="font-medium" data-testid={`text-order-number-${order.id}`}>
                      {order.orderNumber}
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-1 text-sm text-muted-foreground">
                        <Calendar className="h-3 w-3" />
                        {formatDate(order.orderDate)}
                      </div>
                    </TableCell>
                    <TableCell>
                      {order.scheduledAt && (
                        <div className="flex items-center gap-1">
                          <CalendarCheck className="h-4 w-4 text-indigo-600" />
                          <span className="font-medium text-indigo-700 dark:text-indigo-400">
                            {formatDate(order.scheduledAt)}
                          </span>
                        </div>
                      )}
                    </TableCell>
                    <TableCell>
                      {order.client ? (
                        <div className="flex items-center gap-1">
                          <User className="h-4 w-4 text-muted-foreground" />
                          <span>{order.client.nameAr}</span>
                        </div>
                      ) : (
                        <span className="text-muted-foreground">زائر</span>
                      )}
                    </TableCell>
                    <TableCell>
                      <Badge variant="outline">
                        {sourceLabels[order.source] || order.source}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <div className="flex flex-wrap gap-1">
                        {order.services.slice(0, 2).map((svc) => (
                          <Badge key={svc.id} variant="secondary" className="text-xs">
                            {svc.service?.nameAr || `خدمة ${svc.serviceId}`}
                          </Badge>
                        ))}
                        {order.services.length > 2 && (
                          <Badge variant="outline" className="text-xs">
                            +{order.services.length - 2}
                          </Badge>
                        )}
                      </div>
                    </TableCell>
                    <TableCell className="font-medium">
                      {formatNumber(order.totalAmount)} ر.س
                    </TableCell>
                    <TableCell>
                      <Badge className={statusColors[order.status] || "bg-gray-100 text-gray-800"}>
                        {statusLabels[order.status] || order.status}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <Link href={`/orders/${order.id}`}>
                        <Button variant="ghost" size="sm" data-testid={`button-view-order-${order.id}`}>
                          <Eye className="h-4 w-4" />
                        </Button>
                      </Link>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          ) : (
            <div className="p-8 text-center text-muted-foreground">
              <CalendarCheck className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p>لا توجد طلبات مجدولة</p>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
