import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatDate } from "@/lib/formatters";
import { Clock, Plus, Pencil, Trash2, Users, Timer, Calendar, ClockArrowUp } from "lucide-react";
import { format } from "date-fns";
import { ar } from "date-fns/locale";

interface Employee {
  id: number;
  nameAr: string;
}

interface OvertimeEntry {
  id: number;
  employeeId: number;
  date: string;
  startTime: string;
  endTime: string;
  totalMinutes: number;
  reason: string | null;
  approvedByEmployeeId: number | null;
  createdAt: string;
  employee?: Employee;
  approvedBy?: Employee;
}

interface OvertimeFormData {
  employeeId: number | null;
  date: string;
  startTime: string;
  endTime: string;
  totalMinutes: number;
  reason: string;
  approvedByEmployeeId: number | null;
}

export default function OvertimePage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingEntry, setEditingEntry] = useState<OvertimeEntry | null>(null);
  const [formData, setFormData] = useState<OvertimeFormData>({
    employeeId: null,
    date: format(new Date(), "yyyy-MM-dd"),
    startTime: "",
    endTime: "",
    totalMinutes: 0,
    reason: "",
    approvedByEmployeeId: null,
  });
  const { toast } = useToast();

  const { data: entries, isLoading } = useQuery<OvertimeEntry[]>({
    queryKey: ["/api/hr/overtime"],
  });

  const { data: employees } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const createMutation = useMutation({
    mutationFn: (data: OvertimeFormData) =>
      apiRequest("POST", "/api/hr/overtime", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/overtime"] });
      toast({ title: "تم إنشاء سجل الوقت الإضافي بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء سجل الوقت الإضافي",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<OvertimeFormData> }) =>
      apiRequest("PUT", `/api/hr/overtime/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/overtime"] });
      toast({ title: "تم تحديث سجل الوقت الإضافي بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث سجل الوقت الإضافي",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/hr/overtime/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/overtime"] });
      toast({ title: "تم حذف سجل الوقت الإضافي بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف سجل الوقت الإضافي",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      employeeId: null,
      date: format(new Date(), "yyyy-MM-dd"),
      startTime: "",
      endTime: "",
      totalMinutes: 0,
      reason: "",
      approvedByEmployeeId: null,
    });
    setEditingEntry(null);
    setIsDialogOpen(false);
  };

  const handleOpenDialog = (entry?: OvertimeEntry) => {
    if (entry) {
      setEditingEntry(entry);
      setFormData({
        employeeId: entry.employeeId,
        date: entry.date,
        startTime: entry.startTime,
        endTime: entry.endTime,
        totalMinutes: entry.totalMinutes,
        reason: entry.reason || "",
        approvedByEmployeeId: entry.approvedByEmployeeId,
      });
    } else {
      setFormData({
        employeeId: null,
        date: format(new Date(), "yyyy-MM-dd"),
        startTime: "",
        endTime: "",
        totalMinutes: 0,
        reason: "",
        approvedByEmployeeId: null,
      });
      setEditingEntry(null);
    }
    setIsDialogOpen(true);
  };

  const calculateTotalMinutes = (start: string, end: string) => {
    if (!start || !end) return 0;
    const [startH, startM] = start.split(":").map(Number);
    const [endH, endM] = end.split(":").map(Number);
    let minutes = (endH * 60 + endM) - (startH * 60 + startM);
    if (minutes < 0) minutes += 24 * 60;
    return minutes;
  };

  const handleTimeChange = (field: "startTime" | "endTime", value: string) => {
    const newFormData = { ...formData, [field]: value };
    newFormData.totalMinutes = calculateTotalMinutes(
      field === "startTime" ? value : formData.startTime,
      field === "endTime" ? value : formData.endTime
    );
    setFormData(newFormData);
  };

  const handleSubmit = () => {
    if (!formData.employeeId) {
      toast({ title: "الموظف مطلوب", variant: "destructive" });
      return;
    }
    if (!formData.date) {
      toast({ title: "التاريخ مطلوب", variant: "destructive" });
      return;
    }
    if (!formData.startTime || !formData.endTime) {
      toast({ title: "وقت البداية والنهاية مطلوبان", variant: "destructive" });
      return;
    }
    if (formData.totalMinutes <= 0) {
      toast({ title: "مدة الوقت الإضافي يجب أن تكون أكبر من صفر", variant: "destructive" });
      return;
    }

    if (editingEntry) {
      updateMutation.mutate({ id: editingEntry.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const getEmployeeName = (id: number) => {
    return employees?.find(e => e.id === id)?.nameAr || `موظف #${id}`;
  };

  const formatMinutes = (minutes: number) => {
    if (!minutes) return "-";
    const hours = Math.floor(minutes / 60);
    const mins = minutes % 60;
    if (hours > 0) {
      return `${hours}س ${mins > 0 ? mins + "د" : ""}`.trim();
    }
    return `${mins}د`;
  };

  const totalOvertimeMinutes = entries?.reduce((sum, e) => sum + e.totalMinutes, 0) || 0;
  const uniqueEmployees = new Set(entries?.map(e => e.employeeId) || []);

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">الوقت الإضافي</h1>
          <p className="text-muted-foreground">تسجيل ساعات العمل الإضافية</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-overtime">
          <Plus className="h-4 w-4 ml-2" />
          إضافة وقت إضافي
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <ClockArrowUp className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي السجلات</p>
                <p className="text-xl font-bold" data-testid="text-total-entries">
                  {entries?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200">
                <Timer className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الوقت الإضافي</p>
                <p className="text-xl font-bold" data-testid="text-total-overtime">
                  {formatMinutes(totalOvertimeMinutes)}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <Users className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">موظفين بوقت إضافي</p>
                <p className="text-xl font-bold" data-testid="text-employees-with-overtime">
                  {uniqueEmployees.size}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <ClockArrowUp className="h-5 w-5" />
            سجلات الوقت الإضافي
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">الموظف</TableHead>
                  <TableHead className="text-right">التاريخ</TableHead>
                  <TableHead className="text-right">الوقت</TableHead>
                  <TableHead className="text-right">المدة</TableHead>
                  <TableHead className="text-right">السبب</TableHead>
                  <TableHead className="text-right">معتمد من</TableHead>
                  <TableHead className="text-left">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {entries?.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={7} className="text-center py-8 text-muted-foreground">
                      لا توجد سجلات وقت إضافي
                    </TableCell>
                  </TableRow>
                ) : (
                  entries?.map((entry) => (
                    <TableRow key={entry.id} data-testid={`row-overtime-${entry.id}`}>
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <Users className="h-4 w-4 text-muted-foreground" />
                          {entry.employee?.nameAr || getEmployeeName(entry.employeeId)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1 text-sm">
                          <Calendar className="h-3 w-3 text-muted-foreground" />
                          {formatDate(entry.date)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1 text-sm">
                          <Clock className="h-3 w-3 text-muted-foreground" />
                          <span dir="ltr">{entry.startTime} - {entry.endTime}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant="secondary">
                          {formatMinutes(entry.totalMinutes)}
                        </Badge>
                      </TableCell>
                      <TableCell className="max-w-xs truncate">
                        {entry.reason || "-"}
                      </TableCell>
                      <TableCell>
                        {entry.approvedByEmployeeId ? (
                          <span className="text-sm">
                            {entry.approvedBy?.nameAr || getEmployeeName(entry.approvedByEmployeeId)}
                          </span>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleOpenDialog(entry)}
                            data-testid={`button-edit-overtime-${entry.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => deleteMutation.mutate(entry.id)}
                            disabled={deleteMutation.isPending}
                            data-testid={`button-delete-overtime-${entry.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-lg" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingEntry ? "تعديل سجل الوقت الإضافي" : "إضافة وقت إضافي جديد"}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="employeeId">الموظف <span className="text-destructive">*</span></Label>
                <Select
                  value={formData.employeeId?.toString() || ""}
                  onValueChange={(value) => setFormData({ ...formData, employeeId: parseInt(value) })}
                >
                  <SelectTrigger data-testid="select-overtime-employee">
                    <SelectValue placeholder="اختر الموظف" />
                  </SelectTrigger>
                  <SelectContent>
                    {employees?.map((employee) => (
                      <SelectItem key={employee.id} value={employee.id.toString()}>
                        {employee.nameAr}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="date">التاريخ <span className="text-destructive">*</span></Label>
                <Input
                  id="date"
                  type="date"
                  value={formData.date}
                  onChange={(e) => setFormData({ ...formData, date: e.target.value })}
                  data-testid="input-overtime-date"
                />
              </div>
            </div>
            <div className="grid grid-cols-3 gap-4">
              <div className="space-y-2">
                <Label htmlFor="startTime">وقت البداية <span className="text-destructive">*</span></Label>
                <Input
                  id="startTime"
                  type="time"
                  value={formData.startTime}
                  onChange={(e) => handleTimeChange("startTime", e.target.value)}
                  data-testid="input-overtime-start"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="endTime">وقت النهاية <span className="text-destructive">*</span></Label>
                <Input
                  id="endTime"
                  type="time"
                  value={formData.endTime}
                  onChange={(e) => handleTimeChange("endTime", e.target.value)}
                  data-testid="input-overtime-end"
                />
              </div>
              <div className="space-y-2">
                <Label>المدة</Label>
                <div className="h-9 px-3 py-2 border rounded-md bg-muted text-sm">
                  {formatMinutes(formData.totalMinutes)}
                </div>
              </div>
            </div>
            <div className="space-y-2">
              <Label htmlFor="reason">السبب</Label>
              <Textarea
                id="reason"
                value={formData.reason}
                onChange={(e) => setFormData({ ...formData, reason: e.target.value })}
                placeholder="سبب العمل الإضافي"
                rows={2}
                data-testid="input-overtime-reason"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="approvedByEmployeeId">معتمد من</Label>
              <Select
                value={formData.approvedByEmployeeId?.toString() || ""}
                onValueChange={(value) => setFormData({ ...formData, approvedByEmployeeId: value ? parseInt(value) : null })}
              >
                <SelectTrigger data-testid="select-overtime-approver">
                  <SelectValue placeholder="اختر المعتمد" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="">بدون اعتماد</SelectItem>
                  {employees?.map((employee) => (
                    <SelectItem key={employee.id} value={employee.id.toString()}>
                      {employee.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
          <DialogFooter className="gap-2 sm:gap-0">
            <Button
              variant="outline"
              onClick={resetForm}
              data-testid="button-cancel-overtime"
            >
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-save-overtime"
            >
              {createMutation.isPending || updateMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
