import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Textarea } from "@/components/ui/textarea";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Package, Plus, Pencil, Trash2, Calendar, CheckCircle2, XCircle, Gift, Scissors } from "lucide-react";
import { format } from "date-fns";

interface Service {
  id: number;
  nameAr: string;
  price: number;
  isActive: boolean;
}

interface Product {
  id: number;
  nameAr: string;
  defaultSellPrice: number | null;
  isActive: boolean;
}

interface PackageServiceItem {
  id: number;
  packageId: number;
  serviceId: number;
  quantity: number;
  packagePrice: number;
  service?: Service;
}

interface PackageProductItem {
  id: number;
  packageId: number;
  productId: number;
  quantity: number;
  packagePrice: number;
  product?: Product;
}

interface PackageData {
  id: number;
  nameAr: string;
  imageUrl: string | null;
  startDate: string;
  endDate: string;
  descriptionAr: string | null;
  isActive: boolean;
  services: PackageServiceItem[];
  products: PackageProductItem[];
  createdAt: string;
  updatedAt: string;
}

interface PackageServiceFormItem {
  serviceId: number;
  quantity: number;
  packagePrice: number;
}

interface PackageProductFormItem {
  productId: number;
  quantity: number;
  packagePrice: number;
}

interface PackageFormData {
  nameAr: string;
  imageUrl: string;
  startDate: string;
  endDate: string;
  descriptionAr: string;
  isActive: boolean;
  services: PackageServiceFormItem[];
  products: PackageProductFormItem[];
}

export default function PackagesPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingPackage, setEditingPackage] = useState<PackageData | null>(null);
  const [formData, setFormData] = useState<PackageFormData>({
    nameAr: "",
    imageUrl: "",
    startDate: "",
    endDate: "",
    descriptionAr: "",
    isActive: true,
    services: [],
    products: [],
  });
  const { toast } = useToast();

  const { data: packages, isLoading } = useQuery<PackageData[]>({
    queryKey: ["/api/marketing/packages"],
  });

  const { data: servicesData } = useQuery<Service[]>({
    queryKey: ["/api/services"],
  });

  const { data: productsData } = useQuery<Product[]>({
    queryKey: ["/api/products"],
  });

  const createMutation = useMutation({
    mutationFn: (data: PackageFormData) =>
      apiRequest("POST", "/api/marketing/packages", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/marketing/packages"] });
      toast({ title: "تم إنشاء الحزمة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء الحزمة",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<PackageFormData> }) =>
      apiRequest("PUT", `/api/marketing/packages/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/marketing/packages"] });
      toast({ title: "تم تحديث الحزمة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث الحزمة",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/marketing/packages/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/marketing/packages"] });
      toast({ title: "تم إلغاء تفعيل الحزمة بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف الحزمة",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      nameAr: "",
      imageUrl: "",
      startDate: "",
      endDate: "",
      descriptionAr: "",
      isActive: true,
      services: [],
      products: [],
    });
    setEditingPackage(null);
    setIsDialogOpen(false);
  };

  const handleOpenDialog = (pkg?: PackageData) => {
    if (pkg) {
      setEditingPackage(pkg);
      setFormData({
        nameAr: pkg.nameAr,
        imageUrl: pkg.imageUrl || "",
        startDate: pkg.startDate,
        endDate: pkg.endDate,
        descriptionAr: pkg.descriptionAr || "",
        isActive: pkg.isActive,
        services: pkg.services.map(s => ({
          serviceId: s.serviceId,
          quantity: s.quantity,
          packagePrice: s.packagePrice,
        })),
        products: pkg.products.map(p => ({
          productId: p.productId,
          quantity: p.quantity,
          packagePrice: p.packagePrice,
        })),
      });
    } else {
      setFormData({
        nameAr: "",
        imageUrl: "",
        startDate: "",
        endDate: "",
        descriptionAr: "",
        isActive: true,
        services: [],
        products: [],
      });
      setEditingPackage(null);
    }
    setIsDialogOpen(true);
  };

  const handleSubmit = () => {
    if (!formData.nameAr.trim()) {
      toast({ title: "اسم الحزمة مطلوب", variant: "destructive" });
      return;
    }
    if (!formData.startDate || !formData.endDate) {
      toast({ title: "تاريخ البداية والنهاية مطلوبان", variant: "destructive" });
      return;
    }
    if (formData.services.length === 0) {
      toast({ title: "يجب إضافة خدمة واحدة على الأقل", variant: "destructive" });
      return;
    }

    if (editingPackage) {
      updateMutation.mutate({ id: editingPackage.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const addService = () => {
    setFormData(prev => ({
      ...prev,
      services: [...prev.services, { serviceId: 0, quantity: 1, packagePrice: 0 }],
    }));
  };

  const removeService = (index: number) => {
    setFormData(prev => ({
      ...prev,
      services: prev.services.filter((_, i) => i !== index),
    }));
  };

  const updateService = (index: number, field: keyof PackageServiceFormItem, value: number) => {
    setFormData(prev => ({
      ...prev,
      services: prev.services.map((s, i) => i === index ? { ...s, [field]: value } : s),
    }));
  };

  const addProduct = () => {
    setFormData(prev => ({
      ...prev,
      products: [...prev.products, { productId: 0, quantity: 1, packagePrice: 0 }],
    }));
  };

  const removeProduct = (index: number) => {
    setFormData(prev => ({
      ...prev,
      products: prev.products.filter((_, i) => i !== index),
    }));
  };

  const updateProduct = (index: number, field: keyof PackageProductFormItem, value: number) => {
    setFormData(prev => ({
      ...prev,
      products: prev.products.map((p, i) => i === index ? { ...p, [field]: value } : p),
    }));
  };

  const isPackageValid = (pkg: PackageData) => {
    const today = new Date().toISOString().split('T')[0];
    return pkg.isActive && pkg.startDate <= today && pkg.endDate >= today;
  };

  const activePackages = packages?.filter(p => p.isActive) || [];
  const inactivePackages = packages?.filter(p => !p.isActive) || [];
  const availableServices = servicesData?.filter(s => s.isActive) || [];
  const availableProducts = productsData?.filter(p => p.isActive) || [];

  const calculatePackageTotal = (pkg: PackageData) => {
    const servicesTotal = pkg.services.reduce((sum, s) => sum + (s.packagePrice * s.quantity), 0);
    const productsTotal = pkg.products.reduce((sum, p) => sum + (p.packagePrice * p.quantity), 0);
    return servicesTotal + productsTotal;
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">الحزم</h1>
          <p className="text-muted-foreground">إدارة حزم الخدمات والمنتجات</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-package">
          <Plus className="h-4 w-4 ml-2" />
          إضافة حزمة جديدة
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <Package className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الحزم</p>
                <p className="text-xl font-bold" data-testid="text-total-packages">
                  {packages?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-500/10 text-green-600">
                <CheckCircle2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">الحزم النشطة</p>
                <p className="text-xl font-bold" data-testid="text-active-packages">
                  {activePackages.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-blue-500/10 text-blue-600">
                <Gift className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">الحزم الصالحة اليوم</p>
                <p className="text-xl font-bold" data-testid="text-valid-packages">
                  {packages?.filter(p => isPackageValid(p)).length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Package className="h-5 w-5" />
            قائمة الحزم
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              <Skeleton className="h-10 w-full" />
              <Skeleton className="h-10 w-full" />
              <Skeleton className="h-10 w-full" />
            </div>
          ) : packages && packages.length > 0 ? (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>اسم الحزمة</TableHead>
                  <TableHead>تاريخ البداية</TableHead>
                  <TableHead>تاريخ النهاية</TableHead>
                  <TableHead>الخدمات</TableHead>
                  <TableHead>المنتجات</TableHead>
                  <TableHead>السعر الإجمالي</TableHead>
                  <TableHead>الحالة</TableHead>
                  <TableHead>إجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {packages.map((pkg) => (
                  <TableRow key={pkg.id} data-testid={`row-package-${pkg.id}`}>
                    <TableCell className="font-medium">{pkg.nameAr}</TableCell>
                    <TableCell>{format(new Date(pkg.startDate), 'yyyy-MM-dd')}</TableCell>
                    <TableCell>{format(new Date(pkg.endDate), 'yyyy-MM-dd')}</TableCell>
                    <TableCell>
                      <Badge variant="outline">{pkg.services.length} خدمة</Badge>
                    </TableCell>
                    <TableCell>
                      <Badge variant="outline">{pkg.products.length} منتج</Badge>
                    </TableCell>
                    <TableCell className="font-bold">
                      {calculatePackageTotal(pkg).toFixed(2)} ر.س
                    </TableCell>
                    <TableCell>
                      {isPackageValid(pkg) ? (
                        <Badge className="bg-green-500">صالحة</Badge>
                      ) : pkg.isActive ? (
                        <Badge variant="secondary">غير صالحة</Badge>
                      ) : (
                        <Badge variant="destructive">معطلة</Badge>
                      )}
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => handleOpenDialog(pkg)}
                          data-testid={`button-edit-package-${pkg.id}`}
                        >
                          <Pencil className="h-4 w-4" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => deleteMutation.mutate(pkg.id)}
                          disabled={deleteMutation.isPending}
                          data-testid={`button-delete-package-${pkg.id}`}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          ) : (
            <div className="text-center py-8 text-muted-foreground">
              لا توجد حزم. اضغط على "إضافة حزمة جديدة" للبدء.
            </div>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="max-w-3xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>
              {editingPackage ? "تعديل الحزمة" : "إضافة حزمة جديدة"}
            </DialogTitle>
          </DialogHeader>

          <div className="space-y-6 py-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="nameAr">اسم الحزمة *</Label>
                <Input
                  id="nameAr"
                  value={formData.nameAr}
                  onChange={(e) => setFormData(prev => ({ ...prev, nameAr: e.target.value }))}
                  placeholder="مثال: بكج عروس VIP"
                  data-testid="input-package-name"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="imageUrl">رابط الصورة</Label>
                <Input
                  id="imageUrl"
                  value={formData.imageUrl}
                  onChange={(e) => setFormData(prev => ({ ...prev, imageUrl: e.target.value }))}
                  placeholder="https://..."
                  data-testid="input-package-image"
                />
              </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="startDate">تاريخ البداية *</Label>
                <Input
                  id="startDate"
                  type="date"
                  value={formData.startDate}
                  onChange={(e) => setFormData(prev => ({ ...prev, startDate: e.target.value }))}
                  data-testid="input-package-start-date"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="endDate">تاريخ النهاية *</Label>
                <Input
                  id="endDate"
                  type="date"
                  value={formData.endDate}
                  onChange={(e) => setFormData(prev => ({ ...prev, endDate: e.target.value }))}
                  data-testid="input-package-end-date"
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="descriptionAr">الوصف</Label>
              <Textarea
                id="descriptionAr"
                value={formData.descriptionAr}
                onChange={(e) => setFormData(prev => ({ ...prev, descriptionAr: e.target.value }))}
                placeholder="وصف الحزمة..."
                data-testid="input-package-description"
              />
            </div>

            <div className="flex items-center gap-2">
              <Switch
                id="isActive"
                checked={formData.isActive}
                onCheckedChange={(checked) => setFormData(prev => ({ ...prev, isActive: checked }))}
                data-testid="switch-package-active"
              />
              <Label htmlFor="isActive">الحزمة نشطة</Label>
            </div>

            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <Label className="text-lg font-semibold flex items-center gap-2">
                  <Scissors className="h-4 w-4" />
                  الخدمات *
                </Label>
                <Button type="button" variant="outline" size="sm" onClick={addService} data-testid="button-add-service">
                  <Plus className="h-4 w-4 ml-1" />
                  إضافة خدمة
                </Button>
              </div>
              {formData.services.length === 0 ? (
                <p className="text-sm text-muted-foreground">لم يتم إضافة خدمات بعد</p>
              ) : (
                <div className="space-y-3">
                  {formData.services.map((svc, index) => (
                    <div key={index} className="flex items-center gap-2 p-3 border rounded-lg">
                      <Select
                        value={svc.serviceId.toString()}
                        onValueChange={(value) => updateService(index, 'serviceId', parseInt(value))}
                      >
                        <SelectTrigger className="flex-1" data-testid={`select-service-${index}`}>
                          <SelectValue placeholder="اختر خدمة" />
                        </SelectTrigger>
                        <SelectContent>
                          {availableServices.map(s => (
                            <SelectItem key={s.id} value={s.id.toString()}>
                              {s.nameAr} ({s.price} ر.س)
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <Input
                        type="number"
                        min="1"
                        value={svc.quantity}
                        onChange={(e) => updateService(index, 'quantity', parseFloat(e.target.value) || 1)}
                        className="w-20"
                        placeholder="الكمية"
                        data-testid={`input-service-quantity-${index}`}
                      />
                      <Input
                        type="number"
                        min="0"
                        step="0.01"
                        value={svc.packagePrice}
                        onChange={(e) => updateService(index, 'packagePrice', parseFloat(e.target.value) || 0)}
                        className="w-28"
                        placeholder="سعر الحزمة"
                        data-testid={`input-service-price-${index}`}
                      />
                      <Button
                        type="button"
                        variant="ghost"
                        size="icon"
                        onClick={() => removeService(index)}
                        data-testid={`button-remove-service-${index}`}
                      >
                        <Trash2 className="h-4 w-4 text-destructive" />
                      </Button>
                    </div>
                  ))}
                </div>
              )}
            </div>

            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <Label className="text-lg font-semibold flex items-center gap-2">
                  <Package className="h-4 w-4" />
                  المنتجات
                </Label>
                <Button type="button" variant="outline" size="sm" onClick={addProduct} data-testid="button-add-product">
                  <Plus className="h-4 w-4 ml-1" />
                  إضافة منتج
                </Button>
              </div>
              {formData.products.length === 0 ? (
                <p className="text-sm text-muted-foreground">لم يتم إضافة منتجات بعد (اختياري)</p>
              ) : (
                <div className="space-y-3">
                  {formData.products.map((prod, index) => (
                    <div key={index} className="flex items-center gap-2 p-3 border rounded-lg">
                      <Select
                        value={prod.productId.toString()}
                        onValueChange={(value) => updateProduct(index, 'productId', parseInt(value))}
                      >
                        <SelectTrigger className="flex-1" data-testid={`select-product-${index}`}>
                          <SelectValue placeholder="اختر منتج" />
                        </SelectTrigger>
                        <SelectContent>
                          {availableProducts.map(p => (
                            <SelectItem key={p.id} value={p.id.toString()}>
                              {p.nameAr} ({p.defaultSellPrice || 0} ر.س)
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <Input
                        type="number"
                        min="1"
                        value={prod.quantity}
                        onChange={(e) => updateProduct(index, 'quantity', parseFloat(e.target.value) || 1)}
                        className="w-20"
                        placeholder="الكمية"
                        data-testid={`input-product-quantity-${index}`}
                      />
                      <Input
                        type="number"
                        min="0"
                        step="0.01"
                        value={prod.packagePrice}
                        onChange={(e) => updateProduct(index, 'packagePrice', parseFloat(e.target.value) || 0)}
                        className="w-28"
                        placeholder="سعر الحزمة"
                        data-testid={`input-product-price-${index}`}
                      />
                      <Button
                        type="button"
                        variant="ghost"
                        size="icon"
                        onClick={() => removeProduct(index)}
                        data-testid={`button-remove-product-${index}`}
                      >
                        <Trash2 className="h-4 w-4 text-destructive" />
                      </Button>
                    </div>
                  ))}
                </div>
              )}
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={resetForm} data-testid="button-cancel">
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-submit"
            >
              {editingPackage ? "حفظ التغييرات" : "إنشاء الحزمة"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
