import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Link } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";
import {
  Loader2,
  AlertTriangle,
  Clock,
  CreditCard,
  Users,
  FileText,
  ExternalLink,
} from "lucide-react";
import { format } from "date-fns";
import { ar } from "date-fns/locale";
import { formatCurrency } from "@/lib/formatters";

interface SupplierPayable {
  supplierId: number;
  supplierNameAr: string;
  totalOutstanding: number;
  overdueAmount: number;
  invoiceCountUnpaid: number;
  invoiceCountOverdue: number;
}

interface PayablesSummary {
  globalSummary: {
    totalOutstanding: number;
    totalOverdue: number;
    agingBuckets: {
      "0_30": number;
      "31_60": number;
      "61_90": number;
      "90_plus": number;
    };
  };
  suppliers: SupplierPayable[];
}

interface StatementTransaction {
  date: string;
  type: "invoice" | "payment";
  referenceNumber: string;
  descriptionAr: string;
  debitAmount: number;
  creditAmount: number;
  balanceAfter: number;
}

interface SupplierStatement {
  supplier: {
    id: number;
    nameAr: string;
    agreementText: string | null;
    paymentTermDays: number | null;
  };
  openingBalance: {
    amount: number;
    direction: "credit" | "debit";
  };
  transactions: StatementTransaction[];
  closingBalance: {
    amount: number;
    explanation: string;
  };
}

export default function PayablesPage() {
  const [selectedSupplierId, setSelectedSupplierId] = useState<number | null>(null);
  const [statementFilters, setStatementFilters] = useState({
    from_date: "",
    to_date: "",
  });

  const { data: summary, isLoading } = useQuery<PayablesSummary>({
    queryKey: ["/api/reports/payables-summary"],
  });

  const statementQuery = selectedSupplierId
    ? Object.entries(statementFilters)
        .filter(([_, v]) => v)
        .map(([k, v]) => `${k}=${encodeURIComponent(v)}`)
        .join("&")
    : "";

  const { data: statement, isLoading: isLoadingStatement } = useQuery<SupplierStatement>({
    queryKey: ["/api/suppliers", selectedSupplierId, "statement", statementQuery],
    queryFn: async () => {
      const res = await fetch(
        `/api/suppliers/${selectedSupplierId}/statement${statementQuery ? `?${statementQuery}` : ""}`
      );
      if (!res.ok) throw new Error("Failed to fetch statement");
      return res.json();
    },
    enabled: !!selectedSupplierId,
  });

  const formatAmount = formatCurrency;

  const getAgingColor = (bucket: string) => {
    switch (bucket) {
      case "0-30":
        return "bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400";
      case "31-60":
        return "bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400";
      case "61-90":
        return "bg-orange-100 text-orange-800 dark:bg-orange-900/30 dark:text-orange-400";
      case "90+":
        return "bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400";
      default:
        return "";
    }
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-96">
        <Loader2 className="h-8 w-8 animate-spin" />
      </div>
    );
  }

  return (
    <div className="space-y-6" dir="rtl">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold">الديون والمستحقات</h1>
      </div>

      {summary && (
        <>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <Card>
              <CardContent className="p-4">
                <div className="flex items-center gap-4">
                  <div className="p-3 rounded-full bg-primary/10">
                    <CreditCard className="h-6 w-6 text-primary" />
                  </div>
                  <div>
                    <div className="text-sm text-muted-foreground">إجمالي المستحقات</div>
                    <div className="text-2xl font-bold">{formatAmount(summary.globalSummary.totalOutstanding)}</div>
                  </div>
                </div>
              </CardContent>
            </Card>
            <Card>
              <CardContent className="p-4">
                <div className="flex items-center gap-4">
                  <div className="p-3 rounded-full bg-destructive/10">
                    <AlertTriangle className="h-6 w-6 text-destructive" />
                  </div>
                  <div>
                    <div className="text-sm text-muted-foreground">متأخرات</div>
                    <div className="text-2xl font-bold text-destructive">
                      {formatAmount(summary.globalSummary.totalOverdue)}
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
            <Card>
              <CardContent className="p-4">
                <div className="flex items-center gap-4">
                  <div className="p-3 rounded-full bg-muted">
                    <Users className="h-6 w-6" />
                  </div>
                  <div>
                    <div className="text-sm text-muted-foreground">موردين بمستحقات</div>
                    <div className="text-2xl font-bold">{summary.suppliers.length}</div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Clock className="h-5 w-5" />
                تقادم الديون (Aging)
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-4 gap-4 mb-6">
                <div className={`p-4 rounded-lg ${getAgingColor("0-30")}`}>
                  <div className="text-sm font-medium">0-30 يوم</div>
                  <div className="text-2xl font-bold">{formatAmount(summary.globalSummary.agingBuckets["0_30"])}</div>
                </div>
                <div className={`p-4 rounded-lg ${getAgingColor("31-60")}`}>
                  <div className="text-sm font-medium">31-60 يوم</div>
                  <div className="text-2xl font-bold">{formatAmount(summary.globalSummary.agingBuckets["31_60"])}</div>
                </div>
                <div className={`p-4 rounded-lg ${getAgingColor("61-90")}`}>
                  <div className="text-sm font-medium">61-90 يوم</div>
                  <div className="text-2xl font-bold">{formatAmount(summary.globalSummary.agingBuckets["61_90"])}</div>
                </div>
                <div className={`p-4 rounded-lg ${getAgingColor("90+")}`}>
                  <div className="text-sm font-medium">أكثر من 90 يوم</div>
                  <div className="text-2xl font-bold">{formatAmount(summary.globalSummary.agingBuckets["90_plus"])}</div>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <FileText className="h-5 w-5" />
                تفاصيل الموردين
              </CardTitle>
            </CardHeader>
            <CardContent>
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">المورد</TableHead>
                    <TableHead className="text-right">عدد الفواتير</TableHead>
                    <TableHead className="text-right">المستحق</TableHead>
                    <TableHead className="text-right">المتأخر</TableHead>
                    <TableHead className="text-right">الإجراءات</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {summary.suppliers.length === 0 ? (
                    <TableRow>
                      <TableCell colSpan={5} className="text-center text-muted-foreground">
                        لا توجد مستحقات
                      </TableCell>
                    </TableRow>
                  ) : (
                    summary.suppliers.map((supplier) => (
                      <TableRow key={supplier.supplierId} data-testid={`row-supplier-${supplier.supplierId}`}>
                        <TableCell className="font-medium">
                          <Link href={`/suppliers/${supplier.supplierId}`}>
                            <span className="text-primary hover:underline cursor-pointer">
                              {supplier.supplierNameAr}
                            </span>
                          </Link>
                        </TableCell>
                        <TableCell>
                          <Badge variant="secondary">{supplier.invoiceCountUnpaid}</Badge>
                        </TableCell>
                        <TableCell className="font-bold">{formatAmount(supplier.totalOutstanding)}</TableCell>
                        <TableCell>
                          {supplier.overdueAmount > 0 ? (
                            <span className="text-destructive font-medium">
                              {formatAmount(supplier.overdueAmount)}
                            </span>
                          ) : (
                            "-"
                          )}
                        </TableCell>
                        <TableCell>
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={() => setSelectedSupplierId(supplier.supplierId)}
                            data-testid={`button-statement-${supplier.supplierId}`}
                          >
                            <ExternalLink className="h-4 w-4 ml-1" />
                            كشف حساب
                          </Button>
                        </TableCell>
                      </TableRow>
                    ))
                  )}
                </TableBody>
              </Table>
            </CardContent>
          </Card>
        </>
      )}

      <Dialog open={!!selectedSupplierId} onOpenChange={(open) => !open && setSelectedSupplierId(null)}>
        <DialogContent className="max-w-4xl max-h-[80vh] overflow-y-auto" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              كشف حساب المورد - {statement?.supplier.nameAr}
            </DialogTitle>
          </DialogHeader>

          {isLoadingStatement ? (
            <div className="flex items-center justify-center h-48">
              <Loader2 className="h-8 w-8 animate-spin" />
            </div>
          ) : statement ? (
            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4 p-4 bg-muted/50 rounded-lg">
                <div>
                  <div className="text-sm text-muted-foreground">شروط السداد</div>
                  <div className="font-medium">
                    {statement.supplier.paymentTermDays
                      ? `${statement.supplier.paymentTermDays} يوم`
                      : "غير محدد"}
                  </div>
                </div>
                {statement.supplier.agreementText && (
                  <div className="col-span-2">
                    <div className="text-sm text-muted-foreground">نص الاتفاقية</div>
                    <div className="font-medium">{statement.supplier.agreementText}</div>
                  </div>
                )}
              </div>

              <div className="flex gap-4">
                <div className="flex-1 space-y-2">
                  <Label>من تاريخ</Label>
                  <Input
                    type="date"
                    value={statementFilters.from_date}
                    onChange={(e) =>
                      setStatementFilters({ ...statementFilters, from_date: e.target.value })
                    }
                    data-testid="input-statement-from-date"
                  />
                </div>
                <div className="flex-1 space-y-2">
                  <Label>إلى تاريخ</Label>
                  <Input
                    type="date"
                    value={statementFilters.to_date}
                    onChange={(e) =>
                      setStatementFilters({ ...statementFilters, to_date: e.target.value })
                    }
                    data-testid="input-statement-to-date"
                  />
                </div>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <Card>
                  <CardContent className="p-4">
                    <div className="text-sm text-muted-foreground">الرصيد الافتتاحي</div>
                    <div className="text-xl font-bold">
                      {formatAmount(statement.openingBalance.amount)}
                      <span className="text-sm text-muted-foreground mr-2">
                        ({statement.openingBalance.direction === "credit" ? "دائن" : "مدين"})
                      </span>
                    </div>
                  </CardContent>
                </Card>
                <Card>
                  <CardContent className="p-4">
                    <div className="text-sm text-muted-foreground">الرصيد الختامي</div>
                    <div className="text-xl font-bold text-primary">
                      {formatAmount(statement.closingBalance.amount)}
                    </div>
                  </CardContent>
                </Card>
              </div>

              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">التاريخ</TableHead>
                    <TableHead className="text-right">النوع</TableHead>
                    <TableHead className="text-right">المرجع</TableHead>
                    <TableHead className="text-right">الوصف</TableHead>
                    <TableHead className="text-right">مدين</TableHead>
                    <TableHead className="text-right">دائن</TableHead>
                    <TableHead className="text-right">الرصيد</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  <TableRow className="bg-muted/30">
                    <TableCell colSpan={6} className="font-medium">
                      رصيد افتتاحي
                    </TableCell>
                    <TableCell className="font-bold">{formatAmount(statement.openingBalance.amount)}</TableCell>
                  </TableRow>
                  {statement.transactions.map((tx, idx) => (
                    <TableRow key={`${tx.type}-${tx.referenceNumber}-${idx}`}>
                      <TableCell>
                        {format(new Date(tx.date), "dd/MM/yyyy", { locale: ar })}
                      </TableCell>
                      <TableCell>
                        <Badge variant={tx.type === "invoice" ? "default" : "secondary"}>
                          {tx.type === "invoice" ? "فاتورة" : "دفعة"}
                        </Badge>
                      </TableCell>
                      <TableCell>{tx.referenceNumber}</TableCell>
                      <TableCell>{tx.descriptionAr}</TableCell>
                      <TableCell>{tx.debitAmount > 0 ? formatAmount(tx.debitAmount) : "-"}</TableCell>
                      <TableCell className="text-green-600">
                        {tx.creditAmount > 0 ? formatAmount(tx.creditAmount) : "-"}
                      </TableCell>
                      <TableCell className="font-medium">{formatAmount(tx.balanceAfter)}</TableCell>
                    </TableRow>
                  ))}
                  <TableRow className="bg-muted/50 font-bold">
                    <TableCell colSpan={6}>الرصيد الختامي</TableCell>
                    <TableCell className="text-primary">{formatAmount(statement.closingBalance.amount)}</TableCell>
                  </TableRow>
                </TableBody>
              </Table>
            </div>
          ) : null}
        </DialogContent>
      </Dialog>
    </div>
  );
}
