import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Link } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Search, CreditCard, Banknote, Building2 } from "lucide-react";
import type { PurchasePayment, PaymentMethod, Supplier } from "@shared/schema";

type PaymentWithRelations = PurchasePayment & {
  supplier?: Supplier;
  paymentMethod?: PaymentMethod;
};

function PaymentSkeleton() {
  return (
    <TableRow>
      <TableCell><Skeleton className="h-4 w-8" /></TableCell>
      <TableCell><Skeleton className="h-4 w-24" /></TableCell>
      <TableCell><Skeleton className="h-4 w-32" /></TableCell>
      <TableCell><Skeleton className="h-4 w-20" /></TableCell>
      <TableCell><Skeleton className="h-4 w-24" /></TableCell>
      <TableCell><Skeleton className="h-4 w-16" /></TableCell>
    </TableRow>
  );
}

function EmptyState() {
  return (
    <div className="flex flex-col items-center justify-center py-16 text-center">
      <div className="flex h-16 w-16 items-center justify-center rounded-full bg-muted mb-4">
        <CreditCard className="h-8 w-8 text-muted-foreground" />
      </div>
      <h3 className="text-lg font-semibold mb-2">لا توجد مدفوعات</h3>
      <p className="text-muted-foreground mb-4">لم يتم تسجيل أي مدفوعات بعد</p>
      <Button asChild>
        <Link href="/purchase-invoices">عرض الفواتير</Link>
      </Button>
    </div>
  );
}

export default function PaymentsPage() {
  const [searchQuery, setSearchQuery] = useState("");
  const [methodFilter, setMethodFilter] = useState<string>("all");

  const { data: payments, isLoading } = useQuery<PaymentWithRelations[]>({
    queryKey: ["/api/payments"],
  });

  const { data: paymentMethods } = useQuery<PaymentMethod[]>({
    queryKey: ["/api/payment-methods"],
  });

  const filteredPayments = payments?.filter((payment) => {
    const matchesSearch =
      payment.supplier?.nameAr?.includes(searchQuery) ||
      payment.id.toString().includes(searchQuery);
    const matchesMethod =
      methodFilter === "all" || payment.paymentMethodId?.toString() === methodFilter;
    return matchesSearch && matchesMethod;
  });

  const totalPayments = filteredPayments?.reduce((sum, p) => sum + p.amount, 0) || 0;

  const getMethodIcon = (type?: string) => {
    switch (type) {
      case "cash":
        return <Banknote className="h-4 w-4 text-green-600" />;
      case "bank":
        return <Building2 className="h-4 w-4 text-blue-600" />;
      case "card":
        return <CreditCard className="h-4 w-4 text-purple-600" />;
      default:
        return <CreditCard className="h-4 w-4" />;
    }
  };

  return (
    <div className="space-y-6">
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">المدفوعات</h1>
          <p className="text-muted-foreground">سجل المدفوعات للموردين</p>
        </div>
      </div>

      <div className="grid gap-4 md:grid-cols-3">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-green-500/10">
                <Banknote className="h-6 w-6 text-green-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي المدفوعات</p>
                <p className="text-2xl font-bold tabular-nums" data-testid="total-payments">
                  {totalPayments.toLocaleString("ar-SA")} ر.س
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-primary/10">
                <CreditCard className="h-6 w-6 text-primary" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">عدد العمليات</p>
                <p className="text-2xl font-bold">
                  {filteredPayments?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-blue-500/10">
                <Building2 className="h-6 w-6 text-blue-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">طرق الدفع المتاحة</p>
                <p className="text-2xl font-bold">
                  {paymentMethods?.filter(m => m.isActive).length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader className="pb-4">
          <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
            <CardTitle className="text-lg">سجل المدفوعات</CardTitle>
            <div className="flex flex-col sm:flex-row gap-2 w-full sm:w-auto">
              <div className="relative flex-1 sm:w-72">
                <Search className="absolute right-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                <Input
                  placeholder="بحث بالمورد..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pr-10"
                  data-testid="input-search-payments"
                />
              </div>
              <Select value={methodFilter} onValueChange={setMethodFilter}>
                <SelectTrigger className="w-full sm:w-40" data-testid="select-method-filter">
                  <SelectValue placeholder="طريقة الدفع" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">الكل</SelectItem>
                  {paymentMethods?.map((method) => (
                    <SelectItem key={method.id} value={method.id.toString()}>
                      {method.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right w-16">#</TableHead>
                    <TableHead className="text-right">التاريخ</TableHead>
                    <TableHead className="text-right">المورد</TableHead>
                    <TableHead className="text-right">طريقة الدفع</TableHead>
                    <TableHead className="text-right">المبلغ</TableHead>
                    <TableHead className="text-right">الفاتورة</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {[1, 2, 3, 4, 5].map((i) => (
                    <PaymentSkeleton key={i} />
                  ))}
                </TableBody>
              </Table>
            </div>
          ) : !filteredPayments || filteredPayments.length === 0 ? (
            <EmptyState />
          ) : (
            <div className="rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right w-16">#</TableHead>
                    <TableHead className="text-right">التاريخ</TableHead>
                    <TableHead className="text-right">المورد</TableHead>
                    <TableHead className="text-right">طريقة الدفع</TableHead>
                    <TableHead className="text-right">المبلغ</TableHead>
                    <TableHead className="text-right">الفاتورة</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredPayments.map((payment) => (
                    <TableRow key={payment.id} data-testid={`payment-row-${payment.id}`}>
                      <TableCell className="font-mono text-muted-foreground">
                        {payment.id}
                      </TableCell>
                      <TableCell>
                        {new Date(payment.paymentDate).toLocaleDateString("ar-SA")}
                      </TableCell>
                      <TableCell>
                        <Link
                          href={`/suppliers/${payment.supplierId}`}
                          className="text-primary hover:underline"
                        >
                          {payment.supplier?.nameAr}
                        </Link>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          {getMethodIcon(payment.paymentMethod?.type)}
                          <span>{payment.paymentMethod?.nameAr}</span>
                        </div>
                      </TableCell>
                      <TableCell className="font-mono font-medium tabular-nums">
                        {payment.amount.toLocaleString("ar-SA")} ر.س
                      </TableCell>
                      <TableCell>
                        {payment.purchaseInvoiceId ? (
                          <Link
                            href={`/purchase-invoices/${payment.purchaseInvoiceId}`}
                            className="text-primary hover:underline"
                          >
                            #{payment.purchaseInvoiceId}
                          </Link>
                        ) : (
                          <Badge variant="outline">دفع على الحساب</Badge>
                        )}
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
