import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Checkbox } from "@/components/ui/checkbox";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";
import { ScrollArea } from "@/components/ui/scroll-area";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "@/components/ui/alert-dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber } from "@/lib/formatters";
import { 
  Calculator, Plus, Eye, FileText, DollarSign, Users, Clock, 
  CheckCircle2, CreditCard, Pencil, AlertCircle, ArrowLeft, UserCheck
} from "lucide-react";
import { format } from "date-fns";

interface Employee {
  id: number;
  nameAr: string | null;
  fullNameAr: string | null;
}

interface PayrollLine {
  id: number;
  payrollRunId: number;
  employeeId: number;
  basicSalary: number;
  fixedAllowances: number;
  overtimeAmount: number;
  commissionAmount: number;
  otherEarnings: number;
  deductionLate: number;
  deductionOther: number;
  grossSalary: number;
  totalDeductions: number;
  netSalary: number;
  employee?: Employee;
}

interface PayrollRun {
  id: number;
  periodMonth: number;
  periodYear: number;
  status: string;
  notes: string | null;
  lines?: PayrollLine[];
  totalNetSalary?: number;
  employeeCount?: number;
  createdAt: string;
  updatedAt: string;
}

interface PayrollRunFormData {
  periodMonth: number;
  periodYear: number;
  notes: string;
  selectionMode: 'all' | 'selected';
  employeeIds: number[];
}

const statusConfig: Record<string, { label: string; color: string }> = {
  draft: { label: "مسودة", color: "bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-200" },
  approved: { label: "معتمد", color: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200" },
  paid: { label: "مدفوع", color: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200" },
};

const monthNames = [
  "يناير", "فبراير", "مارس", "إبريل", "مايو", "يونيو",
  "يوليو", "أغسطس", "سبتمبر", "أكتوبر", "نوفمبر", "ديسمبر"
];

export default function PayrollPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [selectedRun, setSelectedRun] = useState<PayrollRun | null>(null);
  const [editingLine, setEditingLine] = useState<PayrollLine | null>(null);
  const [isEditLineDialogOpen, setIsEditLineDialogOpen] = useState(false);
  
  const currentDate = new Date();
  const [formData, setFormData] = useState<PayrollRunFormData>({
    periodMonth: currentDate.getMonth() + 1,
    periodYear: currentDate.getFullYear(),
    notes: "",
    selectionMode: 'all',
    employeeIds: [],
  });
  const { toast } = useToast();

  // Fetch all employees for selection
  const { data: allEmployees } = useQuery<any[]>({
    queryKey: ["/api/employees"],
  });
  
  // Filter only active employees
  const activeEmployees = allEmployees?.filter(emp => emp.isActive) || [];

  const { data: payrollRuns, isLoading } = useQuery<PayrollRun[]>({
    queryKey: ["/api/payroll-runs"],
  });

  const { data: runDetails, isLoading: isLoadingDetails } = useQuery<PayrollRun>({
    queryKey: ["/api/payroll-runs", selectedRun?.id],
    enabled: !!selectedRun,
  });

  const createMutation = useMutation({
    mutationFn: (data: PayrollRunFormData) => {
      const payload: any = {
        periodMonth: data.periodMonth,
        periodYear: data.periodYear,
        notes: data.notes,
      };
      
      // If selection mode is 'selected', include the employee IDs
      if (data.selectionMode === 'selected' && data.employeeIds.length > 0) {
        payload.employeeIds = data.employeeIds;
      } else if (data.selectionMode === 'all') {
        // Send all active employee IDs
        payload.employeeIds = activeEmployees.map(e => e.id);
      }
      
      return apiRequest("POST", "/api/payroll-runs", payload);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/payroll-runs"] });
      toast({ title: "تم إنشاء مسيّر الرواتب بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء مسيّر الرواتب",
        variant: "destructive",
      });
    },
  });

  const generateLinesMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("POST", `/api/payroll-runs/${id}/generate-lines`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/payroll-runs"] });
      queryClient.invalidateQueries({ queryKey: ["/api/payroll-runs", selectedRun?.id] });
      toast({ title: "تم توليد بنود الرواتب بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في توليد بنود الرواتب",
        variant: "destructive",
      });
    },
  });

  const approveMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("POST", `/api/payroll-runs/${id}/approve`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/payroll-runs"] });
      queryClient.invalidateQueries({ queryKey: ["/api/payroll-runs", selectedRun?.id] });
      toast({ title: "تم اعتماد مسيّر الرواتب بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في اعتماد مسيّر الرواتب",
        variant: "destructive",
      });
    },
  });

  const markPaidMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("POST", `/api/payroll-runs/${id}/mark-paid`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/payroll-runs"] });
      queryClient.invalidateQueries({ queryKey: ["/api/payroll-runs", selectedRun?.id] });
      toast({ title: "تم وسم مسيّر الرواتب كمدفوع" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في وسم مسيّر الرواتب كمدفوع",
        variant: "destructive",
      });
    },
  });

  const updateLineMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<PayrollLine> }) =>
      apiRequest("PUT", `/api/payroll-lines/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/payroll-runs", selectedRun?.id] });
      toast({ title: "تم تحديث بند الراتب بنجاح" });
      setEditingLine(null);
      setIsEditLineDialogOpen(false);
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث بند الراتب",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      periodMonth: currentDate.getMonth() + 1,
      periodYear: currentDate.getFullYear(),
      notes: "",
      selectionMode: 'all',
      employeeIds: [],
    });
    setIsDialogOpen(false);
  };
  
  const toggleEmployeeSelection = (employeeId: number) => {
    setFormData(prev => ({
      ...prev,
      employeeIds: prev.employeeIds.includes(employeeId)
        ? prev.employeeIds.filter(id => id !== employeeId)
        : [...prev.employeeIds, employeeId]
    }));
  };
  
  const selectAllEmployees = () => {
    setFormData(prev => ({
      ...prev,
      employeeIds: activeEmployees.map(e => e.id)
    }));
  };
  
  const deselectAllEmployees = () => {
    setFormData(prev => ({
      ...prev,
      employeeIds: []
    }));
  };

  const handleSubmit = () => {
    createMutation.mutate(formData);
  };

  const handleEditLine = (line: PayrollLine) => {
    setEditingLine({ ...line });
    setIsEditLineDialogOpen(true);
  };

  const handleSaveLineEdit = () => {
    if (!editingLine) return;
    updateLineMutation.mutate({
      id: editingLine.id,
      data: {
        overtimeAmount: editingLine.overtimeAmount,
        commissionAmount: editingLine.commissionAmount,
        otherEarnings: editingLine.otherEarnings,
        deductionLate: editingLine.deductionLate,
        deductionOther: editingLine.deductionOther,
      },
    });
  };

  const getEmployeeName = (emp?: Employee) => emp?.fullNameAr || emp?.nameAr || "-";

  const getStatusBadge = (status: string) => {
    const config = statusConfig[status] || { label: status, color: "bg-gray-100" };
    return <Badge className={config.color}>{config.label}</Badge>;
  };

  const totalPayroll = payrollRuns?.reduce((sum, run) => sum + (run.totalNetSalary || 0), 0) || 0;
  const draftRuns = payrollRuns?.filter(r => r.status === "draft").length || 0;
  const approvedRuns = payrollRuns?.filter(r => r.status === "approved").length || 0;
  const paidRuns = payrollRuns?.filter(r => r.status === "paid").length || 0;

  if (selectedRun && runDetails) {
    return (
      <div className="space-y-6">
        <div className="flex items-center gap-4">
          <Button variant="ghost" size="icon" onClick={() => setSelectedRun(null)}>
            <ArrowLeft className="h-5 w-5" />
          </Button>
          <div>
            <h1 className="text-2xl font-bold" data-testid="text-run-title">
              مسيّر رواتب {monthNames[runDetails.periodMonth - 1]} {runDetails.periodYear}
            </h1>
            <p className="text-muted-foreground">
              تفاصيل مسيّر الرواتب وبنوده
            </p>
          </div>
          <div className="flex-1" />
          {getStatusBadge(runDetails.status)}
        </div>

        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-3">
                <div className="p-2 rounded-lg bg-primary/10 text-primary">
                  <Users className="h-5 w-5" />
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">عدد الموظفين</p>
                  <p className="text-xl font-bold" data-testid="text-employee-count">
                    {runDetails.lines?.length || 0}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-3">
                <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                  <DollarSign className="h-5 w-5" />
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">صافي الرواتب</p>
                  <p className="text-xl font-bold" data-testid="text-total-net">
                    {formatCurrency(runDetails.totalNetSalary || 0)}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="col-span-2">
            <CardContent className="p-4">
              <div className="flex items-center gap-4 flex-wrap">
                {runDetails.status === "draft" && (
                  <>
                    <Button
                      onClick={() => generateLinesMutation.mutate(runDetails.id)}
                      disabled={generateLinesMutation.isPending}
                      data-testid="button-generate-lines"
                    >
                      <Calculator className="h-4 w-4 ml-2" />
                      توليد بنود الرواتب
                    </Button>
                    <AlertDialog>
                      <AlertDialogTrigger asChild>
                        <Button variant="outline" disabled={!runDetails.lines?.length}>
                          <CheckCircle2 className="h-4 w-4 ml-2" />
                          اعتماد المسيّر
                        </Button>
                      </AlertDialogTrigger>
                      <AlertDialogContent dir="rtl">
                        <AlertDialogHeader>
                          <AlertDialogTitle>تأكيد اعتماد مسيّر الرواتب</AlertDialogTitle>
                          <AlertDialogDescription>
                            هل أنت متأكد من اعتماد مسيّر الرواتب؟ لن تتمكن من تعديل البنود بعد الاعتماد.
                          </AlertDialogDescription>
                        </AlertDialogHeader>
                        <AlertDialogFooter>
                          <AlertDialogCancel>إلغاء</AlertDialogCancel>
                          <AlertDialogAction
                            onClick={() => approveMutation.mutate(runDetails.id)}
                          >
                            اعتماد
                          </AlertDialogAction>
                        </AlertDialogFooter>
                      </AlertDialogContent>
                    </AlertDialog>
                  </>
                )}
                {runDetails.status === "approved" && (
                  <AlertDialog>
                    <AlertDialogTrigger asChild>
                      <Button>
                        <CreditCard className="h-4 w-4 ml-2" />
                        تسجيل كمدفوع
                      </Button>
                    </AlertDialogTrigger>
                    <AlertDialogContent dir="rtl">
                      <AlertDialogHeader>
                        <AlertDialogTitle>تأكيد تسجيل الدفع</AlertDialogTitle>
                        <AlertDialogDescription>
                          هل تم صرف جميع رواتب هذا المسيّر للموظفين؟
                        </AlertDialogDescription>
                      </AlertDialogHeader>
                      <AlertDialogFooter>
                        <AlertDialogCancel>إلغاء</AlertDialogCancel>
                        <AlertDialogAction
                          onClick={() => markPaidMutation.mutate(runDetails.id)}
                        >
                          تأكيد الدفع
                        </AlertDialogAction>
                      </AlertDialogFooter>
                    </AlertDialogContent>
                  </AlertDialog>
                )}
                {runDetails.status === "paid" && (
                  <div className="flex items-center gap-2 text-green-600">
                    <CheckCircle2 className="h-5 w-5" />
                    <span className="font-medium">تم صرف جميع الرواتب</span>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>
        </div>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileText className="h-5 w-5" />
              بنود الرواتب
            </CardTitle>
          </CardHeader>
          <CardContent>
            {isLoadingDetails ? (
              <div className="space-y-2">
                {[1, 2, 3].map((i) => (
                  <Skeleton key={i} className="h-12 w-full" />
                ))}
              </div>
            ) : (
              <div className="overflow-x-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">الموظف</TableHead>
                      <TableHead className="text-right">الراتب الأساسي</TableHead>
                      <TableHead className="text-right">البدلات</TableHead>
                      <TableHead className="text-right">الإضافي</TableHead>
                      <TableHead className="text-right">العمولات</TableHead>
                      <TableHead className="text-right">أخرى</TableHead>
                      <TableHead className="text-right">إجمالي</TableHead>
                      <TableHead className="text-right">الخصومات</TableHead>
                      <TableHead className="text-right">الصافي</TableHead>
                      {runDetails.status === "draft" && (
                        <TableHead className="text-left">تعديل</TableHead>
                      )}
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {!runDetails.lines?.length ? (
                      <TableRow>
                        <TableCell colSpan={10} className="text-center py-8 text-muted-foreground">
                          <div className="flex flex-col items-center gap-2">
                            <AlertCircle className="h-8 w-8" />
                            <p>لم يتم توليد بنود الرواتب بعد</p>
                            <p className="text-sm">اضغط على "توليد بنود الرواتب" للبدء</p>
                          </div>
                        </TableCell>
                      </TableRow>
                    ) : (
                      runDetails.lines.map((line) => (
                        <TableRow key={line.id} data-testid={`row-line-${line.id}`}>
                          <TableCell className="font-medium">
                            {getEmployeeName(line.employee)}
                          </TableCell>
                          <TableCell>{formatCurrency(line.basicSalary)}</TableCell>
                          <TableCell>{formatCurrency(line.fixedAllowances)}</TableCell>
                          <TableCell>{formatCurrency(line.overtimeAmount)}</TableCell>
                          <TableCell>{formatCurrency(line.commissionAmount)}</TableCell>
                          <TableCell>{formatCurrency(line.otherEarnings)}</TableCell>
                          <TableCell className="font-medium">{formatCurrency(line.grossSalary)}</TableCell>
                          <TableCell className="text-red-600">
                            {line.totalDeductions > 0 ? `-${formatCurrency(line.totalDeductions)}` : "-"}
                          </TableCell>
                          <TableCell className="font-bold text-green-600">
                            {formatCurrency(line.netSalary)}
                          </TableCell>
                          {runDetails.status === "draft" && (
                            <TableCell>
                              <Button
                                variant="ghost"
                                size="icon"
                                onClick={() => handleEditLine(line)}
                                data-testid={`button-edit-line-${line.id}`}
                              >
                                <Pencil className="h-4 w-4" />
                              </Button>
                            </TableCell>
                          )}
                        </TableRow>
                      ))
                    )}
                  </TableBody>
                </Table>
              </div>
            )}
          </CardContent>
        </Card>

        <Dialog open={isEditLineDialogOpen} onOpenChange={setIsEditLineDialogOpen}>
          <DialogContent className="sm:max-w-lg" dir="rtl">
            <DialogHeader>
              <DialogTitle>
                تعديل بند راتب - {getEmployeeName(editingLine?.employee)}
              </DialogTitle>
            </DialogHeader>
            {editingLine && (
              <div className="space-y-4 py-4">
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label>الراتب الأساسي</Label>
                    <Input value={formatCurrency(editingLine.basicSalary)} disabled />
                  </div>
                  <div className="space-y-2">
                    <Label>البدلات الثابتة</Label>
                    <Input value={formatCurrency(editingLine.fixedAllowances)} disabled />
                  </div>
                </div>
                <div className="grid grid-cols-3 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="overtimeAmount">بدل الإضافي</Label>
                    <Input
                      id="overtimeAmount"
                      type="number"
                      value={editingLine.overtimeAmount}
                      onChange={(e) => setEditingLine({ ...editingLine, overtimeAmount: parseFloat(e.target.value) || 0 })}
                      min={0}
                      data-testid="input-overtime"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="commissionAmount">العمولات</Label>
                    <Input
                      id="commissionAmount"
                      type="number"
                      value={editingLine.commissionAmount}
                      onChange={(e) => setEditingLine({ ...editingLine, commissionAmount: parseFloat(e.target.value) || 0 })}
                      min={0}
                      data-testid="input-commission"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="otherEarnings">إضافات أخرى</Label>
                    <Input
                      id="otherEarnings"
                      type="number"
                      value={editingLine.otherEarnings}
                      onChange={(e) => setEditingLine({ ...editingLine, otherEarnings: parseFloat(e.target.value) || 0 })}
                      min={0}
                      data-testid="input-other-earnings"
                    />
                  </div>
                </div>
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="deductionLate">خصم التأخير</Label>
                    <Input
                      id="deductionLate"
                      type="number"
                      value={editingLine.deductionLate}
                      onChange={(e) => setEditingLine({ ...editingLine, deductionLate: parseFloat(e.target.value) || 0 })}
                      min={0}
                      data-testid="input-deduction-late"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="deductionOther">خصومات أخرى</Label>
                    <Input
                      id="deductionOther"
                      type="number"
                      value={editingLine.deductionOther}
                      onChange={(e) => setEditingLine({ ...editingLine, deductionOther: parseFloat(e.target.value) || 0 })}
                      min={0}
                      data-testid="input-deduction-other"
                    />
                  </div>
                </div>
              </div>
            )}
            <DialogFooter className="gap-2 sm:gap-0">
              <Button
                variant="outline"
                onClick={() => setIsEditLineDialogOpen(false)}
              >
                إلغاء
              </Button>
              <Button
                onClick={handleSaveLineEdit}
                disabled={updateLineMutation.isPending}
              >
                {updateLineMutation.isPending ? "جاري الحفظ..." : "حفظ"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">إدارة الرواتب</h1>
          <p className="text-muted-foreground">مسيّرات الرواتب الشهرية وصرفها</p>
        </div>
        <Button onClick={() => setIsDialogOpen(true)} data-testid="button-add-payroll">
          <Plus className="h-4 w-4 ml-2" />
          إنشاء مسيّر رواتب جديد
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <FileText className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي المسيّرات</p>
                <p className="text-xl font-bold" data-testid="text-total-runs">
                  {payrollRuns?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-200">
                <Clock className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">مسودات</p>
                <p className="text-xl font-bold" data-testid="text-draft-runs">
                  {draftRuns}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200">
                <CheckCircle2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">معتمدة</p>
                <p className="text-xl font-bold" data-testid="text-approved-runs">
                  {approvedRuns}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <CreditCard className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">مدفوعة</p>
                <p className="text-xl font-bold" data-testid="text-paid-runs">
                  {paidRuns}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Calculator className="h-5 w-5" />
            مسيّرات الرواتب
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">الفترة</TableHead>
                  <TableHead className="text-right">عدد الموظفين</TableHead>
                  <TableHead className="text-right">إجمالي الرواتب</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-right">تاريخ الإنشاء</TableHead>
                  <TableHead className="text-left">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {payrollRuns?.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={6} className="text-center py-8 text-muted-foreground">
                      لا توجد مسيّرات رواتب مسجلة
                    </TableCell>
                  </TableRow>
                ) : (
                  payrollRuns?.map((run) => (
                    <TableRow key={run.id} data-testid={`row-payroll-${run.id}`}>
                      <TableCell className="font-medium">
                        {monthNames[run.periodMonth - 1]} {run.periodYear}
                      </TableCell>
                      <TableCell>{run.employeeCount || 0}</TableCell>
                      <TableCell>{formatCurrency(run.totalNetSalary || 0)}</TableCell>
                      <TableCell>{getStatusBadge(run.status)}</TableCell>
                      <TableCell>
                        {format(new Date(run.createdAt), "dd/MM/yyyy")}
                      </TableCell>
                      <TableCell>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => setSelectedRun(run)}
                          data-testid={`button-view-payroll-${run.id}`}
                        >
                          <Eye className="h-4 w-4" />
                        </Button>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-lg" dir="rtl">
          <DialogHeader>
            <DialogTitle>إنشاء مسيّر رواتب جديد</DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="periodMonth">الشهر</Label>
                <Input
                  id="periodMonth"
                  type="number"
                  value={formData.periodMonth}
                  onChange={(e) => setFormData({ ...formData, periodMonth: parseInt(e.target.value) || 1 })}
                  min={1}
                  max={12}
                  data-testid="input-payroll-month"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="periodYear">السنة</Label>
                <Input
                  id="periodYear"
                  type="number"
                  value={formData.periodYear}
                  onChange={(e) => setFormData({ ...formData, periodYear: parseInt(e.target.value) || 2024 })}
                  min={2020}
                  max={2100}
                  data-testid="input-payroll-year"
                />
              </div>
            </div>
            
            {/* Employee Selection */}
            <div className="space-y-3">
              <Label className="flex items-center gap-2">
                <UserCheck className="h-4 w-4" />
                اختيار الموظفين
              </Label>
              <RadioGroup
                value={formData.selectionMode}
                onValueChange={(value: 'all' | 'selected') => setFormData({ ...formData, selectionMode: value })}
                className="flex gap-4"
              >
                <div className="flex items-center gap-2">
                  <RadioGroupItem value="all" id="all-employees" data-testid="radio-all-employees" />
                  <Label htmlFor="all-employees" className="cursor-pointer">جميع الموظفين النشطين ({activeEmployees.length})</Label>
                </div>
                <div className="flex items-center gap-2">
                  <RadioGroupItem value="selected" id="selected-employees" data-testid="radio-selected-employees" />
                  <Label htmlFor="selected-employees" className="cursor-pointer">اختيار محدد</Label>
                </div>
              </RadioGroup>
              
              {formData.selectionMode === 'selected' && (
                <div className="border rounded-md p-2">
                  <div className="flex items-center justify-between mb-2">
                    <span className="text-sm text-muted-foreground">
                      تم اختيار {formData.employeeIds.length} موظف
                    </span>
                    <div className="flex gap-2">
                      <Button type="button" variant="ghost" size="sm" onClick={selectAllEmployees}>
                        تحديد الكل
                      </Button>
                      <Button type="button" variant="ghost" size="sm" onClick={deselectAllEmployees}>
                        إلغاء الكل
                      </Button>
                    </div>
                  </div>
                  <ScrollArea className="h-40">
                    <div className="space-y-2">
                      {activeEmployees.map((employee) => (
                        <div 
                          key={employee.id} 
                          className="flex items-center gap-2 p-2 hover:bg-muted rounded-md"
                        >
                          <Checkbox
                            id={`employee-${employee.id}`}
                            checked={formData.employeeIds.includes(employee.id)}
                            onCheckedChange={() => toggleEmployeeSelection(employee.id)}
                            data-testid={`checkbox-employee-${employee.id}`}
                          />
                          <Label 
                            htmlFor={`employee-${employee.id}`} 
                            className="flex-1 cursor-pointer"
                          >
                            {employee.fullNameAr || employee.nameAr}
                          </Label>
                          <Badge variant="outline" className="text-xs">
                            {employee.code}
                          </Badge>
                        </div>
                      ))}
                    </div>
                  </ScrollArea>
                </div>
              )}
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="notes">ملاحظات</Label>
              <Textarea
                id="notes"
                value={formData.notes}
                onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
                placeholder="ملاحظات إضافية (اختياري)"
                rows={2}
                data-testid="input-payroll-notes"
              />
            </div>
          </div>
          <DialogFooter className="gap-2 sm:gap-0">
            <Button
              variant="outline"
              onClick={resetForm}
              data-testid="button-cancel-payroll"
            >
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || (formData.selectionMode === 'selected' && formData.employeeIds.length === 0)}
              data-testid="button-create-payroll"
            >
              {createMutation.isPending ? "جاري الإنشاء..." : "إنشاء"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
