import { useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useLocation, useParams } from "wouter";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Skeleton } from "@/components/ui/skeleton";
import { Switch } from "@/components/ui/switch";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
  FormDescription,
} from "@/components/ui/form";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { ArrowRight, Save, Loader2, ArrowLeftRight } from "lucide-react";
import type { ProductWithUnits, Unit } from "@shared/schema";

const usageTypes = [
  { value: "sale", label: "للبيع فقط", description: "منتج مخصص للبيع للعملاء" },
  { value: "consume", label: "للاستهلاك فقط", description: "منتج للاستخدام الداخلي (مواد خام، مستلزمات)" },
  { value: "sale_and_consume", label: "للبيع والاستهلاك", description: "منتج يمكن بيعه واستهلاكه داخلياً" },
] as const;

const productFormSchema = z.object({
  nameAr: z.string().min(1, "اسم المنتج مطلوب"),
  sku: z.string().optional(),
  purchaseUnitId: z.number({ required_error: "وحدة الشراء مطلوبة" }),
  sellUnitId: z.number({ required_error: "وحدة البيع مطلوبة" }),
  conversionFactor: z.number().min(1, "معامل التحويل يجب أن يكون 1 أو أكثر"),
  defaultPurchasePrice: z.number().optional(),
  defaultSellPrice: z.number().optional(),
  hasExpiryDate: z.boolean().default(false),
  usageType: z.enum(["sale", "consume", "sale_and_consume"]).default("sale"),
  isActive: z.boolean().default(true),
}).refine((data) => {
  if ((data.usageType === "sale" || data.usageType === "sale_and_consume") && !data.defaultSellPrice) {
    return false;
  }
  return true;
}, {
  message: "سعر البيع مطلوب لهذا النوع من المنتجات",
  path: ["defaultSellPrice"],
});

type ProductFormValues = z.infer<typeof productFormSchema>;

export default function ProductFormPage() {
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const params = useParams<{ id?: string }>();
  const isEditing = !!params.id;
  const productId = params.id ? parseInt(params.id) : undefined;

  const { data: product, isLoading: isLoadingProduct } = useQuery<ProductWithUnits>({
    queryKey: ["/api/products", productId],
    enabled: isEditing,
  });

  const { data: units } = useQuery<Unit[]>({
    queryKey: ["/api/units"],
  });

  const form = useForm<ProductFormValues>({
    resolver: zodResolver(productFormSchema),
    defaultValues: {
      nameAr: "",
      sku: "",
      purchaseUnitId: undefined,
      sellUnitId: undefined,
      conversionFactor: 1,
      defaultPurchasePrice: undefined,
      defaultSellPrice: undefined,
      hasExpiryDate: false,
      usageType: "sale",
      isActive: true,
    },
  });

  useEffect(() => {
    if (product) {
      form.reset({
        nameAr: product.nameAr,
        sku: product.sku || "",
        purchaseUnitId: product.purchaseUnitId || undefined,
        sellUnitId: product.sellUnitId || undefined,
        conversionFactor: product.conversionFactor,
        defaultPurchasePrice: product.defaultPurchasePrice || undefined,
        defaultSellPrice: product.defaultSellPrice || undefined,
        hasExpiryDate: product.hasExpiryDate,
        usageType: (product as any).usageType || "sale",
        isActive: product.isActive,
      });
    }
  }, [product, form]);

  const createMutation = useMutation({
    mutationFn: async (data: ProductFormValues) => {
      const response = await apiRequest("POST", "/api/products", data);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/products"] });
      toast({
        title: "تم الإنشاء بنجاح",
        description: "تم إضافة المنتج بنجاح",
      });
      navigate("/products");
    },
    onError: () => {
      toast({
        title: "خطأ",
        description: "حدث خطأ أثناء إنشاء المنتج",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async (data: ProductFormValues) => {
      const response = await apiRequest("PUT", `/api/products/${productId}`, data);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/products"] });
      queryClient.invalidateQueries({ queryKey: ["/api/products", productId] });
      toast({
        title: "تم التحديث بنجاح",
        description: "تم تحديث بيانات المنتج",
      });
      navigate("/products");
    },
    onError: () => {
      toast({
        title: "خطأ",
        description: "حدث خطأ أثناء تحديث المنتج",
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: ProductFormValues) => {
    if (isEditing) {
      updateMutation.mutate(data);
    } else {
      createMutation.mutate(data);
    }
  };

  const isPending = createMutation.isPending || updateMutation.isPending;

  const purchaseUnit = units?.find((u) => u.id === form.watch("purchaseUnitId"));
  const sellUnit = units?.find((u) => u.id === form.watch("sellUnitId"));
  const conversionFactor = form.watch("conversionFactor");

  if (isEditing && isLoadingProduct) {
    return (
      <div className="space-y-6">
        <Skeleton className="h-8 w-48" />
        <Card>
          <CardContent className="p-6 space-y-4">
            {[1, 2, 3, 4].map((i) => (
              <div key={i} className="grid gap-4 md:grid-cols-2">
                <Skeleton className="h-10" />
                <Skeleton className="h-10" />
              </div>
            ))}
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center gap-4">
        <Button variant="ghost" size="icon" onClick={() => navigate("/products")} data-testid="button-back">
          <ArrowRight className="h-4 w-4" />
        </Button>
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">
            {isEditing ? "تعديل المنتج" : "إضافة منتج جديد"}
          </h1>
          <p className="text-muted-foreground">
            {isEditing ? "تعديل بيانات المنتج" : "أدخل بيانات المنتج الجديد"}
          </p>
        </div>
      </div>

      <Form {...form}>
        <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>البيانات الأساسية</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid gap-4 md:grid-cols-2">
                <FormField
                  control={form.control}
                  name="nameAr"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>اسم المنتج *</FormLabel>
                      <FormControl>
                        <Input placeholder="أدخل اسم المنتج" {...field} data-testid="input-name" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="sku"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>الكود الداخلي (SKU)</FormLabel>
                      <FormControl>
                        <Input placeholder="PROD-001" {...field} data-testid="input-sku" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="usageType"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>نوع الاستخدام *</FormLabel>
                    <Select onValueChange={field.onChange} value={field.value}>
                      <FormControl>
                        <SelectTrigger data-testid="select-usage-type">
                          <SelectValue placeholder="اختر نوع الاستخدام" />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        {usageTypes.map((type) => (
                          <SelectItem key={type.value} value={type.value}>
                            <div className="flex flex-col">
                              <span>{type.label}</span>
                              <span className="text-xs text-muted-foreground">{type.description}</span>
                            </div>
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <FormDescription>
                      {form.watch("usageType") === "consume" && (
                        <span className="text-amber-600">المنتجات المخصصة للاستهلاك لا يمكن بيعها للعملاء ويمكن استخدامها في العهد</span>
                      )}
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="flex items-center gap-4">
                <FormField
                  control={form.control}
                  name="hasExpiryDate"
                  render={({ field }) => (
                    <FormItem className="flex items-center justify-between rounded-lg border p-4 flex-1">
                      <div>
                        <FormLabel>له تاريخ انتهاء</FormLabel>
                        <FormDescription>
                          هل يحتاج هذا المنتج لتتبع تاريخ الصلاحية؟
                        </FormDescription>
                      </div>
                      <FormControl>
                        <Switch
                          checked={field.value}
                          onCheckedChange={field.onChange}
                          data-testid="switch-expiry"
                        />
                      </FormControl>
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="isActive"
                  render={({ field }) => (
                    <FormItem className="flex items-center justify-between rounded-lg border p-4 flex-1">
                      <div>
                        <FormLabel>الحالة</FormLabel>
                        <FormDescription>
                          هل المنتج نشط ويمكن استخدامه؟
                        </FormDescription>
                      </div>
                      <FormControl>
                        <Switch
                          checked={field.value}
                          onCheckedChange={field.onChange}
                          data-testid="switch-active"
                        />
                      </FormControl>
                    </FormItem>
                  )}
                />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>وحدات القياس والتحويل</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid gap-4 md:grid-cols-3">
                <FormField
                  control={form.control}
                  name="purchaseUnitId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>وحدة الشراء *</FormLabel>
                      <Select
                        onValueChange={(value) => field.onChange(parseInt(value))}
                        value={field.value?.toString()}
                      >
                        <FormControl>
                          <SelectTrigger data-testid="select-purchase-unit">
                            <SelectValue placeholder="اختر وحدة الشراء" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {units?.map((unit) => (
                            <SelectItem key={unit.id} value={unit.id.toString()}>
                              {unit.nameAr}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="conversionFactor"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>معامل التحويل *</FormLabel>
                      <FormControl>
                        <Input
                          type="number"
                          min={1}
                          placeholder="12"
                          {...field}
                          onChange={(e) => field.onChange(parseFloat(e.target.value) || 1)}
                          data-testid="input-conversion"
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="sellUnitId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>وحدة البيع *</FormLabel>
                      <Select
                        onValueChange={(value) => field.onChange(parseInt(value))}
                        value={field.value?.toString()}
                      >
                        <FormControl>
                          <SelectTrigger data-testid="select-sell-unit">
                            <SelectValue placeholder="اختر وحدة البيع" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {units?.map((unit) => (
                            <SelectItem key={unit.id} value={unit.id.toString()}>
                              {unit.nameAr}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              {purchaseUnit && sellUnit && conversionFactor && (
                <div className="rounded-lg bg-muted p-4">
                  <div className="flex items-center justify-center gap-3 text-lg">
                    <span className="font-bold">1</span>
                    <span>{purchaseUnit.nameAr}</span>
                    <ArrowLeftRight className="h-5 w-5 text-muted-foreground" />
                    <span className="font-bold text-primary">{conversionFactor}</span>
                    <span>{sellUnit.nameAr}</span>
                  </div>
                </div>
              )}
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>الأسعار الافتراضية</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid gap-4 md:grid-cols-2">
                <FormField
                  control={form.control}
                  name="defaultPurchasePrice"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>سعر الشراء الافتراضي (لوحدة الشراء)</FormLabel>
                      <FormControl>
                        <div className="relative">
                          <Input
                            type="number"
                            min={0}
                            step={0.01}
                            placeholder="0.00"
                            {...field}
                            value={field.value ?? ""}
                            onChange={(e) => field.onChange(e.target.value ? parseFloat(e.target.value) : undefined)}
                            data-testid="input-purchase-price"
                          />
                          <span className="absolute left-3 top-1/2 -translate-y-1/2 text-sm text-muted-foreground">
                            ر.س
                          </span>
                        </div>
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                {form.watch("usageType") !== "consume" && (
                  <FormField
                    control={form.control}
                    name="defaultSellPrice"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>
                          سعر البيع الافتراضي (لوحدة البيع)
                          {(form.watch("usageType") === "sale" || form.watch("usageType") === "sale_and_consume") && (
                            <span className="text-destructive mr-1">*</span>
                          )}
                        </FormLabel>
                        <FormControl>
                          <div className="relative">
                            <Input
                              type="number"
                              min={0}
                              step={0.01}
                              placeholder="0.00"
                              {...field}
                              value={field.value ?? ""}
                              onChange={(e) => field.onChange(e.target.value ? parseFloat(e.target.value) : undefined)}
                              data-testid="input-sell-price"
                            />
                            <span className="absolute left-3 top-1/2 -translate-y-1/2 text-sm text-muted-foreground">
                              ر.س
                            </span>
                          </div>
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                )}
                {form.watch("usageType") === "consume" && (
                  <div className="flex items-center justify-center rounded-lg border border-dashed p-4 text-muted-foreground">
                    <span>لا يوجد سعر بيع للمنتجات المخصصة للاستهلاك</span>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>

          <div className="flex justify-end gap-4">
            <Button type="button" variant="outline" onClick={() => navigate("/products")} data-testid="button-cancel">
              إلغاء
            </Button>
            <Button type="submit" disabled={isPending} data-testid="button-submit">
              {isPending ? (
                <Loader2 className="h-4 w-4 animate-spin ml-2" />
              ) : (
                <Save className="h-4 w-4 ml-2" />
              )}
              {isEditing ? "حفظ التغييرات" : "إضافة المنتج"}
            </Button>
          </div>
        </form>
      </Form>
    </div>
  );
}
