import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Link, useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Separator } from "@/components/ui/separator";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Plus, Search, FileText, Eye, Check, X, Trash2, Receipt, Package } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { format } from "date-fns";
import type { 
  PurchaseDebitNoteWithDetails, 
  PurchaseInvoiceWithDetails, 
  Supplier,
  Product
} from "@shared/schema";

function formatCurrency(amount: number): string {
  return new Intl.NumberFormat('en-SA', {
    style: 'currency',
    currency: 'SAR',
    minimumFractionDigits: 2,
    maximumFractionDigits: 2,
  }).format(amount);
}

function DebitNoteSkeleton() {
  return (
    <TableRow>
      <TableCell><Skeleton className="h-4 w-20" /></TableCell>
      <TableCell><Skeleton className="h-4 w-32" /></TableCell>
      <TableCell><Skeleton className="h-4 w-24" /></TableCell>
      <TableCell><Skeleton className="h-4 w-24" /></TableCell>
      <TableCell><Skeleton className="h-4 w-20" /></TableCell>
      <TableCell><Skeleton className="h-6 w-20" /></TableCell>
      <TableCell><Skeleton className="h-8 w-20" /></TableCell>
    </TableRow>
  );
}

function EmptyState({ onCreateClick }: { onCreateClick: () => void }) {
  return (
    <div className="flex flex-col items-center justify-center py-16 text-center">
      <div className="flex h-16 w-16 items-center justify-center rounded-full bg-muted mb-4">
        <Receipt className="h-8 w-8 text-muted-foreground" />
      </div>
      <h3 className="text-lg font-semibold mb-2">لا توجد إشعارات خصم</h3>
      <p className="text-muted-foreground mb-4">ابدأ بإنشاء أول إشعار خصم للمشتريات</p>
      <Button onClick={onCreateClick} data-testid="empty-add-debit-note">
        <Plus className="h-4 w-4 ml-2" />
        إنشاء إشعار خصم
      </Button>
    </div>
  );
}

interface DebitNoteLine {
  purchaseInvoiceLineId: number;
  productId: number;
  productName: string;
  quantityReturned: number;
  unitPrice: number;
  vatRate: number;
  vatAmount: number;
  lineSubtotal: number;
  lineTotal: number;
  reason?: string;
}

export default function PurchaseDebitNotesPage() {
  const { toast } = useToast();
  const [searchQuery, setSearchQuery] = useState("");
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [isCreateDialogOpen, setIsCreateDialogOpen] = useState(false);
  const [isDetailDialogOpen, setIsDetailDialogOpen] = useState(false);
  const [selectedNote, setSelectedNote] = useState<PurchaseDebitNoteWithDetails | null>(null);
  
  const [selectedInvoiceId, setSelectedInvoiceId] = useState<string>("");
  const [debitNoteReason, setDebitNoteReason] = useState("");
  const [debitNoteNotes, setDebitNoteNotes] = useState("");
  const [returnLines, setReturnLines] = useState<DebitNoteLine[]>([]);
  
  const { data: debitNotes, isLoading } = useQuery<PurchaseDebitNoteWithDetails[]>({
    queryKey: ["/api/purchase-debit-notes"],
  });
  
  const { data: invoices } = useQuery<PurchaseInvoiceWithDetails[]>({
    queryKey: ["/api/purchase-invoices"],
  });

  const filteredNotes = debitNotes?.filter((note) => {
    const matchesSearch =
      note.supplier?.nameAr?.includes(searchQuery) ||
      note.debitNoteNumber?.includes(searchQuery);
    const matchesStatus =
      statusFilter === "all" || note.status === statusFilter;
    return matchesSearch && matchesStatus;
  });

  const createMutation = useMutation({
    mutationFn: (data: any) => apiRequest("POST", "/api/purchase-debit-notes", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/purchase-debit-notes"] });
      toast({ title: "تم إنشاء إشعار الخصم بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء إشعار الخصم",
        variant: "destructive",
      });
    },
  });

  const confirmMutation = useMutation({
    mutationFn: (id: number) => apiRequest("POST", `/api/purchase-debit-notes/${id}/confirm`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/purchase-debit-notes"] });
      toast({ title: "تم تأكيد إشعار الخصم بنجاح" });
      setIsDetailDialogOpen(false);
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تأكيد إشعار الخصم",
        variant: "destructive",
      });
    },
  });

  const cancelMutation = useMutation({
    mutationFn: (id: number) => apiRequest("POST", `/api/purchase-debit-notes/${id}/cancel`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/purchase-debit-notes"] });
      toast({ title: "تم إلغاء إشعار الخصم" });
      setIsDetailDialogOpen(false);
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إلغاء إشعار الخصم",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setSelectedInvoiceId("");
    setDebitNoteReason("");
    setDebitNoteNotes("");
    setReturnLines([]);
    setIsCreateDialogOpen(false);
  };

  const handleInvoiceSelect = (invoiceId: string) => {
    setSelectedInvoiceId(invoiceId);
    setReturnLines([]);
    
    const invoice = invoices?.find(i => i.id === parseInt(invoiceId));
    if (invoice && invoice.lines) {
      const lines: DebitNoteLine[] = invoice.lines.map(line => ({
        purchaseInvoiceLineId: line.id,
        productId: line.productId,
        productName: line.product?.nameAr || `منتج ${line.productId}`,
        quantityReturned: 0,
        unitPrice: line.purchasePricePerSellUnit,
        vatRate: line.vatRate,
        vatAmount: 0,
        lineSubtotal: 0,
        lineTotal: 0,
      }));
      setReturnLines(lines);
    }
  };

  const updateLineQuantity = (index: number, quantity: number) => {
    setReturnLines(prev => {
      const newLines = [...prev];
      const line = newLines[index];
      const originalLine = invoices?.find(i => i.id === parseInt(selectedInvoiceId))?.lines
        .find(l => l.id === line.purchaseInvoiceLineId);
      
      const maxQty = originalLine?.quantitySellUnit || 0;
      const validQty = Math.min(Math.max(0, quantity), maxQty);
      
      line.quantityReturned = validQty;
      line.lineSubtotal = validQty * line.unitPrice;
      line.vatAmount = (line.lineSubtotal * line.vatRate) / 100;
      line.lineTotal = line.lineSubtotal + line.vatAmount;
      
      return newLines;
    });
  };

  const handleSubmit = () => {
    const validLines = returnLines.filter(l => l.quantityReturned > 0);
    
    if (validLines.length === 0) {
      toast({
        title: "خطأ",
        description: "يجب إضافة منتج واحد على الأقل",
        variant: "destructive",
      });
      return;
    }
    
    const selectedInvoice = invoices?.find(i => i.id === parseInt(selectedInvoiceId));
    
    createMutation.mutate({
      purchaseInvoiceId: parseInt(selectedInvoiceId),
      supplierId: selectedInvoice?.supplierId,
      debitNoteDate: format(new Date(), 'yyyy-MM-dd'),
      returnType: validLines.length === returnLines.length ? 'full' : 'partial',
      reason: debitNoteReason,
      notes: debitNoteNotes,
      lines: validLines.map(l => ({
        purchaseInvoiceLineId: l.purchaseInvoiceLineId,
        productId: l.productId,
        quantityReturned: l.quantityReturned,
        unitPrice: l.unitPrice,
        vatRate: l.vatRate,
        vatAmount: l.vatAmount,
        lineSubtotal: l.lineSubtotal,
        lineTotal: l.lineTotal,
        reason: l.reason,
      })),
    });
  };

  const getStatusBadge = (status: string) => {
    switch (status) {
      case "confirmed":
        return <Badge variant="default" data-testid="badge-confirmed">مؤكد</Badge>;
      case "cancelled":
        return <Badge variant="destructive" data-testid="badge-cancelled">ملغي</Badge>;
      case "draft":
      default:
        return <Badge variant="secondary" data-testid="badge-draft">مسودة</Badge>;
    }
  };

  const calculateTotals = () => {
    const subtotal = returnLines.reduce((sum, l) => sum + l.lineSubtotal, 0);
    const vat = returnLines.reduce((sum, l) => sum + l.vatAmount, 0);
    return { subtotal, vat, total: subtotal + vat };
  };

  const totals = calculateTotals();

  return (
    <div className="space-y-6">
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">إشعارات الخصم للمشتريات</h1>
          <p className="text-muted-foreground">إدارة مرتجعات المشتريات للموردين</p>
        </div>
        <Button onClick={() => setIsCreateDialogOpen(true)} data-testid="button-new-debit-note">
          <Plus className="h-4 w-4 ml-2" />
          إشعار خصم جديد
        </Button>
      </div>

      <Card>
        <CardHeader className="pb-4">
          <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
            <CardTitle className="text-lg">قائمة إشعارات الخصم</CardTitle>
            <div className="flex flex-col sm:flex-row gap-2 w-full sm:w-auto">
              <div className="relative flex-1 sm:w-72">
                <Search className="absolute right-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                <Input
                  placeholder="بحث بالمورد أو رقم الإشعار..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pr-10"
                  data-testid="input-search"
                />
              </div>
              <Select value={statusFilter} onValueChange={setStatusFilter}>
                <SelectTrigger className="w-full sm:w-40" data-testid="select-status-filter">
                  <SelectValue placeholder="الحالة" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">الكل</SelectItem>
                  <SelectItem value="draft">مسودة</SelectItem>
                  <SelectItem value="confirmed">مؤكد</SelectItem>
                  <SelectItem value="cancelled">ملغي</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>رقم الإشعار</TableHead>
                  <TableHead>المورد</TableHead>
                  <TableHead>التاريخ</TableHead>
                  <TableHead>المبلغ</TableHead>
                  <TableHead>النوع</TableHead>
                  <TableHead>الحالة</TableHead>
                  <TableHead>الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {Array(5).fill(0).map((_, i) => <DebitNoteSkeleton key={i} />)}
              </TableBody>
            </Table>
          ) : !filteredNotes?.length ? (
            <EmptyState onCreateClick={() => setIsCreateDialogOpen(true)} />
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>رقم الإشعار</TableHead>
                  <TableHead>المورد</TableHead>
                  <TableHead>التاريخ</TableHead>
                  <TableHead>المبلغ</TableHead>
                  <TableHead>النوع</TableHead>
                  <TableHead>الحالة</TableHead>
                  <TableHead>الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {filteredNotes.map((note) => (
                  <TableRow key={note.id} data-testid={`row-debit-note-${note.id}`}>
                    <TableCell className="font-medium" data-testid={`text-debit-number-${note.id}`}>
                      {note.debitNoteNumber}
                    </TableCell>
                    <TableCell data-testid={`text-supplier-${note.id}`}>
                      {note.supplier?.nameAr || "-"}
                    </TableCell>
                    <TableCell>
                      {format(new Date(note.debitNoteDate), "dd/MM/yyyy")}
                    </TableCell>
                    <TableCell data-testid={`text-amount-${note.id}`}>
                      {formatCurrency(note.totalAmount)}
                    </TableCell>
                    <TableCell>
                      {note.returnType === "full" ? "كامل" : "جزئي"}
                    </TableCell>
                    <TableCell>{getStatusBadge(note.status)}</TableCell>
                    <TableCell>
                      <Button
                        variant="ghost"
                        size="icon"
                        onClick={() => {
                          setSelectedNote(note);
                          setIsDetailDialogOpen(true);
                        }}
                        data-testid={`button-view-${note.id}`}
                      >
                        <Eye className="h-4 w-4" />
                      </Button>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isCreateDialogOpen} onOpenChange={setIsCreateDialogOpen}>
        <DialogContent className="sm:max-w-[700px]" dir="rtl">
          <DialogHeader>
            <DialogTitle>إنشاء إشعار خصم جديد</DialogTitle>
            <DialogDescription>
              اختر فاتورة الشراء وحدد المنتجات المرتجعة
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="invoice">فاتورة الشراء</Label>
              <Select value={selectedInvoiceId} onValueChange={handleInvoiceSelect}>
                <SelectTrigger data-testid="select-invoice">
                  <SelectValue placeholder="اختر فاتورة الشراء" />
                </SelectTrigger>
                <SelectContent>
                  {invoices?.map((invoice) => (
                    <SelectItem key={invoice.id} value={invoice.id.toString()}>
                      #{invoice.id} - {invoice.supplier?.nameAr} - {formatCurrency(invoice.totalAmount)}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            
            {selectedInvoiceId && returnLines.length > 0 && (
              <>
                <div className="space-y-2">
                  <Label>المنتجات</Label>
                  <ScrollArea className="h-48 border rounded-md">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead>المنتج</TableHead>
                          <TableHead>الكمية المتاحة</TableHead>
                          <TableHead>الكمية المرتجعة</TableHead>
                          <TableHead>السعر</TableHead>
                          <TableHead>الإجمالي</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {returnLines.map((line, index) => {
                          const originalLine = invoices?.find(i => i.id === parseInt(selectedInvoiceId))?.lines
                            .find(l => l.id === line.purchaseInvoiceLineId);
                          return (
                            <TableRow key={index}>
                              <TableCell>{line.productName}</TableCell>
                              <TableCell>{originalLine?.quantitySellUnit || 0}</TableCell>
                              <TableCell>
                                <Input
                                  type="number"
                                  value={line.quantityReturned}
                                  onChange={(e) => updateLineQuantity(index, parseFloat(e.target.value) || 0)}
                                  min={0}
                                  max={originalLine?.quantitySellUnit || 0}
                                  className="w-24"
                                  data-testid={`input-qty-${index}`}
                                />
                              </TableCell>
                              <TableCell>{formatCurrency(line.unitPrice)}</TableCell>
                              <TableCell>{formatCurrency(line.lineTotal)}</TableCell>
                            </TableRow>
                          );
                        })}
                      </TableBody>
                    </Table>
                  </ScrollArea>
                </div>
                
                <Separator />
                
                <div className="grid grid-cols-3 gap-4 text-sm">
                  <div className="text-center">
                    <p className="text-muted-foreground">المجموع الفرعي</p>
                    <p className="font-bold">{formatCurrency(totals.subtotal)}</p>
                  </div>
                  <div className="text-center">
                    <p className="text-muted-foreground">الضريبة</p>
                    <p className="font-bold">{formatCurrency(totals.vat)}</p>
                  </div>
                  <div className="text-center">
                    <p className="text-muted-foreground">الإجمالي</p>
                    <p className="font-bold text-lg">{formatCurrency(totals.total)}</p>
                  </div>
                </div>
              </>
            )}
            
            <div className="space-y-2">
              <Label htmlFor="reason">سبب المرتجع</Label>
              <Input
                id="reason"
                value={debitNoteReason}
                onChange={(e) => setDebitNoteReason(e.target.value)}
                placeholder="مثال: منتجات تالفة، أخطاء في الطلب"
                data-testid="input-reason"
              />
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="notes">ملاحظات إضافية</Label>
              <Textarea
                id="notes"
                value={debitNoteNotes}
                onChange={(e) => setDebitNoteNotes(e.target.value)}
                placeholder="ملاحظات إضافية (اختياري)"
                rows={2}
                data-testid="input-notes"
              />
            </div>
          </div>
          
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={resetForm} data-testid="button-cancel">
              إلغاء
            </Button>
            <Button 
              onClick={handleSubmit}
              disabled={createMutation.isPending || !selectedInvoiceId || !debitNoteReason || totals.total === 0}
              data-testid="button-create"
            >
              {createMutation.isPending ? "جاري الإنشاء..." : "إنشاء"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={isDetailDialogOpen} onOpenChange={setIsDetailDialogOpen}>
        <DialogContent className="sm:max-w-[600px]" dir="rtl">
          <DialogHeader>
            <DialogTitle>تفاصيل إشعار الخصم</DialogTitle>
          </DialogHeader>
          
          {selectedNote && (
            <div className="space-y-4 py-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <p className="text-sm text-muted-foreground">رقم الإشعار</p>
                  <p className="font-bold" data-testid="detail-number">{selectedNote.debitNoteNumber}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">التاريخ</p>
                  <p className="font-bold">{format(new Date(selectedNote.debitNoteDate), "dd/MM/yyyy")}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">المورد</p>
                  <p className="font-bold">{selectedNote.supplier?.nameAr}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">الحالة</p>
                  {getStatusBadge(selectedNote.status)}
                </div>
              </div>
              
              <Separator />
              
              <div>
                <p className="text-sm text-muted-foreground mb-2">السبب</p>
                <p>{selectedNote.reason}</p>
              </div>
              
              <Separator />
              
              <div>
                <p className="text-sm text-muted-foreground mb-2">المنتجات المرتجعة</p>
                <ScrollArea className="h-32 border rounded-md">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>المنتج</TableHead>
                        <TableHead>الكمية</TableHead>
                        <TableHead>الإجمالي</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {selectedNote.lines.map((line, index) => (
                        <TableRow key={index}>
                          <TableCell>{line.product?.nameAr || `منتج ${line.productId}`}</TableCell>
                          <TableCell>{line.quantityReturned}</TableCell>
                          <TableCell>{formatCurrency(line.lineTotal)}</TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                </ScrollArea>
              </div>
              
              <Separator />
              
              <div className="grid grid-cols-3 gap-4 text-sm">
                <div className="text-center">
                  <p className="text-muted-foreground">المجموع الفرعي</p>
                  <p className="font-bold">{formatCurrency(selectedNote.subtotalAmount)}</p>
                </div>
                <div className="text-center">
                  <p className="text-muted-foreground">الضريبة</p>
                  <p className="font-bold">{formatCurrency(selectedNote.vatAmount)}</p>
                </div>
                <div className="text-center">
                  <p className="text-muted-foreground">الإجمالي</p>
                  <p className="font-bold text-lg" data-testid="detail-total">{formatCurrency(selectedNote.totalAmount)}</p>
                </div>
              </div>
              
              {selectedNote.status === 'draft' && (
                <div className="flex gap-2 pt-4">
                  <Button
                    variant="default"
                    onClick={() => confirmMutation.mutate(selectedNote.id)}
                    disabled={confirmMutation.isPending}
                    className="flex-1"
                    data-testid="button-confirm"
                  >
                    <Check className="h-4 w-4 ml-2" />
                    تأكيد
                  </Button>
                  <Button
                    variant="destructive"
                    onClick={() => cancelMutation.mutate(selectedNote.id)}
                    disabled={cancelMutation.isPending}
                    className="flex-1"
                    data-testid="button-cancel-note"
                  >
                    <X className="h-4 w-4 ml-2" />
                    إلغاء
                  </Button>
                </div>
              )}
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
