import { useQuery, useMutation } from "@tanstack/react-query";
import { useLocation, useParams, Link } from "wouter";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Separator } from "@/components/ui/separator";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import {
  ArrowRight,
  FileText,
  CreditCard,
  Wallet,
  BookOpen,
  Plus,
  Loader2,
} from "lucide-react";
import type { PurchaseInvoiceWithDetails, PaymentMethod, JournalEntryWithLines } from "@shared/schema";

function PaymentDialog({
  open,
  onOpenChange,
  invoice,
  paymentMethods,
  onSuccess,
}: {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  invoice: PurchaseInvoiceWithDetails;
  paymentMethods?: PaymentMethod[];
  onSuccess: () => void;
}) {
  const { toast } = useToast();
  const [paymentMethodId, setPaymentMethodId] = useState<string>("");
  const [amount, setAmount] = useState<string>("");
  const [paymentDate, setPaymentDate] = useState(new Date().toISOString().split("T")[0]);
  const [notes, setNotes] = useState("");

  const paidAmount = invoice.payments.reduce((sum, p) => sum + p.amount, 0);
  const remainingAmount = invoice.totalAmount - paidAmount;

  const paymentMutation = useMutation({
    mutationFn: async () => {
      const response = await apiRequest("POST", `/api/purchase-invoices/${invoice.id}/payments`, {
        paymentMethodId: parseInt(paymentMethodId),
        amount: parseFloat(amount),
        paymentDate,
        notes: notes || undefined,
      });
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "تم تسجيل الدفعة",
        description: "تم تسجيل الدفعة بنجاح",
      });
      onSuccess();
      onOpenChange(false);
      setPaymentMethodId("");
      setAmount("");
      setNotes("");
    },
    onError: () => {
      toast({
        title: "خطأ",
        description: "حدث خطأ أثناء تسجيل الدفعة",
        variant: "destructive",
      });
    },
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!paymentMethodId || !amount || parseFloat(amount) <= 0) {
      toast({
        title: "خطأ",
        description: "يرجى ملء جميع الحقول المطلوبة",
        variant: "destructive",
      });
      return;
    }
    paymentMutation.mutate();
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent>
        <DialogHeader>
          <DialogTitle>تسجيل دفعة</DialogTitle>
        </DialogHeader>
        <form onSubmit={handleSubmit} className="space-y-4">
          <div className="rounded-lg bg-muted p-4 space-y-2">
            <div className="flex justify-between text-sm">
              <span>إجمالي الفاتورة:</span>
              <span className="font-mono">{invoice.totalAmount.toLocaleString("ar-SA")} ر.س</span>
            </div>
            <div className="flex justify-between text-sm">
              <span>المدفوع:</span>
              <span className="font-mono">{paidAmount.toLocaleString("ar-SA")} ر.س</span>
            </div>
            <Separator />
            <div className="flex justify-between font-bold">
              <span>المتبقي:</span>
              <span className="font-mono text-primary">{remainingAmount.toLocaleString("ar-SA")} ر.س</span>
            </div>
          </div>

          <div>
            <label className="text-sm font-medium">طريقة الدفع *</label>
            <Select value={paymentMethodId} onValueChange={setPaymentMethodId}>
              <SelectTrigger data-testid="select-payment-method">
                <SelectValue placeholder="اختر طريقة الدفع" />
              </SelectTrigger>
              <SelectContent>
                {paymentMethods?.filter(m => m.isActive).map((method) => (
                  <SelectItem key={method.id} value={method.id.toString()}>
                    {method.nameAr}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div>
            <label className="text-sm font-medium">المبلغ *</label>
            <Input
              type="number"
              min={0.01}
              max={remainingAmount}
              step={0.01}
              value={amount}
              onChange={(e) => setAmount(e.target.value)}
              placeholder="0.00"
              data-testid="input-payment-amount"
            />
            <Button
              type="button"
              variant="link"
              size="sm"
              onClick={() => setAmount(remainingAmount.toString())}
              className="text-xs p-0 h-auto"
            >
              دفع المبلغ المتبقي كاملاً
            </Button>
          </div>

          <div>
            <label className="text-sm font-medium">تاريخ الدفع *</label>
            <Input
              type="date"
              value={paymentDate}
              onChange={(e) => setPaymentDate(e.target.value)}
              data-testid="input-payment-date"
            />
          </div>

          <div>
            <label className="text-sm font-medium">ملاحظات</label>
            <Textarea
              value={notes}
              onChange={(e) => setNotes(e.target.value)}
              placeholder="ملاحظات اختيارية..."
            />
          </div>

          <div className="flex justify-end gap-2">
            <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>
              إلغاء
            </Button>
            <Button type="submit" disabled={paymentMutation.isPending} data-testid="button-confirm-payment">
              {paymentMutation.isPending ? (
                <Loader2 className="h-4 w-4 animate-spin ml-2" />
              ) : (
                <CreditCard className="h-4 w-4 ml-2" />
              )}
              تأكيد الدفع
            </Button>
          </div>
        </form>
      </DialogContent>
    </Dialog>
  );
}

export default function PurchaseInvoiceDetailPage() {
  const [, navigate] = useLocation();
  const params = useParams<{ id: string }>();
  const invoiceId = parseInt(params.id);
  const [paymentDialogOpen, setPaymentDialogOpen] = useState(false);

  const { data: invoice, isLoading, refetch } = useQuery<PurchaseInvoiceWithDetails>({
    queryKey: ["/api/purchase-invoices", invoiceId],
  });

  const { data: paymentMethods } = useQuery<PaymentMethod[]>({
    queryKey: ["/api/payment-methods"],
  });

  const { data: journalEntries } = useQuery<JournalEntryWithLines[]>({
    queryKey: ["/api/purchase-invoices", invoiceId, "journal-entries"],
  });

  if (isLoading) {
    return (
      <div className="space-y-6">
        <div className="flex items-center gap-4">
          <Skeleton className="h-10 w-10" />
          <div>
            <Skeleton className="h-8 w-48 mb-2" />
            <Skeleton className="h-4 w-32" />
          </div>
        </div>
        <div className="grid gap-6 md:grid-cols-3">
          <div className="md:col-span-2">
            <Card>
              <CardContent className="p-6">
                <Skeleton className="h-40" />
              </CardContent>
            </Card>
          </div>
          <Card>
            <CardContent className="p-6">
              <Skeleton className="h-40" />
            </CardContent>
          </Card>
        </div>
      </div>
    );
  }

  if (!invoice) {
    return (
      <div className="flex flex-col items-center justify-center py-16 text-center">
        <FileText className="h-16 w-16 text-muted-foreground mb-4" />
        <h2 className="text-xl font-semibold mb-2">الفاتورة غير موجودة</h2>
        <p className="text-muted-foreground mb-4">لم يتم العثور على الفاتورة المطلوبة</p>
        <Button asChild>
          <Link href="/purchase-invoices">العودة للقائمة</Link>
        </Button>
      </div>
    );
  }

  const paidAmount = invoice.payments.reduce((sum, p) => sum + p.amount, 0);
  const remainingAmount = invoice.totalAmount - paidAmount;

  const getStatusBadge = (status: string) => {
    switch (status) {
      case "paid":
        return <Badge variant="default">مدفوع</Badge>;
      case "partially_paid":
        return <Badge variant="secondary">مدفوع جزئياً</Badge>;
      case "unpaid":
        return <Badge variant="destructive">غير مدفوع</Badge>;
      default:
        return <Badge variant="outline">{status}</Badge>;
    }
  };

  const getVatTypeLabel = (vatType: string) => {
    switch (vatType) {
      case "inclusive":
        return "شامل";
      case "exclusive":
        return "غير شامل";
      case "exempt":
        return "معفي";
      default:
        return vatType;
    }
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4">
        <div className="flex items-center gap-4">
          <Button variant="ghost" size="icon" onClick={() => navigate("/purchase-invoices")} data-testid="button-back">
            <ArrowRight className="h-4 w-4" />
          </Button>
          <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-primary/10 text-primary">
            <FileText className="h-6 w-6" />
          </div>
          <div>
            <h1 className="text-2xl font-bold" data-testid="invoice-number">فاتورة #{invoice.id}</h1>
            <div className="flex items-center gap-2">
              {getStatusBadge(invoice.paymentStatus)}
              <span className="text-muted-foreground">
                {new Date(invoice.invoiceDate).toLocaleDateString("ar-SA")}
              </span>
            </div>
          </div>
        </div>
        {invoice.paymentStatus !== "paid" && (
          <Button onClick={() => setPaymentDialogOpen(true)} data-testid="button-add-payment">
            <Plus className="h-4 w-4 ml-2" />
            تسجيل دفعة
          </Button>
        )}
      </div>

      <div className="grid gap-6 lg:grid-cols-3">
        <div className="lg:col-span-2 space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>بيانات المورد</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid gap-4 sm:grid-cols-2">
                <div>
                  <p className="text-sm text-muted-foreground">اسم المورد</p>
                  <Link href={`/suppliers/${invoice.supplierId}`} className="font-medium text-primary hover:underline">
                    {invoice.supplier?.nameAr}
                  </Link>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">تاريخ الاستحقاق</p>
                  <p>
                    {invoice.dueDate
                      ? new Date(invoice.dueDate).toLocaleDateString("ar-SA")
                      : "-"}
                  </p>
                </div>
                {invoice.supplierRepresentativeName && (
                  <div>
                    <p className="text-sm text-muted-foreground">المندوب</p>
                    <p>{invoice.supplierRepresentativeName}</p>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>الأصناف</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="rounded-lg border overflow-x-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">الصنف</TableHead>
                      <TableHead className="text-right">الكمية</TableHead>
                      <TableHead className="text-right">السعر</TableHead>
                      <TableHead className="text-right">الضريبة</TableHead>
                      <TableHead className="text-right">الإجمالي</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {invoice.lines.map((line) => (
                      <TableRow key={line.id}>
                        <TableCell>
                          <div>
                            <p className="font-medium">{line.product?.nameAr}</p>
                            <p className="text-xs text-muted-foreground">
                              {line.quantityPurchaseUnit} {line.purchaseUnit?.nameAr} = {line.quantitySellUnit} {line.sellUnit?.nameAr}
                            </p>
                          </div>
                        </TableCell>
                        <TableCell className="font-mono">
                          {line.quantityPurchaseUnit} {line.purchaseUnit?.nameAr}
                        </TableCell>
                        <TableCell className="font-mono">
                          {line.purchasePricePerPurchaseUnit.toLocaleString("ar-SA")}
                        </TableCell>
                        <TableCell>
                          <div>
                            <Badge variant="outline" className="text-xs">
                              {getVatTypeLabel(line.vatType)}
                            </Badge>
                            <p className="text-xs text-muted-foreground mt-1">
                              {line.vatAmount.toLocaleString("ar-SA")} ر.س
                            </p>
                          </div>
                        </TableCell>
                        <TableCell className="font-mono font-medium">
                          {line.lineTotal.toLocaleString("ar-SA")} ر.س
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>
            </CardContent>
          </Card>

          {invoice.payments.length > 0 && (
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <CreditCard className="h-5 w-5" />
                  المدفوعات
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="rounded-lg border">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead className="text-right">التاريخ</TableHead>
                        <TableHead className="text-right">طريقة الدفع</TableHead>
                        <TableHead className="text-right">المبلغ</TableHead>
                        <TableHead className="text-right">ملاحظات</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {invoice.payments.map((payment) => (
                        <TableRow key={payment.id}>
                          <TableCell>
                            {new Date(payment.paymentDate).toLocaleDateString("ar-SA")}
                          </TableCell>
                          <TableCell>{payment.paymentMethod?.nameAr}</TableCell>
                          <TableCell className="font-mono font-medium">
                            {payment.amount.toLocaleString("ar-SA")} ر.س
                          </TableCell>
                          <TableCell className="text-muted-foreground">
                            {payment.notes || "-"}
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                </div>
              </CardContent>
            </Card>
          )}

          {journalEntries && journalEntries.length > 0 && (
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <BookOpen className="h-5 w-5" />
                  القيود المحاسبية
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                {journalEntries.map((entry) => (
                  <div key={entry.id} className="rounded-lg border p-4">
                    <div className="flex items-center justify-between mb-3">
                      <div>
                        <span className="font-mono font-medium">{entry.entryNumber}</span>
                        <span className="text-muted-foreground mx-2">-</span>
                        <span className="text-sm">{entry.description}</span>
                      </div>
                      <span className="text-sm text-muted-foreground">
                        {new Date(entry.date).toLocaleDateString("ar-SA")}
                      </span>
                    </div>
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead className="text-right">الحساب</TableHead>
                          <TableHead className="text-right">مدين</TableHead>
                          <TableHead className="text-right">دائن</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {entry.lines.map((line) => (
                          <TableRow key={line.id}>
                            <TableCell>
                              <span className="font-mono text-xs text-muted-foreground ml-2">
                                {line.accountCode}
                              </span>
                              {line.accountName}
                            </TableCell>
                            <TableCell className="font-mono">
                              {line.debit > 0 ? line.debit.toLocaleString("ar-SA") : "-"}
                            </TableCell>
                            <TableCell className="font-mono">
                              {line.credit > 0 ? line.credit.toLocaleString("ar-SA") : "-"}
                            </TableCell>
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  </div>
                ))}
              </CardContent>
            </Card>
          )}

          {invoice.notes && (
            <Card>
              <CardHeader>
                <CardTitle>ملاحظات</CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-muted-foreground">{invoice.notes}</p>
              </CardContent>
            </Card>
          )}
        </div>

        <div className="space-y-6">
          <Card className="sticky top-6">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Wallet className="h-5 w-5" />
                ملخص الفاتورة
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <div className="flex justify-between text-sm">
                  <span className="text-muted-foreground">المجموع قبل الضريبة:</span>
                  <span className="font-mono">
                    {invoice.subtotalAmount.toLocaleString("ar-SA")} ر.س
                  </span>
                </div>
                <div className="flex justify-between text-sm">
                  <span className="text-muted-foreground">ضريبة القيمة المضافة:</span>
                  <span className="font-mono">
                    {invoice.vatAmount.toLocaleString("ar-SA")} ر.س
                  </span>
                </div>
                <Separator />
                <div className="flex justify-between font-bold text-lg">
                  <span>الإجمالي:</span>
                  <span className="font-mono" data-testid="total-amount">
                    {invoice.totalAmount.toLocaleString("ar-SA")} ر.س
                  </span>
                </div>
              </div>

              <Separator />

              <div className="space-y-2">
                <div className="flex justify-between text-sm">
                  <span className="text-muted-foreground">المدفوع:</span>
                  <span className="font-mono text-green-600">
                    {paidAmount.toLocaleString("ar-SA")} ر.س
                  </span>
                </div>
                <div className="flex justify-between font-bold">
                  <span>المتبقي:</span>
                  <span className="font-mono text-primary" data-testid="remaining-amount">
                    {remainingAmount.toLocaleString("ar-SA")} ر.س
                  </span>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>

      <PaymentDialog
        open={paymentDialogOpen}
        onOpenChange={setPaymentDialogOpen}
        invoice={invoice}
        paymentMethods={paymentMethods}
        onSuccess={() => {
          refetch();
          queryClient.invalidateQueries({ queryKey: ["/api/purchase-invoices"] });
          queryClient.invalidateQueries({ queryKey: ["/api/dashboard/stats"] });
        }}
      />
    </div>
  );
}
