import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { useForm, useFieldArray } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Separator } from "@/components/ui/separator";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import {
  ArrowRight,
  Plus,
  Trash2,
  Save,
  Loader2,
  ArrowLeftRight,
  Calculator,
  Calendar,
  AlertCircle,
} from "lucide-react";
import type { Supplier, ProductWithUnits, Unit, CreatePurchaseInvoiceRequest } from "@shared/schema";

const lineSchema = z.object({
  productId: z.number().nullable(),
  productNameAr: z.string().optional(),
  purchaseUnitNameAr: z.string().optional(),
  sellUnitNameAr: z.string().optional(),
  conversionFactor: z.number().min(1).default(1),
  hasExpiryDate: z.boolean().default(false),
  defaultSellPrice: z.number().optional(),
  quantityPurchaseUnit: z.number().min(0.01, "الكمية مطلوبة"),
  purchasePricePerPurchaseUnit: z.number().min(0, "السعر مطلوب"),
  vatType: z.enum(["inclusive", "exclusive", "exempt"]),
  vatRate: z.number().min(0).max(1).default(0.15),
  expiryDate: z.string().optional(),
});

const invoiceFormSchema = z.object({
  supplierId: z.number().nullable(),
  supplierNameAr: z.string().optional(),
  supplierCrNumber: z.string().optional(),
  supplierVatNumber: z.string().optional(),
  supplierCityAr: z.string().optional(),
  supplierType: z.string().optional(),
  supplierRepName: z.string().optional(),
  supplierRepPhone: z.string().optional(),
  invoiceDate: z.string().min(1, "تاريخ الفاتورة مطلوب"),
  dueDate: z.string().optional(),
  notes: z.string().optional(),
  lines: z.array(lineSchema).min(1, "يجب إضافة صنف واحد على الأقل"),
});

type InvoiceFormValues = z.infer<typeof invoiceFormSchema>;

const VAT_TYPES = [
  { value: "inclusive", label: "شامل الضريبة" },
  { value: "exclusive", label: "غير شامل الضريبة" },
  { value: "exempt", label: "معفي من الضريبة" },
];

function calculateLineAmounts(
  quantity: number,
  pricePerUnit: number,
  vatType: string,
  vatRate: number
) {
  let lineTotal: number;
  let lineSubtotal: number;
  let vatAmount: number;

  if (vatType === "inclusive") {
    lineTotal = pricePerUnit * quantity;
    lineSubtotal = lineTotal / (1 + vatRate);
    vatAmount = lineTotal - lineSubtotal;
  } else if (vatType === "exclusive") {
    lineSubtotal = pricePerUnit * quantity;
    vatAmount = lineSubtotal * vatRate;
    lineTotal = lineSubtotal + vatAmount;
  } else {
    lineSubtotal = pricePerUnit * quantity;
    vatAmount = 0;
    lineTotal = lineSubtotal;
  }

  return { lineSubtotal, vatAmount, lineTotal };
}

function NewSupplierDialog({
  open,
  onOpenChange,
  onSelect,
}: {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  onSelect: (supplier: { id: null; nameAr: string; crNumber?: string; vatNumber?: string; cityAr?: string; supplierType?: string; representativeName?: string; representativePhone?: string }) => void;
}) {
  const [nameAr, setNameAr] = useState("");
  const [crNumber, setCrNumber] = useState("");
  const [vatNumber, setVatNumber] = useState("");

  const handleSubmit = () => {
    if (nameAr.trim()) {
      onSelect({
        id: null,
        nameAr: nameAr.trim(),
        crNumber: crNumber || undefined,
        vatNumber: vatNumber || undefined,
      });
      onOpenChange(false);
      setNameAr("");
      setCrNumber("");
      setVatNumber("");
    }
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent>
        <DialogHeader>
          <DialogTitle>إضافة مورد جديد</DialogTitle>
        </DialogHeader>
        <div className="space-y-4">
          <div>
            <label className="text-sm font-medium">اسم المورد *</label>
            <Input
              value={nameAr}
              onChange={(e) => setNameAr(e.target.value)}
              placeholder="أدخل اسم المورد"
              data-testid="new-supplier-name"
            />
          </div>
          <div className="grid gap-4 grid-cols-2">
            <div>
              <label className="text-sm font-medium">رقم السجل التجاري</label>
              <Input
                value={crNumber}
                onChange={(e) => setCrNumber(e.target.value)}
                placeholder="1234567890"
              />
            </div>
            <div>
              <label className="text-sm font-medium">الرقم الضريبي</label>
              <Input
                value={vatNumber}
                onChange={(e) => setVatNumber(e.target.value)}
                placeholder="300123456700003"
              />
            </div>
          </div>
          <div className="flex justify-end gap-2">
            <Button variant="outline" onClick={() => onOpenChange(false)}>إلغاء</Button>
            <Button onClick={handleSubmit} disabled={!nameAr.trim()} data-testid="confirm-new-supplier">
              إضافة
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}

function NewProductDialog({
  open,
  onOpenChange,
  onSelect,
  units,
}: {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  onSelect: (product: { id: null; nameAr: string; purchaseUnitNameAr: string; sellUnitNameAr: string; conversionFactor: number; hasExpiryDate: boolean; defaultSellPrice?: number }) => void;
  units?: Unit[];
}) {
  const [nameAr, setNameAr] = useState("");
  const [purchaseUnitNameAr, setPurchaseUnitNameAr] = useState("");
  const [sellUnitNameAr, setSellUnitNameAr] = useState("");
  const [conversionFactor, setConversionFactor] = useState(1);
  const [hasExpiryDate, setHasExpiryDate] = useState(false);

  const handleSubmit = () => {
    if (nameAr.trim() && purchaseUnitNameAr && sellUnitNameAr) {
      onSelect({
        id: null,
        nameAr: nameAr.trim(),
        purchaseUnitNameAr,
        sellUnitNameAr,
        conversionFactor,
        hasExpiryDate,
      });
      onOpenChange(false);
      setNameAr("");
      setPurchaseUnitNameAr("");
      setSellUnitNameAr("");
      setConversionFactor(1);
      setHasExpiryDate(false);
    }
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-lg">
        <DialogHeader>
          <DialogTitle>إضافة منتج جديد</DialogTitle>
        </DialogHeader>
        <div className="space-y-4">
          <div>
            <label className="text-sm font-medium">اسم المنتج *</label>
            <Input
              value={nameAr}
              onChange={(e) => setNameAr(e.target.value)}
              placeholder="أدخل اسم المنتج"
              data-testid="new-product-name"
            />
          </div>
          <div className="grid gap-4 grid-cols-3">
            <div>
              <label className="text-sm font-medium">وحدة الشراء *</label>
              <Select value={purchaseUnitNameAr} onValueChange={setPurchaseUnitNameAr}>
                <SelectTrigger>
                  <SelectValue placeholder="اختر" />
                </SelectTrigger>
                <SelectContent>
                  {units?.map((unit) => (
                    <SelectItem key={unit.id} value={unit.nameAr}>
                      {unit.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div>
              <label className="text-sm font-medium">معامل التحويل *</label>
              <Input
                type="number"
                min={1}
                value={conversionFactor}
                onChange={(e) => setConversionFactor(parseInt(e.target.value) || 1)}
              />
            </div>
            <div>
              <label className="text-sm font-medium">وحدة البيع *</label>
              <Select value={sellUnitNameAr} onValueChange={setSellUnitNameAr}>
                <SelectTrigger>
                  <SelectValue placeholder="اختر" />
                </SelectTrigger>
                <SelectContent>
                  {units?.map((unit) => (
                    <SelectItem key={unit.id} value={unit.nameAr}>
                      {unit.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
          {purchaseUnitNameAr && sellUnitNameAr && (
            <div className="rounded-lg bg-muted p-3 text-center">
              <span className="font-bold">1</span> {purchaseUnitNameAr}
              <ArrowLeftRight className="h-4 w-4 mx-2 inline" />
              <span className="font-bold text-primary">{conversionFactor}</span> {sellUnitNameAr}
            </div>
          )}
          <div className="flex items-center gap-2">
            <input
              type="checkbox"
              id="hasExpiry"
              checked={hasExpiryDate}
              onChange={(e) => setHasExpiryDate(e.target.checked)}
              className="rounded"
            />
            <label htmlFor="hasExpiry" className="text-sm">له تاريخ انتهاء صلاحية</label>
          </div>
          <div className="flex justify-end gap-2">
            <Button variant="outline" onClick={() => onOpenChange(false)}>إلغاء</Button>
            <Button
              onClick={handleSubmit}
              disabled={!nameAr.trim() || !purchaseUnitNameAr || !sellUnitNameAr}
              data-testid="confirm-new-product"
            >
              إضافة
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}

export default function PurchaseInvoiceFormPage() {
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const [newSupplierDialogOpen, setNewSupplierDialogOpen] = useState(false);
  const [newProductDialogOpen, setNewProductDialogOpen] = useState(false);
  const [newProductLineIndex, setNewProductLineIndex] = useState<number>(0);
  const [selectedSupplier, setSelectedSupplier] = useState<Supplier | { id: null; nameAr: string } | null>(null);

  const { data: suppliers } = useQuery<Supplier[]>({
    queryKey: ["/api/suppliers"],
  });

  const { data: products } = useQuery<ProductWithUnits[]>({
    queryKey: ["/api/products"],
  });

  const { data: units } = useQuery<Unit[]>({
    queryKey: ["/api/units"],
  });

  const form = useForm<InvoiceFormValues>({
    resolver: zodResolver(invoiceFormSchema),
    defaultValues: {
      supplierId: null,
      invoiceDate: new Date().toISOString().split("T")[0],
      dueDate: "",
      notes: "",
      lines: [
        {
          productId: null,
          quantityPurchaseUnit: 1,
          purchasePricePerPurchaseUnit: 0,
          vatType: "inclusive",
          vatRate: 0.15,
          conversionFactor: 1,
          hasExpiryDate: false,
        },
      ],
    },
  });

  const { fields, append, remove } = useFieldArray({
    control: form.control,
    name: "lines",
  });

  const createMutation = useMutation({
    mutationFn: async (data: InvoiceFormValues) => {
      const requestData: CreatePurchaseInvoiceRequest = {
        supplier: selectedSupplier?.id
          ? { id: selectedSupplier.id }
          : {
              id: null,
              nameAr: data.supplierNameAr,
              crNumber: data.supplierCrNumber,
              vatNumber: data.supplierVatNumber,
              cityAr: data.supplierCityAr,
              supplierType: data.supplierType,
              representativeName: data.supplierRepName,
              representativePhone: data.supplierRepPhone,
            },
        invoiceDate: data.invoiceDate,
        dueDate: data.dueDate || undefined,
        notes: data.notes || undefined,
        lines: data.lines.map((line) => ({
          product: line.productId
            ? { id: line.productId }
            : {
                id: null,
                nameAr: line.productNameAr,
                purchaseUnitNameAr: line.purchaseUnitNameAr,
                sellUnitNameAr: line.sellUnitNameAr,
                conversionFactor: line.conversionFactor,
                hasExpiryDate: line.hasExpiryDate,
                defaultSellPrice: line.defaultSellPrice,
              },
          quantityPurchaseUnit: line.quantityPurchaseUnit,
          purchasePricePerPurchaseUnit: line.purchasePricePerPurchaseUnit,
          vatType: line.vatType,
          vatRate: line.vatRate,
          expiryDate: line.expiryDate || undefined,
        })),
      };
      const response = await apiRequest("POST", "/api/purchase-invoices", requestData);
      return response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["/api/purchase-invoices"] });
      queryClient.invalidateQueries({ queryKey: ["/api/dashboard/stats"] });
      toast({
        title: "تم الإنشاء بنجاح",
        description: `تم إنشاء الفاتورة رقم #${data.id}`,
      });
      navigate("/purchase-invoices");
    },
    onError: () => {
      toast({
        title: "خطأ",
        description: "حدث خطأ أثناء إنشاء الفاتورة",
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: InvoiceFormValues) => {
    if (!selectedSupplier) {
      toast({
        title: "خطأ",
        description: "يجب اختيار مورد",
        variant: "destructive",
      });
      return;
    }
    
    const missingExpiryLines = data.lines.filter((line) => {
      return line.hasExpiryDate && !line.expiryDate;
    });
    
    if (missingExpiryLines.length > 0) {
      toast({
        title: "خطأ في البيانات",
        description: "بعض الأصناف تتطلب تاريخ انتهاء صلاحية. يرجى إدخال التاريخ لجميع الأصناف المطلوبة.",
        variant: "destructive",
      });
      return;
    }
    
    createMutation.mutate(data);
  };

  const handleSelectSupplier = (supplierId: string) => {
    if (supplierId === "new") {
      setNewSupplierDialogOpen(true);
    } else {
      const supplier = suppliers?.find((s) => s.id === parseInt(supplierId));
      if (supplier) {
        setSelectedSupplier(supplier);
        form.setValue("supplierId", supplier.id);
      }
    }
  };

  const handleSelectProduct = (index: number, productId: string) => {
    if (productId === "new") {
      setNewProductLineIndex(index);
      setNewProductDialogOpen(true);
    } else {
      const product = products?.find((p) => p.id === parseInt(productId));
      if (product) {
        const purchaseUnitName = product.purchaseUnit?.nameAr || "";
        const sellUnitName = product.sellUnit?.nameAr || "";
        form.setValue(`lines.${index}.productId`, product.id);
        form.setValue(`lines.${index}.productNameAr`, product.nameAr);
        form.setValue(`lines.${index}.purchaseUnitNameAr`, purchaseUnitName);
        form.setValue(`lines.${index}.sellUnitNameAr`, sellUnitName);
        form.setValue(`lines.${index}.conversionFactor`, product.conversionFactor);
        form.setValue(`lines.${index}.hasExpiryDate`, product.hasExpiryDate);
        if (product.defaultPurchasePrice) {
          form.setValue(`lines.${index}.purchasePricePerPurchaseUnit`, product.defaultPurchasePrice);
        }
      }
    }
  };

  const watchedLines = form.watch("lines");
  const totals = watchedLines.reduce(
    (acc, line) => {
      const amounts = calculateLineAmounts(
        line.quantityPurchaseUnit || 0,
        line.purchasePricePerPurchaseUnit || 0,
        line.vatType,
        line.vatRate
      );
      return {
        subtotal: acc.subtotal + amounts.lineSubtotal,
        vat: acc.vat + amounts.vatAmount,
        total: acc.total + amounts.lineTotal,
      };
    },
    { subtotal: 0, vat: 0, total: 0 }
  );

  return (
    <div className="space-y-6">
      <div className="flex items-center gap-4">
        <Button variant="ghost" size="icon" onClick={() => navigate("/purchase-invoices")} data-testid="button-back">
          <ArrowRight className="h-4 w-4" />
        </Button>
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">فاتورة شراء جديدة</h1>
          <p className="text-muted-foreground">إنشاء فاتورة شراء من مورد</p>
        </div>
      </div>

      <div className="grid gap-6 lg:grid-cols-3">
        <div className="lg:col-span-2 space-y-6">
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>بيانات المورد</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid gap-4 md:grid-cols-2">
                    <div>
                      <label className="text-sm font-medium">المورد *</label>
                      <Select
                        value={selectedSupplier?.id?.toString() || ""}
                        onValueChange={handleSelectSupplier}
                      >
                        <SelectTrigger data-testid="select-supplier">
                          <SelectValue placeholder="اختر مورد" />
                        </SelectTrigger>
                        <SelectContent>
                          {suppliers?.filter(s => s.isActive).map((supplier) => (
                            <SelectItem key={supplier.id} value={supplier.id.toString()}>
                              {supplier.nameAr}
                            </SelectItem>
                          ))}
                          <SelectItem value="new" className="text-primary">
                            <Plus className="h-4 w-4 ml-1 inline" />
                            إضافة مورد جديد
                          </SelectItem>
                        </SelectContent>
                      </Select>
                      {selectedSupplier && (
                        <p className="text-sm text-muted-foreground mt-1">
                          {selectedSupplier.id === null ? "مورد جديد" : `رقم: ${selectedSupplier.id}`}
                        </p>
                      )}
                    </div>
                    <FormField
                      control={form.control}
                      name="invoiceDate"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>تاريخ الفاتورة *</FormLabel>
                          <FormControl>
                            <Input type="date" {...field} data-testid="input-invoice-date" />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>
                  <FormField
                    control={form.control}
                    name="dueDate"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>تاريخ الاستحقاق</FormLabel>
                        <FormControl>
                          <Input type="date" {...field} data-testid="input-due-date" />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between gap-2">
                  <CardTitle>الأصناف</CardTitle>
                  <Button
                    type="button"
                    variant="outline"
                    size="sm"
                    onClick={() =>
                      append({
                        productId: null,
                        quantityPurchaseUnit: 1,
                        purchasePricePerPurchaseUnit: 0,
                        vatType: "inclusive",
                        vatRate: 0.15,
                        conversionFactor: 1,
                        hasExpiryDate: false,
                      })
                    }
                    data-testid="button-add-line"
                  >
                    <Plus className="h-4 w-4 ml-2" />
                    إضافة صنف
                  </Button>
                </CardHeader>
                <CardContent className="space-y-4">
                  {fields.map((field, index) => {
                    const line = watchedLines[index];
                    const amounts = calculateLineAmounts(
                      line?.quantityPurchaseUnit || 0,
                      line?.purchasePricePerPurchaseUnit || 0,
                      line?.vatType || "inclusive",
                      line?.vatRate || 0.15
                    );
                    const selectedProduct = products?.find((p) => p.id === line?.productId);

                    return (
                      <div key={field.id} className="rounded-lg border p-4 space-y-4">
                        <div className="flex items-center justify-between">
                          <span className="font-medium">صنف {index + 1}</span>
                          {fields.length > 1 && (
                            <Button
                              type="button"
                              variant="ghost"
                              size="icon"
                              onClick={() => remove(index)}
                              data-testid={`remove-line-${index}`}
                            >
                              <Trash2 className="h-4 w-4 text-destructive" />
                            </Button>
                          )}
                        </div>

                        <div className="grid gap-4 md:grid-cols-2">
                          <div>
                            <label className="text-sm font-medium">المنتج *</label>
                            <Select
                              value={line?.productId?.toString() || ""}
                              onValueChange={(value) => handleSelectProduct(index, value)}
                            >
                              <SelectTrigger data-testid={`select-product-${index}`}>
                                <SelectValue placeholder="اختر منتج" />
                              </SelectTrigger>
                              <SelectContent>
                                {products?.filter(p => p.isActive).map((product) => (
                                  <SelectItem key={product.id} value={product.id.toString()}>
                                    {product.nameAr}
                                  </SelectItem>
                                ))}
                                <SelectItem value="new" className="text-primary">
                                  <Plus className="h-4 w-4 ml-1 inline" />
                                  إضافة منتج جديد
                                </SelectItem>
                              </SelectContent>
                            </Select>
                          </div>

                          <FormField
                            control={form.control}
                            name={`lines.${index}.vatType`}
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>نوع الضريبة *</FormLabel>
                                <Select onValueChange={field.onChange} value={field.value}>
                                  <FormControl>
                                    <SelectTrigger data-testid={`select-vat-type-${index}`}>
                                      <SelectValue />
                                    </SelectTrigger>
                                  </FormControl>
                                  <SelectContent>
                                    {VAT_TYPES.map((type) => (
                                      <SelectItem key={type.value} value={type.value}>
                                        {type.label}
                                      </SelectItem>
                                    ))}
                                  </SelectContent>
                                </Select>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        </div>

                        {(line?.productId || line?.productNameAr) && line?.conversionFactor && (line?.purchaseUnitNameAr || line?.sellUnitNameAr) && (
                          <div className="rounded-lg border border-primary/30 bg-primary/5 p-3">
                            <div className="flex items-center gap-4 flex-wrap">
                              <div className="flex items-center gap-2">
                                <ArrowLeftRight className="h-4 w-4 text-primary" />
                                <span className="text-sm font-medium">معامل التحويل:</span>
                              </div>
                              <div className="flex items-center gap-2 text-sm">
                                <span className="font-bold text-lg">1</span>
                                <span className="text-muted-foreground">{line.purchaseUnitNameAr}</span>
                                <span className="mx-1">=</span>
                                <span className="font-bold text-lg text-primary">{line.conversionFactor}</span>
                                <span className="text-muted-foreground">{line.sellUnitNameAr}</span>
                              </div>
                              {line?.quantityPurchaseUnit > 0 && (
                                <div className="mr-auto text-sm bg-background rounded px-2 py-1">
                                  <span className="text-muted-foreground">إجمالي وحدات البيع: </span>
                                  <span className="font-bold text-primary">
                                    {(line.quantityPurchaseUnit * line.conversionFactor).toLocaleString("ar-SA")} {line.sellUnitNameAr}
                                  </span>
                                </div>
                              )}
                            </div>
                          </div>
                        )}

                        <div className="grid gap-4 md:grid-cols-3">
                          <FormField
                            control={form.control}
                            name={`lines.${index}.quantityPurchaseUnit`}
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>الكمية (وحدة شراء) *</FormLabel>
                                <FormControl>
                                  <Input
                                    type="number"
                                    min={0.01}
                                    step={0.01}
                                    {...field}
                                    onChange={(e) => field.onChange(parseFloat(e.target.value) || 0)}
                                    data-testid={`input-quantity-${index}`}
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />

                          <FormField
                            control={form.control}
                            name={`lines.${index}.purchasePricePerPurchaseUnit`}
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>سعر الوحدة *</FormLabel>
                                <FormControl>
                                  <Input
                                    type="number"
                                    min={0}
                                    step={0.01}
                                    {...field}
                                    onChange={(e) => field.onChange(parseFloat(e.target.value) || 0)}
                                    data-testid={`input-price-${index}`}
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />

                          <div>
                            {line?.hasExpiryDate ? (
                              <FormField
                                control={form.control}
                                name={`lines.${index}.expiryDate`}
                                render={({ field }) => (
                                  <FormItem>
                                    <FormLabel className="flex items-center gap-1">
                                      <Calendar className="h-4 w-4 text-amber-500" />
                                      <span>تاريخ انتهاء الصلاحية</span>
                                      <span className="text-destructive">*</span>
                                    </FormLabel>
                                    <FormControl>
                                      <Input 
                                        type="date" 
                                        {...field} 
                                        data-testid={`input-expiry-${index}`}
                                        className="border-amber-300 focus:border-amber-500"
                                      />
                                    </FormControl>
                                    <FormMessage />
                                  </FormItem>
                                )}
                              />
                            ) : (
                              <div className="h-full flex items-end pb-2">
                                <span className="text-xs text-muted-foreground">لا يتطلب تاريخ صلاحية</span>
                              </div>
                            )}
                          </div>
                        </div>

                        {line?.hasExpiryDate && !line?.expiryDate && (
                          <div className="flex items-center gap-2 p-2 rounded-lg bg-amber-50 border border-amber-200 text-amber-700 text-sm">
                            <AlertCircle className="h-4 w-4 flex-shrink-0" />
                            <span>هذا المنتج يتطلب تاريخ انتهاء صلاحية. يرجى إدخال التاريخ قبل حفظ الفاتورة.</span>
                          </div>
                        )}

                        <div className="rounded-lg bg-muted p-3 grid grid-cols-3 gap-4 text-sm">
                          <div>
                            <span className="text-muted-foreground">المبلغ قبل الضريبة:</span>
                            <span className="font-mono font-medium mr-2">
                              {amounts.lineSubtotal.toLocaleString("ar-SA", { minimumFractionDigits: 2 })}
                            </span>
                          </div>
                          <div>
                            <span className="text-muted-foreground">الضريبة:</span>
                            <span className="font-mono font-medium mr-2">
                              {amounts.vatAmount.toLocaleString("ar-SA", { minimumFractionDigits: 2 })}
                            </span>
                          </div>
                          <div>
                            <span className="text-muted-foreground">الإجمالي:</span>
                            <span className="font-mono font-bold mr-2 text-primary">
                              {amounts.lineTotal.toLocaleString("ar-SA", { minimumFractionDigits: 2 })}
                            </span>
                          </div>
                        </div>
                      </div>
                    );
                  })}
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>ملاحظات</CardTitle>
                </CardHeader>
                <CardContent>
                  <FormField
                    control={form.control}
                    name="notes"
                    render={({ field }) => (
                      <FormItem>
                        <FormControl>
                          <Textarea
                            placeholder="أضف ملاحظات على الفاتورة..."
                            {...field}
                            data-testid="input-notes"
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </CardContent>
              </Card>

              <div className="flex justify-end">
                <Button type="submit" disabled={createMutation.isPending} data-testid="button-submit">
                  {createMutation.isPending ? (
                    <Loader2 className="h-4 w-4 animate-spin ml-2" />
                  ) : (
                    <Save className="h-4 w-4 ml-2" />
                  )}
                  حفظ الفاتورة
                </Button>
              </div>
            </form>
          </Form>
        </div>

        <div className="space-y-6">
          <Card className="sticky top-6">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Calculator className="h-5 w-5" />
                ملخص الفاتورة
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <div className="flex justify-between text-sm">
                  <span className="text-muted-foreground">المجموع قبل الضريبة:</span>
                  <span className="font-mono" data-testid="subtotal">
                    {totals.subtotal.toLocaleString("ar-SA", { minimumFractionDigits: 2 })} ر.س
                  </span>
                </div>
                <div className="flex justify-between text-sm">
                  <span className="text-muted-foreground">ضريبة القيمة المضافة:</span>
                  <span className="font-mono" data-testid="vat-total">
                    {totals.vat.toLocaleString("ar-SA", { minimumFractionDigits: 2 })} ر.س
                  </span>
                </div>
                <Separator />
                <div className="flex justify-between font-bold text-lg">
                  <span>الإجمالي:</span>
                  <span className="font-mono text-primary" data-testid="grand-total">
                    {totals.total.toLocaleString("ar-SA", { minimumFractionDigits: 2 })} ر.س
                  </span>
                </div>
              </div>

              <Separator />

              <div className="text-sm text-muted-foreground">
                <p>عدد الأصناف: {fields.length}</p>
                {selectedSupplier && (
                  <p>المورد: {selectedSupplier.nameAr}</p>
                )}
              </div>
            </CardContent>
          </Card>
        </div>
      </div>

      <NewSupplierDialog
        open={newSupplierDialogOpen}
        onOpenChange={setNewSupplierDialogOpen}
        onSelect={(supplier) => {
          setSelectedSupplier(supplier);
          form.setValue("supplierId", null);
          form.setValue("supplierNameAr", supplier.nameAr);
          form.setValue("supplierCrNumber", supplier.crNumber);
          form.setValue("supplierVatNumber", supplier.vatNumber);
        }}
      />

      <NewProductDialog
        open={newProductDialogOpen}
        onOpenChange={setNewProductDialogOpen}
        units={units}
        onSelect={(product) => {
          const targetIndex = newProductLineIndex;
          form.setValue(`lines.${targetIndex}.productId`, null);
          form.setValue(`lines.${targetIndex}.productNameAr`, product.nameAr);
          form.setValue(`lines.${targetIndex}.purchaseUnitNameAr`, product.purchaseUnitNameAr);
          form.setValue(`lines.${targetIndex}.sellUnitNameAr`, product.sellUnitNameAr);
          form.setValue(`lines.${targetIndex}.conversionFactor`, product.conversionFactor);
          form.setValue(`lines.${targetIndex}.hasExpiryDate`, product.hasExpiryDate);
        }}
      />
    </div>
  );
}
