import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Skeleton } from "@/components/ui/skeleton";
import { Separator } from "@/components/ui/separator";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  BarChart3,
  FileText,
  TrendingUp,
  TrendingDown,
  Calendar,
  Download,
  Warehouse,
  Building2,
} from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { format } from "date-fns";
import { AlertTriangle, ArrowUpDown } from "lucide-react";
import type {
  DashboardStats,
  InventorySummaryItem,
  SupplierBalance,
  Supplier,
  StockMovementSummaryItem,
  NearExpiryProductItem,
  Product,
} from "@shared/schema";

type SupplierWithBalance = Supplier & { balance: SupplierBalance };

function ReportCard({
  title,
  description,
  icon: Icon,
  children,
}: {
  title: string;
  description: string;
  icon: React.ComponentType<{ className?: string }>;
  children: React.ReactNode;
}) {
  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Icon className="h-5 w-5" />
          {title}
        </CardTitle>
        <CardDescription>{description}</CardDescription>
      </CardHeader>
      <CardContent>{children}</CardContent>
    </Card>
  );
}

export default function ReportsPage() {
  const [dateFrom, setDateFrom] = useState(() => {
    const date = new Date();
    date.setMonth(date.getMonth() - 1);
    return date.toISOString().split("T")[0];
  });
  const [dateTo, setDateTo] = useState(() => new Date().toISOString().split("T")[0]);
  const [selectedProductId, setSelectedProductId] = useState<string>("all");
  const [expiryDays, setExpiryDays] = useState<number>(30);

  const { data: stats, isLoading: isLoadingStats } = useQuery<DashboardStats>({
    queryKey: ["/api/dashboard/stats"],
  });
  
  const { data: products } = useQuery<Product[]>({
    queryKey: ["/api/products"],
  });

  const stockMovementQueryUrl = selectedProductId === "all" 
    ? `/api/reports/stock-movement-summary?from_date=${dateFrom}&to_date=${dateTo}`
    : `/api/reports/stock-movement-summary?from_date=${dateFrom}&to_date=${dateTo}&product_id=${selectedProductId}`;
  
  const { data: stockMovementSummary, isLoading: isLoadingMovement } = useQuery<StockMovementSummaryItem[]>({
    queryKey: [stockMovementQueryUrl],
  });

  const { data: nearExpiryProducts, isLoading: isLoadingExpiry } = useQuery<NearExpiryProductItem[]>({
    queryKey: [`/api/reports/stock-expiry?days=${expiryDays}`],
  });

  const { data: inventorySummary, isLoading: isLoadingInventory } = useQuery<
    InventorySummaryItem[]
  >({
    queryKey: ["/api/reports/inventory-summary"],
  });

  const { data: suppliersWithBalances, isLoading: isLoadingSuppliers } = useQuery<
    SupplierWithBalance[]
  >({
    queryKey: ["/api/reports/supplier-balances"],
  });

  const totalInventoryValue =
    inventorySummary?.reduce((sum, item) => sum + item.totalCost, 0) || 0;
  const totalUnpaidBalance =
    suppliersWithBalances?.reduce((sum, s) => sum + (s.balance?.balance || 0), 0) || 0;

  return (
    <div className="space-y-6">
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">التقارير</h1>
          <p className="text-muted-foreground">تقارير المشتريات والمالية والمخزون</p>
        </div>
        <div className="flex flex-col sm:flex-row gap-2">
          <div className="flex items-center gap-2">
            <span className="text-sm text-muted-foreground">من:</span>
            <Input
              type="date"
              value={dateFrom}
              onChange={(e) => setDateFrom(e.target.value)}
              className="w-40"
              data-testid="input-date-from"
            />
          </div>
          <div className="flex items-center gap-2">
            <span className="text-sm text-muted-foreground">إلى:</span>
            <Input
              type="date"
              value={dateTo}
              onChange={(e) => setDateTo(e.target.value)}
              className="w-40"
              data-testid="input-date-to"
            />
          </div>
        </div>
      </div>

      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-primary/10">
                <FileText className="h-6 w-6 text-primary" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">عدد الموردين</p>
                <p className="text-2xl font-bold tabular-nums" data-testid="total-suppliers">
                  {(stats?.totalSuppliers || 0).toLocaleString("ar-SA")}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-green-500/10">
                <TrendingDown className="h-6 w-6 text-green-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">عدد المنتجات</p>
                <p className="text-2xl font-bold tabular-nums" data-testid="total-products">
                  {(stats?.totalProducts || 0).toLocaleString("ar-SA")}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-amber-500/10">
                <TrendingUp className="h-6 w-6 text-amber-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">المستحقات المتبقية</p>
                <p className="text-2xl font-bold tabular-nums" data-testid="total-unpaid">
                  {(stats?.totalUnpaidAmount || 0).toLocaleString("ar-SA")} ر.س
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-blue-500/10">
                <Warehouse className="h-6 w-6 text-blue-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">قيمة المخزون</p>
                <p className="text-2xl font-bold tabular-nums" data-testid="inventory-value">
                  {totalInventoryValue.toLocaleString("ar-SA")} ر.س
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="grid gap-6 lg:grid-cols-2">
        <ReportCard
          title="تقرير المخزون"
          description="ملخص المخزون الحالي وقيمة كل منتج"
          icon={Warehouse}
        >
          {isLoadingInventory ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-10" />
              ))}
            </div>
          ) : !inventorySummary || inventorySummary.length === 0 ? (
            <div className="text-center text-muted-foreground py-8">
              لا يوجد مخزون
            </div>
          ) : (
            <div className="rounded-lg border max-h-80 overflow-y-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right sticky top-0 bg-background">المنتج</TableHead>
                    <TableHead className="text-right sticky top-0 bg-background">الكمية</TableHead>
                    <TableHead className="text-right sticky top-0 bg-background">القيمة</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {inventorySummary.slice(0, 10).map((item) => (
                    <TableRow key={item.productId}>
                      <TableCell>{item.productNameAr}</TableCell>
                      <TableCell className="font-mono tabular-nums">
                        {item.quantityOnHand.toLocaleString("ar-SA")} {item.sellUnitNameAr}
                      </TableCell>
                      <TableCell className="font-mono tabular-nums">
                        {item.totalCost.toLocaleString("ar-SA")}
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </ReportCard>

        <ReportCard
          title="أرصدة الموردين"
          description="المبالغ المستحقة لكل مورد"
          icon={Building2}
        >
          {isLoadingSuppliers ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-10" />
              ))}
            </div>
          ) : !suppliersWithBalances || suppliersWithBalances.length === 0 ? (
            <div className="text-center text-muted-foreground py-8">
              لا يوجد موردين
            </div>
          ) : (
            <div className="rounded-lg border max-h-80 overflow-y-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right sticky top-0 bg-background">المورد</TableHead>
                    <TableHead className="text-right sticky top-0 bg-background">إجمالي المشتريات</TableHead>
                    <TableHead className="text-right sticky top-0 bg-background">المتبقي</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {suppliersWithBalances
                    .filter((s) => s.balance?.balance > 0)
                    .slice(0, 10)
                    .map((supplier) => (
                      <TableRow key={supplier.id}>
                        <TableCell>{supplier.nameAr}</TableCell>
                        <TableCell className="font-mono tabular-nums">
                          {(supplier.balance?.totalCredits || 0).toLocaleString("ar-SA")}
                        </TableCell>
                        <TableCell className="font-mono tabular-nums text-primary font-bold">
                          {(supplier.balance?.balance || 0).toLocaleString("ar-SA")}
                        </TableCell>
                      </TableRow>
                    ))}
                </TableBody>
              </Table>
            </div>
          )}
        </ReportCard>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <BarChart3 className="h-5 w-5" />
            ملخص الفواتير
          </CardTitle>
          <CardDescription>
            إحصائيات فواتير المشتريات حسب حالة الدفع
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 sm:grid-cols-3">
            <div className="rounded-lg border p-4 text-center">
              <p className="text-3xl font-bold text-primary tabular-nums" data-testid="invoices-count">
                {stats?.recentInvoices?.length || 0}
              </p>
              <p className="text-sm text-muted-foreground">آخر الفواتير</p>
            </div>
            <div className="rounded-lg border p-4 text-center">
              <p className="text-3xl font-bold text-green-600 tabular-nums" data-testid="paid-invoices">
                {stats?.recentInvoices?.filter(i => i.paymentStatus === "paid").length || 0}
              </p>
              <p className="text-sm text-muted-foreground">فواتير مدفوعة</p>
            </div>
            <div className="rounded-lg border p-4 text-center">
              <p className="text-3xl font-bold text-amber-600 tabular-nums" data-testid="unpaid-invoices">
                {stats?.totalUnpaidInvoices || 0}
              </p>
              <p className="text-sm text-muted-foreground">فواتير غير مدفوعة</p>
            </div>
          </div>
        </CardContent>
      </Card>

      <div className="grid gap-6 lg:grid-cols-2">
        <ReportCard
          title="ملخص حركة المخزون"
          description="ملخص الواردات والصادرات للفترة"
          icon={ArrowUpDown}
        >
          <div className="space-y-4">
            <Select value={selectedProductId} onValueChange={setSelectedProductId}>
              <SelectTrigger data-testid="select-product-movement">
                <SelectValue placeholder="اختر منتج لعرض الملخص" />
              </SelectTrigger>
              <SelectContent>
                {products?.map((product) => (
                  <SelectItem key={product.id} value={product.id.toString()}>
                    {product.nameAr}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
            
            {selectedProductId === "all" ? (
              <div className="text-center text-muted-foreground py-8">
                اختر منتج لعرض ملخص حركته
              </div>
            ) : isLoadingMovement ? (
              <div className="space-y-2">
                {[1, 2, 3].map((i) => (
                  <Skeleton key={i} className="h-10" />
                ))}
              </div>
            ) : !stockMovementSummary || stockMovementSummary.length === 0 ? (
              <div className="text-center text-muted-foreground py-8">
                لا توجد حركات لهذا المنتج
              </div>
            ) : (
              <div className="rounded-lg border max-h-80 overflow-y-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right sticky top-0 bg-background">المنتج</TableHead>
                      <TableHead className="text-right sticky top-0 bg-background">رصيد الافتتاح</TableHead>
                      <TableHead className="text-right sticky top-0 bg-background">الوارد</TableHead>
                      <TableHead className="text-right sticky top-0 bg-background">الصادر</TableHead>
                      <TableHead className="text-right sticky top-0 bg-background">رصيد الإغلاق</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {stockMovementSummary.map((item) => (
                      <TableRow key={item.productId}>
                        <TableCell>{item.productNameAr}</TableCell>
                        <TableCell className="font-mono tabular-nums">
                          {item.openingQuantity} {item.sellUnitNameAr}
                        </TableCell>
                        <TableCell className="font-mono tabular-nums text-green-600">
                          +{item.quantityInPeriod}
                        </TableCell>
                        <TableCell className="font-mono tabular-nums text-red-600">
                          -{item.quantityOutPeriod}
                        </TableCell>
                        <TableCell className="font-mono tabular-nums font-bold">
                          {item.closingQuantity}
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>
            )}
          </div>
        </ReportCard>

        <ReportCard
          title="المنتجات القريبة من الانتهاء"
          description="المنتجات التي ستنتهي صلاحيتها قريباً"
          icon={AlertTriangle}
        >
          <div className="space-y-4">
            <Select value={expiryDays.toString()} onValueChange={(v) => setExpiryDays(parseInt(v))}>
              <SelectTrigger data-testid="select-expiry-days">
                <SelectValue placeholder="اختر فترة الانتهاء" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="7">خلال 7 أيام</SelectItem>
                <SelectItem value="14">خلال 14 يوم</SelectItem>
                <SelectItem value="30">خلال 30 يوم</SelectItem>
                <SelectItem value="60">خلال 60 يوم</SelectItem>
                <SelectItem value="90">خلال 90 يوم</SelectItem>
              </SelectContent>
            </Select>

            {isLoadingExpiry ? (
              <div className="space-y-2">
                {[1, 2, 3].map((i) => (
                  <Skeleton key={i} className="h-10" />
                ))}
              </div>
            ) : !nearExpiryProducts || nearExpiryProducts.length === 0 ? (
              <div className="text-center text-muted-foreground py-8">
                لا توجد منتجات قريبة من الانتهاء
              </div>
            ) : (
              <div className="rounded-lg border max-h-80 overflow-y-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right sticky top-0 bg-background">المنتج</TableHead>
                      <TableHead className="text-right sticky top-0 bg-background">الكمية</TableHead>
                      <TableHead className="text-right sticky top-0 bg-background">تاريخ الانتهاء</TableHead>
                      <TableHead className="text-right sticky top-0 bg-background">الأيام المتبقية</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {nearExpiryProducts.map((item) => (
                      <TableRow key={item.batchId}>
                        <TableCell>{item.productNameAr}</TableCell>
                        <TableCell className="font-mono tabular-nums">
                          {item.quantityOnHand}
                        </TableCell>
                        <TableCell>
                          {format(new Date(item.expiryDate), "yyyy-MM-dd")}
                        </TableCell>
                        <TableCell>
                          <Badge variant={item.daysToExpiry <= 7 ? "destructive" : item.daysToExpiry <= 14 ? "default" : "secondary"}>
                            {item.daysToExpiry} يوم
                          </Badge>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>
            )}
          </div>
        </ReportCard>
      </div>
    </div>
  );
}
