import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Textarea } from "@/components/ui/textarea";
import { Checkbox } from "@/components/ui/checkbox";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { ScrollArea } from "@/components/ui/scroll-area";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { 
  Shield, 
  Plus, 
  Pencil, 
  Trash2, 
  CheckCircle2, 
  XCircle, 
  Code, 
  Users,
  Crown,
  Eye,
  FilePlus,
  FileEdit,
  FileX,
  Download,
  Printer,
  Settings,
  LayoutDashboard,
  ShoppingCart,
  Calendar,
  UserCog,
  Package,
  Warehouse,
  Truck,
  Receipt,
  Banknote,
  Clock,
  DollarSign,
  Calculator,
  BarChart,
  Megaphone,
  Car,
  Coffee,
  Award,
  Scissors,
} from "lucide-react";
import { systemModules, permissionActions } from "@shared/schema";

interface RolePermission {
  id?: number;
  roleId: number;
  moduleCode: string;
  canView: boolean;
  canCreate: boolean;
  canEdit: boolean;
  canDelete: boolean;
  canExport: boolean;
  canPrint: boolean;
}

interface RoleWithPermissions {
  id: number;
  nameAr: string;
  code: string;
  descriptionAr: string | null;
  isSuperAdmin: boolean;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
  permissions: RolePermission[];
  employeeCount?: number;
}

interface RoleFormData {
  nameAr: string;
  code: string;
  descriptionAr: string;
  isSuperAdmin: boolean;
  isActive: boolean;
}

const moduleIcons: Record<string, typeof Shield> = {
  dashboard: LayoutDashboard,
  orders: ShoppingCart,
  appointments: Calendar,
  customers: Users,
  services: Scissors,
  products: Package,
  inventory: Warehouse,
  suppliers: Truck,
  purchases: Receipt,
  employees: UserCog,
  payroll: Banknote,
  attendance: Clock,
  finance: DollarSign,
  accounting: Calculator,
  reports: BarChart,
  marketing: Megaphone,
  drivers: Car,
  cafeteria: Coffee,
  settings: Settings,
  roles: Shield,
  brands: Award,
};

export default function RolesPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingRole, setEditingRole] = useState<RoleWithPermissions | null>(null);
  const [formData, setFormData] = useState<RoleFormData>({
    nameAr: "",
    code: "",
    descriptionAr: "",
    isSuperAdmin: false,
    isActive: true,
  });
  const [permissions, setPermissions] = useState<Record<string, RolePermission>>({});
  const { toast } = useToast();

  const { data: roles, isLoading } = useQuery<RoleWithPermissions[]>({
    queryKey: ["/api/roles", "withPermissions"],
    queryFn: async () => {
      const response = await fetch("/api/roles?withPermissions=true");
      if (!response.ok) throw new Error("Failed to fetch roles");
      return response.json();
    },
  });

  const createMutation = useMutation({
    mutationFn: async (data: RoleFormData) => {
      const role = await apiRequest("POST", "/api/roles", data);
      const roleData = await role.json();
      
      const permissionsList = Object.values(permissions).filter(p => 
        p.canView || p.canCreate || p.canEdit || p.canDelete || p.canExport || p.canPrint
      );
      
      if (permissionsList.length > 0) {
        await apiRequest("PUT", `/api/roles/${roleData.id}/permissions`, {
          permissions: permissionsList,
        });
      }
      
      return roleData;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/roles"] });
      toast({ title: "تم إنشاء الدور بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء الدور",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async ({ id, data }: { id: number; data: Partial<RoleFormData> }) => {
      await apiRequest("PUT", `/api/roles/${id}`, data);
      
      const permissionsList = Object.values(permissions).filter(p => 
        p.canView || p.canCreate || p.canEdit || p.canDelete || p.canExport || p.canPrint
      );
      
      await apiRequest("PUT", `/api/roles/${id}/permissions`, {
        permissions: permissionsList,
      });
      
      return { id };
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/roles"] });
      toast({ title: "تم تحديث الدور بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث الدور",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/roles/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/roles"] });
      toast({ title: "تم تعطيل الدور بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف الدور",
        variant: "destructive",
      });
    },
  });

  const initializePermissions = (rolePermissions?: RolePermission[]) => {
    const perms: Record<string, RolePermission> = {};
    systemModules.forEach(module => {
      const existing = rolePermissions?.find(p => p.moduleCode === module.code);
      perms[module.code] = existing || {
        roleId: 0,
        moduleCode: module.code,
        canView: false,
        canCreate: false,
        canEdit: false,
        canDelete: false,
        canExport: false,
        canPrint: false,
      };
    });
    setPermissions(perms);
  };

  const resetForm = () => {
    setFormData({ nameAr: "", code: "", descriptionAr: "", isSuperAdmin: false, isActive: true });
    setEditingRole(null);
    setIsDialogOpen(false);
    initializePermissions();
  };

  const handleOpenDialog = (role?: RoleWithPermissions) => {
    if (role) {
      setEditingRole(role);
      setFormData({
        nameAr: role.nameAr,
        code: role.code || "",
        descriptionAr: role.descriptionAr || "",
        isSuperAdmin: role.isSuperAdmin,
        isActive: role.isActive,
      });
      initializePermissions(role.permissions);
    } else {
      setFormData({ nameAr: "", code: "", descriptionAr: "", isSuperAdmin: false, isActive: true });
      setEditingRole(null);
      initializePermissions();
    }
    setIsDialogOpen(true);
  };

  const handleSubmit = () => {
    if (!formData.nameAr.trim()) {
      toast({ title: "اسم الدور مطلوب", variant: "destructive" });
      return;
    }
    if (!formData.code.trim()) {
      toast({ title: "رمز الدور مطلوب", variant: "destructive" });
      return;
    }

    if (editingRole) {
      updateMutation.mutate({ id: editingRole.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const togglePermission = (moduleCode: string, action: string) => {
    setPermissions(prev => ({
      ...prev,
      [moduleCode]: {
        ...prev[moduleCode],
        [action]: !prev[moduleCode][action as keyof RolePermission],
      },
    }));
  };

  const toggleAllForModule = (moduleCode: string, value: boolean) => {
    setPermissions(prev => ({
      ...prev,
      [moduleCode]: {
        ...prev[moduleCode],
        canView: value,
        canCreate: value,
        canEdit: value,
        canDelete: value,
        canExport: value,
        canPrint: value,
      },
    }));
  };

  const toggleAllForAction = (action: string, value: boolean) => {
    setPermissions(prev => {
      const updated = { ...prev };
      Object.keys(updated).forEach(moduleCode => {
        updated[moduleCode] = {
          ...updated[moduleCode],
          [action]: value,
        };
      });
      return updated;
    });
  };

  const activeRoles = roles?.filter(r => r.isActive) || [];
  const inactiveRoles = roles?.filter(r => !r.isActive) || [];
  const superAdminRoles = roles?.filter(r => r.isSuperAdmin) || [];

  const getModuleIcon = (code: string) => {
    const Icon = moduleIcons[code] || Shield;
    return <Icon className="h-4 w-4" />;
  };

  const countPermissions = (role: RoleWithPermissions) => {
    if (role.isSuperAdmin) return "كامل الصلاحيات";
    let count = 0;
    role.permissions.forEach(p => {
      if (p.canView) count++;
      if (p.canCreate) count++;
      if (p.canEdit) count++;
      if (p.canDelete) count++;
      if (p.canExport) count++;
      if (p.canPrint) count++;
    });
    return `${count} صلاحية`;
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">إدارة الأدوار والصلاحيات</h1>
          <p className="text-muted-foreground">تعريف أدوار وصلاحيات الموظفين لكل موديول في النظام</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-role">
          <Plus className="h-4 w-4 ml-2" />
          إضافة دور جديد
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <Shield className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الأدوار</p>
                <p className="text-xl font-bold" data-testid="text-total-roles">
                  {roles?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-amber-100 text-amber-800 dark:bg-amber-900 dark:text-amber-200">
                <Crown className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">سوبر أدمن</p>
                <p className="text-xl font-bold" data-testid="text-super-admin-roles">
                  {superAdminRoles.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <CheckCircle2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">أدوار مفعّلة</p>
                <p className="text-xl font-bold" data-testid="text-active-roles">
                  {activeRoles.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                <XCircle className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">أدوار غير مفعّلة</p>
                <p className="text-xl font-bold" data-testid="text-inactive-roles">
                  {inactiveRoles.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Shield className="h-5 w-5" />
            قائمة الأدوار
          </CardTitle>
          <CardDescription>
            جميع الأدوار المعرّفة في النظام مع صلاحياتها
          </CardDescription>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">اسم الدور</TableHead>
                  <TableHead className="text-right">الرمز</TableHead>
                  <TableHead className="text-right">النوع</TableHead>
                  <TableHead className="text-right">الصلاحيات</TableHead>
                  <TableHead className="text-right">الموظفين</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-left">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {roles?.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={7} className="text-center py-8 text-muted-foreground">
                      لا توجد أدوار مسجلة
                    </TableCell>
                  </TableRow>
                ) : (
                  roles?.map((role) => (
                    <TableRow key={role.id} data-testid={`row-role-${role.id}`}>
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          {role.isSuperAdmin ? (
                            <Crown className="h-4 w-4 text-amber-500" />
                          ) : (
                            <Shield className="h-4 w-4 text-muted-foreground" />
                          )}
                          {role.nameAr}
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant="outline" className="font-mono">
                          <Code className="h-3 w-3 ml-1" />
                          {role.code}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        {role.isSuperAdmin ? (
                          <Badge className="bg-amber-100 text-amber-800 dark:bg-amber-900 dark:text-amber-200">
                            سوبر أدمن
                          </Badge>
                        ) : (
                          <Badge variant="secondary">عادي</Badge>
                        )}
                      </TableCell>
                      <TableCell>
                        <span className="text-sm text-muted-foreground">
                          {countPermissions(role)}
                        </span>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1">
                          <Users className="h-4 w-4 text-muted-foreground" />
                          <span>{role.employeeCount || 0}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        {role.isActive ? (
                          <Badge className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                            مفعّل
                          </Badge>
                        ) : (
                          <Badge className="bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                            غير مفعّل
                          </Badge>
                        )}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleOpenDialog(role)}
                            data-testid={`button-edit-role-${role.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => deleteMutation.mutate(role.id)}
                            disabled={deleteMutation.isPending}
                            data-testid={`button-delete-role-${role.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-4xl max-h-[90vh]" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingRole ? "تعديل الدور" : "إضافة دور جديد"}
            </DialogTitle>
            <DialogDescription>
              تعريف الدور وتحديد صلاحياته لكل موديول في النظام
            </DialogDescription>
          </DialogHeader>
          
          <Tabs defaultValue="info" className="w-full">
            <TabsList className="grid w-full grid-cols-2">
              <TabsTrigger value="info">معلومات الدور</TabsTrigger>
              <TabsTrigger value="permissions">الصلاحيات</TabsTrigger>
            </TabsList>
            
            <TabsContent value="info" className="space-y-4 py-4">
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="nameAr">اسم الدور <span className="text-destructive">*</span></Label>
                  <Input
                    id="nameAr"
                    value={formData.nameAr}
                    onChange={(e) => setFormData({ ...formData, nameAr: e.target.value })}
                    placeholder="مثال: مدير النظام"
                    data-testid="input-role-name"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="code">رمز الدور <span className="text-destructive">*</span></Label>
                  <Input
                    id="code"
                    value={formData.code}
                    onChange={(e) => setFormData({ ...formData, code: e.target.value.toUpperCase() })}
                    placeholder="مثال: ADMIN"
                    className="font-mono text-left"
                    dir="ltr"
                    data-testid="input-role-code"
                  />
                </div>
              </div>
              
              <div className="space-y-2">
                <Label htmlFor="descriptionAr">الوصف</Label>
                <Textarea
                  id="descriptionAr"
                  value={formData.descriptionAr}
                  onChange={(e) => setFormData({ ...formData, descriptionAr: e.target.value })}
                  placeholder="وصف مختصر للصلاحيات والمسؤوليات"
                  rows={3}
                  data-testid="input-role-description"
                />
              </div>
              
              <div className="flex items-center justify-between p-4 rounded-lg border bg-amber-50 dark:bg-amber-950/30">
                <div className="flex items-center gap-3">
                  <Crown className="h-5 w-5 text-amber-600" />
                  <div>
                    <Label htmlFor="isSuperAdmin" className="font-medium">سوبر أدمن</Label>
                    <p className="text-sm text-muted-foreground">
                      صلاحية كاملة على جميع موديولات النظام
                    </p>
                  </div>
                </div>
                <Switch
                  id="isSuperAdmin"
                  checked={formData.isSuperAdmin}
                  onCheckedChange={(checked) => setFormData({ ...formData, isSuperAdmin: checked })}
                  data-testid="switch-super-admin"
                />
              </div>
              
              <div className="flex items-center justify-between">
                <Label htmlFor="isActive">الحالة</Label>
                <div className="flex items-center gap-2">
                  <span className="text-sm text-muted-foreground">
                    {formData.isActive ? "مفعّل" : "غير مفعّل"}
                  </span>
                  <Switch
                    id="isActive"
                    checked={formData.isActive}
                    onCheckedChange={(checked) => setFormData({ ...formData, isActive: checked })}
                    data-testid="switch-role-active"
                  />
                </div>
              </div>
            </TabsContent>
            
            <TabsContent value="permissions" className="py-4">
              {formData.isSuperAdmin ? (
                <div className="flex items-center justify-center py-12 text-center">
                  <div>
                    <Crown className="h-16 w-16 mx-auto text-amber-500 mb-4" />
                    <h3 className="text-lg font-semibold">سوبر أدمن</h3>
                    <p className="text-muted-foreground">
                      هذا الدور لديه صلاحية كاملة على جميع موديولات النظام
                    </p>
                  </div>
                </div>
              ) : (
                <ScrollArea className="h-[400px]">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead className="text-right sticky right-0 bg-background">الموديول</TableHead>
                        <TableHead className="text-center">
                          <div className="flex flex-col items-center gap-1">
                            <Eye className="h-4 w-4" />
                            <span className="text-xs">عرض</span>
                          </div>
                        </TableHead>
                        <TableHead className="text-center">
                          <div className="flex flex-col items-center gap-1">
                            <FilePlus className="h-4 w-4" />
                            <span className="text-xs">إضافة</span>
                          </div>
                        </TableHead>
                        <TableHead className="text-center">
                          <div className="flex flex-col items-center gap-1">
                            <FileEdit className="h-4 w-4" />
                            <span className="text-xs">تعديل</span>
                          </div>
                        </TableHead>
                        <TableHead className="text-center">
                          <div className="flex flex-col items-center gap-1">
                            <FileX className="h-4 w-4" />
                            <span className="text-xs">حذف</span>
                          </div>
                        </TableHead>
                        <TableHead className="text-center">
                          <div className="flex flex-col items-center gap-1">
                            <Download className="h-4 w-4" />
                            <span className="text-xs">تصدير</span>
                          </div>
                        </TableHead>
                        <TableHead className="text-center">
                          <div className="flex flex-col items-center gap-1">
                            <Printer className="h-4 w-4" />
                            <span className="text-xs">طباعة</span>
                          </div>
                        </TableHead>
                        <TableHead className="text-center">الكل</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {systemModules.map((module) => {
                        const perm = permissions[module.code];
                        const allChecked = perm?.canView && perm?.canCreate && perm?.canEdit && perm?.canDelete && perm?.canExport && perm?.canPrint;
                        
                        return (
                          <TableRow key={module.code}>
                            <TableCell className="sticky right-0 bg-background">
                              <div className="flex items-center gap-2">
                                {getModuleIcon(module.code)}
                                <span>{module.nameAr}</span>
                              </div>
                            </TableCell>
                            <TableCell className="text-center">
                              <Checkbox
                                checked={perm?.canView || false}
                                onCheckedChange={() => togglePermission(module.code, "canView")}
                                data-testid={`checkbox-${module.code}-view`}
                              />
                            </TableCell>
                            <TableCell className="text-center">
                              <Checkbox
                                checked={perm?.canCreate || false}
                                onCheckedChange={() => togglePermission(module.code, "canCreate")}
                                data-testid={`checkbox-${module.code}-create`}
                              />
                            </TableCell>
                            <TableCell className="text-center">
                              <Checkbox
                                checked={perm?.canEdit || false}
                                onCheckedChange={() => togglePermission(module.code, "canEdit")}
                                data-testid={`checkbox-${module.code}-edit`}
                              />
                            </TableCell>
                            <TableCell className="text-center">
                              <Checkbox
                                checked={perm?.canDelete || false}
                                onCheckedChange={() => togglePermission(module.code, "canDelete")}
                                data-testid={`checkbox-${module.code}-delete`}
                              />
                            </TableCell>
                            <TableCell className="text-center">
                              <Checkbox
                                checked={perm?.canExport || false}
                                onCheckedChange={() => togglePermission(module.code, "canExport")}
                                data-testid={`checkbox-${module.code}-export`}
                              />
                            </TableCell>
                            <TableCell className="text-center">
                              <Checkbox
                                checked={perm?.canPrint || false}
                                onCheckedChange={() => togglePermission(module.code, "canPrint")}
                                data-testid={`checkbox-${module.code}-print`}
                              />
                            </TableCell>
                            <TableCell className="text-center">
                              <Checkbox
                                checked={allChecked}
                                onCheckedChange={(checked) => toggleAllForModule(module.code, !!checked)}
                                data-testid={`checkbox-${module.code}-all`}
                              />
                            </TableCell>
                          </TableRow>
                        );
                      })}
                    </TableBody>
                  </Table>
                </ScrollArea>
              )}
            </TabsContent>
          </Tabs>
          
          <DialogFooter className="gap-2 sm:gap-0">
            <Button
              variant="outline"
              onClick={resetForm}
              data-testid="button-cancel-role"
            >
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-save-role"
            >
              {createMutation.isPending || updateMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
