import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency } from "@/lib/formatters";
import { Wallet, Plus, Pencil, Trash2, DollarSign, Home, Car, Coins, CheckCircle2, XCircle } from "lucide-react";

interface Employee {
  id: number;
  nameAr: string | null;
  fullNameAr: string | null;
}

interface SalaryProfile {
  id: number;
  employeeId: number;
  basicSalary: number;
  housingAllowance: number;
  transportAllowance: number;
  otherFixedAllowances: number;
  isActive: boolean;
  employee?: Employee;
  createdAt: string;
  updatedAt: string;
}

interface SalaryProfileFormData {
  employeeId: number | null;
  basicSalary: number;
  housingAllowance: number;
  transportAllowance: number;
  otherFixedAllowances: number;
  isActive: boolean;
}

export default function SalaryProfilesPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingProfile, setEditingProfile] = useState<SalaryProfile | null>(null);
  const [formData, setFormData] = useState<SalaryProfileFormData>({
    employeeId: null,
    basicSalary: 0,
    housingAllowance: 0,
    transportAllowance: 0,
    otherFixedAllowances: 0,
    isActive: true,
  });
  const { toast } = useToast();

  const { data: profiles, isLoading } = useQuery<SalaryProfile[]>({
    queryKey: ["/api/salary-profiles"],
  });

  const { data: employees } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const saveMutation = useMutation({
    mutationFn: (data: SalaryProfileFormData) =>
      apiRequest("POST", "/api/salary-profiles", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/salary-profiles"] });
      toast({ title: "تم حفظ ملف الراتب بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حفظ ملف الراتب",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (employeeId: number) =>
      apiRequest("DELETE", `/api/salary-profiles/${employeeId}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/salary-profiles"] });
      toast({ title: "تم حذف ملف الراتب بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف ملف الراتب",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      employeeId: null,
      basicSalary: 0,
      housingAllowance: 0,
      transportAllowance: 0,
      otherFixedAllowances: 0,
      isActive: true,
    });
    setEditingProfile(null);
    setIsDialogOpen(false);
  };

  const handleOpenDialog = (profile?: SalaryProfile) => {
    if (profile) {
      setEditingProfile(profile);
      setFormData({
        employeeId: profile.employeeId,
        basicSalary: profile.basicSalary,
        housingAllowance: profile.housingAllowance,
        transportAllowance: profile.transportAllowance,
        otherFixedAllowances: profile.otherFixedAllowances,
        isActive: profile.isActive,
      });
    } else {
      resetForm();
    }
    setIsDialogOpen(true);
  };

  const handleSubmit = () => {
    if (!formData.employeeId) {
      toast({ title: "يجب اختيار الموظف", variant: "destructive" });
      return;
    }

    saveMutation.mutate(formData);
  };

  const getEmployeeName = (emp?: Employee) => emp?.fullNameAr || emp?.nameAr || "-";

  const getTotalSalary = (profile: SalaryProfile) => {
    return profile.basicSalary + profile.housingAllowance + profile.transportAllowance + profile.otherFixedAllowances;
  };

  const employeesWithProfiles = new Set(profiles?.map(p => p.employeeId) || []);
  const employeesWithoutProfiles = employees?.filter(e => !employeesWithProfiles.has(e.id)) || [];

  const totalBasicSalaries = profiles?.reduce((sum, p) => sum + p.basicSalary, 0) || 0;
  const totalAllowances = profiles?.reduce((sum, p) => sum + p.housingAllowance + p.transportAllowance + p.otherFixedAllowances, 0) || 0;
  const activeProfiles = profiles?.filter(p => p.isActive) || [];

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">ملفات الرواتب</h1>
          <p className="text-muted-foreground">تعريف الرواتب الأساسية والبدلات للموظفين</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-profile">
          <Plus className="h-4 w-4 ml-2" />
          إضافة ملف راتب جديد
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <Wallet className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">ملفات الرواتب</p>
                <p className="text-xl font-bold" data-testid="text-total-profiles">
                  {profiles?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <CheckCircle2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">ملفات نشطة</p>
                <p className="text-xl font-bold" data-testid="text-active-profiles">
                  {activeProfiles.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200">
                <DollarSign className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الرواتب الأساسية</p>
                <p className="text-xl font-bold" data-testid="text-total-basic">
                  {formatCurrency(totalBasicSalaries)}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200">
                <Coins className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي البدلات</p>
                <p className="text-xl font-bold" data-testid="text-total-allowances">
                  {formatCurrency(totalAllowances)}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Wallet className="h-5 w-5" />
            قائمة ملفات الرواتب
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">الموظف</TableHead>
                  <TableHead className="text-right">الراتب الأساسي</TableHead>
                  <TableHead className="text-right">بدل السكن</TableHead>
                  <TableHead className="text-right">بدل المواصلات</TableHead>
                  <TableHead className="text-right">بدلات أخرى</TableHead>
                  <TableHead className="text-right">إجمالي الراتب</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-left">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {profiles?.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={8} className="text-center py-8 text-muted-foreground">
                      لا توجد ملفات رواتب مسجلة
                    </TableCell>
                  </TableRow>
                ) : (
                  profiles?.map((profile) => (
                    <TableRow key={profile.id} data-testid={`row-profile-${profile.id}`}>
                      <TableCell className="font-medium">
                        {getEmployeeName(profile.employee)}
                      </TableCell>
                      <TableCell>{formatCurrency(profile.basicSalary)}</TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1">
                          <Home className="h-3 w-3 text-muted-foreground" />
                          {formatCurrency(profile.housingAllowance)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1">
                          <Car className="h-3 w-3 text-muted-foreground" />
                          {formatCurrency(profile.transportAllowance)}
                        </div>
                      </TableCell>
                      <TableCell>{formatCurrency(profile.otherFixedAllowances)}</TableCell>
                      <TableCell className="font-bold">
                        {formatCurrency(getTotalSalary(profile))}
                      </TableCell>
                      <TableCell>
                        {profile.isActive ? (
                          <Badge className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                            نشط
                          </Badge>
                        ) : (
                          <Badge className="bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                            غير نشط
                          </Badge>
                        )}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleOpenDialog(profile)}
                            data-testid={`button-edit-profile-${profile.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => deleteMutation.mutate(profile.employeeId)}
                            disabled={deleteMutation.isPending}
                            data-testid={`button-delete-profile-${profile.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-lg" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingProfile ? "تعديل ملف الراتب" : "إضافة ملف راتب جديد"}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="employeeId">الموظف <span className="text-destructive">*</span></Label>
              <Select
                value={formData.employeeId?.toString() || "_none"}
                onValueChange={(value) => setFormData({ ...formData, employeeId: value === "_none" ? null : parseInt(value) })}
                disabled={!!editingProfile}
              >
                <SelectTrigger data-testid="select-profile-employee">
                  <SelectValue placeholder="اختر الموظف" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="_none">لم يحدد</SelectItem>
                  {editingProfile ? (
                    <SelectItem value={editingProfile.employeeId.toString()}>
                      {getEmployeeName(editingProfile.employee)}
                    </SelectItem>
                  ) : (
                    employeesWithoutProfiles.map((emp) => (
                      <SelectItem key={emp.id} value={emp.id.toString()}>
                        {emp.fullNameAr || emp.nameAr}
                      </SelectItem>
                    ))
                  )}
                </SelectContent>
              </Select>
              {!editingProfile && employeesWithoutProfiles.length === 0 && (
                <p className="text-xs text-muted-foreground">جميع الموظفين لديهم ملفات رواتب</p>
              )}
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="basicSalary">الراتب الأساسي</Label>
                <Input
                  id="basicSalary"
                  type="number"
                  value={formData.basicSalary}
                  onChange={(e) => setFormData({ ...formData, basicSalary: parseFloat(e.target.value) || 0 })}
                  min={0}
                  step={100}
                  data-testid="input-basic-salary"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="housingAllowance">بدل السكن</Label>
                <Input
                  id="housingAllowance"
                  type="number"
                  value={formData.housingAllowance}
                  onChange={(e) => setFormData({ ...formData, housingAllowance: parseFloat(e.target.value) || 0 })}
                  min={0}
                  step={100}
                  data-testid="input-housing-allowance"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="transportAllowance">بدل المواصلات</Label>
                <Input
                  id="transportAllowance"
                  type="number"
                  value={formData.transportAllowance}
                  onChange={(e) => setFormData({ ...formData, transportAllowance: parseFloat(e.target.value) || 0 })}
                  min={0}
                  step={100}
                  data-testid="input-transport-allowance"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="otherFixedAllowances">بدلات أخرى</Label>
                <Input
                  id="otherFixedAllowances"
                  type="number"
                  value={formData.otherFixedAllowances}
                  onChange={(e) => setFormData({ ...formData, otherFixedAllowances: parseFloat(e.target.value) || 0 })}
                  min={0}
                  step={100}
                  data-testid="input-other-allowances"
                />
              </div>
            </div>
            <div className="p-4 bg-muted rounded-lg">
              <div className="flex items-center justify-between">
                <span className="font-medium">إجمالي الراتب الشهري:</span>
                <span className="text-xl font-bold">
                  {formatCurrency(
                    formData.basicSalary +
                    formData.housingAllowance +
                    formData.transportAllowance +
                    formData.otherFixedAllowances
                  )}
                </span>
              </div>
            </div>
            <div className="flex items-center justify-between">
              <Label htmlFor="isActive">الحالة</Label>
              <div className="flex items-center gap-2">
                <span className="text-sm text-muted-foreground">
                  {formData.isActive ? "نشط" : "غير نشط"}
                </span>
                <Switch
                  id="isActive"
                  checked={formData.isActive}
                  onCheckedChange={(checked) => setFormData({ ...formData, isActive: checked })}
                  data-testid="switch-profile-active"
                />
              </div>
            </div>
          </div>
          <DialogFooter className="gap-2 sm:gap-0">
            <Button
              variant="outline"
              onClick={resetForm}
              data-testid="button-cancel-profile"
            >
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={saveMutation.isPending}
              data-testid="button-save-profile"
            >
              {saveMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
