import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber } from "@/lib/formatters";
import { 
  Scissors, 
  Plus, 
  Pencil, 
  Trash2, 
  DollarSign, 
  Percent,
  Package,
  Filter,
  CheckCircle2,
  XCircle,
  ImagePlus,
  X,
  Loader2
} from "lucide-react";

interface ServiceCategory {
  id: number;
  nameAr: string;
  code: string | null;
  isActive: boolean;
}

interface RelatedProduct {
  productId: number;
  productNameAr?: string;
  quantityPerService: number;
  deductFromCommission: boolean;
  notes?: string;
}

interface ServiceImage {
  id: number;
  serviceId: number;
  imageUrl: string;
  filename: string;
  displayOrder: number;
}

interface Service {
  id: number;
  nameAr: string;
  code: string | null;
  descriptionAr: string | null;
  price: number;
  vatType: string;
  vatRate: number;
  isActive: boolean;
  categoryId: number;
  categoryNameAr?: string;
  relatedProducts?: RelatedProduct[];
  images?: ServiceImage[];
}

interface Product {
  id: number;
  nameAr: string;
  isActive: boolean;
  usageType: string; // "sale", "consume", "sale_and_consume"
}

interface ServiceFormData {
  categoryId: number | "";
  nameAr: string;
  code: string;
  descriptionAr: string;
  price: string;
  vatType: string;
  vatRate: string;
  isActive: boolean;
}

const vatTypeLabels: Record<string, string> = {
  inclusive: "شامل الضريبة",
  exclusive: "غير شامل الضريبة",
  exempt: "معفي من الضريبة",
};

export default function ServicesPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingService, setEditingService] = useState<Service | null>(null);
  const [filterCategory, setFilterCategory] = useState<string>("all");
  const [filterActive, setFilterActive] = useState<string>("all");
  const [formData, setFormData] = useState<ServiceFormData>({
    categoryId: "",
    nameAr: "",
    code: "",
    descriptionAr: "",
    price: "",
    vatType: "inclusive",
    vatRate: "0.15",
    isActive: true,
  });
  const [enableRelatedProducts, setEnableRelatedProducts] = useState(false);
  const [relatedProducts, setRelatedProducts] = useState<RelatedProduct[]>([]);
  const [serviceImages, setServiceImages] = useState<ServiceImage[]>([]);
  const [pendingImageFiles, setPendingImageFiles] = useState<File[]>([]);
  const [isUploadingImages, setIsUploadingImages] = useState(false);
  const { toast } = useToast();

  const { data: services, isLoading } = useQuery<Service[]>({
    queryKey: ["/api/services"],
  });

  const { data: categories } = useQuery<ServiceCategory[]>({
    queryKey: ["/api/service-categories"],
  });

  const { data: products } = useQuery<Product[]>({
    queryKey: ["/api/products"],
  });

  const createMutation = useMutation({
    mutationFn: (data: any) =>
      apiRequest("POST", "/api/services", data),
    onSuccess: async (response: any) => {
      // Upload pending images to the newly created service
      if (pendingImageFiles.length > 0 && response?.id) {
        await uploadImages(response.id);
      }
      queryClient.invalidateQueries({ queryKey: ["/api/services"] });
      queryClient.invalidateQueries({ queryKey: ["/api/service-categories"] });
      toast({ title: "تم إنشاء الخدمة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء الخدمة",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: any }) =>
      apiRequest("PUT", `/api/services/${id}`, data),
    onSuccess: async () => {
      // Upload pending images if any
      if (pendingImageFiles.length > 0 && editingService?.id) {
        await uploadImages(editingService.id);
      }
      queryClient.invalidateQueries({ queryKey: ["/api/services"] });
      queryClient.invalidateQueries({ queryKey: ["/api/service-categories"] });
      toast({ title: "تم تحديث الخدمة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث الخدمة",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/services/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/services"] });
      queryClient.invalidateQueries({ queryKey: ["/api/service-categories"] });
      toast({ title: "تم تعطيل الخدمة بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف الخدمة",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      categoryId: "",
      nameAr: "",
      code: "",
      descriptionAr: "",
      price: "",
      vatType: "inclusive",
      vatRate: "0.15",
      isActive: true,
    });
    setEnableRelatedProducts(false);
    setRelatedProducts([]);
    setServiceImages([]);
    setPendingImageFiles([]);
    setEditingService(null);
    setIsDialogOpen(false);
  };

  const handleOpenDialog = (service?: Service) => {
    if (service) {
      setEditingService(service);
      setFormData({
        categoryId: service.categoryId,
        nameAr: service.nameAr,
        code: service.code || "",
        descriptionAr: service.descriptionAr || "",
        price: service.price.toString(),
        vatType: service.vatType,
        vatRate: service.vatRate.toString(),
        isActive: service.isActive,
      });
      // Load related products if they exist
      if (service.relatedProducts && service.relatedProducts.length > 0) {
        setEnableRelatedProducts(true);
        setRelatedProducts(service.relatedProducts.map(rp => ({
          productId: rp.productId,
          productNameAr: rp.productNameAr,
          quantityPerService: rp.quantityPerService,
          deductFromCommission: rp.deductFromCommission || false,
          notes: rp.notes,
        })));
      } else {
        setEnableRelatedProducts(false);
        setRelatedProducts([]);
      }
      // Load existing images
      setServiceImages(service.images || []);
      setPendingImageFiles([]);
    } else {
      resetForm();
    }
    setIsDialogOpen(true);
  };

  const handleSubmit = () => {
    if (!formData.nameAr.trim()) {
      toast({ title: "اسم الخدمة مطلوب", variant: "destructive" });
      return;
    }
    if (!formData.price || parseFloat(formData.price) <= 0) {
      toast({ title: "سعر الخدمة مطلوب", variant: "destructive" });
      return;
    }
    if (!formData.categoryId) {
      toast({ title: "فئة الخدمة مطلوبة", variant: "destructive" });
      return;
    }

    const submitData = {
      categoryId: Number(formData.categoryId),
      nameAr: formData.nameAr.trim(),
      code: formData.code.trim() || null,
      descriptionAr: formData.descriptionAr.trim() || null,
      price: parseFloat(formData.price),
      vatType: formData.vatType,
      vatRate: parseFloat(formData.vatRate),
      isActive: formData.isActive,
      relatedProducts: enableRelatedProducts ? relatedProducts.map(rp => ({
        productId: rp.productId,
        quantityPerService: rp.quantityPerService,
        deductFromCommission: rp.deductFromCommission,
        notes: rp.notes || null,
      })) : [],
    };

    if (editingService) {
      updateMutation.mutate({ id: editingService.id, data: submitData });
    } else {
      createMutation.mutate(submitData);
    }
  };

  // Filter products that can be used for consumption (usageType = consume or sale_and_consume)
  const consumableProducts = products?.filter(p => 
    p.isActive && (p.usageType === "consume" || p.usageType === "sale_and_consume")
  ) || [];

  const addRelatedProduct = () => {
    setRelatedProducts([...relatedProducts, {
      productId: 0,
      quantityPerService: 1,
      deductFromCommission: false,
      notes: "",
    }]);
  };

  const removeRelatedProduct = (index: number) => {
    setRelatedProducts(relatedProducts.filter((_, i) => i !== index));
  };

  const updateRelatedProduct = (index: number, field: keyof RelatedProduct, value: any) => {
    const updated = [...relatedProducts];
    updated[index] = { ...updated[index], [field]: value };
    setRelatedProducts(updated);
  };

  // Image handling functions
  const handleImageSelect = (event: React.ChangeEvent<HTMLInputElement>) => {
    const files = event.target.files;
    if (!files) return;
    
    const validFiles: File[] = [];
    const allowedTypes = ["image/jpeg", "image/png", "image/gif", "image/webp"];
    
    for (let i = 0; i < files.length; i++) {
      const file = files[i];
      if (!allowedTypes.includes(file.type)) {
        toast({
          title: "خطأ في الملف",
          description: `الملف "${file.name}" غير مدعوم. الأنواع المسموحة: JPEG, PNG, GIF, WebP`,
          variant: "destructive",
        });
        continue;
      }
      if (file.size > 5 * 1024 * 1024) {
        toast({
          title: "خطأ في الملف",
          description: `الملف "${file.name}" يتجاوز الحد الأقصى (5 ميجابايت)`,
          variant: "destructive",
        });
        continue;
      }
      validFiles.push(file);
    }
    
    if (validFiles.length > 0) {
      setPendingImageFiles([...pendingImageFiles, ...validFiles]);
    }
    
    // Reset input
    event.target.value = "";
  };

  const removePendingImage = (index: number) => {
    setPendingImageFiles(pendingImageFiles.filter((_, i) => i !== index));
  };

  const uploadImages = async (serviceId: number): Promise<boolean> => {
    if (pendingImageFiles.length === 0) return true;
    
    setIsUploadingImages(true);
    try {
      const formData = new FormData();
      pendingImageFiles.forEach((file) => {
        formData.append("images", file);
      });
      
      const response = await fetch(`/api/services/${serviceId}/images`, {
        method: "POST",
        body: formData,
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || "فشل في رفع الصور");
      }
      
      return true;
    } catch (error: any) {
      toast({
        title: "خطأ في رفع الصور",
        description: error.message || "فشل في رفع الصور",
        variant: "destructive",
      });
      return false;
    } finally {
      setIsUploadingImages(false);
    }
  };

  const deleteServiceImage = async (imageId: number, serviceId: number) => {
    try {
      const response = await fetch(`/api/services/${serviceId}/images/${imageId}`, {
        method: "DELETE",
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || "فشل في حذف الصورة");
      }
      
      setServiceImages(serviceImages.filter(img => img.id !== imageId));
      toast({ title: "تم حذف الصورة بنجاح" });
    } catch (error: any) {
      toast({
        title: "خطأ",
        description: error.message || "فشل في حذف الصورة",
        variant: "destructive",
      });
    }
  };

  const filteredServices = services?.filter(service => {
    if (filterCategory !== "all" && service.categoryId !== parseInt(filterCategory)) {
      return false;
    }
    if (filterActive === "active" && !service.isActive) {
      return false;
    }
    if (filterActive === "inactive" && service.isActive) {
      return false;
    }
    return true;
  }) || [];

  const activeServices = services?.filter(s => s.isActive) || [];
  const inactiveServices = services?.filter(s => !s.isActive) || [];
  const totalRevenue = services?.reduce((sum, s) => sum + s.price, 0) || 0;

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">الخدمات</h1>
          <p className="text-muted-foreground">إدارة خدمات الصالون</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-service">
          <Plus className="h-4 w-4 ml-2" />
          إضافة خدمة جديدة
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <Scissors className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الخدمات</p>
                <p className="text-xl font-bold" data-testid="text-total-services">
                  {services?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <CheckCircle2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">خدمات مفعّلة</p>
                <p className="text-xl font-bold" data-testid="text-active-services">
                  {activeServices.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                <XCircle className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">خدمات غير مفعّلة</p>
                <p className="text-xl font-bold" data-testid="text-inactive-services">
                  {inactiveServices.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200">
                <DollarSign className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">متوسط السعر</p>
                <p className="text-xl font-bold" data-testid="text-avg-price">
                  {services && services.length > 0 
                    ? formatNumber(totalRevenue / services.length) 
                    : "0.00"} ر.س
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between gap-4 flex-wrap">
            <div className="flex items-center gap-2">
              <Scissors className="h-5 w-5" />
              قائمة الخدمات
            </div>
            <div className="flex items-center gap-4">
              <div className="flex items-center gap-2">
                <Filter className="h-4 w-4 text-muted-foreground" />
                <Select value={filterCategory} onValueChange={setFilterCategory}>
                  <SelectTrigger className="w-40" data-testid="select-filter-category">
                    <SelectValue placeholder="جميع الفئات" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">جميع الفئات</SelectItem>
                    {categories?.filter(c => c.isActive).map(cat => (
                      <SelectItem key={cat.id} value={cat.id.toString()}>
                        {cat.nameAr}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <Select value={filterActive} onValueChange={setFilterActive}>
                <SelectTrigger className="w-32" data-testid="select-filter-active">
                  <SelectValue placeholder="الحالة" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">الكل</SelectItem>
                  <SelectItem value="active">مفعّل</SelectItem>
                  <SelectItem value="inactive">غير مفعّل</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : filteredServices.length > 0 ? (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">اسم الخدمة</TableHead>
                  <TableHead className="text-right">الرمز</TableHead>
                  <TableHead className="text-right">الفئة</TableHead>
                  <TableHead className="text-left">السعر</TableHead>
                  <TableHead className="text-right">الضريبة</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-left">إجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {filteredServices.map((service) => (
                  <TableRow key={service.id} data-testid={`row-service-${service.id}`}>
                    <TableCell className="font-medium">
                      <div>
                        {service.nameAr}
                        {service.descriptionAr && (
                          <p className="text-xs text-muted-foreground truncate max-w-xs">
                            {service.descriptionAr}
                          </p>
                        )}
                      </div>
                    </TableCell>
                    <TableCell>
                      {service.code ? (
                        <Badge variant="outline">{service.code}</Badge>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </TableCell>
                    <TableCell>
                      <Badge variant="secondary">{service.categoryNameAr}</Badge>
                    </TableCell>
                    <TableCell className="text-left font-mono">
                      {formatCurrency(service.price)}
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-1">
                        <Percent className="h-3 w-3 text-muted-foreground" />
                        <span className="text-sm">
                          {(service.vatRate * 100).toFixed(0)}% {vatTypeLabels[service.vatType]?.split(" ")[0]}
                        </span>
                      </div>
                    </TableCell>
                    <TableCell>
                      {service.isActive ? (
                        <Badge className="bg-green-100 text-green-800">مفعّل</Badge>
                      ) : (
                        <Badge variant="destructive">غير مفعّل</Badge>
                      )}
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        <Button
                          size="icon"
                          variant="ghost"
                          onClick={() => handleOpenDialog(service)}
                          data-testid={`button-edit-service-${service.id}`}
                        >
                          <Pencil className="h-4 w-4" />
                        </Button>
                        <Button
                          size="icon"
                          variant="ghost"
                          onClick={() => deleteMutation.mutate(service.id)}
                          disabled={deleteMutation.isPending}
                          data-testid={`button-delete-service-${service.id}`}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          ) : (
            <div className="text-center py-8 text-muted-foreground">
              <Scissors className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p>لا توجد خدمات {filterCategory !== "all" || filterActive !== "all" ? "تطابق الفلترة" : "بعد"}</p>
              <p className="text-sm">اضغط على "إضافة خدمة جديدة" لإنشاء خدمة</p>
            </div>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="max-w-lg">
          <DialogHeader>
            <DialogTitle>
              {editingService ? "تعديل الخدمة" : "إضافة خدمة جديدة"}
            </DialogTitle>
          </DialogHeader>

          <div className="space-y-4 max-h-[60vh] overflow-y-auto">
            <div className="space-y-2">
              <Label htmlFor="categoryId">الفئة *</Label>
              <Select
                value={formData.categoryId.toString()}
                onValueChange={(value) => setFormData({ ...formData, categoryId: parseInt(value) })}
              >
                <SelectTrigger data-testid="select-service-category">
                  <SelectValue placeholder="اختر الفئة" />
                </SelectTrigger>
                <SelectContent>
                  {categories?.filter(c => c.isActive).map(cat => (
                    <SelectItem key={cat.id} value={cat.id.toString()}>
                      {cat.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="nameAr">اسم الخدمة *</Label>
              <Input
                id="nameAr"
                value={formData.nameAr}
                onChange={(e) => setFormData({ ...formData, nameAr: e.target.value })}
                placeholder="مثال: قص شعر سيدات"
                data-testid="input-service-name"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="code">رمز الخدمة</Label>
              <Input
                id="code"
                value={formData.code}
                onChange={(e) => setFormData({ ...formData, code: e.target.value.toUpperCase() })}
                placeholder="مثال: HR-CUT-01"
                data-testid="input-service-code"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="descriptionAr">وصف الخدمة</Label>
              <Textarea
                id="descriptionAr"
                value={formData.descriptionAr}
                onChange={(e) => setFormData({ ...formData, descriptionAr: e.target.value })}
                placeholder="وصف تفصيلي للخدمة..."
                rows={2}
                data-testid="input-service-description"
              />
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="price">السعر *</Label>
                <Input
                  id="price"
                  type="number"
                  step="0.01"
                  min="0"
                  value={formData.price}
                  onChange={(e) => setFormData({ ...formData, price: e.target.value })}
                  placeholder="0.00"
                  data-testid="input-service-price"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="vatRate">نسبة الضريبة</Label>
                <Input
                  id="vatRate"
                  type="number"
                  step="0.01"
                  min="0"
                  max="1"
                  value={formData.vatRate}
                  onChange={(e) => setFormData({ ...formData, vatRate: e.target.value })}
                  placeholder="0.15"
                  data-testid="input-service-vat-rate"
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label>نوع الضريبة</Label>
              <Select
                value={formData.vatType}
                onValueChange={(value) => setFormData({ ...formData, vatType: value })}
              >
                <SelectTrigger data-testid="select-service-vat-type">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="inclusive">شامل الضريبة</SelectItem>
                  <SelectItem value="exclusive">غير شامل الضريبة</SelectItem>
                  <SelectItem value="exempt">معفي من الضريبة</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="flex items-center justify-between">
              <Label htmlFor="isActive">حالة التفعيل</Label>
              <Switch
                id="isActive"
                checked={formData.isActive}
                onCheckedChange={(checked) => setFormData({ ...formData, isActive: checked })}
                data-testid="switch-service-active"
              />
            </div>

            {/* Related Products Section */}
            <div className="border rounded-lg p-4 space-y-4">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2">
                  <Package className="h-4 w-4 text-muted-foreground" />
                  <Label htmlFor="enableRelatedProducts">منتجات ذات صلة</Label>
                </div>
                <Switch
                  id="enableRelatedProducts"
                  checked={enableRelatedProducts}
                  onCheckedChange={setEnableRelatedProducts}
                  data-testid="switch-enable-related-products"
                />
              </div>

              {enableRelatedProducts && (
                <div className="space-y-3">
                  <p className="text-sm text-muted-foreground">
                    يتم اختيار منتجات من نوع استهلاك أو صرف لخصمها من المخزون عند تنفيذ الخدمة
                  </p>

                  {relatedProducts.map((rp, index) => (
                    <div key={index} className="border rounded-lg p-3 space-y-3 bg-muted/30">
                      <div className="flex items-center justify-between">
                        <span className="text-sm font-medium">منتج #{index + 1}</span>
                        <Button
                          type="button"
                          size="icon"
                          variant="ghost"
                          onClick={() => removeRelatedProduct(index)}
                          data-testid={`button-remove-related-product-${index}`}
                        >
                          <Trash2 className="h-4 w-4 text-destructive" />
                        </Button>
                      </div>

                      <div className="grid grid-cols-2 gap-3">
                        <div className="col-span-2">
                          <Label>المنتج</Label>
                          <Select
                            value={rp.productId ? rp.productId.toString() : ""}
                            onValueChange={(value) => updateRelatedProduct(index, "productId", parseInt(value))}
                          >
                            <SelectTrigger data-testid={`select-related-product-${index}`}>
                              <SelectValue placeholder="اختر منتج" />
                            </SelectTrigger>
                            <SelectContent>
                              {consumableProducts.map(p => (
                                <SelectItem key={p.id} value={p.id.toString()}>
                                  {p.nameAr}
                                </SelectItem>
                              ))}
                            </SelectContent>
                          </Select>
                        </div>

                        <div>
                          <Label>الكمية لكل خدمة</Label>
                          <Input
                            type="number"
                            step="0.01"
                            min="0.01"
                            value={rp.quantityPerService}
                            onChange={(e) => updateRelatedProduct(index, "quantityPerService", parseFloat(e.target.value) || 1)}
                            data-testid={`input-related-product-qty-${index}`}
                          />
                        </div>

                        <div className="flex items-end">
                          <div className="flex items-center gap-2 pb-2">
                            <Switch
                              checked={rp.deductFromCommission}
                              onCheckedChange={(checked) => updateRelatedProduct(index, "deductFromCommission", checked)}
                              data-testid={`switch-deduct-commission-${index}`}
                            />
                            <Label className="text-sm">خصم من العمولة</Label>
                          </div>
                        </div>
                      </div>
                    </div>
                  ))}

                  <Button
                    type="button"
                    variant="outline"
                    size="sm"
                    onClick={addRelatedProduct}
                    className="w-full"
                    data-testid="button-add-related-product"
                  >
                    <Plus className="h-4 w-4 ml-2" />
                    إضافة منتج
                  </Button>
                </div>
              )}
            </div>

            {/* Images Section */}
            <div className="border rounded-lg p-4 space-y-4">
              <div className="flex items-center gap-2">
                <ImagePlus className="h-4 w-4 text-muted-foreground" />
                <Label>صور الخدمة</Label>
              </div>

              <p className="text-sm text-muted-foreground">
                رفع صور متعددة للخدمة (JPEG, PNG, GIF, WebP - الحد الأقصى 5 ميجابايت لكل صورة)
              </p>

              {/* Existing Images */}
              {serviceImages.length > 0 && (
                <div className="grid grid-cols-3 gap-3">
                  {serviceImages.map((image) => (
                    <div key={image.id} className="relative group">
                      <img
                        src={image.imageUrl}
                        alt={image.filename}
                        className="w-full h-24 object-cover rounded-lg border"
                        data-testid={`img-service-image-${image.id}`}
                      />
                      <Button
                        type="button"
                        size="icon"
                        variant="destructive"
                        className="absolute top-1 left-1 h-6 w-6 opacity-0 group-hover:opacity-100 transition-opacity"
                        onClick={() => editingService && deleteServiceImage(image.id, editingService.id)}
                        data-testid={`button-delete-image-${image.id}`}
                      >
                        <X className="h-3 w-3" />
                      </Button>
                      <p className="text-xs text-center text-muted-foreground truncate mt-1">
                        {image.filename}
                      </p>
                    </div>
                  ))}
                </div>
              )}

              {/* Pending Images (not yet uploaded) */}
              {pendingImageFiles.length > 0 && (
                <div className="grid grid-cols-3 gap-3">
                  {pendingImageFiles.map((file, index) => (
                    <div key={index} className="relative group">
                      <img
                        src={URL.createObjectURL(file)}
                        alt={file.name}
                        className="w-full h-24 object-cover rounded-lg border border-dashed border-primary"
                        data-testid={`img-pending-image-${index}`}
                      />
                      <Button
                        type="button"
                        size="icon"
                        variant="destructive"
                        className="absolute top-1 left-1 h-6 w-6 opacity-0 group-hover:opacity-100 transition-opacity"
                        onClick={() => removePendingImage(index)}
                        data-testid={`button-remove-pending-image-${index}`}
                      >
                        <X className="h-3 w-3" />
                      </Button>
                      <p className="text-xs text-center text-muted-foreground truncate mt-1">
                        {file.name} (جديدة)
                      </p>
                    </div>
                  ))}
                </div>
              )}

              {/* Upload Button */}
              <div>
                <input
                  type="file"
                  accept="image/jpeg,image/png,image/gif,image/webp"
                  multiple
                  onChange={handleImageSelect}
                  className="hidden"
                  id="service-images-input"
                  data-testid="input-service-images"
                />
                <label htmlFor="service-images-input">
                  <Button
                    type="button"
                    variant="outline"
                    size="sm"
                    className="w-full cursor-pointer"
                    asChild
                  >
                    <span>
                      <ImagePlus className="h-4 w-4 ml-2" />
                      اختيار صور من الجهاز
                    </span>
                  </Button>
                </label>
              </div>

              {(serviceImages.length + pendingImageFiles.length) > 0 && (
                <p className="text-xs text-muted-foreground text-center">
                  {serviceImages.length} صورة محفوظة • {pendingImageFiles.length} صورة جديدة للرفع
                </p>
              )}
            </div>
          </div>

          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={resetForm}>
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending || isUploadingImages}
              data-testid="button-save-service"
            >
              {createMutation.isPending || updateMutation.isPending || isUploadingImages ? (
                <>
                  <Loader2 className="h-4 w-4 ml-2 animate-spin" />
                  {isUploadingImages ? "جاري رفع الصور..." : "جاري الحفظ..."}
                </>
              ) : (
                "حفظ"
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
