import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Switch } from "@/components/ui/switch";
import { Skeleton } from "@/components/ui/skeleton";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import {
  Settings as SettingsIcon,
  CreditCard,
  Plus,
  Pencil,
  Trash2,
  Save,
  Loader2,
  Banknote,
  Building2,
} from "lucide-react";
import type { PaymentMethod } from "@shared/schema";

const paymentMethodSchema = z.object({
  nameAr: z.string().min(1, "اسم طريقة الدفع مطلوب"),
  type: z.enum(["cash", "bank", "card"]),
  isActive: z.boolean().default(true),
});

type PaymentMethodFormValues = z.infer<typeof paymentMethodSchema>;

function PaymentMethodDialog({
  open,
  onOpenChange,
  method,
  onSuccess,
}: {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  method?: PaymentMethod;
  onSuccess: () => void;
}) {
  const { toast } = useToast();
  const isEditing = !!method;

  const form = useForm<PaymentMethodFormValues>({
    resolver: zodResolver(paymentMethodSchema),
    defaultValues: {
      nameAr: method?.nameAr || "",
      type: (method?.type as "cash" | "bank" | "card") || "cash",
      isActive: method?.isActive ?? true,
    },
  });

  const createMutation = useMutation({
    mutationFn: async (data: PaymentMethodFormValues) => {
      const response = await apiRequest("POST", "/api/payment-methods", data);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/payment-methods"] });
      toast({
        title: "تم الإنشاء بنجاح",
        description: "تم إضافة طريقة الدفع",
      });
      form.reset();
      onSuccess();
    },
    onError: () => {
      toast({
        title: "خطأ",
        description: "حدث خطأ أثناء إنشاء طريقة الدفع",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async (data: PaymentMethodFormValues) => {
      const response = await apiRequest("PUT", `/api/payment-methods/${method?.id}`, data);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/payment-methods"] });
      toast({
        title: "تم التحديث بنجاح",
        description: "تم تحديث طريقة الدفع",
      });
      onSuccess();
    },
    onError: () => {
      toast({
        title: "خطأ",
        description: "حدث خطأ أثناء تحديث طريقة الدفع",
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: PaymentMethodFormValues) => {
    if (isEditing) {
      updateMutation.mutate(data);
    } else {
      createMutation.mutate(data);
    }
  };

  const isPending = createMutation.isPending || updateMutation.isPending;

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent>
        <DialogHeader>
          <DialogTitle>{isEditing ? "تعديل طريقة الدفع" : "إضافة طريقة دفع"}</DialogTitle>
        </DialogHeader>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
            <FormField
              control={form.control}
              name="nameAr"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>الاسم *</FormLabel>
                  <FormControl>
                    <Input placeholder="مثال: نقدي، بنك الراجحي" {...field} data-testid="input-method-name" />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="type"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>النوع *</FormLabel>
                  <Select onValueChange={field.onChange} value={field.value}>
                    <FormControl>
                      <SelectTrigger data-testid="select-method-type">
                        <SelectValue />
                      </SelectTrigger>
                    </FormControl>
                    <SelectContent>
                      <SelectItem value="cash">
                        <div className="flex items-center gap-2">
                          <Banknote className="h-4 w-4 text-green-600" />
                          نقدي
                        </div>
                      </SelectItem>
                      <SelectItem value="bank">
                        <div className="flex items-center gap-2">
                          <Building2 className="h-4 w-4 text-blue-600" />
                          تحويل بنكي
                        </div>
                      </SelectItem>
                      <SelectItem value="card">
                        <div className="flex items-center gap-2">
                          <CreditCard className="h-4 w-4 text-purple-600" />
                          بطاقة ائتمان
                        </div>
                      </SelectItem>
                    </SelectContent>
                  </Select>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="isActive"
              render={({ field }) => (
                <FormItem className="flex items-center justify-between rounded-lg border p-4">
                  <div>
                    <FormLabel>نشط</FormLabel>
                  </div>
                  <FormControl>
                    <Switch
                      checked={field.value}
                      onCheckedChange={field.onChange}
                      data-testid="switch-method-active"
                    />
                  </FormControl>
                </FormItem>
              )}
            />
            <div className="flex justify-end gap-2">
              <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>
                إلغاء
              </Button>
              <Button type="submit" disabled={isPending} data-testid="button-save-method">
                {isPending ? (
                  <Loader2 className="h-4 w-4 animate-spin ml-2" />
                ) : (
                  <Save className="h-4 w-4 ml-2" />
                )}
                {isEditing ? "حفظ" : "إضافة"}
              </Button>
            </div>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  );
}

export default function SettingsPage() {
  const { toast } = useToast();
  const [dialogOpen, setDialogOpen] = useState(false);
  const [editingMethod, setEditingMethod] = useState<PaymentMethod | undefined>();

  const { data: paymentMethods, isLoading } = useQuery<PaymentMethod[]>({
    queryKey: ["/api/payment-methods"],
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      await apiRequest("DELETE", `/api/payment-methods/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/payment-methods"] });
      toast({
        title: "تم الحذف بنجاح",
        description: "تم حذف طريقة الدفع",
      });
    },
    onError: () => {
      toast({
        title: "خطأ",
        description: "لا يمكن حذف طريقة الدفع لأنها مستخدمة في مدفوعات",
        variant: "destructive",
      });
    },
  });

  const handleEdit = (method: PaymentMethod) => {
    setEditingMethod(method);
    setDialogOpen(true);
  };

  const handleDialogClose = () => {
    setDialogOpen(false);
    setEditingMethod(undefined);
  };

  const getTypeIcon = (type: string) => {
    switch (type) {
      case "cash":
        return <Banknote className="h-4 w-4 text-green-600" />;
      case "bank":
        return <Building2 className="h-4 w-4 text-blue-600" />;
      case "card":
        return <CreditCard className="h-4 w-4 text-purple-600" />;
      default:
        return <CreditCard className="h-4 w-4" />;
    }
  };

  const getTypeLabel = (type: string) => {
    switch (type) {
      case "cash":
        return "نقدي";
      case "bank":
        return "تحويل بنكي";
      case "card":
        return "بطاقة ائتمان";
      default:
        return type;
    }
  };

  return (
    <div className="space-y-6">
      <div>
        <h1 className="text-2xl font-bold" data-testid="page-title">الإعدادات</h1>
        <p className="text-muted-foreground">إدارة إعدادات النظام وطرق الدفع</p>
      </div>

      <div className="grid gap-6 lg:grid-cols-2">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2">
            <div>
              <CardTitle className="flex items-center gap-2">
                <CreditCard className="h-5 w-5" />
                طرق الدفع
              </CardTitle>
              <CardDescription>إدارة طرق الدفع المتاحة للمدفوعات</CardDescription>
            </div>
            <Button size="sm" onClick={() => setDialogOpen(true)} data-testid="button-add-payment-method">
              <Plus className="h-4 w-4 ml-2" />
              إضافة
            </Button>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="space-y-2">
                {[1, 2, 3].map((i) => (
                  <Skeleton key={i} className="h-12" />
                ))}
              </div>
            ) : !paymentMethods || paymentMethods.length === 0 ? (
              <div className="text-center text-muted-foreground py-8">
                <CreditCard className="h-12 w-12 mx-auto mb-4 opacity-50" />
                <p>لا توجد طرق دفع</p>
                <Button
                  variant="link"
                  onClick={() => setDialogOpen(true)}
                  className="mt-2"
                >
                  إضافة طريقة دفع
                </Button>
              </div>
            ) : (
              <div className="rounded-lg border">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">الاسم</TableHead>
                      <TableHead className="text-right">النوع</TableHead>
                      <TableHead className="text-right">الحالة</TableHead>
                      <TableHead className="w-24"></TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {paymentMethods.map((method) => (
                      <TableRow key={method.id} data-testid={`payment-method-row-${method.id}`}>
                        <TableCell className="font-medium">{method.nameAr}</TableCell>
                        <TableCell>
                          <div className="flex items-center gap-2">
                            {getTypeIcon(method.type)}
                            <span>{getTypeLabel(method.type)}</span>
                          </div>
                        </TableCell>
                        <TableCell>
                          <span
                            className={`inline-flex items-center rounded-full px-2 py-1 text-xs ${
                              method.isActive
                                ? "bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400"
                                : "bg-gray-100 text-gray-600 dark:bg-gray-800 dark:text-gray-400"
                            }`}
                          >
                            {method.isActive ? "نشط" : "غير نشط"}
                          </span>
                        </TableCell>
                        <TableCell>
                          <div className="flex items-center gap-1">
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => handleEdit(method)}
                              data-testid={`edit-method-${method.id}`}
                            >
                              <Pencil className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => deleteMutation.mutate(method.id)}
                              data-testid={`delete-method-${method.id}`}
                            >
                              <Trash2 className="h-4 w-4 text-destructive" />
                            </Button>
                          </div>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>
            )}
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <SettingsIcon className="h-5 w-5" />
              إعدادات الضريبة
            </CardTitle>
            <CardDescription>إعدادات ضريبة القيمة المضافة</CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="rounded-lg border p-4">
              <div className="flex items-center justify-between mb-4">
                <div>
                  <p className="font-medium">نسبة ضريبة القيمة المضافة</p>
                  <p className="text-sm text-muted-foreground">النسبة المطبقة على الفواتير</p>
                </div>
                <div className="text-2xl font-bold text-primary">15%</div>
              </div>
              <p className="text-xs text-muted-foreground">
                يتم تطبيق هذه النسبة تلقائياً على جميع الفواتير غير المعفاة من الضريبة
              </p>
            </div>

            <div className="rounded-lg border p-4">
              <p className="font-medium mb-2">أنواع الضريبة المدعومة</p>
              <div className="space-y-2 text-sm">
                <div className="flex items-center justify-between">
                  <span>شامل الضريبة</span>
                  <span className="text-muted-foreground">السعر يتضمن الضريبة</span>
                </div>
                <div className="flex items-center justify-between">
                  <span>غير شامل الضريبة</span>
                  <span className="text-muted-foreground">يضاف 15% على السعر</span>
                </div>
                <div className="flex items-center justify-between">
                  <span>معفي من الضريبة</span>
                  <span className="text-muted-foreground">بدون ضريبة</span>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="lg:col-span-2">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <SettingsIcon className="h-5 w-5" />
              معلومات النظام
            </CardTitle>
            <CardDescription>معلومات عامة عن نظام إدارة المشتريات</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid gap-4 sm:grid-cols-2 lg:grid-cols-4">
              <div className="rounded-lg border p-4">
                <p className="text-sm text-muted-foreground">إصدار النظام</p>
                <p className="text-lg font-bold">1.0.0</p>
              </div>
              <div className="rounded-lg border p-4">
                <p className="text-sm text-muted-foreground">قاعدة البيانات</p>
                <p className="text-lg font-bold">PostgreSQL</p>
              </div>
              <div className="rounded-lg border p-4">
                <p className="text-sm text-muted-foreground">اللغة</p>
                <p className="text-lg font-bold">العربية (RTL)</p>
              </div>
              <div className="rounded-lg border p-4">
                <p className="text-sm text-muted-foreground">العملة</p>
                <p className="text-lg font-bold">ريال سعودي (SAR)</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <PaymentMethodDialog
        open={dialogOpen}
        onOpenChange={handleDialogClose}
        method={editingMethod}
        onSuccess={handleDialogClose}
      />
    </div>
  );
}
