import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatDate } from "@/lib/formatters";
import { CalendarDays, Plus, Pencil, Trash2, Users, Timer, Calendar } from "lucide-react";
import { format } from "date-fns";
import { ar } from "date-fns/locale";

interface Employee {
  id: number;
  nameAr: string;
}

interface WorkShift {
  id: number;
  nameAr: string;
  startTime: string;
  endTime: string;
  isActive: boolean;
}

interface ShiftAssignment {
  id: number;
  employeeId: number;
  workShiftId: number;
  startDate: string;
  endDate: string;
  createdAt: string;
  employee?: Employee;
  workShift?: WorkShift;
}

interface ShiftAssignmentFormData {
  employeeId: number | null;
  workShiftId: number | null;
  startDate: string;
  endDate: string;
}

export default function ShiftAssignmentsPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingAssignment, setEditingAssignment] = useState<ShiftAssignment | null>(null);
  const [formData, setFormData] = useState<ShiftAssignmentFormData>({
    employeeId: null,
    workShiftId: null,
    startDate: "",
    endDate: "",
  });
  const { toast } = useToast();

  const { data: assignments, isLoading } = useQuery<ShiftAssignment[]>({
    queryKey: ["/api/hr/shift-assignments"],
  });

  const { data: employees } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const { data: shifts } = useQuery<WorkShift[]>({
    queryKey: ["/api/hr/work-shifts"],
  });

  const createMutation = useMutation({
    mutationFn: (data: ShiftAssignmentFormData) =>
      apiRequest("POST", "/api/hr/shift-assignments", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/shift-assignments"] });
      toast({ title: "تم تعيين المناوبة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تعيين المناوبة",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<ShiftAssignmentFormData> }) =>
      apiRequest("PUT", `/api/hr/shift-assignments/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/shift-assignments"] });
      toast({ title: "تم تحديث المناوبة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث المناوبة",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/hr/shift-assignments/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/shift-assignments"] });
      toast({ title: "تم حذف المناوبة بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف المناوبة",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      employeeId: null,
      workShiftId: null,
      startDate: "",
      endDate: "",
    });
    setEditingAssignment(null);
    setIsDialogOpen(false);
  };

  const handleOpenDialog = (assignment?: ShiftAssignment) => {
    if (assignment) {
      setEditingAssignment(assignment);
      setFormData({
        employeeId: assignment.employeeId,
        workShiftId: assignment.workShiftId,
        startDate: assignment.startDate,
        endDate: assignment.endDate,
      });
    } else {
      setFormData({
        employeeId: null,
        workShiftId: null,
        startDate: "",
        endDate: "",
      });
      setEditingAssignment(null);
    }
    setIsDialogOpen(true);
  };

  const handleSubmit = () => {
    if (!formData.employeeId) {
      toast({ title: "الموظف مطلوب", variant: "destructive" });
      return;
    }
    if (!formData.workShiftId) {
      toast({ title: "الوردية مطلوبة", variant: "destructive" });
      return;
    }
    if (!formData.startDate || !formData.endDate) {
      toast({ title: "تاريخ البداية والنهاية مطلوبان", variant: "destructive" });
      return;
    }
    if (formData.startDate > formData.endDate) {
      toast({ title: "تاريخ البداية يجب أن يكون قبل تاريخ النهاية", variant: "destructive" });
      return;
    }

    if (editingAssignment) {
      updateMutation.mutate({ id: editingAssignment.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const getEmployeeName = (id: number) => {
    return employees?.find(e => e.id === id)?.nameAr || `موظف #${id}`;
  };

  const getShiftName = (id: number) => {
    return shifts?.find(s => s.id === id)?.nameAr || `وردية #${id}`;
  };

  const uniqueEmployees = new Set(assignments?.map(a => a.employeeId) || []);

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">مناوبات الموظفين</h1>
          <p className="text-muted-foreground">تعيين الورديات للموظفين</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-shift-assignment">
          <Plus className="h-4 w-4 ml-2" />
          تعيين مناوبة جديدة
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <CalendarDays className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي المناوبات</p>
                <p className="text-xl font-bold" data-testid="text-total-assignments">
                  {assignments?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200">
                <Users className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">موظفين معينين</p>
                <p className="text-xl font-bold" data-testid="text-assigned-employees">
                  {uniqueEmployees.size}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <Timer className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">الورديات المستخدمة</p>
                <p className="text-xl font-bold" data-testid="text-used-shifts">
                  {new Set(assignments?.map(a => a.workShiftId) || []).size}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <CalendarDays className="h-5 w-5" />
            قائمة المناوبات
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">الموظف</TableHead>
                  <TableHead className="text-right">الوردية</TableHead>
                  <TableHead className="text-right">تاريخ البداية</TableHead>
                  <TableHead className="text-right">تاريخ النهاية</TableHead>
                  <TableHead className="text-left">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {assignments?.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={5} className="text-center py-8 text-muted-foreground">
                      لا توجد مناوبات مسجلة
                    </TableCell>
                  </TableRow>
                ) : (
                  assignments?.map((assignment) => (
                    <TableRow key={assignment.id} data-testid={`row-shift-assignment-${assignment.id}`}>
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <Users className="h-4 w-4 text-muted-foreground" />
                          {assignment.employee?.nameAr || getEmployeeName(assignment.employeeId)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant="outline">
                          <Timer className="h-3 w-3 ml-1" />
                          {assignment.workShift?.nameAr || getShiftName(assignment.workShiftId)}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1 text-sm">
                          <Calendar className="h-3 w-3 text-muted-foreground" />
                          {formatDate(assignment.startDate)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1 text-sm">
                          <Calendar className="h-3 w-3 text-muted-foreground" />
                          {formatDate(assignment.endDate)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleOpenDialog(assignment)}
                            data-testid={`button-edit-shift-assignment-${assignment.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => deleteMutation.mutate(assignment.id)}
                            disabled={deleteMutation.isPending}
                            data-testid={`button-delete-shift-assignment-${assignment.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-lg" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingAssignment ? "تعديل المناوبة" : "تعيين مناوبة جديدة"}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="employeeId">الموظف <span className="text-destructive">*</span></Label>
              <Select
                value={formData.employeeId?.toString() || ""}
                onValueChange={(value) => setFormData({ ...formData, employeeId: parseInt(value) })}
              >
                <SelectTrigger data-testid="select-assignment-employee">
                  <SelectValue placeholder="اختر الموظف" />
                </SelectTrigger>
                <SelectContent>
                  {employees?.map((employee) => (
                    <SelectItem key={employee.id} value={employee.id.toString()}>
                      {employee.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label htmlFor="workShiftId">الوردية <span className="text-destructive">*</span></Label>
              <Select
                value={formData.workShiftId?.toString() || ""}
                onValueChange={(value) => setFormData({ ...formData, workShiftId: parseInt(value) })}
              >
                <SelectTrigger data-testid="select-assignment-shift">
                  <SelectValue placeholder="اختر الوردية" />
                </SelectTrigger>
                <SelectContent>
                  {shifts?.filter(s => s.isActive !== false).map((shift) => (
                    <SelectItem key={shift.id} value={shift.id.toString()}>
                      {shift.nameAr} ({shift.startTime} - {shift.endTime})
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="startDate">تاريخ البداية <span className="text-destructive">*</span></Label>
                <Input
                  id="startDate"
                  type="date"
                  value={formData.startDate}
                  onChange={(e) => setFormData({ ...formData, startDate: e.target.value })}
                  data-testid="input-assignment-start-date"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="endDate">تاريخ النهاية <span className="text-destructive">*</span></Label>
                <Input
                  id="endDate"
                  type="date"
                  value={formData.endDate}
                  onChange={(e) => setFormData({ ...formData, endDate: e.target.value })}
                  data-testid="input-assignment-end-date"
                />
              </div>
            </div>
          </div>
          <DialogFooter className="gap-2 sm:gap-0">
            <Button
              variant="outline"
              onClick={resetForm}
              data-testid="button-cancel-shift-assignment"
            >
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-save-shift-assignment"
            >
              {createMutation.isPending || updateMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
