import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { UserCheck, Plus, Pencil, Trash2, Users, ArrowRight } from "lucide-react";

interface Employee {
  id: number;
  nameAr: string;
  role?: { nameAr: string } | null;
}

interface Supervisor {
  id: number;
  employeeId: number;
  supervisorId: number;
  createdAt: string;
  employee?: Employee;
  supervisor?: Employee;
}

interface SupervisorFormData {
  employeeId: number | null;
  supervisorId: number | null;
}

export default function SupervisorsPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingSupervisor, setEditingSupervisor] = useState<Supervisor | null>(null);
  const [formData, setFormData] = useState<SupervisorFormData>({
    employeeId: null,
    supervisorId: null,
  });
  const { toast } = useToast();

  const { data: supervisors, isLoading } = useQuery<Supervisor[]>({
    queryKey: ["/api/hr/supervisors"],
  });

  const { data: employees } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const createMutation = useMutation({
    mutationFn: (data: SupervisorFormData) =>
      apiRequest("POST", "/api/hr/supervisors", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/supervisors"] });
      toast({ title: "تم تعيين المشرف بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تعيين المشرف",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<SupervisorFormData> }) =>
      apiRequest("PUT", `/api/hr/supervisors/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/supervisors"] });
      toast({ title: "تم تحديث المشرف بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث المشرف",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/hr/supervisors/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/supervisors"] });
      toast({ title: "تم إزالة المشرف بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إزالة المشرف",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({ employeeId: null, supervisorId: null });
    setEditingSupervisor(null);
    setIsDialogOpen(false);
  };

  const handleOpenDialog = (supervisor?: Supervisor) => {
    if (supervisor) {
      setEditingSupervisor(supervisor);
      setFormData({
        employeeId: supervisor.employeeId,
        supervisorId: supervisor.supervisorId,
      });
    } else {
      setFormData({ employeeId: null, supervisorId: null });
      setEditingSupervisor(null);
    }
    setIsDialogOpen(true);
  };

  const handleSubmit = () => {
    if (!formData.employeeId) {
      toast({ title: "الموظف مطلوب", variant: "destructive" });
      return;
    }
    if (!formData.supervisorId) {
      toast({ title: "المشرف مطلوب", variant: "destructive" });
      return;
    }
    if (formData.employeeId === formData.supervisorId) {
      toast({ title: "لا يمكن للموظف أن يكون مشرفاً على نفسه", variant: "destructive" });
      return;
    }

    if (editingSupervisor) {
      updateMutation.mutate({ id: editingSupervisor.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const getEmployeeName = (id: number) => {
    return employees?.find(e => e.id === id)?.nameAr || `موظف #${id}`;
  };

  const uniqueSupervisors = new Set(supervisors?.map(s => s.supervisorId) || []);
  const uniqueEmployees = new Set(supervisors?.map(s => s.employeeId) || []);

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">إدارة المشرفين</h1>
          <p className="text-muted-foreground">تعيين المشرفين على الموظفين</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-supervisor">
          <Plus className="h-4 w-4 ml-2" />
          تعيين مشرف جديد
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <UserCheck className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي العلاقات</p>
                <p className="text-xl font-bold" data-testid="text-total-supervisors">
                  {supervisors?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200">
                <UserCheck className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">عدد المشرفين</p>
                <p className="text-xl font-bold" data-testid="text-unique-supervisors">
                  {uniqueSupervisors.size}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <Users className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">موظفين تحت إشراف</p>
                <p className="text-xl font-bold" data-testid="text-supervised-employees">
                  {uniqueEmployees.size}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <UserCheck className="h-5 w-5" />
            علاقات الإشراف
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">الموظف</TableHead>
                  <TableHead className="text-center">العلاقة</TableHead>
                  <TableHead className="text-right">المشرف</TableHead>
                  <TableHead className="text-left">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {supervisors?.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={4} className="text-center py-8 text-muted-foreground">
                      لا توجد علاقات إشراف مسجلة
                    </TableCell>
                  </TableRow>
                ) : (
                  supervisors?.map((supervisor) => (
                    <TableRow key={supervisor.id} data-testid={`row-supervisor-${supervisor.id}`}>
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <Users className="h-4 w-4 text-muted-foreground" />
                          {supervisor.employee?.nameAr || getEmployeeName(supervisor.employeeId)}
                        </div>
                      </TableCell>
                      <TableCell className="text-center">
                        <Badge variant="outline">
                          <ArrowRight className="h-3 w-3 mx-1" />
                          يشرف عليه
                        </Badge>
                      </TableCell>
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <UserCheck className="h-4 w-4 text-primary" />
                          {supervisor.supervisor?.nameAr || getEmployeeName(supervisor.supervisorId)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleOpenDialog(supervisor)}
                            data-testid={`button-edit-supervisor-${supervisor.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => deleteMutation.mutate(supervisor.id)}
                            disabled={deleteMutation.isPending}
                            data-testid={`button-delete-supervisor-${supervisor.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-md" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingSupervisor ? "تعديل علاقة الإشراف" : "تعيين مشرف جديد"}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="employeeId">الموظف <span className="text-destructive">*</span></Label>
              <Select
                value={formData.employeeId?.toString() || ""}
                onValueChange={(value) => setFormData({ ...formData, employeeId: parseInt(value) })}
              >
                <SelectTrigger data-testid="select-employee">
                  <SelectValue placeholder="اختر الموظف" />
                </SelectTrigger>
                <SelectContent>
                  {employees?.map((employee) => (
                    <SelectItem key={employee.id} value={employee.id.toString()}>
                      {employee.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label htmlFor="supervisorId">المشرف <span className="text-destructive">*</span></Label>
              <Select
                value={formData.supervisorId?.toString() || ""}
                onValueChange={(value) => setFormData({ ...formData, supervisorId: parseInt(value) })}
              >
                <SelectTrigger data-testid="select-supervisor">
                  <SelectValue placeholder="اختر المشرف" />
                </SelectTrigger>
                <SelectContent>
                  {employees?.filter(e => e.id !== formData.employeeId).map((employee) => (
                    <SelectItem key={employee.id} value={employee.id.toString()}>
                      {employee.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <p className="text-xs text-muted-foreground">
                المشرف المسؤول عن متابعة أداء الموظف
              </p>
            </div>
          </div>
          <DialogFooter className="gap-2 sm:gap-0">
            <Button
              variant="outline"
              onClick={resetForm}
              data-testid="button-cancel-supervisor"
            >
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-save-supervisor"
            >
              {createMutation.isPending || updateMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
