import { useQuery } from "@tanstack/react-query";
import { useLocation, useParams, Link } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Separator } from "@/components/ui/separator";
import {
  ArrowRight,
  Pencil,
  Building2,
  Phone,
  FileText,
  CreditCard,
  Wallet,
} from "lucide-react";
import type { SupplierWithBankAccounts, SupplierBalance, PurchaseInvoice } from "@shared/schema";

export default function SupplierDetailPage() {
  const [, navigate] = useLocation();
  const params = useParams<{ id: string }>();
  const supplierId = parseInt(params.id);

  const { data: supplier, isLoading } = useQuery<SupplierWithBankAccounts>({
    queryKey: ["/api/suppliers", supplierId],
  });

  const { data: balance } = useQuery<SupplierBalance>({
    queryKey: ["/api/suppliers", supplierId, "balance"],
  });

  const { data: invoices } = useQuery<PurchaseInvoice[]>({
    queryKey: ["/api/suppliers", supplierId, "invoices"],
  });

  if (isLoading) {
    return (
      <div className="space-y-6">
        <div className="flex items-center gap-4">
          <Skeleton className="h-10 w-10" />
          <div>
            <Skeleton className="h-8 w-48 mb-2" />
            <Skeleton className="h-4 w-32" />
          </div>
        </div>
        <div className="grid gap-6 md:grid-cols-2">
          <Card>
            <CardContent className="p-6">
              <Skeleton className="h-40" />
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-6">
              <Skeleton className="h-40" />
            </CardContent>
          </Card>
        </div>
      </div>
    );
  }

  if (!supplier) {
    return (
      <div className="flex flex-col items-center justify-center py-16 text-center">
        <Building2 className="h-16 w-16 text-muted-foreground mb-4" />
        <h2 className="text-xl font-semibold mb-2">المورد غير موجود</h2>
        <p className="text-muted-foreground mb-4">لم يتم العثور على المورد المطلوب</p>
        <Button asChild>
          <Link href="/suppliers">العودة للقائمة</Link>
        </Button>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4">
        <div className="flex items-center gap-4">
          <Button variant="ghost" size="icon" onClick={() => navigate("/suppliers")} data-testid="button-back">
            <ArrowRight className="h-4 w-4" />
          </Button>
          <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-primary/10 text-primary">
            <Building2 className="h-6 w-6" />
          </div>
          <div>
            <h1 className="text-2xl font-bold" data-testid="supplier-name">{supplier.nameAr}</h1>
            <div className="flex items-center gap-2">
              <Badge variant={supplier.isActive ? "default" : "secondary"}>
                {supplier.isActive ? "نشط" : "غير نشط"}
              </Badge>
              {supplier.supplierType && (
                <Badge variant="outline">{supplier.supplierType}</Badge>
              )}
            </div>
          </div>
        </div>
        <Button asChild data-testid="button-edit">
          <Link href={`/suppliers/${supplierId}/edit`}>
            <Pencil className="h-4 w-4 ml-2" />
            تعديل
          </Link>
        </Button>
      </div>

      <div className="grid gap-6 lg:grid-cols-3">
        <Card className="lg:col-span-2">
          <CardHeader>
            <CardTitle>البيانات الأساسية</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid gap-4 sm:grid-cols-2">
              <div>
                <p className="text-sm text-muted-foreground">رقم السجل التجاري</p>
                <p className="font-mono" data-testid="cr-number">{supplier.crNumber || "-"}</p>
              </div>
              <div>
                <p className="text-sm text-muted-foreground">الرقم الضريبي</p>
                <p className="font-mono" data-testid="vat-number">{supplier.vatNumber || "-"}</p>
              </div>
              <div>
                <p className="text-sm text-muted-foreground">المدينة</p>
                <p>{supplier.cityAr || "-"}</p>
              </div>
              <div>
                <p className="text-sm text-muted-foreground">نوع المورد</p>
                <p>{supplier.supplierType || "-"}</p>
              </div>
            </div>

            <Separator />

            <div>
              <h4 className="font-medium mb-3 flex items-center gap-2">
                <Phone className="h-4 w-4" />
                بيانات المندوب
              </h4>
              <div className="grid gap-4 sm:grid-cols-2">
                <div>
                  <p className="text-sm text-muted-foreground">اسم المندوب</p>
                  <p>{supplier.representativeName || "-"}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">رقم الجوال</p>
                  <p className="font-mono" dir="ltr">{supplier.representativePhone || "-"}</p>
                </div>
              </div>
            </div>

            {supplier.bankAccounts && supplier.bankAccounts.length > 0 && (
              <>
                <Separator />
                <div>
                  <h4 className="font-medium mb-3 flex items-center gap-2">
                    <CreditCard className="h-4 w-4" />
                    الحسابات البنكية
                  </h4>
                  <div className="space-y-3">
                    {supplier.bankAccounts.map((account) => (
                      <div key={account.id} className="rounded-lg border p-4">
                        <div className="flex items-center justify-between mb-2">
                          <span className="font-medium">{account.bankNameAr || "حساب بنكي"}</span>
                          {account.isDefault && (
                            <Badge variant="secondary" className="text-xs">افتراضي</Badge>
                          )}
                        </div>
                        <div className="grid gap-2 text-sm">
                          <div className="flex justify-between">
                            <span className="text-muted-foreground">رقم الحساب:</span>
                            <span className="font-mono" dir="ltr">{account.accountNumber || "-"}</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-muted-foreground">الآيبان:</span>
                            <span className="font-mono text-xs" dir="ltr">{account.iban || "-"}</span>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              </>
            )}
          </CardContent>
        </Card>

        <div className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Wallet className="h-5 w-5" />
                الرصيد
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-center">
                <p className="text-3xl font-bold tabular-nums" data-testid="balance-amount">
                  {(balance?.balance ?? 0).toLocaleString("ar-SA")}
                </p>
                <p className="text-sm text-muted-foreground">ريال سعودي</p>
                {balance && balance.balance > 0 && (
                  <Badge variant="secondary" className="mt-2">مستحق للمورد</Badge>
                )}
                {balance && balance.balance < 0 && (
                  <Badge variant="outline" className="mt-2">رصيد دائن للمنشأة</Badge>
                )}
              </div>
              <Separator className="my-4" />
              <div className="space-y-2 text-sm">
                <div className="flex justify-between">
                  <span className="text-muted-foreground">إجمالي المشتريات:</span>
                  <span className="tabular-nums">{(balance?.totalCredits ?? 0).toLocaleString("ar-SA")}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">إجمالي المدفوعات:</span>
                  <span className="tabular-nums">{(balance?.totalDebits ?? 0).toLocaleString("ar-SA")}</span>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between gap-2">
              <CardTitle className="flex items-center gap-2">
                <FileText className="h-5 w-5" />
                الفواتير
              </CardTitle>
              <Button variant="ghost" size="sm" asChild>
                <Link href={`/purchase-invoices?supplier=${supplierId}`}>عرض الكل</Link>
              </Button>
            </CardHeader>
            <CardContent>
              {!invoices || invoices.length === 0 ? (
                <div className="text-center text-muted-foreground py-4">
                  لا توجد فواتير لهذا المورد
                </div>
              ) : (
                <div className="space-y-3">
                  {invoices.slice(0, 5).map((invoice) => (
                    <Link
                      key={invoice.id}
                      href={`/purchase-invoices/${invoice.id}`}
                      className="flex items-center justify-between hover-elevate rounded-lg p-2 -mx-2 cursor-pointer"
                      data-testid={`invoice-link-${invoice.id}`}
                    >
                      <div>
                        <p className="text-sm">فاتورة #{invoice.id}</p>
                        <p className="text-xs text-muted-foreground">
                          {new Date(invoice.invoiceDate).toLocaleDateString("ar-SA")}
                        </p>
                      </div>
                      <div className="text-left">
                        <p className="font-medium tabular-nums">
                          {invoice.totalAmount.toLocaleString("ar-SA")}
                        </p>
                        <Badge
                          variant={
                            invoice.paymentStatus === "paid"
                              ? "default"
                              : invoice.paymentStatus === "partially_paid"
                              ? "secondary"
                              : "destructive"
                          }
                          className="text-xs"
                        >
                          {invoice.paymentStatus === "paid"
                            ? "مدفوع"
                            : invoice.paymentStatus === "partially_paid"
                            ? "جزئي"
                            : "غير مدفوع"}
                        </Badge>
                      </div>
                    </Link>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
