import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useLocation, useParams } from "wouter";
import { useForm, useFieldArray } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Skeleton } from "@/components/ui/skeleton";
import { Switch } from "@/components/ui/switch";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { ArrowRight, Plus, Trash2, Save, Loader2 } from "lucide-react";
import type { SupplierWithBankAccounts } from "@shared/schema";

const supplierFormSchema = z.object({
  nameAr: z.string().min(1, "اسم المورد مطلوب"),
  crNumber: z.string().optional(),
  vatNumber: z.string().optional(),
  cityAr: z.string().optional(),
  addressAr: z.string().optional(),
  supplierType: z.string().default("normal"),
  imageUrl: z.string().optional(),
  representativeName: z.string().optional(),
  representativePhone: z.string().optional(),
  agreementStatus: z.string().optional(),
  paymentDueDays: z.coerce.number().min(0).optional(),
  monthlyPaymentDay: z.coerce.number().min(1).max(28).optional(),
  isActive: z.boolean().default(true),
  bankAccounts: z.array(z.object({
    id: z.number().optional(),
    bankNameAr: z.string().optional(),
    iban: z.string().optional(),
    accountNumber: z.string().optional(),
    isDefault: z.boolean().default(false),
  })).default([]),
});

type SupplierFormValues = z.infer<typeof supplierFormSchema>;

const SUPPLIER_TYPES = [
  { value: "normal", label: "عادي" },
  { value: "online", label: "أونلاين" },
  { value: "cafeteria", label: "كافتيريا" },
];

const AGREEMENT_STATUSES = [
  { value: "none", label: "لا توجد اتفاقية" },
  { value: "active", label: "اتفاقية سارية" },
  { value: "pending", label: "قيد المراجعة" },
  { value: "expired", label: "منتهية" },
];

const CITIES = [
  "الرياض",
  "جدة",
  "مكة المكرمة",
  "المدينة المنورة",
  "الدمام",
  "الخبر",
  "تبوك",
  "أبها",
  "الطائف",
];

export default function SupplierFormPage() {
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const params = useParams<{ id?: string }>();
  const isEditing = !!params.id;
  const supplierId = params.id ? parseInt(params.id) : undefined;

  const { data: supplier, isLoading: isLoadingSupplier } = useQuery<SupplierWithBankAccounts>({
    queryKey: ["/api/suppliers", supplierId],
    enabled: isEditing,
  });

  const form = useForm<SupplierFormValues>({
    resolver: zodResolver(supplierFormSchema),
    defaultValues: {
      nameAr: "",
      crNumber: "",
      vatNumber: "",
      cityAr: "",
      addressAr: "",
      supplierType: "normal",
      imageUrl: "",
      representativeName: "",
      representativePhone: "",
      agreementStatus: "none",
      paymentDueDays: undefined,
      monthlyPaymentDay: undefined,
      isActive: true,
      bankAccounts: [],
    },
  });

  const { fields, append, remove } = useFieldArray({
    control: form.control,
    name: "bankAccounts",
  });

  useEffect(() => {
    if (supplier) {
      const sup = supplier as any;
      form.reset({
        nameAr: supplier.nameAr,
        crNumber: supplier.crNumber || "",
        vatNumber: supplier.vatNumber || "",
        cityAr: supplier.cityAr || "",
        addressAr: sup.addressAr || "",
        supplierType: supplier.supplierType || "normal",
        imageUrl: supplier.imageUrl || "",
        representativeName: supplier.representativeName || "",
        representativePhone: supplier.representativePhone || "",
        agreementStatus: sup.agreementStatus || "none",
        paymentDueDays: sup.paymentDueDays ?? undefined,
        monthlyPaymentDay: sup.monthlyPaymentDay ?? undefined,
        isActive: supplier.isActive,
        bankAccounts: supplier.bankAccounts?.map((acc) => ({
          id: acc.id,
          bankNameAr: acc.bankNameAr || "",
          iban: acc.iban || "",
          accountNumber: acc.accountNumber || "",
          isDefault: acc.isDefault,
        })) || [],
      });
    }
  }, [supplier, form]);

  const createMutation = useMutation({
    mutationFn: async (data: SupplierFormValues) => {
      const response = await apiRequest("POST", "/api/suppliers", data);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/suppliers"] });
      toast({
        title: "تم الإنشاء بنجاح",
        description: "تم إضافة المورد بنجاح",
      });
      navigate("/suppliers");
    },
    onError: () => {
      toast({
        title: "خطأ",
        description: "حدث خطأ أثناء إنشاء المورد",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async (data: SupplierFormValues) => {
      const response = await apiRequest("PUT", `/api/suppliers/${supplierId}`, data);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/suppliers"] });
      queryClient.invalidateQueries({ queryKey: ["/api/suppliers", supplierId] });
      toast({
        title: "تم التحديث بنجاح",
        description: "تم تحديث بيانات المورد",
      });
      navigate("/suppliers");
    },
    onError: () => {
      toast({
        title: "خطأ",
        description: "حدث خطأ أثناء تحديث المورد",
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: SupplierFormValues) => {
    if (isEditing) {
      updateMutation.mutate(data);
    } else {
      createMutation.mutate(data);
    }
  };

  const isPending = createMutation.isPending || updateMutation.isPending;

  if (isEditing && isLoadingSupplier) {
    return (
      <div className="space-y-6">
        <Skeleton className="h-8 w-48" />
        <Card>
          <CardContent className="p-6 space-y-4">
            {[1, 2, 3, 4].map((i) => (
              <div key={i} className="grid gap-4 md:grid-cols-2">
                <Skeleton className="h-10" />
                <Skeleton className="h-10" />
              </div>
            ))}
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center gap-4">
        <Button variant="ghost" size="icon" onClick={() => navigate("/suppliers")} data-testid="button-back">
          <ArrowRight className="h-4 w-4" />
        </Button>
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">
            {isEditing ? "تعديل المورد" : "إضافة مورد جديد"}
          </h1>
          <p className="text-muted-foreground">
            {isEditing ? "تعديل بيانات المورد" : "أدخل بيانات المورد الجديد"}
          </p>
        </div>
      </div>

      <Form {...form}>
        <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>البيانات الأساسية</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid gap-4 md:grid-cols-2">
                <FormField
                  control={form.control}
                  name="nameAr"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>اسم المورد *</FormLabel>
                      <FormControl>
                        <Input placeholder="أدخل اسم المورد" {...field} data-testid="input-name" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="supplierType"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>نوع المورد *</FormLabel>
                      <Select onValueChange={field.onChange} value={field.value}>
                        <FormControl>
                          <SelectTrigger data-testid="select-type">
                            <SelectValue placeholder="اختر نوع المورد" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {SUPPLIER_TYPES.map((type) => (
                            <SelectItem key={type.value} value={type.value}>
                              {type.label}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="grid gap-4 md:grid-cols-2">
                <FormField
                  control={form.control}
                  name="crNumber"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>رقم السجل التجاري</FormLabel>
                      <FormControl>
                        <Input placeholder="1234567890" {...field} data-testid="input-cr" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="vatNumber"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>الرقم الضريبي</FormLabel>
                      <FormControl>
                        <Input placeholder="300123456700003" {...field} data-testid="input-vat" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="grid gap-4 md:grid-cols-2">
                <FormField
                  control={form.control}
                  name="cityAr"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>المدينة</FormLabel>
                      <Select onValueChange={field.onChange} value={field.value}>
                        <FormControl>
                          <SelectTrigger data-testid="select-city">
                            <SelectValue placeholder="اختر المدينة" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {CITIES.map((city) => (
                            <SelectItem key={city} value={city}>
                              {city}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="addressAr"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>العنوان</FormLabel>
                      <FormControl>
                        <Input placeholder="أدخل عنوان المورد" {...field} data-testid="input-address" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="isActive"
                render={({ field }) => (
                  <FormItem className="flex items-center justify-between rounded-lg border p-4">
                    <div>
                      <FormLabel>الحالة</FormLabel>
                      <p className="text-sm text-muted-foreground">
                        هل المورد نشط ويمكن التعامل معه؟
                      </p>
                    </div>
                    <FormControl>
                      <Switch
                        checked={field.value}
                        onCheckedChange={field.onChange}
                        data-testid="switch-active"
                      />
                    </FormControl>
                  </FormItem>
                )}
              />
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>بيانات المندوب</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid gap-4 md:grid-cols-2">
                <FormField
                  control={form.control}
                  name="representativeName"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>اسم المندوب</FormLabel>
                      <FormControl>
                        <Input placeholder="أدخل اسم المندوب" {...field} data-testid="input-rep-name" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="representativePhone"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>رقم جوال المندوب</FormLabel>
                      <FormControl>
                        <Input placeholder="0550000000" {...field} data-testid="input-rep-phone" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>الاتفاقية الشهرية</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid gap-4 md:grid-cols-2">
                <FormField
                  control={form.control}
                  name="agreementStatus"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>حالة الاتفاقية</FormLabel>
                      <Select onValueChange={field.onChange} value={field.value || "none"}>
                        <FormControl>
                          <SelectTrigger data-testid="select-agreement-status">
                            <SelectValue placeholder="اختر حالة الاتفاقية" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {AGREEMENT_STATUSES.map((status) => (
                            <SelectItem key={status.value} value={status.value}>
                              {status.label}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="paymentDueDays"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>مدة استحقاق السداد (بالأيام)</FormLabel>
                      <FormControl>
                        <Input 
                          type="number" 
                          placeholder="مثال: 30" 
                          {...field}
                          value={field.value ?? ""}
                          data-testid="input-payment-due-days" 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="monthlyPaymentDay"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>يوم السداد الشهري</FormLabel>
                    <FormControl>
                      <Input 
                        type="number" 
                        min="1" 
                        max="28" 
                        placeholder="يوم من الشهر (1-28)" 
                        {...field}
                        value={field.value ?? ""}
                        data-testid="input-monthly-payment-day" 
                      />
                    </FormControl>
                    <p className="text-sm text-muted-foreground">
                      اليوم من كل شهر الذي يستحق فيه السداد للمورد
                    </p>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between gap-2">
              <CardTitle>الحسابات البنكية</CardTitle>
              <Button
                type="button"
                variant="outline"
                size="sm"
                onClick={() =>
                  append({ bankNameAr: "", iban: "", accountNumber: "", isDefault: false })
                }
                data-testid="button-add-bank"
              >
                <Plus className="h-4 w-4 ml-2" />
                إضافة حساب
              </Button>
            </CardHeader>
            <CardContent className="space-y-4">
              {fields.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  لا توجد حسابات بنكية. اضغط على "إضافة حساب" لإضافة حساب جديد.
                </div>
              ) : (
                fields.map((field, index) => (
                  <div key={field.id} className="rounded-lg border p-4 space-y-4">
                    <div className="flex items-center justify-between">
                      <span className="font-medium">حساب {index + 1}</span>
                      <Button
                        type="button"
                        variant="ghost"
                        size="icon"
                        onClick={() => remove(index)}
                        data-testid={`button-remove-bank-${index}`}
                      >
                        <Trash2 className="h-4 w-4 text-destructive" />
                      </Button>
                    </div>
                    <div className="grid gap-4 md:grid-cols-2">
                      <FormField
                        control={form.control}
                        name={`bankAccounts.${index}.bankNameAr`}
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>اسم البنك</FormLabel>
                            <FormControl>
                              <Input placeholder="بنك الراجحي" {...field} data-testid={`input-bank-name-${index}`} />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                      <FormField
                        control={form.control}
                        name={`bankAccounts.${index}.accountNumber`}
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>رقم الحساب</FormLabel>
                            <FormControl>
                              <Input placeholder="1234567890" {...field} data-testid={`input-account-${index}`} />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    </div>
                    <div className="grid gap-4 md:grid-cols-2">
                      <FormField
                        control={form.control}
                        name={`bankAccounts.${index}.iban`}
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>رقم الآيبان</FormLabel>
                            <FormControl>
                              <Input placeholder="SA0000000000000000000000" {...field} data-testid={`input-iban-${index}`} />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                      <FormField
                        control={form.control}
                        name={`bankAccounts.${index}.isDefault`}
                        render={({ field }) => (
                          <FormItem className="flex items-center justify-between rounded-lg border p-3">
                            <FormLabel className="text-sm">الحساب الافتراضي</FormLabel>
                            <FormControl>
                              <Switch
                                checked={field.value}
                                onCheckedChange={field.onChange}
                                data-testid={`switch-default-${index}`}
                              />
                            </FormControl>
                          </FormItem>
                        )}
                      />
                    </div>
                  </div>
                ))
              )}
            </CardContent>
          </Card>

          <div className="flex justify-end gap-4">
            <Button type="button" variant="outline" onClick={() => navigate("/suppliers")} data-testid="button-cancel">
              إلغاء
            </Button>
            <Button type="submit" disabled={isPending} data-testid="button-submit">
              {isPending ? (
                <Loader2 className="h-4 w-4 animate-spin ml-2" />
              ) : (
                <Save className="h-4 w-4 ml-2" />
              )}
              {isEditing ? "حفظ التغييرات" : "إضافة المورد"}
            </Button>
          </div>
        </form>
      </Form>
    </div>
  );
}
