import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Link, useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import {
  Plus,
  Search,
  MoreHorizontal,
  Eye,
  Pencil,
  Trash2,
  Users,
  Phone,
  Building2,
} from "lucide-react";
import type { SupplierWithBankAccounts } from "@shared/schema";

const SUPPLIER_TYPE_LABELS: Record<string, string> = {
  normal: "عادي",
  online: "أونلاين",
  cafeteria: "كافتيريا",
};

const AGREEMENT_STATUS_LABELS: Record<string, string> = {
  none: "لا توجد",
  active: "سارية",
  pending: "قيد المراجعة",
  expired: "منتهية",
};

function SupplierSkeleton() {
  return (
    <TableRow>
      <TableCell><Skeleton className="h-4 w-32" /></TableCell>
      <TableCell><Skeleton className="h-4 w-24" /></TableCell>
      <TableCell><Skeleton className="h-4 w-28" /></TableCell>
      <TableCell><Skeleton className="h-4 w-20" /></TableCell>
      <TableCell><Skeleton className="h-4 w-24" /></TableCell>
      <TableCell><Skeleton className="h-6 w-16" /></TableCell>
      <TableCell><Skeleton className="h-8 w-8" /></TableCell>
    </TableRow>
  );
}

function EmptyState() {
  return (
    <div className="flex flex-col items-center justify-center py-16 text-center">
      <div className="flex h-16 w-16 items-center justify-center rounded-full bg-muted mb-4">
        <Users className="h-8 w-8 text-muted-foreground" />
      </div>
      <h3 className="text-lg font-semibold mb-2">لا يوجد موردين</h3>
      <p className="text-muted-foreground mb-4">ابدأ بإضافة أول مورد للنظام</p>
      <Button asChild>
        <Link href="/suppliers/new" data-testid="empty-add-supplier">
          <Plus className="h-4 w-4 ml-2" />
          إضافة مورد
        </Link>
      </Button>
    </div>
  );
}

export default function SuppliersPage() {
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const [searchQuery, setSearchQuery] = useState("");
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [supplierToDelete, setSupplierToDelete] = useState<SupplierWithBankAccounts | null>(null);

  const { data: suppliers, isLoading } = useQuery<SupplierWithBankAccounts[]>({
    queryKey: ["/api/suppliers"],
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      await apiRequest("DELETE", `/api/suppliers/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/suppliers"] });
      toast({
        title: "تم الحذف بنجاح",
        description: "تم إلغاء تنشيط المورد",
      });
      setDeleteDialogOpen(false);
      setSupplierToDelete(null);
    },
    onError: () => {
      toast({
        title: "خطأ",
        description: "حدث خطأ أثناء حذف المورد",
        variant: "destructive",
      });
    },
  });

  const filteredSuppliers = suppliers?.filter(
    (supplier) =>
      supplier.nameAr.includes(searchQuery) ||
      supplier.crNumber?.includes(searchQuery) ||
      supplier.vatNumber?.includes(searchQuery) ||
      supplier.cityAr?.includes(searchQuery)
  );

  const handleDelete = (supplier: SupplierWithBankAccounts) => {
    setSupplierToDelete(supplier);
    setDeleteDialogOpen(true);
  };

  return (
    <div className="space-y-6">
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">الموردين</h1>
          <p className="text-muted-foreground">إدارة الموردين وبياناتهم المصرفية</p>
        </div>
        <Button asChild data-testid="button-add-supplier">
          <Link href="/suppliers/new">
            <Plus className="h-4 w-4 ml-2" />
            إضافة مورد
          </Link>
        </Button>
      </div>

      <Card>
        <CardHeader className="pb-4">
          <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
            <CardTitle className="text-lg">قائمة الموردين</CardTitle>
            <div className="relative w-full sm:w-72">
              <Search className="absolute right-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
              <Input
                placeholder="بحث بالاسم أو السجل التجاري..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pr-10"
                data-testid="input-search-suppliers"
              />
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">اسم المورد</TableHead>
                    <TableHead className="text-right">رقم السجل</TableHead>
                    <TableHead className="text-right">النوع</TableHead>
                    <TableHead className="text-right">الاتفاقية</TableHead>
                    <TableHead className="text-right">السداد</TableHead>
                    <TableHead className="text-right">الحالة</TableHead>
                    <TableHead className="w-12"></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {[1, 2, 3, 4, 5].map((i) => (
                    <SupplierSkeleton key={i} />
                  ))}
                </TableBody>
              </Table>
            </div>
          ) : !filteredSuppliers || filteredSuppliers.length === 0 ? (
            <EmptyState />
          ) : (
            <div className="rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">اسم المورد</TableHead>
                    <TableHead className="text-right">رقم السجل</TableHead>
                    <TableHead className="text-right">النوع</TableHead>
                    <TableHead className="text-right">الاتفاقية</TableHead>
                    <TableHead className="text-right">السداد</TableHead>
                    <TableHead className="text-right">الحالة</TableHead>
                    <TableHead className="w-12"></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredSuppliers.map((supplier) => (
                    <TableRow
                      key={supplier.id}
                      className="hover-elevate cursor-pointer"
                      onClick={() => navigate(`/suppliers/${supplier.id}`)}
                      data-testid={`supplier-row-${supplier.id}`}
                    >
                      <TableCell>
                        <div className="flex items-center gap-3">
                          <div className="flex h-10 w-10 items-center justify-center rounded-lg bg-primary/10 text-primary">
                            <Building2 className="h-5 w-5" />
                          </div>
                          <div>
                            <p className="font-medium">{supplier.nameAr}</p>
                            {supplier.representativeName && (
                              <div className="flex items-center gap-1 text-xs text-muted-foreground">
                                <Phone className="h-3 w-3" />
                                {supplier.representativeName}
                              </div>
                            )}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell className="font-mono text-sm">
                        {supplier.crNumber || "-"}
                      </TableCell>
                      <TableCell>
                        {supplier.supplierType 
                          ? (SUPPLIER_TYPE_LABELS[supplier.supplierType] || supplier.supplierType)
                          : "-"
                        }
                      </TableCell>
                      <TableCell>
                        {(() => {
                          const sup = supplier as any;
                          const status = sup.agreementStatus || "none";
                          const label = AGREEMENT_STATUS_LABELS[status] || status;
                          const variant = status === "active" ? "default" : 
                                        status === "expired" ? "destructive" : 
                                        status === "pending" ? "secondary" : "outline";
                          return <Badge variant={variant as any}>{label}</Badge>;
                        })()}
                      </TableCell>
                      <TableCell>
                        {(() => {
                          const sup = supplier as any;
                          if (sup.paymentDueDays) {
                            return `${sup.paymentDueDays} يوم`;
                          }
                          if (sup.monthlyPaymentDay) {
                            return `يوم ${sup.monthlyPaymentDay}`;
                          }
                          return "-";
                        })()}
                      </TableCell>
                      <TableCell>
                        <Badge variant={supplier.isActive ? "default" : "secondary"}>
                          {supplier.isActive ? "نشط" : "غير نشط"}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild onClick={(e) => e.stopPropagation()}>
                            <Button variant="ghost" size="icon" data-testid={`supplier-menu-${supplier.id}`}>
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuItem onClick={(e) => {
                              e.stopPropagation();
                              navigate(`/suppliers/${supplier.id}`);
                            }}>
                              <Eye className="h-4 w-4 ml-2" />
                              عرض
                            </DropdownMenuItem>
                            <DropdownMenuItem onClick={(e) => {
                              e.stopPropagation();
                              navigate(`/suppliers/${supplier.id}/edit`);
                            }}>
                              <Pencil className="h-4 w-4 ml-2" />
                              تعديل
                            </DropdownMenuItem>
                            <DropdownMenuItem
                              className="text-destructive"
                              onClick={(e) => {
                                e.stopPropagation();
                                handleDelete(supplier);
                              }}
                            >
                              <Trash2 className="h-4 w-4 ml-2" />
                              حذف
                            </DropdownMenuItem>
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </CardContent>
      </Card>

      <AlertDialog open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>هل أنت متأكد من الحذف؟</AlertDialogTitle>
            <AlertDialogDescription>
              سيتم إلغاء تنشيط المورد "{supplierToDelete?.nameAr}". يمكنك إعادة تنشيطه لاحقاً.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter className="gap-2">
            <AlertDialogCancel>إلغاء</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => supplierToDelete && deleteMutation.mutate(supplierToDelete.id)}
              className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
            >
              حذف
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
