import { useQuery } from "@tanstack/react-query";
import { useState, Fragment } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
  TableFooter,
} from "@/components/ui/table";
import { Skeleton } from "@/components/ui/skeleton";
import { Scale, Calendar, Download, RefreshCw } from "lucide-react";
import { formatCurrency, formatNumber } from "@/lib/formatters";

interface TrialBalanceItem {
  accountCode: string;
  accountName: string;
  accountType: string;
  debit: number;
  credit: number;
  balance: number;
  balanceType: "debit" | "credit";
}

const accountTypeLabels: Record<string, string> = {
  asset: "أصول",
  liability: "خصوم",
  equity: "حقوق ملكية",
  revenue: "إيرادات",
  expense: "مصروفات",
};

export default function TrialBalancePage() {
  const [asOfDate, setAsOfDate] = useState<string>(
    new Date().toISOString().split("T")[0]
  );

  const { data: trialBalance, isLoading, refetch } = useQuery<TrialBalanceItem[]>({
    queryKey: ["/api/reports/trial-balance", asOfDate],
    queryFn: async () => {
      const params = new URLSearchParams();
      if (asOfDate) params.set("asOfDate", asOfDate);
      const res = await fetch(`/api/reports/trial-balance?${params.toString()}`);
      if (!res.ok) throw new Error("Failed to fetch trial balance");
      return res.json();
    },
  });

  const totalDebit = trialBalance?.reduce((sum, item) => sum + item.debit, 0) || 0;
  const totalCredit = trialBalance?.reduce((sum, item) => sum + item.credit, 0) || 0;
  const isBalanced = Math.abs(totalDebit - totalCredit) < 0.01;

  const groupedByType = trialBalance?.reduce((acc, item) => {
    if (!acc[item.accountType]) {
      acc[item.accountType] = [];
    }
    acc[item.accountType].push(item);
    return acc;
  }, {} as Record<string, TrialBalanceItem[]>);

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">ميزان المراجعة</h1>
          <p className="text-muted-foreground">عرض أرصدة الحسابات المحاسبية</p>
        </div>
        <div className="flex items-center gap-4">
          <div className="flex items-center gap-2">
            <Label htmlFor="asOfDate">حتى تاريخ:</Label>
            <Input
              id="asOfDate"
              type="date"
              value={asOfDate}
              onChange={(e) => setAsOfDate(e.target.value)}
              className="w-40"
              data-testid="input-as-of-date"
            />
          </div>
          <Button variant="outline" onClick={() => refetch()} data-testid="button-refresh">
            <RefreshCw className="h-4 w-4 ml-2" />
            تحديث
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <Scale className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي المدين</p>
                <p className="text-xl font-bold" data-testid="text-total-debit">
                  {formatNumber(totalDebit)} ر.س
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                <Scale className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الدائن</p>
                <p className="text-xl font-bold" data-testid="text-total-credit">
                  {formatNumber(totalCredit)} ر.س
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className={`p-2 rounded-lg ${isBalanced ? "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200" : "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200"}`}>
                <Scale className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">حالة الميزان</p>
                <div className="flex items-center gap-2">
                  {isBalanced ? (
                    <Badge className="bg-green-100 text-green-800" data-testid="text-balance-status">
                      متوازن
                    </Badge>
                  ) : (
                    <Badge variant="destructive" data-testid="text-balance-status">
                      غير متوازن - الفرق: {formatNumber(Math.abs(totalDebit - totalCredit))}
                    </Badge>
                  )}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Calendar className="h-5 w-5" />
            ميزان المراجعة حتى {asOfDate}
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3, 4, 5].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : trialBalance && trialBalance.length > 0 ? (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">رمز الحساب</TableHead>
                  <TableHead className="text-right">اسم الحساب</TableHead>
                  <TableHead className="text-right">النوع</TableHead>
                  <TableHead className="text-left">مدين</TableHead>
                  <TableHead className="text-left">دائن</TableHead>
                  <TableHead className="text-left">الرصيد</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {Object.entries(groupedByType || {}).map(([type, items]) => (
                  <Fragment key={type}>
                    <TableRow className="bg-muted/50">
                      <TableCell colSpan={6} className="font-bold">
                        {accountTypeLabels[type]}
                      </TableCell>
                    </TableRow>
                    {items.map((item) => (
                      <TableRow key={item.accountCode} data-testid={`row-account-${item.accountCode}`}>
                        <TableCell className="font-mono">{item.accountCode}</TableCell>
                        <TableCell>{item.accountName}</TableCell>
                        <TableCell>
                          <Badge variant="outline">{accountTypeLabels[item.accountType]}</Badge>
                        </TableCell>
                        <TableCell className="text-left font-mono">
                          {item.debit > 0 ? formatNumber(item.debit) : "-"}
                        </TableCell>
                        <TableCell className="text-left font-mono">
                          {item.credit > 0 ? formatNumber(item.credit) : "-"}
                        </TableCell>
                        <TableCell className="text-left font-mono">
                          <span className={item.balanceType === "debit" ? "text-green-600" : "text-red-600"}>
                            {formatNumber(item.balance)} {item.balanceType === "debit" ? "مدين" : "دائن"}
                          </span>
                        </TableCell>
                      </TableRow>
                    ))}
                  </Fragment>
                ))}
              </TableBody>
              <TableFooter>
                <TableRow className="bg-muted font-bold">
                  <TableCell colSpan={3}>الإجمالي</TableCell>
                  <TableCell className="text-left font-mono" data-testid="text-footer-debit">
                    {formatNumber(totalDebit)}
                  </TableCell>
                  <TableCell className="text-left font-mono" data-testid="text-footer-credit">
                    {formatNumber(totalCredit)}
                  </TableCell>
                  <TableCell className="text-left font-mono">
                    {isBalanced ? (
                      <Badge className="bg-green-100 text-green-800">متوازن</Badge>
                    ) : (
                      <Badge variant="destructive">
                        فرق: {formatNumber(Math.abs(totalDebit - totalCredit))}
                      </Badge>
                    )}
                  </TableCell>
                </TableRow>
              </TableFooter>
            </Table>
          ) : (
            <div className="text-center py-8 text-muted-foreground">
              <Scale className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p>لا توجد حركات محاسبية بعد</p>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
