import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Skeleton } from "@/components/ui/skeleton";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Plus, Pencil, Trash2, Ruler, Save, Loader2 } from "lucide-react";
import type { Unit } from "@shared/schema";

const unitFormSchema = z.object({
  nameAr: z.string().min(1, "اسم الوحدة مطلوب"),
  code: z.string().optional(),
});

type UnitFormValues = z.infer<typeof unitFormSchema>;

function UnitSkeleton() {
  return (
    <TableRow>
      <TableCell><Skeleton className="h-4 w-8" /></TableCell>
      <TableCell><Skeleton className="h-4 w-24" /></TableCell>
      <TableCell><Skeleton className="h-4 w-16" /></TableCell>
      <TableCell><Skeleton className="h-8 w-20" /></TableCell>
    </TableRow>
  );
}

function UnitDialog({
  open,
  onOpenChange,
  unit,
  onSuccess,
}: {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  unit?: Unit;
  onSuccess: () => void;
}) {
  const { toast } = useToast();
  const isEditing = !!unit;

  const form = useForm<UnitFormValues>({
    resolver: zodResolver(unitFormSchema),
    defaultValues: {
      nameAr: unit?.nameAr || "",
      code: unit?.code || "",
    },
  });

  const createMutation = useMutation({
    mutationFn: async (data: UnitFormValues) => {
      const response = await apiRequest("POST", "/api/units", data);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/units"] });
      toast({
        title: "تم الإنشاء بنجاح",
        description: "تم إضافة الوحدة بنجاح",
      });
      form.reset();
      onSuccess();
    },
    onError: () => {
      toast({
        title: "خطأ",
        description: "حدث خطأ أثناء إنشاء الوحدة",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async (data: UnitFormValues) => {
      const response = await apiRequest("PUT", `/api/units/${unit?.id}`, data);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/units"] });
      toast({
        title: "تم التحديث بنجاح",
        description: "تم تحديث الوحدة بنجاح",
      });
      onSuccess();
    },
    onError: () => {
      toast({
        title: "خطأ",
        description: "حدث خطأ أثناء تحديث الوحدة",
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: UnitFormValues) => {
    if (isEditing) {
      updateMutation.mutate(data);
    } else {
      createMutation.mutate(data);
    }
  };

  const isPending = createMutation.isPending || updateMutation.isPending;

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent>
        <DialogHeader>
          <DialogTitle>{isEditing ? "تعديل الوحدة" : "إضافة وحدة جديدة"}</DialogTitle>
        </DialogHeader>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
            <FormField
              control={form.control}
              name="nameAr"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>اسم الوحدة *</FormLabel>
                  <FormControl>
                    <Input placeholder="مثال: كرتون، حبة، عبوة" {...field} data-testid="input-unit-name" />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="code"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>الكود</FormLabel>
                  <FormControl>
                    <Input placeholder="مثال: CTN, PC, PKG" {...field} data-testid="input-unit-code" />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <div className="flex justify-end gap-2">
              <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>
                إلغاء
              </Button>
              <Button type="submit" disabled={isPending} data-testid="button-save-unit">
                {isPending ? (
                  <Loader2 className="h-4 w-4 animate-spin ml-2" />
                ) : (
                  <Save className="h-4 w-4 ml-2" />
                )}
                {isEditing ? "حفظ" : "إضافة"}
              </Button>
            </div>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  );
}

export default function UnitsPage() {
  const { toast } = useToast();
  const [dialogOpen, setDialogOpen] = useState(false);
  const [editingUnit, setEditingUnit] = useState<Unit | undefined>();
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [unitToDelete, setUnitToDelete] = useState<Unit | null>(null);

  const { data: units, isLoading } = useQuery<Unit[]>({
    queryKey: ["/api/units"],
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      await apiRequest("DELETE", `/api/units/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/units"] });
      toast({
        title: "تم الحذف بنجاح",
        description: "تم حذف الوحدة",
      });
      setDeleteDialogOpen(false);
      setUnitToDelete(null);
    },
    onError: () => {
      toast({
        title: "خطأ",
        description: "لا يمكن حذف الوحدة لأنها مستخدمة في منتجات",
        variant: "destructive",
      });
    },
  });

  const handleEdit = (unit: Unit) => {
    setEditingUnit(unit);
    setDialogOpen(true);
  };

  const handleDelete = (unit: Unit) => {
    setUnitToDelete(unit);
    setDeleteDialogOpen(true);
  };

  const handleDialogClose = () => {
    setDialogOpen(false);
    setEditingUnit(undefined);
  };

  return (
    <div className="space-y-6">
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">وحدات القياس</h1>
          <p className="text-muted-foreground">إدارة وحدات الشراء والبيع</p>
        </div>
        <Button onClick={() => setDialogOpen(true)} data-testid="button-add-unit">
          <Plus className="h-4 w-4 ml-2" />
          إضافة وحدة
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="text-lg">قائمة الوحدات</CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right w-16">#</TableHead>
                    <TableHead className="text-right">اسم الوحدة</TableHead>
                    <TableHead className="text-right">الكود</TableHead>
                    <TableHead className="w-24"></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {[1, 2, 3, 4].map((i) => (
                    <UnitSkeleton key={i} />
                  ))}
                </TableBody>
              </Table>
            </div>
          ) : !units || units.length === 0 ? (
            <div className="flex flex-col items-center justify-center py-16 text-center">
              <div className="flex h-16 w-16 items-center justify-center rounded-full bg-muted mb-4">
                <Ruler className="h-8 w-8 text-muted-foreground" />
              </div>
              <h3 className="text-lg font-semibold mb-2">لا توجد وحدات</h3>
              <p className="text-muted-foreground mb-4">ابدأ بإضافة وحدات القياس</p>
              <Button onClick={() => setDialogOpen(true)} data-testid="empty-add-unit">
                <Plus className="h-4 w-4 ml-2" />
                إضافة وحدة
              </Button>
            </div>
          ) : (
            <div className="rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right w-16">#</TableHead>
                    <TableHead className="text-right">اسم الوحدة</TableHead>
                    <TableHead className="text-right">الكود</TableHead>
                    <TableHead className="w-24"></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {units.map((unit, index) => (
                    <TableRow key={unit.id} data-testid={`unit-row-${unit.id}`}>
                      <TableCell className="font-mono text-muted-foreground">
                        {index + 1}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Ruler className="h-4 w-4 text-muted-foreground" />
                          <span className="font-medium">{unit.nameAr}</span>
                        </div>
                      </TableCell>
                      <TableCell className="font-mono text-sm text-muted-foreground">
                        {unit.code || "-"}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleEdit(unit)}
                            data-testid={`edit-unit-${unit.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleDelete(unit)}
                            data-testid={`delete-unit-${unit.id}`}
                          >
                            <Trash2 className="h-4 w-4 text-destructive" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </CardContent>
      </Card>

      <UnitDialog
        open={dialogOpen}
        onOpenChange={handleDialogClose}
        unit={editingUnit}
        onSuccess={handleDialogClose}
      />

      <AlertDialog open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>هل أنت متأكد من الحذف؟</AlertDialogTitle>
            <AlertDialogDescription>
              سيتم حذف الوحدة "{unitToDelete?.nameAr}" بشكل نهائي.
              لا يمكن حذف الوحدة إذا كانت مستخدمة في منتجات.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter className="gap-2">
            <AlertDialogCancel>إلغاء</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => unitToDelete && deleteMutation.mutate(unitToDelete.id)}
              className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
            >
              حذف
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
