import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Separator } from "@/components/ui/separator";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import {
  FileText,
  Calculator,
  TrendingUp,
  TrendingDown,
  Receipt,
  ArrowLeft,
  Calendar,
  Building2,
  Printer,
  Download,
} from "lucide-react";
import { format, startOfQuarter, endOfQuarter, startOfMonth, endOfMonth, subQuarters } from "date-fns";
import { ar } from "date-fns/locale";
import { formatCurrency } from "@/lib/formatters";

interface VatDeclarationData {
  period: { from: string; to: string };
  sales: {
    standardRatedSales: number;
    vatOnStandardRatedSales: number;
    zeroRatedSales: number;
    exemptSales: number;
    totalSales: number;
  };
  adjustments: {
    salesReturns: number;
    vatOnSalesReturns: number;
  };
  purchases: {
    standardRatedPurchases: number;
    vatOnStandardRatedPurchases: number;
    zeroRatedPurchases: number;
    exemptPurchases: number;
    totalPurchases: number;
  };
  summary: {
    totalVatDue: number;
    totalVatRecoverable: number;
    netVatPayable: number;
  };
  breakdown: {
    servicesSales: { subtotal: number; vat: number; total: number };
    productsSales: { subtotal: number; vat: number; total: number };
    purchaseInvoices: { subtotal: number; vat: number; total: number };
  };
}

const quarterOptions = [
  { value: "q1", label: "الربع الأول (يناير - مارس)" },
  { value: "q2", label: "الربع الثاني (أبريل - يونيو)" },
  { value: "q3", label: "الربع الثالث (يوليو - سبتمبر)" },
  { value: "q4", label: "الربع الرابع (أكتوبر - ديسمبر)" },
];

const currentYear = new Date().getFullYear();
const yearOptions = [currentYear - 1, currentYear, currentYear + 1];

export default function VatDeclarationPage() {
  const [periodType, setPeriodType] = useState<"quarterly" | "monthly" | "custom">("quarterly");
  const [selectedQuarter, setSelectedQuarter] = useState<string>("q4");
  const [selectedYear, setSelectedYear] = useState<number>(currentYear);
  const [selectedMonth, setSelectedMonth] = useState<number>(new Date().getMonth());
  const [customFromDate, setCustomFromDate] = useState<string>("");
  const [customToDate, setCustomToDate] = useState<string>("");

  const getDateRange = () => {
    if (periodType === "quarterly") {
      const quarterIndex = parseInt(selectedQuarter.replace("q", "")) - 1;
      const quarterStart = new Date(selectedYear, quarterIndex * 3, 1);
      const quarterEnd = endOfQuarter(quarterStart);
      return {
        from: format(quarterStart, "yyyy-MM-dd"),
        to: format(quarterEnd, "yyyy-MM-dd"),
      };
    } else if (periodType === "monthly") {
      const monthStart = new Date(selectedYear, selectedMonth, 1);
      const monthEnd = endOfMonth(monthStart);
      return {
        from: format(monthStart, "yyyy-MM-dd"),
        to: format(monthEnd, "yyyy-MM-dd"),
      };
    } else {
      return { from: customFromDate, to: customToDate };
    }
  };

  const dateRange = getDateRange();
  const canFetch = Boolean(dateRange.from && dateRange.to);

  const { data, isLoading, error, refetch } = useQuery<VatDeclarationData>({
    queryKey: [`/api/reports/vat-declaration?fromDate=${dateRange.from}&toDate=${dateRange.to}`],
    enabled: canFetch,
  });

  const handlePrint = () => {
    window.print();
  };

  return (
    <div className="space-y-6 print:space-y-4">
      <div className="flex items-center justify-between print:hidden">
        <div>
          <h1 className="text-2xl font-bold flex items-center gap-2">
            <FileText className="h-6 w-6 text-primary" />
            تقرير الإقرار الضريبي
          </h1>
          <p className="text-muted-foreground">
            متوافق مع متطلبات هيئة الزكاة والضريبة والجمارك (ZATCA)
          </p>
        </div>
        <div className="flex gap-2">
          <Button variant="outline" onClick={handlePrint} data-testid="button-print">
            <Printer className="h-4 w-4 ml-2" />
            طباعة
          </Button>
        </div>
      </div>

      <Card className="print:hidden">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Calendar className="h-5 w-5" />
            تحديد الفترة الضريبية
          </CardTitle>
          <CardDescription>
            اختر الفترة الضريبية لإعداد الإقرار (ربع سنوي حسب متطلبات ZATCA)
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-4">
            <div className="space-y-2">
              <Label>نوع الفترة</Label>
              <Select value={periodType} onValueChange={(v) => setPeriodType(v as any)}>
                <SelectTrigger data-testid="select-period-type">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="quarterly">ربع سنوي</SelectItem>
                  <SelectItem value="monthly">شهري</SelectItem>
                  <SelectItem value="custom">مخصص</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {periodType === "quarterly" && (
              <>
                <div className="space-y-2">
                  <Label>السنة</Label>
                  <Select value={selectedYear.toString()} onValueChange={(v) => setSelectedYear(parseInt(v))}>
                    <SelectTrigger data-testid="select-year">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      {yearOptions.map((year) => (
                        <SelectItem key={year} value={year.toString()}>
                          {year}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label>الربع</Label>
                  <Select value={selectedQuarter} onValueChange={setSelectedQuarter}>
                    <SelectTrigger data-testid="select-quarter">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      {quarterOptions.map((q) => (
                        <SelectItem key={q.value} value={q.value}>
                          {q.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
              </>
            )}

            {periodType === "monthly" && (
              <>
                <div className="space-y-2">
                  <Label>السنة</Label>
                  <Select value={selectedYear.toString()} onValueChange={(v) => setSelectedYear(parseInt(v))}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      {yearOptions.map((year) => (
                        <SelectItem key={year} value={year.toString()}>
                          {year}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label>الشهر</Label>
                  <Select value={selectedMonth.toString()} onValueChange={(v) => setSelectedMonth(parseInt(v))}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      {Array.from({ length: 12 }, (_, i) => (
                        <SelectItem key={i} value={i.toString()}>
                          {format(new Date(2024, i, 1), "MMMM", { locale: ar })}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
              </>
            )}

            {periodType === "custom" && (
              <>
                <div className="space-y-2">
                  <Label>من تاريخ</Label>
                  <Input
                    type="date"
                    value={customFromDate}
                    onChange={(e) => setCustomFromDate(e.target.value)}
                    data-testid="input-from-date"
                  />
                </div>
                <div className="space-y-2">
                  <Label>إلى تاريخ</Label>
                  <Input
                    type="date"
                    value={customToDate}
                    onChange={(e) => setCustomToDate(e.target.value)}
                    data-testid="input-to-date"
                  />
                </div>
              </>
            )}

            <div className="flex items-end">
              <Button onClick={() => refetch()} disabled={!canFetch} data-testid="button-generate">
                <Calculator className="h-4 w-4 ml-2" />
                إنشاء التقرير
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>

      {isLoading && (
        <div className="space-y-4">
          <Skeleton className="h-32 w-full" />
          <Skeleton className="h-64 w-full" />
        </div>
      )}

      {error && (
        <Card className="border-destructive">
          <CardContent className="pt-6">
            <p className="text-destructive">حدث خطأ في جلب البيانات. يرجى المحاولة مرة أخرى.</p>
          </CardContent>
        </Card>
      )}

      {data && (
        <>
          <div className="print:block hidden text-center mb-6">
            <h1 className="text-xl font-bold">الإقرار الضريبي لضريبة القيمة المضافة</h1>
            <p className="text-sm text-muted-foreground">
              الفترة من {format(new Date(data.period.from), "dd/MM/yyyy")} إلى {format(new Date(data.period.to), "dd/MM/yyyy")}
            </p>
          </div>

          <div className="grid gap-4 md:grid-cols-3 print:grid-cols-3">
            <Card className="border-green-200 bg-green-50 dark:bg-green-950/20">
              <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">ضريبة المبيعات المستحقة</CardTitle>
                <TrendingUp className="h-4 w-4 text-green-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold text-green-600" data-testid="text-vat-due">
                  {formatCurrency(data.summary.totalVatDue)}
                </div>
                <p className="text-xs text-muted-foreground">ضريبة محصلة من العملاء</p>
              </CardContent>
            </Card>

            <Card className="border-blue-200 bg-blue-50 dark:bg-blue-950/20">
              <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">ضريبة المشتريات القابلة للاسترداد</CardTitle>
                <TrendingDown className="h-4 w-4 text-blue-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold text-blue-600" data-testid="text-vat-recoverable">
                  {formatCurrency(data.summary.totalVatRecoverable)}
                </div>
                <p className="text-xs text-muted-foreground">ضريبة مدفوعة للموردين</p>
              </CardContent>
            </Card>

            <Card className={`${data.summary.netVatPayable >= 0 ? "border-primary bg-primary/5" : "border-amber-200 bg-amber-50 dark:bg-amber-950/20"}`}>
              <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">
                  {data.summary.netVatPayable >= 0 ? "صافي الضريبة المستحقة للسداد" : "صافي الضريبة المستردة"}
                </CardTitle>
                <Receipt className="h-4 w-4 text-primary" />
              </CardHeader>
              <CardContent>
                <div className={`text-2xl font-bold ${data.summary.netVatPayable >= 0 ? "text-primary" : "text-amber-600"}`} data-testid="text-net-vat">
                  {formatCurrency(Math.abs(data.summary.netVatPayable))}
                </div>
                <p className="text-xs text-muted-foreground">
                  {data.summary.netVatPayable >= 0 ? "مبلغ يجب سداده لهيئة الزكاة" : "مبلغ قابل للاسترداد"}
                </p>
              </CardContent>
            </Card>
          </div>

          <div className="grid gap-6 lg:grid-cols-2 print:grid-cols-2">
            <Card>
              <CardHeader className="bg-green-50 dark:bg-green-950/20 border-b">
                <CardTitle className="flex items-center gap-2 text-green-700 dark:text-green-400">
                  <TrendingUp className="h-5 w-5" />
                  المبيعات والضريبة المحصلة
                </CardTitle>
              </CardHeader>
              <CardContent className="pt-4">
                <div className="space-y-4">
                  <div className="flex justify-between items-center py-2 border-b">
                    <span className="font-medium">1. المبيعات الخاضعة للضريبة بالنسبة الأساسية (15%)</span>
                    <span className="font-bold tabular-nums" data-testid="text-standard-sales">
                      {formatCurrency(data.sales.standardRatedSales)}
                    </span>
                  </div>
                  <div className="flex justify-between items-center py-2 border-b">
                    <span className="font-medium">2. ضريبة القيمة المضافة على المبيعات</span>
                    <span className="font-bold text-green-600 tabular-nums" data-testid="text-sales-vat">
                      {formatCurrency(data.sales.vatOnStandardRatedSales)}
                    </span>
                  </div>
                  <div className="flex justify-between items-center py-2 border-b text-muted-foreground">
                    <span>3. المبيعات الخاضعة للضريبة بنسبة صفر</span>
                    <span className="tabular-nums">{formatCurrency(data.sales.zeroRatedSales)}</span>
                  </div>
                  <div className="flex justify-between items-center py-2 border-b text-muted-foreground">
                    <span>4. المبيعات المعفاة من الضريبة</span>
                    <span className="tabular-nums">{formatCurrency(data.sales.exemptSales)}</span>
                  </div>
                  <div className="flex justify-between items-center py-2 bg-muted/50 rounded px-2">
                    <span className="font-bold">5. إجمالي المبيعات</span>
                    <span className="font-bold text-lg tabular-nums">{formatCurrency(data.sales.totalSales)}</span>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="bg-blue-50 dark:bg-blue-950/20 border-b">
                <CardTitle className="flex items-center gap-2 text-blue-700 dark:text-blue-400">
                  <TrendingDown className="h-5 w-5" />
                  المشتريات والضريبة القابلة للخصم
                </CardTitle>
              </CardHeader>
              <CardContent className="pt-4">
                <div className="space-y-4">
                  <div className="flex justify-between items-center py-2 border-b">
                    <span className="font-medium">6. المشتريات الخاضعة للضريبة بالنسبة الأساسية (15%)</span>
                    <span className="font-bold tabular-nums" data-testid="text-standard-purchases">
                      {formatCurrency(data.purchases.standardRatedPurchases)}
                    </span>
                  </div>
                  <div className="flex justify-between items-center py-2 border-b">
                    <span className="font-medium">7. ضريبة القيمة المضافة على المشتريات</span>
                    <span className="font-bold text-blue-600 tabular-nums" data-testid="text-purchases-vat">
                      {formatCurrency(data.purchases.vatOnStandardRatedPurchases)}
                    </span>
                  </div>
                  <div className="flex justify-between items-center py-2 border-b text-muted-foreground">
                    <span>8. المشتريات الخاضعة للضريبة بنسبة صفر</span>
                    <span className="tabular-nums">{formatCurrency(data.purchases.zeroRatedPurchases)}</span>
                  </div>
                  <div className="flex justify-between items-center py-2 border-b text-muted-foreground">
                    <span>9. المشتريات المعفاة من الضريبة</span>
                    <span className="tabular-nums">{formatCurrency(data.purchases.exemptPurchases)}</span>
                  </div>
                  <div className="flex justify-between items-center py-2 bg-muted/50 rounded px-2">
                    <span className="font-bold">10. إجمالي المشتريات</span>
                    <span className="font-bold text-lg tabular-nums">{formatCurrency(data.purchases.totalPurchases)}</span>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>

          <Card>
            <CardHeader className="bg-amber-50 dark:bg-amber-950/20 border-b">
              <CardTitle className="flex items-center gap-2 text-amber-700 dark:text-amber-400">
                <ArrowLeft className="h-5 w-5" />
                التعديلات (المرتجعات)
              </CardTitle>
            </CardHeader>
            <CardContent className="pt-4">
              <div className="grid gap-4 md:grid-cols-2">
                <div className="flex justify-between items-center py-2 border-b">
                  <span>مرتجعات المبيعات</span>
                  <span className="font-bold text-amber-600 tabular-nums" data-testid="text-sales-returns">
                    {formatCurrency(data.adjustments.salesReturns)}
                  </span>
                </div>
                <div className="flex justify-between items-center py-2 border-b">
                  <span>ضريبة المرتجعات</span>
                  <span className="font-bold text-amber-600 tabular-nums" data-testid="text-returns-vat">
                    {formatCurrency(data.adjustments.vatOnSalesReturns)}
                  </span>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="bg-primary/10 border-b">
              <CardTitle className="flex items-center gap-2">
                <Calculator className="h-5 w-5 text-primary" />
                ملخص الإقرار الضريبي
              </CardTitle>
            </CardHeader>
            <CardContent className="pt-4">
              <div className="space-y-4">
                <div className="flex justify-between items-center py-3 border-b">
                  <span className="font-medium">11. إجمالي ضريبة القيمة المضافة المستحقة</span>
                  <span className="font-bold text-lg text-green-600 tabular-nums">
                    {formatCurrency(data.summary.totalVatDue)}
                  </span>
                </div>
                <div className="flex justify-between items-center py-3 border-b">
                  <span className="font-medium">12. إجمالي ضريبة القيمة المضافة القابلة للاسترداد</span>
                  <span className="font-bold text-lg text-blue-600 tabular-nums">
                    ({formatCurrency(data.summary.totalVatRecoverable)})
                  </span>
                </div>
                <Separator />
                <div className="flex justify-between items-center py-3 bg-primary/10 rounded-lg px-4">
                  <span className="font-bold text-lg">
                    13. صافي ضريبة القيمة المضافة {data.summary.netVatPayable >= 0 ? "المستحقة للسداد" : "القابلة للاسترداد"}
                  </span>
                  <span className={`font-bold text-2xl tabular-nums ${data.summary.netVatPayable >= 0 ? "text-primary" : "text-amber-600"}`}>
                    {formatCurrency(Math.abs(data.summary.netVatPayable))}
                  </span>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Building2 className="h-5 w-5" />
                تفصيل حسب نوع النشاط
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="overflow-x-auto">
                <table className="w-full text-sm">
                  <thead>
                    <tr className="border-b bg-muted/50">
                      <th className="text-right py-3 px-4 font-medium">النشاط</th>
                      <th className="text-left py-3 px-4 font-medium">المبلغ قبل الضريبة</th>
                      <th className="text-left py-3 px-4 font-medium">الضريبة</th>
                      <th className="text-left py-3 px-4 font-medium">الإجمالي</th>
                    </tr>
                  </thead>
                  <tbody>
                    <tr className="border-b">
                      <td className="py-3 px-4">مبيعات الخدمات</td>
                      <td className="py-3 px-4 tabular-nums">{formatCurrency(data.breakdown.servicesSales.subtotal)}</td>
                      <td className="py-3 px-4 tabular-nums text-green-600">{formatCurrency(data.breakdown.servicesSales.vat)}</td>
                      <td className="py-3 px-4 tabular-nums font-medium">{formatCurrency(data.breakdown.servicesSales.total)}</td>
                    </tr>
                    <tr className="border-b">
                      <td className="py-3 px-4">مبيعات المنتجات</td>
                      <td className="py-3 px-4 tabular-nums">{formatCurrency(data.breakdown.productsSales.subtotal)}</td>
                      <td className="py-3 px-4 tabular-nums text-green-600">{formatCurrency(data.breakdown.productsSales.vat)}</td>
                      <td className="py-3 px-4 tabular-nums font-medium">{formatCurrency(data.breakdown.productsSales.total)}</td>
                    </tr>
                    <tr className="border-b bg-muted/30">
                      <td className="py-3 px-4 font-medium">إجمالي المبيعات</td>
                      <td className="py-3 px-4 tabular-nums font-medium">
                        {formatCurrency(data.breakdown.servicesSales.subtotal + data.breakdown.productsSales.subtotal)}
                      </td>
                      <td className="py-3 px-4 tabular-nums font-medium text-green-600">
                        {formatCurrency(data.breakdown.servicesSales.vat + data.breakdown.productsSales.vat)}
                      </td>
                      <td className="py-3 px-4 tabular-nums font-bold">
                        {formatCurrency(data.breakdown.servicesSales.total + data.breakdown.productsSales.total)}
                      </td>
                    </tr>
                    <tr className="border-b">
                      <td className="py-3 px-4">فواتير المشتريات</td>
                      <td className="py-3 px-4 tabular-nums">{formatCurrency(data.breakdown.purchaseInvoices.subtotal)}</td>
                      <td className="py-3 px-4 tabular-nums text-blue-600">{formatCurrency(data.breakdown.purchaseInvoices.vat)}</td>
                      <td className="py-3 px-4 tabular-nums font-medium">{formatCurrency(data.breakdown.purchaseInvoices.total)}</td>
                    </tr>
                  </tbody>
                </table>
              </div>
            </CardContent>
          </Card>

          <Card className="print:hidden">
            <CardContent className="pt-6">
              <div className="flex items-start gap-3 text-sm text-muted-foreground">
                <Badge variant="outline" className="shrink-0">ملاحظة</Badge>
                <div>
                  <p className="mb-2">
                    هذا التقرير يُعد لأغراض المراجعة الداخلية فقط. يجب تقديم الإقرار الضريبي الرسمي عبر بوابة هيئة الزكاة والضريبة والجمارك الإلكترونية.
                  </p>
                  <p>
                    نسبة ضريبة القيمة المضافة الأساسية في المملكة العربية السعودية هي 15%.
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </>
      )}
    </div>
  );
}
