import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Timer, Plus, Pencil, Trash2, CheckCircle2, XCircle, Clock, Coffee } from "lucide-react";

interface WorkShift {
  id: number;
  nameAr: string;
  shiftType: string;
  startTime: string;
  endTime: string;
  breakStartTime: string | null;
  breakEndTime: string | null;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
}

interface WorkShiftFormData {
  nameAr: string;
  shiftType: string;
  startTime: string;
  endTime: string;
  breakStartTime: string;
  breakEndTime: string;
  isActive: boolean;
}

const shiftTypes = [
  { value: "morning", label: "صباحية" },
  { value: "evening", label: "مسائية" },
  { value: "night", label: "ليلية" },
  { value: "split", label: "مقسمة" },
  { value: "flexible", label: "مرنة" },
];

export default function WorkShiftsPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingShift, setEditingShift] = useState<WorkShift | null>(null);
  const [formData, setFormData] = useState<WorkShiftFormData>({
    nameAr: "",
    shiftType: "morning",
    startTime: "09:00",
    endTime: "17:00",
    breakStartTime: "",
    breakEndTime: "",
    isActive: true,
  });
  const { toast } = useToast();

  const { data: shifts, isLoading } = useQuery<WorkShift[]>({
    queryKey: ["/api/hr/work-shifts"],
  });

  const createMutation = useMutation({
    mutationFn: (data: WorkShiftFormData) =>
      apiRequest("POST", "/api/hr/work-shifts", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/work-shifts"] });
      toast({ title: "تم إنشاء الوردية بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء الوردية",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<WorkShiftFormData> }) =>
      apiRequest("PUT", `/api/hr/work-shifts/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/work-shifts"] });
      toast({ title: "تم تحديث الوردية بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث الوردية",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/hr/work-shifts/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/work-shifts"] });
      toast({ title: "تم حذف الوردية بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف الوردية",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      nameAr: "",
      shiftType: "morning",
      startTime: "09:00",
      endTime: "17:00",
      breakStartTime: "",
      breakEndTime: "",
      isActive: true,
    });
    setEditingShift(null);
    setIsDialogOpen(false);
  };

  const handleOpenDialog = (shift?: WorkShift) => {
    if (shift) {
      setEditingShift(shift);
      setFormData({
        nameAr: shift.nameAr,
        shiftType: shift.shiftType,
        startTime: shift.startTime,
        endTime: shift.endTime,
        breakStartTime: shift.breakStartTime || "",
        breakEndTime: shift.breakEndTime || "",
        isActive: shift.isActive,
      });
    } else {
      resetForm();
      setIsDialogOpen(true);
      return;
    }
    setIsDialogOpen(true);
  };

  const handleSubmit = () => {
    if (!formData.nameAr.trim()) {
      toast({ title: "اسم الوردية مطلوب", variant: "destructive" });
      return;
    }
    if (!formData.startTime || !formData.endTime) {
      toast({ title: "وقت البداية والنهاية مطلوبان", variant: "destructive" });
      return;
    }

    if (editingShift) {
      updateMutation.mutate({ id: editingShift.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const getShiftTypeLabel = (type: string) => {
    return shiftTypes.find(t => t.value === type)?.label || type;
  };

  const getShiftDuration = (start: string, end: string) => {
    const [startH, startM] = start.split(":").map(Number);
    const [endH, endM] = end.split(":").map(Number);
    let hours = endH - startH;
    let minutes = endM - startM;
    if (minutes < 0) {
      hours--;
      minutes += 60;
    }
    if (hours < 0) hours += 24;
    return `${hours}س ${minutes > 0 ? minutes + "د" : ""}`.trim();
  };

  const activeShifts = shifts?.filter(s => s.isActive) || [];
  const inactiveShifts = shifts?.filter(s => !s.isActive) || [];

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">إدارة الورديات</h1>
          <p className="text-muted-foreground">تعريف أوقات العمل والورديات</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-work-shift">
          <Plus className="h-4 w-4 ml-2" />
          إضافة وردية جديدة
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <Timer className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الورديات</p>
                <p className="text-xl font-bold" data-testid="text-total-shifts">
                  {shifts?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <CheckCircle2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">ورديات مفعّلة</p>
                <p className="text-xl font-bold" data-testid="text-active-shifts">
                  {activeShifts.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                <XCircle className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">ورديات غير مفعّلة</p>
                <p className="text-xl font-bold" data-testid="text-inactive-shifts">
                  {inactiveShifts.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Timer className="h-5 w-5" />
            قائمة الورديات
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">اسم الوردية</TableHead>
                  <TableHead className="text-right">النوع</TableHead>
                  <TableHead className="text-right">وقت العمل</TableHead>
                  <TableHead className="text-right">الاستراحة</TableHead>
                  <TableHead className="text-right">المدة</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-left">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {shifts?.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={7} className="text-center py-8 text-muted-foreground">
                      لا توجد ورديات مسجلة
                    </TableCell>
                  </TableRow>
                ) : (
                  shifts?.map((shift) => (
                    <TableRow key={shift.id} data-testid={`row-work-shift-${shift.id}`}>
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <Timer className="h-4 w-4 text-muted-foreground" />
                          {shift.nameAr}
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant="outline">
                          {getShiftTypeLabel(shift.shiftType)}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1 text-sm">
                          <Clock className="h-3 w-3 text-muted-foreground" />
                          <span dir="ltr">{shift.startTime} - {shift.endTime}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        {shift.breakStartTime && shift.breakEndTime ? (
                          <div className="flex items-center gap-1 text-sm">
                            <Coffee className="h-3 w-3 text-muted-foreground" />
                            <span dir="ltr">{shift.breakStartTime} - {shift.breakEndTime}</span>
                          </div>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell>
                        <Badge variant="secondary">
                          {getShiftDuration(shift.startTime, shift.endTime)}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        {shift.isActive ? (
                          <Badge className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                            مفعّل
                          </Badge>
                        ) : (
                          <Badge className="bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                            غير مفعّل
                          </Badge>
                        )}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleOpenDialog(shift)}
                            data-testid={`button-edit-work-shift-${shift.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => deleteMutation.mutate(shift.id)}
                            disabled={deleteMutation.isPending}
                            data-testid={`button-delete-work-shift-${shift.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-lg" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingShift ? "تعديل الوردية" : "إضافة وردية جديدة"}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="nameAr">اسم الوردية <span className="text-destructive">*</span></Label>
                <Input
                  id="nameAr"
                  value={formData.nameAr}
                  onChange={(e) => setFormData({ ...formData, nameAr: e.target.value })}
                  placeholder="مثال: الفترة الصباحية"
                  data-testid="input-work-shift-name"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="shiftType">نوع الوردية <span className="text-destructive">*</span></Label>
                <Select
                  value={formData.shiftType}
                  onValueChange={(value) => setFormData({ ...formData, shiftType: value })}
                >
                  <SelectTrigger data-testid="select-shift-type">
                    <SelectValue placeholder="اختر النوع" />
                  </SelectTrigger>
                  <SelectContent>
                    {shiftTypes.map((type) => (
                      <SelectItem key={type.value} value={type.value}>
                        {type.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="startTime">وقت البداية <span className="text-destructive">*</span></Label>
                <Input
                  id="startTime"
                  type="time"
                  value={formData.startTime}
                  onChange={(e) => setFormData({ ...formData, startTime: e.target.value })}
                  data-testid="input-start-time"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="endTime">وقت النهاية <span className="text-destructive">*</span></Label>
                <Input
                  id="endTime"
                  type="time"
                  value={formData.endTime}
                  onChange={(e) => setFormData({ ...formData, endTime: e.target.value })}
                  data-testid="input-end-time"
                />
              </div>
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="breakStartTime">بداية الاستراحة</Label>
                <Input
                  id="breakStartTime"
                  type="time"
                  value={formData.breakStartTime}
                  onChange={(e) => setFormData({ ...formData, breakStartTime: e.target.value })}
                  data-testid="input-break-start-time"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="breakEndTime">نهاية الاستراحة</Label>
                <Input
                  id="breakEndTime"
                  type="time"
                  value={formData.breakEndTime}
                  onChange={(e) => setFormData({ ...formData, breakEndTime: e.target.value })}
                  data-testid="input-break-end-time"
                />
              </div>
            </div>
            <div className="flex items-center justify-between">
              <Label htmlFor="isActive">الحالة</Label>
              <div className="flex items-center gap-2">
                <span className="text-sm text-muted-foreground">
                  {formData.isActive ? "مفعّل" : "غير مفعّل"}
                </span>
                <Switch
                  id="isActive"
                  checked={formData.isActive}
                  onCheckedChange={(checked) => setFormData({ ...formData, isActive: checked })}
                  data-testid="switch-work-shift-active"
                />
              </div>
            </div>
          </div>
          <DialogFooter className="gap-2 sm:gap-0">
            <Button
              variant="outline"
              onClick={resetForm}
              data-testid="button-cancel-work-shift"
            >
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-save-work-shift"
            >
              {createMutation.isPending || updateMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
