/**
 * Mobile API Routes for Flutter Application
 * ==========================================
 * جميع الـ endpoints الخاصة بتطبيق الجوال تحت prefix: /mobile
 */

import { Express, Request, Response, NextFunction } from "express";
import bcrypt from "bcryptjs";
import jwt from "jsonwebtoken";
import { storage } from "./storage";
import type {
  MobileUserRole,
  MobileLoginRequest,
  MobileLoginResponse,
  MobileUser,
  MobileClientHomeResponse,
  MobileOrdersListResponse,
  MobileOrder,
  MobileCreateOrderRequest,
  MobileAppointmentsListResponse,
  MobileEmployeeTodayServicesResponse,
  MobileEmployeeAttendanceSummary,
  MobileDriverOrdersResponse,
  MobileLoyaltyResponse,
  MobileValidateCouponRequest,
  MobileValidateCouponResponse,
  MobileApiDocsResponse,
} from "@shared/mobile-api-types";

// JWT Secret - must be set via environment variable
const JWT_SECRET = process.env.SESSION_SECRET;
if (!JWT_SECRET) {
  console.warn("WARNING: SESSION_SECRET not set. Mobile API authentication may not work properly.");
}
const JWT_EXPIRES_IN = "30d";

// ========================================
// MIDDLEWARE
// ========================================

interface AuthenticatedRequest extends Request {
  user?: {
    id: number;
    role: MobileUserRole;
    phone: string;
  };
}

function authenticateToken(req: AuthenticatedRequest, res: Response, next: NextFunction) {
  if (!JWT_SECRET) {
    return res.status(500).json({
      success: false,
      error: "خطأ في إعدادات الخادم",
      errorCode: "SERVER_CONFIG_ERROR",
    });
  }

  const authHeader = req.headers.authorization;
  const token = authHeader?.split(" ")[1];

  if (!token) {
    return res.status(401).json({
      success: false,
      error: "التوكن مطلوب للوصول",
      errorCode: "TOKEN_REQUIRED",
    });
  }

  try {
    const decoded = jwt.verify(token, JWT_SECRET) as {
      id: number;
      role: MobileUserRole;
      phone: string;
    };
    req.user = decoded;
    next();
  } catch (error) {
    return res.status(401).json({
      success: false,
      error: "التوكن غير صالح أو منتهي الصلاحية",
      errorCode: "INVALID_TOKEN",
    });
  }
}

function requireRole(...roles: MobileUserRole[]) {
  return (req: AuthenticatedRequest, res: Response, next: NextFunction) => {
    if (!req.user || !roles.includes(req.user.role)) {
      return res.status(403).json({
        success: false,
        error: "غير مصرح لك بالوصول إلى هذا المورد",
        errorCode: "FORBIDDEN",
      });
    }
    next();
  };
}

// ========================================
// HELPER FUNCTIONS
// ========================================

function toISOString(date: Date | string | null | undefined): string | null {
  if (!date) return null;
  if (typeof date === 'string') return date;
  return date.toISOString();
}

async function generateToken(user: { id: number; role: MobileUserRole; phone: string }) {
  if (!JWT_SECRET) throw new Error("JWT_SECRET not configured");
  return jwt.sign(user, JWT_SECRET, { expiresIn: JWT_EXPIRES_IN });
}

// ========================================
// REGISTER MOBILE ROUTES
// ========================================

export function registerMobileRoutes(app: Express) {
  // ========================================
  // AUTH ENDPOINTS (المصادقة)
  // ========================================

  /**
   * POST /mobile/login
   * تسجيل الدخول بواسطة رقم الجوال + كلمة المرور + نوع الدور
   */
  app.post("/mobile/login", async (req: Request, res: Response) => {
    try {
      if (!JWT_SECRET) {
        return res.status(500).json({
          success: false,
          error: "خطأ في إعدادات الخادم",
          errorCode: "SERVER_CONFIG_ERROR",
        });
      }

      const { phone, password, role, phoneCountryCode = "966" }: MobileLoginRequest = req.body;

      if (!phone || !password || !role) {
        return res.status(400).json({
          success: false,
          error: "رقم الجوال وكلمة المرور ونوع الحساب مطلوبة",
          errorCode: "MISSING_FIELDS",
        });
      }

      if (!["client", "employee", "driver"].includes(role)) {
        return res.status(400).json({
          success: false,
          error: "نوع الحساب غير صحيح",
          errorCode: "INVALID_ROLE",
        });
      }

      let user: MobileUser | null = null;
      let passwordValid = false;

      if (role === "client") {
        // Find client by phone
        const customers = await storage.getCustomers();
        const customer = customers.find(
          (c: any) => c.phone === phone
        );

        if (!customer) {
          return res.status(401).json({
            success: false,
            error: "رقم الجوال غير مسجل",
            errorCode: "USER_NOT_FOUND",
          });
        }

        if (!customer.isActive) {
          return res.status(401).json({
            success: false,
            error: "الحساب غير مفعل",
            errorCode: "ACCOUNT_INACTIVE",
          });
        }

        // Check password using bcrypt - password must be stored as hash
        if (customer.passwordHash) {
          passwordValid = await bcrypt.compare(password, customer.passwordHash);
        }

        if (!passwordValid) {
          return res.status(401).json({
            success: false,
            error: "كلمة المرور غير صحيحة",
            errorCode: "INVALID_CREDENTIALS",
          });
        }

        user = {
          id: customer.id,
          role: "client",
          nameAr: customer.nameAr,
          phone: customer.phone || phone,
          phoneCountryCode: customer.phoneCountryCode || phoneCountryCode,
          email: customer.email,
          profileImageUrl: customer.profileImageUrl,
          loyaltyPointsBalance: customer.loyaltyPointsBalance || 0,
        };
      } else if (role === "employee") {
        // Find employee by phone
        const employees = await storage.getEmployees();
        const employee = employees.find(
          (e: any) => e.phone === phone || e.phoneNumber === phone
        );

        if (!employee) {
          return res.status(401).json({
            success: false,
            error: "رقم الجوال غير مسجل",
            errorCode: "USER_NOT_FOUND",
          });
        }

        if (!employee.isActive) {
          return res.status(401).json({
            success: false,
            error: "الحساب غير مفعل",
            errorCode: "ACCOUNT_INACTIVE",
          });
        }

        // Check password using bcrypt - if employee has passwordHash
        if (employee.passwordHash) {
          passwordValid = await bcrypt.compare(password, employee.passwordHash);
        } else {
          // Fallback: No passwordHash set yet - require proper password setup
          return res.status(401).json({
            success: false,
            error: "لم يتم تعيين كلمة مرور للموظف. يرجى التواصل مع المسؤول.",
            errorCode: "PASSWORD_NOT_SET",
          });
        }

        if (!passwordValid) {
          return res.status(401).json({
            success: false,
            error: "كلمة المرور غير صحيحة",
            errorCode: "INVALID_CREDENTIALS",
          });
        }

        user = {
          id: employee.id,
          role: "employee",
          nameAr: employee.fullNameAr || employee.nameAr || "موظف",
          phone: employee.phone || employee.phoneNumber || phone,
          phoneCountryCode: employee.phoneCountryCode || phoneCountryCode,
          email: employee.email,
          profileImageUrl: employee.profileImageUrl,
          jobTitleAr: employee.jobTitleAr,
          branchId: employee.branchId,
        };
      } else if (role === "driver") {
        // Find driver by phone
        const drivers = await storage.getDrivers();
        const driver = drivers.find(
          (d: any) => d.phone === phone || d.phoneNumber === phone
        );

        if (!driver) {
          return res.status(401).json({
            success: false,
            error: "رقم الجوال غير مسجل",
            errorCode: "USER_NOT_FOUND",
          });
        }

        if (!driver.isActive) {
          return res.status(401).json({
            success: false,
            error: "الحساب غير مفعل",
            errorCode: "ACCOUNT_INACTIVE",
          });
        }

        // Check password using bcrypt - if driver has passwordHash
        if (driver.passwordHash) {
          passwordValid = await bcrypt.compare(password, driver.passwordHash);
        } else {
          // Fallback: No passwordHash set yet - require proper password setup
          return res.status(401).json({
            success: false,
            error: "لم يتم تعيين كلمة مرور للسائق. يرجى التواصل مع المسؤول.",
            errorCode: "PASSWORD_NOT_SET",
          });
        }

        if (!passwordValid) {
          return res.status(401).json({
            success: false,
            error: "كلمة المرور غير صحيحة",
            errorCode: "INVALID_CREDENTIALS",
          });
        }

        user = {
          id: driver.id,
          role: "driver",
          nameAr: driver.fullNameAr || "سائق",
          phone: driver.phone || driver.phoneNumber || phone,
          phoneCountryCode: driver.phoneCountryCode || phoneCountryCode,
          email: driver.email,
          profileImageUrl: driver.profileImageUrl,
          licenseNumber: driver.licenseNumber,
          vehiclePlateNumber: driver.vehiclePlateNumber,
        };
      }

      if (!user) {
        return res.status(401).json({
          success: false,
          error: "حدث خطأ في تسجيل الدخول",
          errorCode: "LOGIN_ERROR",
        });
      }

      const token = await generateToken({
        id: user.id,
        role: user.role,
        phone: user.phone,
      });

      const response: MobileLoginResponse = {
        success: true,
        token,
        user,
        message: "تم تسجيل الدخول بنجاح",
      };

      res.json(response);
    } catch (error) {
      console.error("Mobile login error:", error);
      res.status(500).json({
        success: false,
        error: "حدث خطأ في الخادم",
        errorCode: "SERVER_ERROR",
      });
    }
  });

  /**
   * GET /mobile/me
   * يرجع بيانات المستخدم الحالي حسب التوكن
   */
  app.get("/mobile/me", authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      const { id, role } = req.user!;

      let user: MobileUser | null = null;

      if (role === "client") {
        const customer = await storage.getCustomer(id);
        if (customer) {
          user = {
            id: customer.id,
            role: "client",
            nameAr: customer.nameAr,
            phone: customer.phone || "",
            phoneCountryCode: customer.phoneCountryCode || "966",
            email: customer.email,
            profileImageUrl: customer.profileImageUrl,
            loyaltyPointsBalance: customer.loyaltyPointsBalance || 0,
          };
        }
      } else if (role === "employee") {
        const employee = await storage.getEmployee(id);
        if (employee) {
          user = {
            id: employee.id,
            role: "employee",
            nameAr: employee.fullNameAr || employee.nameAr || "موظف",
            phone: employee.phone || employee.phoneNumber || "",
            phoneCountryCode: employee.phoneCountryCode || "966",
            email: employee.email,
            profileImageUrl: employee.profileImageUrl,
            jobTitleAr: employee.jobTitleAr,
            branchId: employee.branchId,
          };
        }
      } else if (role === "driver") {
        const driver = await storage.getDriver(id);
        if (driver) {
          user = {
            id: driver.id,
            role: "driver",
            nameAr: driver.fullNameAr || "سائق",
            phone: driver.phone || driver.phoneNumber || "",
            phoneCountryCode: driver.phoneCountryCode || "966",
            email: driver.email,
            profileImageUrl: driver.profileImageUrl,
            licenseNumber: driver.licenseNumber,
            vehiclePlateNumber: driver.vehiclePlateNumber,
          };
        }
      }

      if (!user) {
        return res.status(404).json({
          success: false,
          error: "المستخدم غير موجود",
          errorCode: "USER_NOT_FOUND",
        });
      }

      res.json({
        success: true,
        data: user,
      });
    } catch (error) {
      console.error("Mobile /me error:", error);
      res.status(500).json({
        success: false,
        error: "حدث خطأ في الخادم",
        errorCode: "SERVER_ERROR",
      });
    }
  });

  // ========================================
  // CLIENT ENDPOINTS (العميل)
  // ========================================

  /**
   * GET /mobile/client/home
   * الصفحة الرئيسية للعميل
   */
  app.get(
    "/mobile/client/home",
    authenticateToken,
    requireRole("client"),
    async (req: AuthenticatedRequest, res: Response) => {
      try {
        const clientId = req.user!.id;

        // Fetch all required data in parallel using existing storage methods
        const [
          customer,
          categories,
          services,
          packages,
          coupons,
          settings,
        ] = await Promise.all([
          storage.getCustomer(clientId),
          storage.getServiceCategories(),
          storage.getServices(),
          storage.getPackages(),
          storage.getCoupons(),
          storage.getSettings(),
        ]);

        // Get delivery fee setting
        let homeServiceDeliveryFee = 50;
        const feeSetting = settings.find((s: any) => s.key === "home_service_delivery_fee");
        if (feeSetting) homeServiceDeliveryFee = Number(feeSetting.value) || 50;

        // Get active categories
        const activeCategories = categories
          .filter((c: any) => c.isActive)
          .map((c: any) => ({
            id: c.id,
            nameAr: c.nameAr,
            code: c.code,
            imageUrl: null,
          }));

        // Get featured/active services
        const activeServices = services
          .filter((s: any) => s.isActive)
          .slice(0, 10)
          .map((s: any) => ({
            id: s.id,
            categoryId: s.categoryId,
            nameAr: s.nameAr,
            descriptionAr: s.descriptionAr,
            price: Number(s.price) || 0,
            vatType: s.vatType as "inclusive" | "exclusive",
            vatRate: Number(s.vatRate) || 0.15,
            finalPrice: s.vatType === "inclusive" 
              ? Number(s.price) 
              : Number(s.price) * (1 + Number(s.vatRate)),
            images: [],
            isActive: s.isActive,
          }));

        // Get active packages
        const now = new Date();
        const activePackages = packages
          .filter((p: any) => {
            if (!p.isActive) return false;
            const validFrom = new Date(p.validFrom);
            const validTo = new Date(p.validTo);
            return now >= validFrom && now <= validTo;
          })
          .map((p: any) => ({
            id: p.id,
            nameAr: p.nameAr,
            descriptionAr: p.descriptionAr,
            price: Number(p.price) || 0,
            discountPercent: Number(p.discountPercent) || 0,
            originalPrice: Number(p.originalPrice) || 0,
            validFrom: toISOString(p.validFrom) || "",
            validTo: toISOString(p.validTo) || "",
            services: p.services?.map((s: any) => ({
              id: s.serviceId,
              nameAr: s.service?.nameAr || "خدمة",
              quantity: s.quantity,
            })) || [],
          }));

        // Get available coupons
        const availableCoupons = coupons
          .filter((c: any) => {
            if (!c.isActive) return false;
            const startDate = new Date(c.startDate);
            const endDate = new Date(c.endDate);
            return now >= startDate && now <= endDate;
          })
          .map((c: any) => ({
            id: c.id,
            code: c.code,
            nameAr: c.nameAr,
            descriptionAr: c.descriptionAr,
            discountType: c.discountType as "percent" | "fixed",
            discountValue: Number(c.discountValue) || 0,
            startDate: toISOString(c.startDate) || "",
            endDate: toISOString(c.endDate) || "",
          }));

        // Build offers from packages and coupons
        const offers = [
          ...activePackages.slice(0, 3).map((p: any) => ({
            id: p.id,
            type: "package" as const,
            nameAr: p.nameAr,
            descriptionAr: p.descriptionAr,
            imageUrl: null,
            discountPercent: p.discountPercent,
            validUntil: p.validTo,
          })),
          ...availableCoupons.slice(0, 3).map((c: any) => ({
            id: c.id,
            type: "coupon" as const,
            nameAr: c.nameAr,
            descriptionAr: c.descriptionAr,
            imageUrl: null,
            discountPercent: c.discountType === "percent" ? c.discountValue : undefined,
            validUntil: c.endDate,
          })),
        ];

        const response: MobileClientHomeResponse = {
          categories: activeCategories,
          featuredServices: activeServices,
          activePackages,
          offers,
          loyaltyPoints: customer?.loyaltyPointsBalance || 0,
          availableCoupons,
          settings: {
            homeServiceDeliveryFee,
            currency: "ر.س",
            vatRate: 0.15,
          },
        };

        res.json({
          success: true,
          data: response,
        });
      } catch (error) {
        console.error("Mobile client home error:", error);
        res.status(500).json({
          success: false,
          error: "حدث خطأ في جلب البيانات",
          errorCode: "SERVER_ERROR",
        });
      }
    }
  );

  /**
   * GET /mobile/client/orders
   * قائمة طلبات العميل
   */
  app.get(
    "/mobile/client/orders",
    authenticateToken,
    requireRole("client"),
    async (req: AuthenticatedRequest, res: Response) => {
      try {
        const clientId = req.user!.id;
        const page = parseInt(req.query.page as string) || 1;
        const limit = parseInt(req.query.limit as string) || 20;

        // Use existing getOrders method without filter, then filter manually
        const allOrders = await storage.getOrders({});
        
        // Filter by client
        const clientOrders = allOrders.filter((o: any) => o.customerId === clientId);
        
        // Sort by createdAt descending
        clientOrders.sort((a: any, b: any) => 
          new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime()
        );

        const total = clientOrders.length;
        const totalPages = Math.ceil(total / limit);
        const offset = (page - 1) * limit;
        const paginatedOrders = clientOrders.slice(offset, offset + limit);

        const orders = paginatedOrders.map((o: any) => ({
          id: o.id,
          orderNumber: o.orderNumber,
          status: o.status,
          totalAmount: Number(o.totalAmount) || 0,
          itemsCount: (o.services?.length || 0) + (o.products?.length || 0),
          scheduledAt: toISOString(o.scheduledAt),
          createdAt: toISOString(o.createdAt) || new Date().toISOString(),
        }));

        const response: MobileOrdersListResponse = {
          orders,
          pagination: {
            page,
            limit,
            total,
            totalPages,
          },
        };

        res.json({
          success: true,
          data: response,
        });
      } catch (error) {
        console.error("Mobile client orders error:", error);
        res.status(500).json({
          success: false,
          error: "حدث خطأ في جلب الطلبات",
          errorCode: "SERVER_ERROR",
        });
      }
    }
  );

  /**
   * GET /mobile/client/orders/:id
   * تفاصيل طلب معين
   */
  app.get(
    "/mobile/client/orders/:id",
    authenticateToken,
    requireRole("client"),
    async (req: AuthenticatedRequest, res: Response) => {
      try {
        const clientId = req.user!.id;
        const orderId = parseInt(req.params.id);

        const order: any = await storage.getOrder(orderId);

        if (!order) {
          return res.status(404).json({
            success: false,
            error: "الطلب غير موجود",
            errorCode: "ORDER_NOT_FOUND",
          });
        }

        // Verify order belongs to client
        if (order.customerId !== clientId) {
          return res.status(403).json({
            success: false,
            error: "غير مصرح لك بالوصول لهذا الطلب",
            errorCode: "FORBIDDEN",
          });
        }

        const items: any[] = [];

        // Add services
        if (order.services) {
          for (const s of order.services) {
            items.push({
              id: s.id,
              type: "service",
              nameAr: s.service?.nameAr || "خدمة",
              quantity: s.quantity || 1,
              unitPrice: Number(s.unitPrice || s.lineTotal / (s.quantity || 1)) || 0,
              lineTotal: Number(s.lineTotal) || 0,
              vatAmount: Number(s.vatAmount) || 0,
              employeeId: s.employeeId || s.preferredEmployeeId,
              employeeNameAr: s.employee?.fullNameAr || s.employee?.nameAr || s.preferredEmployee?.fullNameAr,
              status: s.status,
            });
          }
        }

        // Add products
        if (order.products) {
          for (const p of order.products) {
            items.push({
              id: p.id,
              type: "product",
              nameAr: p.product?.nameAr || "منتج",
              quantity: p.quantity || 1,
              unitPrice: Number(p.unitPrice || p.lineTotal / (p.quantity || 1)) || 0,
              lineTotal: Number(p.lineTotal) || 0,
              vatAmount: Number(p.vatAmount) || 0,
            });
          }
        }

        const mobileOrder: MobileOrder = {
          id: order.id,
          orderNumber: order.orderNumber,
          source: order.source as "pos" | "app",
          orderType: (order.orderType || "in_salon") as "in_salon" | "home_service",
          status: order.status,
          paymentStatus: order.paymentStatus || "pending",
          scheduledAt: toISOString(order.scheduledAt),
          subtotalAmount: Number(order.subtotalAmount) || 0,
          vatAmount: Number(order.vatAmount) || 0,
          discountAmount: Number(order.discountAmount || 0),
          couponCode: order.couponCode,
          couponDiscountAmount: Number(order.couponDiscountAmount) || 0,
          deliveryFee: Number(order.deliveryFee || 0),
          totalAmount: Number(order.totalAmount) || 0,
          paidAmount: Number(order.paidAmount) || 0,
          remainingAmount: Number(order.remainingAmount || order.totalAmount - order.paidAmount || 0),
          addressText: order.addressText || null,
          notes: order.notes,
          items,
          createdAt: toISOString(order.createdAt) || new Date().toISOString(),
          completedAt: toISOString(order.completedAt),
        };

        res.json({
          success: true,
          data: mobileOrder,
        });
      } catch (error) {
        console.error("Mobile client order detail error:", error);
        res.status(500).json({
          success: false,
          error: "حدث خطأ في جلب تفاصيل الطلب",
          errorCode: "SERVER_ERROR",
        });
      }
    }
  );

  /**
   * POST /mobile/client/orders
   * إنشاء طلب جديد
   */
  app.post(
    "/mobile/client/orders",
    authenticateToken,
    requireRole("client"),
    async (req: AuthenticatedRequest, res: Response) => {
      try {
        const clientId = req.user!.id;
        const orderData: MobileCreateOrderRequest = req.body;

        if (!orderData.items || orderData.items.length === 0) {
          return res.status(400).json({
            success: false,
            error: "يجب إضافة خدمة أو منتج على الأقل",
            errorCode: "EMPTY_ORDER",
          });
        }

        // Validate order type
        if (!["in_salon", "home_service"].includes(orderData.orderType)) {
          return res.status(400).json({
            success: false,
            error: "نوع الطلب غير صحيح",
            errorCode: "INVALID_ORDER_TYPE",
          });
        }

        // If home service, address is required
        if (orderData.orderType === "home_service" && !orderData.addressText) {
          return res.status(400).json({
            success: false,
            error: "العنوان مطلوب للخدمة المنزلية",
            errorCode: "ADDRESS_REQUIRED",
          });
        }

        // Prepare order request for existing createOrder method
        const services: any[] = [];
        const products: any[] = [];

        for (const item of orderData.items) {
          if (item.type === "service") {
            services.push({
              serviceId: item.itemId,
              quantity: item.quantity,
              employeeId: item.preferredEmployeeId,
            });
          } else if (item.type === "product") {
            products.push({
              productId: item.itemId,
              quantity: item.quantity,
            });
          }
        }

        // Create order using existing storage.createOrder method
        const newOrder = await storage.createOrder({
          customerId: clientId,
          source: "app",
          orderType: "normal", // Map to existing order type
          scheduledAt: orderData.scheduledAt,
          services,
          products,
          packageId: orderData.packageId,
          couponCode: orderData.couponCode,
          loyaltyPointsToRedeem: orderData.loyaltyPointsToRedeem,
          notes: orderData.notes,
          // Additional mobile-specific fields stored in notes
          addressText: orderData.addressText,
        } as any);

        res.status(201).json({
          success: true,
          data: {
            orderId: newOrder.id,
            orderNumber: newOrder.orderNumber,
            totalAmount: Number(newOrder.totalAmount) || 0,
          },
          message: "تم إنشاء الطلب بنجاح",
        });
      } catch (error) {
        console.error("Mobile create order error:", error);
        res.status(500).json({
          success: false,
          error: "حدث خطأ في إنشاء الطلب",
          errorCode: "SERVER_ERROR",
        });
      }
    }
  );

  /**
   * GET /mobile/client/appointments
   * قائمة حجوزات العميل
   */
  app.get(
    "/mobile/client/appointments",
    authenticateToken,
    requireRole("client"),
    async (req: AuthenticatedRequest, res: Response) => {
      try {
        const clientId = req.user!.id;
        const page = parseInt(req.query.page as string) || 1;
        const limit = parseInt(req.query.limit as string) || 20;

        const allAppointments = await storage.getAppointments({ clientId });
        
        // Sort by scheduledAt descending
        allAppointments.sort((a: any, b: any) => 
          new Date(b.scheduledAt).getTime() - new Date(a.scheduledAt).getTime()
        );

        const total = allAppointments.length;
        const totalPages = Math.ceil(total / limit);
        const offset = (page - 1) * limit;
        const paginatedAppointments = allAppointments.slice(offset, offset + limit);

        const appointments = paginatedAppointments.map((a: any) => ({
          id: a.id,
          appointmentNumber: a.appointmentNumber,
          appointmentType: a.appointmentType as "in_salon" | "home_service",
          status: a.status,
          scheduledAt: toISOString(a.scheduledAt) || new Date().toISOString(),
          subtotalAmount: Number(a.subtotalAmount) || 0,
          vatAmount: Number(a.vatAmount) || 0,
          totalAmount: Number(a.totalAmount) || 0,
          deliveryFee: Number(a.deliveryFee) || 0,
          couponCode: a.couponCode,
          couponDiscountAmount: Number(a.couponDiscountAmount) || 0,
          addressText: a.addressText,
          notes: a.notes,
          services: a.services?.map((s: any) => ({
            id: s.id,
            serviceId: s.serviceId,
            nameAr: s.service?.nameAr || "خدمة",
            quantity: s.quantity || 1,
            lineTotal: Number(s.lineTotal) || 0,
            preferredEmployeeId: s.preferredEmployeeId,
            preferredEmployeeNameAr: s.preferredEmployee?.fullNameAr || s.preferredEmployee?.nameAr,
          })) || [],
          relatedOrderId: a.relatedOrderId,
          ratingValue: a.ratingValue,
          ratingComment: a.ratingComment,
          createdAt: toISOString(a.createdAt) || new Date().toISOString(),
          completedAt: toISOString(a.completedAt),
        }));

        const response: MobileAppointmentsListResponse = {
          appointments,
          pagination: {
            page,
            limit,
            total,
            totalPages,
          },
        };

        res.json({
          success: true,
          data: response,
        });
      } catch (error) {
        console.error("Mobile client appointments error:", error);
        res.status(500).json({
          success: false,
          error: "حدث خطأ في جلب الحجوزات",
          errorCode: "SERVER_ERROR",
        });
      }
    }
  );

  // ========================================
  // EMPLOYEE ENDPOINTS (الموظفة)
  // ========================================

  /**
   * GET /mobile/employee/today-services
   * جميع الخدمات المسندة للموظفة اليوم
   */
  app.get(
    "/mobile/employee/today-services",
    authenticateToken,
    requireRole("employee"),
    async (req: AuthenticatedRequest, res: Response) => {
      try {
        const employeeId = req.user!.id;
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        const tomorrow = new Date(today);
        tomorrow.setDate(tomorrow.getDate() + 1);

        // Get all orders using existing storage method
        const allOrders = await storage.getOrders({});
        
        // Filter orders for today and extract services assigned to this employee
        const todayServices: any[] = [];
        
        for (const order of allOrders) {
          const orderDate = new Date(order.createdAt);
          if (orderDate >= today && orderDate < tomorrow) {
            if (order.services) {
              for (const service of order.services) {
                const svc = service as any;
                if (svc.employeeId === employeeId || svc.preferredEmployeeId === employeeId) {
                  todayServices.push({
                    id: svc.id,
                    orderId: order.id,
                    orderNumber: order.orderNumber,
                    serviceId: svc.serviceId,
                    serviceNameAr: svc.service?.nameAr || "خدمة",
                    quantity: svc.quantity || 1,
                    lineTotal: Number(svc.lineTotal) || 0,
                    status: svc.status || "pending",
                    customerNameAr: (order as any).customer?.nameAr || "عميل",
                    customerPhone: (order as any).customer?.phone,
                    scheduledAt: toISOString(order.scheduledAt),
                    notes: order.notes,
                    orderType: order.orderType || "in_salon",
                    addressText: (order as any).addressText || null,
                  });
                }
              }
            }
          }
        }

        // Calculate summary
        const summary = {
          total: todayServices.length,
          pending: todayServices.filter((s: any) => s.status === "pending").length,
          inProgress: todayServices.filter((s: any) => s.status === "in_progress").length,
          completed: todayServices.filter((s: any) => s.status === "completed").length,
          cancelled: todayServices.filter((s: any) => s.status === "cancelled").length,
          totalEarnings: todayServices
            .filter((s: any) => s.status === "completed")
            .reduce((sum: number, s: any) => sum + s.lineTotal, 0),
        };

        const response: MobileEmployeeTodayServicesResponse = {
          date: today.toISOString().split("T")[0],
          services: todayServices,
          summary,
        };

        res.json({
          success: true,
          data: response,
        });
      } catch (error) {
        console.error("Mobile employee today-services error:", error);
        res.status(500).json({
          success: false,
          error: "حدث خطأ في جلب الخدمات",
          errorCode: "SERVER_ERROR",
        });
      }
    }
  );

  /**
   * POST /mobile/employee/services/:serviceLineId/status
   * تغيير حالة خدمة
   */
  app.post(
    "/mobile/employee/services/:serviceLineId/status",
    authenticateToken,
    requireRole("employee"),
    async (req: AuthenticatedRequest, res: Response) => {
      try {
        const serviceLineId = parseInt(req.params.serviceLineId);
        const { status, notes } = req.body;

        if (!["pending", "in_progress", "completed", "cancelled"].includes(status)) {
          return res.status(400).json({
            success: false,
            error: "حالة الخدمة غير صحيحة",
            errorCode: "INVALID_STATUS",
          });
        }

        // Update service status using existing updateOrderService method
        const updated = await storage.updateOrderService(serviceLineId, {
          status,
          notes,
        });

        if (!updated) {
          return res.status(404).json({
            success: false,
            error: "الخدمة غير موجودة",
            errorCode: "SERVICE_NOT_FOUND",
          });
        }

        res.json({
          success: true,
          message: "تم تحديث حالة الخدمة بنجاح",
        });
      } catch (error) {
        console.error("Mobile employee update service status error:", error);
        res.status(500).json({
          success: false,
          error: "حدث خطأ في تحديث الحالة",
          errorCode: "SERVER_ERROR",
        });
      }
    }
  );

  /**
   * GET /mobile/employee/attendance-summary
   * ملخص الحضور والإجازات
   */
  app.get(
    "/mobile/employee/attendance-summary",
    authenticateToken,
    requireRole("employee"),
    async (req: AuthenticatedRequest, res: Response) => {
      try {
        const employeeId = req.user!.id;
        const now = new Date();
        const year = now.getFullYear();

        // Get employee for leave balance using existing storage method
        const employee = await storage.getEmployee(employeeId);
        const remainingLeaveDays = employee?.annualLeaveDays || 0;

        // Return summary - in production, implement proper attendance tracking
        const response: MobileEmployeeAttendanceSummary = {
          month: now.toLocaleString("ar-SA", { month: "long" }),
          year,
          totalWorkDays: 30,
          presentDays: 25,
          absentDays: 2,
          lateDays: 1,
          leaveDays: 2,
          overtimeHours: 10,
          remainingLeaveDays,
          attendancePercentage: 83,
        };

        res.json({
          success: true,
          data: response,
        });
      } catch (error) {
        console.error("Mobile employee attendance summary error:", error);
        res.status(500).json({
          success: false,
          error: "حدث خطأ في جلب ملخص الحضور",
          errorCode: "SERVER_ERROR",
        });
      }
    }
  );

  // ========================================
  // DRIVER ENDPOINTS (السائق)
  // ========================================

  /**
   * GET /mobile/driver/orders
   * الطلبات التي تحتاج توصيل
   */
  app.get(
    "/mobile/driver/orders",
    authenticateToken,
    requireRole("driver"),
    async (req: AuthenticatedRequest, res: Response) => {
      try {
        const driverId = req.user!.id;

        // Get trips assigned to this driver using existing storage method
        const trips = await storage.getTrips({ driverId });

        // Get appointments that need delivery using existing storage method
        const appointments = await storage.getAppointments({
          appointmentType: "home_service",
        });

        const driverOrders: any[] = [];

        // Add trips
        for (const trip of trips) {
          if (trip.status !== "completed" && trip.status !== "cancelled") {
            driverOrders.push({
              id: trip.id,
              type: "trip",
              tripNumber: trip.tripNumber,
              status: trip.status,
              addressText: trip.endAddressText || "",
              locationLat: trip.endLocationLat,
              locationLng: trip.endLocationLng,
              scheduledAt: toISOString(trip.scheduledAt),
              notes: trip.notes,
              createdAt: toISOString(trip.createdAt) || new Date().toISOString(),
            });
          }
        }

        // Add pending home service appointments
        for (const appt of appointments) {
          if (appt.status === "confirmed" || appt.status === "pending") {
            driverOrders.push({
              id: appt.id,
              appointmentNumber: appt.appointmentNumber,
              type: "appointment",
              status: "pending",
              customerNameAr: appt.client?.nameAr || "عميل",
              customerPhone: appt.client?.phone,
              addressText: appt.addressText || "",
              locationLat: appt.locationLat,
              locationLng: appt.locationLng,
              scheduledAt: toISOString(appt.scheduledAt),
              deliveryFee: Number(appt.deliveryFee) || 0,
              notes: appt.notes,
              createdAt: toISOString(appt.createdAt) || new Date().toISOString(),
            });
          }
        }

        // Sort by scheduledAt
        driverOrders.sort((a: any, b: any) => {
          const aDate = a.scheduledAt ? new Date(a.scheduledAt) : new Date(a.createdAt);
          const bDate = b.scheduledAt ? new Date(b.scheduledAt) : new Date(b.createdAt);
          return aDate.getTime() - bDate.getTime();
        });

        const summary = {
          total: driverOrders.length,
          pending: driverOrders.filter((o: any) => o.status === "pending").length,
          onTheWay: driverOrders.filter((o: any) => o.status === "on_the_way" || o.status === "in_progress").length,
          delivered: driverOrders.filter((o: any) => o.status === "completed" || o.status === "delivered").length,
        };

        const response: MobileDriverOrdersResponse = {
          orders: driverOrders,
          summary,
        };

        res.json({
          success: true,
          data: response,
        });
      } catch (error) {
        console.error("Mobile driver orders error:", error);
        res.status(500).json({
          success: false,
          error: "حدث خطأ في جلب الطلبات",
          errorCode: "SERVER_ERROR",
        });
      }
    }
  );

  /**
   * POST /mobile/driver/orders/:id/status
   * تغيير حالة توصيل
   */
  app.post(
    "/mobile/driver/orders/:id/status",
    authenticateToken,
    requireRole("driver"),
    async (req: AuthenticatedRequest, res: Response) => {
      try {
        const orderId = parseInt(req.params.id);
        const { status, notes, type } = req.body;

        if (!["pending", "assigned", "picked_up", "on_the_way", "delivered", "cancelled"].includes(status)) {
          return res.status(400).json({
            success: false,
            error: "حالة التوصيل غير صحيحة",
            errorCode: "INVALID_STATUS",
          });
        }

        if (type === "trip") {
          // Update trip status using existing storage method
          await storage.updateTrip(orderId, {
            status: status === "delivered" ? "completed" : status,
            notes,
          });
        } else {
          // Update appointment status using existing storage method
          await storage.updateAppointment(orderId, {
            status: status === "delivered" ? "completed" : status,
            notes,
          });
        }

        res.json({
          success: true,
          message: "تم تحديث حالة التوصيل بنجاح",
        });
      } catch (error) {
        console.error("Mobile driver update status error:", error);
        res.status(500).json({
          success: false,
          error: "حدث خطأ في تحديث الحالة",
          errorCode: "SERVER_ERROR",
        });
      }
    }
  );

  // ========================================
  // LOYALTY & COUPONS ENDPOINTS
  // ========================================

  /**
   * GET /mobile/client/loyalty
   * رصيد النقاط والحركات
   */
  app.get(
    "/mobile/client/loyalty",
    authenticateToken,
    requireRole("client"),
    async (req: AuthenticatedRequest, res: Response) => {
      try {
        const clientId = req.user!.id;

        const customer = await storage.getCustomer(clientId);
        
        // Get loyalty transactions using existing storage method
        const allTransactions = await storage.getAllLoyaltyTransactions();
        const transactions = allTransactions.filter((t: any) => t.customerId === clientId);

        // Get loyalty settings using existing storage method
        const settings = await storage.getSettings();
        let pointsValue = 0.1;
        const valueSetting = settings.find((s: any) => s.key === "loyalty_points_value");
        if (valueSetting) pointsValue = Number(valueSetting.value) || 0.1;

        const loyaltyTransactions = transactions.map((t: any) => ({
          id: t.id,
          type: t.type as "earn" | "redeem" | "expire" | "adjust",
          points: t.points,
          balanceAfter: t.balanceAfter,
          description: t.description || "",
          orderId: t.orderId,
          createdAt: toISOString(t.createdAt) || new Date().toISOString(),
        }));

        const totalEarned = transactions
          .filter((t: any) => t.type === "earn")
          .reduce((sum: number, t: any) => sum + t.points, 0);
        const totalRedeemed = transactions
          .filter((t: any) => t.type === "redeem")
          .reduce((sum: number, t: any) => sum + Math.abs(t.points), 0);

        const response: MobileLoyaltyResponse = {
          balance: customer?.loyaltyPointsBalance || 0,
          totalEarned,
          totalRedeemed,
          pointsValue,
          transactions: loyaltyTransactions,
        };

        res.json({
          success: true,
          data: response,
        });
      } catch (error) {
        console.error("Mobile client loyalty error:", error);
        res.status(500).json({
          success: false,
          error: "حدث خطأ في جلب بيانات الولاء",
          errorCode: "SERVER_ERROR",
        });
      }
    }
  );

  /**
   * GET /mobile/client/coupons
   * الكوبونات المتاحة
   */
  app.get(
    "/mobile/client/coupons",
    authenticateToken,
    requireRole("client"),
    async (req: AuthenticatedRequest, res: Response) => {
      try {
        const coupons = await storage.getCoupons();
        const now = new Date();

        const availableCoupons = coupons
          .filter((c: any) => {
            if (!c.isActive) return false;
            const startDate = new Date(c.startDate);
            const endDate = new Date(c.endDate);
            return now >= startDate && now <= endDate;
          })
          .map((c: any) => ({
            id: c.id,
            code: c.code,
            nameAr: c.nameAr,
            descriptionAr: c.descriptionAr,
            discountType: c.discountType as "percent" | "fixed",
            discountValue: Number(c.discountValue) || 0,
            startDate: toISOString(c.startDate) || "",
            endDate: toISOString(c.endDate) || "",
          }));

        res.json({
          success: true,
          data: availableCoupons,
        });
      } catch (error) {
        console.error("Mobile client coupons error:", error);
        res.status(500).json({
          success: false,
          error: "حدث خطأ في جلب الكوبونات",
          errorCode: "SERVER_ERROR",
        });
      }
    }
  );

  /**
   * POST /mobile/client/validate-coupon
   * التحقق من صلاحية الكوبون
   */
  app.post(
    "/mobile/client/validate-coupon",
    authenticateToken,
    requireRole("client"),
    async (req: AuthenticatedRequest, res: Response) => {
      try {
        const { couponCode, orderSubtotal }: MobileValidateCouponRequest = req.body;

        if (!couponCode) {
          return res.status(400).json({
            success: false,
            error: "رمز الكوبون مطلوب",
            errorCode: "MISSING_COUPON_CODE",
          });
        }

        // Find coupon by code using existing storage method
        const coupons = await storage.getCoupons();
        const coupon = coupons.find(
          (c: any) => c.code.toLowerCase() === couponCode.toLowerCase()
        );

        if (!coupon) {
          const response: MobileValidateCouponResponse = {
            valid: false,
            message: "رمز الكوبون غير صحيح",
            errorCode: "INVALID_CODE",
          };
          return res.json(response);
        }

        // Check if active
        if (!coupon.isActive) {
          const response: MobileValidateCouponResponse = {
            valid: false,
            message: "الكوبون غير مفعل",
            errorCode: "INVALID_CODE",
          };
          return res.json(response);
        }

        // Check dates
        const now = new Date();
        const startDate = new Date(coupon.startDate);
        const endDate = new Date(coupon.endDate);

        if (now < startDate || now > endDate) {
          const response: MobileValidateCouponResponse = {
            valid: false,
            message: "الكوبون منتهي الصلاحية",
            errorCode: "EXPIRED",
          };
          return res.json(response);
        }

        // Calculate discount
        let discountAmount = 0;
        if (coupon.discountType === "percent") {
          discountAmount = (orderSubtotal * Number(coupon.discountValue)) / 100;
        } else {
          discountAmount = Math.min(Number(coupon.discountValue), orderSubtotal);
        }

        const response: MobileValidateCouponResponse = {
          valid: true,
          coupon: {
            id: coupon.id,
            code: coupon.code,
            nameAr: coupon.nameAr,
            descriptionAr: coupon.descriptionAr,
            discountType: coupon.discountType as "percent" | "fixed",
            discountValue: Number(coupon.discountValue),
            startDate: toISOString(coupon.startDate) || "",
            endDate: toISOString(coupon.endDate) || "",
          },
          discountAmount,
          message: "الكوبون صالح للاستخدام",
        };

        res.json(response);
      } catch (error) {
        console.error("Mobile validate coupon error:", error);
        res.status(500).json({
          success: false,
          error: "حدث خطأ في التحقق من الكوبون",
          errorCode: "SERVER_ERROR",
        });
      }
    }
  );

  // ========================================
  // API DOCUMENTATION ENDPOINT
  // ========================================

  /**
   * GET /mobile/docs
   * توثيق الـ API
   */
  app.get("/mobile/docs", (_req: Request, res: Response) => {
    const endpoints: MobileApiDocsResponse = {
      version: "1.0.0",
      baseUrl: "/mobile",
      endpoints: {
        auth: [
          {
            method: "POST",
            path: "/mobile/login",
            description: "User login with phone, password, and role",
            descriptionAr: "تسجيل الدخول برقم الجوال وكلمة المرور ونوع الحساب",
            requiresAuth: false,
            allowedRoles: [],
            requestBody: JSON.stringify({
              phone: "0500000000",
              password: "password123",
              role: "client | employee | driver",
              phoneCountryCode: "966",
            }, null, 2),
            responseExample: JSON.stringify({
              success: true,
              token: "eyJhbGciOiJIUzI1NiIs...",
              user: {
                id: 1,
                role: "client",
                nameAr: "أحمد محمد",
                phone: "0500000000",
              },
            }, null, 2),
          },
          {
            method: "GET",
            path: "/mobile/me",
            description: "Get current user profile",
            descriptionAr: "جلب بيانات المستخدم الحالي",
            requiresAuth: true,
            allowedRoles: ["client", "employee", "driver"],
          },
        ],
        client: [
          {
            method: "GET",
            path: "/mobile/client/home",
            description: "Get client home page data",
            descriptionAr: "جلب بيانات الصفحة الرئيسية للعميل",
            requiresAuth: true,
            allowedRoles: ["client"],
          },
          {
            method: "GET",
            path: "/mobile/client/orders",
            description: "Get client orders list",
            descriptionAr: "جلب قائمة طلبات العميل",
            requiresAuth: true,
            allowedRoles: ["client"],
          },
          {
            method: "GET",
            path: "/mobile/client/orders/:id",
            description: "Get order details",
            descriptionAr: "جلب تفاصيل طلب معين",
            requiresAuth: true,
            allowedRoles: ["client"],
          },
          {
            method: "POST",
            path: "/mobile/client/orders",
            description: "Create new order",
            descriptionAr: "إنشاء طلب جديد",
            requiresAuth: true,
            allowedRoles: ["client"],
            requestBody: JSON.stringify({
              orderType: "in_salon | home_service",
              scheduledAt: "2024-12-04T10:00:00Z",
              items: [
                { type: "service", itemId: 1, quantity: 1, preferredEmployeeId: 2 },
                { type: "product", itemId: 5, quantity: 2 },
              ],
              couponCode: "DISCOUNT10",
              addressText: "الرياض، حي النخيل",
            }, null, 2),
          },
          {
            method: "GET",
            path: "/mobile/client/appointments",
            description: "Get client appointments",
            descriptionAr: "جلب قائمة حجوزات العميل",
            requiresAuth: true,
            allowedRoles: ["client"],
          },
        ],
        employee: [
          {
            method: "GET",
            path: "/mobile/employee/today-services",
            description: "Get today's assigned services",
            descriptionAr: "جلب خدمات اليوم المسندة للموظفة",
            requiresAuth: true,
            allowedRoles: ["employee"],
          },
          {
            method: "POST",
            path: "/mobile/employee/services/:serviceLineId/status",
            description: "Update service status",
            descriptionAr: "تحديث حالة الخدمة",
            requiresAuth: true,
            allowedRoles: ["employee"],
            requestBody: JSON.stringify({
              status: "pending | in_progress | completed | cancelled",
              notes: "ملاحظات اختيارية",
            }, null, 2),
          },
          {
            method: "GET",
            path: "/mobile/employee/attendance-summary",
            description: "Get attendance summary",
            descriptionAr: "جلب ملخص الحضور والإجازات",
            requiresAuth: true,
            allowedRoles: ["employee"],
          },
        ],
        driver: [
          {
            method: "GET",
            path: "/mobile/driver/orders",
            description: "Get delivery orders",
            descriptionAr: "جلب طلبات التوصيل",
            requiresAuth: true,
            allowedRoles: ["driver"],
          },
          {
            method: "POST",
            path: "/mobile/driver/orders/:id/status",
            description: "Update delivery status",
            descriptionAr: "تحديث حالة التوصيل",
            requiresAuth: true,
            allowedRoles: ["driver"],
            requestBody: JSON.stringify({
              status: "pending | assigned | picked_up | on_the_way | delivered | cancelled",
              type: "order | appointment | trip",
              notes: "ملاحظات اختيارية",
            }, null, 2),
          },
        ],
        loyalty: [
          {
            method: "GET",
            path: "/mobile/client/loyalty",
            description: "Get loyalty points and transactions",
            descriptionAr: "جلب رصيد النقاط وحركاتها",
            requiresAuth: true,
            allowedRoles: ["client"],
          },
          {
            method: "GET",
            path: "/mobile/client/coupons",
            description: "Get available coupons",
            descriptionAr: "جلب الكوبونات المتاحة",
            requiresAuth: true,
            allowedRoles: ["client"],
          },
          {
            method: "POST",
            path: "/mobile/client/validate-coupon",
            description: "Validate coupon for order",
            descriptionAr: "التحقق من صلاحية الكوبون",
            requiresAuth: true,
            allowedRoles: ["client"],
            requestBody: JSON.stringify({
              couponCode: "DISCOUNT10",
              orderSubtotal: 500,
              orderItems: [
                { type: "service", itemId: 1, quantity: 1, price: 300 },
                { type: "product", itemId: 5, quantity: 2, price: 100 },
              ],
            }, null, 2),
          },
        ],
      },
    };

    res.json(endpoints);
  });

  console.log("Mobile API routes registered successfully");
}
