import type { Express, Request, Response } from "express";
import { createServer, type Server } from "http";
import { storage } from "./storage";
import multer from "multer";
import path from "path";
import fs from "fs";
import { registerMobileRoutes } from "./mobile-api";
import { 
  insertSupplierSchema, 
  insertUnitSchema, 
  insertProductSchema,
  insertPaymentMethodSchema,
  insertPurchaseInvoiceSchema,
  insertPurchaseInvoiceLineSchema,
  insertPurchasePaymentSchema,
  insertServiceCategorySchema,
  insertServiceSchema,
  insertServiceProductSchema,
  updateServiceCategorySchema,
  updateServiceSchema,
  relatedProductsArraySchema,
  insertEmployeeSchema,
  updateEmployeeSchema,
  insertOrderRatingSchema,
  updateOrderSchema,
  updateOrderServiceSchema,
  insertOrderReturnSchema,
  orderStatuses,
  orderSources,
  orderTypes,
  insertRoleSchema,
  updateRoleSchema,
  insertBrandSchema,
  updateBrandSchema,
  systemModules,
  insertEmployeeTaskSchema,
  updateEmployeeTaskSchema,
  insertSalaryProfileSchema,
  insertPayrollRunSchema,
  insertCafeteriaProductSchema,
  insertCafeteriaPurchasePaymentSchema,
  updateCafeteriaSalesOrderSchema,
  cafeteriaOrderStatuses,
  insertDriverSchema,
  insertTripSchema,
  insertCashDisbursementSchema,
  type CreatePurchaseInvoiceRequest,
  type CreateOrderRequest,
  type OrderFilters,
  type CreateCafeteriaPurchaseInvoiceRequest,
  type CreateCafeteriaSalesOrderRequest,
  type CafeteriaOrderFilters,
} from "@shared/schema";
import { z } from "zod";

// Helper to translate database constraint errors to Arabic
function translateDbError(error: any): string | null {
  const message = error?.message || "";
  if (message.includes("duplicate key") || message.includes("unique constraint")) {
    if (message.includes("code")) {
      return "الرمز مستخدم مسبقاً";
    }
    return "القيمة مكررة";
  }
  if (message.includes("foreign key")) {
    return "السجل المرتبط غير موجود";
  }
  return null;
}

// Multer configuration for service image uploads
const uploadsDir = path.join(process.cwd(), "uploads", "services");
if (!fs.existsSync(uploadsDir)) {
  fs.mkdirSync(uploadsDir, { recursive: true });
}

const serviceImageStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, uploadsDir);
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
    const ext = path.extname(file.originalname);
    cb(null, `service-${uniqueSuffix}${ext}`);
  },
});

const uploadServiceImage = multer({
  storage: serviceImageStorage,
  limits: { fileSize: 5 * 1024 * 1024 }, // 5MB limit
  fileFilter: (req, file, cb) => {
    const allowedTypes = ["image/jpeg", "image/png", "image/gif", "image/webp"];
    if (allowedTypes.includes(file.mimetype)) {
      cb(null, true);
    } else {
      cb(new Error("نوع الملف غير مدعوم. الأنواع المسموحة: JPEG, PNG, GIF, WebP"));
    }
  },
});

// Multer configuration for brand image uploads
const brandUploadsDir = path.join(process.cwd(), "uploads", "brands");
if (!fs.existsSync(brandUploadsDir)) {
  fs.mkdirSync(brandUploadsDir, { recursive: true });
}

const brandImageStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, brandUploadsDir);
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
    const ext = path.extname(file.originalname);
    cb(null, `brand-${uniqueSuffix}${ext}`);
  },
});

const uploadBrandImage = multer({
  storage: brandImageStorage,
  limits: { fileSize: 5 * 1024 * 1024 }, // 5MB limit
  fileFilter: (req, file, cb) => {
    const allowedTypes = ["image/jpeg", "image/png", "image/gif", "image/webp"];
    if (allowedTypes.includes(file.mimetype)) {
      cb(null, true);
    } else {
      cb(new Error("نوع الملف غير مدعوم. الأنواع المسموحة: JPEG, PNG, GIF, WebP"));
    }
  },
});

export async function registerRoutes(
  httpServer: Server,
  app: Express
): Promise<Server> {
  
  // ========================================
  // MOBILE API
  // ========================================
  registerMobileRoutes(app);
  
  // ========================================
  // DASHBOARD
  // ========================================
  
  app.get("/api/dashboard/stats", async (req: Request, res: Response) => {
    try {
      const stats = await storage.getDashboardStats();
      res.json(stats);
    } catch (error) {
      console.error("Error fetching dashboard stats:", error);
      res.status(500).json({ error: "Failed to fetch dashboard stats" });
    }
  });
  
  app.get("/api/dashboard/summary", async (req: Request, res: Response) => {
    try {
      const { from, to, branch_id, source } = req.query;
      
      const filters = {
        fromDate: from ? String(from) : undefined,
        toDate: to ? String(to) : undefined,
        branchId: branch_id ? parseInt(String(branch_id)) : undefined,
        source: source && ["all", "pos", "app"].includes(String(source)) 
          ? String(source) as "all" | "pos" | "app" 
          : "all" as const,
      };
      
      const summary = await storage.getSalonDashboardSummary(filters);
      res.json(summary);
    } catch (error) {
      console.error("Error fetching salon dashboard summary:", error);
      res.status(500).json({ error: "فشل في جلب إحصائيات لوحة التحكم" });
    }
  });
  
  // Daily Report (تقرير اليومية)
  app.get("/api/reports/daily", async (req: Request, res: Response) => {
    try {
      const { date, branch_id } = req.query;
      
      if (!date) {
        return res.status(400).json({ error: "التاريخ مطلوب" });
      }
      
      const filters = {
        date: String(date),
        branchId: branch_id ? parseInt(String(branch_id)) : undefined,
      };
      
      const report = await storage.getSalonDailyReport(filters);
      res.json(report);
    } catch (error) {
      console.error("Error fetching daily report:", error);
      res.status(500).json({ error: "فشل في جلب تقرير اليومية" });
    }
  });
  
  // Marketing Report (تقارير التسويق)
  app.get("/api/reports/marketing", async (req: Request, res: Response) => {
    try {
      const { from_date, to_date } = req.query;
      
      const filters = {
        fromDate: from_date ? String(from_date) : undefined,
        toDate: to_date ? String(to_date) : undefined,
      };
      
      const report = await storage.getMarketingReport(filters);
      res.json(report);
    } catch (error) {
      console.error("Error fetching marketing report:", error);
      res.status(500).json({ error: "فشل في جلب تقرير التسويق" });
    }
  });
  
  // ========================================
  // SUPPLIERS
  // ========================================
  
  app.get("/api/suppliers", async (req: Request, res: Response) => {
    try {
      const suppliers = await storage.getSuppliers();
      res.json(suppliers);
    } catch (error) {
      console.error("Error fetching suppliers:", error);
      res.status(500).json({ error: "Failed to fetch suppliers" });
    }
  });
  
  app.get("/api/suppliers/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const supplier = await storage.getSupplier(id);
      if (!supplier) {
        return res.status(404).json({ error: "Supplier not found" });
      }
      res.json(supplier);
    } catch (error) {
      console.error("Error fetching supplier:", error);
      res.status(500).json({ error: "Failed to fetch supplier" });
    }
  });
  
  app.post("/api/suppliers", async (req: Request, res: Response) => {
    try {
      const { bankAccounts, ...supplierData } = req.body;
      const validated = insertSupplierSchema.parse(supplierData);
      const supplier = await storage.createSupplier(validated, bankAccounts);
      res.status(201).json(supplier);
    } catch (error) {
      console.error("Error creating supplier:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "Validation error", details: error.errors });
      }
      res.status(500).json({ error: "Failed to create supplier" });
    }
  });
  
  app.put("/api/suppliers/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { bankAccounts, ...supplierData } = req.body;
      const supplier = await storage.updateSupplier(id, supplierData, bankAccounts);
      if (!supplier) {
        return res.status(404).json({ error: "Supplier not found" });
      }
      res.json(supplier);
    } catch (error) {
      console.error("Error updating supplier:", error);
      res.status(500).json({ error: "Failed to update supplier" });
    }
  });
  
  app.delete("/api/suppliers/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteSupplier(id);
      if (!success) {
        return res.status(404).json({ error: "Supplier not found" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting supplier:", error);
      res.status(500).json({ error: "Failed to delete supplier" });
    }
  });
  
  // ========================================
  // UNITS
  // ========================================
  
  app.get("/api/units", async (req: Request, res: Response) => {
    try {
      const units = await storage.getUnits();
      res.json(units);
    } catch (error) {
      console.error("Error fetching units:", error);
      res.status(500).json({ error: "Failed to fetch units" });
    }
  });
  
  app.get("/api/units/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const unit = await storage.getUnit(id);
      if (!unit) {
        return res.status(404).json({ error: "Unit not found" });
      }
      res.json(unit);
    } catch (error) {
      console.error("Error fetching unit:", error);
      res.status(500).json({ error: "Failed to fetch unit" });
    }
  });
  
  app.post("/api/units", async (req: Request, res: Response) => {
    try {
      const validated = insertUnitSchema.parse(req.body);
      const unit = await storage.createUnit(validated);
      res.status(201).json(unit);
    } catch (error) {
      console.error("Error creating unit:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "Validation error", details: error.errors });
      }
      res.status(500).json({ error: "Failed to create unit" });
    }
  });
  
  app.put("/api/units/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const unit = await storage.updateUnit(id, req.body);
      if (!unit) {
        return res.status(404).json({ error: "Unit not found" });
      }
      res.json(unit);
    } catch (error) {
      console.error("Error updating unit:", error);
      res.status(500).json({ error: "Failed to update unit" });
    }
  });
  
  app.delete("/api/units/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteUnit(id);
      if (!success) {
        return res.status(404).json({ error: "Unit not found" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting unit:", error);
      res.status(500).json({ error: "Failed to delete unit" });
    }
  });
  
  // Seed default units
  app.post("/api/units/seed", async (req: Request, res: Response) => {
    try {
      const defaultUnits = [
        { nameAr: "قطعة", code: "PC" },
        { nameAr: "علبة", code: "BOX" },
        { nameAr: "كرتون", code: "CTN" },
        { nameAr: "زجاجة", code: "BTL" },
        { nameAr: "أنبوب", code: "TUBE" },
        { nameAr: "كيلوجرام", code: "KG" },
        { nameAr: "جرام", code: "G" },
        { nameAr: "لتر", code: "L" },
        { nameAr: "مليلتر", code: "ML" },
        { nameAr: "متر", code: "M" },
        { nameAr: "درزن", code: "DZ" },
        { nameAr: "طقم", code: "SET" },
      ];
      
      const existingUnits = await storage.getUnits();
      const newUnits = [];
      
      for (const unit of defaultUnits) {
        const exists = existingUnits.some(u => u.code === unit.code);
        if (!exists) {
          const created = await storage.createUnit(unit);
          newUnits.push(created);
        }
      }
      
      res.json({ created: newUnits.length, units: newUnits });
    } catch (error) {
      console.error("Error seeding units:", error);
      res.status(500).json({ error: "Failed to seed units" });
    }
  });
  
  // ========================================
  // PRODUCTS
  // ========================================
  
  app.get("/api/products", async (req: Request, res: Response) => {
    try {
      const products = await storage.getProducts();
      res.json(products);
    } catch (error) {
      console.error("Error fetching products:", error);
      res.status(500).json({ error: "Failed to fetch products" });
    }
  });
  
  app.get("/api/products/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const product = await storage.getProduct(id);
      if (!product) {
        return res.status(404).json({ error: "Product not found" });
      }
      res.json(product);
    } catch (error) {
      console.error("Error fetching product:", error);
      res.status(500).json({ error: "Failed to fetch product" });
    }
  });
  
  app.post("/api/products", async (req: Request, res: Response) => {
    try {
      const validated = insertProductSchema.parse(req.body);
      const product = await storage.createProduct(validated);
      res.status(201).json(product);
    } catch (error) {
      console.error("Error creating product:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "Validation error", details: error.errors });
      }
      res.status(500).json({ error: "Failed to create product" });
    }
  });
  
  app.put("/api/products/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const product = await storage.updateProduct(id, req.body);
      if (!product) {
        return res.status(404).json({ error: "Product not found" });
      }
      res.json(product);
    } catch (error) {
      console.error("Error updating product:", error);
      res.status(500).json({ error: "Failed to update product" });
    }
  });
  
  app.delete("/api/products/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteProduct(id);
      if (!success) {
        return res.status(404).json({ error: "Product not found" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting product:", error);
      res.status(500).json({ error: "Failed to delete product" });
    }
  });
  
  // ========================================
  // PAYMENT METHODS
  // ========================================
  
  app.get("/api/payment-methods", async (req: Request, res: Response) => {
    try {
      const methods = await storage.getPaymentMethods();
      res.json(methods);
    } catch (error) {
      console.error("Error fetching payment methods:", error);
      res.status(500).json({ error: "Failed to fetch payment methods" });
    }
  });
  
  app.post("/api/payment-methods", async (req: Request, res: Response) => {
    try {
      const validated = insertPaymentMethodSchema.parse(req.body);
      const method = await storage.createPaymentMethod(validated);
      res.status(201).json(method);
    } catch (error) {
      console.error("Error creating payment method:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "Validation error", details: error.errors });
      }
      res.status(500).json({ error: "Failed to create payment method" });
    }
  });
  
  app.put("/api/payment-methods/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const method = await storage.updatePaymentMethod(id, req.body);
      if (!method) {
        return res.status(404).json({ error: "Payment method not found" });
      }
      res.json(method);
    } catch (error) {
      console.error("Error updating payment method:", error);
      res.status(500).json({ error: "Failed to update payment method" });
    }
  });
  
  app.delete("/api/payment-methods/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deletePaymentMethod(id);
      if (!success) {
        return res.status(404).json({ error: "Payment method not found" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting payment method:", error);
      res.status(500).json({ error: "Failed to delete payment method" });
    }
  });
  
  // Seed default payment methods
  app.post("/api/payment-methods/seed", async (req: Request, res: Response) => {
    try {
      const defaultMethods = [
        { nameAr: "نقداً", code: "CASH", type: "cash" },
        { nameAr: "تحويل بنكي", code: "BANK", type: "bank" },
        { nameAr: "بطاقة ائتمان", code: "CARD", type: "card" },
        { nameAr: "شيك", code: "CHECK", type: "bank" },
      ];
      
      const existingMethods = await storage.getPaymentMethods();
      const newMethods = [];
      
      for (const method of defaultMethods) {
        const exists = existingMethods.some(m => m.code === method.code);
        if (!exists) {
          const created = await storage.createPaymentMethod(method);
          newMethods.push(created);
        }
      }
      
      res.json({ created: newMethods.length, methods: newMethods });
    } catch (error) {
      console.error("Error seeding payment methods:", error);
      res.status(500).json({ error: "Failed to seed payment methods" });
    }
  });
  
  // ========================================
  // PURCHASE INVOICES
  // ========================================
  
  app.get("/api/purchase-invoices", async (req: Request, res: Response) => {
    try {
      const invoices = await storage.getPurchaseInvoices();
      res.json(invoices);
    } catch (error) {
      console.error("Error fetching purchase invoices:", error);
      res.status(500).json({ error: "Failed to fetch purchase invoices" });
    }
  });
  
  app.get("/api/purchase-invoices/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const invoice = await storage.getPurchaseInvoice(id);
      if (!invoice) {
        return res.status(404).json({ error: "Invoice not found" });
      }
      res.json(invoice);
    } catch (error) {
      console.error("Error fetching purchase invoice:", error);
      res.status(500).json({ error: "Failed to fetch purchase invoice" });
    }
  });
  
  app.post("/api/purchase-invoices", async (req: Request, res: Response) => {
    try {
      const request = req.body as CreatePurchaseInvoiceRequest;
      
      // Create or get supplier
      let supplierId = request.supplier.id;
      if (!supplierId && request.supplier.nameAr) {
        const newSupplier = await storage.createSupplier({
          nameAr: request.supplier.nameAr,
          crNumber: request.supplier.crNumber,
          vatNumber: request.supplier.vatNumber,
          cityAr: request.supplier.cityAr,
          supplierType: request.supplier.supplierType,
          imageUrl: request.supplier.imageUrl,
          representativeName: request.supplier.representativeName,
          representativePhone: request.supplier.representativePhone,
          isActive: true,
        });
        supplierId = newSupplier.id;
      }
      
      if (!supplierId) {
        return res.status(400).json({ error: "Supplier is required" });
      }
      
      // Get existing units
      const allUnits = await storage.getUnits();
      
      // Process lines and create products/units as needed
      const processedLines = [];
      let subtotalAmount = 0;
      let vatAmount = 0;
      
      for (const line of request.lines) {
        // Create or get product
        let productId = line.product.id;
        let purchaseUnitId: number | null = null;
        let sellUnitId: number | null = null;
        let conversionFactor = line.product.conversionFactor || 1;
        
        if (!productId && line.product.nameAr) {
          // Create purchase unit if needed
          if (line.product.purchaseUnitNameAr) {
            const existingPurchaseUnit = allUnits.find(u => u.nameAr === line.product.purchaseUnitNameAr);
            if (existingPurchaseUnit) {
              purchaseUnitId = existingPurchaseUnit.id;
            } else {
              const newUnit = await storage.createUnit({ nameAr: line.product.purchaseUnitNameAr });
              purchaseUnitId = newUnit.id;
              allUnits.push(newUnit);
            }
          }
          
          // Create sell unit if needed
          if (line.product.sellUnitNameAr) {
            const existingSellUnit = allUnits.find(u => u.nameAr === line.product.sellUnitNameAr);
            if (existingSellUnit) {
              sellUnitId = existingSellUnit.id;
            } else {
              const newUnit = await storage.createUnit({ nameAr: line.product.sellUnitNameAr });
              sellUnitId = newUnit.id;
              allUnits.push(newUnit);
            }
          }
          
          // Create product
          const newProduct = await storage.createProduct({
            nameAr: line.product.nameAr,
            purchaseUnitId,
            sellUnitId,
            conversionFactor,
            hasExpiryDate: line.product.hasExpiryDate || false,
            defaultSellPrice: line.product.defaultSellPrice,
            usageType: "sale_and_consume",
            isActive: true,
          });
          productId = newProduct.id;
        } else if (productId) {
          const existingProduct = await storage.getProduct(productId);
          if (existingProduct) {
            purchaseUnitId = existingProduct.purchaseUnitId;
            sellUnitId = existingProduct.sellUnitId;
            conversionFactor = existingProduct.conversionFactor;
          }
        }
        
        if (!productId) {
          return res.status(400).json({ error: "Product is required for each line" });
        }
        
        // Calculate quantities and prices
        const quantityPurchaseUnit = line.quantityPurchaseUnit;
        const quantitySellUnit = quantityPurchaseUnit * conversionFactor;
        const purchasePricePerPurchaseUnit = line.purchasePricePerPurchaseUnit;
        const purchasePricePerSellUnit = purchasePricePerPurchaseUnit / conversionFactor;
        
        // Calculate VAT
        let lineSubtotal = quantityPurchaseUnit * purchasePricePerPurchaseUnit;
        let lineVatAmount = 0;
        
        if (line.vatType === "exclusive") {
          lineVatAmount = lineSubtotal * line.vatRate;
        } else if (line.vatType === "inclusive") {
          const priceWithoutVat = lineSubtotal / (1 + line.vatRate);
          lineVatAmount = lineSubtotal - priceWithoutVat;
          lineSubtotal = priceWithoutVat;
        }
        
        const lineTotal = lineSubtotal + lineVatAmount;
        
        subtotalAmount += lineSubtotal;
        vatAmount += lineVatAmount;
        
        processedLines.push({
          productId,
          purchaseUnitId: purchaseUnitId || 1,
          sellUnitId: sellUnitId || 1,
          quantityPurchaseUnit,
          conversionFactor,
          quantitySellUnit,
          purchasePricePerPurchaseUnit,
          purchasePricePerSellUnit,
          vatType: line.vatType,
          vatRate: line.vatRate,
          vatAmount: lineVatAmount,
          lineSubtotal,
          lineTotal,
          expiryDate: line.expiryDate,
        });
      }
      
      const totalAmount = subtotalAmount + vatAmount;
      
      // Auto-calculate due_date from supplier's payment_term_days if not provided
      let dueDate = request.dueDate;
      if (!dueDate) {
        const supplier = await storage.getSupplier(supplierId);
        if (supplier && supplier.paymentTermDays !== null && supplier.paymentTermDays !== undefined) {
          if (!request.invoiceDate) {
            return res.status(400).json({ error: "تاريخ الفاتورة مطلوب لحساب تاريخ الاستحقاق" });
          }
          const invoiceDate = new Date(request.invoiceDate);
          invoiceDate.setDate(invoiceDate.getDate() + supplier.paymentTermDays);
          dueDate = invoiceDate.toISOString().split("T")[0];
        }
      }
      
      // Create invoice
      const invoice = await storage.createPurchaseInvoice(
        {
          supplierId,
          supplierRepresentativeName: request.supplier.representativeName,
          invoiceDate: request.invoiceDate,
          dueDate,
          paymentStatus: "unpaid",
          subtotalAmount,
          vatAmount,
          totalAmount,
          notes: request.notes,
        },
        processedLines
      );
      
      res.status(201).json(invoice);
    } catch (error) {
      console.error("Error creating purchase invoice:", error);
      res.status(500).json({ error: "Failed to create purchase invoice" });
    }
  });
  
  // Add payment to invoice
  app.post("/api/purchase-invoices/:id/payments", async (req: Request, res: Response) => {
    try {
      const invoiceId = parseInt(req.params.id);
      const invoice = await storage.getPurchaseInvoice(invoiceId);
      
      if (!invoice) {
        return res.status(404).json({ error: "Invoice not found" });
      }
      
      const paymentData = {
        purchaseInvoiceId: invoiceId,
        supplierId: invoice.supplierId,
        paymentMethodId: req.body.paymentMethodId,
        paymentDate: req.body.paymentDate,
        amount: req.body.amount,
        notes: req.body.notes,
      };
      
      const payment = await storage.createPayment(paymentData);
      res.status(201).json(payment);
    } catch (error) {
      console.error("Error creating payment:", error);
      res.status(500).json({ error: "Failed to create payment" });
    }
  });
  
  // ========================================
  // PAYMENTS
  // ========================================
  
  app.get("/api/payments", async (req: Request, res: Response) => {
    try {
      const supplierId = req.query.supplierId ? parseInt(req.query.supplierId as string) : undefined;
      const payments = await storage.getPayments(supplierId);
      res.json(payments);
    } catch (error) {
      console.error("Error fetching payments:", error);
      res.status(500).json({ error: "Failed to fetch payments" });
    }
  });
  
  app.post("/api/payments", async (req: Request, res: Response) => {
    try {
      const validated = insertPurchasePaymentSchema.parse(req.body);
      const payment = await storage.createPayment(validated);
      res.status(201).json(payment);
    } catch (error) {
      console.error("Error creating payment:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "Validation error", details: error.errors });
      }
      res.status(500).json({ error: "Failed to create payment" });
    }
  });
  
  // ========================================
  // PURCHASE DEBIT NOTES (إشعارات الخصم للمشتريات)
  // ========================================
  
  app.get("/api/purchase-debit-notes", async (req: Request, res: Response) => {
    try {
      const debitNotes = await storage.getPurchaseDebitNotes();
      res.json(debitNotes);
    } catch (error) {
      console.error("Error fetching purchase debit notes:", error);
      res.status(500).json({ error: "فشل في جلب إشعارات الخصم" });
    }
  });
  
  app.get("/api/purchase-debit-notes/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const debitNote = await storage.getPurchaseDebitNote(id);
      if (!debitNote) {
        return res.status(404).json({ error: "إشعار الخصم غير موجود" });
      }
      res.json(debitNote);
    } catch (error) {
      console.error("Error fetching purchase debit note:", error);
      res.status(500).json({ error: "فشل في جلب إشعار الخصم" });
    }
  });
  
  app.post("/api/purchase-debit-notes", async (req: Request, res: Response) => {
    try {
      const { lines, ...debitNoteData } = req.body;
      
      if (!lines || !Array.isArray(lines) || lines.length === 0) {
        return res.status(400).json({ error: "يجب إضافة منتج واحد على الأقل" });
      }
      
      // Calculate totals from lines
      let subtotalAmount = 0;
      let vatAmount = 0;
      
      for (const line of lines) {
        subtotalAmount += line.lineSubtotal;
        vatAmount += line.vatAmount;
      }
      
      const totalAmount = subtotalAmount + vatAmount;
      
      const debitNote = await storage.createPurchaseDebitNote(
        {
          ...debitNoteData,
          debitNoteNumber: '', // Will be generated
          subtotalAmount,
          vatAmount,
          totalAmount,
          status: 'draft',
        },
        lines
      );
      
      res.status(201).json(debitNote);
    } catch (error) {
      console.error("Error creating purchase debit note:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "خطأ في البيانات", details: error.errors });
      }
      res.status(500).json({ error: "فشل في إنشاء إشعار الخصم" });
    }
  });
  
  app.post("/api/purchase-debit-notes/:id/confirm", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const debitNote = await storage.confirmPurchaseDebitNote(id);
      if (!debitNote) {
        return res.status(404).json({ error: "إشعار الخصم غير موجود أو تم تأكيده مسبقاً" });
      }
      res.json(debitNote);
    } catch (error) {
      console.error("Error confirming purchase debit note:", error);
      res.status(500).json({ error: "فشل في تأكيد إشعار الخصم" });
    }
  });
  
  app.post("/api/purchase-debit-notes/:id/cancel", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const debitNote = await storage.cancelPurchaseDebitNote(id);
      if (!debitNote) {
        return res.status(404).json({ error: "إشعار الخصم غير موجود أو لا يمكن إلغاؤه" });
      }
      res.json(debitNote);
    } catch (error) {
      console.error("Error cancelling purchase debit note:", error);
      res.status(500).json({ error: "فشل في إلغاء إشعار الخصم" });
    }
  });

  // ========================================
  // INVENTORY WITHDRAWAL ORDERS (أوامر سحب المخزون)
  // ========================================

  app.get("/api/inventory-withdrawals", async (req: Request, res: Response) => {
    try {
      const withdrawals = await storage.getInventoryWithdrawalOrders();
      res.json(withdrawals);
    } catch (error) {
      console.error("Error fetching inventory withdrawals:", error);
      res.status(500).json({ error: "فشل في جلب أوامر سحب المخزون" });
    }
  });

  app.get("/api/inventory-withdrawals/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const withdrawal = await storage.getInventoryWithdrawalOrder(id);
      if (!withdrawal) {
        return res.status(404).json({ error: "أمر السحب غير موجود" });
      }
      res.json(withdrawal);
    } catch (error) {
      console.error("Error fetching inventory withdrawal:", error);
      res.status(500).json({ error: "فشل في جلب أمر السحب" });
    }
  });

  app.post("/api/inventory-withdrawals", async (req: Request, res: Response) => {
    try {
      const { lines, ...orderData } = req.body;
      
      if (!lines || !Array.isArray(lines) || lines.length === 0) {
        return res.status(400).json({ error: "يجب إضافة منتج واحد على الأقل" });
      }
      
      const withdrawal = await storage.createInventoryWithdrawalOrder(orderData, lines);
      res.status(201).json(withdrawal);
    } catch (error: any) {
      console.error("Error creating inventory withdrawal:", error);
      res.status(500).json({ error: error.message || "فشل في إنشاء أمر السحب" });
    }
  });

  app.patch("/api/inventory-withdrawals/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const withdrawal = await storage.updateInventoryWithdrawalOrder(id, req.body);
      if (!withdrawal) {
        return res.status(404).json({ error: "أمر السحب غير موجود" });
      }
      res.json(withdrawal);
    } catch (error: any) {
      console.error("Error updating inventory withdrawal:", error);
      res.status(500).json({ error: error.message || "فشل في تحديث أمر السحب" });
    }
  });

  app.delete("/api/inventory-withdrawals/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteInventoryWithdrawalOrder(id);
      if (!success) {
        return res.status(404).json({ error: "أمر السحب غير موجود" });
      }
      res.json({ success: true });
    } catch (error: any) {
      console.error("Error deleting inventory withdrawal:", error);
      res.status(500).json({ error: error.message || "فشل في حذف أمر السحب" });
    }
  });

  app.post("/api/inventory-withdrawals/:id/lines", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const line = await storage.addInventoryWithdrawalLine(id, req.body);
      res.status(201).json(line);
    } catch (error: any) {
      console.error("Error adding inventory withdrawal line:", error);
      res.status(500).json({ error: error.message || "فشل في إضافة السطر" });
    }
  });

  app.delete("/api/inventory-withdrawal-lines/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.removeInventoryWithdrawalLine(id);
      if (!success) {
        return res.status(404).json({ error: "السطر غير موجود" });
      }
      res.json({ success: true });
    } catch (error: any) {
      console.error("Error removing inventory withdrawal line:", error);
      res.status(500).json({ error: error.message || "فشل في حذف السطر" });
    }
  });

  app.post("/api/inventory-withdrawals/:id/confirm", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const confirmedByEmployeeId = req.body.confirmedByEmployeeId;
      const withdrawal = await storage.confirmInventoryWithdrawalOrder(id, confirmedByEmployeeId);
      res.json(withdrawal);
    } catch (error: any) {
      console.error("Error confirming inventory withdrawal:", error);
      res.status(500).json({ error: error.message || "فشل في تأكيد أمر السحب" });
    }
  });

  app.post("/api/inventory-withdrawals/:id/cancel", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const withdrawal = await storage.cancelInventoryWithdrawalOrder(id);
      res.json(withdrawal);
    } catch (error: any) {
      console.error("Error cancelling inventory withdrawal:", error);
      res.status(500).json({ error: error.message || "فشل في إلغاء أمر السحب" });
    }
  });
  
  // ========================================
  // INVENTORY
  // ========================================
  
  app.get("/api/inventory", async (req: Request, res: Response) => {
    try {
      const inventory = await storage.getInventorySummary();
      res.json(inventory);
    } catch (error) {
      console.error("Error fetching inventory:", error);
      res.status(500).json({ error: "Failed to fetch inventory" });
    }
  });
  
  app.get("/api/stock-batches", async (req: Request, res: Response) => {
    try {
      const productId = req.query.productId ? parseInt(req.query.productId as string) : undefined;
      const batches = await storage.getStockBatches(productId);
      res.json(batches);
    } catch (error) {
      console.error("Error fetching stock batches:", error);
      res.status(500).json({ error: "Failed to fetch stock batches" });
    }
  });
  
  app.get("/api/stock-movements", async (req: Request, res: Response) => {
    try {
      const productId = req.query.productId ? parseInt(req.query.productId as string) : undefined;
      const movements = await storage.getStockMovements(productId);
      res.json(movements);
    } catch (error) {
      console.error("Error fetching stock movements:", error);
      res.status(500).json({ error: "Failed to fetch stock movements" });
    }
  });

  // ========================================
  // EMPLOYEE CUSTODIES (عهدة الموظف)
  // ========================================

  app.get("/api/employee-custodies", async (req: Request, res: Response) => {
    try {
      const employeeId = req.query.employeeId ? parseInt(req.query.employeeId as string) : undefined;
      const status = req.query.status as string | undefined;
      const productId = req.query.productId ? parseInt(req.query.productId as string) : undefined;
      const custodies = await storage.getEmployeeCustodies({ employeeId, status, productId });
      res.json(custodies);
    } catch (error) {
      console.error("Error fetching employee custodies:", error);
      res.status(500).json({ error: "Failed to fetch employee custodies" });
    }
  });

  app.get("/api/employee-custodies/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const custody = await storage.getEmployeeCustody(id);
      if (!custody) {
        return res.status(404).json({ error: "Custody not found" });
      }
      res.json(custody);
    } catch (error) {
      console.error("Error fetching employee custody:", error);
      res.status(500).json({ error: "Failed to fetch employee custody" });
    }
  });

  app.post("/api/employee-custodies", async (req: Request, res: Response) => {
    try {
      const custody = await storage.createEmployeeCustody(req.body);
      res.status(201).json(custody);
    } catch (error: any) {
      console.error("Error creating employee custody:", error);
      res.status(400).json({ error: error.message || "Failed to create employee custody" });
    }
  });

  app.put("/api/employee-custodies/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const custody = await storage.updateEmployeeCustody(id, req.body);
      if (!custody) {
        return res.status(404).json({ error: "Custody not found" });
      }
      res.json(custody);
    } catch (error) {
      console.error("Error updating employee custody:", error);
      res.status(500).json({ error: "Failed to update employee custody" });
    }
  });

  app.delete("/api/employee-custodies/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const result = await storage.deleteEmployeeCustody(id);
      if (!result) {
        return res.status(404).json({ error: "Custody not found" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting employee custody:", error);
      res.status(500).json({ error: "Failed to delete employee custody" });
    }
  });

  app.get("/api/employees/:id/custodies", async (req: Request, res: Response) => {
    try {
      const employeeId = parseInt(req.params.id);
      // Return all custodies for the employee (both open and closed) to show full history
      const custodies = await storage.getEmployeeCustodies({ employeeId });
      res.json(custodies);
    } catch (error) {
      console.error("Error fetching employee custodies:", error);
      res.status(500).json({ error: "Failed to fetch employee custodies" });
    }
  });

  // ========================================
  // EMPLOYEE COMMISSIONS REPORT (تقرير العمولات)
  // ========================================

  app.get("/api/employees/:id/commissions", async (req: Request, res: Response) => {
    try {
      const employeeId = parseInt(req.params.id);
      const startDate = req.query.startDate as string;
      const endDate = req.query.endDate as string;
      
      if (!startDate || !endDate) {
        return res.status(400).json({ error: "يجب تحديد تاريخ البداية والنهاية" });
      }
      
      const commissions = await storage.getEmployeeCommissions(employeeId, startDate, endDate);
      res.json(commissions);
    } catch (error) {
      console.error("Error fetching employee commissions:", error);
      res.status(500).json({ error: "فشل في جلب تقرير العمولات" });
    }
  });

  // ========================================
  // INVENTORY REPORTS (تقارير المخزون)
  // ========================================

  app.get("/api/reports/stock-movement-summary", async (req: Request, res: Response) => {
    try {
      const fromDate = req.query.from_date as string;
      const toDate = req.query.to_date as string;
      const productId = req.query.product_id ? parseInt(req.query.product_id as string) : undefined;
      
      if (!fromDate || !toDate) {
        return res.status(400).json({ error: "from_date and to_date are required" });
      }
      
      const summary = await storage.getStockMovementSummary(fromDate, toDate, productId);
      res.json(summary);
    } catch (error) {
      console.error("Error fetching stock movement summary:", error);
      res.status(500).json({ error: "Failed to fetch stock movement summary" });
    }
  });

  app.get("/api/reports/stock-expiry", async (req: Request, res: Response) => {
    try {
      const days = req.query.days ? parseInt(req.query.days as string) : 30;
      const expiringProducts = await storage.getNearExpiryProducts(days);
      res.json(expiringProducts);
    } catch (error) {
      console.error("Error fetching near-expiry products:", error);
      res.status(500).json({ error: "Failed to fetch near-expiry products" });
    }
  });
  
  // ========================================
  // JOURNAL ENTRIES
  // ========================================
  
  app.get("/api/journal-entries", async (req: Request, res: Response) => {
    try {
      const entries = await storage.getJournalEntries();
      res.json(entries);
    } catch (error) {
      console.error("Error fetching journal entries:", error);
      res.status(500).json({ error: "Failed to fetch journal entries" });
    }
  });
  
  app.get("/api/journal-entries/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const entry = await storage.getJournalEntry(id);
      if (!entry) {
        return res.status(404).json({ error: "Journal entry not found" });
      }
      res.json(entry);
    } catch (error) {
      console.error("Error fetching journal entry:", error);
      res.status(500).json({ error: "Failed to fetch journal entry" });
    }
  });
  
  // ========================================
  // REPORTS
  // ========================================
  
  app.get("/api/reports/supplier-balances", async (req: Request, res: Response) => {
    try {
      const balances = await storage.getSupplierBalances();
      res.json(balances);
    } catch (error) {
      console.error("Error fetching supplier balances:", error);
      res.status(500).json({ error: "Failed to fetch supplier balances" });
    }
  });
  
  // ========================================
  // CHART OF ACCOUNTS
  // ========================================
  
  app.get("/api/accounts", async (req: Request, res: Response) => {
    try {
      const tree = req.query.tree === "true";
      const accounts = tree ? await storage.getAccountsTree() : await storage.getAccounts();
      res.json(accounts);
    } catch (error) {
      console.error("Error fetching accounts:", error);
      res.status(500).json({ error: "Failed to fetch accounts" });
    }
  });
  
  app.get("/api/accounts/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const account = await storage.getAccount(id);
      if (!account) {
        return res.status(404).json({ error: "Account not found" });
      }
      res.json(account);
    } catch (error) {
      console.error("Error fetching account:", error);
      res.status(500).json({ error: "Failed to fetch account" });
    }
  });
  
  app.post("/api/accounts", async (req: Request, res: Response) => {
    try {
      const account = await storage.createAccount(req.body);
      res.status(201).json(account);
    } catch (error) {
      console.error("Error creating account:", error);
      res.status(500).json({ error: "Failed to create account" });
    }
  });
  
  app.patch("/api/accounts/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const account = await storage.updateAccount(id, req.body);
      if (!account) {
        return res.status(404).json({ error: "Account not found" });
      }
      res.json(account);
    } catch (error) {
      console.error("Error updating account:", error);
      res.status(500).json({ error: "Failed to update account" });
    }
  });
  
  app.delete("/api/accounts/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteAccount(id);
      if (!success) {
        return res.status(400).json({ error: "Cannot delete account with children or journal entries" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting account:", error);
      res.status(500).json({ error: "Failed to delete account" });
    }
  });
  
  // Seed default accounts
  app.post("/api/accounts/seed", async (req: Request, res: Response) => {
    try {
      const existingAccounts = await storage.getAccounts();
      if (existingAccounts.length > 0) {
        return res.json({ message: "Accounts already exist", count: existingAccounts.length });
      }
      
      // Create default chart of accounts
      const defaultAccounts = [
        // Level 1 - Main categories
        { code: "1", nameAr: "الأصول", accountType: "asset", isPostable: false, level: 1 },
        { code: "2", nameAr: "الخصوم", accountType: "liability", isPostable: false, level: 1 },
        { code: "3", nameAr: "حقوق الملكية", accountType: "equity", isPostable: false, level: 1 },
        { code: "4", nameAr: "الإيرادات", accountType: "revenue", isPostable: false, level: 1 },
        { code: "5", nameAr: "المصروفات", accountType: "expense", isPostable: false, level: 1 },
      ];
      
      const createdLevel1: any[] = [];
      for (const acc of defaultAccounts) {
        const created = await storage.createAccount(acc as any);
        createdLevel1.push(created);
      }
      
      // Level 2 - Sub categories
      const assetsId = createdLevel1.find(a => a.code === "1")?.id;
      const liabilitiesId = createdLevel1.find(a => a.code === "2")?.id;
      const equityId = createdLevel1.find(a => a.code === "3")?.id;
      const revenueId = createdLevel1.find(a => a.code === "4")?.id;
      const expensesId = createdLevel1.find(a => a.code === "5")?.id;
      
      const level2Accounts = [
        // Assets sub-categories
        { code: "11", nameAr: "الأصول المتداولة", accountType: "asset", parentId: assetsId, isPostable: false },
        { code: "12", nameAr: "الأصول الثابتة", accountType: "asset", parentId: assetsId, isPostable: false },
        { code: "14", nameAr: "أصول أخرى", accountType: "asset", parentId: assetsId, isPostable: false },
        // Liabilities sub-categories  
        { code: "21", nameAr: "الخصوم المتداولة", accountType: "liability", parentId: liabilitiesId, isPostable: false },
        { code: "24", nameAr: "الضرائب المستحقة", accountType: "liability", parentId: liabilitiesId, isPostable: false },
        // Equity sub-categories
        { code: "31", nameAr: "رأس المال", accountType: "equity", parentId: equityId, isPostable: false },
        { code: "32", nameAr: "الأرباح", accountType: "equity", parentId: equityId, isPostable: false },
        // Revenue sub-categories
        { code: "41", nameAr: "إيرادات المبيعات", accountType: "revenue", parentId: revenueId, isPostable: false },
        { code: "42", nameAr: "إيرادات الخدمات", accountType: "revenue", parentId: revenueId, isPostable: false },
        // Expense sub-categories
        { code: "51", nameAr: "تكلفة المبيعات", accountType: "expense", parentId: expensesId, isPostable: false },
        { code: "52", nameAr: "المشتريات", accountType: "expense", parentId: expensesId, isPostable: false },
        { code: "53", nameAr: "المصروفات التشغيلية", accountType: "expense", parentId: expensesId, isPostable: false },
      ];
      
      const createdLevel2: any[] = [];
      for (const acc of level2Accounts) {
        const created = await storage.createAccount(acc as any);
        createdLevel2.push(created);
      }
      
      // Level 3 - Postable accounts
      const currentAssetsId = createdLevel2.find(a => a.code === "11")?.id;
      const currentLiabId = createdLevel2.find(a => a.code === "21")?.id;
      const taxesId = createdLevel2.find(a => a.code === "24")?.id;
      const capitalId = createdLevel2.find(a => a.code === "31")?.id;
      const salesRevId = createdLevel2.find(a => a.code === "41")?.id;
      const serviceRevId = createdLevel2.find(a => a.code === "42")?.id;
      const cogsId = createdLevel2.find(a => a.code === "51")?.id;
      const purchasesId = createdLevel2.find(a => a.code === "52")?.id;
      const otherAssetsId = createdLevel2.find(a => a.code === "14")?.id;
      
      const level3Accounts = [
        // Current Assets
        { code: "1110", nameAr: "النقدية", accountType: "asset", parentId: currentAssetsId, isPostable: true },
        { code: "1120", nameAr: "البنك", accountType: "asset", parentId: currentAssetsId, isPostable: true },
        { code: "1200", nameAr: "المدينون", accountType: "asset", parentId: currentAssetsId, isPostable: true },
        { code: "1201", nameAr: "مخزون البضائع", accountType: "asset", parentId: currentAssetsId, isPostable: true },
        // Other Assets
        { code: "1410", nameAr: "ضريبة القيمة المضافة - مدخلات", accountType: "asset", parentId: otherAssetsId, isPostable: true },
        // Current Liabilities
        { code: "2100", nameAr: "الموردين", accountType: "liability", parentId: currentLiabId, isPostable: true },
        // Tax Liabilities
        { code: "2410", nameAr: "ضريبة القيمة المضافة - مخرجات", accountType: "liability", parentId: taxesId, isPostable: true },
        // Capital
        { code: "3100", nameAr: "رأس المال", accountType: "equity", parentId: capitalId, isPostable: true },
        { code: "3200", nameAr: "الأرباح المحتجزة", accountType: "equity", parentId: capitalId, isPostable: true },
        // Revenue
        { code: "4100", nameAr: "إيرادات المبيعات", accountType: "revenue", parentId: salesRevId, isPostable: true },
        { code: "4200", nameAr: "إيرادات الخدمات", accountType: "revenue", parentId: serviceRevId, isPostable: true },
        // Cost of Goods Sold
        { code: "5100", nameAr: "تكلفة البضاعة المباعة", accountType: "expense", parentId: cogsId, isPostable: true },
        // Purchases
        { code: "5200", nameAr: "المشتريات", accountType: "expense", parentId: purchasesId, isPostable: true },
      ];
      
      for (const acc of level3Accounts) {
        await storage.createAccount(acc as any);
      }
      
      res.json({ success: true, message: "Default chart of accounts created" });
    } catch (error) {
      console.error("Error seeding accounts:", error);
      res.status(500).json({ error: "Failed to seed accounts" });
    }
  });
  
  // ========================================
  // SERVICE CATEGORIES (فئات الخدمات)
  // ========================================
  
  // GET /api/service-categories - List all categories with services_count
  app.get("/api/service-categories", async (req: Request, res: Response) => {
    try {
      const categories = await storage.getServiceCategories();
      res.json(categories);
    } catch (error) {
      console.error("Error fetching service categories:", error);
      res.status(500).json({ error: "فشل في جلب فئات الخدمات" });
    }
  });
  
  // GET /api/service-categories/:id - Get category with its services
  app.get("/api/service-categories/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const category = await storage.getServiceCategoryWithServices(id);
      if (!category) {
        return res.status(404).json({ error: "فئة الخدمة غير موجودة" });
      }
      res.json(category);
    } catch (error) {
      console.error("Error fetching service category:", error);
      res.status(500).json({ error: "فشل في جلب فئة الخدمة" });
    }
  });
  
  // POST /api/service-categories - Create new category
  app.post("/api/service-categories", async (req: Request, res: Response) => {
    try {
      // Preprocess data
      const rawData = {
        nameAr: req.body.nameAr?.trim() || "",
        code: req.body.code?.trim() || null,
        imageUrl: req.body.imageUrl?.trim() || null,
        isActive: req.body.isActive !== false,
      };
      
      // Validate with Zod schema
      const validated = insertServiceCategorySchema.parse(rawData);
      
      const category = await storage.createServiceCategory(validated);
      res.status(201).json(category);
    } catch (error: any) {
      console.error("Error creating service category:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0]?.message || "خطأ في البيانات";
        return res.status(400).json({ error: firstError, details: error.errors });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في إنشاء فئة الخدمة" });
    }
  });
  
  // PUT /api/service-categories/:id - Update category
  app.put("/api/service-categories/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const existing = await storage.getServiceCategory(id);
      if (!existing) {
        return res.status(404).json({ error: "فئة الخدمة غير موجودة" });
      }
      
      // Preprocess only provided fields
      const rawData: Record<string, any> = {};
      if (req.body.nameAr !== undefined) rawData.nameAr = req.body.nameAr.trim();
      if (req.body.code !== undefined) rawData.code = req.body.code?.trim() || null;
      if (req.body.imageUrl !== undefined) rawData.imageUrl = req.body.imageUrl?.trim() || null;
      if (req.body.isActive !== undefined) rawData.isActive = req.body.isActive;
      
      // Validate with partial Zod schema
      const validated = updateServiceCategorySchema.parse(rawData);
      
      const category = await storage.updateServiceCategory(id, validated);
      res.json(category);
    } catch (error: any) {
      console.error("Error updating service category:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0]?.message || "خطأ في البيانات";
        return res.status(400).json({ error: firstError, details: error.errors });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في تحديث فئة الخدمة" });
    }
  });
  
  // DELETE /api/service-categories/:id - Soft delete (set isActive = false)
  app.delete("/api/service-categories/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteServiceCategory(id);
      if (!success) {
        return res.status(404).json({ error: "فئة الخدمة غير موجودة" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting service category:", error);
      res.status(500).json({ error: "فشل في حذف فئة الخدمة" });
    }
  });
  
  // ========================================
  // SERVICES (الخدمات)
  // ========================================
  
  // GET /api/services - List services with optional filters
  app.get("/api/services", async (req: Request, res: Response) => {
    try {
      const categoryId = req.query.category_id ? parseInt(req.query.category_id as string) : undefined;
      const isActive = req.query.is_active !== undefined 
        ? req.query.is_active === "true" 
        : undefined;
      
      const services = await storage.getServices({ categoryId, isActive });
      
      // Map to API response format
      const response = services.map(s => ({
        id: s.id,
        nameAr: s.nameAr,
        code: s.code,
        descriptionAr: s.descriptionAr,
        price: s.price,
        vatType: s.vatType,
        vatRate: s.vatRate,
        isActive: s.isActive,
        categoryId: s.categoryId,
        categoryNameAr: s.category?.nameAr,
        relatedProducts: s.relatedProducts?.map(rp => ({
          productId: rp.productId,
          productNameAr: rp.product?.nameAr,
          quantityPerService: rp.quantityPerService,
        })),
      }));
      
      res.json(response);
    } catch (error) {
      console.error("Error fetching services:", error);
      res.status(500).json({ error: "فشل في جلب الخدمات" });
    }
  });
  
  // GET /api/services/:id - Get service with full details
  app.get("/api/services/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const service = await storage.getService(id);
      if (!service) {
        return res.status(404).json({ error: "الخدمة غير موجودة" });
      }
      
      // Map to API response format
      const response = {
        id: service.id,
        categoryId: service.categoryId,
        nameAr: service.nameAr,
        code: service.code,
        descriptionAr: service.descriptionAr,
        price: service.price,
        vatType: service.vatType,
        vatRate: service.vatRate,
        isActive: service.isActive,
        categoryNameAr: service.category?.nameAr,
        relatedProducts: service.relatedProducts?.map(rp => ({
          productId: rp.productId,
          productNameAr: rp.product?.nameAr,
          quantityPerService: rp.quantityPerService,
          notes: rp.notes,
        })),
      };
      
      res.json(response);
    } catch (error) {
      console.error("Error fetching service:", error);
      res.status(500).json({ error: "فشل في جلب الخدمة" });
    }
  });
  
  // POST /api/services - Create new service with optional related products
  app.post("/api/services", async (req: Request, res: Response) => {
    try {
      const { related_products, relatedProducts, ...serviceData } = req.body;
      
      // Preprocess data
      const rawData = {
        categoryId: Number(serviceData.categoryId) || 0,
        nameAr: serviceData.nameAr?.trim() || "",
        code: serviceData.code?.trim() || null,
        descriptionAr: serviceData.descriptionAr?.trim() || null,
        price: Number(serviceData.price) || 0,
        vatType: serviceData.vatType || "inclusive",
        vatRate: Number(serviceData.vatRate) ?? 0.15,
        isActive: serviceData.isActive !== false,
      };
      
      // Validate with Zod schema
      const validated = insertServiceSchema.parse(rawData);
      
      // Validate category exists and is active
      const category = await storage.getServiceCategory(validated.categoryId);
      if (!category) {
        return res.status(400).json({ error: "فئة الخدمة غير موجودة" });
      }
      if (!category.isActive) {
        return res.status(400).json({ error: "فئة الخدمة غير مفعّلة" });
      }
      
      // Process and validate related products
      const productsData = related_products || relatedProducts;
      let validatedProducts: { productId: number; quantityPerService: number; deductFromCommission: boolean; notes?: string }[] | undefined;
      
      if (productsData && productsData.length > 0) {
        // Normalize product data for validation
        const normalizedProducts = productsData.map((rp: any) => {
          // Handle boolean conversion properly (handle string "true"/"false" and actual booleans)
          const deductValue = rp.deduct_from_commission ?? rp.deductFromCommission;
          const deductFromCommission = deductValue === true || deductValue === "true";
          
          return {
            productId: Number(rp.product_id || rp.productId),
            quantityPerService: Number(rp.quantity_per_service || rp.quantityPerService) || 1,
            deductFromCommission,
            notes: rp.notes || undefined,
          };
        });
        
        // Validate with Zod schema
        const parsedProducts = relatedProductsArraySchema.parse(normalizedProducts);
        
        if (parsedProducts) {
          validatedProducts = [];
          for (const rp of parsedProducts) {
            const product = await storage.getProduct(rp.productId);
            if (!product) {
              return res.status(400).json({ error: `المنتج رقم ${rp.productId} غير موجود` });
            }
            validatedProducts.push({
              productId: rp.productId,
              quantityPerService: rp.quantityPerService,
              deductFromCommission: rp.deductFromCommission === true,
              notes: rp.notes || undefined,
            });
          }
        }
      }
      
      const service = await storage.createService(validated, validatedProducts);
      
      res.status(201).json(service);
    } catch (error: any) {
      console.error("Error creating service:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0]?.message || "خطأ في البيانات";
        return res.status(400).json({ error: firstError, details: error.errors });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في إنشاء الخدمة" });
    }
  });
  
  // PUT /api/services/:id - Update service with optional related products replacement
  app.put("/api/services/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const existing = await storage.getService(id);
      if (!existing) {
        return res.status(404).json({ error: "الخدمة غير موجودة" });
      }
      
      const { related_products, relatedProducts, ...serviceData } = req.body;
      
      // Preprocess only provided fields
      const rawData: Record<string, any> = {};
      if (serviceData.categoryId !== undefined) rawData.categoryId = Number(serviceData.categoryId);
      if (serviceData.nameAr !== undefined) rawData.nameAr = serviceData.nameAr.trim();
      if (serviceData.code !== undefined) rawData.code = serviceData.code?.trim() || null;
      if (serviceData.descriptionAr !== undefined) rawData.descriptionAr = serviceData.descriptionAr?.trim() || null;
      if (serviceData.price !== undefined) rawData.price = Number(serviceData.price);
      if (serviceData.vatType !== undefined) rawData.vatType = serviceData.vatType;
      if (serviceData.vatRate !== undefined) rawData.vatRate = Number(serviceData.vatRate);
      if (serviceData.isActive !== undefined) rawData.isActive = serviceData.isActive;
      
      // Validate with partial Zod schema
      const validated = updateServiceSchema.parse(rawData);
      
      // Validate category if changing
      const newCategoryId = validated.categoryId ?? existing.categoryId;
      if (validated.categoryId && validated.categoryId !== existing.categoryId) {
        const category = await storage.getServiceCategory(newCategoryId);
        if (!category) {
          return res.status(400).json({ error: "فئة الخدمة غير موجودة" });
        }
        if (!category.isActive) {
          return res.status(400).json({ error: "فئة الخدمة غير مفعّلة" });
        }
      }
      
      // Process and validate related products
      const productsData = related_products || relatedProducts;
      let validatedProducts: { productId: number; quantityPerService: number; deductFromCommission: boolean; notes?: string }[] | undefined;
      
      if (productsData !== undefined) {
        // Normalize product data for validation
        const normalizedProducts = productsData.map((rp: any) => {
          // Handle boolean conversion properly (handle string "true"/"false" and actual booleans)
          const deductValue = rp.deduct_from_commission ?? rp.deductFromCommission;
          const deductFromCommission = deductValue === true || deductValue === "true";
          
          return {
            productId: Number(rp.product_id || rp.productId),
            quantityPerService: Number(rp.quantity_per_service || rp.quantityPerService) || 1,
            deductFromCommission,
            notes: rp.notes || undefined,
          };
        });
        
        // Validate with Zod schema
        const parsedProducts = relatedProductsArraySchema.parse(normalizedProducts);
        
        if (parsedProducts) {
          validatedProducts = [];
          for (const rp of parsedProducts) {
            const product = await storage.getProduct(rp.productId);
            if (!product) {
              return res.status(400).json({ error: `المنتج رقم ${rp.productId} غير موجود` });
            }
            validatedProducts.push({
              productId: rp.productId,
              quantityPerService: rp.quantityPerService,
              deductFromCommission: rp.deductFromCommission === true,
              notes: rp.notes || undefined,
            });
          }
        }
      }
      
      const service = await storage.updateService(id, validated, validatedProducts);
      
      res.json(service);
    } catch (error: any) {
      console.error("Error updating service:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0]?.message || "خطأ في البيانات";
        return res.status(400).json({ error: firstError, details: error.errors });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في تحديث الخدمة" });
    }
  });
  
  // DELETE /api/services/:id - Soft delete (set isActive = false)
  app.delete("/api/services/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteService(id);
      if (!success) {
        return res.status(404).json({ error: "الخدمة غير موجودة" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting service:", error);
      res.status(500).json({ error: "فشل في حذف الخدمة" });
    }
  });
  
  // ========================================
  // SERVICE IMAGES
  // ========================================
  
  // Serve uploaded images statically
  app.use("/uploads/services", (req, res, next) => {
    const filePath = path.join(uploadsDir, req.path);
    if (fs.existsSync(filePath)) {
      res.sendFile(filePath);
    } else {
      res.status(404).json({ error: "الصورة غير موجودة" });
    }
  });
  
  // Serve brand uploaded images statically
  app.use("/uploads/brands", (req, res, next) => {
    const filePath = path.join(brandUploadsDir, req.path);
    if (fs.existsSync(filePath)) {
      res.sendFile(filePath);
    } else {
      res.status(404).json({ error: "الصورة غير موجودة" });
    }
  });
  
  // GET /api/services/:serviceId/images - Get all images for a service
  app.get("/api/services/:serviceId/images", async (req: Request, res: Response) => {
    try {
      const serviceId = parseInt(req.params.serviceId);
      const service = await storage.getService(serviceId);
      if (!service) {
        return res.status(404).json({ error: "الخدمة غير موجودة" });
      }
      const images = await storage.getServiceImages(serviceId);
      res.json(images);
    } catch (error) {
      console.error("Error fetching service images:", error);
      res.status(500).json({ error: "فشل في جلب صور الخدمة" });
    }
  });
  
  // POST /api/services/:serviceId/images - Upload multiple images to a service
  app.post("/api/services/:serviceId/images", uploadServiceImage.array("images", 10), async (req: Request, res: Response) => {
    try {
      const serviceId = parseInt(req.params.serviceId);
      const service = await storage.getService(serviceId);
      if (!service) {
        return res.status(404).json({ error: "الخدمة غير موجودة" });
      }
      
      const files = req.files as Express.Multer.File[];
      if (!files || files.length === 0) {
        return res.status(400).json({ error: "لم يتم تحديد أي صور للرفع" });
      }
      
      // Get current highest display order
      const existingImages = await storage.getServiceImages(serviceId);
      let maxOrder = existingImages.length > 0 
        ? Math.max(...existingImages.map(img => img.displayOrder)) 
        : -1;
      
      const uploadedImages = [];
      for (const file of files) {
        maxOrder++;
        const imageUrl = `/uploads/services/${file.filename}`;
        const newImage = await storage.addServiceImage({
          serviceId,
          imageUrl,
          filename: file.originalname,
          displayOrder: maxOrder,
        });
        uploadedImages.push(newImage);
      }
      
      res.status(201).json(uploadedImages);
    } catch (error: any) {
      console.error("Error uploading service images:", error);
      if (error.message?.includes("نوع الملف غير مدعوم")) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في رفع الصور" });
    }
  });
  
  // DELETE /api/services/:serviceId/images/:imageId - Delete a service image
  app.delete("/api/services/:serviceId/images/:imageId", async (req: Request, res: Response) => {
    try {
      const serviceId = parseInt(req.params.serviceId);
      const imageId = parseInt(req.params.imageId);
      
      const service = await storage.getService(serviceId);
      if (!service) {
        return res.status(404).json({ error: "الخدمة غير موجودة" });
      }
      
      const images = await storage.getServiceImages(serviceId);
      const image = images.find(img => img.id === imageId);
      if (!image) {
        return res.status(404).json({ error: "الصورة غير موجودة" });
      }
      
      // Delete the file from disk
      const filePath = path.join(process.cwd(), image.imageUrl);
      if (fs.existsSync(filePath)) {
        fs.unlinkSync(filePath);
      }
      
      // Delete from database
      await storage.deleteServiceImage(imageId);
      
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting service image:", error);
      res.status(500).json({ error: "فشل في حذف الصورة" });
    }
  });
  
  // PUT /api/services/:serviceId/images/order - Reorder service images
  app.put("/api/services/:serviceId/images/order", async (req: Request, res: Response) => {
    try {
      const serviceId = parseInt(req.params.serviceId);
      const { imageIds } = req.body;
      
      if (!Array.isArray(imageIds)) {
        return res.status(400).json({ error: "يجب توفير قائمة معرفات الصور" });
      }
      
      const service = await storage.getService(serviceId);
      if (!service) {
        return res.status(404).json({ error: "الخدمة غير موجودة" });
      }
      
      await storage.updateServiceImagesOrder(serviceId, imageIds);
      
      const updatedImages = await storage.getServiceImages(serviceId);
      res.json(updatedImages);
    } catch (error) {
      console.error("Error reordering service images:", error);
      res.status(500).json({ error: "فشل في إعادة ترتيب الصور" });
    }
  });
  
  // ========================================
  // CUSTOMERS
  // ========================================
  
  app.get("/api/customers", async (req: Request, res: Response) => {
    try {
      const customers = await storage.getCustomers();
      res.json(customers);
    } catch (error) {
      console.error("Error fetching customers:", error);
      res.status(500).json({ error: "Failed to fetch customers" });
    }
  });
  
  app.get("/api/customers/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const customer = await storage.getCustomer(id);
      if (!customer) {
        return res.status(404).json({ error: "Customer not found" });
      }
      res.json(customer);
    } catch (error) {
      console.error("Error fetching customer:", error);
      res.status(500).json({ error: "Failed to fetch customer" });
    }
  });
  
  app.post("/api/customers", async (req: Request, res: Response) => {
    try {
      const { password, ...customerData } = req.body;
      let dataToInsert = { ...customerData };
      
      // Validate required fields
      if (!dataToInsert.nameAr || !dataToInsert.nameAr.trim()) {
        return res.status(400).json({ error: "اسم العميل مطلوب" });
      }
      
      // Trim and clean data
      dataToInsert.nameAr = dataToInsert.nameAr.trim();
      
      // Handle phone - allow null/empty but validate non-empty phones
      const phone = dataToInsert.phone?.trim();
      if (phone) {
        dataToInsert.phone = phone;
        
        // Validate phone format (basic validation - digits, +, -, spaces, parens)
        const phonePattern = /^[0-9+\-\s()]+$/;
        if (!phonePattern.test(phone)) {
          return res.status(400).json({ error: "رقم الهاتف غير صالح" });
        }
        
        // Check for duplicate phone
        const existingByPhone = await storage.getCustomerByPhone(phone);
        if (existingByPhone) {
          return res.status(400).json({ error: "رقم الهاتف مسجل بالفعل لعميل آخر" });
        }
      } else {
        // Clear empty string to null
        dataToInsert.phone = null;
      }
      
      // Hash password if provided
      if (password && password.trim()) {
        const bcrypt = await import("bcryptjs");
        const passwordHash = await bcrypt.hash(password, 10);
        dataToInsert = { ...dataToInsert, passwordHash };
      }
      
      const customer = await storage.createCustomer(dataToInsert);
      res.status(201).json(customer);
    } catch (error) {
      console.error("Error creating customer:", error);
      res.status(500).json({ error: "Failed to create customer" });
    }
  });
  
  app.patch("/api/customers/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { password, ...updateData } = req.body;
      let dataToUpdate = { ...updateData };
      
      // Hash password if provided
      if (password && password.trim()) {
        const bcrypt = await import("bcryptjs");
        const passwordHash = await bcrypt.hash(password, 10);
        dataToUpdate = { ...dataToUpdate, passwordHash };
      }
      
      const customer = await storage.updateCustomer(id, dataToUpdate);
      if (!customer) {
        return res.status(404).json({ error: "Customer not found" });
      }
      res.json(customer);
    } catch (error) {
      console.error("Error updating customer:", error);
      res.status(500).json({ error: "Failed to update customer" });
    }
  });
  
  app.delete("/api/customers/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteCustomer(id);
      if (!success) {
        return res.status(404).json({ error: "Customer not found" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting customer:", error);
      res.status(500).json({ error: "Failed to delete customer" });
    }
  });

  // Customer search
  app.get("/api/customers/search", async (req: Request, res: Response) => {
    try {
      const query = req.query.q as string || "";
      const customers = await storage.searchCustomers(query);
      res.json(customers);
    } catch (error) {
      console.error("Error searching customers:", error);
      res.status(500).json({ error: "فشل في البحث عن العملاء" });
    }
  });

  // Customer summary (stats)
  app.get("/api/customers/:id/summary", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const summary = await storage.getCustomerSummary(id);
      res.json(summary);
    } catch (error) {
      console.error("Error fetching customer summary:", error);
      res.status(500).json({ error: "فشل في جلب ملخص العميل" });
    }
  });

  // Customer orders
  app.get("/api/customers/:id/orders", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const orders = await storage.getCustomerOrders(id);
      res.json(orders);
    } catch (error) {
      console.error("Error fetching customer orders:", error);
      res.status(500).json({ error: "فشل في جلب طلبات العميل" });
    }
  });

  // Customer ratings
  app.get("/api/customers/:id/ratings", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const ratings = await storage.getCustomerRatings(id);
      res.json(ratings);
    } catch (error) {
      console.error("Error fetching customer ratings:", error);
      res.status(500).json({ error: "فشل في جلب تقييمات العميل" });
    }
  });

  // Customer loyalty points history
  app.get("/api/customers/:id/loyalty", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const history = await storage.getCustomerLoyaltyHistory(id);
      res.json(history);
    } catch (error) {
      console.error("Error fetching customer loyalty history:", error);
      res.status(500).json({ error: "فشل في جلب سجل نقاط الولاء" });
    }
  });

  // Add loyalty points to customer
  app.post("/api/customers/:id/loyalty", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { points, transactionType, referenceType, referenceId, notes } = req.body;
      const transaction = await storage.addLoyaltyPoints(id, points, transactionType, referenceType, referenceId, notes);
      res.status(201).json(transaction);
    } catch (error: any) {
      console.error("Error adding loyalty points:", error);
      res.status(400).json({ error: error.message || "فشل في إضافة نقاط الولاء" });
    }
  });

  // Redeem loyalty points
  app.post("/api/customers/:id/loyalty/redeem", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { points, notes } = req.body;
      const transaction = await storage.redeemLoyaltyPoints(id, points, notes);
      res.status(201).json(transaction);
    } catch (error: any) {
      console.error("Error redeeming loyalty points:", error);
      res.status(400).json({ error: error.message || "فشل في استبدال نقاط الولاء" });
    }
  });

  // ========================================
  // LOYALTY SETTINGS
  // ========================================

  app.get("/api/loyalty/settings", async (req: Request, res: Response) => {
    try {
      const settings = await storage.getLoyaltySettings();
      res.json(settings || {
        pointsPerCurrency: 1,
        currencyPerPoint: 0.1,
        expiryMonths: 12,
        minRedeemPoints: 100,
        isActive: true,
      });
    } catch (error) {
      console.error("Error fetching loyalty settings:", error);
      res.status(500).json({ error: "فشل في جلب إعدادات الولاء" });
    }
  });

  app.patch("/api/loyalty/settings", async (req: Request, res: Response) => {
    try {
      const settings = await storage.updateLoyaltySettings(req.body);
      res.json(settings);
    } catch (error) {
      console.error("Error updating loyalty settings:", error);
      res.status(500).json({ error: "فشل في تحديث إعدادات الولاء" });
    }
  });

  // Get loyalty summary/statistics
  app.get("/api/loyalty/summary", async (req: Request, res: Response) => {
    try {
      const summary = await storage.getLoyaltySummary();
      res.json(summary);
    } catch (error) {
      console.error("Error fetching loyalty summary:", error);
      res.status(500).json({ error: "فشل في جلب ملخص نقاط الولاء" });
    }
  });

  // Get all loyalty transactions with filters
  app.get("/api/loyalty/transactions", async (req: Request, res: Response) => {
    try {
      const customerId = req.query.customerId ? parseInt(req.query.customerId as string) : undefined;
      const transactionType = req.query.transactionType as string | undefined;
      const startDate = req.query.startDate as string | undefined;
      const endDate = req.query.endDate as string | undefined;
      const limit = req.query.limit ? parseInt(req.query.limit as string) : undefined;
      const offset = req.query.offset ? parseInt(req.query.offset as string) : undefined;

      const result = await storage.getAllLoyaltyTransactions({
        customerId,
        transactionType,
        startDate,
        endDate,
        limit,
        offset,
      });
      res.json(result);
    } catch (error) {
      console.error("Error fetching loyalty transactions:", error);
      res.status(500).json({ error: "فشل في جلب حركات نقاط الولاء" });
    }
  });

  // Manual adjustment of loyalty points (accepts positive or negative points)
  app.post("/api/loyalty/adjust", async (req: Request, res: Response) => {
    try {
      const { customerId, points, notes } = req.body;
      if (!customerId || points === undefined || points === 0) {
        return res.status(400).json({ error: "يجب تحديد العميل وعدد النقاط (غير صفر)" });
      }
      const transaction = await storage.adjustLoyaltyPoints(customerId, points, notes);
      res.status(201).json(transaction);
    } catch (error: any) {
      console.error("Error adjusting loyalty points:", error);
      res.status(400).json({ error: error.message || "فشل في تعديل نقاط الولاء" });
    }
  });

  // Expire loyalty points (batch operation)
  app.post("/api/loyalty/expire", async (req: Request, res: Response) => {
    try {
      const result = await storage.expireLoyaltyPoints();
      res.json(result);
    } catch (error) {
      console.error("Error expiring loyalty points:", error);
      res.status(500).json({ error: "فشل في إنهاء صلاحية نقاط الولاء" });
    }
  });
  
  // ========================================
  // GENERAL LEDGER & TRIAL BALANCE
  // ========================================
  
  app.get("/api/reports/general-ledger", async (req: Request, res: Response) => {
    try {
      const accountCode = req.query.accountCode as string | undefined;
      const startDate = req.query.startDate as string | undefined;
      const endDate = req.query.endDate as string | undefined;
      const ledger = await storage.getGeneralLedger(accountCode, startDate, endDate);
      res.json(ledger);
    } catch (error) {
      console.error("Error fetching general ledger:", error);
      res.status(500).json({ error: "Failed to fetch general ledger" });
    }
  });
  
  app.get("/api/reports/trial-balance", async (req: Request, res: Response) => {
    try {
      const asOfDate = req.query.asOfDate as string | undefined;
      const trialBalance = await storage.getTrialBalance(asOfDate);
      res.json(trialBalance);
    } catch (error) {
      console.error("Error fetching trial balance:", error);
      res.status(500).json({ error: "Failed to fetch trial balance" });
    }
  });
  
  app.get("/api/reports/income-statement", async (req: Request, res: Response) => {
    try {
      const startDate = req.query.startDate as string | undefined;
      const endDate = req.query.endDate as string | undefined;
      const incomeStatement = await storage.getIncomeStatement(startDate, endDate);
      res.json(incomeStatement);
    } catch (error) {
      console.error("Error fetching income statement:", error);
      res.status(500).json({ error: "فشل في جلب قائمة الدخل" });
    }
  });
  
  app.get("/api/reports/balance-sheet", async (req: Request, res: Response) => {
    try {
      const asOfDate = req.query.asOfDate as string | undefined;
      const balanceSheet = await storage.getBalanceSheet(asOfDate);
      res.json(balanceSheet);
    } catch (error) {
      console.error("Error fetching balance sheet:", error);
      res.status(500).json({ error: "فشل في جلب الميزانية العمومية" });
    }
  });
  
  app.get("/api/reports/cash-flow", async (req: Request, res: Response) => {
    try {
      const startDate = req.query.startDate as string | undefined;
      const endDate = req.query.endDate as string | undefined;
      const cashFlow = await storage.getCashFlowStatement(startDate, endDate);
      res.json(cashFlow);
    } catch (error) {
      console.error("Error fetching cash flow statement:", error);
      res.status(500).json({ error: "فشل في جلب التدفقات النقدية" });
    }
  });

  // VAT Declaration Report (ZATCA Compliant)
  app.get("/api/reports/vat-declaration", async (req: Request, res: Response) => {
    try {
      const fromDate = req.query.fromDate as string;
      const toDate = req.query.toDate as string;
      
      if (!fromDate || !toDate) {
        return res.status(400).json({ error: "يجب تحديد تاريخ البداية والنهاية" });
      }
      
      const vatReport = await storage.getVatDeclarationReport(fromDate, toDate);
      res.json(vatReport);
    } catch (error) {
      console.error("Error fetching VAT declaration report:", error);
      res.status(500).json({ error: "فشل في جلب تقرير الإقرار الضريبي" });
    }
  });
  
  app.post("/api/journal-entries/manual", async (req: Request, res: Response) => {
    try {
      const { date, description, lines } = req.body;
      
      if (!date || !description || !lines || !Array.isArray(lines) || lines.length < 2) {
        return res.status(400).json({ error: "يجب توفير التاريخ والوصف وسطرين على الأقل" });
      }
      
      const entry = await storage.createManualJournalEntry({
        date,
        description,
        lines,
      });
      
      res.status(201).json(entry);
    } catch (error: any) {
      console.error("Error creating manual journal entry:", error);
      res.status(400).json({ error: error.message || "فشل في إنشاء القيد اليدوي" });
    }
  });
  
  // ========================================
  // ROLES (الأدوار)
  // ========================================
  
  app.get("/api/roles", async (req: Request, res: Response) => {
    try {
      const isActive = req.query.isActive === 'true' ? true : req.query.isActive === 'false' ? false : undefined;
      const withPermissions = req.query.withPermissions === 'true';
      
      if (withPermissions) {
        const roles = await storage.getRolesWithPermissions(isActive);
        res.json(roles);
      } else {
        const roles = await storage.getRoles(isActive);
        res.json(roles);
      }
    } catch (error) {
      console.error("Error fetching roles:", error);
      res.status(500).json({ error: "فشل في جلب الأدوار" });
    }
  });
  
  app.get("/api/roles/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const withPermissions = req.query.withPermissions === 'true';
      
      if (withPermissions) {
        const role = await storage.getRoleWithPermissions(id);
        if (!role) {
          return res.status(404).json({ error: "الدور غير موجود" });
        }
        res.json(role);
      } else {
        const role = await storage.getRole(id);
        if (!role) {
          return res.status(404).json({ error: "الدور غير موجود" });
        }
        res.json(role);
      }
    } catch (error) {
      console.error("Error fetching role:", error);
      res.status(500).json({ error: "فشل في جلب الدور" });
    }
  });
  
  app.post("/api/roles", async (req: Request, res: Response) => {
    try {
      const validated = insertRoleSchema.parse(req.body);
      const role = await storage.createRole(validated);
      res.status(201).json(role);
    } catch (error) {
      console.error("Error creating role:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في إنشاء الدور" });
    }
  });
  
  app.put("/api/roles/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const validated = updateRoleSchema.parse(req.body);
      const role = await storage.updateRole(id, validated);
      if (!role) {
        return res.status(404).json({ error: "الدور غير موجود" });
      }
      res.json(role);
    } catch (error) {
      console.error("Error updating role:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في تحديث الدور" });
    }
  });
  
  app.delete("/api/roles/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteRole(id);
      if (!success) {
        return res.status(404).json({ error: "الدور غير موجود" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting role:", error);
      res.status(500).json({ error: "فشل في حذف الدور" });
    }
  });
  
  // Role Permissions
  app.get("/api/roles/:id/permissions", async (req: Request, res: Response) => {
    try {
      const roleId = parseInt(req.params.id);
      const permissions = await storage.getRolePermissions(roleId);
      res.json(permissions);
    } catch (error) {
      console.error("Error fetching role permissions:", error);
      res.status(500).json({ error: "فشل في جلب صلاحيات الدور" });
    }
  });
  
  app.put("/api/roles/:id/permissions", async (req: Request, res: Response) => {
    try {
      const roleId = parseInt(req.params.id);
      const { permissions } = req.body;
      
      if (!Array.isArray(permissions)) {
        return res.status(400).json({ error: "يجب توفير قائمة الصلاحيات" });
      }
      
      const updatedPermissions = await storage.setRolePermissions(roleId, permissions);
      res.json(updatedPermissions);
    } catch (error) {
      console.error("Error updating role permissions:", error);
      res.status(500).json({ error: "فشل في تحديث صلاحيات الدور" });
    }
  });
  
  // ========================================
  // BRANDS (العلامات التجارية)
  // ========================================
  
  app.get("/api/brands", async (req: Request, res: Response) => {
    try {
      const isActive = req.query.isActive === 'true' ? true : req.query.isActive === 'false' ? false : undefined;
      const brands = await storage.getBrands(isActive);
      res.json(brands);
    } catch (error) {
      console.error("Error fetching brands:", error);
      res.status(500).json({ error: "فشل في جلب العلامات التجارية" });
    }
  });
  
  app.get("/api/brands/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const brand = await storage.getBrand(id);
      if (!brand) {
        return res.status(404).json({ error: "العلامة التجارية غير موجودة" });
      }
      res.json(brand);
    } catch (error) {
      console.error("Error fetching brand:", error);
      res.status(500).json({ error: "فشل في جلب العلامة التجارية" });
    }
  });
  
  app.post("/api/brands", async (req: Request, res: Response) => {
    try {
      const validated = insertBrandSchema.parse(req.body);
      const brand = await storage.createBrand(validated);
      res.status(201).json(brand);
    } catch (error) {
      console.error("Error creating brand:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في إنشاء العلامة التجارية" });
    }
  });
  
  app.put("/api/brands/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const validated = updateBrandSchema.parse(req.body);
      const brand = await storage.updateBrand(id, validated);
      if (!brand) {
        return res.status(404).json({ error: "العلامة التجارية غير موجودة" });
      }
      res.json(brand);
    } catch (error) {
      console.error("Error updating brand:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في تحديث العلامة التجارية" });
    }
  });
  
  app.delete("/api/brands/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteBrand(id);
      if (!success) {
        return res.status(404).json({ error: "العلامة التجارية غير موجودة" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting brand:", error);
      res.status(500).json({ error: "فشل في حذف العلامة التجارية" });
    }
  });
  
  // Brand image upload
  app.post("/api/upload/brand-image", uploadBrandImage.single("image"), async (req: Request, res: Response) => {
    try {
      if (!req.file) {
        return res.status(400).json({ error: "لم يتم رفع صورة" });
      }
      const url = `/uploads/brands/${req.file.filename}`;
      res.json({ url });
    } catch (error) {
      console.error("Error uploading brand image:", error);
      res.status(500).json({ error: "فشل في رفع الصورة" });
    }
  });
  
  // ========================================
  // EMPLOYEES (الموظفين) - Extended HR
  // ========================================
  
  app.get("/api/employees", async (req: Request, res: Response) => {
    try {
      const filters: { isActive?: boolean; roleId?: number; branchId?: number; search?: string } = {};
      if (req.query.isActive === 'true') filters.isActive = true;
      if (req.query.isActive === 'false') filters.isActive = false;
      if (req.query.roleId) filters.roleId = parseInt(req.query.roleId as string);
      if (req.query.branchId) filters.branchId = parseInt(req.query.branchId as string);
      if (req.query.search) filters.search = req.query.search as string;
      
      const employees = await storage.getEmployees(filters);
      res.json(employees);
    } catch (error) {
      console.error("Error fetching employees:", error);
      res.status(500).json({ error: "فشل في جلب الموظفين" });
    }
  });
  
  app.get("/api/employees/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const employee = await storage.getEmployee(id);
      if (!employee) {
        return res.status(404).json({ error: "الموظف غير موجود" });
      }
      res.json(employee);
    } catch (error) {
      console.error("Error fetching employee:", error);
      res.status(500).json({ error: "فشل في جلب الموظف" });
    }
  });
  
  // Employee payroll history
  app.get("/api/employees/:id/payroll-history", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const employee = await storage.getEmployee(id);
      if (!employee) {
        return res.status(404).json({ error: "الموظف غير موجود" });
      }
      const payrollHistory = await storage.getEmployeePayrollHistory(id);
      res.json(payrollHistory);
    } catch (error) {
      console.error("Error fetching employee payroll history:", error);
      res.status(500).json({ error: "فشل في جلب سجل رواتب الموظف" });
    }
  });
  
  app.post("/api/employees", async (req: Request, res: Response) => {
    try {
      const validated = insertEmployeeSchema.parse(req.body);
      const employee = await storage.createEmployee(validated);
      res.status(201).json(employee);
    } catch (error) {
      console.error("Error creating employee:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في إنشاء الموظف" });
    }
  });
  
  app.put("/api/employees/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const validated = updateEmployeeSchema.parse(req.body);
      const employee = await storage.updateEmployee(id, validated);
      if (!employee) {
        return res.status(404).json({ error: "الموظف غير موجود" });
      }
      res.json(employee);
    } catch (error) {
      console.error("Error updating employee:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في تحديث الموظف" });
    }
  });
  
  app.delete("/api/employees/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteEmployee(id);
      if (!success) {
        return res.status(404).json({ error: "الموظف غير موجود" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting employee:", error);
      res.status(500).json({ error: "فشل في حذف الموظف" });
    }
  });
  
  // ========================================
  // EMPLOYEE TASKS (مهام الموظفين)
  // ========================================
  
  app.get("/api/employee-tasks", async (req: Request, res: Response) => {
    try {
      const filters: { assignedToEmployeeId?: number; status?: string; dueDate?: string } = {};
      if (req.query.assignedToEmployeeId) filters.assignedToEmployeeId = parseInt(req.query.assignedToEmployeeId as string);
      if (req.query.status) filters.status = req.query.status as string;
      if (req.query.dueDate) filters.dueDate = req.query.dueDate as string;
      
      const tasks = await storage.getEmployeeTasks(filters);
      res.json(tasks);
    } catch (error) {
      console.error("Error fetching employee tasks:", error);
      res.status(500).json({ error: "فشل في جلب المهام" });
    }
  });
  
  app.get("/api/employee-tasks/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const task = await storage.getEmployeeTask(id);
      if (!task) {
        return res.status(404).json({ error: "المهمة غير موجودة" });
      }
      res.json(task);
    } catch (error) {
      console.error("Error fetching employee task:", error);
      res.status(500).json({ error: "فشل في جلب المهمة" });
    }
  });
  
  app.post("/api/employee-tasks", async (req: Request, res: Response) => {
    try {
      const validated = insertEmployeeTaskSchema.parse(req.body);
      const task = await storage.createEmployeeTask(validated);
      res.status(201).json(task);
    } catch (error) {
      console.error("Error creating employee task:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في إنشاء المهمة" });
    }
  });
  
  app.put("/api/employee-tasks/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const validated = updateEmployeeTaskSchema.parse(req.body);
      const task = await storage.updateEmployeeTask(id, validated);
      if (!task) {
        return res.status(404).json({ error: "المهمة غير موجودة" });
      }
      res.json(task);
    } catch (error) {
      console.error("Error updating employee task:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في تحديث المهمة" });
    }
  });
  
  app.delete("/api/employee-tasks/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteEmployeeTask(id);
      if (!success) {
        return res.status(404).json({ error: "المهمة غير موجودة" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting employee task:", error);
      res.status(500).json({ error: "فشل في حذف المهمة" });
    }
  });
  
  // ========================================
  // SALARY PROFILES (ملفات الرواتب)
  // ========================================
  
  app.get("/api/salary-profiles", async (req: Request, res: Response) => {
    try {
      const profiles = await storage.getSalaryProfiles();
      res.json(profiles);
    } catch (error) {
      console.error("Error fetching salary profiles:", error);
      res.status(500).json({ error: "فشل في جلب ملفات الرواتب" });
    }
  });
  
  app.get("/api/salary-profiles/:employeeId", async (req: Request, res: Response) => {
    try {
      const employeeId = parseInt(req.params.employeeId);
      const profile = await storage.getSalaryProfile(employeeId);
      if (!profile) {
        return res.status(404).json({ error: "ملف الراتب غير موجود" });
      }
      res.json(profile);
    } catch (error) {
      console.error("Error fetching salary profile:", error);
      res.status(500).json({ error: "فشل في جلب ملف الراتب" });
    }
  });
  
  app.post("/api/salary-profiles", async (req: Request, res: Response) => {
    try {
      const validated = insertSalaryProfileSchema.parse(req.body);
      const profile = await storage.createOrUpdateSalaryProfile(validated);
      res.status(201).json(profile);
    } catch (error) {
      console.error("Error creating/updating salary profile:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في حفظ ملف الراتب" });
    }
  });
  
  app.delete("/api/salary-profiles/:employeeId", async (req: Request, res: Response) => {
    try {
      const employeeId = parseInt(req.params.employeeId);
      const success = await storage.deleteSalaryProfile(employeeId);
      if (!success) {
        return res.status(404).json({ error: "ملف الراتب غير موجود" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting salary profile:", error);
      res.status(500).json({ error: "فشل في حذف ملف الراتب" });
    }
  });
  
  // ========================================
  // PAYROLL RUNS (مسيّرات الرواتب)
  // ========================================
  
  app.get("/api/payroll-runs", async (req: Request, res: Response) => {
    try {
      const runs = await storage.getPayrollRuns();
      res.json(runs);
    } catch (error) {
      console.error("Error fetching payroll runs:", error);
      res.status(500).json({ error: "فشل في جلب مسيّرات الرواتب" });
    }
  });
  
  app.get("/api/payroll-runs/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const run = await storage.getPayrollRun(id);
      if (!run) {
        return res.status(404).json({ error: "مسيّر الرواتب غير موجود" });
      }
      res.json(run);
    } catch (error) {
      console.error("Error fetching payroll run:", error);
      res.status(500).json({ error: "فشل في جلب مسيّر الرواتب" });
    }
  });
  
  app.post("/api/payroll-runs", async (req: Request, res: Response) => {
    try {
      const { periodMonth, periodYear, notes, employeeIds } = req.body;
      
      // Convert month/year to proper format
      const month = parseInt(periodMonth);
      const year = parseInt(periodYear);
      
      if (isNaN(month) || month < 1 || month > 12) {
        return res.status(400).json({ error: "الشهر غير صحيح" });
      }
      if (isNaN(year) || year < 2020 || year > 2100) {
        return res.status(400).json({ error: "السنة غير صحيحة" });
      }
      
      // Arabic month names
      const monthNames = ["يناير", "فبراير", "مارس", "إبريل", "مايو", "يونيو",
                          "يوليو", "أغسطس", "سبتمبر", "أكتوبر", "نوفمبر", "ديسمبر"];
      
      const periodName = `${monthNames[month - 1]} ${year}`;
      const periodStartDate = `${year}-${String(month).padStart(2, '0')}-01`;
      const lastDay = new Date(year, month, 0).getDate();
      const periodEndDate = `${year}-${String(month).padStart(2, '0')}-${String(lastDay).padStart(2, '0')}`;
      
      const validated = insertPayrollRunSchema.parse({
        periodName,
        periodStartDate,
        periodEndDate,
        status: 'draft',
        notes: notes || null,
      });
      
      // Store employee selection for later use in generatePayrollLines
      const run = await storage.createPayrollRun(validated);
      
      // If employeeIds provided, generate lines immediately for selected employees
      if (employeeIds && Array.isArray(employeeIds) && employeeIds.length > 0) {
        await storage.generatePayrollLines(run.id, employeeIds);
      }
      
      res.status(201).json(run);
    } catch (error) {
      console.error("Error creating payroll run:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في إنشاء مسيّر الرواتب" });
    }
  });
  
  app.post("/api/payroll-runs/:id/generate-lines", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { employeeIds } = req.body || {};
      const lines = await storage.generatePayrollLines(id, employeeIds);
      res.status(201).json(lines);
    } catch (error: any) {
      console.error("Error generating payroll lines:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في توليد بنود الرواتب" });
    }
  });
  
  app.post("/api/payroll-runs/:id/approve", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const run = await storage.approvePayrollRun(id);
      if (!run) {
        return res.status(404).json({ error: "مسيّر الرواتب غير موجود" });
      }
      res.json(run);
    } catch (error: any) {
      console.error("Error approving payroll run:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في اعتماد مسيّر الرواتب" });
    }
  });
  
  app.post("/api/payroll-runs/:id/mark-paid", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const run = await storage.markPayrollRunAsPaid(id);
      if (!run) {
        return res.status(404).json({ error: "مسيّر الرواتب غير موجود" });
      }
      res.json(run);
    } catch (error: any) {
      console.error("Error marking payroll run as paid:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في وسم مسيّر الرواتب كمدفوع" });
    }
  });
  
  // ========================================
  // PAYROLL LINES (بنود الرواتب)
  // ========================================
  
  app.get("/api/payroll-runs/:id/lines", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const lines = await storage.getPayrollLines(id);
      res.json(lines);
    } catch (error) {
      console.error("Error fetching payroll lines:", error);
      res.status(500).json({ error: "فشل في جلب بنود الرواتب" });
    }
  });
  
  app.put("/api/payroll-lines/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const line = await storage.updatePayrollLine(id, req.body);
      if (!line) {
        return res.status(404).json({ error: "بند الراتب غير موجود" });
      }
      res.json(line);
    } catch (error) {
      console.error("Error updating payroll line:", error);
      res.status(500).json({ error: "فشل في تحديث بند الراتب" });
    }
  });
  
  // ========================================
  // ORDERS (طلبات الصالون)
  // ========================================
  
  // GET /api/orders - List orders with filters
  app.get("/api/orders", async (req: Request, res: Response) => {
    try {
      const filters: OrderFilters = {};
      
      if (req.query.date) {
        filters.date = req.query.date as string;
      }
      if (req.query.status && orderStatuses.includes(req.query.status as any)) {
        filters.status = req.query.status as any;
      }
      if (req.query.source && orderSources.includes(req.query.source as any)) {
        filters.source = req.query.source as any;
      }
      if (req.query.orderType && orderTypes.includes(req.query.orderType as any)) {
        filters.orderType = req.query.orderType as any;
      }
      if (req.query.clientId) {
        filters.clientId = parseInt(req.query.clientId as string);
      }
      if (req.query.branchId) {
        filters.branchId = parseInt(req.query.branchId as string);
      }
      if (req.query.hasRating !== undefined) {
        filters.hasRating = req.query.hasRating === 'true';
      }
      if (req.query.isScheduled !== undefined) {
        filters.isScheduled = req.query.isScheduled === 'true';
      }
      
      const orders = await storage.getOrders(filters);
      res.json(orders);
    } catch (error) {
      console.error("Error fetching orders:", error);
      res.status(500).json({ error: "فشل في جلب الطلبات" });
    }
  });
  
  // GET /api/orders/views/today - Today's orders
  app.get("/api/orders/views/today", async (req: Request, res: Response) => {
    try {
      const today = new Date().toISOString().split('T')[0];
      const orders = await storage.getOrders({ date: today });
      res.json(orders);
    } catch (error) {
      console.error("Error fetching today's orders:", error);
      res.status(500).json({ error: "فشل في جلب طلبات اليوم" });
    }
  });
  
  // GET /api/orders/views/app - App orders (mobile app source)
  app.get("/api/orders/views/app", async (req: Request, res: Response) => {
    try {
      const orders = await storage.getOrders({ source: 'app' });
      res.json(orders);
    } catch (error) {
      console.error("Error fetching app orders:", error);
      res.status(500).json({ error: "فشل في جلب طلبات التطبيق" });
    }
  });
  
  // GET /api/orders/views/cancelled - Cancelled orders
  app.get("/api/orders/views/cancelled", async (req: Request, res: Response) => {
    try {
      const orders = await storage.getOrders({ status: 'canceled' });
      res.json(orders);
    } catch (error) {
      console.error("Error fetching cancelled orders:", error);
      res.status(500).json({ error: "فشل في جلب الطلبات الملغية" });
    }
  });
  
  // GET /api/orders/views/scheduled - Scheduled orders
  app.get("/api/orders/views/scheduled", async (req: Request, res: Response) => {
    try {
      const orders = await storage.getOrders({ isScheduled: true });
      res.json(orders);
    } catch (error) {
      console.error("Error fetching scheduled orders:", error);
      res.status(500).json({ error: "فشل في جلب الطلبات المجدولة" });
    }
  });
  
  // GET /api/orders/views/rated - Orders with ratings
  app.get("/api/orders/views/rated", async (req: Request, res: Response) => {
    try {
      const orders = await storage.getOrders({ hasRating: true });
      res.json(orders);
    } catch (error) {
      console.error("Error fetching rated orders:", error);
      res.status(500).json({ error: "فشل في جلب الطلبات المقيمة" });
    }
  });
  
  // GET /api/orders/views/gifts - Gift orders
  app.get("/api/orders/views/gifts", async (req: Request, res: Response) => {
    try {
      const orders = await storage.getOrders({ orderType: 'gift' });
      res.json(orders);
    } catch (error) {
      console.error("Error fetching gift orders:", error);
      res.status(500).json({ error: "فشل في جلب طلبات الهدايا" });
    }
  });
  
  // GET /api/orders/:id - Get order details
  app.get("/api/orders/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const order = await storage.getOrder(id);
      if (!order) {
        return res.status(404).json({ error: "الطلب غير موجود" });
      }
      res.json(order);
    } catch (error) {
      console.error("Error fetching order:", error);
      res.status(500).json({ error: "فشل في جلب الطلب" });
    }
  });
  
  // POST /api/orders - Create new order
  app.post("/api/orders", async (req: Request, res: Response) => {
    try {
      const request: CreateOrderRequest = req.body;
      
      // Validate required fields
      if (!request.services || request.services.length === 0) {
        return res.status(400).json({ error: "يجب إضافة خدمة واحدة على الأقل" });
      }
      if (!request.createdByEmployeeId) {
        return res.status(400).json({ error: "الموظف المنشئ مطلوب" });
      }
      
      // Validate gift order fields
      if (request.orderType === "gift") {
        if (!request.giftToCustomerId) {
          return res.status(400).json({ error: "يجب تحديد العميل المُهدى إليه للهدية" });
        }
        if (!request.giftExpiryDate) {
          return res.status(400).json({ error: "يجب تحديد تاريخ انتهاء الهدية" });
        }
        
        // Validate expiry date is in the future
        const expiryDate = new Date(request.giftExpiryDate);
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        if (expiryDate < today) {
          return res.status(400).json({ error: "تاريخ انتهاء الهدية يجب أن يكون في المستقبل" });
        }
        
        // Validate gift source
        if (request.giftFromType === "customer" && !request.giftFromCustomerId) {
          return res.status(400).json({ error: "يجب تحديد العميل المُهدي" });
        }
        if (request.giftFromType === "supervisor" && !request.giftFromSupervisorId) {
          return res.status(400).json({ error: "يجب تحديد المشرف/المدير المُهدي" });
        }
        
        // Verify gift recipient exists
        const recipient = await storage.getCustomer(request.giftToCustomerId);
        if (!recipient) {
          return res.status(400).json({ error: "العميل المُهدى إليه غير موجود" });
        }
        
        // Verify gift sender exists (if customer)
        if (request.giftFromType === "customer" && request.giftFromCustomerId) {
          const sender = await storage.getCustomer(request.giftFromCustomerId);
          if (!sender) {
            return res.status(400).json({ error: "العميل المُهدي غير موجود" });
          }
        }
      }
      
      const order = await storage.createOrder(request);
      res.status(201).json(order);
    } catch (error: any) {
      console.error("Error creating order:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في إنشاء الطلب" });
    }
  });
  
  // PUT /api/orders/:id - Update order
  app.put("/api/orders/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const validated = updateOrderSchema.parse(req.body);
      
      const order = await storage.updateOrder(id, validated);
      if (!order) {
        return res.status(404).json({ error: "الطلب غير موجود" });
      }
      res.json(order);
    } catch (error: any) {
      console.error("Error updating order:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في تحديث الطلب" });
    }
  });
  
  // POST /api/orders/:id/rate - Rate a completed order
  app.post("/api/orders/:id/rate", async (req: Request, res: Response) => {
    try {
      const orderId = parseInt(req.params.id);
      const ratingData = insertOrderRatingSchema.omit({ orderId: true }).parse(req.body);
      
      const rating = await storage.createOrUpdateOrderRating(orderId, ratingData);
      res.status(201).json(rating);
    } catch (error: any) {
      console.error("Error rating order:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في تقييم الطلب" });
    }
  });
  
  // GET /api/orders/:id/rating - Get order rating
  app.get("/api/orders/:id/rating", async (req: Request, res: Response) => {
    try {
      const orderId = parseInt(req.params.id);
      const rating = await storage.getOrderRating(orderId);
      if (!rating) {
        return res.status(404).json({ error: "لا يوجد تقييم لهذا الطلب" });
      }
      res.json(rating);
    } catch (error) {
      console.error("Error fetching order rating:", error);
      res.status(500).json({ error: "فشل في جلب التقييم" });
    }
  });
  
  // GET /api/orders/:id/payments - Get order payments
  app.get("/api/orders/:id/payments", async (req: Request, res: Response) => {
    try {
      const orderId = parseInt(req.params.id);
      const payments = await storage.getOrderPayments(orderId);
      res.json(payments);
    } catch (error) {
      console.error("Error fetching order payments:", error);
      res.status(500).json({ error: "فشل في جلب مدفوعات الطلب" });
    }
  });
  
  // POST /api/orders/:id/payments - Create order payment
  app.post("/api/orders/:id/payments", async (req: Request, res: Response) => {
    try {
      const orderId = parseInt(req.params.id);
      const paymentData = {
        orderId,
        paymentMethodId: req.body.paymentMethodId,
        paymentDate: req.body.paymentDate,
        amount: req.body.amount,
        notes: req.body.notes,
      };
      
      const payment = await storage.createOrderPayment(paymentData);
      res.status(201).json(payment);
    } catch (error: any) {
      console.error("Error creating order payment:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في إضافة الدفعة" });
    }
  });
  
  // GET /api/orders/:id/print-data - Get order print data for thermal receipt
  app.get("/api/orders/:id/print-data", async (req: Request, res: Response) => {
    try {
      const orderId = parseInt(req.params.id);
      const order = await storage.getOrder(orderId);
      
      if (!order) {
        return res.status(404).json({ error: "الطلب غير موجود" });
      }
      
      // Get company settings
      const settings = await storage.getSystemSettings();
      const getSettingValue = (key: string) => {
        const setting = settings.find(s => s.key === key);
        return setting?.value || "";
      };
      
      // Get payments
      const payments = await storage.getOrderPayments(orderId);
      
      // Get credit note info for cancelled orders
      let creditNoteInfo: { creditNoteNumber: string; originalInvoiceNumber: string; refundPaymentMethodName?: string } | null = null;
      if (order.status === "canceled") {
        const returns = await storage.getOrderReturns(orderId);
        const cancellationReturn = returns.find(r => r.returnType === "cancellation");
        if (cancellationReturn) {
          // Get payment method name for refund
          let refundPaymentMethodName: string | undefined;
          if (cancellationReturn.refundPaymentMethodId) {
            const paymentMethods = await storage.getPaymentMethods();
            const method = paymentMethods.find(m => m.id === cancellationReturn.refundPaymentMethodId);
            refundPaymentMethodName = method?.nameAr;
          }
          creditNoteInfo = {
            creditNoteNumber: cancellationReturn.creditNoteNumber,
            originalInvoiceNumber: cancellationReturn.originalInvoiceNumber || order.orderNumber,
            refundPaymentMethodName,
          };
        }
      }
      
      // Status labels
      const statusLabels: Record<string, string> = {
        new: "جديدة",
        in_progress: "قيد التنفيذ",
        completed: "مكتملة",
        canceled: "ملغاة",
        scheduled: "مجدولة",
        postponed: "مؤجلة",
        deferred: "مؤجلة",
        returned: "مرتجعة",
      };
      
      // Build order lines
      const lines: any[] = [];
      
      // Service lines
      for (const svc of order.services) {
        lines.push({
          type: "service",
          lineId: svc.id,
          name: svc.service?.nameAr || "خدمة",
          executionTime: svc.scheduledAt || svc.completedAt || order.orderDate,
          employeeName: svc.executingEmployee 
            ? `${svc.executingEmployee.nameAr || ''}`.trim() 
            : null,
          discount: 0, // Line-level discount if available
          price: svc.lineTotal,
          quantity: svc.quantity,
        });
      }
      
      // Product lines
      for (const prod of order.products) {
        lines.push({
          type: "product",
          lineId: prod.id,
          name: prod.product?.nameAr || "منتج",
          executionTime: null,
          employeeName: null,
          discount: 0,
          price: prod.lineTotal,
          quantity: prod.quantity,
        });
      }
      
      // Calculate totals
      const paidAmount = payments.reduce((sum, p) => sum + p.amount, 0);
      const remainingAmount = order.totalAmount - paidAmount;
      
      // Build print data
      const now = new Date();
      const printData = {
        companyInfo: {
          companyNameAr: getSettingValue("company_name_ar"),
          phone: getSettingValue("company_phone"),
          branchName: getSettingValue("company_branch_name"),
          crNumber: getSettingValue("company_cr_number"),
          vatNumber: getSettingValue("company_vat_number"),
          addressLine: getSettingValue("company_address"),
          city: getSettingValue("company_city"),
          extraNote: getSettingValue("invoice_extra_note"),
        },
        invoiceNumber: order.orderNumber,
        status: order.status,
        statusLabel: statusLabels[order.status] || order.status,
        invoiceDate: new Date(order.orderDate).toLocaleDateString("ar-SA"),
        invoiceTime: new Date(order.orderDate).toLocaleTimeString("ar-SA", { hour: "2-digit", minute: "2-digit" }),
        invoiceTimestampISO: new Date(order.orderDate).toISOString(),
        bookingDate: null as string | null,
        bookingTime: null as string | null,
        printDate: now.toLocaleDateString("ar-SA"),
        printTime: now.toLocaleTimeString("ar-SA", { hour: "2-digit", minute: "2-digit" }),
        customerName: order.client?.nameAr || "عميل نقدي",
        payments: payments.map(p => ({
          methodName: p.paymentMethod?.nameAr || "طريقة دفع",
          amount: p.amount,
        })),
        cashierName: order.createdByEmployee?.nameAr || "—",
        lines,
        subtotalBeforeDiscount: order.subtotalAmount + order.couponDiscountAmount,
        discountAmount: order.couponDiscountAmount,
        vatRate: 0.15,
        vatAmount: order.vatAmount,
        totalWithVat: order.totalAmount,
        paidAmount,
        remainingAmount,
        // Credit note info for cancelled orders
        creditNoteInfo,
        isCancelled: order.status === "canceled",
      };
      
      res.json(printData);
    } catch (error) {
      console.error("Error fetching order print data:", error);
      res.status(500).json({ error: "فشل في جلب بيانات الطباعة" });
    }
  });
  
  // GET /api/orders/:orderId/services - Get services for an order
  app.get("/api/orders/:orderId/services", async (req: Request, res: Response) => {
    try {
      const orderId = parseInt(req.params.orderId);
      const services = await storage.getOrderServices(orderId);
      res.json(services);
    } catch (error) {
      console.error("Error fetching order services:", error);
      res.status(500).json({ error: "فشل في جلب خدمات الطلب" });
    }
  });
  
  // PUT /api/order-services/:id - Update a specific service in an order
  app.put("/api/order-services/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const validated = updateOrderServiceSchema.parse(req.body);
      
      const service = await storage.updateOrderService(id, validated);
      if (!service) {
        return res.status(404).json({ error: "الخدمة غير موجودة" });
      }
      res.json(service);
    } catch (error) {
      console.error("Error updating order service:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      res.status(500).json({ error: "فشل في تحديث الخدمة" });
    }
  });
  
  // POST /api/orders/:id/defer - Defer an order
  app.post("/api/orders/:id/defer", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { reason } = req.body;
      
      const order = await storage.updateOrder(id, {
        status: "deferred",
        deferredAt: new Date(),
        deferredReason: reason || null,
      });
      
      if (!order) {
        return res.status(404).json({ error: "الطلب غير موجود" });
      }
      res.json(order);
    } catch (error) {
      console.error("Error deferring order:", error);
      res.status(500).json({ error: "فشل في تأجيل الطلب" });
    }
  });
  
  // POST /api/orders/:id/cancel - Cancel an order (with reversal entries if completed)
  app.post("/api/orders/:id/cancel", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { reason, refundPaymentMethodId } = req.body;
      
      const order = await storage.cancelOrder(id, reason, refundPaymentMethodId);
      
      if (!order) {
        return res.status(404).json({ error: "الطلب غير موجود" });
      }
      res.json(order);
    } catch (error: any) {
      console.error("Error cancelling order:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في إلغاء الطلب" });
    }
  });
  
  // POST /api/orders/:id/pay-cancelled - Pay a cancelled order (creates revenue journal entry)
  app.post("/api/orders/:id/pay-cancelled", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { amount, paymentMethodId, notes } = req.body;
      
      // Validate required fields
      if (!amount || amount <= 0) {
        return res.status(400).json({ error: "مبلغ السداد مطلوب ويجب أن يكون أكبر من صفر" });
      }
      if (!paymentMethodId) {
        return res.status(400).json({ error: "طريقة الدفع مطلوبة" });
      }
      
      const result = await storage.payCancelledOrder(id, {
        amount: parseFloat(amount),
        paymentMethodId: parseInt(paymentMethodId),
        notes: notes || undefined,
      });
      
      res.json({
        success: true,
        message: "تم سداد الطلب الملغي بنجاح",
        journalEntryId: result.journalEntryId,
      });
    } catch (error: any) {
      console.error("Error paying cancelled order:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في سداد الطلب الملغي" });
    }
  });
  
  // GET /api/orders/:orderId/returns - Get returns for an order
  app.get("/api/orders/:orderId/returns", async (req: Request, res: Response) => {
    try {
      const orderId = parseInt(req.params.orderId);
      const returns = await storage.getOrderReturns(orderId);
      res.json(returns);
    } catch (error) {
      console.error("Error fetching order returns:", error);
      res.status(500).json({ error: "فشل في جلب مرتجعات الطلب" });
    }
  });
  
  // POST /api/orders/:orderId/returns - Create a return for an order
  app.post("/api/orders/:orderId/returns", async (req: Request, res: Response) => {
    try {
      const orderId = parseInt(req.params.orderId);
      const { lines, ...returnData } = req.body;
      
      const validated = insertOrderReturnSchema.parse({
        ...returnData,
        orderId,
      });
      
      const orderReturn = await storage.createOrderReturn(validated, lines || []);
      res.status(201).json(orderReturn);
    } catch (error: any) {
      console.error("Error creating order return:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في إنشاء المرتجع" });
    }
  });
  
  // GET /api/order-returns - Get all returns
  app.get("/api/order-returns", async (req: Request, res: Response) => {
    try {
      const returns = await storage.getOrderReturns();
      res.json(returns);
    } catch (error) {
      console.error("Error fetching order returns:", error);
      res.status(500).json({ error: "فشل في جلب المرتجعات" });
    }
  });
  
  // GET /api/order-returns/:id - Get specific return
  app.get("/api/order-returns/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const orderReturn = await storage.getOrderReturn(id);
      if (!orderReturn) {
        return res.status(404).json({ error: "المرتجع غير موجود" });
      }
      res.json(orderReturn);
    } catch (error) {
      console.error("Error fetching order return:", error);
      res.status(500).json({ error: "فشل في جلب المرتجع" });
    }
  });
  
  // ========================================
  // APPOINTMENTS (حجوزات التطبيق)
  // ========================================
  
  // GET /api/appointments - List appointments with filters
  app.get("/api/appointments", async (req: Request, res: Response) => {
    try {
      const { date, status, type, source, clientId, branchId } = req.query;
      
      const filters: any = {};
      if (date) filters.date = date as string;
      if (status) filters.status = status as string;
      if (type) filters.appointmentType = type as string;
      if (source) filters.source = source as string;
      if (clientId) filters.clientId = parseInt(clientId as string);
      if (branchId) filters.branchId = parseInt(branchId as string);
      
      const appointments = await storage.getAppointments(filters);
      res.json(appointments);
    } catch (error) {
      console.error("Error fetching appointments:", error);
      res.status(500).json({ error: "فشل في جلب الحجوزات" });
    }
  });
  
  // GET /api/appointments/:id - Get single appointment
  app.get("/api/appointments/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const appointment = await storage.getAppointment(id);
      if (!appointment) {
        return res.status(404).json({ error: "الحجز غير موجود" });
      }
      res.json(appointment);
    } catch (error) {
      console.error("Error fetching appointment:", error);
      res.status(500).json({ error: "فشل في جلب الحجز" });
    }
  });
  
  // POST /api/appointments - Create new appointment
  app.post("/api/appointments", async (req: Request, res: Response) => {
    try {
      const request = req.body;
      
      // Basic validation
      if (!request.clientId) {
        return res.status(400).json({ error: "العميل مطلوب" });
      }
      if (!request.scheduledAt) {
        return res.status(400).json({ error: "موعد الحجز مطلوب" });
      }
      if (!request.appointmentType) {
        return res.status(400).json({ error: "نوع الحجز مطلوب" });
      }
      if (!request.services || request.services.length === 0) {
        return res.status(400).json({ error: "يجب اختيار خدمة واحدة على الأقل" });
      }
      
      const appointment = await storage.createAppointment(request);
      res.status(201).json(appointment);
    } catch (error: any) {
      console.error("Error creating appointment:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في إنشاء الحجز" });
    }
  });
  
  // PUT /api/appointments/:id - Update appointment
  app.put("/api/appointments/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const updates = req.body;
      
      const appointment = await storage.updateAppointment(id, updates);
      if (!appointment) {
        return res.status(404).json({ error: "الحجز غير موجود" });
      }
      res.json(appointment);
    } catch (error: any) {
      console.error("Error updating appointment:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في تحديث الحجز" });
    }
  });
  
  // POST /api/appointments/:id/confirm - Confirm appointment
  app.post("/api/appointments/:id/confirm", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const appointment = await storage.confirmAppointment(id);
      if (!appointment) {
        return res.status(404).json({ error: "الحجز غير موجود" });
      }
      res.json(appointment);
    } catch (error: any) {
      console.error("Error confirming appointment:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في تأكيد الحجز" });
    }
  });
  
  // POST /api/appointments/:id/cancel - Cancel appointment
  app.post("/api/appointments/:id/cancel", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { reason } = req.body;
      
      const appointment = await storage.cancelAppointment(id, reason);
      if (!appointment) {
        return res.status(404).json({ error: "الحجز غير موجود" });
      }
      res.json(appointment);
    } catch (error: any) {
      console.error("Error canceling appointment:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في إلغاء الحجز" });
    }
  });
  
  // ========================================
  // SYSTEM SETTINGS (إعدادات النظام)
  // ========================================
  
  // GET /api/settings - Get all system settings
  app.get("/api/settings", async (req: Request, res: Response) => {
    try {
      const settings = await storage.getSystemSettings();
      res.json(settings);
    } catch (error) {
      console.error("Error fetching settings:", error);
      res.status(500).json({ error: "فشل في جلب الإعدادات" });
    }
  });
  
  // GET /api/settings/:key - Get single setting by key
  app.get("/api/settings/:key", async (req: Request, res: Response) => {
    try {
      const setting = await storage.getSystemSetting(req.params.key);
      if (!setting) {
        return res.status(404).json({ error: "الإعداد غير موجود" });
      }
      res.json(setting);
    } catch (error) {
      console.error("Error fetching setting:", error);
      res.status(500).json({ error: "فشل في جلب الإعداد" });
    }
  });
  
  // PUT /api/settings/:key - Set or update a setting
  app.put("/api/settings/:key", async (req: Request, res: Response) => {
    try {
      const { value, description, valueType } = req.body;
      
      if (value === undefined || value === null) {
        return res.status(400).json({ error: "القيمة مطلوبة" });
      }
      
      const setting = await storage.setSystemSetting(
        req.params.key,
        String(value),
        description,
        valueType
      );
      res.json(setting);
    } catch (error) {
      console.error("Error setting:", error);
      res.status(500).json({ error: "فشل في تحديث الإعداد" });
    }
  });
  
  // ========================================
  // CAFETERIA PRODUCTS (منتجات الكافتيريا)
  // ========================================
  
  app.get("/api/cafeteria/products", async (req: Request, res: Response) => {
    try {
      const products = await storage.getCafeteriaProducts();
      res.json(products);
    } catch (error) {
      console.error("Error fetching cafeteria products:", error);
      res.status(500).json({ error: "فشل في جلب منتجات الكافتيريا" });
    }
  });
  
  app.get("/api/cafeteria/products/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const product = await storage.getCafeteriaProduct(id);
      if (!product) {
        return res.status(404).json({ error: "المنتج غير موجود" });
      }
      res.json(product);
    } catch (error) {
      console.error("Error fetching cafeteria product:", error);
      res.status(500).json({ error: "فشل في جلب المنتج" });
    }
  });
  
  app.post("/api/cafeteria/products", async (req: Request, res: Response) => {
    try {
      const validated = insertCafeteriaProductSchema.parse(req.body);
      const product = await storage.createCafeteriaProduct(validated);
      res.status(201).json(product);
    } catch (error) {
      console.error("Error creating cafeteria product:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في إنشاء المنتج" });
    }
  });
  
  app.put("/api/cafeteria/products/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const product = await storage.updateCafeteriaProduct(id, req.body);
      if (!product) {
        return res.status(404).json({ error: "المنتج غير موجود" });
      }
      res.json(product);
    } catch (error) {
      console.error("Error updating cafeteria product:", error);
      const dbError = translateDbError(error);
      if (dbError) {
        return res.status(400).json({ error: dbError });
      }
      res.status(500).json({ error: "فشل في تحديث المنتج" });
    }
  });
  
  app.delete("/api/cafeteria/products/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteCafeteriaProduct(id);
      if (!success) {
        return res.status(404).json({ error: "المنتج غير موجود" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting cafeteria product:", error);
      res.status(500).json({ error: "فشل في حذف المنتج" });
    }
  });
  
  // ========================================
  // CAFETERIA PURCHASE INVOICES (فواتير مشتريات الكافتيريا)
  // ========================================
  
  app.get("/api/cafeteria/purchase-invoices", async (req: Request, res: Response) => {
    try {
      const invoices = await storage.getCafeteriaPurchaseInvoices();
      res.json(invoices);
    } catch (error) {
      console.error("Error fetching cafeteria purchase invoices:", error);
      res.status(500).json({ error: "فشل في جلب فواتير المشتريات" });
    }
  });
  
  app.get("/api/cafeteria/purchase-invoices/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const invoice = await storage.getCafeteriaPurchaseInvoice(id);
      if (!invoice) {
        return res.status(404).json({ error: "الفاتورة غير موجودة" });
      }
      res.json(invoice);
    } catch (error) {
      console.error("Error fetching cafeteria purchase invoice:", error);
      res.status(500).json({ error: "فشل في جلب الفاتورة" });
    }
  });
  
  app.post("/api/cafeteria/purchase-invoices", async (req: Request, res: Response) => {
    try {
      const request: CreateCafeteriaPurchaseInvoiceRequest = req.body;
      
      if (!request.lines || request.lines.length === 0) {
        return res.status(400).json({ error: "يجب إضافة منتج واحد على الأقل" });
      }
      if (!request.supplierId) {
        return res.status(400).json({ error: "المورد مطلوب" });
      }
      
      const invoice = await storage.createCafeteriaPurchaseInvoice(request);
      res.status(201).json(invoice);
    } catch (error: any) {
      console.error("Error creating cafeteria purchase invoice:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في إنشاء الفاتورة" });
    }
  });
  
  // ========================================
  // CAFETERIA PURCHASE PAYMENTS (مدفوعات مشتريات الكافتيريا)
  // ========================================
  
  app.get("/api/cafeteria/purchase-payments", async (req: Request, res: Response) => {
    try {
      const invoiceId = req.query.invoiceId ? parseInt(req.query.invoiceId as string) : undefined;
      const payments = await storage.getCafeteriaPurchasePayments(invoiceId);
      res.json(payments);
    } catch (error) {
      console.error("Error fetching cafeteria purchase payments:", error);
      res.status(500).json({ error: "فشل في جلب المدفوعات" });
    }
  });
  
  app.post("/api/cafeteria/purchase-payments", async (req: Request, res: Response) => {
    try {
      const validated = insertCafeteriaPurchasePaymentSchema.parse(req.body);
      const payment = await storage.createCafeteriaPurchasePayment(validated);
      res.status(201).json(payment);
    } catch (error) {
      console.error("Error creating cafeteria purchase payment:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      res.status(500).json({ error: "فشل في إنشاء الدفعة" });
    }
  });
  
  // ========================================
  // CAFETERIA SALES ORDERS (طلبات مبيعات الكافتيريا)
  // ========================================
  
  app.get("/api/cafeteria/sales-orders", async (req: Request, res: Response) => {
    try {
      const filters: CafeteriaOrderFilters = {};
      
      if (req.query.date) {
        filters.date = req.query.date as string;
      }
      if (req.query.status && cafeteriaOrderStatuses.includes(req.query.status as any)) {
        filters.status = req.query.status as any;
      }
      if (req.query.branchId) {
        filters.branchId = parseInt(req.query.branchId as string);
      }
      if (req.query.createdByEmployeeId) {
        filters.createdByEmployeeId = parseInt(req.query.createdByEmployeeId as string);
      }
      
      const orders = await storage.getCafeteriaSalesOrders(filters);
      res.json(orders);
    } catch (error) {
      console.error("Error fetching cafeteria sales orders:", error);
      res.status(500).json({ error: "فشل في جلب طلبات المبيعات" });
    }
  });
  
  app.get("/api/cafeteria/sales-orders/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const order = await storage.getCafeteriaSalesOrder(id);
      if (!order) {
        return res.status(404).json({ error: "الطلب غير موجود" });
      }
      res.json(order);
    } catch (error) {
      console.error("Error fetching cafeteria sales order:", error);
      res.status(500).json({ error: "فشل في جلب الطلب" });
    }
  });
  
  app.post("/api/cafeteria/sales-orders", async (req: Request, res: Response) => {
    try {
      const request: CreateCafeteriaSalesOrderRequest = req.body;
      
      if (!request.lines || request.lines.length === 0) {
        return res.status(400).json({ error: "يجب إضافة منتج واحد على الأقل" });
      }
      if (!request.createdByEmployeeId) {
        return res.status(400).json({ error: "الموظف المنشئ مطلوب" });
      }
      
      const order = await storage.createCafeteriaSalesOrder(request);
      res.status(201).json(order);
    } catch (error: any) {
      console.error("Error creating cafeteria sales order:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في إنشاء الطلب" });
    }
  });
  
  app.put("/api/cafeteria/sales-orders/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const validated = updateCafeteriaSalesOrderSchema.parse(req.body);
      
      const order = await storage.updateCafeteriaSalesOrder(id, validated);
      if (!order) {
        return res.status(404).json({ error: "الطلب غير موجود" });
      }
      res.json(order);
    } catch (error) {
      console.error("Error updating cafeteria sales order:", error);
      if (error instanceof z.ZodError) {
        const firstError = error.errors[0];
        return res.status(400).json({ error: firstError.message });
      }
      res.status(500).json({ error: "فشل في تحديث الطلب" });
    }
  });
  
  // ========================================
  // CAFETERIA INVENTORY REPORTS (تقارير مخزون الكافتيريا)
  // ========================================
  
  app.get("/api/cafeteria/inventory/summary", async (req: Request, res: Response) => {
    try {
      const summary = await storage.getCafeteriaInventorySummary();
      res.json(summary);
    } catch (error) {
      console.error("Error fetching cafeteria inventory summary:", error);
      res.status(500).json({ error: "فشل في جلب ملخص المخزون" });
    }
  });
  
  app.get("/api/cafeteria/inventory/movements/:productId", async (req: Request, res: Response) => {
    try {
      const productId = parseInt(req.params.productId);
      const movements = await storage.getCafeteriaInventoryMovements(productId);
      res.json(movements);
    } catch (error) {
      console.error("Error fetching cafeteria inventory movements:", error);
      res.status(500).json({ error: "فشل في جلب حركة المخزون" });
    }
  });
  
  app.get("/api/cafeteria/stock-batches", async (req: Request, res: Response) => {
    try {
      const productId = req.query.productId ? parseInt(req.query.productId as string) : undefined;
      const batches = await storage.getCafeteriaStockBatches(productId);
      res.json(batches);
    } catch (error) {
      console.error("Error fetching cafeteria stock batches:", error);
      res.status(500).json({ error: "فشل في جلب دفعات المخزون" });
    }
  });
  
  // ========================================
  // CAFETERIA EMPLOYEE CUSTODIES (عهد موظفي الكافتيريا)
  // ========================================
  
  app.get("/api/cafeteria/custodies", async (req: Request, res: Response) => {
    try {
      const filters: { employeeId?: number; status?: string } = {};
      if (req.query.employeeId) {
        filters.employeeId = parseInt(req.query.employeeId as string);
      }
      if (req.query.status) {
        filters.status = req.query.status as string;
      }
      const custodies = await storage.getCafeteriaEmployeeCustodies(filters);
      res.json(custodies);
    } catch (error) {
      console.error("Error fetching cafeteria employee custodies:", error);
      res.status(500).json({ error: "فشل في جلب العهد" });
    }
  });
  
  app.get("/api/cafeteria/custodies/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const custody = await storage.getCafeteriaEmployeeCustody(id);
      if (!custody) {
        return res.status(404).json({ error: "العهدة غير موجودة" });
      }
      res.json(custody);
    } catch (error) {
      console.error("Error fetching cafeteria employee custody:", error);
      res.status(500).json({ error: "فشل في جلب العهدة" });
    }
  });
  
  app.post("/api/cafeteria/custodies", async (req: Request, res: Response) => {
    try {
      const custody = await storage.createCafeteriaEmployeeCustody(req.body);
      res.status(201).json(custody);
    } catch (error: any) {
      console.error("Error creating cafeteria employee custody:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في إنشاء العهدة" });
    }
  });
  
  app.put("/api/cafeteria/custodies/:id/close", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { closedByEmployeeId, closureReason, returnedQuantity, closureNotes } = req.body;
      
      if (!closedByEmployeeId) {
        return res.status(400).json({ error: "الموظف المغلق مطلوب" });
      }
      if (!closureReason) {
        return res.status(400).json({ error: "سبب الإغلاق مطلوب" });
      }
      
      const custody = await storage.closeCafeteriaEmployeeCustody(id, {
        closedByEmployeeId,
        closureReason,
        returnedQuantity,
        closureNotes,
      });
      
      if (!custody) {
        return res.status(404).json({ error: "العهدة غير موجودة" });
      }
      res.json(custody);
    } catch (error) {
      console.error("Error closing cafeteria employee custody:", error);
      res.status(500).json({ error: "فشل في إغلاق العهدة" });
    }
  });
  
  app.delete("/api/cafeteria/custodies/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const deleted = await storage.deleteCafeteriaEmployeeCustody(id);
      if (!deleted) {
        return res.status(400).json({ error: "لا يمكن حذف العهدة - العهدة غير موجودة أو مغلقة" });
      }
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting cafeteria employee custody:", error);
      res.status(500).json({ error: "فشل في حذف العهدة" });
    }
  });
  
  // GET /api/cafeteria/sales-orders/:id/print-data - Get cafeteria order print data for thermal receipt (ZATCA compliant)
  app.get("/api/cafeteria/sales-orders/:id/print-data", async (req: Request, res: Response) => {
    try {
      const orderId = parseInt(req.params.id);
      const order = await storage.getCafeteriaSalesOrder(orderId);
      
      if (!order) {
        return res.status(404).json({ error: "الطلب غير موجود" });
      }
      
      // Get company settings
      const settings = await storage.getSystemSettings();
      const getSettingValue = (key: string) => {
        const setting = settings.find(s => s.key === key);
        return setting?.value || "";
      };
      
      // Status labels
      const statusLabels: Record<string, string> = {
        new: "جديد",
        completed: "مكتمل",
        canceled: "ملغي",
      };
      
      // Get credit note info for canceled orders
      let creditNoteInfo: { creditNoteNumber: string; originalInvoiceNumber: string } | null = null;
      if (order.status === "canceled") {
        const returns = await storage.getCafeteriaSalesReturns(orderId);
        const cancellationReturn = returns.find(r => r.returnType === "cancellation");
        if (cancellationReturn) {
          creditNoteInfo = {
            creditNoteNumber: cancellationReturn.creditNoteNumber,
            originalInvoiceNumber: cancellationReturn.originalInvoiceNumber || order.orderNumber,
          };
        }
      }
      
      // Build order lines
      const lines = order.lines.map(line => ({
        type: "product" as const,
        name: line.cafeteriaProduct?.nameAr || "منتج",
        executionTime: null,
        employeeName: null,
        discount: 0,
        price: line.lineTotal,
        quantity: line.quantity,
      }));
      
      // Build print data
      const now = new Date();
      const orderDate = new Date(order.orderDate);
      const printData = {
        companyInfo: {
          companyNameAr: getSettingValue("company_name_ar"),
          phone: getSettingValue("company_phone"),
          branchName: getSettingValue("company_branch_name") || "الكافتيريا",
          crNumber: getSettingValue("company_cr_number"),
          vatNumber: getSettingValue("company_vat_number"),
          addressLine: getSettingValue("company_address"),
          city: getSettingValue("company_city"),
          extraNote: getSettingValue("invoice_extra_note"),
        },
        invoiceNumber: order.orderNumber,
        status: order.status,
        statusLabel: statusLabels[order.status] || order.status,
        invoiceDate: orderDate.toLocaleDateString("ar-SA"),
        invoiceTime: orderDate.toLocaleTimeString("ar-SA", { hour: "2-digit", minute: "2-digit" }),
        invoiceTimestampISO: orderDate.toISOString(),
        bookingDate: null as string | null,
        bookingTime: null as string | null,
        printDate: now.toLocaleDateString("ar-SA"),
        printTime: now.toLocaleTimeString("ar-SA", { hour: "2-digit", minute: "2-digit" }),
        customerName: order.client?.nameAr || "عميل نقدي",
        payments: [] as { methodName: string; amount: number }[],
        cashierName: order.createdByEmployee?.nameAr || "—",
        lines,
        subtotalBeforeDiscount: order.subtotalAmount,
        discountAmount: 0,
        vatRate: 0.15,
        vatAmount: order.vatAmount,
        totalWithVat: order.totalAmount,
        paidAmount: order.totalAmount,
        remainingAmount: 0,
        creditNoteInfo,
        isCancelled: order.status === "canceled",
        isCafeteria: true,
      };
      
      res.json(printData);
    } catch (error) {
      console.error("Error fetching cafeteria order print data:", error);
      res.status(500).json({ error: "فشل في جلب بيانات الطباعة" });
    }
  });
  
  // GET /api/cafeteria/sales-orders/:id/returns - Get cafeteria order returns/credit notes
  app.get("/api/cafeteria/sales-orders/:id/returns", async (req: Request, res: Response) => {
    try {
      const orderId = parseInt(req.params.id);
      const returns = await storage.getCafeteriaSalesReturns(orderId);
      res.json(returns);
    } catch (error) {
      console.error("Error fetching cafeteria order returns:", error);
      res.status(500).json({ error: "فشل في جلب مرتجعات الطلب" });
    }
  });
  
  // POST /api/cafeteria/sales-orders/:id/cancel - Cancel cafeteria order and create ZATCA-compliant credit note
  app.post("/api/cafeteria/sales-orders/:id/cancel", async (req: Request, res: Response) => {
    try {
      const orderId = parseInt(req.params.id);
      const { reason, processedByEmployeeId } = req.body;
      
      if (!reason) {
        return res.status(400).json({ error: "سبب الإلغاء مطلوب" });
      }
      
      const returnRecord = await storage.createCafeteriaSalesCancellationReturn(
        orderId, 
        reason, 
        processedByEmployeeId
      );
      
      res.status(201).json(returnRecord);
    } catch (error: any) {
      console.error("Error canceling cafeteria order:", error);
      if (error.message) {
        return res.status(400).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في إلغاء الطلب" });
    }
  });
  
  // ========================================
  // MARKETING - PACKAGES (الحزم)
  // ========================================
  
  app.get("/api/marketing/packages", async (req: Request, res: Response) => {
    try {
      const filters: any = {};
      if (req.query.is_active !== undefined) {
        filters.isActive = req.query.is_active === "true";
      }
      if (req.query.valid_today === "true") {
        filters.validToday = true;
      }
      const packages = await storage.getPackages(filters);
      res.json(packages);
    } catch (error) {
      console.error("Error fetching packages:", error);
      res.status(500).json({ error: "فشل في جلب الحزم" });
    }
  });
  
  app.get("/api/marketing/packages/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const pkg = await storage.getPackage(id);
      if (!pkg) {
        return res.status(404).json({ error: "الحزمة غير موجودة" });
      }
      res.json(pkg);
    } catch (error) {
      console.error("Error fetching package:", error);
      res.status(500).json({ error: "فشل في جلب الحزمة" });
    }
  });
  
  app.post("/api/marketing/packages", async (req: Request, res: Response) => {
    try {
      const request = req.body;
      
      // Validate required fields
      if (!request.nameAr) {
        return res.status(400).json({ error: "اسم الحزمة مطلوب" });
      }
      if (!request.startDate || !request.endDate) {
        return res.status(400).json({ error: "تاريخ البداية والنهاية مطلوبان" });
      }
      if (request.startDate > request.endDate) {
        return res.status(400).json({ error: "تاريخ البداية يجب أن يكون قبل تاريخ النهاية" });
      }
      if (!request.services || request.services.length === 0) {
        return res.status(400).json({ error: "يجب إضافة خدمة واحدة على الأقل" });
      }
      
      const pkg = await storage.createPackage(request);
      res.status(201).json(pkg);
    } catch (error: any) {
      console.error("Error creating package:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في إنشاء الحزمة" });
    }
  });
  
  app.put("/api/marketing/packages/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const request = req.body;
      
      // Validate dates if both provided
      if (request.startDate && request.endDate && request.startDate > request.endDate) {
        return res.status(400).json({ error: "تاريخ البداية يجب أن يكون قبل تاريخ النهاية" });
      }
      
      const pkg = await storage.updatePackage(id, request);
      if (!pkg) {
        return res.status(404).json({ error: "الحزمة غير موجودة" });
      }
      res.json(pkg);
    } catch (error: any) {
      console.error("Error updating package:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في تحديث الحزمة" });
    }
  });
  
  app.delete("/api/marketing/packages/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deletePackage(id);
      if (!success) {
        return res.status(404).json({ error: "الحزمة غير موجودة" });
      }
      res.json({ message: "تم إلغاء تفعيل الحزمة بنجاح" });
    } catch (error) {
      console.error("Error deleting package:", error);
      res.status(500).json({ error: "فشل في حذف الحزمة" });
    }
  });
  
  // ========================================
  // MARKETING - COUPONS (الكوبونات)
  // ========================================
  
  app.get("/api/marketing/coupons", async (req: Request, res: Response) => {
    try {
      const filters: any = {};
      if (req.query.is_active !== undefined) {
        filters.isActive = req.query.is_active === "true";
      }
      if (req.query.valid_today === "true") {
        filters.validToday = true;
      }
      const coupons = await storage.getCoupons(filters);
      res.json(coupons);
    } catch (error) {
      console.error("Error fetching coupons:", error);
      res.status(500).json({ error: "فشل في جلب الكوبونات" });
    }
  });
  
  app.get("/api/marketing/coupons/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const coupon = await storage.getCoupon(id);
      if (!coupon) {
        return res.status(404).json({ error: "الكوبون غير موجود" });
      }
      res.json(coupon);
    } catch (error) {
      console.error("Error fetching coupon:", error);
      res.status(500).json({ error: "فشل في جلب الكوبون" });
    }
  });
  
  app.get("/api/marketing/coupons/code/:code", async (req: Request, res: Response) => {
    try {
      const code = req.params.code;
      const coupon = await storage.getCouponByCode(code);
      if (!coupon) {
        return res.status(404).json({ error: "الكوبون غير موجود" });
      }
      res.json(coupon);
    } catch (error) {
      console.error("Error fetching coupon by code:", error);
      res.status(500).json({ error: "فشل في جلب الكوبون" });
    }
  });
  
  app.get("/api/marketing/coupons/validate/:code", async (req: Request, res: Response) => {
    try {
      const code = req.params.code;
      const result = await storage.validateCoupon(code);
      if (!result.valid) {
        return res.status(400).json({ error: result.error || "الكوبون غير صالح" });
      }
      res.json(result);
    } catch (error) {
      console.error("Error validating coupon:", error);
      res.status(500).json({ error: "فشل في التحقق من الكوبون" });
    }
  });
  
  // POST /api/marketing/coupons/validate - Validate coupon with order items context
  app.post("/api/marketing/coupons/validate", async (req: Request, res: Response) => {
    try {
      const { code, services, products } = req.body;
      
      if (!code) {
        return res.status(400).json({ error: "كود الكوبون مطلوب" });
      }
      
      const result = await storage.validateCoupon(code, services, products);
      
      if (!result.valid) {
        return res.status(400).json({ 
          valid: false,
          error: result.error || "الكوبون غير صالح" 
        });
      }
      
      res.json({
        valid: true,
        coupon: result.coupon,
        discountAmount: result.discountAmount,
      });
    } catch (error) {
      console.error("Error validating coupon:", error);
      res.status(500).json({ error: "فشل في التحقق من الكوبون" });
    }
  });
  
  // GET /api/marketing/packages/:id/items - Get package items for order creation
  app.get("/api/marketing/packages/:id/items", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const items = await storage.getPackageItems(id);
      res.json(items);
    } catch (error) {
      console.error("Error fetching package items:", error);
      res.status(500).json({ error: "فشل في جلب عناصر الباقة" });
    }
  });
  
  app.post("/api/marketing/coupons", async (req: Request, res: Response) => {
    try {
      const request = req.body;
      
      // Validate required fields
      if (!request.nameAr) {
        return res.status(400).json({ error: "اسم الكوبون مطلوب" });
      }
      if (!request.code) {
        return res.status(400).json({ error: "رمز الكوبون مطلوب" });
      }
      if (!request.discountType || !["percent", "amount"].includes(request.discountType)) {
        return res.status(400).json({ error: "نوع الخصم غير صحيح" });
      }
      if (!request.discountValue || request.discountValue <= 0) {
        return res.status(400).json({ error: "قيمة الخصم مطلوبة" });
      }
      if (!request.startDate || !request.endDate) {
        return res.status(400).json({ error: "تاريخ البداية والنهاية مطلوبان" });
      }
      if (request.startDate > request.endDate) {
        return res.status(400).json({ error: "تاريخ البداية يجب أن يكون قبل تاريخ النهاية" });
      }
      
      const coupon = await storage.createCoupon(request);
      res.status(201).json(coupon);
    } catch (error: any) {
      console.error("Error creating coupon:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        if (translatedError === "القيمة مكررة") {
          return res.status(400).json({ error: "رمز الكوبون مستخدم من قبل" });
        }
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في إنشاء الكوبون" });
    }
  });
  
  app.put("/api/marketing/coupons/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const request = req.body;
      
      // Validate discount type if provided
      if (request.discountType && !["percent", "amount"].includes(request.discountType)) {
        return res.status(400).json({ error: "نوع الخصم غير صحيح" });
      }
      // Validate dates if both provided
      if (request.startDate && request.endDate && request.startDate > request.endDate) {
        return res.status(400).json({ error: "تاريخ البداية يجب أن يكون قبل تاريخ النهاية" });
      }
      
      const coupon = await storage.updateCoupon(id, request);
      if (!coupon) {
        return res.status(404).json({ error: "الكوبون غير موجود" });
      }
      res.json(coupon);
    } catch (error: any) {
      console.error("Error updating coupon:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        if (translatedError === "القيمة مكررة") {
          return res.status(400).json({ error: "رمز الكوبون مستخدم من قبل" });
        }
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في تحديث الكوبون" });
    }
  });
  
  app.delete("/api/marketing/coupons/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteCoupon(id);
      if (!success) {
        return res.status(404).json({ error: "الكوبون غير موجود" });
      }
      res.json({ message: "تم إلغاء تفعيل الكوبون بنجاح" });
    } catch (error) {
      console.error("Error deleting coupon:", error);
      res.status(500).json({ error: "فشل في حذف الكوبون" });
    }
  });
  
  // ========================================
  // CUSTOMER REVIEWS (تقييمات العملاء)
  // ========================================

  // Review Reasons
  app.get("/api/review-reasons", async (req: Request, res: Response) => {
    try {
      const reasonType = req.query.reasonType as string | undefined;
      const isActive = req.query.isActive === "true" ? true : req.query.isActive === "false" ? false : undefined;
      const reasons = await storage.getReviewReasons({ reasonType, isActive });
      res.json(reasons);
    } catch (error) {
      console.error("Error fetching review reasons:", error);
      res.status(500).json({ error: "فشل في جلب أسباب التقييم" });
    }
  });

  app.get("/api/review-reasons/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const reason = await storage.getReviewReason(id);
      if (!reason) {
        return res.status(404).json({ error: "سبب التقييم غير موجود" });
      }
      res.json(reason);
    } catch (error) {
      console.error("Error fetching review reason:", error);
      res.status(500).json({ error: "فشل في جلب سبب التقييم" });
    }
  });

  app.post("/api/review-reasons", async (req: Request, res: Response) => {
    try {
      const reason = await storage.createReviewReason(req.body);
      res.status(201).json(reason);
    } catch (error) {
      console.error("Error creating review reason:", error);
      res.status(500).json({ error: "فشل في إنشاء سبب التقييم" });
    }
  });

  app.put("/api/review-reasons/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const reason = await storage.updateReviewReason(id, req.body);
      if (!reason) {
        return res.status(404).json({ error: "سبب التقييم غير موجود" });
      }
      res.json(reason);
    } catch (error) {
      console.error("Error updating review reason:", error);
      res.status(500).json({ error: "فشل في تحديث سبب التقييم" });
    }
  });

  app.delete("/api/review-reasons/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteReviewReason(id);
      if (!success) {
        return res.status(404).json({ error: "سبب التقييم غير موجود" });
      }
      res.json({ message: "تم إلغاء تفعيل سبب التقييم" });
    } catch (error) {
      console.error("Error deleting review reason:", error);
      res.status(500).json({ error: "فشل في حذف سبب التقييم" });
    }
  });

  // Customer Reviews
  app.get("/api/customer-reviews", async (req: Request, res: Response) => {
    try {
      const customerId = req.query.customerId ? parseInt(req.query.customerId as string) : undefined;
      const orderId = req.query.orderId ? parseInt(req.query.orderId as string) : undefined;
      const reviewReasonId = req.query.reviewReasonId ? parseInt(req.query.reviewReasonId as string) : undefined;
      const status = req.query.status as string | undefined;
      const reviews = await storage.getCustomerReviews({ customerId, orderId, reviewReasonId, status });
      res.json(reviews);
    } catch (error) {
      console.error("Error fetching customer reviews:", error);
      res.status(500).json({ error: "فشل في جلب تقييمات العملاء" });
    }
  });

  app.get("/api/customer-reviews/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const review = await storage.getCustomerReview(id);
      if (!review) {
        return res.status(404).json({ error: "التقييم غير موجود" });
      }
      res.json(review);
    } catch (error) {
      console.error("Error fetching customer review:", error);
      res.status(500).json({ error: "فشل في جلب التقييم" });
    }
  });

  app.post("/api/customer-reviews", async (req: Request, res: Response) => {
    try {
      const review = await storage.createCustomerReview(req.body);
      res.status(201).json(review);
    } catch (error) {
      console.error("Error creating customer review:", error);
      res.status(500).json({ error: "فشل في إنشاء التقييم" });
    }
  });

  app.put("/api/customer-reviews/:id/status", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { status, reviewedByEmployeeId, responseNotes } = req.body;
      const review = await storage.updateCustomerReviewStatus(id, status, reviewedByEmployeeId, responseNotes);
      if (!review) {
        return res.status(404).json({ error: "التقييم غير موجود" });
      }
      res.json(review);
    } catch (error) {
      console.error("Error updating customer review status:", error);
      res.status(500).json({ error: "فشل في تحديث حالة التقييم" });
    }
  });

  app.delete("/api/customer-reviews/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteCustomerReview(id);
      if (!success) {
        return res.status(404).json({ error: "التقييم غير موجود" });
      }
      res.json({ message: "تم حذف التقييم بنجاح" });
    } catch (error) {
      console.error("Error deleting customer review:", error);
      res.status(500).json({ error: "فشل في حذف التقييم" });
    }
  });
  
  // ========================================
  // HR - JOB POSITIONS (الوظائف)
  // ========================================
  
  app.get("/api/hr/job-positions", async (req: Request, res: Response) => {
    try {
      const isActive = req.query.isActive === "true" ? true : req.query.isActive === "false" ? false : undefined;
      const positions = await storage.getJobPositions(isActive);
      res.json(positions);
    } catch (error) {
      console.error("Error fetching job positions:", error);
      res.status(500).json({ error: "فشل في جلب الوظائف" });
    }
  });
  
  app.get("/api/hr/job-positions/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const position = await storage.getJobPosition(id);
      if (!position) {
        return res.status(404).json({ error: "الوظيفة غير موجودة" });
      }
      res.json(position);
    } catch (error) {
      console.error("Error fetching job position:", error);
      res.status(500).json({ error: "فشل في جلب الوظيفة" });
    }
  });
  
  app.post("/api/hr/job-positions", async (req: Request, res: Response) => {
    try {
      const { nameAr, code, descriptionAr, isActive } = req.body;
      if (!nameAr) {
        return res.status(400).json({ error: "اسم الوظيفة بالعربية مطلوب" });
      }
      const position = await storage.createJobPosition({ nameAr, code, descriptionAr, isActive: isActive ?? true });
      res.status(201).json(position);
    } catch (error: any) {
      console.error("Error creating job position:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في إنشاء الوظيفة" });
    }
  });
  
  app.put("/api/hr/job-positions/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { nameAr, code, descriptionAr, isActive } = req.body;
      const position = await storage.updateJobPosition(id, { nameAr, code, descriptionAr, isActive });
      if (!position) {
        return res.status(404).json({ error: "الوظيفة غير موجودة" });
      }
      res.json(position);
    } catch (error: any) {
      console.error("Error updating job position:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في تحديث الوظيفة" });
    }
  });
  
  app.delete("/api/hr/job-positions/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteJobPosition(id);
      if (!success) {
        return res.status(404).json({ error: "الوظيفة غير موجودة" });
      }
      res.json({ message: "تم إلغاء تفعيل الوظيفة بنجاح" });
    } catch (error) {
      console.error("Error deleting job position:", error);
      res.status(500).json({ error: "فشل في حذف الوظيفة" });
    }
  });
  
  // ========================================
  // HR - NATIONALITIES (الجنسيات)
  // ========================================
  
  app.get("/api/hr/nationalities", async (req: Request, res: Response) => {
    try {
      const isActive = req.query.isActive === "true" ? true : req.query.isActive === "false" ? false : undefined;
      const nationalities = await storage.getNationalities(isActive);
      res.json(nationalities);
    } catch (error) {
      console.error("Error fetching nationalities:", error);
      res.status(500).json({ error: "فشل في جلب الجنسيات" });
    }
  });
  
  app.get("/api/hr/nationalities/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const nationality = await storage.getNationality(id);
      if (!nationality) {
        return res.status(404).json({ error: "الجنسية غير موجودة" });
      }
      res.json(nationality);
    } catch (error) {
      console.error("Error fetching nationality:", error);
      res.status(500).json({ error: "فشل في جلب الجنسية" });
    }
  });
  
  app.post("/api/hr/nationalities", async (req: Request, res: Response) => {
    try {
      const { nameAr, code, isActive } = req.body;
      if (!nameAr) {
        return res.status(400).json({ error: "اسم الجنسية بالعربية مطلوب" });
      }
      const nationality = await storage.createNationality({ nameAr, code, isActive: isActive ?? true });
      res.status(201).json(nationality);
    } catch (error: any) {
      console.error("Error creating nationality:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في إنشاء الجنسية" });
    }
  });
  
  app.put("/api/hr/nationalities/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { nameAr, code, isActive } = req.body;
      const nationality = await storage.updateNationality(id, { nameAr, code, isActive });
      if (!nationality) {
        return res.status(404).json({ error: "الجنسية غير موجودة" });
      }
      res.json(nationality);
    } catch (error: any) {
      console.error("Error updating nationality:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في تحديث الجنسية" });
    }
  });
  
  app.delete("/api/hr/nationalities/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteNationality(id);
      if (!success) {
        return res.status(404).json({ error: "الجنسية غير موجودة" });
      }
      res.json({ message: "تم إلغاء تفعيل الجنسية بنجاح" });
    } catch (error) {
      console.error("Error deleting nationality:", error);
      res.status(500).json({ error: "فشل في حذف الجنسية" });
    }
  });
  
  // ========================================
  // HR - EMPLOYEE SUPERVISORS (المشرفين)
  // ========================================
  
  app.get("/api/hr/supervisors", async (req: Request, res: Response) => {
    try {
      const supervisors = await storage.getSupervisorsWithSubordinates();
      res.json(supervisors);
    } catch (error) {
      console.error("Error fetching supervisors:", error);
      res.status(500).json({ error: "فشل في جلب المشرفين" });
    }
  });
  
  app.get("/api/hr/supervisors/:employeeId/subordinates", async (req: Request, res: Response) => {
    try {
      const employeeId = parseInt(req.params.employeeId);
      const subordinates = await storage.getSubordinates(employeeId);
      res.json(subordinates);
    } catch (error) {
      console.error("Error fetching subordinates:", error);
      res.status(500).json({ error: "فشل في جلب المرؤوسين" });
    }
  });
  
  app.get("/api/hr/employees/:employeeId/supervisor", async (req: Request, res: Response) => {
    try {
      const employeeId = parseInt(req.params.employeeId);
      const supervisor = await storage.getSupervisorFor(employeeId);
      res.json(supervisor || null);
    } catch (error) {
      console.error("Error fetching supervisor:", error);
      res.status(500).json({ error: "فشل في جلب المشرف" });
    }
  });
  
  app.post("/api/hr/supervisors", async (req: Request, res: Response) => {
    try {
      const { employeeId, supervisorEmployeeId } = req.body;
      if (!employeeId || !supervisorEmployeeId) {
        return res.status(400).json({ error: "الموظف والمشرف مطلوبان" });
      }
      if (employeeId === supervisorEmployeeId) {
        return res.status(400).json({ error: "لا يمكن أن يكون الموظف مشرفاً على نفسه" });
      }
      const relation = await storage.assignSupervisor({ employeeId, supervisorEmployeeId });
      res.status(201).json(relation);
    } catch (error: any) {
      console.error("Error assigning supervisor:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في تعيين المشرف" });
    }
  });
  
  app.delete("/api/hr/employees/:employeeId/supervisor", async (req: Request, res: Response) => {
    try {
      const employeeId = parseInt(req.params.employeeId);
      const success = await storage.removeSupervisor(employeeId);
      if (!success) {
        return res.status(404).json({ error: "لا يوجد مشرف لهذا الموظف" });
      }
      res.json({ message: "تم إزالة المشرف بنجاح" });
    } catch (error) {
      console.error("Error removing supervisor:", error);
      res.status(500).json({ error: "فشل في إزالة المشرف" });
    }
  });
  
  // ========================================
  // HR - WORK SHIFTS (الورديات)
  // ========================================
  
  app.get("/api/hr/work-shifts", async (req: Request, res: Response) => {
    try {
      const isActive = req.query.isActive === "true" ? true : req.query.isActive === "false" ? false : undefined;
      const shifts = await storage.getWorkShifts(isActive);
      res.json(shifts);
    } catch (error) {
      console.error("Error fetching work shifts:", error);
      res.status(500).json({ error: "فشل في جلب الورديات" });
    }
  });
  
  app.get("/api/hr/work-shifts/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const shift = await storage.getWorkShift(id);
      if (!shift) {
        return res.status(404).json({ error: "الوردية غير موجودة" });
      }
      res.json(shift);
    } catch (error) {
      console.error("Error fetching work shift:", error);
      res.status(500).json({ error: "فشل في جلب الوردية" });
    }
  });
  
  app.post("/api/hr/work-shifts", async (req: Request, res: Response) => {
    try {
      const { nameAr, shiftType, startTime, endTime, breakStartTime, breakEndTime, isActive } = req.body;
      if (!nameAr) {
        return res.status(400).json({ error: "اسم الوردية بالعربية مطلوب" });
      }
      if (!startTime || !endTime) {
        return res.status(400).json({ error: "وقت البداية والنهاية مطلوبان" });
      }
      if (!shiftType) {
        return res.status(400).json({ error: "نوع الوردية مطلوب" });
      }
      const shift = await storage.createWorkShift({ 
        nameAr, 
        shiftType,
        startTime, 
        endTime, 
        breakStartTime,
        breakEndTime,
        isActive: isActive ?? true 
      });
      res.status(201).json(shift);
    } catch (error: any) {
      console.error("Error creating work shift:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في إنشاء الوردية" });
    }
  });
  
  app.put("/api/hr/work-shifts/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { nameAr, shiftType, startTime, endTime, breakStartTime, breakEndTime, isActive } = req.body;
      const shift = await storage.updateWorkShift(id, { nameAr, shiftType, startTime, endTime, breakStartTime, breakEndTime, isActive });
      if (!shift) {
        return res.status(404).json({ error: "الوردية غير موجودة" });
      }
      res.json(shift);
    } catch (error: any) {
      console.error("Error updating work shift:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في تحديث الوردية" });
    }
  });
  
  app.delete("/api/hr/work-shifts/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteWorkShift(id);
      if (!success) {
        return res.status(404).json({ error: "الوردية غير موجودة" });
      }
      res.json({ message: "تم إلغاء تفعيل الوردية بنجاح" });
    } catch (error) {
      console.error("Error deleting work shift:", error);
      res.status(500).json({ error: "فشل في حذف الوردية" });
    }
  });
  
  // ========================================
  // HR - SHIFT ASSIGNMENTS (مناوبات الموظفين)
  // ========================================
  
  app.get("/api/hr/shift-assignments", async (req: Request, res: Response) => {
    try {
      const filters: { employeeId?: number; workShiftId?: number; date?: string } = {};
      if (req.query.employeeId) filters.employeeId = parseInt(req.query.employeeId as string);
      if (req.query.workShiftId) filters.workShiftId = parseInt(req.query.workShiftId as string);
      if (req.query.date) filters.date = req.query.date as string;
      
      const assignments = await storage.getShiftAssignments(Object.keys(filters).length > 0 ? filters : undefined);
      res.json(assignments);
    } catch (error) {
      console.error("Error fetching shift assignments:", error);
      res.status(500).json({ error: "فشل في جلب مناوبات الموظفين" });
    }
  });
  
  app.get("/api/hr/shift-assignments/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const assignment = await storage.getShiftAssignment(id);
      if (!assignment) {
        return res.status(404).json({ error: "المناوبة غير موجودة" });
      }
      res.json(assignment);
    } catch (error) {
      console.error("Error fetching shift assignment:", error);
      res.status(500).json({ error: "فشل في جلب المناوبة" });
    }
  });
  
  app.post("/api/hr/shift-assignments", async (req: Request, res: Response) => {
    try {
      const { employeeId, workShiftId, startDate, endDate } = req.body;
      if (!employeeId || !workShiftId || !startDate || !endDate) {
        return res.status(400).json({ error: "الموظف والوردية وتاريخ البداية والنهاية مطلوبان" });
      }
      if (startDate > endDate) {
        return res.status(400).json({ error: "تاريخ البداية يجب أن يكون قبل تاريخ النهاية" });
      }
      const assignment = await storage.createShiftAssignment({ employeeId, workShiftId, startDate, endDate });
      res.status(201).json(assignment);
    } catch (error: any) {
      console.error("Error creating shift assignment:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في إنشاء المناوبة" });
    }
  });
  
  app.put("/api/hr/shift-assignments/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { employeeId, workShiftId, startDate, endDate } = req.body;
      if (startDate && endDate && startDate > endDate) {
        return res.status(400).json({ error: "تاريخ البداية يجب أن يكون قبل تاريخ النهاية" });
      }
      const assignment = await storage.updateShiftAssignment(id, { employeeId, workShiftId, startDate, endDate });
      if (!assignment) {
        return res.status(404).json({ error: "المناوبة غير موجودة" });
      }
      res.json(assignment);
    } catch (error: any) {
      console.error("Error updating shift assignment:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في تحديث المناوبة" });
    }
  });
  
  app.delete("/api/hr/shift-assignments/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteShiftAssignment(id);
      if (!success) {
        return res.status(404).json({ error: "المناوبة غير موجودة" });
      }
      res.json({ message: "تم حذف المناوبة بنجاح" });
    } catch (error) {
      console.error("Error deleting shift assignment:", error);
      res.status(500).json({ error: "فشل في حذف المناوبة" });
    }
  });
  
  // ========================================
  // HR - ATTENDANCE RAW (البصمات الخام)
  // ========================================
  
  app.get("/api/hr/attendance-raw", async (req: Request, res: Response) => {
    try {
      const filters: { employeeId?: number; startDate?: string; endDate?: string } = {};
      if (req.query.employeeId) filters.employeeId = parseInt(req.query.employeeId as string);
      if (req.query.startDate) filters.startDate = req.query.startDate as string;
      if (req.query.endDate) filters.endDate = req.query.endDate as string;
      
      const records = await storage.getAttendanceRaw(Object.keys(filters).length > 0 ? filters : undefined);
      res.json(records);
    } catch (error) {
      console.error("Error fetching attendance raw:", error);
      res.status(500).json({ error: "فشل في جلب البصمات الخام" });
    }
  });
  
  app.post("/api/hr/attendance-raw", async (req: Request, res: Response) => {
    try {
      const { employeeId, timestamp, direction, deviceId } = req.body;
      if (!employeeId || !timestamp) {
        return res.status(400).json({ error: "الموظف والوقت مطلوبان" });
      }
      const record = await storage.createAttendanceRaw({ employeeId, timestamp, direction, deviceId });
      res.status(201).json(record);
    } catch (error: any) {
      console.error("Error creating attendance raw:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في إنشاء البصمة" });
    }
  });
  
  // ========================================
  // HR - ATTENDANCE LOGS (سجل الحضور)
  // ========================================
  
  app.get("/api/hr/attendance-logs", async (req: Request, res: Response) => {
    try {
      const filters: { employeeId?: number; startDate?: string; endDate?: string; status?: string } = {};
      if (req.query.employeeId) filters.employeeId = parseInt(req.query.employeeId as string);
      if (req.query.startDate) filters.startDate = req.query.startDate as string;
      if (req.query.endDate) filters.endDate = req.query.endDate as string;
      if (req.query.status) filters.status = req.query.status as string;
      
      const logs = await storage.getAttendanceLogs(Object.keys(filters).length > 0 ? filters : undefined);
      res.json(logs);
    } catch (error) {
      console.error("Error fetching attendance logs:", error);
      res.status(500).json({ error: "فشل في جلب سجل الحضور" });
    }
  });
  
  app.get("/api/hr/attendance-logs/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const log = await storage.getAttendanceLog(id);
      if (!log) {
        return res.status(404).json({ error: "سجل الحضور غير موجود" });
      }
      res.json(log);
    } catch (error) {
      console.error("Error fetching attendance log:", error);
      res.status(500).json({ error: "فشل في جلب سجل الحضور" });
    }
  });
  
  app.post("/api/hr/attendance-logs", async (req: Request, res: Response) => {
    try {
      const { employeeId, date, workShiftId, scheduledStartTime, scheduledEndTime, actualCheckIn, actualCheckOut, lateMinutes, overtimeMinutes, totalWorkMinutes, status, notes } = req.body;
      if (!employeeId || !date) {
        return res.status(400).json({ error: "الموظف والتاريخ مطلوبان" });
      }
      const log = await storage.createAttendanceLog({ 
        employeeId, 
        date, 
        workShiftId,
        scheduledStartTime,
        scheduledEndTime,
        actualCheckIn,
        actualCheckOut,
        lateMinutes: lateMinutes ?? 0,
        overtimeMinutes: overtimeMinutes ?? 0,
        totalWorkMinutes: totalWorkMinutes ?? 0,
        status: status ?? "present",
        notes 
      });
      res.status(201).json(log);
    } catch (error: any) {
      console.error("Error creating attendance log:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في إنشاء سجل الحضور" });
    }
  });
  
  app.put("/api/hr/attendance-logs/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { employeeId, date, workShiftId, scheduledStartTime, scheduledEndTime, actualCheckIn, actualCheckOut, lateMinutes, overtimeMinutes, totalWorkMinutes, status, notes } = req.body;
      const log = await storage.updateAttendanceLog(id, { employeeId, date, workShiftId, scheduledStartTime, scheduledEndTime, actualCheckIn, actualCheckOut, lateMinutes, overtimeMinutes, totalWorkMinutes, status, notes });
      if (!log) {
        return res.status(404).json({ error: "سجل الحضور غير موجود" });
      }
      res.json(log);
    } catch (error: any) {
      console.error("Error updating attendance log:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في تحديث سجل الحضور" });
    }
  });
  
  app.delete("/api/hr/attendance-logs/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteAttendanceLog(id);
      if (!success) {
        return res.status(404).json({ error: "سجل الحضور غير موجود" });
      }
      res.json({ message: "تم حذف سجل الحضور بنجاح" });
    } catch (error) {
      console.error("Error deleting attendance log:", error);
      res.status(500).json({ error: "فشل في حذف سجل الحضور" });
    }
  });
  
  // ========================================
  // HR - OVERTIME (الوقت الإضافي)
  // ========================================
  
  app.get("/api/hr/overtime", async (req: Request, res: Response) => {
    try {
      const filters: { employeeId?: number; startDate?: string; endDate?: string } = {};
      if (req.query.employeeId) filters.employeeId = parseInt(req.query.employeeId as string);
      if (req.query.startDate) filters.startDate = req.query.startDate as string;
      if (req.query.endDate) filters.endDate = req.query.endDate as string;
      
      const entries = await storage.getOvertimeEntries(Object.keys(filters).length > 0 ? filters : undefined);
      res.json(entries);
    } catch (error) {
      console.error("Error fetching overtime entries:", error);
      res.status(500).json({ error: "فشل في جلب سجلات الوقت الإضافي" });
    }
  });
  
  app.get("/api/hr/overtime/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const entry = await storage.getOvertimeEntry(id);
      if (!entry) {
        return res.status(404).json({ error: "سجل الوقت الإضافي غير موجود" });
      }
      res.json(entry);
    } catch (error) {
      console.error("Error fetching overtime entry:", error);
      res.status(500).json({ error: "فشل في جلب سجل الوقت الإضافي" });
    }
  });
  
  app.post("/api/hr/overtime", async (req: Request, res: Response) => {
    try {
      const { employeeId, date, startTime, endTime, totalMinutes, reason, approvedByEmployeeId } = req.body;
      if (!employeeId || !date || !startTime || !endTime || !totalMinutes) {
        return res.status(400).json({ error: "الموظف والتاريخ ووقت البداية والنهاية ومدة الوقت الإضافي مطلوبان" });
      }
      if (totalMinutes <= 0) {
        return res.status(400).json({ error: "مدة الوقت الإضافي يجب أن تكون أكبر من صفر" });
      }
      const entry = await storage.createOvertimeEntry({ 
        employeeId, 
        date, 
        startTime, 
        endTime,
        totalMinutes,
        reason,
        approvedByEmployeeId 
      });
      res.status(201).json(entry);
    } catch (error: any) {
      console.error("Error creating overtime entry:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في إنشاء سجل الوقت الإضافي" });
    }
  });
  
  app.put("/api/hr/overtime/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { employeeId, date, startTime, endTime, totalMinutes, reason, approvedByEmployeeId } = req.body;
      if (totalMinutes !== undefined && totalMinutes <= 0) {
        return res.status(400).json({ error: "مدة الوقت الإضافي يجب أن تكون أكبر من صفر" });
      }
      const entry = await storage.updateOvertimeEntry(id, { employeeId, date, startTime, endTime, totalMinutes, reason, approvedByEmployeeId });
      if (!entry) {
        return res.status(404).json({ error: "سجل الوقت الإضافي غير موجود" });
      }
      res.json(entry);
    } catch (error: any) {
      console.error("Error updating overtime entry:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في تحديث سجل الوقت الإضافي" });
    }
  });
  
  app.delete("/api/hr/overtime/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteOvertimeEntry(id);
      if (!success) {
        return res.status(404).json({ error: "سجل الوقت الإضافي غير موجود" });
      }
      res.json({ message: "تم حذف سجل الوقت الإضافي بنجاح" });
    } catch (error) {
      console.error("Error deleting overtime entry:", error);
      res.status(500).json({ error: "فشل في حذف سجل الوقت الإضافي" });
    }
  });
  
  // ========================================
  // HR - LEAVES (الإجازات)
  // ========================================
  
  app.get("/api/hr/leaves", async (req: Request, res: Response) => {
    try {
      const filters: { employeeId?: number; status?: string; leaveType?: string; startDate?: string; endDate?: string } = {};
      if (req.query.employeeId) filters.employeeId = parseInt(req.query.employeeId as string);
      if (req.query.status) filters.status = req.query.status as string;
      if (req.query.leaveType) filters.leaveType = req.query.leaveType as string;
      if (req.query.startDate) filters.startDate = req.query.startDate as string;
      if (req.query.endDate) filters.endDate = req.query.endDate as string;
      
      const leaves = await storage.getLeaves(Object.keys(filters).length > 0 ? filters : undefined);
      res.json(leaves);
    } catch (error) {
      console.error("Error fetching leaves:", error);
      res.status(500).json({ error: "فشل في جلب الإجازات" });
    }
  });
  
  app.get("/api/hr/leaves/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const leave = await storage.getLeave(id);
      if (!leave) {
        return res.status(404).json({ error: "الإجازة غير موجودة" });
      }
      res.json(leave);
    } catch (error) {
      console.error("Error fetching leave:", error);
      res.status(500).json({ error: "فشل في جلب الإجازة" });
    }
  });
  
  app.post("/api/hr/leaves", async (req: Request, res: Response) => {
    try {
      const { employeeId, leaveType, startDate, endDate, startTime, endTime, durationDays, durationHours, reason, requestedByEmployeeId, attachmentUrl, status } = req.body;
      if (!employeeId || !leaveType || !startDate || !endDate) {
        return res.status(400).json({ error: "الموظف ونوع الإجازة وتاريخ البداية والنهاية مطلوبان" });
      }
      if (startDate > endDate) {
        return res.status(400).json({ error: "تاريخ البداية يجب أن يكون قبل تاريخ النهاية" });
      }
      const leave = await storage.createLeave({ 
        employeeId, 
        leaveType, 
        startDate, 
        endDate,
        startTime,
        endTime,
        durationDays: durationDays ?? 0,
        durationHours: durationHours ?? 0,
        reason,
        requestedByEmployeeId,
        attachmentUrl,
        status: status ?? "pending"
      });
      res.status(201).json(leave);
    } catch (error: any) {
      console.error("Error creating leave:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في إنشاء الإجازة" });
    }
  });
  
  app.put("/api/hr/leaves/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { employeeId, leaveType, startDate, endDate, startTime, endTime, durationDays, durationHours, reason, requestedByEmployeeId, attachmentUrl, status } = req.body;
      if (startDate && endDate && startDate > endDate) {
        return res.status(400).json({ error: "تاريخ البداية يجب أن يكون قبل تاريخ النهاية" });
      }
      const leave = await storage.updateLeave(id, { employeeId, leaveType, startDate, endDate, startTime, endTime, durationDays, durationHours, reason, requestedByEmployeeId, attachmentUrl, status });
      if (!leave) {
        return res.status(404).json({ error: "الإجازة غير موجودة" });
      }
      res.json(leave);
    } catch (error: any) {
      console.error("Error updating leave:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في تحديث الإجازة" });
    }
  });
  
  app.delete("/api/hr/leaves/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteLeave(id);
      if (!success) {
        return res.status(404).json({ error: "الإجازة غير موجودة" });
      }
      res.json({ message: "تم حذف الإجازة بنجاح" });
    } catch (error) {
      console.error("Error deleting leave:", error);
      res.status(500).json({ error: "فشل في حذف الإجازة" });
    }
  });
  
  app.post("/api/hr/leaves/:id/approve", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const leave = await storage.approveLeave(id);
      if (!leave) {
        return res.status(404).json({ error: "الإجازة غير موجودة" });
      }
      res.json(leave);
    } catch (error) {
      console.error("Error approving leave:", error);
      res.status(500).json({ error: "فشل في الموافقة على الإجازة" });
    }
  });
  
  app.post("/api/hr/leaves/:id/reject", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const leave = await storage.rejectLeave(id);
      if (!leave) {
        return res.status(404).json({ error: "الإجازة غير موجودة" });
      }
      res.json(leave);
    } catch (error) {
      console.error("Error rejecting leave:", error);
      res.status(500).json({ error: "فشل في رفض الإجازة" });
    }
  });
  
  // ========================================
  // EXPENSE TYPES (أنواع النفقات)
  // ========================================
  
  app.get("/api/expenses/types", async (req: Request, res: Response) => {
    try {
      const isActive = req.query.isActive === "true" ? true : req.query.isActive === "false" ? false : undefined;
      const types = await storage.getExpenseTypes(isActive);
      res.json(types);
    } catch (error) {
      console.error("Error fetching expense types:", error);
      res.status(500).json({ error: "فشل في جلب أنواع النفقات" });
    }
  });
  
  app.get("/api/expenses/types/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const expenseType = await storage.getExpenseType(id);
      if (!expenseType) {
        return res.status(404).json({ error: "نوع النفقة غير موجود" });
      }
      res.json(expenseType);
    } catch (error) {
      console.error("Error fetching expense type:", error);
      res.status(500).json({ error: "فشل في جلب نوع النفقة" });
    }
  });
  
  app.post("/api/expenses/types", async (req: Request, res: Response) => {
    try {
      const { nameAr, descriptionAr, isFixedAmount, fixedAmount, isActive } = req.body;
      
      if (!nameAr || !nameAr.trim()) {
        return res.status(400).json({ error: "اسم النفقة مطلوب" });
      }
      
      if (isFixedAmount && (!fixedAmount || fixedAmount <= 0)) {
        return res.status(400).json({ error: "المبلغ الثابت يجب أن يكون أكبر من صفر" });
      }
      
      const expenseType = await storage.createExpenseType({
        nameAr: nameAr.trim(),
        descriptionAr: descriptionAr?.trim() || null,
        isFixedAmount: isFixedAmount ?? false,
        fixedAmount: isFixedAmount ? fixedAmount : null,
        isActive: isActive ?? true,
      });
      res.status(201).json(expenseType);
    } catch (error: any) {
      console.error("Error creating expense type:", error);
      if (error?.message?.includes("unique") || error?.message?.includes("duplicate")) {
        return res.status(400).json({ error: "اسم النفقة موجود مسبقاً" });
      }
      res.status(500).json({ error: "فشل في إنشاء نوع النفقة" });
    }
  });
  
  app.put("/api/expenses/types/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { nameAr, descriptionAr, isFixedAmount, fixedAmount, isActive } = req.body;
      
      if (nameAr !== undefined && (!nameAr || !nameAr.trim())) {
        return res.status(400).json({ error: "اسم النفقة مطلوب" });
      }
      
      if (isFixedAmount && (!fixedAmount || fixedAmount <= 0)) {
        return res.status(400).json({ error: "المبلغ الثابت يجب أن يكون أكبر من صفر" });
      }
      
      const expenseType = await storage.updateExpenseType(id, {
        nameAr: nameAr?.trim(),
        descriptionAr: descriptionAr?.trim(),
        isFixedAmount,
        fixedAmount: isFixedAmount ? fixedAmount : null,
        isActive,
      });
      
      if (!expenseType) {
        return res.status(404).json({ error: "نوع النفقة غير موجود" });
      }
      res.json(expenseType);
    } catch (error: any) {
      console.error("Error updating expense type:", error);
      if (error?.message?.includes("unique") || error?.message?.includes("duplicate")) {
        return res.status(400).json({ error: "اسم النفقة موجود مسبقاً" });
      }
      res.status(500).json({ error: "فشل في تحديث نوع النفقة" });
    }
  });
  
  app.delete("/api/expenses/types/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteExpenseType(id);
      if (!success) {
        return res.status(404).json({ error: "نوع النفقة غير موجود" });
      }
      res.json({ message: "تم حذف نوع النفقة بنجاح" });
    } catch (error) {
      console.error("Error deleting expense type:", error);
      res.status(500).json({ error: "فشل في حذف نوع النفقة" });
    }
  });
  
  // ========================================
  // EXPENSES (النفقات)
  // ========================================
  
  app.get("/api/expenses", async (req: Request, res: Response) => {
    try {
      const filters = {
        fromDate: req.query.from_date as string | undefined,
        toDate: req.query.to_date as string | undefined,
        expenseTypeId: req.query.expense_type_id ? parseInt(req.query.expense_type_id as string) : undefined,
        supplierId: req.query.supplier_id ? parseInt(req.query.supplier_id as string) : undefined,
        paymentMethodId: req.query.payment_method_id ? parseInt(req.query.payment_method_id as string) : undefined,
      };
      const expenses = await storage.getExpenses(filters);
      res.json(expenses);
    } catch (error) {
      console.error("Error fetching expenses:", error);
      res.status(500).json({ error: "فشل في جلب النفقات" });
    }
  });
  
  app.get("/api/expenses/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const expense = await storage.getExpense(id);
      if (!expense) {
        return res.status(404).json({ error: "النفقة غير موجودة" });
      }
      res.json(expense);
    } catch (error) {
      console.error("Error fetching expense:", error);
      res.status(500).json({ error: "فشل في جلب النفقة" });
    }
  });
  
  app.post("/api/expenses", async (req: Request, res: Response) => {
    try {
      const {
        expense_type_id,
        supplier_id,
        invoice_reference,
        description_ar,
        amount,
        vat_type,
        vat_rate,
        payment_method_id,
        payment_date,
        attachment_url,
        created_by_employee_id,
      } = req.body;
      
      if (!expense_type_id) {
        return res.status(400).json({ error: "نوع النفقة مطلوب" });
      }
      
      if (!payment_date) {
        return res.status(400).json({ error: "تاريخ الدفع مطلوب" });
      }
      
      // Get expense type to check if fixed amount
      const expenseType = await storage.getExpenseType(expense_type_id);
      if (!expenseType) {
        return res.status(400).json({ error: "نوع النفقة غير موجود" });
      }
      
      // Determine the amount to use
      let finalAmount = amount;
      if (expenseType.isFixedAmount && expenseType.fixedAmount) {
        finalAmount = expenseType.fixedAmount;
      }
      
      if (!finalAmount || finalAmount <= 0) {
        return res.status(400).json({ error: "المبلغ مطلوب ويجب أن يكون أكبر من صفر" });
      }
      
      // Calculate VAT based on vat_type
      const vatType = vat_type || "inclusive";
      const vatRate = vat_rate ?? 0.15;
      
      let amountWithoutVat: number;
      let vatAmount: number;
      let totalAmount: number;
      
      if (vatType === "inclusive") {
        totalAmount = finalAmount;
        amountWithoutVat = totalAmount / (1 + vatRate);
        vatAmount = totalAmount - amountWithoutVat;
      } else if (vatType === "exclusive") {
        amountWithoutVat = finalAmount;
        vatAmount = amountWithoutVat * vatRate;
        totalAmount = amountWithoutVat + vatAmount;
      } else { // exempt
        amountWithoutVat = finalAmount;
        vatAmount = 0;
        totalAmount = amountWithoutVat;
      }
      
      const expense = await storage.createExpense({
        expenseTypeId: expense_type_id,
        supplierId: supplier_id || null,
        invoiceReference: invoice_reference || null,
        descriptionAr: description_ar || null,
        amountWithoutVat,
        vatType,
        vatRate,
        vatAmount,
        totalAmount,
        paymentMethodId: payment_method_id || null,
        paymentDate: payment_date,
        isPaid: true,
        attachmentUrl: attachment_url || null,
        createdByEmployeeId: created_by_employee_id || null,
      });
      
      res.status(201).json(expense);
    } catch (error) {
      console.error("Error creating expense:", error);
      res.status(500).json({ error: "فشل في إنشاء النفقة" });
    }
  });
  
  app.put("/api/expenses/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const {
        expense_type_id,
        supplier_id,
        invoice_reference,
        description_ar,
        amount,
        vat_type,
        vat_rate,
        payment_method_id,
        payment_date,
        attachment_url,
      } = req.body;
      
      // Get existing expense
      const existing = await storage.getExpense(id);
      if (!existing) {
        return res.status(404).json({ error: "النفقة غير موجودة" });
      }
      
      // Calculate VAT if amount or vat_type changed
      let updateData: any = {};
      
      if (expense_type_id !== undefined) updateData.expenseTypeId = expense_type_id;
      if (supplier_id !== undefined) updateData.supplierId = supplier_id;
      if (invoice_reference !== undefined) updateData.invoiceReference = invoice_reference;
      if (description_ar !== undefined) updateData.descriptionAr = description_ar;
      if (payment_method_id !== undefined) updateData.paymentMethodId = payment_method_id;
      if (payment_date !== undefined) updateData.paymentDate = payment_date;
      if (attachment_url !== undefined) updateData.attachmentUrl = attachment_url;
      
      // Recalculate VAT if amount or vat_type changed
      if (amount !== undefined || vat_type !== undefined || vat_rate !== undefined) {
        const finalAmount = amount ?? existing.totalAmount;
        const vatType = vat_type ?? existing.vatType;
        const finalVatRate = vat_rate ?? existing.vatRate;
        
        let amountWithoutVat: number;
        let vatAmount: number;
        let totalAmount: number;
        
        if (vatType === "inclusive") {
          totalAmount = finalAmount;
          amountWithoutVat = totalAmount / (1 + finalVatRate);
          vatAmount = totalAmount - amountWithoutVat;
        } else if (vatType === "exclusive") {
          amountWithoutVat = finalAmount;
          vatAmount = amountWithoutVat * finalVatRate;
          totalAmount = amountWithoutVat + vatAmount;
        } else { // exempt
          amountWithoutVat = finalAmount;
          vatAmount = 0;
          totalAmount = amountWithoutVat;
        }
        
        updateData.amountWithoutVat = amountWithoutVat;
        updateData.vatType = vatType;
        updateData.vatRate = finalVatRate;
        updateData.vatAmount = vatAmount;
        updateData.totalAmount = totalAmount;
      }
      
      const expense = await storage.updateExpense(id, updateData);
      res.json(expense);
    } catch (error) {
      console.error("Error updating expense:", error);
      res.status(500).json({ error: "فشل في تحديث النفقة" });
    }
  });
  
  app.delete("/api/expenses/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteExpense(id);
      if (!success) {
        return res.status(404).json({ error: "النفقة غير موجودة" });
      }
      res.json({ message: "تم حذف النفقة بنجاح" });
    } catch (error) {
      console.error("Error deleting expense:", error);
      res.status(500).json({ error: "فشل في حذف النفقة" });
    }
  });
  
  // ========================================
  // EXPENSE REPORTS (تقارير النفقات)
  // ========================================
  
  app.get("/api/reports/expenses-summary", async (req: Request, res: Response) => {
    try {
      const filters = {
        fromDate: req.query.from_date as string | undefined,
        toDate: req.query.to_date as string | undefined,
        expenseTypeId: req.query.expense_type_id ? parseInt(req.query.expense_type_id as string) : undefined,
        supplierId: req.query.supplier_id ? parseInt(req.query.supplier_id as string) : undefined,
      };
      const summary = await storage.getExpensesSummary(filters);
      res.json(summary);
    } catch (error) {
      console.error("Error fetching expenses summary:", error);
      res.status(500).json({ error: "فشل في جلب تقرير النفقات" });
    }
  });
  
  // ========================================
  // PAYABLES / DEBTS (الديون)
  // ========================================
  
  app.get("/api/reports/payables-summary", async (req: Request, res: Response) => {
    try {
      const summary = await storage.getPayablesSummary();
      res.json(summary);
    } catch (error) {
      console.error("Error fetching payables summary:", error);
      res.status(500).json({ error: "فشل في جلب تقرير الديون" });
    }
  });
  
  // ========================================
  // SUPPLIER STATEMENT (كشف حساب المورد)
  // ========================================
  
  app.get("/api/suppliers/:id/statement", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const fromDate = req.query.from_date as string | undefined;
      const toDate = req.query.to_date as string | undefined;
      
      const statement = await storage.getSupplierStatement(id, fromDate, toDate);
      res.json(statement);
    } catch (error: any) {
      console.error("Error fetching supplier statement:", error);
      if (error?.message === "المورد غير موجود") {
        return res.status(404).json({ error: error.message });
      }
      res.status(500).json({ error: "فشل في جلب كشف حساب المورد" });
    }
  });
  
  // ========================================
  // DRIVERS (السائقون)
  // ========================================
  
  app.get("/api/drivers", async (req: Request, res: Response) => {
    try {
      const drivers = await storage.getDrivers();
      res.json(drivers);
    } catch (error) {
      console.error("Error fetching drivers:", error);
      res.status(500).json({ error: "فشل في جلب السائقين" });
    }
  });
  
  app.get("/api/drivers/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const driver = await storage.getDriver(id);
      if (!driver) {
        return res.status(404).json({ error: "السائق غير موجود" });
      }
      res.json(driver);
    } catch (error) {
      console.error("Error fetching driver:", error);
      res.status(500).json({ error: "فشل في جلب السائق" });
    }
  });
  
  app.post("/api/drivers", async (req: Request, res: Response) => {
    try {
      const validated = insertDriverSchema.parse(req.body);
      const driver = await storage.createDriver(validated);
      res.status(201).json(driver);
    } catch (error) {
      console.error("Error creating driver:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "خطأ في البيانات", details: error.errors });
      }
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في إنشاء السائق" });
    }
  });
  
  app.put("/api/drivers/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const driver = await storage.updateDriver(id, req.body);
      if (!driver) {
        return res.status(404).json({ error: "السائق غير موجود" });
      }
      res.json(driver);
    } catch (error) {
      console.error("Error updating driver:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في تحديث السائق" });
    }
  });
  
  app.delete("/api/drivers/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteDriver(id);
      if (!success) {
        return res.status(404).json({ error: "السائق غير موجود" });
      }
      res.json({ message: "تم حذف السائق بنجاح" });
    } catch (error) {
      console.error("Error deleting driver:", error);
      res.status(500).json({ error: "فشل في حذف السائق" });
    }
  });
  
  // ========================================
  // TRIPS (الرحلات)
  // ========================================
  
  app.get("/api/trips", async (req: Request, res: Response) => {
    try {
      const filters = {
        driverId: req.query.driver_id ? parseInt(req.query.driver_id as string) : undefined,
        tripType: req.query.trip_type as string | undefined,
        status: req.query.status as string | undefined,
      };
      const trips = await storage.getTrips(filters);
      res.json(trips);
    } catch (error) {
      console.error("Error fetching trips:", error);
      res.status(500).json({ error: "فشل في جلب الرحلات" });
    }
  });
  
  app.get("/api/trips/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const trip = await storage.getTrip(id);
      if (!trip) {
        return res.status(404).json({ error: "الرحلة غير موجودة" });
      }
      res.json(trip);
    } catch (error) {
      console.error("Error fetching trip:", error);
      res.status(500).json({ error: "فشل في جلب الرحلة" });
    }
  });
  
  app.post("/api/trips", async (req: Request, res: Response) => {
    try {
      // Generate trip number
      const tripNumber = await storage.generateTripNumber();
      
      // Preprocess date/timestamp fields - convert strings to Date or null
      const body = req.body;
      const tripData = { 
        ...body, 
        tripNumber,
        scheduledAt: body.scheduledAt ? new Date(body.scheduledAt) : null,
        startedAt: body.startedAt ? new Date(body.startedAt) : null,
        completedAt: body.completedAt ? new Date(body.completedAt) : null,
      };
      
      const validated = insertTripSchema.parse(tripData);
      const trip = await storage.createTrip(validated);
      res.status(201).json(trip);
    } catch (error) {
      console.error("Error creating trip:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "خطأ في البيانات", details: error.errors });
      }
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في إنشاء الرحلة" });
    }
  });
  
  app.put("/api/trips/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      
      // Preprocess date/timestamp fields - convert strings to Date or null
      const body = req.body;
      const tripData: Record<string, any> = { ...body };
      
      if ('scheduledAt' in body) {
        tripData.scheduledAt = body.scheduledAt ? new Date(body.scheduledAt) : null;
      }
      if ('startedAt' in body) {
        tripData.startedAt = body.startedAt ? new Date(body.startedAt) : null;
      }
      if ('completedAt' in body) {
        tripData.completedAt = body.completedAt ? new Date(body.completedAt) : null;
      }
      
      const trip = await storage.updateTrip(id, tripData);
      if (!trip) {
        return res.status(404).json({ error: "الرحلة غير موجودة" });
      }
      res.json(trip);
    } catch (error) {
      console.error("Error updating trip:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في تحديث الرحلة" });
    }
  });
  
  app.delete("/api/trips/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteTrip(id);
      if (!success) {
        return res.status(404).json({ error: "الرحلة غير موجودة" });
      }
      res.json({ message: "تم حذف الرحلة بنجاح" });
    } catch (error) {
      console.error("Error deleting trip:", error);
      res.status(500).json({ error: "فشل في حذف الرحلة" });
    }
  });
  
  // ========================================
  // CASH DISBURSEMENTS (صرف نقدي)
  // ========================================
  
  app.get("/api/cash-disbursements", async (req: Request, res: Response) => {
    try {
      const filters = {
        disbursementType: req.query.type as string | undefined,
        employeeId: req.query.employee_id ? parseInt(req.query.employee_id as string) : undefined,
        driverId: req.query.driver_id ? parseInt(req.query.driver_id as string) : undefined,
        fromDate: req.query.from_date as string | undefined,
        toDate: req.query.to_date as string | undefined,
      };
      const disbursements = await storage.getCashDisbursements(filters);
      res.json(disbursements);
    } catch (error) {
      console.error("Error fetching cash disbursements:", error);
      res.status(500).json({ error: "فشل في جلب سجلات الصرف النقدي" });
    }
  });
  
  app.get("/api/cash-disbursements/generate-number", async (req: Request, res: Response) => {
    try {
      const number = await storage.generateDisbursementNumber();
      res.json({ number });
    } catch (error) {
      console.error("Error generating disbursement number:", error);
      res.status(500).json({ error: "فشل في إنشاء رقم الصرف" });
    }
  });
  
  app.get("/api/cash-disbursements/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const disbursement = await storage.getCashDisbursement(id);
      if (!disbursement) {
        return res.status(404).json({ error: "سجل الصرف غير موجود" });
      }
      res.json(disbursement);
    } catch (error) {
      console.error("Error fetching cash disbursement:", error);
      res.status(500).json({ error: "فشل في جلب سجل الصرف" });
    }
  });
  
  app.post("/api/cash-disbursements", async (req: Request, res: Response) => {
    try {
      const validated = insertCashDisbursementSchema.parse(req.body);
      const disbursement = await storage.createCashDisbursement(validated);
      res.status(201).json(disbursement);
    } catch (error) {
      console.error("Error creating cash disbursement:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ error: "خطأ في البيانات", details: error.errors });
      }
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في إنشاء سجل الصرف" });
    }
  });
  
  app.put("/api/cash-disbursements/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const disbursement = await storage.updateCashDisbursement(id, req.body);
      if (!disbursement) {
        return res.status(404).json({ error: "سجل الصرف غير موجود" });
      }
      res.json(disbursement);
    } catch (error) {
      console.error("Error updating cash disbursement:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في تحديث سجل الصرف" });
    }
  });
  
  app.delete("/api/cash-disbursements/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteCashDisbursement(id);
      if (!success) {
        return res.status(404).json({ error: "سجل الصرف غير موجود" });
      }
      res.json({ message: "تم حذف سجل الصرف بنجاح" });
    } catch (error) {
      console.error("Error deleting cash disbursement:", error);
      res.status(500).json({ error: "فشل في حذف سجل الصرف" });
    }
  });
  
  // Get disbursements for a specific employee (for employee detail page)
  app.get("/api/employees/:id/disbursements", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const disbursements = await storage.getEmployeeDisbursements(id);
      res.json(disbursements);
    } catch (error) {
      console.error("Error fetching employee disbursements:", error);
      res.status(500).json({ error: "فشل في جلب سجلات الصرف للموظف" });
    }
  });
  
  // Get disbursements for a specific driver
  app.get("/api/drivers/:id/disbursements", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const disbursements = await storage.getDriverDisbursements(id);
      res.json(disbursements);
    } catch (error) {
      console.error("Error fetching driver disbursements:", error);
      res.status(500).json({ error: "فشل في جلب سجلات الصرف للسائق" });
    }
  });
  
  // Installment management
  app.get("/api/cash-disbursements/:id/installments", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const installments = await storage.getAdvanceInstallments(id);
      res.json(installments);
    } catch (error) {
      console.error("Error fetching installments:", error);
      res.status(500).json({ error: "فشل في جلب الأقساط" });
    }
  });
  
  app.put("/api/installments/:id/status", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { status, payrollLineId } = req.body;
      const installment = await storage.updateInstallmentStatus(id, status, payrollLineId);
      if (!installment) {
        return res.status(404).json({ error: "القسط غير موجود" });
      }
      res.json(installment);
    } catch (error) {
      console.error("Error updating installment status:", error);
      res.status(500).json({ error: "فشل في تحديث حالة القسط" });
    }
  });
  
  // ========================================
  // COMMISSION MODULE (نظام العمولات)
  // ========================================
  
  // Get all commission profiles
  app.get("/api/commissions/profiles", async (req: Request, res: Response) => {
    try {
      const profiles = await storage.getCommissionProfiles();
      res.json(profiles);
    } catch (error) {
      console.error("Error fetching commission profiles:", error);
      res.status(500).json({ error: "فشل في جلب ملفات العمولات" });
    }
  });
  
  // Get commission profile by ID
  app.get("/api/commissions/profiles/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const profile = await storage.getCommissionProfile(id);
      if (!profile) {
        return res.status(404).json({ error: "ملف العمولات غير موجود" });
      }
      res.json(profile);
    } catch (error) {
      console.error("Error fetching commission profile:", error);
      res.status(500).json({ error: "فشل في جلب ملف العمولات" });
    }
  });
  
  // Get commission profile by employee ID
  app.get("/api/employees/:id/commission-profile", async (req: Request, res: Response) => {
    try {
      const employeeId = parseInt(req.params.id);
      const profile = await storage.getCommissionProfileByEmployee(employeeId);
      res.json(profile || null);
    } catch (error) {
      console.error("Error fetching employee commission profile:", error);
      res.status(500).json({ error: "فشل في جلب ملف عمولات الموظف" });
    }
  });
  
  // Create commission profile
  app.post("/api/commissions/profiles", async (req: Request, res: Response) => {
    try {
      const { targetSources, ...profileData } = req.body;
      const profile = await storage.createCommissionProfile(profileData, targetSources);
      res.status(201).json(profile);
    } catch (error: any) {
      console.error("Error creating commission profile:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في إنشاء ملف العمولات" });
    }
  });
  
  // Update commission profile
  app.put("/api/commissions/profiles/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const { targetSources, ...profileData } = req.body;
      const profile = await storage.updateCommissionProfile(id, profileData, targetSources);
      if (!profile) {
        return res.status(404).json({ error: "ملف العمولات غير موجود" });
      }
      res.json(profile);
    } catch (error: any) {
      console.error("Error updating commission profile:", error);
      const translatedError = translateDbError(error);
      if (translatedError) {
        return res.status(400).json({ error: translatedError });
      }
      res.status(500).json({ error: "فشل في تحديث ملف العمولات" });
    }
  });
  
  // Delete commission profile
  app.delete("/api/commissions/profiles/:id", async (req: Request, res: Response) => {
    try {
      const id = parseInt(req.params.id);
      const success = await storage.deleteCommissionProfile(id);
      if (!success) {
        return res.status(404).json({ error: "ملف العمولات غير موجود" });
      }
      res.json({ message: "تم حذف ملف العمولات بنجاح" });
    } catch (error) {
      console.error("Error deleting commission profile:", error);
      res.status(500).json({ error: "فشل في حذف ملف العمولات" });
    }
  });
  
  // Get employee commission report
  app.get("/api/employees/:id/commission-report", async (req: Request, res: Response) => {
    try {
      const employeeId = parseInt(req.params.id);
      const month = parseInt(req.query.month as string) || new Date().getMonth() + 1;
      const year = parseInt(req.query.year as string) || new Date().getFullYear();
      
      const report = await storage.getEmployeeCommissionReport(employeeId, month, year);
      res.json(report);
    } catch (error: any) {
      console.error("Error fetching employee commission report:", error);
      res.status(500).json({ error: error.message || "فشل في جلب تقرير العمولات" });
    }
  });
  
  // Get commission transactions for a profile
  app.get("/api/commissions/profiles/:id/transactions", async (req: Request, res: Response) => {
    try {
      const profileId = parseInt(req.params.id);
      const month = req.query.month ? parseInt(req.query.month as string) : undefined;
      const year = req.query.year ? parseInt(req.query.year as string) : undefined;
      
      const transactions = await storage.getCommissionTransactions(profileId, month, year);
      res.json(transactions);
    } catch (error) {
      console.error("Error fetching commission transactions:", error);
      res.status(500).json({ error: "فشل في جلب معاملات العمولات" });
    }
  });
  
  // Settle commission (create snapshot and accrual transaction)
  app.post("/api/commissions/profiles/:id/settle", async (req: Request, res: Response) => {
    try {
      const profileId = parseInt(req.params.id);
      const { periodMonth, periodYear, processedByEmployeeId } = req.body;
      
      if (!periodMonth || !periodYear) {
        return res.status(400).json({ error: "الشهر والسنة مطلوبان" });
      }
      
      const snapshot = await storage.settleCommission(profileId, periodMonth, periodYear, processedByEmployeeId);
      res.status(201).json(snapshot);
    } catch (error: any) {
      console.error("Error settling commission:", error);
      res.status(500).json({ error: error.message || "فشل في تسوية العمولة" });
    }
  });
  
  // Pay commission
  app.post("/api/commissions/snapshots/:id/pay", async (req: Request, res: Response) => {
    try {
      const snapshotId = parseInt(req.params.id);
      const { paidByEmployeeId } = req.body;
      
      if (!paidByEmployeeId) {
        return res.status(400).json({ error: "معرف الموظف المعالج مطلوب" });
      }
      
      const snapshot = await storage.payCommission(snapshotId, paidByEmployeeId);
      res.json(snapshot);
    } catch (error: any) {
      console.error("Error paying commission:", error);
      res.status(500).json({ error: error.message || "فشل في سداد العمولة" });
    }
  });
  
  // Get commission snapshots
  app.get("/api/commissions/snapshots", async (req: Request, res: Response) => {
    try {
      const profileId = req.query.profileId ? parseInt(req.query.profileId as string) : undefined;
      const snapshots = await storage.getCommissionSnapshots(profileId);
      res.json(snapshots);
    } catch (error) {
      console.error("Error fetching commission snapshots:", error);
      res.status(500).json({ error: "فشل في جلب سجلات العمولات" });
    }
  });
  
  return httpServer;
}
