import { eq, sql, and, desc, asc, gte, lte, ne, isNotNull, isNull, sum, between, or, like } from "drizzle-orm";
import { db } from "./db";
import {
  suppliers,
  supplierBankAccounts,
  units,
  products,
  stockBatches,
  stockMovements,
  paymentMethods,
  purchaseInvoices,
  purchaseInvoiceLines,
  purchasePayments,
  journalEntries,
  journalLines,
  accounts,
  serviceCategories,
  services,
  serviceProducts,
  serviceImages,
  customers,
  loyaltySettings,
  customerLoyaltyPoints,
  employees,
  orders,
  orderServices,
  orderProducts,
  orderPackages,
  orderRatings,
  orderReturns,
  orderReturnLines,
  orderPayments,
  roles,
  employeeTasks,
  salaryProfiles,
  payrollRuns,
  payrollLines,
  cafeteriaProducts,
  cafeteriaStockBatches,
  cafeteriaStockMovements,
  cafeteriaPurchaseInvoices,
  cafeteriaPurchaseInvoiceLines,
  cafeteriaPurchasePayments,
  cafeteriaSalesOrders,
  cafeteriaSalesOrderLines,
  cafeteriaSalesReturns,
  cafeteriaSalesReturnLines,
  cafeteriaEmployeeCustodies,
  appointments,
  appointmentServices,
  systemSettings,
  packages,
  packageServices,
  packageProducts,
  coupons,
  couponServices,
  couponProducts,
  drivers,
  trips,
  ACCOUNT_CODES,
  ACCOUNT_NAMES,
  SALES_ACCOUNT_CODES,
  type InsertSupplier,
  type Supplier,
  type SupplierWithBankAccounts,
  type InsertSupplierBankAccount,
  type SupplierBankAccount,
  type InsertUnit,
  type Unit,
  type InsertProduct,
  type Product,
  type ProductWithUnits,
  type InsertStockBatch,
  type StockBatch,
  type InsertStockMovement,
  type StockMovement,
  type InsertPaymentMethod,
  type PaymentMethod,
  type InsertPurchaseInvoice,
  type PurchaseInvoice,
  type InsertPurchaseInvoiceLine,
  type PurchaseInvoiceLine,
  type PurchaseInvoiceWithDetails,
  type InsertPurchasePayment,
  type PurchasePayment,
  purchaseDebitNotes,
  purchaseDebitNoteLines,
  type InsertPurchaseDebitNote,
  type PurchaseDebitNote,
  type InsertPurchaseDebitNoteLine,
  type PurchaseDebitNoteLine,
  type PurchaseDebitNoteLineWithProduct,
  type PurchaseDebitNoteWithDetails,
  inventoryWithdrawalOrders,
  inventoryWithdrawalLines,
  type InsertInventoryWithdrawalOrder,
  type InventoryWithdrawalOrder,
  type InsertInventoryWithdrawalLine,
  type InventoryWithdrawalLine,
  type InventoryWithdrawalLineWithDetails,
  type InventoryWithdrawalOrderWithDetails,
  type InsertJournalEntry,
  type JournalEntry,
  type InsertJournalLine,
  type JournalLine,
  type JournalEntryWithLines,
  type Account,
  type AccountWithChildren,
  type InsertAccount,
  type ServiceCategory,
  type ServiceCategoryWithCount,
  type ServiceCategoryWithServices,
  type InsertServiceCategory,
  type Service,
  type ServiceWithDetails,
  type InsertService,
  type ServiceProduct,
  type ServiceProductWithProduct,
  type InsertServiceProduct,
  type ServiceImage,
  type InsertServiceImage,
  type Customer,
  type InsertCustomer,
  type UpdateCustomer,
  type LoyaltySettings,
  type InsertLoyaltySettings,
  type CustomerLoyaltyPoints,
  type InsertCustomerLoyaltyPoints,
  type Employee,
  type InsertEmployee,
  type EmployeeWithRole,
  type Order,
  type OrderWithDetails,
  type OrderService,
  type OrderServiceWithDetails,
  type OrderProduct,
  type OrderProductWithDetails,
  type OrderRating,
  type InsertOrderRating,
  type OrderPackage,
  type OrderPackageWithDetails,
  type InsertOrderPackage,
  type OrderReturn,
  type OrderReturnWithDetails,
  type InsertOrderReturn,
  type OrderReturnLine,
  type OrderReturnLineWithDetails,
  type InsertOrderPayment,
  type OrderPayment,
  type InsertOrderReturnLine,
  type CreateOrderRequest,
  type OrderFilters,
  type DashboardStats,
  type SalonDashboardFilters,
  type SalonDashboardSummary,
  type InventorySummaryItem,
  type SupplierBalance,
  type Role,
  type InsertRole,
  type RoleWithPermissions,
  type RolePermission,
  type InsertRolePermission,
  rolePermissions,
  brands,
  type Brand,
  type InsertBrand,
  type EmployeeTask,
  type EmployeeTaskWithDetails,
  type InsertEmployeeTask,
  type SalaryProfile,
  type SalaryProfileWithEmployee,
  type InsertSalaryProfile,
  type PayrollRun,
  type PayrollRunWithLines,
  type InsertPayrollRun,
  type PayrollLine,
  type PayrollLineWithEmployee,
  type InsertPayrollLine,
  type CafeteriaProduct,
  type CafeteriaProductWithUnit,
  type InsertCafeteriaProduct,
  type CafeteriaStockBatch,
  type InsertCafeteriaStockBatch,
  type CafeteriaStockMovement,
  type InsertCafeteriaStockMovement,
  type CafeteriaPurchaseInvoice,
  type CafeteriaPurchaseInvoiceWithDetails,
  type InsertCafeteriaPurchaseInvoice,
  type CafeteriaPurchaseInvoiceLine,
  type CafeteriaPurchaseInvoiceLineWithProduct,
  type InsertCafeteriaPurchaseInvoiceLine,
  type CafeteriaPurchasePayment,
  type InsertCafeteriaPurchasePayment,
  type CafeteriaSalesOrder,
  type CafeteriaSalesOrderWithDetails,
  type CafeteriaEmployeeCustody,
  type CafeteriaEmployeeCustodyWithDetails,
  type InsertCafeteriaEmployeeCustody,
  type InsertCafeteriaSalesOrder,
  type CafeteriaSalesOrderLine,
  type CafeteriaSalesOrderLineWithProduct,
  type InsertCafeteriaSalesOrderLine,
  type CafeteriaSalesReturn,
  type CafeteriaSalesReturnWithDetails,
  type InsertCafeteriaSalesReturn,
  type CafeteriaSalesReturnLine,
  type InsertCafeteriaSalesReturnLine,
  type CreateCafeteriaPurchaseInvoiceRequest,
  type CreateCafeteriaSalesOrderRequest,
  type CafeteriaOrderFilters,
  type Appointment,
  type AppointmentService,
  type AppointmentWithDetails,
  type AppointmentServiceWithDetails,
  type InsertAppointment,
  type InsertAppointmentService,
  type CreateAppointmentRequest,
  type AppointmentFilters,
  type CafeteriaInventorySummaryItem,
  type CafeteriaStockMovementReportItem,
  type SystemSetting,
  type InsertSystemSetting,
  type Package,
  type PackageService,
  type PackageProduct,
  type PackageWithDetails,
  type PackageServiceWithDetails,
  type PackageProductWithDetails,
  type InsertPackage,
  type InsertPackageService,
  type InsertPackageProduct,
  type CreatePackageRequest,
  type PackageFilters,
  type Coupon,
  type CouponService,
  type CouponProduct,
  type CouponWithDetails,
  type CouponServiceWithDetails,
  type CouponProductWithDetails,
  type InsertCoupon,
  type InsertCouponService,
  type InsertCouponProduct,
  type CreateCouponRequest,
  type CouponFilters,
  reviewReasons,
  customerReviews,
  type ReviewReason,
  type InsertReviewReason,
  type CustomerReview,
  type CustomerReviewWithDetails,
  type InsertCustomerReview,
  jobPositions,
  nationalities,
  employeeSupervisors,
  workShifts,
  employeeShiftAssignments,
  attendanceRaw,
  attendanceLogs,
  overtimeEntries,
  leaves,
  type JobPosition,
  type InsertJobPosition,
  type Nationality,
  type InsertNationality,
  type EmployeeSupervisor,
  type EmployeeSupervisorWithDetails,
  type InsertEmployeeSupervisor,
  type SupervisorWithSubordinates,
  type WorkShift,
  type InsertWorkShift,
  type ShiftAssignment,
  type ShiftAssignmentWithDetails,
  type InsertShiftAssignment,
  type AttendanceRaw,
  type InsertAttendanceRaw,
  type AttendanceLog,
  type AttendanceLogWithDetails,
  type InsertAttendanceLog,
  type OvertimeEntry,
  type OvertimeEntryWithDetails,
  type InsertOvertimeEntry,
  type Leave,
  type LeaveWithDetails,
  type InsertLeave,
  expenseTypes,
  expenses,
  employeeCustodies,
  type ExpenseType,
  type InsertExpenseType,
  type Expense,
  type InsertExpense,
  type ExpenseWithDetails,
  type ExpenseFilters,
  type ExpensesSummary,
  type PayablesSummary,
  type SupplierStatement,
  type SupplierStatementTransaction,
  type EmployeeCustody,
  type EmployeeCustodyWithDetails,
  type InsertEmployeeCustody,
  type StockMovementSummaryItem,
  type Driver,
  type InsertDriver,
  type Trip,
  type InsertTrip,
  type TripWithDetails,
  type NearExpiryProductItem,
  type CommissionRecord,
  cashDisbursements,
  advanceInstallments,
  type CashDisbursement,
  type CashDisbursementWithDetails,
  type InsertCashDisbursement,
  type AdvanceInstallment,
  type AdvanceInstallmentWithDetails,
  type InsertAdvanceInstallment,
  type DailyReportFilters,
  type SalonDailyReport,
  type DailyReportPaymentMethodLine,
  type DailyReportCashSummary,
  type MarketingReportFilters,
  type MarketingReportSummary,
  type CouponUsageStat,
  type PackageUsageStat,
  type PackageTopService,
  type PackageTopProduct,
  commissionProfiles,
  commissionTargetSources,
  commissionTransactions,
  commissionPerformanceSnapshots,
  type CommissionProfile,
  type CommissionProfileWithSources,
  type InsertCommissionProfile,
  type CommissionTargetSource,
  type InsertCommissionTargetSource,
  type CommissionTransaction,
  type InsertCommissionTransaction,
  type CommissionPerformanceSnapshot,
  type EmployeeServiceActivity,
  type EmployeeOrderActivity,
  type EmployeeCommissionReport,
} from "@shared/schema";

export interface IStorage {
  // Suppliers
  getSuppliers(): Promise<SupplierWithBankAccounts[]>;
  getSupplier(id: number): Promise<SupplierWithBankAccounts | undefined>;
  createSupplier(supplier: InsertSupplier, bankAccounts?: Omit<InsertSupplierBankAccount, 'supplierId'>[]): Promise<SupplierWithBankAccounts>;
  updateSupplier(id: number, supplier: Partial<InsertSupplier>, bankAccounts?: (Omit<InsertSupplierBankAccount, 'supplierId'> & { id?: number })[]): Promise<SupplierWithBankAccounts | undefined>;
  deleteSupplier(id: number): Promise<boolean>;
  
  // Units
  getUnits(): Promise<Unit[]>;
  getUnit(id: number): Promise<Unit | undefined>;
  createUnit(unit: InsertUnit): Promise<Unit>;
  updateUnit(id: number, unit: Partial<InsertUnit>): Promise<Unit | undefined>;
  deleteUnit(id: number): Promise<boolean>;
  
  // Products
  getProducts(): Promise<ProductWithUnits[]>;
  getProduct(id: number): Promise<ProductWithUnits | undefined>;
  createProduct(product: InsertProduct): Promise<Product>;
  updateProduct(id: number, product: Partial<InsertProduct>): Promise<Product | undefined>;
  deleteProduct(id: number): Promise<boolean>;
  
  // Stock Batches
  getStockBatches(productId?: number): Promise<(StockBatch & { product?: Product })[]>;
  createStockBatch(batch: InsertStockBatch): Promise<StockBatch>;
  updateStockBatch(id: number, batch: Partial<InsertStockBatch>): Promise<StockBatch | undefined>;
  
  // Stock Movements
  getStockMovements(productId?: number): Promise<StockMovement[]>;
  createStockMovement(movement: InsertStockMovement): Promise<StockMovement>;
  
  // Payment Methods
  getPaymentMethods(): Promise<PaymentMethod[]>;
  getPaymentMethod(id: number): Promise<PaymentMethod | undefined>;
  createPaymentMethod(method: InsertPaymentMethod): Promise<PaymentMethod>;
  updatePaymentMethod(id: number, method: Partial<InsertPaymentMethod>): Promise<PaymentMethod | undefined>;
  deletePaymentMethod(id: number): Promise<boolean>;
  
  // Purchase Invoices
  getPurchaseInvoices(): Promise<PurchaseInvoiceWithDetails[]>;
  getPurchaseInvoice(id: number): Promise<PurchaseInvoiceWithDetails | undefined>;
  createPurchaseInvoice(invoice: InsertPurchaseInvoice, lines: Omit<InsertPurchaseInvoiceLine, 'purchaseInvoiceId'>[]): Promise<PurchaseInvoiceWithDetails>;
  updatePurchaseInvoice(id: number, invoice: Partial<InsertPurchaseInvoice>): Promise<PurchaseInvoice | undefined>;
  
  // Payments
  getPayments(supplierId?: number): Promise<(PurchasePayment & { paymentMethod?: PaymentMethod; supplier?: Supplier; invoice?: PurchaseInvoice })[]>;
  createPayment(payment: InsertPurchasePayment): Promise<PurchasePayment>;
  
  // Purchase Debit Notes (إشعارات الخصم للمشتريات)
  getPurchaseDebitNotes(): Promise<PurchaseDebitNoteWithDetails[]>;
  getPurchaseDebitNote(id: number): Promise<PurchaseDebitNoteWithDetails | undefined>;
  createPurchaseDebitNote(debitNote: InsertPurchaseDebitNote, lines: Omit<InsertPurchaseDebitNoteLine, 'debitNoteId'>[]): Promise<PurchaseDebitNoteWithDetails>;
  confirmPurchaseDebitNote(id: number): Promise<PurchaseDebitNoteWithDetails | undefined>;
  cancelPurchaseDebitNote(id: number): Promise<PurchaseDebitNoteWithDetails | undefined>;
  generateDebitNoteNumber(): Promise<string>;
  
  // Journal Entries
  getJournalEntries(): Promise<JournalEntryWithLines[]>;
  getJournalEntry(id: number): Promise<JournalEntryWithLines | undefined>;
  createJournalEntry(entry: InsertJournalEntry, lines: InsertJournalLine[]): Promise<JournalEntryWithLines>;
  
  // Dashboard & Reports
  getDashboardStats(): Promise<DashboardStats>;
  getSalonDashboardSummary(filters?: SalonDashboardFilters): Promise<SalonDashboardSummary>;
  getInventorySummary(): Promise<InventorySummaryItem[]>;
  getSupplierBalances(): Promise<SupplierBalance[]>;
  
  // Accounts (Chart of Accounts)
  getAccounts(): Promise<Account[]>;
  getAccountsTree(): Promise<AccountWithChildren[]>;
  getAccount(id: number): Promise<Account | undefined>;
  getAccountByCode(code: string): Promise<Account | undefined>;
  createAccount(account: InsertAccount): Promise<Account>;
  updateAccount(id: number, account: Partial<InsertAccount>): Promise<Account | undefined>;
  deleteAccount(id: number): Promise<boolean>;
  
  // Service Categories
  getServiceCategories(): Promise<ServiceCategoryWithCount[]>;
  getServiceCategoryWithServices(id: number): Promise<ServiceCategoryWithServices | undefined>;
  getServiceCategory(id: number): Promise<ServiceCategory | undefined>;
  createServiceCategory(category: InsertServiceCategory): Promise<ServiceCategory>;
  updateServiceCategory(id: number, category: Partial<InsertServiceCategory>): Promise<ServiceCategory | undefined>;
  deleteServiceCategory(id: number): Promise<boolean>;
  
  // Services
  getServices(filters?: { categoryId?: number; isActive?: boolean }): Promise<ServiceWithDetails[]>;
  getService(id: number): Promise<ServiceWithDetails | undefined>;
  createService(service: InsertService, relatedProducts?: { productId: number; quantityPerService: number; deductFromCommission: boolean; notes?: string }[]): Promise<ServiceWithDetails>;
  updateService(id: number, service: Partial<InsertService>, relatedProducts?: { productId: number; quantityPerService: number; deductFromCommission: boolean; notes?: string }[]): Promise<ServiceWithDetails | undefined>;
  deleteService(id: number): Promise<boolean>;
  
  // Service Products
  getServiceProducts(serviceId: number): Promise<ServiceProductWithProduct[]>;
  
  // Service Images
  getServiceImages(serviceId: number): Promise<ServiceImage[]>;
  addServiceImage(image: InsertServiceImage): Promise<ServiceImage>;
  deleteServiceImage(id: number): Promise<boolean>;
  updateServiceImagesOrder(serviceId: number, imageIds: number[]): Promise<void>;
  
  // Customers
  getCustomers(): Promise<Customer[]>;
  getCustomer(id: number): Promise<Customer | undefined>;
  getCustomerByPhone(phone: string): Promise<Customer | undefined>;
  createCustomer(customer: InsertCustomer): Promise<Customer>;
  updateCustomer(id: number, customer: Partial<InsertCustomer>): Promise<Customer | undefined>;
  deleteCustomer(id: number): Promise<boolean>;
  
  // General Ledger & Trial Balance
  getGeneralLedger(accountCode?: string, startDate?: string, endDate?: string): Promise<any[]>;
  getTrialBalance(asOfDate?: string): Promise<any[]>;
  getIncomeStatement(startDate?: string, endDate?: string): Promise<any>;
  getBalanceSheet(asOfDate?: string): Promise<any>;
  getCashFlowStatement(startDate?: string, endDate?: string): Promise<any>;
  
  // Manual Journal Entry
  createManualJournalEntry(data: { date: string; description: string; lines: { accountId: number; description?: string; debit: number; credit: number }[] }): Promise<JournalEntryWithLines>;
  
  // Roles
  getRoles(isActive?: boolean): Promise<Role[]>;
  getRolesWithPermissions(isActive?: boolean): Promise<RoleWithPermissions[]>;
  getRole(id: number): Promise<Role | undefined>;
  getRoleWithPermissions(id: number): Promise<RoleWithPermissions | undefined>;
  createRole(role: InsertRole): Promise<Role>;
  updateRole(id: number, role: Partial<InsertRole>): Promise<Role | undefined>;
  deleteRole(id: number): Promise<boolean>;
  
  // Role Permissions
  getRolePermissions(roleId: number): Promise<RolePermission[]>;
  setRolePermissions(roleId: number, permissions: InsertRolePermission[]): Promise<RolePermission[]>;
  
  // Brands
  getBrands(isActive?: boolean): Promise<Brand[]>;
  getBrand(id: number): Promise<Brand | undefined>;
  createBrand(brand: InsertBrand): Promise<Brand>;
  updateBrand(id: number, brand: Partial<InsertBrand>): Promise<Brand | undefined>;
  deleteBrand(id: number): Promise<boolean>;
  
  // Employees (extended HR)
  getEmployees(filters?: { isActive?: boolean; roleId?: number; branchId?: number; search?: string }): Promise<EmployeeWithRole[]>;
  getEmployee(id: number): Promise<EmployeeWithRole | undefined>;
  createEmployee(employee: InsertEmployee): Promise<Employee>;
  updateEmployee(id: number, employee: Partial<InsertEmployee>): Promise<Employee | undefined>;
  deleteEmployee(id: number): Promise<boolean>;
  
  // Employee Tasks
  getEmployeeTasks(filters?: { assignedToEmployeeId?: number; status?: string; dueDate?: string }): Promise<EmployeeTaskWithDetails[]>;
  getEmployeeTask(id: number): Promise<EmployeeTaskWithDetails | undefined>;
  createEmployeeTask(task: InsertEmployeeTask): Promise<EmployeeTask>;
  updateEmployeeTask(id: number, task: Partial<InsertEmployeeTask>): Promise<EmployeeTask | undefined>;
  deleteEmployeeTask(id: number): Promise<boolean>;
  
  // Salary Profiles
  getSalaryProfiles(): Promise<SalaryProfileWithEmployee[]>;
  getSalaryProfile(employeeId: number): Promise<SalaryProfile | undefined>;
  createOrUpdateSalaryProfile(profile: InsertSalaryProfile): Promise<SalaryProfile>;
  deleteSalaryProfile(employeeId: number): Promise<boolean>;
  
  // Payroll Runs
  getPayrollRuns(): Promise<PayrollRun[]>;
  getPayrollRun(id: number): Promise<PayrollRunWithLines | undefined>;
  createPayrollRun(run: InsertPayrollRun): Promise<PayrollRun>;
  updatePayrollRun(id: number, run: Partial<InsertPayrollRun>): Promise<PayrollRun | undefined>;
  generatePayrollLines(payrollRunId: number, employeeIds?: number[]): Promise<PayrollLine[]>;
  approvePayrollRun(id: number): Promise<PayrollRun | undefined>;
  markPayrollRunAsPaid(id: number): Promise<PayrollRun | undefined>;
  
  // Payroll Lines
  getPayrollLines(payrollRunId: number): Promise<PayrollLineWithEmployee[]>;
  getEmployeePayrollHistory(employeeId: number): Promise<PayrollLineWithEmployee[]>;
  updatePayrollLine(id: number, line: Partial<InsertPayrollLine>): Promise<PayrollLine | undefined>;
  
  // Orders
  getOrders(filters?: OrderFilters): Promise<OrderWithDetails[]>;
  getOrder(id: number): Promise<OrderWithDetails | undefined>;
  createOrder(request: CreateOrderRequest): Promise<OrderWithDetails>;
  updateOrder(id: number, updates: { status?: string; notes?: string | null; couponCode?: string | null; couponDiscountAmount?: number; deferredAt?: Date | null; deferredReason?: string | null; cancelledAt?: Date | null; cancelledReason?: string | null; returnedAt?: Date | null; returnedReason?: string | null; scheduledAt?: Date | null; ratingValue?: number | null; ratingComment?: string | null }): Promise<OrderWithDetails | undefined>;
  
  // Order Service Status
  updateOrderService(orderServiceId: number, updates: { executingEmployeeId?: number | null; status?: string; deferredAt?: Date | null; deferredReason?: string | null; completedAt?: Date | null; scheduledAt?: Date | null }): Promise<OrderService | undefined>;
  getOrderServices(orderId: number): Promise<OrderServiceWithDetails[]>;
  
  // Order Returns
  getOrderReturns(orderId?: number): Promise<OrderReturnWithDetails[]>;
  getOrderReturn(id: number): Promise<OrderReturnWithDetails | undefined>;
  createOrderReturn(returnData: InsertOrderReturn, lines: Omit<InsertOrderReturnLine, 'returnId'>[]): Promise<OrderReturnWithDetails>;
  
  // Order Ratings
  createOrUpdateOrderRating(orderId: number, rating: Omit<InsertOrderRating, 'orderId'>): Promise<OrderRating>;
  getOrderRating(orderId: number): Promise<OrderRating | undefined>;
  
  // Order Payments
  getOrderPayments(orderId: number): Promise<(OrderPayment & { paymentMethod?: PaymentMethod })[]>;
  createOrderPayment(payment: InsertOrderPayment): Promise<OrderPayment>;
  
  // Order Cancellation
  cancelOrder(id: number, reason?: string, refundPaymentMethodId?: number): Promise<OrderWithDetails | undefined>;
  
  // Cancelled Order Payment (سداد الطلبات الملغاة)
  payCancelledOrder(orderId: number, payment: { amount: number; paymentMethodId: number; notes?: string }): Promise<{ success: boolean; journalEntryId: number }>;
  
  // Coupon Validation
  validateCoupon(code: string, orderServices?: { serviceId: number; quantity: number; unitPrice: number }[], orderProducts?: { productId: number; quantity: number; unitPrice: number }[]): Promise<{ valid: boolean; error?: string; coupon?: CouponWithDetails; discountAmount?: number }>;
  applyCouponToOrder(couponId: number, orderServices?: { serviceId: number; quantity: number; unitPrice: number }[], orderProducts?: { productId: number; quantity: number; unitPrice: number }[]): Promise<number>;
  
  // Package Application
  getPackageItems(packageId: number): Promise<{ services: { serviceId: number; quantity: number; unitPrice: number }[]; products: { productId: number; quantity: number; unitPrice: number }[] }>;
  
  // Appointments
  getAppointments(filters?: AppointmentFilters): Promise<AppointmentWithDetails[]>;
  getAppointment(id: number): Promise<AppointmentWithDetails | undefined>;
  createAppointment(request: CreateAppointmentRequest): Promise<AppointmentWithDetails>;
  updateAppointment(id: number, updates: {
    scheduledAt?: Date;
    branchId?: number | null;
    appointmentType?: string;
    deliveryFee?: number;
    locationLat?: number | null;
    locationLng?: number | null;
    addressText?: string | null;
    status?: string;
    notes?: string | null;
    cancelledReason?: string | null;
    relatedOrderId?: number | null;
  }): Promise<AppointmentWithDetails | undefined>;
  confirmAppointment(id: number): Promise<AppointmentWithDetails | undefined>;
  cancelAppointment(id: number, reason?: string): Promise<AppointmentWithDetails | undefined>;
  generateAppointmentNumber(): Promise<string>;
  
  // System Settings
  getSystemSetting(key: string): Promise<SystemSetting | undefined>;
  setSystemSetting(key: string, value: string, description?: string, valueType?: string): Promise<SystemSetting>;
  getSystemSettings(): Promise<SystemSetting[]>;
  
  // Cafeteria Products
  getCafeteriaProducts(): Promise<CafeteriaProductWithUnit[]>;
  getCafeteriaProduct(id: number): Promise<CafeteriaProductWithUnit | undefined>;
  createCafeteriaProduct(product: InsertCafeteriaProduct): Promise<CafeteriaProduct>;
  updateCafeteriaProduct(id: number, product: Partial<InsertCafeteriaProduct>): Promise<CafeteriaProduct | undefined>;
  deleteCafeteriaProduct(id: number): Promise<boolean>;
  
  // Cafeteria Stock
  getCafeteriaStockBatches(productId?: number): Promise<(CafeteriaStockBatch & { cafeteriaProduct?: CafeteriaProduct })[]>;
  createCafeteriaStockBatch(batch: InsertCafeteriaStockBatch): Promise<CafeteriaStockBatch>;
  updateCafeteriaStockBatch(id: number, batch: Partial<InsertCafeteriaStockBatch>): Promise<CafeteriaStockBatch | undefined>;
  getCafeteriaStockMovements(productId?: number): Promise<CafeteriaStockMovement[]>;
  createCafeteriaStockMovement(movement: InsertCafeteriaStockMovement): Promise<CafeteriaStockMovement>;
  
  // Cafeteria Purchase Invoices
  getCafeteriaPurchaseInvoices(): Promise<CafeteriaPurchaseInvoiceWithDetails[]>;
  getCafeteriaPurchaseInvoice(id: number): Promise<CafeteriaPurchaseInvoiceWithDetails | undefined>;
  createCafeteriaPurchaseInvoice(request: CreateCafeteriaPurchaseInvoiceRequest): Promise<CafeteriaPurchaseInvoiceWithDetails>;
  
  // Cafeteria Purchase Payments
  getCafeteriaPurchasePayments(invoiceId?: number): Promise<(CafeteriaPurchasePayment & { paymentMethod?: PaymentMethod; supplier?: Supplier })[]>;
  createCafeteriaPurchasePayment(payment: InsertCafeteriaPurchasePayment): Promise<CafeteriaPurchasePayment>;
  
  // Cafeteria Sales Orders
  getCafeteriaSalesOrders(filters?: CafeteriaOrderFilters): Promise<CafeteriaSalesOrderWithDetails[]>;
  getCafeteriaSalesOrder(id: number): Promise<CafeteriaSalesOrderWithDetails | undefined>;
  createCafeteriaSalesOrder(request: CreateCafeteriaSalesOrderRequest): Promise<CafeteriaSalesOrderWithDetails>;
  updateCafeteriaSalesOrder(id: number, updates: { status?: string; notes?: string | null }): Promise<CafeteriaSalesOrderWithDetails | undefined>;
  
  // Cafeteria Reports
  getCafeteriaInventorySummary(): Promise<CafeteriaInventorySummaryItem[]>;
  getCafeteriaInventoryMovements(productId: number): Promise<CafeteriaStockMovementReportItem[]>;
  
  // Cafeteria Employee Custodies (عهد موظفي الكافتيريا)
  getCafeteriaEmployeeCustodies(filters?: { employeeId?: number; status?: string }): Promise<CafeteriaEmployeeCustodyWithDetails[]>;
  getCafeteriaEmployeeCustody(id: number): Promise<CafeteriaEmployeeCustodyWithDetails | undefined>;
  createCafeteriaEmployeeCustody(custody: InsertCafeteriaEmployeeCustody): Promise<CafeteriaEmployeeCustodyWithDetails>;
  closeCafeteriaEmployeeCustody(id: number, closureData: { closedByEmployeeId: number; closureReason: string; returnedQuantity?: number; closureNotes?: string }): Promise<CafeteriaEmployeeCustodyWithDetails | undefined>;
  deleteCafeteriaEmployeeCustody(id: number): Promise<boolean>;
  
  // Cafeteria Sales Returns (مرتجعات مبيعات الكافتيريا)
  getCafeteriaSalesReturns(orderId?: number): Promise<CafeteriaSalesReturnWithDetails[]>;
  getCafeteriaSalesReturn(id: number): Promise<CafeteriaSalesReturnWithDetails | undefined>;
  createCafeteriaSalesCancellationReturn(orderId: number, reason: string, processedByEmployeeId?: number): Promise<CafeteriaSalesReturnWithDetails>;
  
  // Marketing - Packages
  getPackages(filters?: PackageFilters): Promise<PackageWithDetails[]>;
  getPackage(id: number): Promise<PackageWithDetails | undefined>;
  getValidPackage(id: number): Promise<PackageWithDetails | undefined>;
  createPackage(request: CreatePackageRequest): Promise<PackageWithDetails>;
  updatePackage(id: number, request: Partial<CreatePackageRequest>): Promise<PackageWithDetails | undefined>;
  deletePackage(id: number): Promise<boolean>;
  
  // Marketing - Coupons
  getCoupons(filters?: CouponFilters): Promise<CouponWithDetails[]>;
  getCoupon(id: number): Promise<CouponWithDetails | undefined>;
  getCouponByCode(code: string): Promise<CouponWithDetails | undefined>;
  getValidCoupon(id: number): Promise<CouponWithDetails | undefined>;
  getValidCouponByCode(code: string): Promise<CouponWithDetails | undefined>;
  createCoupon(request: CreateCouponRequest): Promise<CouponWithDetails>;
  updateCoupon(id: number, request: Partial<CreateCouponRequest>): Promise<CouponWithDetails | undefined>;
  deleteCoupon(id: number): Promise<boolean>;
  
  // Customer Reviews (تقييمات العملاء)
  getReviewReasons(filters?: { reasonType?: string; isActive?: boolean }): Promise<ReviewReason[]>;
  getReviewReason(id: number): Promise<ReviewReason | undefined>;
  createReviewReason(reason: InsertReviewReason): Promise<ReviewReason>;
  updateReviewReason(id: number, reason: Partial<InsertReviewReason>): Promise<ReviewReason | undefined>;
  deleteReviewReason(id: number): Promise<boolean>;
  
  getCustomerReviews(filters?: { customerId?: number; orderId?: number; reviewReasonId?: number; status?: string }): Promise<CustomerReviewWithDetails[]>;
  getCustomerReview(id: number): Promise<CustomerReviewWithDetails | undefined>;
  createCustomerReview(review: InsertCustomerReview): Promise<CustomerReview>;
  updateCustomerReviewStatus(id: number, status: string, reviewedByEmployeeId?: number, responseNotes?: string): Promise<CustomerReview | undefined>;
  deleteCustomerReview(id: number): Promise<boolean>;
  
  // HR - Job Positions (الوظائف)
  getJobPositions(isActive?: boolean): Promise<JobPosition[]>;
  getJobPosition(id: number): Promise<JobPosition | undefined>;
  createJobPosition(position: InsertJobPosition): Promise<JobPosition>;
  updateJobPosition(id: number, position: Partial<InsertJobPosition>): Promise<JobPosition | undefined>;
  deleteJobPosition(id: number): Promise<boolean>;
  
  // HR - Nationalities (الجنسيات)
  getNationalities(isActive?: boolean): Promise<Nationality[]>;
  getNationality(id: number): Promise<Nationality | undefined>;
  createNationality(nationality: InsertNationality): Promise<Nationality>;
  updateNationality(id: number, nationality: Partial<InsertNationality>): Promise<Nationality | undefined>;
  deleteNationality(id: number): Promise<boolean>;
  
  // HR - Employee Supervisors (المشرفين)
  getEmployeeSupervisors(): Promise<EmployeeSupervisorWithDetails[]>;
  getSupervisorsWithSubordinates(): Promise<SupervisorWithSubordinates[]>;
  getSubordinates(supervisorEmployeeId: number): Promise<EmployeeSupervisorWithDetails[]>;
  getSupervisorFor(employeeId: number): Promise<EmployeeSupervisorWithDetails | undefined>;
  assignSupervisor(data: InsertEmployeeSupervisor): Promise<EmployeeSupervisor>;
  removeSupervisor(employeeId: number): Promise<boolean>;
  
  // HR - Work Shifts (الورديات)
  getWorkShifts(isActive?: boolean): Promise<WorkShift[]>;
  getWorkShift(id: number): Promise<WorkShift | undefined>;
  createWorkShift(shift: InsertWorkShift): Promise<WorkShift>;
  updateWorkShift(id: number, shift: Partial<InsertWorkShift>): Promise<WorkShift | undefined>;
  deleteWorkShift(id: number): Promise<boolean>;
  
  // HR - Shift Assignments (مناوبات الموظفين)
  getShiftAssignments(filters?: { employeeId?: number; workShiftId?: number; date?: string }): Promise<ShiftAssignmentWithDetails[]>;
  getShiftAssignment(id: number): Promise<ShiftAssignmentWithDetails | undefined>;
  getActiveShiftAssignment(employeeId: number, date: string): Promise<ShiftAssignmentWithDetails | undefined>;
  createShiftAssignment(assignment: InsertShiftAssignment): Promise<ShiftAssignment>;
  updateShiftAssignment(id: number, assignment: Partial<InsertShiftAssignment>): Promise<ShiftAssignment | undefined>;
  deleteShiftAssignment(id: number): Promise<boolean>;
  
  // HR - Attendance Raw (البصمات الخام)
  getAttendanceRaw(filters?: { employeeId?: number; startDate?: string; endDate?: string }): Promise<AttendanceRaw[]>;
  createAttendanceRaw(record: InsertAttendanceRaw): Promise<AttendanceRaw>;
  
  // HR - Attendance Logs (سجل الحضور)
  getAttendanceLogs(filters?: { employeeId?: number; startDate?: string; endDate?: string; status?: string }): Promise<AttendanceLogWithDetails[]>;
  getAttendanceLog(id: number): Promise<AttendanceLogWithDetails | undefined>;
  getEmployeeAttendanceForDate(employeeId: number, date: string): Promise<AttendanceLog | undefined>;
  createAttendanceLog(log: InsertAttendanceLog): Promise<AttendanceLog>;
  updateAttendanceLog(id: number, log: Partial<InsertAttendanceLog>): Promise<AttendanceLog | undefined>;
  deleteAttendanceLog(id: number): Promise<boolean>;
  
  // HR - Overtime (الوقت الإضافي)
  getOvertimeEntries(filters?: { employeeId?: number; startDate?: string; endDate?: string }): Promise<OvertimeEntryWithDetails[]>;
  getOvertimeEntry(id: number): Promise<OvertimeEntryWithDetails | undefined>;
  createOvertimeEntry(entry: InsertOvertimeEntry): Promise<OvertimeEntry>;
  updateOvertimeEntry(id: number, entry: Partial<InsertOvertimeEntry>): Promise<OvertimeEntry | undefined>;
  deleteOvertimeEntry(id: number): Promise<boolean>;
  
  // HR - Leaves (الإجازات)
  getLeaves(filters?: { employeeId?: number; status?: string; leaveType?: string; startDate?: string; endDate?: string }): Promise<LeaveWithDetails[]>;
  getLeave(id: number): Promise<LeaveWithDetails | undefined>;
  createLeave(leave: InsertLeave): Promise<Leave>;
  updateLeave(id: number, leave: Partial<InsertLeave>): Promise<Leave | undefined>;
  deleteLeave(id: number): Promise<boolean>;
  approveLeave(id: number): Promise<Leave | undefined>;
  rejectLeave(id: number): Promise<Leave | undefined>;
  
  // Expense Types (أنواع النفقات)
  getExpenseTypes(isActive?: boolean): Promise<ExpenseType[]>;
  getExpenseType(id: number): Promise<ExpenseType | undefined>;
  createExpenseType(expenseType: InsertExpenseType): Promise<ExpenseType>;
  updateExpenseType(id: number, expenseType: Partial<InsertExpenseType>): Promise<ExpenseType | undefined>;
  deleteExpenseType(id: number): Promise<boolean>;
  
  // Expenses (النفقات)
  getExpenses(filters?: ExpenseFilters): Promise<ExpenseWithDetails[]>;
  getExpense(id: number): Promise<ExpenseWithDetails | undefined>;
  createExpense(expense: InsertExpense): Promise<Expense>;
  updateExpense(id: number, expense: Partial<InsertExpense>): Promise<Expense | undefined>;
  deleteExpense(id: number): Promise<boolean>;
  getExpensesSummary(filters?: ExpenseFilters): Promise<ExpensesSummary>;
  
  // Payables (الديون)
  getPayablesSummary(): Promise<PayablesSummary>;
  getSupplierStatement(supplierId: number, fromDate?: string, toDate?: string): Promise<SupplierStatement>;
  
  // Employee Custodies (عهدة الموظف)
  getEmployeeCustodies(filters?: { employeeId?: number; status?: string; productId?: number }): Promise<EmployeeCustodyWithDetails[]>;
  getEmployeeCustody(id: number): Promise<EmployeeCustodyWithDetails | undefined>;
  createEmployeeCustody(custody: InsertEmployeeCustody): Promise<EmployeeCustodyWithDetails>;
  updateEmployeeCustody(id: number, custody: Partial<InsertEmployeeCustody>): Promise<EmployeeCustody | undefined>;
  deleteEmployeeCustody(id: number): Promise<boolean>;
  getEmployeeActiveCustodies(employeeId: number): Promise<EmployeeCustodyWithDetails[]>;
  
  // Employee Commissions Report (تقرير عمولات الموظف)
  getEmployeeCommissions(employeeId: number, startDate: string, endDate: string): Promise<CommissionRecord[]>;
  
  // Inventory Reports (تقارير المخزون)
  getStockMovementSummary(fromDate: string, toDate: string, productId?: number): Promise<StockMovementSummaryItem[]>;
  getNearExpiryProducts(days?: number): Promise<NearExpiryProductItem[]>;
  
  // Moving Average Cost (متوسط التكلفة)
  updateProductAverageCost(productId: number, newQuantity: number, newCost: number): Promise<Product | undefined>;
  getProductAverageCost(productId: number): Promise<number>;
  
  // Drivers (السائقون)
  getDrivers(): Promise<Driver[]>;
  getDriver(id: number): Promise<Driver | undefined>;
  createDriver(driver: InsertDriver): Promise<Driver>;
  updateDriver(id: number, driver: Partial<InsertDriver>): Promise<Driver | undefined>;
  deleteDriver(id: number): Promise<boolean>;
  
  // Trips (الرحلات)
  getTrips(filters?: { driverId?: number; tripType?: string; status?: string }): Promise<TripWithDetails[]>;
  getTrip(id: number): Promise<TripWithDetails | undefined>;
  createTrip(trip: InsertTrip): Promise<Trip>;
  updateTrip(id: number, trip: Partial<InsertTrip>): Promise<Trip | undefined>;
  deleteTrip(id: number): Promise<boolean>;
  generateTripNumber(): Promise<string>;

  // Cash Disbursements (صرف نقدي)
  getCashDisbursements(filters?: { 
    disbursementType?: string; 
    employeeId?: number; 
    driverId?: number;
    fromDate?: string;
    toDate?: string;
  }): Promise<CashDisbursementWithDetails[]>;
  getCashDisbursement(id: number): Promise<CashDisbursementWithDetails | undefined>;
  createCashDisbursement(disbursement: InsertCashDisbursement): Promise<CashDisbursementWithDetails>;
  updateCashDisbursement(id: number, disbursement: Partial<InsertCashDisbursement>): Promise<CashDisbursement | undefined>;
  deleteCashDisbursement(id: number): Promise<boolean>;
  generateDisbursementNumber(): Promise<string>;
  
  // Get disbursements for employee/driver (for employee detail page)
  getEmployeeDisbursements(employeeId: number): Promise<CashDisbursementWithDetails[]>;
  getDriverDisbursements(driverId: number): Promise<CashDisbursementWithDetails[]>;
  
  // Advance installments
  getAdvanceInstallments(cashDisbursementId: number): Promise<AdvanceInstallment[]>;
  updateInstallmentStatus(id: number, status: string, payrollLineId?: number): Promise<AdvanceInstallment | undefined>;
}

export class DatabaseStorage implements IStorage {
  // ========================================
  // SUPPLIERS
  // ========================================
  
  async getSuppliers(): Promise<SupplierWithBankAccounts[]> {
    const allSuppliers = await db.select().from(suppliers).orderBy(desc(suppliers.createdAt));
    const allBankAccounts = await db.select().from(supplierBankAccounts);
    
    return allSuppliers.map(supplier => ({
      ...supplier,
      bankAccounts: allBankAccounts.filter(acc => acc.supplierId === supplier.id),
    }));
  }
  
  async getSupplier(id: number): Promise<SupplierWithBankAccounts | undefined> {
    const [supplier] = await db.select().from(suppliers).where(eq(suppliers.id, id));
    if (!supplier) return undefined;
    
    const bankAccountsList = await db.select().from(supplierBankAccounts).where(eq(supplierBankAccounts.supplierId, id));
    
    return {
      ...supplier,
      bankAccounts: bankAccountsList,
    };
  }
  
  async createSupplier(supplier: InsertSupplier, bankAccountsData?: Omit<InsertSupplierBankAccount, 'supplierId'>[]): Promise<SupplierWithBankAccounts> {
    const [newSupplier] = await db.insert(suppliers).values(supplier).returning();
    
    let bankAccountsList: SupplierBankAccount[] = [];
    if (bankAccountsData && bankAccountsData.length > 0) {
      bankAccountsList = await db.insert(supplierBankAccounts).values(
        bankAccountsData.map(acc => ({ ...acc, supplierId: newSupplier.id }))
      ).returning();
    }
    
    return {
      ...newSupplier,
      bankAccounts: bankAccountsList,
    };
  }
  
  async updateSupplier(id: number, supplier: Partial<InsertSupplier>, bankAccountsData?: (Omit<InsertSupplierBankAccount, 'supplierId'> & { id?: number })[]): Promise<SupplierWithBankAccounts | undefined> {
    const [updatedSupplier] = await db.update(suppliers)
      .set({ ...supplier, updatedAt: new Date() })
      .where(eq(suppliers.id, id))
      .returning();
    
    if (!updatedSupplier) return undefined;
    
    if (bankAccountsData) {
      // Delete old bank accounts that are not in the new list
      const existingIds = bankAccountsData.filter(acc => acc.id).map(acc => acc.id!);
      await db.delete(supplierBankAccounts).where(
        and(
          eq(supplierBankAccounts.supplierId, id),
          existingIds.length > 0 ? sql`${supplierBankAccounts.id} NOT IN (${sql.join(existingIds.map(i => sql`${i}`), sql`, `)})` : sql`1=1`
        )
      );
      
      // Update or create bank accounts
      for (const acc of bankAccountsData) {
        if (acc.id) {
          await db.update(supplierBankAccounts)
            .set({ bankNameAr: acc.bankNameAr, iban: acc.iban, accountNumber: acc.accountNumber, isDefault: acc.isDefault })
            .where(eq(supplierBankAccounts.id, acc.id));
        } else {
          await db.insert(supplierBankAccounts).values({ ...acc, supplierId: id });
        }
      }
    }
    
    return this.getSupplier(id);
  }
  
  async deleteSupplier(id: number): Promise<boolean> {
    // Soft delete by setting isActive to false
    const [result] = await db.update(suppliers)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(suppliers.id, id))
      .returning();
    return !!result;
  }
  
  // ========================================
  // UNITS
  // ========================================
  
  async getUnits(): Promise<Unit[]> {
    return db.select().from(units).orderBy(asc(units.nameAr));
  }
  
  async getUnit(id: number): Promise<Unit | undefined> {
    const [unit] = await db.select().from(units).where(eq(units.id, id));
    return unit;
  }
  
  async createUnit(unit: InsertUnit): Promise<Unit> {
    const [newUnit] = await db.insert(units).values(unit).returning();
    return newUnit;
  }
  
  async updateUnit(id: number, unit: Partial<InsertUnit>): Promise<Unit | undefined> {
    const [updatedUnit] = await db.update(units)
      .set({ ...unit, updatedAt: new Date() })
      .where(eq(units.id, id))
      .returning();
    return updatedUnit;
  }
  
  async deleteUnit(id: number): Promise<boolean> {
    const [result] = await db.delete(units).where(eq(units.id, id)).returning();
    return !!result;
  }
  
  // ========================================
  // PRODUCTS
  // ========================================
  
  async getProducts(): Promise<ProductWithUnits[]> {
    const allProducts = await db.select().from(products).orderBy(desc(products.createdAt));
    const allUnits = await db.select().from(units);
    
    return allProducts.map(product => ({
      ...product,
      purchaseUnit: allUnits.find(u => u.id === product.purchaseUnitId),
      sellUnit: allUnits.find(u => u.id === product.sellUnitId),
    }));
  }
  
  async getProduct(id: number): Promise<ProductWithUnits | undefined> {
    const [product] = await db.select().from(products).where(eq(products.id, id));
    if (!product) return undefined;
    
    const allUnits = await db.select().from(units);
    
    return {
      ...product,
      purchaseUnit: allUnits.find(u => u.id === product.purchaseUnitId),
      sellUnit: allUnits.find(u => u.id === product.sellUnitId),
    };
  }
  
  async createProduct(product: InsertProduct): Promise<Product> {
    const [newProduct] = await db.insert(products).values(product).returning();
    return newProduct;
  }
  
  async updateProduct(id: number, product: Partial<InsertProduct>): Promise<Product | undefined> {
    const [updatedProduct] = await db.update(products)
      .set({ ...product, updatedAt: new Date() })
      .where(eq(products.id, id))
      .returning();
    return updatedProduct;
  }
  
  async deleteProduct(id: number): Promise<boolean> {
    // Soft delete
    const [result] = await db.update(products)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(products.id, id))
      .returning();
    return !!result;
  }
  
  // ========================================
  // STOCK BATCHES
  // ========================================
  
  async getStockBatches(productId?: number): Promise<(StockBatch & { product?: Product })[]> {
    let query = db.select().from(stockBatches);
    if (productId) {
      query = query.where(eq(stockBatches.productId, productId)) as any;
    }
    
    const batches = await query.orderBy(asc(stockBatches.expiryDate));
    const allProducts = await db.select().from(products);
    
    return batches.map(batch => ({
      ...batch,
      product: allProducts.find(p => p.id === batch.productId),
    }));
  }
  
  async createStockBatch(batch: InsertStockBatch): Promise<StockBatch> {
    const [newBatch] = await db.insert(stockBatches).values(batch).returning();
    return newBatch;
  }
  
  async updateStockBatch(id: number, batch: Partial<InsertStockBatch>): Promise<StockBatch | undefined> {
    const [updatedBatch] = await db.update(stockBatches)
      .set({ ...batch, updatedAt: new Date() })
      .where(eq(stockBatches.id, id))
      .returning();
    return updatedBatch;
  }
  
  // ========================================
  // STOCK MOVEMENTS
  // ========================================
  
  async getStockMovements(productId?: number): Promise<StockMovement[]> {
    let query = db.select().from(stockMovements);
    if (productId) {
      query = query.where(eq(stockMovements.productId, productId)) as any;
    }
    return query.orderBy(desc(stockMovements.createdAt));
  }
  
  async createStockMovement(movement: InsertStockMovement): Promise<StockMovement> {
    const [newMovement] = await db.insert(stockMovements).values(movement).returning();
    return newMovement;
  }
  
  // ========================================
  // PAYMENT METHODS
  // ========================================
  
  async getPaymentMethods(): Promise<PaymentMethod[]> {
    return db.select().from(paymentMethods).orderBy(asc(paymentMethods.nameAr));
  }
  
  async getPaymentMethod(id: number): Promise<PaymentMethod | undefined> {
    const [method] = await db.select().from(paymentMethods).where(eq(paymentMethods.id, id));
    return method;
  }
  
  async createPaymentMethod(method: InsertPaymentMethod): Promise<PaymentMethod> {
    const [newMethod] = await db.insert(paymentMethods).values(method).returning();
    return newMethod;
  }
  
  async updatePaymentMethod(id: number, method: Partial<InsertPaymentMethod>): Promise<PaymentMethod | undefined> {
    const [updatedMethod] = await db.update(paymentMethods)
      .set({ ...method, updatedAt: new Date() })
      .where(eq(paymentMethods.id, id))
      .returning();
    return updatedMethod;
  }
  
  async deletePaymentMethod(id: number): Promise<boolean> {
    const [result] = await db.update(paymentMethods)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(paymentMethods.id, id))
      .returning();
    return !!result;
  }
  
  // ========================================
  // PURCHASE INVOICES
  // ========================================
  
  async getPurchaseInvoices(): Promise<PurchaseInvoiceWithDetails[]> {
    const invoices = await db.select().from(purchaseInvoices).orderBy(desc(purchaseInvoices.createdAt));
    const allSuppliers = await db.select().from(suppliers);
    const allLines = await db.select().from(purchaseInvoiceLines);
    const allPayments = await db.select().from(purchasePayments);
    const allProducts = await db.select().from(products);
    const allUnits = await db.select().from(units);
    const allPaymentMethods = await db.select().from(paymentMethods);
    
    return invoices.map(invoice => ({
      ...invoice,
      supplier: allSuppliers.find(s => s.id === invoice.supplierId),
      lines: allLines.filter(l => l.purchaseInvoiceId === invoice.id).map(line => ({
        ...line,
        product: allProducts.find(p => p.id === line.productId),
        purchaseUnit: allUnits.find(u => u.id === line.purchaseUnitId),
        sellUnit: allUnits.find(u => u.id === line.sellUnitId),
      })),
      payments: allPayments.filter(p => p.purchaseInvoiceId === invoice.id).map(payment => ({
        ...payment,
        paymentMethod: allPaymentMethods.find(m => m.id === payment.paymentMethodId),
      })),
    }));
  }
  
  async getPurchaseInvoice(id: number): Promise<PurchaseInvoiceWithDetails | undefined> {
    const [invoice] = await db.select().from(purchaseInvoices).where(eq(purchaseInvoices.id, id));
    if (!invoice) return undefined;
    
    const [supplier] = await db.select().from(suppliers).where(eq(suppliers.id, invoice.supplierId));
    const lines = await db.select().from(purchaseInvoiceLines).where(eq(purchaseInvoiceLines.purchaseInvoiceId, id));
    const payments = await db.select().from(purchasePayments).where(eq(purchasePayments.purchaseInvoiceId, id));
    const allProducts = await db.select().from(products);
    const allUnits = await db.select().from(units);
    const allPaymentMethods = await db.select().from(paymentMethods);
    
    return {
      ...invoice,
      supplier,
      lines: lines.map(line => ({
        ...line,
        product: allProducts.find(p => p.id === line.productId),
        purchaseUnit: allUnits.find(u => u.id === line.purchaseUnitId),
        sellUnit: allUnits.find(u => u.id === line.sellUnitId),
      })),
      payments: payments.map(payment => ({
        ...payment,
        paymentMethod: allPaymentMethods.find(m => m.id === payment.paymentMethodId),
      })),
    };
  }
  
  async createPurchaseInvoice(invoice: InsertPurchaseInvoice, lines: Omit<InsertPurchaseInvoiceLine, 'purchaseInvoiceId'>[]): Promise<PurchaseInvoiceWithDetails> {
    const [newInvoice] = await db.insert(purchaseInvoices).values(invoice).returning();
    
    const linesWithInvoiceId = lines.map(line => ({
      ...line,
      purchaseInvoiceId: newInvoice.id,
    }));
    
    await db.insert(purchaseInvoiceLines).values(linesWithInvoiceId);
    
    // Create stock batches and movements for each line
    for (const line of linesWithInvoiceId) {
      // Create stock batch
      const [batch] = await db.insert(stockBatches).values({
        productId: line.productId,
        expiryDate: line.expiryDate,
        quantityOnHand: line.quantitySellUnit,
        unitCost: line.purchasePricePerSellUnit,
      }).returning();
      
      // Create stock movement
      await db.insert(stockMovements).values({
        productId: line.productId,
        movementDate: invoice.invoiceDate,
        movementType: "purchase",
        quantityIn: line.quantitySellUnit,
        quantityOut: 0,
        unitCost: line.purchasePricePerSellUnit,
        referenceType: "PurchaseInvoice",
        referenceId: newInvoice.id,
        batchId: batch.id,
      });
      
      // Update product moving average cost
      await this.updateProductAverageCost(line.productId, line.quantitySellUnit, line.purchasePricePerSellUnit);
    }
    
    // Create journal entry for purchase
    await this.createPurchaseJournalEntry(newInvoice, lines);
    
    return this.getPurchaseInvoice(newInvoice.id) as Promise<PurchaseInvoiceWithDetails>;
  }
  
  private async createPurchaseJournalEntry(invoice: PurchaseInvoice, lines: Omit<InsertPurchaseInvoiceLine, 'purchaseInvoiceId'>[]): Promise<void> {
    // Generate entry number
    const count = await db.select({ count: sql<number>`count(*)` }).from(journalEntries);
    const entryNumber = `JE-${String((count[0]?.count || 0) + 1).padStart(6, '0')}`;
    
    const [entry] = await db.insert(journalEntries).values({
      entryNumber,
      date: invoice.invoiceDate,
      description: `قيد مشتريات - فاتورة رقم ${invoice.id}`,
      referenceType: "PurchaseInvoice",
      referenceId: invoice.id,
    }).returning();
    
    // Calculate totals
    const subtotal = invoice.subtotalAmount;
    const vatAmount = invoice.vatAmount;
    const total = invoice.totalAmount;
    
    const journalLinesData: InsertJournalLine[] = [
      // Debit Inventory
      {
        journalEntryId: entry.id,
        accountCode: ACCOUNT_CODES.INVENTORY,
        accountName: ACCOUNT_NAMES[ACCOUNT_CODES.INVENTORY],
        debit: subtotal,
        credit: 0,
      },
    ];
    
    // Debit VAT if applicable
    if (vatAmount > 0) {
      journalLinesData.push({
        journalEntryId: entry.id,
        accountCode: ACCOUNT_CODES.VAT_INPUT,
        accountName: ACCOUNT_NAMES[ACCOUNT_CODES.VAT_INPUT],
        debit: vatAmount,
        credit: 0,
      });
    }
    
    // Credit Supplier
    journalLinesData.push({
      journalEntryId: entry.id,
      accountCode: ACCOUNT_CODES.SUPPLIER_CONTROL,
      accountName: ACCOUNT_NAMES[ACCOUNT_CODES.SUPPLIER_CONTROL],
      debit: 0,
      credit: total,
      supplierId: invoice.supplierId,
    });
    
    await db.insert(journalLines).values(journalLinesData);
  }
  
  async updatePurchaseInvoice(id: number, invoice: Partial<InsertPurchaseInvoice>): Promise<PurchaseInvoice | undefined> {
    const [updatedInvoice] = await db.update(purchaseInvoices)
      .set({ ...invoice, updatedAt: new Date() })
      .where(eq(purchaseInvoices.id, id))
      .returning();
    return updatedInvoice;
  }
  
  // ========================================
  // PAYMENTS
  // ========================================
  
  async getPayments(supplierId?: number): Promise<(PurchasePayment & { paymentMethod?: PaymentMethod; supplier?: Supplier; invoice?: PurchaseInvoice })[]> {
    let query = db.select().from(purchasePayments);
    if (supplierId) {
      query = query.where(eq(purchasePayments.supplierId, supplierId)) as any;
    }
    
    const payments = await query.orderBy(desc(purchasePayments.createdAt));
    const allMethods = await db.select().from(paymentMethods);
    const allSuppliers = await db.select().from(suppliers);
    const allInvoices = await db.select().from(purchaseInvoices);
    
    return payments.map(payment => ({
      ...payment,
      paymentMethod: allMethods.find(m => m.id === payment.paymentMethodId),
      supplier: allSuppliers.find(s => s.id === payment.supplierId),
      invoice: allInvoices.find(i => i.id === payment.purchaseInvoiceId),
    }));
  }
  
  async createPayment(payment: InsertPurchasePayment): Promise<PurchasePayment> {
    const [newPayment] = await db.insert(purchasePayments).values(payment).returning();
    
    // Update invoice payment status if linked to an invoice
    if (payment.purchaseInvoiceId) {
      const invoice = await this.getPurchaseInvoice(payment.purchaseInvoiceId);
      if (invoice) {
        const totalPaid = invoice.payments.reduce((sum, p) => sum + p.amount, 0) + payment.amount;
        let paymentStatus: "unpaid" | "partially_paid" | "paid" = "unpaid";
        
        if (totalPaid >= invoice.totalAmount) {
          paymentStatus = "paid";
        } else if (totalPaid > 0) {
          paymentStatus = "partially_paid";
        }
        
        await this.updatePurchaseInvoice(payment.purchaseInvoiceId, { paymentStatus });
      }
    }
    
    // Create journal entry for payment
    await this.createPaymentJournalEntry(newPayment);
    
    return newPayment;
  }
  
  private async createPaymentJournalEntry(payment: PurchasePayment): Promise<void> {
    const paymentMethod = await this.getPaymentMethod(payment.paymentMethodId);
    
    // Generate entry number
    const count = await db.select({ count: sql<number>`count(*)` }).from(journalEntries);
    const entryNumber = `JE-${String((count[0]?.count || 0) + 1).padStart(6, '0')}`;
    
    const [entry] = await db.insert(journalEntries).values({
      entryNumber,
      date: payment.paymentDate,
      description: `قيد سداد للمورد - دفعة رقم ${payment.id}`,
      referenceType: "Payment",
      referenceId: payment.id,
    }).returning();
    
    // Determine account based on payment method type
    let accountCode: string = ACCOUNT_CODES.CASH;
    if (paymentMethod?.type === "bank") {
      accountCode = ACCOUNT_CODES.BANK;
    }
    
    const journalLinesData: InsertJournalLine[] = [
      // Debit Supplier (reduce liability)
      {
        journalEntryId: entry.id,
        accountCode: ACCOUNT_CODES.SUPPLIER_CONTROL,
        accountName: ACCOUNT_NAMES[ACCOUNT_CODES.SUPPLIER_CONTROL],
        debit: payment.amount,
        credit: 0,
        supplierId: payment.supplierId,
      },
      // Credit Cash/Bank
      {
        journalEntryId: entry.id,
        accountCode,
        accountName: ACCOUNT_NAMES[accountCode as keyof typeof ACCOUNT_NAMES],
        debit: 0,
        credit: payment.amount,
      },
    ];
    
    await db.insert(journalLines).values(journalLinesData);
  }
  
  // ========================================
  // PURCHASE DEBIT NOTES (إشعارات الخصم للمشتريات)
  // ========================================
  
  async generateDebitNoteNumber(): Promise<string> {
    const count = await db.select({ count: sql<number>`count(*)` }).from(purchaseDebitNotes);
    return `DN-${String((count[0]?.count || 0) + 1).padStart(4, '0')}`;
  }
  
  async getPurchaseDebitNotes(): Promise<PurchaseDebitNoteWithDetails[]> {
    const notes = await db.select().from(purchaseDebitNotes).orderBy(desc(purchaseDebitNotes.createdAt));
    const allSuppliers = await db.select().from(suppliers);
    const allInvoices = await db.select().from(purchaseInvoices);
    const allLines = await db.select().from(purchaseDebitNoteLines);
    const allProducts = await db.select().from(products);
    const allEmployees = await db.select().from(employees);
    
    return notes.map(note => ({
      ...note,
      supplier: allSuppliers.find(s => s.id === note.supplierId),
      purchaseInvoice: allInvoices.find(i => i.id === note.purchaseInvoiceId),
      processedByEmployee: note.processedByEmployeeId ? allEmployees.find(e => e.id === note.processedByEmployeeId) : undefined,
      lines: allLines
        .filter(l => l.debitNoteId === note.id)
        .map(line => ({
          ...line,
          product: allProducts.find(p => p.id === line.productId),
        })),
    }));
  }
  
  async getPurchaseDebitNote(id: number): Promise<PurchaseDebitNoteWithDetails | undefined> {
    const [note] = await db.select().from(purchaseDebitNotes).where(eq(purchaseDebitNotes.id, id));
    if (!note) return undefined;
    
    const [supplier] = await db.select().from(suppliers).where(eq(suppliers.id, note.supplierId));
    const [invoice] = await db.select().from(purchaseInvoices).where(eq(purchaseInvoices.id, note.purchaseInvoiceId));
    
    let processedByEmployee: Employee | undefined;
    if (note.processedByEmployeeId) {
      const [emp] = await db.select().from(employees).where(eq(employees.id, note.processedByEmployeeId));
      processedByEmployee = emp;
    }
    
    const lines = await db.select().from(purchaseDebitNoteLines).where(eq(purchaseDebitNoteLines.debitNoteId, id));
    const linesWithProducts = await Promise.all(lines.map(async (line) => {
      const [product] = await db.select().from(products).where(eq(products.id, line.productId));
      return { ...line, product };
    }));
    
    return {
      ...note,
      supplier,
      purchaseInvoice: invoice,
      processedByEmployee,
      lines: linesWithProducts,
    };
  }
  
  async createPurchaseDebitNote(
    debitNote: InsertPurchaseDebitNote, 
    lines: Omit<InsertPurchaseDebitNoteLine, 'debitNoteId'>[]
  ): Promise<PurchaseDebitNoteWithDetails> {
    // Generate debit note number
    const debitNoteNumber = await this.generateDebitNoteNumber();
    
    const [newNote] = await db.insert(purchaseDebitNotes).values({
      ...debitNote,
      debitNoteNumber,
    }).returning();
    
    // Insert lines
    const linesWithNoteId = lines.map(line => ({
      ...line,
      debitNoteId: newNote.id,
    }));
    
    await db.insert(purchaseDebitNoteLines).values(linesWithNoteId);
    
    return this.getPurchaseDebitNote(newNote.id) as Promise<PurchaseDebitNoteWithDetails>;
  }
  
  async confirmPurchaseDebitNote(id: number): Promise<PurchaseDebitNoteWithDetails | undefined> {
    const note = await this.getPurchaseDebitNote(id);
    if (!note || note.status !== 'draft') return undefined;
    
    // Update status to confirmed
    await db.update(purchaseDebitNotes)
      .set({ status: 'confirmed', updatedAt: new Date() })
      .where(eq(purchaseDebitNotes.id, id));
    
    // Create reverse stock movements for each line
    for (const line of note.lines) {
      // Find the original stock batch and reduce quantity
      const batches = await db.select().from(stockBatches)
        .where(eq(stockBatches.productId, line.productId))
        .orderBy(asc(stockBatches.expiryDate));
      
      // Create stock movement for the return (quantity out)
      await db.insert(stockMovements).values({
        productId: line.productId,
        movementDate: note.debitNoteDate,
        movementType: "purchase_return",
        quantityIn: 0,
        quantityOut: line.quantityReturned,
        unitCost: line.unitPrice,
        referenceType: "PurchaseDebitNote",
        referenceId: id,
      });
      
      // Reduce quantity from the first available batch
      if (batches.length > 0) {
        const batch = batches[0];
        const newQuantity = Math.max(0, batch.quantityOnHand - line.quantityReturned);
        await db.update(stockBatches)
          .set({ quantityOnHand: newQuantity })
          .where(eq(stockBatches.id, batch.id));
      }
    }
    
    // Create journal entry to reverse the purchase
    await this.createDebitNoteJournalEntry(note);
    
    return this.getPurchaseDebitNote(id);
  }
  
  private async createDebitNoteJournalEntry(note: PurchaseDebitNoteWithDetails): Promise<void> {
    // Generate entry number
    const count = await db.select({ count: sql<number>`count(*)` }).from(journalEntries);
    const entryNumber = `JE-${String((count[0]?.count || 0) + 1).padStart(6, '0')}`;
    
    const [entry] = await db.insert(journalEntries).values({
      entryNumber,
      date: note.debitNoteDate,
      description: `قيد إشعار خصم للمشتريات - ${note.debitNoteNumber}`,
      referenceType: "PurchaseDebitNote",
      referenceId: note.id,
    }).returning();
    
    const journalLinesData: InsertJournalLine[] = [
      // Debit Supplier (reduce payable)
      {
        journalEntryId: entry.id,
        accountCode: ACCOUNT_CODES.SUPPLIER_CONTROL,
        accountName: ACCOUNT_NAMES[ACCOUNT_CODES.SUPPLIER_CONTROL],
        debit: note.totalAmount,
        credit: 0,
        supplierId: note.supplierId,
      },
      // Credit Inventory (reduce inventory value)
      {
        journalEntryId: entry.id,
        accountCode: ACCOUNT_CODES.INVENTORY,
        accountName: ACCOUNT_NAMES[ACCOUNT_CODES.INVENTORY],
        debit: 0,
        credit: note.subtotalAmount,
      },
    ];
    
    // Credit VAT Input (if applicable)
    if (note.vatAmount > 0) {
      journalLinesData.push({
        journalEntryId: entry.id,
        accountCode: ACCOUNT_CODES.VAT_INPUT,
        accountName: ACCOUNT_NAMES[ACCOUNT_CODES.VAT_INPUT],
        debit: 0,
        credit: note.vatAmount,
      });
    }
    
    await db.insert(journalLines).values(journalLinesData);
  }
  
  async cancelPurchaseDebitNote(id: number): Promise<PurchaseDebitNoteWithDetails | undefined> {
    const note = await this.getPurchaseDebitNote(id);
    if (!note || note.status !== 'draft') return undefined;
    
    await db.update(purchaseDebitNotes)
      .set({ status: 'cancelled', updatedAt: new Date() })
      .where(eq(purchaseDebitNotes.id, id));
    
    return this.getPurchaseDebitNote(id);
  }
  
  // ========================================
  // JOURNAL ENTRIES
  // ========================================
  
  async getJournalEntries(): Promise<JournalEntryWithLines[]> {
    const entries = await db.select().from(journalEntries).orderBy(desc(journalEntries.createdAt));
    const allLines = await db.select().from(journalLines);
    
    return entries.map(entry => ({
      ...entry,
      lines: allLines.filter(l => l.journalEntryId === entry.id),
    }));
  }
  
  async getJournalEntry(id: number): Promise<JournalEntryWithLines | undefined> {
    const [entry] = await db.select().from(journalEntries).where(eq(journalEntries.id, id));
    if (!entry) return undefined;
    
    const lines = await db.select().from(journalLines).where(eq(journalLines.journalEntryId, id));
    
    return {
      ...entry,
      lines,
    };
  }
  
  async createJournalEntry(entry: InsertJournalEntry, lines: InsertJournalLine[]): Promise<JournalEntryWithLines> {
    const [newEntry] = await db.insert(journalEntries).values(entry).returning();
    
    const linesWithEntryId = lines.map(line => ({
      ...line,
      journalEntryId: newEntry.id,
    }));
    
    const newLines = await db.insert(journalLines).values(linesWithEntryId).returning();
    
    return {
      ...newEntry,
      lines: newLines,
    };
  }
  
  // ========================================
  // DASHBOARD & REPORTS
  // ========================================
  
  async getDashboardStats(): Promise<DashboardStats> {
    const allSuppliers = await db.select().from(suppliers);
    const allProducts = await db.select().from(products);
    const allBatches = await db.select().from(stockBatches);
    const unpaidInvoices = await db.select().from(purchaseInvoices)
      .where(ne(purchaseInvoices.paymentStatus, "paid"));
    
    const totalInventoryValue = allBatches.reduce((sum, batch) => sum + (batch.quantityOnHand * batch.unitCost), 0);
    const totalUnpaidAmount = unpaidInvoices.reduce((sum, inv) => sum + inv.totalAmount, 0);
    
    // Get recent invoices
    const recentInvoices = await this.getPurchaseInvoices();
    
    // Get low stock products (quantity < 10)
    const productBatches = allBatches.reduce((acc, batch) => {
      if (!acc[batch.productId]) acc[batch.productId] = 0;
      acc[batch.productId] += batch.quantityOnHand;
      return acc;
    }, {} as Record<number, number>);
    
    const lowStockProducts = allProducts
      .filter(p => p.isActive && (productBatches[p.id] || 0) < 10)
      .map(p => ({ ...p, quantityOnHand: productBatches[p.id] || 0 }));
    
    // Get expiring products (within 30 days)
    const thirtyDaysFromNow = new Date();
    thirtyDaysFromNow.setDate(thirtyDaysFromNow.getDate() + 30);
    
    const expiringBatches = allBatches
      .filter(b => b.expiryDate && new Date(b.expiryDate) <= thirtyDaysFromNow && b.quantityOnHand > 0)
      .map(b => ({ ...b, product: allProducts.find(p => p.id === b.productId) }));
    
    return {
      totalSuppliers: allSuppliers.length,
      activeSuppliers: allSuppliers.filter(s => s.isActive).length,
      totalProducts: allProducts.length,
      activeProducts: allProducts.filter(p => p.isActive).length,
      totalInventoryValue,
      totalUnpaidInvoices: unpaidInvoices.length,
      totalUnpaidAmount,
      recentInvoices: recentInvoices.slice(0, 5),
      lowStockProducts: lowStockProducts.slice(0, 5),
      expiringProducts: expiringBatches.slice(0, 5),
    };
  }
  
  async getSalonDashboardSummary(filters?: SalonDashboardFilters): Promise<SalonDashboardSummary> {
    const today = new Date();
    const todayStart = new Date(today.getFullYear(), today.getMonth(), today.getDate());
    const todayEnd = new Date(today.getFullYear(), today.getMonth(), today.getDate(), 23, 59, 59);
    
    const fromDate = filters?.fromDate ? new Date(filters.fromDate) : new Date(today.getFullYear(), today.getMonth(), 1);
    const toDate = filters?.toDate ? new Date(filters.toDate) : today;
    toDate.setHours(23, 59, 59);
    
    const thirtyDaysAgo = new Date();
    thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);
    
    // Build source filter
    const sourceFilter = filters?.source && filters.source !== "all" ? filters.source : null;
    
    // ======== SALES TODAY ========
    let todayOrdersQuery = db.select().from(orders)
      .where(and(
        gte(orders.orderDate, todayStart),
        lte(orders.orderDate, todayEnd),
        eq(orders.status, "completed")
      ));
    
    const todayOrders = await todayOrdersQuery;
    const filteredTodayOrders = sourceFilter ? todayOrders.filter(o => o.source === sourceFilter) : todayOrders;
    
    const salesToday = {
      totalAmount: filteredTodayOrders.reduce((sum, o) => sum + o.totalAmount, 0),
      ordersCount: filteredTodayOrders.length,
      avgOrderValue: filteredTodayOrders.length > 0 
        ? filteredTodayOrders.reduce((sum, o) => sum + o.totalAmount, 0) / filteredTodayOrders.length 
        : 0,
    };
    
    // ======== BOOKINGS TODAY ========
    const todayAppointments = await db.select().from(appointments)
      .where(and(
        gte(appointments.scheduledAt, todayStart),
        lte(appointments.scheduledAt, todayEnd)
      ));
    
    const now = new Date();
    const bookingsToday = {
      total: todayAppointments.length,
      cancelled: todayAppointments.filter(a => a.status === "canceled").length,
      upcoming: todayAppointments.filter(a => 
        new Date(a.scheduledAt) > now && 
        !["canceled", "completed", "no_show"].includes(a.status)
      ).length,
    };
    
    // ======== CANCELLED ORDERS (Period) ========
    const cancelledOrdersData = await db.select().from(orders)
      .where(and(
        gte(orders.orderDate, fromDate),
        lte(orders.orderDate, toDate),
        eq(orders.status, "canceled")
      ));
    
    const filteredCancelled = sourceFilter ? cancelledOrdersData.filter(o => o.source === sourceFilter) : cancelledOrdersData;
    
    const cancelledOrders = {
      count: filteredCancelled.length,
      lostAmount: filteredCancelled.reduce((sum, o) => sum + o.totalAmount, 0),
    };
    
    // ======== CUSTOMERS ========
    const allCustomers = await db.select().from(customers);
    const newCustomers = allCustomers.filter(c => c.createdAt && new Date(c.createdAt) >= thirtyDaysAgo);
    
    const customersData = {
      totalCustomers: allCustomers.length,
      newLast30Days: newCustomers.length,
    };
    
    // ======== TOP SERVICES (Period) ========
    const periodOrders = await db.select().from(orders)
      .where(and(
        gte(orders.orderDate, fromDate),
        lte(orders.orderDate, toDate),
        eq(orders.status, "completed")
      ));
    
    const filteredPeriodOrders = sourceFilter ? periodOrders.filter(o => o.source === sourceFilter) : periodOrders;
    const periodOrderIds = filteredPeriodOrders.map(o => o.id);
    
    const allOrderServices = periodOrderIds.length > 0 
      ? await db.select().from(orderServices).where(sql`${orderServices.orderId} IN (${sql.join(periodOrderIds.map(id => sql`${id}`), sql`, `)})`)
      : [];
    
    const allServices = await db.select().from(services);
    
    const serviceStats = allOrderServices.reduce((acc, os) => {
      if (!acc[os.serviceId]) {
        acc[os.serviceId] = { count: 0, totalAmount: 0 };
      }
      acc[os.serviceId].count += os.quantity;
      acc[os.serviceId].totalAmount += os.lineTotal;
      return acc;
    }, {} as Record<number, { count: number; totalAmount: number }>);
    
    const topServices = Object.entries(serviceStats)
      .map(([serviceId, stats]) => {
        const service = allServices.find(s => s.id === parseInt(serviceId));
        return {
          serviceId: parseInt(serviceId),
          nameAr: service?.nameAr || `خدمة ${serviceId}`,
          count: stats.count,
          totalAmount: stats.totalAmount,
        };
      })
      .sort((a, b) => b.totalAmount - a.totalAmount)
      .slice(0, 5);
    
    // ======== TOP EMPLOYEES (Period) ========
    const allEmployees = await db.select().from(employees);
    
    const employeeStats = allOrderServices.reduce((acc, os) => {
      if (os.executingEmployeeId) {
        if (!acc[os.executingEmployeeId]) {
          acc[os.executingEmployeeId] = { servicesCount: 0, totalSales: 0 };
        }
        acc[os.executingEmployeeId].servicesCount += os.quantity;
        acc[os.executingEmployeeId].totalSales += os.lineTotal;
      }
      return acc;
    }, {} as Record<number, { servicesCount: number; totalSales: number }>);
    
    const topEmployees = Object.entries(employeeStats)
      .map(([empId, stats]) => {
        const emp = allEmployees.find(e => e.id === parseInt(empId));
        return {
          employeeId: parseInt(empId),
          name: emp?.fullNameAr || emp?.nameAr || `موظف ${empId}`,
          servicesCount: stats.servicesCount,
          totalSales: stats.totalSales,
        };
      })
      .sort((a, b) => b.totalSales - a.totalSales)
      .slice(0, 5);
    
    // ======== INVENTORY SUMMARY ========
    const allProducts = await db.select().from(products).where(eq(products.isActive, true));
    const allBatches = await db.select().from(stockBatches);
    
    const productBatches = allBatches.reduce((acc, batch) => {
      if (!acc[batch.productId]) acc[batch.productId] = 0;
      acc[batch.productId] += batch.quantityOnHand;
      return acc;
    }, {} as Record<number, number>);
    
    const lowStockCount = allProducts.filter(p => (productBatches[p.id] || 0) < 10).length;
    
    const thirtyDaysFromNow = new Date();
    thirtyDaysFromNow.setDate(thirtyDaysFromNow.getDate() + 30);
    
    const nearExpiryCount = allBatches.filter(b => 
      b.expiryDate && 
      new Date(b.expiryDate) <= thirtyDaysFromNow && 
      b.quantityOnHand > 0
    ).length;
    
    const estimatedValue = allBatches.reduce((sum, b) => sum + (b.quantityOnHand * b.unitCost), 0);
    
    const inventory = {
      lowStockCount,
      nearExpiryCount,
      estimatedValue,
    };
    
    // ======== SALES OVER TIME (Last 30 days or period) ========
    const salesDates: Record<string, number> = {};
    
    filteredPeriodOrders.forEach(order => {
      const dateStr = new Date(order.orderDate).toISOString().split('T')[0];
      if (!salesDates[dateStr]) salesDates[dateStr] = 0;
      salesDates[dateStr] += order.totalAmount;
    });
    
    const salesOverTime = Object.entries(salesDates)
      .map(([date, amount]) => ({ date, amount }))
      .sort((a, b) => a.date.localeCompare(b.date));
    
    // ======== ORDERS STATUS DISTRIBUTION (Period) ========
    const periodAllOrders = await db.select().from(orders)
      .where(and(
        gte(orders.orderDate, fromDate),
        lte(orders.orderDate, toDate)
      ));
    
    const filteredAllPeriodOrders = sourceFilter ? periodAllOrders.filter(o => o.source === sourceFilter) : periodAllOrders;
    
    const ordersStatusDistribution = {
      completed: filteredAllPeriodOrders.filter(o => o.status === "completed").length,
      inProgress: filteredAllPeriodOrders.filter(o => o.status === "in_progress").length,
      cancelled: filteredAllPeriodOrders.filter(o => o.status === "canceled").length,
      scheduled: filteredAllPeriodOrders.filter(o => o.status === "scheduled" || o.status === "postponed").length,
    };
    
    // ======== LATEST ORDERS ========
    const latestOrdersRaw = await db.select().from(orders)
      .orderBy(desc(orders.orderDate))
      .limit(10);
    
    const latestOrders = await Promise.all(latestOrdersRaw.map(async o => {
      let customerName = "زائر";
      if (o.clientId) {
        const customer = allCustomers.find(c => c.id === o.clientId);
        if (customer) customerName = customer.nameAr;
      }
      
      const statusLabels: Record<string, string> = {
        new: "جديد",
        in_progress: "قيد التنفيذ",
        completed: "مكتمل",
        canceled: "ملغي",
        scheduled: "مجدول",
        postponed: "مؤجل",
        returned: "مرتجع",
      };
      
      return {
        id: o.id,
        orderNumber: o.orderNumber,
        date: o.orderDate.toISOString(),
        customer: customerName,
        status: statusLabels[o.status] || o.status,
        total: o.totalAmount,
      };
    }));
    
    // ======== SALES REVENUE (Period) ========
    const salesRevenue = {
      totalSales: filteredPeriodOrders.reduce((sum, o) => sum + o.totalAmount, 0),
      vatCollected: filteredPeriodOrders.reduce((sum, o) => sum + o.vatAmount, 0),
      totalDiscounts: filteredPeriodOrders.reduce((sum, o) => sum + (o.couponDiscountAmount || 0), 0),
    };
    
    // ======== TOP CUSTOMERS (Period) ========
    const customerOrderStats = filteredPeriodOrders.reduce((acc, o) => {
      if (o.clientId) {
        if (!acc[o.clientId]) {
          acc[o.clientId] = { totalSpent: 0, ordersCount: 0 };
        }
        acc[o.clientId].totalSpent += o.totalAmount;
        acc[o.clientId].ordersCount += 1;
      }
      return acc;
    }, {} as Record<number, { totalSpent: number; ordersCount: number }>);
    
    const topCustomers = Object.entries(customerOrderStats)
      .map(([customerId, stats]) => {
        const customer = allCustomers.find(c => c.id === parseInt(customerId));
        return {
          customerId: parseInt(customerId),
          name: customer?.nameAr || `عميل ${customerId}`,
          totalSpent: stats.totalSpent,
          ordersCount: stats.ordersCount,
        };
      })
      .sort((a, b) => b.totalSpent - a.totalSpent)
      .slice(0, 5);
    
    return {
      salesToday,
      bookingsToday,
      cancelledOrders,
      customers: customersData,
      topServices,
      topEmployees,
      inventory,
      salesOverTime,
      ordersStatusDistribution,
      latestOrders,
      salesRevenue,
      topCustomers,
    };
  }
  
  async getInventorySummary(): Promise<InventorySummaryItem[]> {
    const allProducts = await db.select().from(products).where(eq(products.isActive, true));
    const allBatches = await db.select().from(stockBatches);
    const allUnits = await db.select().from(units);
    const allBrands = await db.select().from(brands);
    
    return allProducts.map(product => {
      const productBatches = allBatches.filter(b => b.productId === product.id);
      const quantityOnHand = productBatches.reduce((sum, b) => sum + b.quantityOnHand, 0);
      const totalCost = productBatches.reduce((sum, b) => sum + (b.quantityOnHand * b.unitCost), 0);
      const sellUnit = allUnits.find(u => u.id === product.sellUnitId);
      const brand = allBrands.find(b => b.id === product.brandId);
      
      return {
        productId: product.id,
        productNameAr: product.nameAr,
        sku: product.sku || undefined,
        quantityOnHand,
        totalCost,
        sellUnitNameAr: sellUnit?.nameAr,
        usageType: product.usageType,
        brandNameAr: brand?.nameAr,
      };
    });
  }
  
  async getSupplierBalances(): Promise<SupplierBalance[]> {
    const allSuppliers = await db.select().from(suppliers).where(eq(suppliers.isActive, true));
    const allJournalLines = await db.select().from(journalLines)
      .where(eq(journalLines.accountCode, ACCOUNT_CODES.SUPPLIER_CONTROL));
    
    return allSuppliers.map(supplier => {
      const supplierLines = allJournalLines.filter(l => l.supplierId === supplier.id);
      const totalCredits = supplierLines.reduce((sum, l) => sum + l.credit, 0);
      const totalDebits = supplierLines.reduce((sum, l) => sum + l.debit, 0);
      
      return {
        supplierId: supplier.id,
        supplierNameAr: supplier.nameAr,
        totalCredits,
        totalDebits,
        balance: totalCredits - totalDebits, // positive = we owe supplier
      };
    });
  }
  
  // ========================================
  // CHART OF ACCOUNTS
  // ========================================
  
  async getAccounts(): Promise<Account[]> {
    return db.select().from(accounts).orderBy(asc(accounts.code));
  }
  
  async getAccountsTree(): Promise<AccountWithChildren[]> {
    const allAccounts = await this.getAccounts();
    
    const buildTree = (parentId: number | null): AccountWithChildren[] => {
      return allAccounts
        .filter(acc => acc.parentId === parentId)
        .map(acc => ({
          ...acc,
          children: buildTree(acc.id),
        }));
    };
    
    return buildTree(null);
  }
  
  async getAccount(id: number): Promise<Account | undefined> {
    const [account] = await db.select().from(accounts).where(eq(accounts.id, id));
    return account;
  }
  
  async getAccountByCode(code: string): Promise<Account | undefined> {
    const [account] = await db.select().from(accounts).where(eq(accounts.code, code));
    return account;
  }
  
  async createAccount(account: InsertAccount): Promise<Account> {
    // Calculate level based on parent
    let level = 1;
    if (account.parentId) {
      const parent = await this.getAccount(account.parentId);
      if (parent) {
        level = parent.level + 1;
      }
    }
    
    // Determine normal balance based on account type
    let normalBalance = "debit";
    if (account.accountType === "liability" || account.accountType === "equity" || account.accountType === "revenue") {
      normalBalance = "credit";
    }
    
    const [newAccount] = await db.insert(accounts).values({
      ...account,
      level,
      normalBalance: account.normalBalance || normalBalance,
    }).returning();
    return newAccount;
  }
  
  async updateAccount(id: number, account: Partial<InsertAccount>): Promise<Account | undefined> {
    const [updatedAccount] = await db.update(accounts)
      .set({ ...account, updatedAt: new Date() })
      .where(eq(accounts.id, id))
      .returning();
    return updatedAccount;
  }
  
  async deleteAccount(id: number): Promise<boolean> {
    // Check if account has children or journal lines
    const children = await db.select().from(accounts).where(eq(accounts.parentId, id));
    if (children.length > 0) {
      return false; // Cannot delete account with children
    }
    
    const lines = await db.select().from(journalLines).where(eq(journalLines.accountCode, 
      (await this.getAccount(id))?.code || ""));
    if (lines.length > 0) {
      // Soft delete
      const [result] = await db.update(accounts)
        .set({ isActive: false, updatedAt: new Date() })
        .where(eq(accounts.id, id))
        .returning();
      return !!result;
    }
    
    // Hard delete if no journal lines
    const [result] = await db.delete(accounts).where(eq(accounts.id, id)).returning();
    return !!result;
  }
  
  // ========================================
  // SERVICE CATEGORIES
  // ========================================
  
  async getServiceCategories(): Promise<ServiceCategoryWithCount[]> {
    const allCategories = await db.select().from(serviceCategories).orderBy(asc(serviceCategories.nameAr));
    const allServices = await db.select().from(services).where(eq(services.isActive, true));
    
    return allCategories.map(cat => ({
      ...cat,
      servicesCount: allServices.filter(s => s.categoryId === cat.id).length,
    }));
  }
  
  async getServiceCategoryWithServices(id: number): Promise<ServiceCategoryWithServices | undefined> {
    const [category] = await db.select().from(serviceCategories).where(eq(serviceCategories.id, id));
    if (!category) return undefined;
    
    const categoryServices = await db.select().from(services).where(eq(services.categoryId, id));
    
    return {
      ...category,
      services: categoryServices,
    };
  }
  
  async getServiceCategory(id: number): Promise<ServiceCategory | undefined> {
    const [category] = await db.select().from(serviceCategories).where(eq(serviceCategories.id, id));
    return category;
  }
  
  async createServiceCategory(category: InsertServiceCategory): Promise<ServiceCategory> {
    const [newCategory] = await db.insert(serviceCategories).values(category).returning();
    return newCategory;
  }
  
  async updateServiceCategory(id: number, category: Partial<InsertServiceCategory>): Promise<ServiceCategory | undefined> {
    const [updatedCategory] = await db.update(serviceCategories)
      .set({ ...category, updatedAt: new Date() })
      .where(eq(serviceCategories.id, id))
      .returning();
    return updatedCategory;
  }
  
  async deleteServiceCategory(id: number): Promise<boolean> {
    // Soft delete by setting isActive to false
    const [result] = await db.update(serviceCategories)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(serviceCategories.id, id))
      .returning();
    return !!result;
  }
  
  // ========================================
  // SERVICES
  // ========================================
  
  async getServices(filters?: { categoryId?: number; isActive?: boolean }): Promise<ServiceWithDetails[]> {
    let query = db.select().from(services);
    
    // Apply filters
    const conditions: any[] = [];
    if (filters?.categoryId) {
      conditions.push(eq(services.categoryId, filters.categoryId));
    }
    if (filters?.isActive !== undefined) {
      conditions.push(eq(services.isActive, filters.isActive));
    }
    
    if (conditions.length > 0) {
      query = query.where(and(...conditions)) as any;
    }
    
    const allServices = await query.orderBy(asc(services.nameAr));
    const allCategories = await db.select().from(serviceCategories);
    const allServiceProducts = await db.select().from(serviceProducts);
    const allProducts = await db.select().from(products);
    const allServiceImages = await db.select().from(serviceImages).orderBy(asc(serviceImages.displayOrder));
    
    return allServices.map(service => ({
      ...service,
      category: allCategories.find(c => c.id === service.categoryId),
      relatedProducts: allServiceProducts
        .filter(sp => sp.serviceId === service.id)
        .map(sp => ({
          ...sp,
          product: allProducts.find(p => p.id === sp.productId),
        })),
      images: allServiceImages.filter(img => img.serviceId === service.id),
    }));
  }
  
  async getService(id: number): Promise<ServiceWithDetails | undefined> {
    const [service] = await db.select().from(services).where(eq(services.id, id));
    if (!service) return undefined;
    
    const [category] = await db.select().from(serviceCategories).where(eq(serviceCategories.id, service.categoryId));
    const relatedProducts = await this.getServiceProducts(id);
    const images = await this.getServiceImages(id);
    
    return {
      ...service,
      category,
      relatedProducts,
      images,
    };
  }
  
  async createService(
    service: InsertService, 
    relatedProducts?: { productId: number; quantityPerService: number; deductFromCommission: boolean; notes?: string }[]
  ): Promise<ServiceWithDetails> {
    const [newService] = await db.insert(services).values(service).returning();
    
    // Insert related products
    if (relatedProducts && relatedProducts.length > 0) {
      await db.insert(serviceProducts).values(
        relatedProducts.map(rp => ({
          serviceId: newService.id,
          productId: rp.productId,
          quantityPerService: rp.quantityPerService,
          deductFromCommission: rp.deductFromCommission || false,
          notes: rp.notes,
        }))
      );
    }
    
    return this.getService(newService.id) as Promise<ServiceWithDetails>;
  }
  
  async updateService(
    id: number, 
    service: Partial<InsertService>, 
    relatedProducts?: { productId: number; quantityPerService: number; deductFromCommission: boolean; notes?: string }[]
  ): Promise<ServiceWithDetails | undefined> {
    const [updatedService] = await db.update(services)
      .set({ ...service, updatedAt: new Date() })
      .where(eq(services.id, id))
      .returning();
    
    if (!updatedService) return undefined;
    
    // Replace related products if provided
    if (relatedProducts !== undefined) {
      // Delete existing related products
      await db.delete(serviceProducts).where(eq(serviceProducts.serviceId, id));
      
      // Insert new related products
      if (relatedProducts.length > 0) {
        await db.insert(serviceProducts).values(
          relatedProducts.map(rp => ({
            serviceId: id,
            productId: rp.productId,
            quantityPerService: rp.quantityPerService,
            deductFromCommission: rp.deductFromCommission || false,
            notes: rp.notes,
          }))
        );
      }
    }
    
    return this.getService(id);
  }
  
  async deleteService(id: number): Promise<boolean> {
    // Soft delete by setting isActive to false
    const [result] = await db.update(services)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(services.id, id))
      .returning();
    return !!result;
  }
  
  // ========================================
  // SERVICE PRODUCTS
  // ========================================
  
  async getServiceProducts(serviceId: number): Promise<ServiceProductWithProduct[]> {
    const svcProducts = await db.select().from(serviceProducts).where(eq(serviceProducts.serviceId, serviceId));
    const allProducts = await db.select().from(products);
    
    return svcProducts.map(sp => ({
      ...sp,
      product: allProducts.find(p => p.id === sp.productId),
    }));
  }
  
  // ========================================
  // SERVICE IMAGES
  // ========================================
  
  async getServiceImages(serviceId: number): Promise<ServiceImage[]> {
    return db.select()
      .from(serviceImages)
      .where(eq(serviceImages.serviceId, serviceId))
      .orderBy(asc(serviceImages.displayOrder));
  }
  
  async addServiceImage(image: InsertServiceImage): Promise<ServiceImage> {
    const [newImage] = await db.insert(serviceImages).values(image).returning();
    return newImage;
  }
  
  async deleteServiceImage(id: number): Promise<boolean> {
    const [result] = await db.delete(serviceImages)
      .where(eq(serviceImages.id, id))
      .returning();
    return !!result;
  }
  
  async updateServiceImagesOrder(serviceId: number, imageIds: number[]): Promise<void> {
    for (let i = 0; i < imageIds.length; i++) {
      await db.update(serviceImages)
        .set({ displayOrder: i })
        .where(and(eq(serviceImages.id, imageIds[i]), eq(serviceImages.serviceId, serviceId)));
    }
  }
  
  // ========================================
  // CUSTOMERS
  // ========================================
  
  async getCustomers(): Promise<Customer[]> {
    return db.select().from(customers).orderBy(desc(customers.createdAt));
  }
  
  async getCustomer(id: number): Promise<Customer | undefined> {
    const [customer] = await db.select().from(customers).where(eq(customers.id, id));
    return customer;
  }
  
  async getCustomerByPhone(phone: string): Promise<Customer | undefined> {
    const trimmedPhone = phone?.trim();
    if (!trimmedPhone) return undefined;
    const [customer] = await db.select().from(customers).where(eq(customers.phone, trimmedPhone));
    return customer;
  }
  
  async createCustomer(customer: InsertCustomer): Promise<Customer> {
    const [newCustomer] = await db.insert(customers).values(customer).returning();
    return newCustomer;
  }
  
  async updateCustomer(id: number, customer: Partial<InsertCustomer>): Promise<Customer | undefined> {
    const [updatedCustomer] = await db.update(customers)
      .set({ ...customer, updatedAt: new Date() })
      .where(eq(customers.id, id))
      .returning();
    return updatedCustomer;
  }
  
  async deleteCustomer(id: number): Promise<boolean> {
    const [result] = await db.update(customers)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(customers.id, id))
      .returning();
    return !!result;
  }

  async searchCustomers(query: string): Promise<Customer[]> {
    const allCustomers = await db.select().from(customers).where(eq(customers.isActive, true));
    const lowerQuery = query.toLowerCase();
    return allCustomers.filter(c => 
      c.nameAr?.toLowerCase().includes(lowerQuery) ||
      c.phone?.toLowerCase().includes(lowerQuery) ||
      c.email?.toLowerCase().includes(lowerQuery)
    );
  }

  async getCustomerOrders(customerId: number): Promise<OrderWithDetails[]> {
    const customerOrders = await db.select().from(orders).where(eq(orders.clientId, customerId)).orderBy(desc(orders.createdAt));
    const allOrderServices = await db.select().from(orderServices);
    const allOrderProducts = await db.select().from(orderProducts);
    const allServices = await db.select().from(services);
    const allServiceCategories = await db.select().from(serviceCategories);
    const allProducts = await db.select().from(products);
    const allRatings = await db.select().from(orderRatings);
    const allEmployees = await db.select().from(employees);

    return customerOrders.map(order => {
      const svc = allOrderServices.filter(s => s.orderId === order.id);
      const prd = allOrderProducts.filter(p => p.orderId === order.id);
      const rating = allRatings.find(r => r.orderId === order.id);
      const createdByEmployee = allEmployees.find(e => e.id === order.createdByEmployeeId);

      return {
        ...order,
        createdByEmployee,
        services: svc.map(s => {
          const service = allServices.find(sv => sv.id === s.serviceId);
          const category = service ? allServiceCategories.find(c => c.id === service.categoryId) : undefined;
          const executingEmployee = s.executingEmployeeId ? allEmployees.find(e => e.id === s.executingEmployeeId) : undefined;
          return {
            ...s,
            service: service ? { ...service, category } : undefined,
            executingEmployee,
          };
        }),
        products: prd.map(p => ({
          ...p,
          product: allProducts.find(pr => pr.id === p.productId),
        })),
        rating,
      };
    });
  }

  async getCustomerRatings(customerId: number): Promise<(OrderRating & { order?: OrderWithDetails })[]> {
    const ratings = await db.select().from(orderRatings).where(eq(orderRatings.clientId, customerId)).orderBy(desc(orderRatings.createdAt));
    const customerOrders = await this.getCustomerOrders(customerId);
    
    return ratings.map(r => ({
      ...r,
      order: customerOrders.find(o => o.id === r.orderId),
    }));
  }
  
  // ========================================
  // LOYALTY POINTS
  // ========================================

  async getLoyaltySettings(): Promise<LoyaltySettings | undefined> {
    const [settings] = await db.select().from(loyaltySettings).limit(1);
    return settings;
  }

  async updateLoyaltySettings(settings: Partial<InsertLoyaltySettings>): Promise<LoyaltySettings> {
    const existing = await this.getLoyaltySettings();
    if (existing) {
      const [updated] = await db.update(loyaltySettings)
        .set({ ...settings, updatedAt: new Date() })
        .where(eq(loyaltySettings.id, existing.id))
        .returning();
      return updated;
    } else {
      const [created] = await db.insert(loyaltySettings).values(settings as InsertLoyaltySettings).returning();
      return created;
    }
  }

  async getCustomerLoyaltyHistory(customerId: number): Promise<CustomerLoyaltyPoints[]> {
    return db.select().from(customerLoyaltyPoints)
      .where(eq(customerLoyaltyPoints.customerId, customerId))
      .orderBy(desc(customerLoyaltyPoints.createdAt));
  }

  async addLoyaltyPoints(customerId: number, points: number, transactionType: string, referenceType?: string, referenceId?: number, notes?: string): Promise<CustomerLoyaltyPoints> {
    const customer = await this.getCustomer(customerId);
    if (!customer) throw new Error("Customer not found");

    const settings = await this.getLoyaltySettings();
    const expiryMonths = settings?.expiryMonths || 12;
    const expiresAt = transactionType === 'earn' ? new Date(Date.now() + expiryMonths * 30 * 24 * 60 * 60 * 1000) : null;

    const newBalance = customer.loyaltyPointsBalance + points;

    await db.update(customers)
      .set({ loyaltyPointsBalance: newBalance, updatedAt: new Date() })
      .where(eq(customers.id, customerId));

    const [transaction] = await db.insert(customerLoyaltyPoints).values({
      customerId,
      transactionType,
      points,
      balanceAfter: newBalance,
      referenceType,
      referenceId,
      notes,
      expiresAt,
    }).returning();

    return transaction;
  }

  async redeemLoyaltyPoints(customerId: number, points: number, notes?: string): Promise<CustomerLoyaltyPoints> {
    const customer = await this.getCustomer(customerId);
    if (!customer) throw new Error("Customer not found");
    if (customer.loyaltyPointsBalance < points) throw new Error("Insufficient points");

    const settings = await this.getLoyaltySettings();
    if (settings && points < settings.minRedeemPoints) {
      throw new Error(`Minimum redeemable points is ${settings.minRedeemPoints}`);
    }

    return this.addLoyaltyPoints(customerId, -points, 'redeem', 'manual', undefined, notes);
  }

  async getCustomerSummary(customerId: number): Promise<{
    customer: Customer;
    ordersCount: number;
    totalSpent: number;
    ratingsCount: number;
    averageRating: number;
    loyaltyPoints: number;
    scheduledOrdersCount: number;
  }> {
    const customer = await this.getCustomer(customerId);
    if (!customer) throw new Error("Customer not found");

    const customerOrders = await db.select().from(orders).where(eq(orders.clientId, customerId));
    const completedOrders = customerOrders.filter(o => o.status === 'completed');
    const totalSpent = completedOrders.reduce((sum, o) => sum + o.totalAmount, 0);
    const scheduledOrders = customerOrders.filter(o => o.status === 'scheduled');

    const ratings = await db.select().from(orderRatings).where(eq(orderRatings.clientId, customerId));
    const averageRating = ratings.length > 0 ? ratings.reduce((sum, r) => sum + r.rating, 0) / ratings.length : 0;

    return {
      customer,
      ordersCount: customerOrders.length,
      totalSpent,
      ratingsCount: ratings.length,
      averageRating,
      loyaltyPoints: customer.loyaltyPointsBalance,
      scheduledOrdersCount: scheduledOrders.length,
    };
  }

  // Get all loyalty transactions with filters
  async getAllLoyaltyTransactions(filters?: {
    customerId?: number;
    transactionType?: string;
    startDate?: string;
    endDate?: string;
    limit?: number;
    offset?: number;
  }): Promise<{ transactions: (CustomerLoyaltyPoints & { customerName?: string })[], total: number }> {
    // Get all transactions
    let allTransactions = await db.select().from(customerLoyaltyPoints)
      .orderBy(desc(customerLoyaltyPoints.createdAt));
    
    // Apply filters
    if (filters?.customerId) {
      allTransactions = allTransactions.filter(t => t.customerId === filters.customerId);
    }
    if (filters?.transactionType) {
      allTransactions = allTransactions.filter(t => t.transactionType === filters.transactionType);
    }
    if (filters?.startDate) {
      allTransactions = allTransactions.filter(t => {
        const txDate = new Date(t.createdAt).toISOString().split('T')[0];
        return txDate >= filters.startDate!;
      });
    }
    if (filters?.endDate) {
      allTransactions = allTransactions.filter(t => {
        const txDate = new Date(t.createdAt).toISOString().split('T')[0];
        return txDate <= filters.endDate!;
      });
    }

    const total = allTransactions.length;
    
    // Apply pagination
    if (filters?.offset !== undefined) {
      allTransactions = allTransactions.slice(filters.offset);
    }
    if (filters?.limit !== undefined) {
      allTransactions = allTransactions.slice(0, filters.limit);
    }

    // Get customer names
    const allCustomers = await db.select().from(customers);
    const customerMap: Record<number, string> = {};
    for (const c of allCustomers) {
      customerMap[c.id] = c.nameAr;
    }

    const transactions = allTransactions.map(t => ({
      ...t,
      customerName: customerMap[t.customerId] || 'غير معروف',
    }));

    return { transactions, total };
  }

  // Get loyalty points summary/statistics
  async getLoyaltySummary(): Promise<{
    totalCustomersWithPoints: number;
    totalActivePoints: number;
    totalPointsEarned: number;
    totalPointsRedeemed: number;
    totalPointsExpired: number;
    totalPointsAdjusted: number;
    topCustomers: { customerId: number; customerName: string; points: number }[];
  }> {
    const allCustomers = await db.select().from(customers);
    const allTransactions = await db.select().from(customerLoyaltyPoints);

    const customersWithPoints = allCustomers.filter(c => c.loyaltyPointsBalance > 0);
    const totalActivePoints = customersWithPoints.reduce((sum, c) => sum + c.loyaltyPointsBalance, 0);

    const earnedTransactions = allTransactions.filter(t => t.transactionType === 'earn');
    const redeemedTransactions = allTransactions.filter(t => t.transactionType === 'redeem');
    const expiredTransactions = allTransactions.filter(t => t.transactionType === 'expire');
    const adjustedTransactions = allTransactions.filter(t => t.transactionType === 'adjust');

    const totalPointsEarned = earnedTransactions.reduce((sum, t) => sum + Math.abs(t.points), 0);
    const totalPointsRedeemed = redeemedTransactions.reduce((sum, t) => sum + Math.abs(t.points), 0);
    const totalPointsExpired = expiredTransactions.reduce((sum, t) => sum + Math.abs(t.points), 0);
    const totalPointsAdjusted = adjustedTransactions.reduce((sum, t) => sum + t.points, 0);

    // Top 10 customers by points
    const topCustomers = customersWithPoints
      .sort((a, b) => b.loyaltyPointsBalance - a.loyaltyPointsBalance)
      .slice(0, 10)
      .map(c => ({
        customerId: c.id,
        customerName: c.nameAr,
        points: c.loyaltyPointsBalance,
      }));

    return {
      totalCustomersWithPoints: customersWithPoints.length,
      totalActivePoints,
      totalPointsEarned,
      totalPointsRedeemed,
      totalPointsExpired,
      totalPointsAdjusted,
      topCustomers,
    };
  }

  // Expire points for all customers based on expiry date
  async expireLoyaltyPoints(): Promise<{ expiredCount: number; totalPointsExpired: number }> {
    const now = new Date();
    
    // Get all unexpired earn transactions that have passed their expiry date
    const allTransactions = await db.select().from(customerLoyaltyPoints);
    const expiredEarnTransactions = allTransactions.filter(t => 
      t.transactionType === 'earn' && 
      t.expiresAt && 
      new Date(t.expiresAt) < now &&
      t.points > 0 // Only positive earn transactions
    );

    // Group by customer and calculate points to expire
    const customerExpiry: Record<number, number> = {};
    for (const tx of expiredEarnTransactions) {
      // Check if already expired (look for matching expire transaction)
      const alreadyExpired = allTransactions.some(t =>
        t.transactionType === 'expire' &&
        t.customerId === tx.customerId &&
        t.referenceId === tx.id
      );
      if (!alreadyExpired) {
        const current = customerExpiry[tx.customerId] || 0;
        customerExpiry[tx.customerId] = current + tx.points;
      }
    }

    let totalPointsExpired = 0;
    let expiredCount = 0;

    for (const customerIdStr of Object.keys(customerExpiry)) {
      const customerId = parseInt(customerIdStr);
      const pointsToExpire = customerExpiry[customerId];
      const customer = await this.getCustomer(customerId);
      if (customer && customer.loyaltyPointsBalance > 0) {
        const actualExpire = Math.min(pointsToExpire, customer.loyaltyPointsBalance);
        if (actualExpire > 0) {
          await this.addLoyaltyPoints(customerId, -actualExpire, 'expire', 'system', undefined, 'انتهاء صلاحية النقاط');
          totalPointsExpired += actualExpire;
          expiredCount++;
        }
      }
    }

    return { expiredCount, totalPointsExpired };
  }

  // Manual adjustment of loyalty points
  async adjustLoyaltyPoints(customerId: number, points: number, notes?: string): Promise<CustomerLoyaltyPoints> {
    const customer = await this.getCustomer(customerId);
    if (!customer) throw new Error("Customer not found");

    // For negative adjustments, check balance
    if (points < 0 && customer.loyaltyPointsBalance + points < 0) {
      throw new Error("لا يمكن خصم نقاط أكثر من الرصيد الحالي");
    }

    return this.addLoyaltyPoints(customerId, points, 'adjust', 'manual', undefined, notes);
  }
  
  // ========================================
  // GENERAL LEDGER & TRIAL BALANCE
  // ========================================
  
  async getGeneralLedger(accountCode?: string, startDate?: string, endDate?: string): Promise<any[]> {
    const allEntries = await db.select().from(journalEntries).orderBy(asc(journalEntries.date));
    const allLines = await db.select().from(journalLines);
    
    let filteredEntries = allEntries;
    
    // Filter by date range
    if (startDate) {
      filteredEntries = filteredEntries.filter(e => e.date >= startDate);
    }
    if (endDate) {
      filteredEntries = filteredEntries.filter(e => e.date <= endDate);
    }
    
    // Build ledger entries
    const ledger = filteredEntries.flatMap(entry => {
      let entryLines = allLines.filter(l => l.journalEntryId === entry.id);
      
      if (accountCode) {
        entryLines = entryLines.filter(l => l.accountCode === accountCode);
      }
      
      return entryLines.map(line => ({
        date: entry.date,
        entryNumber: entry.entryNumber,
        description: entry.description,
        accountCode: line.accountCode,
        accountName: line.accountName,
        debit: line.debit,
        credit: line.credit,
        referenceType: entry.referenceType,
        referenceId: entry.referenceId,
      }));
    });
    
    return ledger;
  }
  
  async getTrialBalance(asOfDate?: string): Promise<any[]> {
    const allAccounts = await this.getAccounts();
    const allEntries = await db.select().from(journalEntries);
    const allLines = await db.select().from(journalLines);
    
    // Filter entries by date if provided
    let filteredEntries = allEntries;
    if (asOfDate) {
      filteredEntries = allEntries.filter(e => e.date <= asOfDate);
    }
    
    const entryIds = new Set(filteredEntries.map(e => e.id));
    const filteredLines = allLines.filter(l => entryIds.has(l.journalEntryId));
    
    // Group by account code
    const balances: Record<string, { debit: number; credit: number }> = {};
    
    for (const line of filteredLines) {
      if (!balances[line.accountCode]) {
        balances[line.accountCode] = { debit: 0, credit: 0 };
      }
      balances[line.accountCode].debit += line.debit;
      balances[line.accountCode].credit += line.credit;
    }
    
    // Build trial balance
    const trialBalance = allAccounts
      .filter(acc => balances[acc.code])
      .map(acc => {
        const bal = balances[acc.code];
        const netBalance = bal.debit - bal.credit;
        
        return {
          accountCode: acc.code,
          accountName: acc.nameAr,
          accountType: acc.accountType,
          debit: bal.debit,
          credit: bal.credit,
          balance: Math.abs(netBalance),
          balanceType: netBalance >= 0 ? "debit" : "credit",
        };
      })
      .sort((a, b) => a.accountCode.localeCompare(b.accountCode));
    
    return trialBalance;
  }
  
  async getIncomeStatement(startDate?: string, endDate?: string): Promise<any> {
    const allAccounts = await this.getAccounts();
    const allEntries = await db.select().from(journalEntries);
    const allLines = await db.select().from(journalLines);
    
    // Filter entries by date range
    let filteredEntries = allEntries;
    if (startDate) {
      filteredEntries = filteredEntries.filter(e => e.date >= startDate);
    }
    if (endDate) {
      filteredEntries = filteredEntries.filter(e => e.date <= endDate);
    }
    
    const entryIds = new Set(filteredEntries.map(e => e.id));
    const filteredLines = allLines.filter(l => entryIds.has(l.journalEntryId));
    
    // Group by account code - calculate direct balances for all accounts
    const directBalances: Record<string, { debit: number; credit: number }> = {};
    for (const line of filteredLines) {
      if (!directBalances[line.accountCode]) {
        directBalances[line.accountCode] = { debit: 0, credit: 0 };
      }
      directBalances[line.accountCode].debit += line.debit;
      directBalances[line.accountCode].credit += line.credit;
    }
    
    // Get all revenue accounts (code starts with 4) and expense accounts (code starts with 5)
    const revenueAccounts = allAccounts.filter(acc => acc.code.startsWith("4"));
    const expenseAccounts = allAccounts.filter(acc => acc.code.startsWith("5"));
    
    // Helper function to build hierarchical amounts using bottom-up aggregation
    const buildHierarchicalAmounts = (accounts: any[], isRevenue: boolean): any[] => {
      // Create account map by code
      const accountMap = new Map<string, any>();
      accounts.forEach(acc => {
        accountMap.set(acc.code, {
          ...acc,
          amount: 0,
          isParent: false,
        });
      });
      
      // Sort accounts by code length descending (process leaves first, then parents)
      const sortedByDepth = [...accounts].sort((a, b) => b.code.length - a.code.length);
      
      // First pass: calculate direct amounts for leaf accounts only
      for (const acc of sortedByDepth) {
        const hasChildren = accounts.some(child => 
          child.code.startsWith(acc.code) && child.code !== acc.code
        );
        
        const item = accountMap.get(acc.code)!;
        item.isParent = hasChildren;
        
        // Only set direct balance for leaf (postable) accounts
        if (!hasChildren) {
          const bal = directBalances[acc.code] || { debit: 0, credit: 0 };
          item.amount = isRevenue ? (bal.credit - bal.debit) : (bal.debit - bal.credit);
        }
      }
      
      // Second pass: aggregate from leaves to parents (bottom-up)
      // Process from longest code to shortest (children before parents)
      for (const acc of sortedByDepth) {
        const item = accountMap.get(acc.code)!;
        
        if (item.isParent) {
          // Find direct children only (immediate children, not grandchildren)
          const directChildren = accounts.filter(child => {
            if (child.code === acc.code) return false;
            if (!child.code.startsWith(acc.code)) return false;
            
            // Check if there's any intermediate parent between acc and child
            const hasIntermediateParent = accounts.some(other => 
              other.code !== acc.code &&
              other.code !== child.code &&
              other.code.startsWith(acc.code) &&
              child.code.startsWith(other.code)
            );
            
            return !hasIntermediateParent;
          });
          
          // Sum amounts from direct children (which already include their descendants' amounts)
          item.amount = directChildren.reduce((sum, child) => {
            const childItem = accountMap.get(child.code)!;
            return sum + childItem.amount;
          }, 0);
        }
      }
      
      // Convert to array and sort by account code
      return [...accountMap.values()]
        .map(item => ({
          accountCode: item.code,
          accountName: item.nameAr,
          amount: item.amount,
          level: item.level || 1,
          isParent: item.isParent,
          isPostable: item.isPostable,
        }))
        .sort((a, b) => a.accountCode.localeCompare(b.accountCode));
    };
    
    // Build revenues and expenses with proper hierarchical aggregation
    const revenues = buildHierarchicalAmounts(revenueAccounts, true);
    const expenses = buildHierarchicalAmounts(expenseAccounts, false);
    
    // Get totals from root accounts
    const totalRevenue = revenues.find(r => r.accountCode === "4")?.amount || 0;
    const totalExpenses = expenses.find(e => e.accountCode === "5")?.amount || 0;
    const netIncome = totalRevenue - totalExpenses;
    
    return {
      revenues,
      expenses,
      totalRevenue,
      totalExpenses,
      netIncome,
      startDate,
      endDate,
    };
  }
  
  async getBalanceSheet(asOfDate?: string): Promise<any> {
    const allAccounts = await this.getAccounts();
    const allEntries = await db.select().from(journalEntries);
    const allLines = await db.select().from(journalLines);
    
    // Filter entries by date
    let filteredEntries = allEntries;
    if (asOfDate) {
      filteredEntries = allEntries.filter(e => e.date <= asOfDate);
    }
    
    const entryIds = new Set(filteredEntries.map(e => e.id));
    const filteredLines = allLines.filter(l => entryIds.has(l.journalEntryId));
    
    // Group by account code
    const balances: Record<string, { debit: number; credit: number }> = {};
    for (const line of filteredLines) {
      if (!balances[line.accountCode]) {
        balances[line.accountCode] = { debit: 0, credit: 0 };
      }
      balances[line.accountCode].debit += line.debit;
      balances[line.accountCode].credit += line.credit;
    }
    
    // Build balance sheet - assets, liabilities, equity
    const assetAccounts = allAccounts.filter(acc => acc.accountType === "asset");
    const liabilityAccounts = allAccounts.filter(acc => acc.accountType === "liability");
    const equityAccounts = allAccounts.filter(acc => acc.accountType === "equity");
    
    const assets = assetAccounts
      .filter(acc => balances[acc.code])
      .map(acc => {
        const bal = balances[acc.code];
        // Assets have debit normal balance
        const amount = bal.debit - bal.credit;
        return {
          accountCode: acc.code,
          accountName: acc.nameAr,
          amount,
        };
      })
      .filter(a => a.amount !== 0)
      .sort((a, b) => a.accountCode.localeCompare(b.accountCode));
    
    const liabilities = liabilityAccounts
      .filter(acc => balances[acc.code])
      .map(acc => {
        const bal = balances[acc.code];
        // Liabilities have credit normal balance
        const amount = bal.credit - bal.debit;
        return {
          accountCode: acc.code,
          accountName: acc.nameAr,
          amount,
        };
      })
      .filter(l => l.amount !== 0)
      .sort((a, b) => a.accountCode.localeCompare(b.accountCode));
    
    const equity = equityAccounts
      .filter(acc => balances[acc.code])
      .map(acc => {
        const bal = balances[acc.code];
        // Equity has credit normal balance
        const amount = bal.credit - bal.debit;
        return {
          accountCode: acc.code,
          accountName: acc.nameAr,
          amount,
        };
      })
      .filter(e => e.amount !== 0)
      .sort((a, b) => a.accountCode.localeCompare(b.accountCode));
    
    // Calculate retained earnings (net income from income statement)
    const incomeStatement = await this.getIncomeStatement(undefined, asOfDate);
    const retainedEarnings = incomeStatement.netIncome;
    
    const totalAssets = assets.reduce((sum, a) => sum + a.amount, 0);
    const totalLiabilities = liabilities.reduce((sum, l) => sum + l.amount, 0);
    const totalEquity = equity.reduce((sum, e) => sum + e.amount, 0) + retainedEarnings;
    
    return {
      assets,
      liabilities,
      equity,
      retainedEarnings,
      totalAssets,
      totalLiabilities,
      totalEquity,
      totalLiabilitiesAndEquity: totalLiabilities + totalEquity,
      isBalanced: Math.abs(totalAssets - (totalLiabilities + totalEquity)) < 0.01,
      asOfDate,
    };
  }
  
  async getCashFlowStatement(startDate?: string, endDate?: string): Promise<any> {
    const allEntries = await db.select().from(journalEntries);
    const allLines = await db.select().from(journalLines);
    
    // Filter entries by date range
    let filteredEntries = allEntries;
    if (startDate) {
      filteredEntries = filteredEntries.filter(e => e.date >= startDate);
    }
    if (endDate) {
      filteredEntries = filteredEntries.filter(e => e.date <= endDate);
    }
    
    const entryIds = new Set(filteredEntries.map(e => e.id));
    const filteredLines = allLines.filter(l => entryIds.has(l.journalEntryId));
    
    // Cash accounts (النقدية والبنك)
    const cashAccountCodes: string[] = [ACCOUNT_CODES.CASH, ACCOUNT_CODES.BANK];
    
    // Get all cash movements
    const cashMovements = filteredLines
      .filter(l => cashAccountCodes.includes(l.accountCode))
      .map(line => {
        const entry = filteredEntries.find(e => e.id === line.journalEntryId);
        return {
          date: entry?.date || '',
          entryNumber: entry?.entryNumber || '',
          description: entry?.description || '',
          accountCode: line.accountCode,
          accountName: line.accountName,
          debit: line.debit,
          credit: line.credit,
          netChange: line.debit - line.credit,
          referenceType: entry?.referenceType || '',
          referenceId: entry?.referenceId || null,
        };
      })
      .sort((a, b) => a.date.localeCompare(b.date));
    
    // Categorize by activity type
    const operatingActivities = cashMovements.filter(m => 
      ['PurchaseInvoice', 'Payment', 'SalesOrder', 'Order', 'Manual'].includes(m.referenceType) ||
      !m.referenceType
    );
    
    const investingActivities = cashMovements.filter(m => 
      m.referenceType === 'Investment' || m.referenceType === 'AssetPurchase'
    );
    
    const financingActivities = cashMovements.filter(m => 
      m.referenceType === 'CapitalContribution' || m.referenceType === 'OwnerWithdrawal' ||
      m.referenceType === 'Loan'
    );
    
    const totalOperating = operatingActivities.reduce((sum, m) => sum + m.netChange, 0);
    const totalInvesting = investingActivities.reduce((sum, m) => sum + m.netChange, 0);
    const totalFinancing = financingActivities.reduce((sum, m) => sum + m.netChange, 0);
    const netCashChange = totalOperating + totalInvesting + totalFinancing;
    
    // Get opening and closing balances
    let openingBalance = 0;
    if (startDate) {
      const previousEntries = allEntries.filter(e => e.date < startDate);
      const previousEntryIds = new Set(previousEntries.map(e => e.id));
      const previousCashLines = allLines.filter(l => 
        previousEntryIds.has(l.journalEntryId) && cashAccountCodes.includes(l.accountCode)
      );
      openingBalance = previousCashLines.reduce((sum, l) => sum + l.debit - l.credit, 0);
    }
    
    const closingBalance = openingBalance + netCashChange;
    
    return {
      operatingActivities,
      investingActivities,
      financingActivities,
      totalOperating,
      totalInvesting,
      totalFinancing,
      netCashChange,
      openingBalance,
      closingBalance,
      startDate,
      endDate,
    };
  }
  
  async createManualJournalEntry(data: { date: string; description: string; lines: { accountId: number; description?: string; debit: number; credit: number }[] }): Promise<JournalEntryWithLines> {
    // Validate that debits equal credits
    const totalDebit = data.lines.reduce((sum, l) => sum + l.debit, 0);
    const totalCredit = data.lines.reduce((sum, l) => sum + l.credit, 0);
    
    if (Math.abs(totalDebit - totalCredit) > 0.01) {
      throw new Error("القيد غير متوازن: مجموع المدين يجب أن يساوي مجموع الدائن");
    }
    
    // Get accounts for the lines
    const allAccounts = await this.getAccounts();
    const accountMap = new Map(allAccounts.map(a => [a.id, a]));
    
    // Validate all accounts exist and are postable
    for (const line of data.lines) {
      const account = accountMap.get(line.accountId);
      if (!account) {
        throw new Error(`الحساب غير موجود: ${line.accountId}`);
      }
      if (!account.isPostable) {
        throw new Error(`الحساب ${account.nameAr} غير قابل للترحيل`);
      }
    }
    
    // Generate entry number
    const count = await db.select({ count: sql<number>`count(*)` }).from(journalEntries);
    const entryNumber = `JE-${String((count[0]?.count || 0) + 1).padStart(6, '0')}`;
    
    // Create the journal entry
    const [entry] = await db.insert(journalEntries).values({
      entryNumber,
      date: data.date,
      description: data.description,
      referenceType: "Manual",
      referenceId: null,
    }).returning();
    
    // Create journal lines
    const journalLinesData: InsertJournalLine[] = data.lines.map(line => {
      const account = accountMap.get(line.accountId)!;
      return {
        journalEntryId: entry.id,
        accountCode: account.code,
        accountName: account.nameAr,
        description: line.description || null,
        debit: line.debit,
        credit: line.credit,
      };
    });
    
    const newLines = await db.insert(journalLines).values(journalLinesData).returning();
    
    return {
      ...entry,
      lines: newLines,
    };
  }
  
  // ========================================
  // ROLES
  // ========================================
  
  async getRoles(isActive?: boolean): Promise<Role[]> {
    if (isActive !== undefined) {
      return db.select().from(roles).where(eq(roles.isActive, isActive)).orderBy(asc(roles.nameAr));
    }
    return db.select().from(roles).orderBy(asc(roles.nameAr));
  }
  
  async getRole(id: number): Promise<Role | undefined> {
    const [role] = await db.select().from(roles).where(eq(roles.id, id));
    return role;
  }
  
  async createRole(role: InsertRole): Promise<Role> {
    const [newRole] = await db.insert(roles).values(role).returning();
    return newRole;
  }
  
  async updateRole(id: number, role: Partial<InsertRole>): Promise<Role | undefined> {
    const [updatedRole] = await db.update(roles)
      .set({ ...role, updatedAt: new Date() })
      .where(eq(roles.id, id))
      .returning();
    return updatedRole;
  }
  
  async deleteRole(id: number): Promise<boolean> {
    const [result] = await db.update(roles)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(roles.id, id))
      .returning();
    return !!result;
  }
  
  async getRolesWithPermissions(isActive?: boolean): Promise<RoleWithPermissions[]> {
    const allRoles = isActive !== undefined 
      ? await db.select().from(roles).where(eq(roles.isActive, isActive)).orderBy(asc(roles.nameAr))
      : await db.select().from(roles).orderBy(asc(roles.nameAr));
    
    const allPermissions = await db.select().from(rolePermissions);
    const allEmployees = await db.select().from(employees);
    
    return allRoles.map(role => ({
      ...role,
      permissions: allPermissions.filter(p => p.roleId === role.id),
      employeeCount: allEmployees.filter(e => e.roleId === role.id && e.isActive).length,
    }));
  }
  
  async getRoleWithPermissions(id: number): Promise<RoleWithPermissions | undefined> {
    const [role] = await db.select().from(roles).where(eq(roles.id, id));
    if (!role) return undefined;
    
    const permissions = await db.select().from(rolePermissions).where(eq(rolePermissions.roleId, id));
    const employeeCount = await db.select().from(employees).where(and(eq(employees.roleId, id), eq(employees.isActive, true)));
    
    return {
      ...role,
      permissions,
      employeeCount: employeeCount.length,
    };
  }
  
  async getRolePermissions(roleId: number): Promise<RolePermission[]> {
    return db.select().from(rolePermissions).where(eq(rolePermissions.roleId, roleId));
  }
  
  async setRolePermissions(roleId: number, permissions: InsertRolePermission[]): Promise<RolePermission[]> {
    // Delete existing permissions
    await db.delete(rolePermissions).where(eq(rolePermissions.roleId, roleId));
    
    // Insert new permissions
    if (permissions.length === 0) return [];
    
    const permissionsWithRoleId = permissions.map(p => ({
      ...p,
      roleId,
    }));
    
    return db.insert(rolePermissions).values(permissionsWithRoleId).returning();
  }
  
  // ========================================
  // BRANDS
  // ========================================
  
  async getBrands(isActive?: boolean): Promise<Brand[]> {
    if (isActive !== undefined) {
      return db.select().from(brands).where(eq(brands.isActive, isActive)).orderBy(asc(brands.nameAr));
    }
    return db.select().from(brands).orderBy(asc(brands.nameAr));
  }
  
  async getBrand(id: number): Promise<Brand | undefined> {
    const [brand] = await db.select().from(brands).where(eq(brands.id, id));
    return brand;
  }
  
  async createBrand(brand: InsertBrand): Promise<Brand> {
    const [newBrand] = await db.insert(brands).values(brand).returning();
    return newBrand;
  }
  
  async updateBrand(id: number, brand: Partial<InsertBrand>): Promise<Brand | undefined> {
    const [updatedBrand] = await db.update(brands)
      .set({ ...brand, updatedAt: new Date() })
      .where(eq(brands.id, id))
      .returning();
    return updatedBrand;
  }
  
  async deleteBrand(id: number): Promise<boolean> {
    const [result] = await db.update(brands)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(brands.id, id))
      .returning();
    return !!result;
  }
  
  // ========================================
  // EMPLOYEES (Extended HR)
  // ========================================
  
  async getEmployees(filters?: { isActive?: boolean; roleId?: number; branchId?: number; search?: string }): Promise<EmployeeWithRole[]> {
    const allEmployees = await db.select().from(employees).orderBy(desc(employees.createdAt));
    const allRoles = await db.select().from(roles);
    
    let result = allEmployees.map(emp => ({
      ...emp,
      role: allRoles.find(r => r.id === emp.roleId),
    }));
    
    // Apply filters
    if (filters?.isActive !== undefined) {
      result = result.filter(e => e.isActive === filters.isActive);
    }
    if (filters?.roleId) {
      result = result.filter(e => e.roleId === filters.roleId);
    }
    if (filters?.branchId) {
      result = result.filter(e => e.branchId === filters.branchId);
    }
    if (filters?.search) {
      const searchLower = filters.search.toLowerCase();
      result = result.filter(e => 
        (e.fullNameAr && e.fullNameAr.toLowerCase().includes(searchLower)) ||
        (e.nameAr && e.nameAr.toLowerCase().includes(searchLower)) ||
        (e.email && e.email.toLowerCase().includes(searchLower)) ||
        (e.phoneNumber && e.phoneNumber.includes(searchLower)) ||
        (e.phone && e.phone.includes(searchLower))
      );
    }
    
    return result;
  }
  
  async getEmployee(id: number): Promise<EmployeeWithRole | undefined> {
    const [employee] = await db.select().from(employees).where(eq(employees.id, id));
    if (!employee) return undefined;
    
    let role: Role | undefined;
    if (employee.roleId) {
      [role] = await db.select().from(roles).where(eq(roles.id, employee.roleId));
    }
    
    return { ...employee, role };
  }
  
  async createEmployee(employee: InsertEmployee): Promise<Employee> {
    // Also set nameAr from fullNameAr for backward compatibility
    const employeeData = {
      ...employee,
      nameAr: employee.fullNameAr,
      phone: employee.phoneNumber,
    };
    const [newEmployee] = await db.insert(employees).values(employeeData).returning();
    return newEmployee;
  }
  
  async updateEmployee(id: number, employee: Partial<InsertEmployee>): Promise<Employee | undefined> {
    // Also update legacy columns for backward compatibility
    const updateData: any = { ...employee, updatedAt: new Date() };
    if (employee.fullNameAr) updateData.nameAr = employee.fullNameAr;
    if (employee.phoneNumber) updateData.phone = employee.phoneNumber;
    
    const [updatedEmployee] = await db.update(employees)
      .set(updateData)
      .where(eq(employees.id, id))
      .returning();
    return updatedEmployee;
  }
  
  async deleteEmployee(id: number): Promise<boolean> {
    const [result] = await db.update(employees)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(employees.id, id))
      .returning();
    return !!result;
  }
  
  // ========================================
  // EMPLOYEE TASKS
  // ========================================
  
  async getEmployeeTasks(filters?: { assignedToEmployeeId?: number; status?: string; dueDate?: string }): Promise<EmployeeTaskWithDetails[]> {
    let query = db.select().from(employeeTasks);
    const conditions: any[] = [];
    
    if (filters?.assignedToEmployeeId) {
      conditions.push(eq(employeeTasks.assignedToEmployeeId, filters.assignedToEmployeeId));
    }
    if (filters?.status) {
      conditions.push(eq(employeeTasks.status, filters.status));
    }
    if (filters?.dueDate) {
      conditions.push(eq(employeeTasks.dueDate, filters.dueDate));
    }
    
    const tasks = conditions.length > 0
      ? await db.select().from(employeeTasks).where(and(...conditions)).orderBy(desc(employeeTasks.createdAt))
      : await db.select().from(employeeTasks).orderBy(desc(employeeTasks.createdAt));
    
    const allEmployees = await db.select().from(employees);
    
    return tasks.map(task => ({
      ...task,
      assignedToEmployee: allEmployees.find(e => e.id === task.assignedToEmployeeId),
      assignedByEmployee: task.assignedByEmployeeId ? allEmployees.find(e => e.id === task.assignedByEmployeeId) : undefined,
    }));
  }
  
  async getEmployeeTask(id: number): Promise<EmployeeTaskWithDetails | undefined> {
    const [task] = await db.select().from(employeeTasks).where(eq(employeeTasks.id, id));
    if (!task) return undefined;
    
    const [assignedTo] = await db.select().from(employees).where(eq(employees.id, task.assignedToEmployeeId));
    let assignedBy: Employee | undefined;
    if (task.assignedByEmployeeId) {
      [assignedBy] = await db.select().from(employees).where(eq(employees.id, task.assignedByEmployeeId));
    }
    
    return {
      ...task,
      assignedToEmployee: assignedTo,
      assignedByEmployee: assignedBy,
    };
  }
  
  async createEmployeeTask(task: InsertEmployeeTask): Promise<EmployeeTask> {
    const [newTask] = await db.insert(employeeTasks).values(task).returning();
    return newTask;
  }
  
  async updateEmployeeTask(id: number, task: Partial<InsertEmployeeTask>): Promise<EmployeeTask | undefined> {
    const updateData: any = { ...task, updatedAt: new Date() };
    
    // Set completedAt if status is changing to completed
    if (task.status === 'completed') {
      updateData.completedAt = new Date();
    }
    
    const [updatedTask] = await db.update(employeeTasks)
      .set(updateData)
      .where(eq(employeeTasks.id, id))
      .returning();
    return updatedTask;
  }
  
  async deleteEmployeeTask(id: number): Promise<boolean> {
    const result = await db.delete(employeeTasks).where(eq(employeeTasks.id, id));
    return true;
  }
  
  // ========================================
  // SALARY PROFILES
  // ========================================
  
  async getSalaryProfiles(): Promise<SalaryProfileWithEmployee[]> {
    const profiles = await db.select().from(salaryProfiles).orderBy(desc(salaryProfiles.createdAt));
    const allEmployees = await db.select().from(employees);
    
    return profiles.map(profile => ({
      ...profile,
      employee: allEmployees.find(e => e.id === profile.employeeId),
    }));
  }
  
  async getSalaryProfile(employeeId: number): Promise<SalaryProfile | undefined> {
    const [profile] = await db.select().from(salaryProfiles).where(eq(salaryProfiles.employeeId, employeeId));
    return profile;
  }
  
  async createOrUpdateSalaryProfile(profile: InsertSalaryProfile): Promise<SalaryProfile> {
    // Check if profile exists
    const existing = await this.getSalaryProfile(profile.employeeId);
    
    if (existing) {
      const [updated] = await db.update(salaryProfiles)
        .set({ ...profile, updatedAt: new Date() })
        .where(eq(salaryProfiles.employeeId, profile.employeeId))
        .returning();
      return updated;
    } else {
      const [created] = await db.insert(salaryProfiles).values(profile).returning();
      return created;
    }
  }
  
  async deleteSalaryProfile(employeeId: number): Promise<boolean> {
    await db.delete(salaryProfiles).where(eq(salaryProfiles.employeeId, employeeId));
    return true;
  }
  
  // ========================================
  // PAYROLL RUNS
  // ========================================
  
  async getPayrollRuns(): Promise<PayrollRun[]> {
    return db.select().from(payrollRuns).orderBy(desc(payrollRuns.createdAt));
  }
  
  async getPayrollRun(id: number): Promise<PayrollRunWithLines | undefined> {
    const [run] = await db.select().from(payrollRuns).where(eq(payrollRuns.id, id));
    if (!run) return undefined;
    
    const lines = await this.getPayrollLines(id);
    
    return {
      ...run,
      lines,
      totalNetSalary: lines.reduce((sum, line) => sum + (line.netSalary || 0), 0),
      employeeCount: lines.length,
    };
  }
  
  async createPayrollRun(run: InsertPayrollRun): Promise<PayrollRun> {
    const [newRun] = await db.insert(payrollRuns).values({
      ...run,
      status: run.status || 'draft',
    }).returning();
    return newRun;
  }
  
  async updatePayrollRun(id: number, run: Partial<InsertPayrollRun>): Promise<PayrollRun | undefined> {
    const [updated] = await db.update(payrollRuns)
      .set({ ...run, updatedAt: new Date() })
      .where(eq(payrollRuns.id, id))
      .returning();
    return updated;
  }
  
  async generatePayrollLines(payrollRunId: number, employeeIds?: number[]): Promise<PayrollLine[]> {
    // Get active employees with salary profiles
    let activeEmployees = await db.select().from(employees).where(eq(employees.isActive, true));
    
    // Filter by employee IDs if provided
    if (employeeIds && employeeIds.length > 0) {
      activeEmployees = activeEmployees.filter(emp => employeeIds.includes(emp.id));
    }
    
    const profiles = await db.select().from(salaryProfiles).where(eq(salaryProfiles.isActive, true));
    
    const linesToInsert: InsertPayrollLine[] = [];
    
    for (const emp of activeEmployees) {
      const profile = profiles.find(p => p.employeeId === emp.id);
      
      const basicSalary = profile?.basicSalary || 0;
      const fixedAllowances = (profile?.housingAllowance || 0) + 
                              (profile?.transportAllowance || 0) + 
                              (profile?.otherFixedAllowances || 0);
      const overtimeAmount = 0;
      const commissionAmount = 0;
      const otherEarnings = 0;
      const deductionLate = 0;
      const deductionOther = 0;
      
      const grossSalary = basicSalary + fixedAllowances + overtimeAmount + commissionAmount + otherEarnings;
      const totalDeductions = deductionLate + deductionOther;
      const netSalary = grossSalary - totalDeductions;
      
      linesToInsert.push({
        payrollRunId,
        employeeId: emp.id,
        basicSalary,
        fixedAllowances,
        overtimeAmount,
        commissionAmount,
        otherEarnings,
        deductionLate,
        deductionOther,
      });
    }
    
    if (linesToInsert.length === 0) {
      return [];
    }
    
    // Insert lines with calculated values
    const insertedLines: PayrollLine[] = [];
    for (const line of linesToInsert) {
      const grossSalary = (line.basicSalary || 0) + (line.fixedAllowances || 0) + 
                         (line.overtimeAmount || 0) + (line.commissionAmount || 0) + (line.otherEarnings || 0);
      const totalDeductions = (line.deductionLate || 0) + (line.deductionOther || 0);
      const netSalary = grossSalary - totalDeductions;
      
      const [inserted] = await db.insert(payrollLines).values({
        ...line,
        grossSalary,
        totalDeductions,
        netSalary,
      }).returning();
      insertedLines.push(inserted);
    }
    
    return insertedLines;
  }
  
  async approvePayrollRun(id: number): Promise<PayrollRun | undefined> {
    // Check if there are payroll lines
    const lines = await this.getPayrollLines(id);
    if (lines.length === 0) {
      throw new Error("لا يمكن اعتماد الرواتب قبل توليد بنود الرواتب للموظفين");
    }
    
    const [run] = await db.select().from(payrollRuns).where(eq(payrollRuns.id, id));
    if (!run) return undefined;
    if (run.status === 'paid') {
      throw new Error("لا يمكن تغيير حالة الرواتب بعد وسمها كمدفوعة");
    }
    
    return this.updatePayrollRun(id, { status: 'approved' });
  }
  
  async markPayrollRunAsPaid(id: number): Promise<PayrollRun | undefined> {
    const [run] = await db.select().from(payrollRuns).where(eq(payrollRuns.id, id));
    if (!run) return undefined;
    if (run.status !== 'approved') {
      throw new Error("يجب اعتماد الرواتب قبل وسمها كمدفوعة");
    }
    
    return this.updatePayrollRun(id, { status: 'paid' });
  }
  
  // ========================================
  // PAYROLL LINES
  // ========================================
  
  async getPayrollLines(payrollRunId: number): Promise<PayrollLineWithEmployee[]> {
    const lines = await db.select().from(payrollLines).where(eq(payrollLines.payrollRunId, payrollRunId));
    const allEmployees = await db.select().from(employees);
    
    return lines.map(line => ({
      ...line,
      employee: allEmployees.find(e => e.id === line.employeeId),
    }));
  }
  
  async getEmployeePayrollHistory(employeeId: number): Promise<(PayrollLineWithEmployee & { payrollRun?: PayrollRun })[]> {
    const lines = await db.select().from(payrollLines)
      .where(eq(payrollLines.employeeId, employeeId))
      .orderBy(desc(payrollLines.id));
    
    const allRuns = await db.select().from(payrollRuns);
    const [employee] = await db.select().from(employees).where(eq(employees.id, employeeId));
    
    return lines.map(line => {
      const payrollRun = allRuns.find(r => r.id === line.payrollRunId);
      return {
        ...line,
        employee: employee || undefined,
        payrollRun: payrollRun || undefined,
      };
    });
  }
  
  async updatePayrollLine(id: number, line: Partial<InsertPayrollLine>): Promise<PayrollLine | undefined> {
    // Get current line to recalculate
    const [current] = await db.select().from(payrollLines).where(eq(payrollLines.id, id));
    if (!current) return undefined;
    
    // Merge updates
    const basicSalary = line.basicSalary ?? current.basicSalary;
    const fixedAllowances = line.fixedAllowances ?? current.fixedAllowances;
    const overtimeAmount = line.overtimeAmount ?? current.overtimeAmount;
    const commissionAmount = line.commissionAmount ?? current.commissionAmount;
    const otherEarnings = line.otherEarnings ?? current.otherEarnings;
    const deductionLate = line.deductionLate ?? current.deductionLate;
    const deductionOther = line.deductionOther ?? current.deductionOther;
    
    // Recalculate totals
    const grossSalary = basicSalary + fixedAllowances + overtimeAmount + commissionAmount + otherEarnings;
    const totalDeductions = deductionLate + deductionOther;
    const netSalary = grossSalary - totalDeductions;
    
    const [updated] = await db.update(payrollLines)
      .set({
        ...line,
        grossSalary,
        totalDeductions,
        netSalary,
        updatedAt: new Date(),
      })
      .where(eq(payrollLines.id, id))
      .returning();
    
    return updated;
  }
  
  // ========================================
  // ORDERS
  // ========================================
  
  async getOrders(filters?: OrderFilters): Promise<OrderWithDetails[]> {
    let query = db.select().from(orders);
    
    const conditions: any[] = [];
    
    if (filters?.date) {
      // Filter by date (same day)
      const startOfDay = new Date(filters.date);
      startOfDay.setHours(0, 0, 0, 0);
      const endOfDay = new Date(filters.date);
      endOfDay.setHours(23, 59, 59, 999);
      conditions.push(and(
        gte(orders.orderDate, startOfDay),
        lte(orders.orderDate, endOfDay)
      ));
    }
    
    if (filters?.status) {
      conditions.push(eq(orders.status, filters.status));
    }
    
    if (filters?.source) {
      conditions.push(eq(orders.source, filters.source));
    }
    
    if (filters?.clientId) {
      conditions.push(eq(orders.clientId, filters.clientId));
    }
    
    if (filters?.branchId) {
      conditions.push(eq(orders.branchId, filters.branchId));
    }
    
    if (filters?.orderType) {
      conditions.push(eq(orders.orderType, filters.orderType));
    }
    
    if (filters?.isScheduled === true) {
      conditions.push(isNotNull(orders.scheduledAt));
    } else if (filters?.isScheduled === false) {
      conditions.push(isNull(orders.scheduledAt));
    }
    
    if (conditions.length > 0) {
      query = query.where(and(...conditions)) as any;
    }
    
    const allOrders = await query.orderBy(desc(orders.orderDate));
    
    // Fetch related data
    const allCustomers = await db.select().from(customers);
    const allEmployees = await db.select().from(employees);
    const allOrderServices = await db.select().from(orderServices);
    const allOrderProducts = await db.select().from(orderProducts);
    const allOrderRatings = await db.select().from(orderRatings);
    const allServices = await db.select().from(services);
    const allServiceCategories = await db.select().from(serviceCategories);
    const allProducts = await db.select().from(products);
    const allUnits = await db.select().from(units);
    
    return allOrders.map(order => ({
      ...order,
      client: allCustomers.find(c => c.id === order.clientId),
      createdByEmployee: allEmployees.find(e => e.id === order.createdByEmployeeId),
      services: allOrderServices
        .filter(os => os.orderId === order.id)
        .map(os => {
          const service = allServices.find(s => s.id === os.serviceId);
          return {
            ...os,
            service: service ? {
              ...service,
              category: allServiceCategories.find(c => c.id === service.categoryId),
            } : undefined,
            executingEmployee: allEmployees.find(e => e.id === os.executingEmployeeId),
          };
        }),
      products: allOrderProducts
        .filter(op => op.orderId === order.id)
        .map(op => {
          const product = allProducts.find(p => p.id === op.productId);
          return {
            ...op,
            product: product ? {
              ...product,
              sellUnit: allUnits.find(u => u.id === product.sellUnitId),
            } : undefined,
          };
        }),
      rating: allOrderRatings.find(r => r.orderId === order.id),
    })).filter(order => {
      // Apply hasRating filter on the mapped results
      if (filters?.hasRating === true) {
        return order.rating !== undefined;
      } else if (filters?.hasRating === false) {
        return order.rating === undefined;
      }
      return true;
    });
  }
  
  async getOrder(id: number): Promise<OrderWithDetails | undefined> {
    const [order] = await db.select().from(orders).where(eq(orders.id, id));
    if (!order) return undefined;
    
    const [client] = order.clientId ? await db.select().from(customers).where(eq(customers.id, order.clientId)) : [undefined];
    const [createdByEmployee] = await db.select().from(employees).where(eq(employees.id, order.createdByEmployeeId));
    
    const svcLines = await db.select().from(orderServices).where(eq(orderServices.orderId, id));
    const productLines = await db.select().from(orderProducts).where(eq(orderProducts.orderId, id));
    const packageLines = await db.select().from(orderPackages).where(eq(orderPackages.orderId, id));
    const [rating] = await db.select().from(orderRatings).where(eq(orderRatings.orderId, id));
    
    // Fetch payments for this order
    const orderPaymentsList = await db.select().from(orderPayments).where(eq(orderPayments.orderId, id));
    const allPaymentMethods = await db.select().from(paymentMethods);
    
    const allServices = await db.select().from(services);
    const allCategories = await db.select().from(serviceCategories);
    const allProducts = await db.select().from(products);
    const allUnits = await db.select().from(units);
    const allEmployees = await db.select().from(employees);
    const allPackages = await db.select().from(packages);
    
    return {
      ...order,
      client,
      createdByEmployee,
      services: svcLines.map(os => {
        const service = allServices.find(s => s.id === os.serviceId);
        return {
          ...os,
          service: service ? {
            ...service,
            category: allCategories.find(c => c.id === service.categoryId),
          } : undefined,
          executingEmployee: allEmployees.find(e => e.id === os.executingEmployeeId),
        };
      }),
      products: productLines.map(op => {
        const product = allProducts.find(p => p.id === op.productId);
        return {
          ...op,
          product: product ? {
            ...product,
            sellUnit: allUnits.find(u => u.id === product.sellUnitId),
          } : undefined,
        };
      }),
      packages: packageLines.map(pk => ({
        ...pk,
        package: allPackages.find(p => p.id === pk.packageId),
      })),
      payments: orderPaymentsList.map(op => ({
        ...op,
        paymentMethod: allPaymentMethods.find(pm => pm.id === op.paymentMethodId),
      })),
      rating,
    };
  }
  
  async createOrder(request: CreateOrderRequest): Promise<OrderWithDetails> {
    // Generate order number
    const count = await db.select({ count: sql<number>`count(*)` }).from(orders);
    const orderNumber = `SO-${String((count[0]?.count || 0) + 1).padStart(4, '0')}`;
    
    // Fetch services and products for calculations
    const allServices = await db.select().from(services);
    const allProducts = await db.select().from(products);
    
    // Calculate service lines
    const serviceLines: {
      serviceId: number;
      quantity: number;
      basePrice: number;
      vatType: string;
      vatRate: number;
      lineSubtotal: number;
      vatAmount: number;
      lineTotal: number;
      scheduledAt: Date | null;
      executingEmployeeId: number | null;
      packageId: number | null;
    }[] = [];
    
    let totalServiceSubtotal = 0;
    let totalServiceVat = 0;
    let hasScheduledServices = false;
    
    for (const svc of request.services) {
      const service = allServices.find(s => s.id === svc.serviceId);
      if (!service || !service.isActive) {
        throw new Error(`الخدمة غير موجودة أو غير نشطة: ${svc.serviceId}`);
      }
      
      const quantity = svc.quantity || 1;
      const vatRate = 0.15; // Standard VAT rate
      
      // Calculate line totals
      let lineSubtotal: number;
      let vatAmount: number;
      let lineTotal: number;
      let basePrice: number;
      let vatType: string;
      
      // If packagePrice is provided, use it (VAT-inclusive package pricing)
      if (svc.packagePrice != null && svc.packageId != null) {
        // Package price is VAT-inclusive
        const gross = svc.packagePrice * quantity;
        lineTotal = gross;
        lineSubtotal = gross / (1 + vatRate);
        vatAmount = gross - lineSubtotal;
        basePrice = svc.packagePrice;
        vatType = "inclusive";
      } else {
        // Use service's own pricing
        basePrice = service.price;
        vatType = service.vatType;
        
        if (vatType === "inclusive") {
          const gross = basePrice * quantity;
          lineSubtotal = gross / (1 + service.vatRate);
          vatAmount = gross - lineSubtotal;
          lineTotal = gross;
        } else if (vatType === "exclusive") {
          lineSubtotal = basePrice * quantity;
          vatAmount = lineSubtotal * service.vatRate;
          lineTotal = lineSubtotal + vatAmount;
        } else {
          // exempt
          lineSubtotal = basePrice * quantity;
          vatAmount = 0;
          lineTotal = lineSubtotal;
        }
      }
      
      const scheduledAt = svc.scheduledAt ? new Date(svc.scheduledAt) : null;
      if (scheduledAt && scheduledAt > new Date()) {
        hasScheduledServices = true;
      }
      
      serviceLines.push({
        serviceId: svc.serviceId,
        quantity,
        basePrice,
        vatType,
        vatRate: svc.packageId != null ? vatRate : service.vatRate,
        lineSubtotal,
        vatAmount,
        lineTotal,
        scheduledAt,
        executingEmployeeId: svc.executingEmployeeId || null,
        packageId: svc.packageId || null,
      });
      
      totalServiceSubtotal += lineSubtotal;
      totalServiceVat += vatAmount;
    }
    
    // Calculate product lines
    const productLines: {
      productId: number;
      quantity: number;
      unitPrice: number;
      vatType: string;
      vatRate: number;
      lineSubtotal: number;
      vatAmount: number;
      lineTotal: number;
      packageId: number | null;
    }[] = [];
    
    let totalProductSubtotal = 0;
    let totalProductVat = 0;
    
    if (request.products && request.products.length > 0) {
      for (const prod of request.products) {
        const product = allProducts.find(p => p.id === prod.productId);
        if (!product || !product.isActive) {
          throw new Error(`المنتج غير موجود أو غير نشط: ${prod.productId}`);
        }
        
        const quantity = prod.quantity;
        const vatRate = 0.15; // Standard VAT rate
        
        let unitPrice: number;
        let vatType: string;
        let lineSubtotal: number;
        let vatAmount: number;
        let lineTotal: number;
        
        // If packagePrice is provided, use it (VAT-inclusive package pricing)
        if (prod.packagePrice != null && prod.packageId != null) {
          // Package price is VAT-inclusive
          unitPrice = prod.packagePrice;
          vatType = "inclusive";
          const gross = unitPrice * quantity;
          lineTotal = gross;
          lineSubtotal = gross / (1 + vatRate);
          vatAmount = gross - lineSubtotal;
        } else {
          // Use product's own pricing (exclusive VAT)
          unitPrice = product.defaultSellPrice || 0;
          vatType = "exclusive";
          lineSubtotal = unitPrice * quantity;
          vatAmount = lineSubtotal * vatRate;
          lineTotal = lineSubtotal + vatAmount;
        }
        
        productLines.push({
          productId: prod.productId,
          quantity,
          unitPrice,
          vatType,
          vatRate,
          lineSubtotal,
          vatAmount,
          lineTotal,
          packageId: prod.packageId || null,
        });
        
        totalProductSubtotal += lineSubtotal;
        totalProductVat += vatAmount;
      }
    }
    
    // Calculate order totals
    const subtotalAmount = totalServiceSubtotal + totalProductSubtotal;
    const vatAmount = totalServiceVat + totalProductVat;
    const couponDiscountAmount = request.couponDiscountAmount || 0;
    
    // Calculate loyalty points value on the server (validate against settings)
    let loyaltyPointsValue = 0;
    let actualPointsToRedeem = 0;
    
    if (request.redeemLoyaltyPoints && request.redeemLoyaltyPoints > 0 && request.clientId) {
      const loyaltySettingsData = await this.getLoyaltySettings();
      const customer = await this.getCustomer(request.clientId);
      
      if (loyaltySettingsData?.isActive && customer) {
        const currencyPerPoint = loyaltySettingsData.currencyPerPoint || 0;
        const customerBalance = customer.loyaltyPointsBalance || 0;
        const minRedeemPoints = loyaltySettingsData.minRedeemPoints || 0;
        
        // Validate minimum redeemable points
        if (request.redeemLoyaltyPoints >= minRedeemPoints) {
          // Cap points to customer balance
          actualPointsToRedeem = Math.min(request.redeemLoyaltyPoints, customerBalance);
          
          // Calculate the max value that can be redeemed (cannot exceed order total before loyalty)
          const orderTotalBeforeLoyalty = subtotalAmount + vatAmount - couponDiscountAmount;
          const maxRedeemValue = orderTotalBeforeLoyalty;
          
          // Calculate value and cap to order total
          const calculatedValue = actualPointsToRedeem * currencyPerPoint;
          loyaltyPointsValue = Math.min(calculatedValue, maxRedeemValue);
          
          // Recalculate actual points based on capped value
          if (calculatedValue > maxRedeemValue) {
            actualPointsToRedeem = Math.floor(maxRedeemValue / currencyPerPoint);
            loyaltyPointsValue = actualPointsToRedeem * currencyPerPoint;
          }
        }
      }
    }
    
    const totalAmount = subtotalAmount + vatAmount - couponDiscountAmount - loyaltyPointsValue;
    
    // Determine status
    let status = "new";
    if (hasScheduledServices && serviceLines.every(sl => sl.scheduledAt && sl.scheduledAt > new Date())) {
      status = "scheduled";
    }
    
    // Insert order
    const [newOrder] = await db.insert(orders).values({
      orderNumber,
      orderDate: new Date(),
      source: request.source,
      orderType: request.orderType,
      clientId: request.clientId || null,
      createdByEmployeeId: request.createdByEmployeeId,
      branchId: request.branchId || null,
      couponCode: request.couponCode || null,
      couponDiscountAmount,
      status,
      notes: request.notes || null,
      subtotalAmount,
      vatAmount,
      totalAmount,
    }).returning();
    
    // Insert service lines
    if (serviceLines.length > 0) {
      await db.insert(orderServices).values(
        serviceLines.map(sl => ({
          orderId: newOrder.id,
          ...sl,
        }))
      );
    }
    
    // Insert product lines
    if (productLines.length > 0) {
      await db.insert(orderProducts).values(
        productLines.map(pl => ({
          orderId: newOrder.id,
          ...pl,
        }))
      );
    }
    
    // Insert order packages (multi-package support)
    const packageIdsToInsert = request.packageIds && request.packageIds.length > 0 
      ? request.packageIds 
      : (request.packageId ? [request.packageId] : []);
    
    if (packageIdsToInsert.length > 0) {
      await db.insert(orderPackages).values(
        packageIdsToInsert.map(pkgId => ({
          orderId: newOrder.id,
          packageId: pkgId,
        }))
      );
    }
    
    // Handle loyalty points redemption (only if we have valid points to redeem)
    if (actualPointsToRedeem > 0 && loyaltyPointsValue > 0 && request.clientId) {
      try {
        // Deduct points using the server-validated amount
        await this.addLoyaltyPoints(
          request.clientId,
          -actualPointsToRedeem,
          'redeem',
          'order',
          newOrder.id,
          `استبدال ${actualPointsToRedeem} نقطة للطلب رقم ${orderNumber}`
        );
      } catch (error) {
        console.error("Failed to redeem loyalty points:", error);
        // If points redemption fails, update the order total to remove the discount
        const correctedTotal = subtotalAmount + vatAmount - couponDiscountAmount;
        await db.update(orders)
          .set({ totalAmount: correctedTotal, updatedAt: new Date() })
          .where(eq(orders.id, newOrder.id));
      }
    }
    
    // Create payment records - supports multiple partial payments
    let totalPaymentsAmount = 0;
    const paymentsToProcess = request.payments && request.payments.length > 0 
      ? request.payments 
      : (request.paymentMethodId && request.paymentAmount && request.paymentAmount > 0 
          ? [{ paymentMethodId: request.paymentMethodId, amount: request.paymentAmount, notes: null }]
          : []);
    
    for (const payment of paymentsToProcess) {
      if (payment.amount <= 0) continue;
      
      try {
        // Get payment method for validation
        const paymentMethod = await this.getPaymentMethod(payment.paymentMethodId);
        if (paymentMethod) {
          // Calculate remaining amount that can be paid
          const remainingAmount = totalAmount - totalPaymentsAmount;
          if (remainingAmount <= 0) break; // Order is fully paid
          
          // Cap payment amount to remaining balance (prevent overpayment)
          const cappedPaymentAmount = Math.min(payment.amount, remainingAmount);
          
          if (cappedPaymentAmount > 0) {
            await this.createOrderPayment({
              orderId: newOrder.id,
              paymentMethodId: payment.paymentMethodId,
              amount: cappedPaymentAmount,
              paymentDate: new Date().toISOString().split('T')[0],
              notes: payment.notes || `دفعة عند إنشاء الطلب رقم ${orderNumber}`,
            });
            totalPaymentsAmount += cappedPaymentAmount;
          }
        }
      } catch (error) {
        console.error("Failed to create payment:", error);
        // Continue with next payment even if this one fails
      }
    }
    
    return this.getOrder(newOrder.id) as Promise<OrderWithDetails>;
  }
  
  async updateOrder(id: number, updates: { status?: string; notes?: string | null; couponCode?: string | null; couponDiscountAmount?: number; deferredAt?: Date | null; deferredReason?: string | null; cancelledAt?: Date | null; cancelledReason?: string | null; returnedAt?: Date | null; returnedReason?: string | null; scheduledAt?: Date | null; ratingValue?: number | null; ratingComment?: string | null }): Promise<OrderWithDetails | undefined> {
    const existingOrder = await this.getOrder(id);
    if (!existingOrder) return undefined;
    
    // Prevent any status changes for cancelled or returned orders (ZATCA compliance)
    if (existingOrder.status === "canceled" || existingOrder.status === "returned") {
      throw new Error("لا يمكن تعديل طلب ملغي أو مرتجع");
    }
    
    const wasCompleted = existingOrder.status === "completed";
    const isNowCompleted = updates.status === "completed";
    
    // Add completedAt timestamp if completing the order
    const orderUpdates: any = { ...updates, updatedAt: new Date() };
    if (!wasCompleted && isNowCompleted) {
      orderUpdates.completedAt = new Date();
    }
    
    // Update order
    const [updatedOrder] = await db.update(orders)
      .set(orderUpdates)
      .where(eq(orders.id, id))
      .returning();
    
    if (!updatedOrder) return undefined;
    
    // If status changed to "completed", process inventory and create journal entries
    if (!wasCompleted && isNowCompleted) {
      // Get full order details for journal entries
      const fullOrder = await this.getOrder(id);
      if (fullOrder) {
        // Create revenue journal entry
        const journalEntryId = await this.createOrderRevenueJournalEntry(fullOrder);
        
        // Create COGS journal entry
        const cogsJournalEntryId = await this.createOrderCOGSJournalEntry(fullOrder);
        
        // Update order with journal entry IDs
        await db.update(orders)
          .set({ 
            journalEntryId, 
            cogsJournalEntryId: cogsJournalEntryId || null,
          })
          .where(eq(orders.id, id));
        
        // Process inventory consumption
        await this.processOrderInventoryConsumption(id);
      }
    }
    
    return this.getOrder(id);
  }
  
  async updateOrderService(orderServiceId: number, updates: { executingEmployeeId?: number | null; status?: string; deferredAt?: Date | null; deferredReason?: string | null; completedAt?: Date | null; scheduledAt?: Date | null }): Promise<OrderService | undefined> {
    const [updatedService] = await db.update(orderServices)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(orderServices.id, orderServiceId))
      .returning();
    
    if (!updatedService) return undefined;
    
    // If service is marked as completed, check if all services are completed to auto-complete order
    if (updates.status === "completed") {
      const order = await db.select().from(orders).where(eq(orders.id, updatedService.orderId));
      if (order.length > 0) {
        const allServices = await db.select().from(orderServices).where(eq(orderServices.orderId, updatedService.orderId));
        const allCompleted = allServices.every(s => s.status === "completed");
        
        if (allCompleted) {
          await this.updateOrder(updatedService.orderId, { status: "completed" });
        }
      }
    }
    
    return updatedService;
  }
  
  async getOrderServices(orderId: number): Promise<OrderServiceWithDetails[]> {
    const servicesList = await db.select().from(orderServices).where(eq(orderServices.orderId, orderId));
    
    if (servicesList.length === 0) return [];
    
    const serviceIds = Array.from(new Set(servicesList.map(s => s.serviceId)));
    const employeeIds = Array.from(new Set(servicesList.filter(s => s.executingEmployeeId).map(s => s.executingEmployeeId!)));
    
    const servicesData = await db.select().from(services).where(sql`${services.id} IN ${serviceIds}`);
    const categoryIds = Array.from(new Set(servicesData.filter(s => s.categoryId).map(s => s.categoryId!)));
    const categoriesData = categoryIds.length > 0 
      ? await db.select().from(serviceCategories).where(sql`${serviceCategories.id} IN ${categoryIds}`)
      : [];
    const employeesData = employeeIds.length > 0
      ? await db.select().from(employees).where(sql`${employees.id} IN ${employeeIds}`)
      : [];
    
    return servicesList.map(sl => ({
      ...sl,
      service: servicesData.find(s => s.id === sl.serviceId) 
        ? {
            ...servicesData.find(s => s.id === sl.serviceId)!,
            category: categoriesData.find(c => c.id === servicesData.find(s => s.id === sl.serviceId)?.categoryId),
          }
        : undefined,
      executingEmployee: employeesData.find(e => e.id === sl.executingEmployeeId),
    }));
  }
  
  private async processOrderInventoryConsumption(orderId: number): Promise<void> {
    const order = await this.getOrder(orderId);
    if (!order) return;
    
    const today = new Date().toISOString().split('T')[0];
    
    // Process service-related product consumption
    for (const serviceLine of order.services) {
      const serviceProductsList = await this.getServiceProducts(serviceLine.serviceId);
      
      for (const sp of serviceProductsList) {
        const totalQuantity = sp.quantityPerService * serviceLine.quantity;
        
        // Create stock movement for consumption
        await db.insert(stockMovements).values({
          productId: sp.productId,
          movementDate: today,
          movementType: "consume",
          quantityIn: 0,
          quantityOut: totalQuantity,
          referenceType: "OrderService",
          referenceId: serviceLine.id,
        });
        
        // Decrease stock from batches (FIFO)
        await this.decreaseStockFIFO(sp.productId, totalQuantity);
      }
    }
    
    // Process direct product sales
    for (const productLine of order.products) {
      // Create stock movement for sale
      await db.insert(stockMovements).values({
        productId: productLine.productId,
        movementDate: today,
        movementType: "sale",
        quantityIn: 0,
        quantityOut: productLine.quantity,
        referenceType: "OrderProduct",
        referenceId: productLine.id,
      });
      
      // Decrease stock from batches (FIFO)
      await this.decreaseStockFIFO(productLine.productId, productLine.quantity);
    }
  }
  
  /**
   * Create revenue journal entry when order is completed
   * DR Cash/Bank/Receivable (totalAmount)
   * CR Sales Revenue (subtotalAmount)
   * CR VAT Output (vatAmount)
   */
  private async createOrderRevenueJournalEntry(order: OrderWithDetails): Promise<number> {
    // Generate entry number
    const count = await db.select({ count: sql<number>`count(*)` }).from(journalEntries);
    const entryNumber = `JE-${String((count[0]?.count || 0) + 1).padStart(6, '0')}`;
    
    const today = new Date().toISOString().split('T')[0];
    
    const [entry] = await db.insert(journalEntries).values({
      entryNumber,
      date: today,
      description: `قيد إيرادات مبيعات - طلب رقم ${order.orderNumber}`,
      referenceType: "Order",
      referenceId: order.id,
    }).returning();
    
    const journalLinesData: InsertJournalLine[] = [];
    
    // Debit - Cash/Receivable (depends on payment status, for now we use receivable)
    journalLinesData.push({
      journalEntryId: entry.id,
      accountCode: SALES_ACCOUNT_CODES.ACCOUNTS_RECEIVABLE,
      accountName: "العملاء (المدينون)",
      debit: order.totalAmount,
      credit: 0,
      customerId: order.clientId,
    });
    
    // Credit - Sales Revenue
    journalLinesData.push({
      journalEntryId: entry.id,
      accountCode: SALES_ACCOUNT_CODES.SALON_REVENUE,
      accountName: "إيرادات الصالون",
      debit: 0,
      credit: order.subtotalAmount,
    });
    
    // Credit - VAT Output (if applicable)
    if (order.vatAmount > 0) {
      journalLinesData.push({
        journalEntryId: entry.id,
        accountCode: SALES_ACCOUNT_CODES.VAT_OUTPUT,
        accountName: "ضريبة القيمة المضافة المحصلة",
        debit: 0,
        credit: order.vatAmount,
      });
    }
    
    await db.insert(journalLines).values(journalLinesData);
    
    return entry.id;
  }
  
  /**
   * Create COGS journal entry when order is completed
   * DR COGS (cost of products sold + consumed)
   * CR Inventory (same amount)
   */
  private async createOrderCOGSJournalEntry(order: OrderWithDetails): Promise<number | null> {
    let totalCOGS = 0;
    
    // Calculate COGS for direct product sales
    for (const productLine of order.products) {
      const product = await this.getProduct(productLine.productId);
      if (product) {
        // Use average unit cost if available, otherwise use product sell price as fallback
        const unitCost = product.averageUnitCost || productLine.unitPrice || 0;
        totalCOGS += unitCost * productLine.quantity;
      }
    }
    
    // Calculate COGS for service-related product consumption
    for (const serviceLine of order.services) {
      const serviceProductsList = await this.getServiceProducts(serviceLine.serviceId);
      for (const sp of serviceProductsList) {
        const product = await this.getProduct(sp.productId);
        if (product) {
          const unitCost = product.averageUnitCost || 0;
          const consumedQty = sp.quantityPerService * serviceLine.quantity;
          totalCOGS += unitCost * consumedQty;
        }
      }
    }
    
    // Only create journal entry if there's COGS to record
    if (totalCOGS <= 0) return null;
    
    // Generate entry number
    const count = await db.select({ count: sql<number>`count(*)` }).from(journalEntries);
    const entryNumber = `JE-${String((count[0]?.count || 0) + 1).padStart(6, '0')}`;
    
    const today = new Date().toISOString().split('T')[0];
    
    const [entry] = await db.insert(journalEntries).values({
      entryNumber,
      date: today,
      description: `قيد تكلفة بضاعة مباعة - طلب رقم ${order.orderNumber}`,
      referenceType: "Order",
      referenceId: order.id,
    }).returning();
    
    const journalLinesData: InsertJournalLine[] = [
      // Debit COGS
      {
        journalEntryId: entry.id,
        accountCode: SALES_ACCOUNT_CODES.COGS,
        accountName: "تكلفة البضاعة المباعة",
        debit: totalCOGS,
        credit: 0,
      },
      // Credit Inventory
      {
        journalEntryId: entry.id,
        accountCode: SALES_ACCOUNT_CODES.INVENTORY,
        accountName: "المخزون",
        debit: 0,
        credit: totalCOGS,
      },
    ];
    
    await db.insert(journalLines).values(journalLinesData);
    
    return entry.id;
  }
  
  private async decreaseStockFIFO(productId: number, quantity: number): Promise<void> {
    // Get batches ordered by expiry date (FEFO/FIFO - earliest expiry first)
    const batches = await db.select()
      .from(stockBatches)
      .where(and(
        eq(stockBatches.productId, productId),
        sql`${stockBatches.quantityOnHand} > 0`
      ))
      .orderBy(asc(stockBatches.expiryDate), asc(stockBatches.createdAt));
    
    let remainingQuantity = quantity;
    
    for (const batch of batches) {
      if (remainingQuantity <= 0) break;
      
      const deductQuantity = Math.min(batch.quantityOnHand, remainingQuantity);
      const newQuantity = batch.quantityOnHand - deductQuantity;
      
      await db.update(stockBatches)
        .set({ quantityOnHand: newQuantity, updatedAt: new Date() })
        .where(eq(stockBatches.id, batch.id));
      
      remainingQuantity -= deductQuantity;
    }
  }
  
  // ========================================
  // ORDER RETURNS
  // ========================================
  
  async getOrderReturns(orderId?: number): Promise<OrderReturnWithDetails[]> {
    let query = db.select().from(orderReturns).orderBy(desc(orderReturns.createdAt));
    
    const returnsList = orderId 
      ? await db.select().from(orderReturns).where(eq(orderReturns.orderId, orderId)).orderBy(desc(orderReturns.createdAt))
      : await db.select().from(orderReturns).orderBy(desc(orderReturns.createdAt));
    
    if (returnsList.length === 0) return [];
    
    const returnIds = returnsList.map(r => r.id);
    const orderIds = Array.from(new Set(returnsList.map(r => r.orderId)));
    const employeeIds = Array.from(new Set(returnsList.map(r => r.processedByEmployeeId)));
    
    const ordersData = await db.select().from(orders).where(sql`${orders.id} IN ${orderIds}`);
    const employeesData = await db.select().from(employees).where(sql`${employees.id} IN ${employeeIds}`);
    const linesData = await db.select().from(orderReturnLines).where(sql`${orderReturnLines.returnId} IN ${returnIds}`);
    
    return returnsList.map(ret => ({
      ...ret,
      order: ordersData.find(o => o.id === ret.orderId),
      processedByEmployee: employeesData.find(e => e.id === ret.processedByEmployeeId),
      lines: linesData.filter(l => l.returnId === ret.id).map(line => ({
        ...line,
        orderService: undefined,
        orderProduct: undefined,
      })),
    }));
  }
  
  async getOrderReturn(id: number): Promise<OrderReturnWithDetails | undefined> {
    const [ret] = await db.select().from(orderReturns).where(eq(orderReturns.id, id));
    if (!ret) return undefined;
    
    const [order] = await db.select().from(orders).where(eq(orders.id, ret.orderId));
    const [employee] = await db.select().from(employees).where(eq(employees.id, ret.processedByEmployeeId));
    const lines = await db.select().from(orderReturnLines).where(eq(orderReturnLines.returnId, id));
    
    // Get service and product details for lines
    const serviceIds = lines.filter(l => l.orderServiceId).map(l => l.orderServiceId!) as number[];
    const productLineIds = lines.filter(l => l.orderProductId).map(l => l.orderProductId!) as number[];
    
    const serviceLines = serviceIds.length > 0
      ? await db.select().from(orderServices).where(sql`${orderServices.id} IN ${serviceIds}`)
      : [];
    const productLines = productLineIds.length > 0
      ? await db.select().from(orderProducts).where(sql`${orderProducts.id} IN ${productLineIds}`)
      : [];
    
    return {
      ...ret,
      order,
      processedByEmployee: employee,
      lines: lines.map(line => ({
        ...line,
        orderService: serviceLines.find(s => s.id === line.orderServiceId) as any,
        orderProduct: productLines.find(p => p.id === line.orderProductId) as any,
      })),
    };
  }
  
  async createOrderReturn(returnData: InsertOrderReturn, lines: Omit<InsertOrderReturnLine, 'returnId'>[]): Promise<OrderReturnWithDetails> {
    // Generate return number
    const lastReturn = await db.select().from(orderReturns).orderBy(desc(orderReturns.id)).limit(1);
    const lastNumber = lastReturn.length > 0 ? parseInt(lastReturn[0].returnNumber.split('-')[1] || '0') : 0;
    const returnNumber = `RET-${String(lastNumber + 1).padStart(4, '0')}`;
    
    // Generate credit note number
    const lastCNNumber = lastReturn.length > 0 && lastReturn[0].creditNoteNumber
      ? parseInt(lastReturn[0].creditNoteNumber.split('-')[1] || '0') 
      : 0;
    const creditNoteNumber = `CN-${String(lastCNNumber + 1).padStart(4, '0')}`;
    
    // Get order for original invoice number
    const order = await this.getOrder(returnData.orderId);
    
    // Create return header
    const [newReturn] = await db.insert(orderReturns).values({
      ...returnData,
      returnNumber,
      creditNoteNumber,
      originalInvoiceNumber: order?.orderNumber || null,
    }).returning();
    
    // Create return lines
    if (lines.length > 0) {
      await db.insert(orderReturnLines).values(
        lines.map(line => ({
          ...line,
          returnId: newReturn.id,
        }))
      );
    }
    
    // Update order status to returned if full return
    if (returnData.returnType === "full") {
      await this.updateOrder(returnData.orderId, { 
        status: "returned" as any,
        returnedAt: new Date(),
        returnedReason: returnData.reason,
      });
    }
    
    return this.getOrderReturn(newReturn.id) as Promise<OrderReturnWithDetails>;
  }
  
  // ========================================
  // ORDER RATINGS
  // ========================================
  
  async createOrUpdateOrderRating(orderId: number, ratingData: Omit<InsertOrderRating, 'orderId'>): Promise<OrderRating> {
    // Check if order exists and is completed
    const order = await this.getOrder(orderId);
    if (!order) {
      throw new Error("الطلب غير موجود");
    }
    if (order.status !== "completed") {
      throw new Error("لا يمكن تقييم الطلب إلا بعد اكتماله");
    }
    
    // Check if rating already exists
    const existingRating = await this.getOrderRating(orderId);
    
    if (existingRating) {
      // Update existing rating
      const [updated] = await db.update(orderRatings)
        .set({
          rating: ratingData.rating,
          comment: ratingData.comment,
          clientId: ratingData.clientId,
        })
        .where(eq(orderRatings.orderId, orderId))
        .returning();
      return updated;
    } else {
      // Create new rating
      const [newRating] = await db.insert(orderRatings).values({
        orderId,
        ...ratingData,
      }).returning();
      return newRating;
    }
  }
  
  async getOrderRating(orderId: number): Promise<OrderRating | undefined> {
    const [rating] = await db.select().from(orderRatings).where(eq(orderRatings.orderId, orderId));
    return rating;
  }
  
  // ========================================
  // ORDER PAYMENTS (مدفوعات الطلبات)
  // ========================================
  
  async getOrderPayments(orderId: number): Promise<(OrderPayment & { paymentMethod?: PaymentMethod })[]> {
    const payments = await db.select().from(orderPayments).where(eq(orderPayments.orderId, orderId)).orderBy(desc(orderPayments.createdAt));
    const allPaymentMethods = await db.select().from(paymentMethods);
    
    return payments.map(payment => ({
      ...payment,
      paymentMethod: allPaymentMethods.find(m => m.id === payment.paymentMethodId),
    }));
  }
  
  async createOrderPayment(payment: InsertOrderPayment): Promise<OrderPayment> {
    // Get the order to validate and update
    const order = await this.getOrder(payment.orderId);
    if (!order) {
      throw new Error("الطلب غير موجود");
    }
    
    // Calculate new paid amount
    const existingPayments = await this.getOrderPayments(payment.orderId);
    const existingPaidAmount = existingPayments.reduce((sum, p) => sum + p.amount, 0);
    const newPaidAmount = existingPaidAmount + payment.amount;
    
    // Validate payment amount
    if (newPaidAmount > order.totalAmount) {
      throw new Error("المبلغ المدفوع أكبر من قيمة الطلب");
    }
    
    // Create the payment
    const [newPayment] = await db.insert(orderPayments).values(payment).returning();
    
    // Create journal entry for the payment
    const journalEntryId = await this.createOrderPaymentJournalEntry(newPayment, order);
    
    // Update payment with journal entry ID
    await db.update(orderPayments)
      .set({ journalEntryId })
      .where(eq(orderPayments.id, newPayment.id));
    
    // Determine new payment status
    let newPaymentStatus: string;
    if (newPaidAmount >= order.totalAmount) {
      newPaymentStatus = "paid";
    } else if (newPaidAmount > 0) {
      newPaymentStatus = "partially_paid";
    } else {
      newPaymentStatus = "unpaid";
    }
    
    // Update order with new paid amount and payment status
    await db.update(orders)
      .set({ 
        paidAmount: newPaidAmount,
        paymentStatus: newPaymentStatus,
        updatedAt: new Date(),
      })
      .where(eq(orders.id, payment.orderId));
    
    return { ...newPayment, journalEntryId };
  }
  
  /**
   * Create journal entry for order payment
   * DR Cash/Bank/Card (based on payment method's linked account)
   * CR Accounts Receivable (العملاء/المدينون)
   */
  private async createOrderPaymentJournalEntry(payment: OrderPayment, order: OrderWithDetails): Promise<number> {
    const paymentMethod = await this.getPaymentMethod(payment.paymentMethodId);
    
    // Generate entry number
    const count = await db.select({ count: sql<number>`count(*)` }).from(journalEntries);
    const entryNumber = `JE-${String((count[0]?.count || 0) + 1).padStart(6, '0')}`;
    
    const [entry] = await db.insert(journalEntries).values({
      entryNumber,
      date: payment.paymentDate,
      description: `قيد سداد عميل - طلب رقم ${order.orderNumber}`,
      referenceType: "OrderPayment",
      referenceId: payment.id,
    }).returning();
    
    // Get the linked account from payment method (use accountId if set)
    let debitAccountCode: string;
    let debitAccountName: string;
    
    if (paymentMethod?.accountId) {
      // Use the linked account from payment method
      const [linkedAccount] = await db.select().from(accounts).where(eq(accounts.id, paymentMethod.accountId));
      if (linkedAccount) {
        debitAccountCode = linkedAccount.code;
        debitAccountName = linkedAccount.nameAr;
      } else {
        // Fallback to cash if linked account not found
        debitAccountCode = SALES_ACCOUNT_CODES.CASH;
        debitAccountName = "النقدية";
      }
    } else {
      // Fallback: Determine account based on payment method type (legacy behavior)
      debitAccountCode = SALES_ACCOUNT_CODES.CASH;
      debitAccountName = "النقدية";
      if (paymentMethod?.type === "bank") {
        debitAccountCode = SALES_ACCOUNT_CODES.BANK;
        debitAccountName = "البنك";
      } else if (paymentMethod?.type === "card") {
        debitAccountCode = SALES_ACCOUNT_CODES.CARD;
        debitAccountName = "شبكة";
      }
    }
    
    // Accounts Receivable (المدينون) - use code from SALES_ACCOUNT_CODES
    const ACCOUNTS_RECEIVABLE_CODE = SALES_ACCOUNT_CODES.ACCOUNTS_RECEIVABLE;
    const ACCOUNTS_RECEIVABLE_NAME = "العملاء (المدينون)";
    
    const journalLinesData: InsertJournalLine[] = [
      // Debit Cash/Bank/Card - based on payment method's linked account
      {
        journalEntryId: entry.id,
        accountCode: debitAccountCode,
        accountName: debitAccountName,
        debit: payment.amount,
        credit: 0,
      },
      // Credit Accounts Receivable (العملاء)
      {
        journalEntryId: entry.id,
        accountCode: ACCOUNTS_RECEIVABLE_CODE,
        accountName: ACCOUNTS_RECEIVABLE_NAME,
        debit: 0,
        credit: payment.amount,
        customerId: order.clientId,
      },
    ];
    
    await db.insert(journalLines).values(journalLinesData);
    
    return entry.id;
  }
  
  /**
   * Cancel an order and create reversal journal entries if order was completed
   * Creates a credit note (إشعار دائن) for ZATCA compliance
   */
  async cancelOrder(id: number, reason?: string, refundPaymentMethodId?: number): Promise<OrderWithDetails | undefined> {
    const order = await this.getOrder(id);
    if (!order) return undefined;
    
    // Don't allow canceling already cancelled or returned orders
    if (order.status === "canceled") {
      throw new Error("الطلب ملغي بالفعل");
    }
    if (order.status === "returned") {
      throw new Error("الطلب مرتجع بالفعل");
    }
    
    // If order was completed, create reversal entries
    if (order.status === "completed") {
      await this.createOrderCancellationReversalEntries(order);
    }
    
    // Create order return record with credit note for ZATCA compliance
    await this.createCancellationReturn(order, reason || "إلغاء الطلب", refundPaymentMethodId);
    
    // Update order status
    const [updated] = await db.update(orders)
      .set({
        status: "canceled",
        cancelledAt: new Date(),
        cancelledReason: reason || null,
        updatedAt: new Date(),
      })
      .where(eq(orders.id, id))
      .returning();
    
    return this.getOrder(id);
  }
  
  /**
   * Pay a cancelled order - creates a direct revenue entry
   * Journal Entry:
   *   DR Cash/Bank/Card (based on payment method)
   *   CR Other Revenue - Cancelled Order Payments (4300)
   */
  async payCancelledOrder(orderId: number, payment: { amount: number; paymentMethodId: number; notes?: string }): Promise<{ success: boolean; journalEntryId: number }> {
    // Get the order
    const order = await this.getOrder(orderId);
    if (!order) {
      throw new Error("الطلب غير موجود");
    }
    
    // Verify order is cancelled
    if (order.status !== "canceled") {
      throw new Error("يمكن سداد الطلبات الملغاة فقط");
    }
    
    // Validate payment amount
    if (payment.amount <= 0) {
      throw new Error("مبلغ السداد يجب أن يكون أكبر من صفر");
    }
    
    // Calculate remaining balance
    const paidAmount = order.paidAmount || 0;
    const remainingBalance = order.totalAmount - paidAmount;
    
    if (remainingBalance <= 0) {
      throw new Error("الطلب مسدد بالكامل بالفعل");
    }
    
    if (payment.amount > remainingBalance) {
      throw new Error(`مبلغ السداد (${payment.amount.toFixed(2)}) أكبر من المبلغ المتبقي (${remainingBalance.toFixed(2)})`);
    }
    
    // Get payment method
    const paymentMethod = await this.getPaymentMethod(payment.paymentMethodId);
    if (!paymentMethod) {
      throw new Error("طريقة الدفع غير موجودة");
    }
    
    // Generate entry number
    const count = await db.select({ count: sql<number>`count(*)` }).from(journalEntries);
    const entryNumber = `JE-${String((count[0]?.count || 0) + 1).padStart(6, '0')}`;
    
    const today = new Date().toISOString().split('T')[0];
    
    // Create journal entry
    const [entry] = await db.insert(journalEntries).values({
      entryNumber,
      date: today,
      description: `سداد طلب ملغي - طلب رقم ${order.orderNumber}${payment.notes ? ` - ${payment.notes}` : ''}`,
      referenceType: "CancelledOrderPayment",
      referenceId: orderId,
    }).returning();
    
    // Determine debit account (Cash/Bank/Card) based on payment method
    let debitAccountCode: string;
    let debitAccountName: string;
    
    if (paymentMethod?.accountId) {
      const [linkedAccount] = await db.select().from(accounts).where(eq(accounts.id, paymentMethod.accountId));
      if (linkedAccount) {
        debitAccountCode = linkedAccount.code;
        debitAccountName = linkedAccount.nameAr;
      } else {
        debitAccountCode = SALES_ACCOUNT_CODES.CASH;
        debitAccountName = "النقدية";
      }
    } else {
      debitAccountCode = SALES_ACCOUNT_CODES.CASH;
      debitAccountName = "النقدية";
      if (paymentMethod?.type === "bank") {
        debitAccountCode = SALES_ACCOUNT_CODES.BANK;
        debitAccountName = "البنك";
      } else if (paymentMethod?.type === "card") {
        debitAccountCode = SALES_ACCOUNT_CODES.CARD;
        debitAccountName = "شبكة";
      }
    }
    
    // Credit account: Other Revenue - Cancelled Order Payments (4300)
    const CANCELLED_ORDER_REVENUE_CODE = "4300";
    const CANCELLED_ORDER_REVENUE_NAME = "سداد طلبات ملغاة";
    
    const journalLinesData: InsertJournalLine[] = [
      // Debit Cash/Bank/Card
      {
        journalEntryId: entry.id,
        accountCode: debitAccountCode,
        accountName: debitAccountName,
        debit: payment.amount,
        credit: 0,
      },
      // Credit Other Revenue - Cancelled Order Payments
      {
        journalEntryId: entry.id,
        accountCode: CANCELLED_ORDER_REVENUE_CODE,
        accountName: CANCELLED_ORDER_REVENUE_NAME,
        debit: 0,
        credit: payment.amount,
      },
    ];
    
    await db.insert(journalLines).values(journalLinesData);
    
    // Update order with paid amount and payment status
    const currentPaidAmount = order.paidAmount || 0;
    const newPaidAmount = currentPaidAmount + payment.amount;
    
    let newPaymentStatus: string;
    if (newPaidAmount >= order.totalAmount) {
      newPaymentStatus = "paid";
    } else if (newPaidAmount > 0) {
      newPaymentStatus = "partially_paid";
    } else {
      newPaymentStatus = "unpaid";
    }
    
    await db.update(orders)
      .set({
        paidAmount: newPaidAmount,
        paymentStatus: newPaymentStatus,
        updatedAt: new Date(),
      })
      .where(eq(orders.id, orderId));
    
    return { success: true, journalEntryId: entry.id };
  }
  
  /**
   * Create order return record with credit note number for cancelled order
   * Required for ZATCA compliance - cancelled orders must have a credit note
   */
  private async createCancellationReturn(order: OrderWithDetails, reason: string, refundPaymentMethodId?: number): Promise<void> {
    // Generate return number
    const lastReturn = await db.select().from(orderReturns).orderBy(desc(orderReturns.id)).limit(1);
    const lastReturnNumber = lastReturn.length > 0 ? parseInt(lastReturn[0].returnNumber.split('-')[1] || '0') : 0;
    const returnNumber = `RET-${String(lastReturnNumber + 1).padStart(4, '0')}`;
    
    // Generate credit note number (ZATCA format)
    const lastCreditNote = await db.select().from(orderReturns).orderBy(desc(orderReturns.id)).limit(1);
    const lastCNNumber = lastCreditNote.length > 0 && lastCreditNote[0].creditNoteNumber
      ? parseInt(lastCreditNote[0].creditNoteNumber.split('-')[1] || '0') 
      : 0;
    const creditNoteNumber = `CN-${String(lastCNNumber + 1).padStart(4, '0')}`;
    
    // Calculate amounts
    const subtotal = order.subtotalAmount || 0;
    const vatAmount = order.vatAmount || 0;
    const total = order.totalAmount || 0;
    
    // Create return header
    const [newReturn] = await db.insert(orderReturns).values({
      orderId: order.id,
      returnNumber,
      creditNoteNumber,
      returnDate: new Date(),
      processedByEmployeeId: order.createdByEmployeeId || null,
      refundPaymentMethodId: refundPaymentMethodId || null,
      reason,
      returnType: "cancellation",
      originalInvoiceNumber: order.orderNumber,
      subtotalAmount: subtotal,
      vatAmount: vatAmount,
      refundAmount: total,
      notes: `إشعار دائن تلقائي - إلغاء طلب رقم ${order.orderNumber}`,
    }).returning();
    
    // Create return lines for all services
    const serviceLines = order.services || [];
    for (const service of serviceLines) {
      await db.insert(orderReturnLines).values({
        returnId: newReturn.id,
        orderServiceId: service.id,
        orderProductId: null,
        quantity: service.quantity || 1,
        refundAmount: service.lineTotal || 0,
        reason: reason,
      });
    }
    
    // Create return lines for all products
    const productLines = order.products || [];
    for (const product of productLines) {
      await db.insert(orderReturnLines).values({
        returnId: newReturn.id,
        orderServiceId: null,
        orderProductId: product.id,
        quantity: product.quantity || 1,
        refundAmount: product.lineTotal || 0,
        reason: reason,
      });
    }
  }
  
  /**
   * Create reversal journal entries for cancelled order
   * This reverses both revenue/VAT entries and COGS entries
   */
  private async createOrderCancellationReversalEntries(order: OrderWithDetails): Promise<void> {
    // Generate entry number
    const count = await db.select({ count: sql<number>`count(*)` }).from(journalEntries);
    const entryNumber = `JE-${String((count[0]?.count || 0) + 1).padStart(6, '0')}`;
    
    const today = new Date().toISOString().split('T')[0];
    
    // Create the reversal journal entry
    const [entry] = await db.insert(journalEntries).values({
      entryNumber,
      date: today,
      description: `قيد عكسي - إلغاء طلب رقم ${order.orderNumber}`,
      referenceType: "SalesOrderCancellation",
      referenceId: order.id,
    }).returning();
    
    // Calculate values from order
    const subtotal = order.subtotalAmount || 0;
    const vatAmount = order.vatAmount || 0;
    const total = order.totalAmount || 0;
    
    const journalLinesData: InsertJournalLine[] = [];
    
    // === Reverse Revenue Entry ===
    // CR Accounts Receivable (reverse original debit)
    journalLinesData.push({
      journalEntryId: entry.id,
      accountCode: SALES_ACCOUNT_CODES.ACCOUNTS_RECEIVABLE,
      accountName: "العملاء (المدينون)",
      debit: 0,
      credit: total,
      customerId: order.clientId,
    });
    
    // DR Sales Revenue (reverse original credit)
    journalLinesData.push({
      journalEntryId: entry.id,
      accountCode: SALES_ACCOUNT_CODES.SALON_REVENUE,
      accountName: "إيرادات الصالون",
      debit: subtotal,
      credit: 0,
    });
    
    // DR VAT Output (reverse original credit)
    if (vatAmount > 0) {
      journalLinesData.push({
        journalEntryId: entry.id,
        accountCode: SALES_ACCOUNT_CODES.VAT_OUTPUT,
        accountName: "ضريبة القيمة المضافة المستحقة",
        debit: vatAmount,
        credit: 0,
      });
    }
    
    // === Reverse COGS Entry and Restock Inventory ===
    // Get order products for COGS reversal and restocking
    const orderProductsList = order.products || [];
    
    for (const op of orderProductsList) {
      if (op.productId && op.quantity > 0) {
        // Get product for cost calculation
        const product = await this.getProduct(op.productId);
        if (product) {
          // Use average unit cost or unit price as fallback
          const unitCost = product.averageUnitCost || op.unitPrice || 0;
          const costAmount = op.quantity * unitCost;
          
          if (costAmount > 0) {
            // CR COGS (reverse original debit)
            journalLinesData.push({
              journalEntryId: entry.id,
              accountCode: SALES_ACCOUNT_CODES.COGS,
              accountName: "تكلفة البضاعة المباعة",
              debit: 0,
              credit: costAmount,
            });
            
            // DR Inventory (reverse original credit)
            journalLinesData.push({
              journalEntryId: entry.id,
              accountCode: SALES_ACCOUNT_CODES.INVENTORY,
              accountName: "المخزون",
              debit: costAmount,
              credit: 0,
            });
            
            // Create stock movement to restore inventory
            await db.insert(stockMovements).values({
              productId: op.productId,
              movementDate: today,
              movementType: "return",
              quantityIn: op.quantity,
              quantityOut: 0,
              referenceType: "SalesOrderCancellation",
              referenceId: order.id,
            });
          }
        }
      }
    }
    
    if (journalLinesData.length > 0) {
      await db.insert(journalLines).values(journalLinesData);
    }
  }
  
  // ========================================
  // COUPON VALIDATION & APPLICATION
  // ========================================
  
  async validateCoupon(
    code: string,
    orderServices?: { serviceId: number; quantity: number; unitPrice: number }[],
    orderProducts?: { productId: number; quantity: number; unitPrice: number }[]
  ): Promise<{ valid: boolean; error?: string; coupon?: CouponWithDetails; discountAmount?: number }> {
    // Get coupon by code
    const coupon = await this.getCouponByCode(code);
    
    if (!coupon) {
      return { valid: false, error: "كود الكوبون غير صحيح" };
    }
    
    // Check if coupon is active
    if (!coupon.isActive) {
      return { valid: false, error: "الكوبون غير نشط" };
    }
    
    // Check date validity
    const now = new Date();
    const startDate = new Date(coupon.startDate);
    const endDate = new Date(coupon.endDate);
    
    if (now < startDate) {
      return { valid: false, error: "الكوبون لم يبدأ بعد" };
    }
    
    if (now > endDate) {
      return { valid: false, error: "الكوبون منتهي الصلاحية" };
    }
    
    // Calculate discount amount
    const discountAmount = await this.applyCouponToOrder(coupon.id, orderServices, orderProducts);
    
    if (discountAmount === 0) {
      return { valid: false, error: "الكوبون لا ينطبق على عناصر الطلب" };
    }
    
    return { valid: true, coupon, discountAmount };
  }
  
  async applyCouponToOrder(
    couponId: number,
    orderServices?: { serviceId: number; quantity: number; unitPrice: number }[],
    orderProducts?: { productId: number; quantity: number; unitPrice: number }[]
  ): Promise<number> {
    const coupon = await this.getCoupon(couponId);
    if (!coupon) return 0;
    
    let applicableTotal = 0;
    
    // Calculate total for applicable services
    if (coupon.appliesToServices && orderServices && orderServices.length > 0) {
      // If specific services are defined, only apply to those
      if (coupon.allowedServices && coupon.allowedServices.length > 0) {
        const allowedServiceIds = new Set(coupon.allowedServices.map(s => s.serviceId));
        for (const os of orderServices) {
          if (allowedServiceIds.has(os.serviceId)) {
            applicableTotal += os.quantity * os.unitPrice;
          }
        }
      } else {
        // Apply to all services
        for (const os of orderServices) {
          applicableTotal += os.quantity * os.unitPrice;
        }
      }
    }
    
    // Calculate total for applicable products
    if (coupon.appliesToProducts && orderProducts && orderProducts.length > 0) {
      // If specific products are defined, only apply to those
      if (coupon.allowedProducts && coupon.allowedProducts.length > 0) {
        const allowedProductIds = new Set(coupon.allowedProducts.map(p => p.productId));
        for (const op of orderProducts) {
          if (allowedProductIds.has(op.productId)) {
            applicableTotal += op.quantity * op.unitPrice;
          }
        }
      } else {
        // Apply to all products
        for (const op of orderProducts) {
          applicableTotal += op.quantity * op.unitPrice;
        }
      }
    }
    
    if (applicableTotal === 0) return 0;
    
    // Calculate discount
    if (coupon.discountType === "percent") {
      return Math.round((applicableTotal * coupon.discountValue / 100) * 100) / 100;
    } else {
      // Fixed amount discount
      return Math.min(coupon.discountValue, applicableTotal);
    }
  }
  
  // ========================================
  // PACKAGE APPLICATION
  // ========================================
  
  async getPackageItems(packageId: number): Promise<{ 
    services: { serviceId: number; quantity: number; unitPrice: number }[]; 
    products: { productId: number; quantity: number; unitPrice: number }[] 
  }> {
    const pkg = await this.getPackage(packageId);
    if (!pkg) {
      return { services: [], products: [] };
    }
    
    const serviceItems = pkg.services.map(s => ({
      serviceId: s.serviceId,
      quantity: s.quantity,
      unitPrice: s.packagePrice,
    }));
    
    const productItems = pkg.products.map(p => ({
      productId: p.productId,
      quantity: p.quantity,
      unitPrice: p.packagePrice,
    }));
    
    return { services: serviceItems, products: productItems };
  }
  
  // ========================================
  // APPOINTMENTS (حجوزات التطبيق)
  // ========================================
  
  async generateAppointmentNumber(): Promise<string> {
    const result = await db.select({ count: sql<number>`count(*)` }).from(appointments);
    const count = Number(result[0]?.count || 0);
    return `AP-${String(count + 1).padStart(4, '0')}`;
  }
  
  async getAppointments(filters?: AppointmentFilters): Promise<AppointmentWithDetails[]> {
    let query = db.select().from(appointments);
    
    const conditions: any[] = [];
    
    if (filters?.date) {
      conditions.push(sql`DATE(${appointments.scheduledAt}) = ${filters.date}`);
    }
    if (filters?.status) {
      conditions.push(eq(appointments.status, filters.status));
    }
    if (filters?.appointmentType) {
      conditions.push(eq(appointments.appointmentType, filters.appointmentType));
    }
    if (filters?.source) {
      conditions.push(eq(appointments.source, filters.source));
    }
    if (filters?.clientId) {
      conditions.push(eq(appointments.clientId, filters.clientId));
    }
    if (filters?.branchId) {
      conditions.push(eq(appointments.branchId, filters.branchId));
    }
    
    let appointmentRows: Appointment[];
    if (conditions.length > 0) {
      appointmentRows = await db.select().from(appointments).where(and(...conditions)).orderBy(desc(appointments.scheduledAt));
    } else {
      appointmentRows = await db.select().from(appointments).orderBy(desc(appointments.scheduledAt));
    }
    
    // Fetch related data for each appointment
    const result: AppointmentWithDetails[] = [];
    for (const appt of appointmentRows) {
      const full = await this.getAppointment(appt.id);
      if (full) result.push(full);
    }
    
    return result;
  }
  
  async getAppointment(id: number): Promise<AppointmentWithDetails | undefined> {
    const [appt] = await db.select().from(appointments).where(eq(appointments.id, id));
    if (!appt) return undefined;
    
    // Get client
    const [client] = appt.clientId ? await db.select().from(customers).where(eq(customers.id, appt.clientId)) : [undefined];
    
    // Get services
    const serviceRows = await db.select().from(appointmentServices).where(eq(appointmentServices.appointmentId, id));
    const servicesWithDetails: AppointmentServiceWithDetails[] = [];
    
    for (const svc of serviceRows) {
      const [service] = await db.select().from(services).where(eq(services.id, svc.serviceId));
      const [preferredEmployee] = svc.preferredEmployeeId
        ? await db.select().from(employees).where(eq(employees.id, svc.preferredEmployeeId))
        : [undefined];
      
      servicesWithDetails.push({
        ...svc,
        service,
        preferredEmployee,
      });
    }
    
    // Get related order if exists
    let relatedOrder: Order | undefined;
    if (appt.relatedOrderId) {
      const [order] = await db.select().from(orders).where(eq(orders.id, appt.relatedOrderId));
      relatedOrder = order;
    }
    
    return {
      ...appt,
      client,
      services: servicesWithDetails,
      relatedOrder,
    };
  }
  
  async createAppointment(request: CreateAppointmentRequest): Promise<AppointmentWithDetails> {
    // Validate client exists
    const [client] = await db.select().from(customers).where(eq(customers.id, request.clientId));
    if (!client) {
      throw new Error("العميل غير موجود");
    }
    
    // Validate services
    if (!request.services || request.services.length === 0) {
      throw new Error("يجب اختيار خدمة واحدة على الأقل");
    }
    
    // Validate home service requirements - need address OR coordinates
    if (request.appointmentType === "home_service") {
      const hasAddress = request.addressText && request.addressText.trim();
      const hasCoordinates = request.locationLat && request.locationLng;
      if (!hasAddress && !hasCoordinates) {
        throw new Error("يجب إدخال العنوان أو إحداثيات الموقع للخدمة المنزلية");
      }
    }
    
    // Calculate service totals
    let subtotalAmount = 0;
    let vatAmountTotal = 0;
    const serviceLines: Omit<InsertAppointmentService, "appointmentId">[] = [];
    
    for (const svcReq of request.services) {
      const [service] = await db.select().from(services).where(and(
        eq(services.id, svcReq.serviceId),
        eq(services.isActive, true)
      ));
      
      if (!service) {
        throw new Error("الخدمة غير موجودة أو غير مفعّلة");
      }
      
      const quantity = svcReq.quantity || 1;
      const basePrice = service.price || 0;
      const vatType = service.vatType || "exempt";
      const vatRate = service.vatRate || 0;
      
      let lineSubtotal: number;
      let lineVatAmount: number;
      let lineTotal: number;
      
      if (vatType === "inclusive") {
        const gross = basePrice * quantity;
        lineSubtotal = gross / (1 + vatRate);
        lineVatAmount = gross - lineSubtotal;
        lineTotal = gross;
      } else if (vatType === "exclusive") {
        lineSubtotal = basePrice * quantity;
        lineVatAmount = lineSubtotal * vatRate;
        lineTotal = lineSubtotal + lineVatAmount;
      } else {
        // exempt
        lineSubtotal = basePrice * quantity;
        lineVatAmount = 0;
        lineTotal = lineSubtotal;
      }
      
      subtotalAmount += lineSubtotal;
      vatAmountTotal += lineVatAmount;
      
      serviceLines.push({
        serviceId: svcReq.serviceId,
        quantity,
        basePrice,
        vatType,
        vatRate,
        lineSubtotal,
        vatAmount: lineVatAmount,
        lineTotal,
        preferredEmployeeId: svcReq.preferredEmployeeId || null,
      });
    }
    
    // Get delivery fee from system settings for home service
    let deliveryFee = 0;
    if (request.appointmentType === "home_service") {
      const setting = await this.getSystemSetting("home_service_delivery_fee");
      deliveryFee = setting ? parseFloat(setting.value) || 0 : 0;
    }
    
    const totalAmount = subtotalAmount + vatAmountTotal + deliveryFee;
    
    // Generate appointment number
    const appointmentNumber = await this.generateAppointmentNumber();
    
    // Create appointment
    const [appt] = await db.insert(appointments).values({
      appointmentNumber,
      source: request.source || "app",
      clientId: request.clientId,
      branchId: request.branchId || null,
      appointmentType: request.appointmentType,
      scheduledAt: new Date(request.scheduledAt),
      status: "pending",
      deliveryFee,
      locationLat: request.locationLat || null,
      locationLng: request.locationLng || null,
      addressText: request.addressText || null,
      notes: request.notes || null,
      subtotalAmount,
      vatAmount: vatAmountTotal,
      totalAmount,
    }).returning();
    
    // Create service lines
    for (const line of serviceLines) {
      await db.insert(appointmentServices).values({
        appointmentId: appt.id,
        ...line,
      });
    }
    
    return (await this.getAppointment(appt.id))!;
  }
  
  async updateAppointment(id: number, updates: {
    scheduledAt?: Date;
    branchId?: number | null;
    appointmentType?: string;
    deliveryFee?: number;
    locationLat?: number | null;
    locationLng?: number | null;
    addressText?: string | null;
    status?: string;
    notes?: string | null;
    cancelledReason?: string | null;
    relatedOrderId?: number | null;
  }): Promise<AppointmentWithDetails | undefined> {
    const [appt] = await db.select().from(appointments).where(eq(appointments.id, id));
    if (!appt) return undefined;
    
    // Validate status if provided
    if (updates.status) {
      const validStatuses = ["pending", "confirmed", "in_progress", "completed", "canceled", "no_show", "postponed"];
      if (!validStatuses.includes(updates.status)) {
        throw new Error("حالة الحجز غير صحيحة");
      }
      
      // Check for completion without services
      if (updates.status === "completed") {
        const services = await db.select().from(appointmentServices).where(eq(appointmentServices.appointmentId, id));
        if (services.length === 0) {
          throw new Error("لا يمكن إكمال الحجز بدون خدمات");
        }
      }
    }
    
    // Validate home service fields if changing to home_service
    if (updates.appointmentType === "home_service") {
      const newLat = updates.locationLat ?? appt.locationLat;
      const newLng = updates.locationLng ?? appt.locationLng;
      const newAddr = updates.addressText ?? appt.addressText;
      
      if (!newLat || !newLng || !newAddr) {
        throw new Error("موقع العميل مطلوب للخدمة المنزلية");
      }
    }
    
    // Recalculate total if delivery fee changed
    let totalAmount = appt.totalAmount;
    if (updates.deliveryFee !== undefined && updates.deliveryFee !== appt.deliveryFee) {
      totalAmount = appt.subtotalAmount + appt.vatAmount + updates.deliveryFee;
    }
    
    await db.update(appointments)
      .set({
        ...updates,
        totalAmount,
        updatedAt: new Date(),
      })
      .where(eq(appointments.id, id));
    
    return await this.getAppointment(id);
  }
  
  async confirmAppointment(id: number): Promise<AppointmentWithDetails | undefined> {
    const [appt] = await db.select().from(appointments).where(eq(appointments.id, id));
    if (!appt) return undefined;
    
    if (appt.status !== "pending") {
      throw new Error("لا يمكن تأكيد هذا الحجز في حالته الحالية");
    }
    
    await db.update(appointments)
      .set({ status: "confirmed", updatedAt: new Date() })
      .where(eq(appointments.id, id));
    
    return await this.getAppointment(id);
  }
  
  async cancelAppointment(id: number, reason?: string): Promise<AppointmentWithDetails | undefined> {
    const [appt] = await db.select().from(appointments).where(eq(appointments.id, id));
    if (!appt) return undefined;
    
    await db.update(appointments)
      .set({ 
        status: "canceled", 
        cancelledReason: reason || null,
        updatedAt: new Date() 
      })
      .where(eq(appointments.id, id));
    
    return await this.getAppointment(id);
  }
  
  // ========================================
  // CAFETERIA PRODUCTS
  // ========================================
  
  async getCafeteriaProducts(): Promise<CafeteriaProductWithUnit[]> {
    const allProducts = await db.select().from(cafeteriaProducts).orderBy(desc(cafeteriaProducts.createdAt));
    const allUnits = await db.select().from(units);
    
    return allProducts.map(product => ({
      ...product,
      unit: allUnits.find(u => u.id === product.unitId),
    }));
  }
  
  async getCafeteriaProduct(id: number): Promise<CafeteriaProductWithUnit | undefined> {
    const [product] = await db.select().from(cafeteriaProducts).where(eq(cafeteriaProducts.id, id));
    if (!product) return undefined;
    
    let unit: Unit | undefined;
    if (product.unitId) {
      [unit] = await db.select().from(units).where(eq(units.id, product.unitId));
    }
    
    return { ...product, unit };
  }
  
  async createCafeteriaProduct(product: InsertCafeteriaProduct): Promise<CafeteriaProduct> {
    const [newProduct] = await db.insert(cafeteriaProducts).values(product).returning();
    return newProduct;
  }
  
  async updateCafeteriaProduct(id: number, product: Partial<InsertCafeteriaProduct>): Promise<CafeteriaProduct | undefined> {
    const [updated] = await db.update(cafeteriaProducts)
      .set({ ...product, updatedAt: new Date() })
      .where(eq(cafeteriaProducts.id, id))
      .returning();
    return updated;
  }
  
  async deleteCafeteriaProduct(id: number): Promise<boolean> {
    // Soft delete by setting isActive to false
    const [updated] = await db.update(cafeteriaProducts)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(cafeteriaProducts.id, id))
      .returning();
    return !!updated;
  }
  
  // ========================================
  // CAFETERIA STOCK
  // ========================================
  
  async getCafeteriaStockBatches(productId?: number): Promise<(CafeteriaStockBatch & { cafeteriaProduct?: CafeteriaProduct })[]> {
    let batches: CafeteriaStockBatch[];
    if (productId) {
      batches = await db.select().from(cafeteriaStockBatches)
        .where(eq(cafeteriaStockBatches.cafeteriaProductId, productId))
        .orderBy(desc(cafeteriaStockBatches.createdAt));
    } else {
      batches = await db.select().from(cafeteriaStockBatches).orderBy(desc(cafeteriaStockBatches.createdAt));
    }
    
    const allProducts = await db.select().from(cafeteriaProducts);
    
    return batches.map(batch => ({
      ...batch,
      cafeteriaProduct: allProducts.find(p => p.id === batch.cafeteriaProductId),
    }));
  }
  
  async createCafeteriaStockBatch(batch: InsertCafeteriaStockBatch): Promise<CafeteriaStockBatch> {
    const [newBatch] = await db.insert(cafeteriaStockBatches).values(batch).returning();
    return newBatch;
  }
  
  async updateCafeteriaStockBatch(id: number, batch: Partial<InsertCafeteriaStockBatch>): Promise<CafeteriaStockBatch | undefined> {
    const [updated] = await db.update(cafeteriaStockBatches)
      .set({ ...batch, updatedAt: new Date() })
      .where(eq(cafeteriaStockBatches.id, id))
      .returning();
    return updated;
  }
  
  async getCafeteriaStockMovements(productId?: number): Promise<CafeteriaStockMovement[]> {
    if (productId) {
      return await db.select().from(cafeteriaStockMovements)
        .where(eq(cafeteriaStockMovements.cafeteriaProductId, productId))
        .orderBy(desc(cafeteriaStockMovements.movementDate));
    }
    return await db.select().from(cafeteriaStockMovements).orderBy(desc(cafeteriaStockMovements.movementDate));
  }
  
  async createCafeteriaStockMovement(movement: InsertCafeteriaStockMovement): Promise<CafeteriaStockMovement> {
    const [newMovement] = await db.insert(cafeteriaStockMovements).values(movement).returning();
    return newMovement;
  }
  
  // ========================================
  // CAFETERIA PURCHASE INVOICES
  // ========================================
  
  async getCafeteriaPurchaseInvoices(): Promise<CafeteriaPurchaseInvoiceWithDetails[]> {
    const invoices = await db.select().from(cafeteriaPurchaseInvoices).orderBy(desc(cafeteriaPurchaseInvoices.createdAt));
    const allSuppliers = await db.select().from(suppliers);
    const allLines = await db.select().from(cafeteriaPurchaseInvoiceLines);
    const allProducts = await db.select().from(cafeteriaProducts);
    const allPayments = await db.select().from(cafeteriaPurchasePayments);
    const allPaymentMethods = await db.select().from(paymentMethods);
    
    return invoices.map(invoice => ({
      ...invoice,
      supplier: allSuppliers.find(s => s.id === invoice.supplierId),
      lines: allLines
        .filter(l => l.cafeteriaPurchaseInvoiceId === invoice.id)
        .map(line => ({
          ...line,
          cafeteriaProduct: allProducts.find(p => p.id === line.cafeteriaProductId),
        })),
      payments: allPayments
        .filter(p => p.cafeteriaPurchaseInvoiceId === invoice.id)
        .map(payment => ({
          ...payment,
          paymentMethod: allPaymentMethods.find(m => m.id === payment.paymentMethodId),
        })),
    }));
  }
  
  async getCafeteriaPurchaseInvoice(id: number): Promise<CafeteriaPurchaseInvoiceWithDetails | undefined> {
    const [invoice] = await db.select().from(cafeteriaPurchaseInvoices).where(eq(cafeteriaPurchaseInvoices.id, id));
    if (!invoice) return undefined;
    
    const [supplier] = await db.select().from(suppliers).where(eq(suppliers.id, invoice.supplierId));
    const lines = await db.select().from(cafeteriaPurchaseInvoiceLines)
      .where(eq(cafeteriaPurchaseInvoiceLines.cafeteriaPurchaseInvoiceId, id));
    const allProducts = await db.select().from(cafeteriaProducts);
    const payments = await db.select().from(cafeteriaPurchasePayments)
      .where(eq(cafeteriaPurchasePayments.cafeteriaPurchaseInvoiceId, id));
    const allPaymentMethods = await db.select().from(paymentMethods);
    
    return {
      ...invoice,
      supplier,
      lines: lines.map(line => ({
        ...line,
        cafeteriaProduct: allProducts.find(p => p.id === line.cafeteriaProductId),
      })),
      payments: payments.map(payment => ({
        ...payment,
        paymentMethod: allPaymentMethods.find(m => m.id === payment.paymentMethodId),
      })),
    };
  }
  
  async createCafeteriaPurchaseInvoice(request: CreateCafeteriaPurchaseInvoiceRequest): Promise<CafeteriaPurchaseInvoiceWithDetails> {
    // Calculate line totals
    let subtotalAmount = 0;
    let vatAmount = 0;
    
    const calculatedLines = request.lines.map(line => {
      let lineSubtotal: number;
      let lineVatAmount: number;
      let lineTotal: number;
      
      const rawAmount = line.quantity * line.purchasePricePerUnit;
      
      if (line.vatType === "inclusive") {
        lineTotal = rawAmount;
        lineSubtotal = rawAmount / (1 + line.vatRate);
        lineVatAmount = lineTotal - lineSubtotal;
      } else if (line.vatType === "exclusive") {
        lineSubtotal = rawAmount;
        lineVatAmount = rawAmount * line.vatRate;
        lineTotal = lineSubtotal + lineVatAmount;
      } else {
        lineSubtotal = rawAmount;
        lineVatAmount = 0;
        lineTotal = rawAmount;
      }
      
      subtotalAmount += lineSubtotal;
      vatAmount += lineVatAmount;
      
      return {
        ...line,
        lineSubtotal,
        vatAmount: lineVatAmount,
        lineTotal,
      };
    });
    
    const totalAmount = subtotalAmount + vatAmount;
    
    // Create invoice
    const [invoice] = await db.insert(cafeteriaPurchaseInvoices).values({
      supplierId: request.supplierId,
      invoiceDate: request.invoiceDate,
      dueDate: request.dueDate,
      subtotalAmount,
      vatAmount,
      totalAmount,
      paymentStatus: "unpaid",
      notes: request.notes,
    }).returning();
    
    // Create lines
    const insertedLines: CafeteriaPurchaseInvoiceLine[] = [];
    for (const line of calculatedLines) {
      const [insertedLine] = await db.insert(cafeteriaPurchaseInvoiceLines).values({
        cafeteriaPurchaseInvoiceId: invoice.id,
        cafeteriaProductId: line.cafeteriaProductId,
        quantity: line.quantity,
        purchasePricePerUnit: line.purchasePricePerUnit,
        vatType: line.vatType,
        vatRate: line.vatRate,
        vatAmount: line.vatAmount,
        lineSubtotal: line.lineSubtotal,
        lineTotal: line.lineTotal,
        expiryDate: line.expiryDate,
      }).returning();
      insertedLines.push(insertedLine);
      
      // Create or update stock batch
      const existingBatches = await db.select().from(cafeteriaStockBatches)
        .where(and(
          eq(cafeteriaStockBatches.cafeteriaProductId, line.cafeteriaProductId),
          line.expiryDate ? eq(cafeteriaStockBatches.expiryDate, line.expiryDate) : isNull(cafeteriaStockBatches.expiryDate)
        ));
      
      if (existingBatches.length > 0) {
        const batch = existingBatches[0];
        await db.update(cafeteriaStockBatches)
          .set({
            quantityOnHand: batch.quantityOnHand + line.quantity,
            unitCost: line.purchasePricePerUnit,
            updatedAt: new Date(),
          })
          .where(eq(cafeteriaStockBatches.id, batch.id));
      } else {
        await db.insert(cafeteriaStockBatches).values({
          cafeteriaProductId: line.cafeteriaProductId,
          expiryDate: line.expiryDate,
          quantityOnHand: line.quantity,
          unitCost: line.purchasePricePerUnit,
        });
      }
      
      // Create stock movement
      await db.insert(cafeteriaStockMovements).values({
        cafeteriaProductId: line.cafeteriaProductId,
        movementDate: request.invoiceDate,
        movementType: "purchase",
        quantityIn: line.quantity,
        quantityOut: 0,
        unitCost: line.purchasePricePerUnit,
        referenceType: "CafeteriaPurchaseInvoice",
        referenceId: invoice.id,
      });
    }
    
    return this.getCafeteriaPurchaseInvoice(invoice.id) as Promise<CafeteriaPurchaseInvoiceWithDetails>;
  }
  
  // ========================================
  // CAFETERIA PURCHASE PAYMENTS
  // ========================================
  
  async getCafeteriaPurchasePayments(invoiceId?: number): Promise<(CafeteriaPurchasePayment & { paymentMethod?: PaymentMethod; supplier?: Supplier })[]> {
    let payments: CafeteriaPurchasePayment[];
    if (invoiceId) {
      payments = await db.select().from(cafeteriaPurchasePayments)
        .where(eq(cafeteriaPurchasePayments.cafeteriaPurchaseInvoiceId, invoiceId))
        .orderBy(desc(cafeteriaPurchasePayments.paymentDate));
    } else {
      payments = await db.select().from(cafeteriaPurchasePayments).orderBy(desc(cafeteriaPurchasePayments.paymentDate));
    }
    
    const allPaymentMethods = await db.select().from(paymentMethods);
    const allSuppliers = await db.select().from(suppliers);
    
    return payments.map(payment => ({
      ...payment,
      paymentMethod: allPaymentMethods.find(m => m.id === payment.paymentMethodId),
      supplier: allSuppliers.find(s => s.id === payment.supplierId),
    }));
  }
  
  async createCafeteriaPurchasePayment(payment: InsertCafeteriaPurchasePayment): Promise<CafeteriaPurchasePayment> {
    const [newPayment] = await db.insert(cafeteriaPurchasePayments).values(payment).returning();
    
    // Update invoice payment status if linked to an invoice
    if (payment.cafeteriaPurchaseInvoiceId) {
      const [invoice] = await db.select().from(cafeteriaPurchaseInvoices)
        .where(eq(cafeteriaPurchaseInvoices.id, payment.cafeteriaPurchaseInvoiceId));
      
      if (invoice) {
        const allPayments = await db.select().from(cafeteriaPurchasePayments)
          .where(eq(cafeteriaPurchasePayments.cafeteriaPurchaseInvoiceId, invoice.id));
        
        const totalPaid = allPayments.reduce((sum, p) => sum + p.amount, 0);
        
        let paymentStatus: string;
        if (totalPaid >= invoice.totalAmount) {
          paymentStatus = "paid";
        } else if (totalPaid > 0) {
          paymentStatus = "partially_paid";
        } else {
          paymentStatus = "unpaid";
        }
        
        await db.update(cafeteriaPurchaseInvoices)
          .set({ paymentStatus, updatedAt: new Date() })
          .where(eq(cafeteriaPurchaseInvoices.id, invoice.id));
      }
    }
    
    return newPayment;
  }
  
  // ========================================
  // CAFETERIA SALES ORDERS
  // ========================================
  
  private async generateCafeteriaOrderNumber(): Promise<string> {
    const [result] = await db.select({ count: sql<number>`count(*)` }).from(cafeteriaSalesOrders);
    const count = Number(result?.count || 0) + 1;
    return `CF-${count.toString().padStart(4, '0')}`;
  }
  
  async getCafeteriaSalesOrders(filters?: CafeteriaOrderFilters): Promise<CafeteriaSalesOrderWithDetails[]> {
    let query = db.select().from(cafeteriaSalesOrders);
    
    const conditions: any[] = [];
    if (filters?.status) {
      conditions.push(eq(cafeteriaSalesOrders.status, filters.status));
    }
    if (filters?.branchId) {
      conditions.push(eq(cafeteriaSalesOrders.branchId, filters.branchId));
    }
    if (filters?.createdByEmployeeId) {
      conditions.push(eq(cafeteriaSalesOrders.createdByEmployeeId, filters.createdByEmployeeId));
    }
    if (filters?.date) {
      conditions.push(sql`DATE(${cafeteriaSalesOrders.orderDate}) = ${filters.date}`);
    }
    
    const orders = conditions.length > 0
      ? await db.select().from(cafeteriaSalesOrders).where(and(...conditions)).orderBy(desc(cafeteriaSalesOrders.orderDate))
      : await db.select().from(cafeteriaSalesOrders).orderBy(desc(cafeteriaSalesOrders.orderDate));
    
    const allCustomers = await db.select().from(customers);
    const allEmployees = await db.select().from(employees);
    const allLines = await db.select().from(cafeteriaSalesOrderLines);
    const allProducts = await db.select().from(cafeteriaProducts);
    
    return orders.map(order => ({
      ...order,
      client: allCustomers.find(c => c.id === order.clientId),
      createdByEmployee: allEmployees.find(e => e.id === order.createdByEmployeeId),
      lines: allLines
        .filter(l => l.cafeteriaSalesOrderId === order.id)
        .map(line => ({
          ...line,
          cafeteriaProduct: allProducts.find(p => p.id === line.cafeteriaProductId),
        })),
    }));
  }
  
  async getCafeteriaSalesOrder(id: number): Promise<CafeteriaSalesOrderWithDetails | undefined> {
    const [order] = await db.select().from(cafeteriaSalesOrders).where(eq(cafeteriaSalesOrders.id, id));
    if (!order) return undefined;
    
    let client: Customer | undefined;
    if (order.clientId) {
      [client] = await db.select().from(customers).where(eq(customers.id, order.clientId));
    }
    
    const [createdByEmployee] = await db.select().from(employees).where(eq(employees.id, order.createdByEmployeeId));
    const lines = await db.select().from(cafeteriaSalesOrderLines)
      .where(eq(cafeteriaSalesOrderLines.cafeteriaSalesOrderId, id));
    const allProducts = await db.select().from(cafeteriaProducts);
    
    return {
      ...order,
      client,
      createdByEmployee,
      lines: lines.map(line => ({
        ...line,
        cafeteriaProduct: allProducts.find(p => p.id === line.cafeteriaProductId),
      })),
    };
  }
  
  async createCafeteriaSalesOrder(request: CreateCafeteriaSalesOrderRequest): Promise<CafeteriaSalesOrderWithDetails> {
    const orderNumber = await this.generateCafeteriaOrderNumber();
    
    // Calculate line totals
    let subtotalAmount = 0;
    let vatAmount = 0;
    
    const calculatedLines: Array<{
      cafeteriaProductId: number;
      quantity: number;
      unitPrice: number;
      vatType: string;
      vatRate: number;
      lineSubtotal: number;
      vatAmount: number;
      lineTotal: number;
    }> = [];
    
    for (const line of request.lines) {
      const product = await this.getCafeteriaProduct(line.cafeteriaProductId);
      if (!product) {
        throw new Error(`منتج الكافتيريا غير موجود: ${line.cafeteriaProductId}`);
      }
      
      const unitPrice = product.defaultSellPrice || 0;
      const vatType = "inclusive";
      const vatRate = 0.15;
      
      const rawAmount = line.quantity * unitPrice;
      let lineSubtotal: number;
      let lineVatAmount: number;
      let lineTotal: number;
      
      if (vatType === "inclusive") {
        lineTotal = rawAmount;
        lineSubtotal = rawAmount / (1 + vatRate);
        lineVatAmount = lineTotal - lineSubtotal;
      } else {
        lineSubtotal = rawAmount;
        lineVatAmount = rawAmount * vatRate;
        lineTotal = lineSubtotal + lineVatAmount;
      }
      
      subtotalAmount += lineSubtotal;
      vatAmount += lineVatAmount;
      
      calculatedLines.push({
        cafeteriaProductId: line.cafeteriaProductId,
        quantity: line.quantity,
        unitPrice,
        vatType,
        vatRate,
        lineSubtotal,
        vatAmount: lineVatAmount,
        lineTotal,
      });
    }
    
    const totalAmount = subtotalAmount + vatAmount;
    
    // Create order
    const [order] = await db.insert(cafeteriaSalesOrders).values({
      orderNumber,
      orderDate: new Date(),
      source: "pos",
      clientId: request.clientId,
      createdByEmployeeId: request.createdByEmployeeId,
      branchId: request.branchId,
      status: "new",
      notes: request.notes,
      subtotalAmount,
      vatAmount,
      totalAmount,
    }).returning();
    
    // Create lines
    for (const line of calculatedLines) {
      await db.insert(cafeteriaSalesOrderLines).values({
        cafeteriaSalesOrderId: order.id,
        cafeteriaProductId: line.cafeteriaProductId,
        quantity: line.quantity,
        unitPrice: line.unitPrice,
        vatType: line.vatType,
        vatRate: line.vatRate,
        lineSubtotal: line.lineSubtotal,
        vatAmount: line.vatAmount,
        lineTotal: line.lineTotal,
      });
    }
    
    return this.getCafeteriaSalesOrder(order.id) as Promise<CafeteriaSalesOrderWithDetails>;
  }
  
  async updateCafeteriaSalesOrder(id: number, updates: { status?: string; notes?: string | null }): Promise<CafeteriaSalesOrderWithDetails | undefined> {
    const order = await this.getCafeteriaSalesOrder(id);
    if (!order) return undefined;
    
    // If completing order, process stock consumption
    if (updates.status === "completed" && order.status !== "completed") {
      await this.processCafeteriaOrderCompletion(order);
    }
    
    const [updated] = await db.update(cafeteriaSalesOrders)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(cafeteriaSalesOrders.id, id))
      .returning();
    
    return this.getCafeteriaSalesOrder(id);
  }
  
  private async processCafeteriaOrderCompletion(order: CafeteriaSalesOrderWithDetails): Promise<void> {
    const today = new Date().toISOString().split('T')[0];
    
    for (const line of order.lines) {
      // Create stock movement for sale
      await db.insert(cafeteriaStockMovements).values({
        cafeteriaProductId: line.cafeteriaProductId,
        movementDate: today,
        movementType: "sale",
        quantityIn: 0,
        quantityOut: line.quantity,
        referenceType: "CafeteriaSaleOrder",
        referenceId: order.id,
      });
      
      // Decrease stock from batches (FIFO)
      await this.decreaseCafeteriaStockFIFO(line.cafeteriaProductId, line.quantity);
    }
  }
  
  private async decreaseCafeteriaStockFIFO(productId: number, quantity: number): Promise<void> {
    const batches = await db.select()
      .from(cafeteriaStockBatches)
      .where(and(
        eq(cafeteriaStockBatches.cafeteriaProductId, productId),
        sql`${cafeteriaStockBatches.quantityOnHand} > 0`
      ))
      .orderBy(asc(cafeteriaStockBatches.expiryDate), asc(cafeteriaStockBatches.createdAt));
    
    let remainingQuantity = quantity;
    
    for (const batch of batches) {
      if (remainingQuantity <= 0) break;
      
      const deductQuantity = Math.min(batch.quantityOnHand, remainingQuantity);
      const newQuantity = batch.quantityOnHand - deductQuantity;
      
      await db.update(cafeteriaStockBatches)
        .set({ quantityOnHand: newQuantity, updatedAt: new Date() })
        .where(eq(cafeteriaStockBatches.id, batch.id));
      
      remainingQuantity -= deductQuantity;
    }
  }
  
  // ========================================
  // CAFETERIA REPORTS
  // ========================================
  
  async getCafeteriaInventorySummary(): Promise<CafeteriaInventorySummaryItem[]> {
    const productsList = await db.select().from(cafeteriaProducts).where(eq(cafeteriaProducts.isActive, true));
    const batches = await db.select().from(cafeteriaStockBatches);
    const unitsList = await db.select().from(units);
    
    return productsList.map(product => {
      const productBatches = batches.filter(b => b.cafeteriaProductId === product.id);
      const quantityOnHand = productBatches.reduce((sum, b) => sum + b.quantityOnHand, 0);
      const totalCost = productBatches.reduce((sum, b) => sum + (b.quantityOnHand * b.unitCost), 0);
      const unit = unitsList.find(u => u.id === product.unitId);
      
      return {
        cafeteriaProductId: product.id,
        productNameAr: product.nameAr,
        sku: product.sku || undefined,
        quantityOnHand,
        totalCost,
        unitNameAr: unit?.nameAr,
      };
    });
  }
  
  async getCafeteriaInventoryMovements(productId: number): Promise<CafeteriaStockMovementReportItem[]> {
    const movements = await db.select().from(cafeteriaStockMovements)
      .where(eq(cafeteriaStockMovements.cafeteriaProductId, productId))
      .orderBy(asc(cafeteriaStockMovements.movementDate), asc(cafeteriaStockMovements.id));
    
    let runningBalance = 0;
    
    return movements.map(m => {
      runningBalance = runningBalance + m.quantityIn - m.quantityOut;
      return {
        id: m.id,
        movementDate: m.movementDate,
        movementType: m.movementType,
        quantityIn: m.quantityIn,
        quantityOut: m.quantityOut,
        unitCost: m.unitCost || undefined,
        referenceType: m.referenceType || undefined,
        referenceId: m.referenceId || undefined,
        balance: runningBalance,
      };
    });
  }
  
  // ========================================
  // CAFETERIA EMPLOYEE CUSTODIES (عهد موظفي الكافتيريا)
  // ========================================
  
  async getCafeteriaEmployeeCustodies(filters?: { employeeId?: number; status?: string }): Promise<CafeteriaEmployeeCustodyWithDetails[]> {
    let custodies: CafeteriaEmployeeCustody[];
    const conditions: any[] = [];
    
    if (filters?.employeeId) {
      conditions.push(eq(cafeteriaEmployeeCustodies.employeeId, filters.employeeId));
    }
    if (filters?.status) {
      conditions.push(eq(cafeteriaEmployeeCustodies.status, filters.status));
    }
    
    if (conditions.length > 0) {
      custodies = await db.select().from(cafeteriaEmployeeCustodies)
        .where(and(...conditions))
        .orderBy(desc(cafeteriaEmployeeCustodies.createdAt));
    } else {
      custodies = await db.select().from(cafeteriaEmployeeCustodies)
        .orderBy(desc(cafeteriaEmployeeCustodies.createdAt));
    }
    
    const allEmployees = await db.select().from(employees);
    const allProducts = await db.select().from(cafeteriaProducts);
    
    return custodies.map(custody => ({
      ...custody,
      employee: allEmployees.find(e => e.id === custody.employeeId),
      cafeteriaProduct: allProducts.find(p => p.id === custody.cafeteriaProductId),
      createdByEmployee: custody.createdByEmployeeId ? allEmployees.find(e => e.id === custody.createdByEmployeeId) : undefined,
      closedByEmployee: custody.closedByEmployeeId ? allEmployees.find(e => e.id === custody.closedByEmployeeId) : undefined,
    }));
  }
  
  async getCafeteriaEmployeeCustody(id: number): Promise<CafeteriaEmployeeCustodyWithDetails | undefined> {
    const [custody] = await db.select().from(cafeteriaEmployeeCustodies).where(eq(cafeteriaEmployeeCustodies.id, id));
    if (!custody) return undefined;
    
    const allEmployees = await db.select().from(employees);
    const allProducts = await db.select().from(cafeteriaProducts);
    
    return {
      ...custody,
      employee: allEmployees.find(e => e.id === custody.employeeId),
      cafeteriaProduct: allProducts.find(p => p.id === custody.cafeteriaProductId),
      createdByEmployee: custody.createdByEmployeeId ? allEmployees.find(e => e.id === custody.createdByEmployeeId) : undefined,
      closedByEmployee: custody.closedByEmployeeId ? allEmployees.find(e => e.id === custody.closedByEmployeeId) : undefined,
    };
  }
  
  async createCafeteriaEmployeeCustody(custody: InsertCafeteriaEmployeeCustody): Promise<CafeteriaEmployeeCustodyWithDetails> {
    // Calculate average cost from stock batches
    const batches = await db.select().from(cafeteriaStockBatches)
      .where(and(
        eq(cafeteriaStockBatches.cafeteriaProductId, custody.cafeteriaProductId),
        sql`${cafeteriaStockBatches.quantityOnHand} > 0`
      ));
    
    let unitCost = custody.unitCost || 0;
    if (!custody.unitCost && batches.length > 0) {
      const totalQty = batches.reduce((sum, b) => sum + b.quantityOnHand, 0);
      const totalCost = batches.reduce((sum, b) => sum + (b.quantityOnHand * b.unitCost), 0);
      unitCost = totalQty > 0 ? totalCost / totalQty : 0;
    }
    
    const totalCost = custody.quantity * unitCost;
    
    const [newCustody] = await db.insert(cafeteriaEmployeeCustodies).values({
      ...custody,
      unitCost,
      totalCost,
    }).returning();
    
    // Record stock movement for custody issue
    const today = new Date().toISOString().split('T')[0];
    await db.insert(cafeteriaStockMovements).values({
      cafeteriaProductId: custody.cafeteriaProductId,
      movementDate: today,
      movementType: "custody_issue",
      quantityIn: 0,
      quantityOut: custody.quantity,
      unitCost,
      referenceType: "CafeteriaEmployeeCustody",
      referenceId: newCustody.id,
    });
    
    // Decrease stock using FIFO
    await this.decreaseCafeteriaStockFIFO(custody.cafeteriaProductId, custody.quantity);
    
    return this.getCafeteriaEmployeeCustody(newCustody.id) as Promise<CafeteriaEmployeeCustodyWithDetails>;
  }
  
  async closeCafeteriaEmployeeCustody(
    id: number,
    closureData: {
      closedByEmployeeId: number;
      closureReason: string;
      returnedQuantity?: number;
      closureNotes?: string;
    }
  ): Promise<CafeteriaEmployeeCustodyWithDetails | undefined> {
    const custody = await this.getCafeteriaEmployeeCustody(id);
    if (!custody) return undefined;
    
    const today = new Date().toISOString().split('T')[0];
    
    // Determine new status based on closure reason
    let newStatus: string;
    const returnedQty = closureData.returnedQuantity || 0;
    
    if (closureData.closureReason === "returned") {
      if (returnedQty >= custody.quantity) {
        newStatus = "returned";
      } else if (returnedQty > 0) {
        newStatus = "partially_returned";
      } else {
        newStatus = "returned";
      }
      
      // If items returned, add back to stock
      if (returnedQty > 0) {
        await db.insert(cafeteriaStockMovements).values({
          cafeteriaProductId: custody.cafeteriaProductId,
          movementDate: today,
          movementType: "custody_return",
          quantityIn: returnedQty,
          quantityOut: 0,
          unitCost: custody.unitCost,
          referenceType: "CafeteriaEmployeeCustody",
          referenceId: id,
        });
        
        // Add quantity back to stock batch
        const existingBatches = await db.select().from(cafeteriaStockBatches)
          .where(eq(cafeteriaStockBatches.cafeteriaProductId, custody.cafeteriaProductId))
          .orderBy(desc(cafeteriaStockBatches.createdAt))
          .limit(1);
        
        if (existingBatches.length > 0) {
          await db.update(cafeteriaStockBatches)
            .set({ 
              quantityOnHand: existingBatches[0].quantityOnHand + returnedQty,
              updatedAt: new Date()
            })
            .where(eq(cafeteriaStockBatches.id, existingBatches[0].id));
        } else {
          await db.insert(cafeteriaStockBatches).values({
            cafeteriaProductId: custody.cafeteriaProductId,
            quantityOnHand: returnedQty,
            unitCost: custody.unitCost,
          });
        }
      }
    } else if (closureData.closureReason === "consumed") {
      newStatus = "consumed";
    } else if (closureData.closureReason === "lost") {
      newStatus = "lost";
    } else if (closureData.closureReason === "damaged") {
      newStatus = "damaged";
    } else {
      newStatus = "consumed";
    }
    
    const [updated] = await db.update(cafeteriaEmployeeCustodies)
      .set({
        status: newStatus,
        closedDate: today,
        closedByEmployeeId: closureData.closedByEmployeeId,
        closureReason: closureData.closureReason,
        returnedQuantity: returnedQty || null,
        closureNotes: closureData.closureNotes || null,
        updatedAt: new Date(),
      })
      .where(eq(cafeteriaEmployeeCustodies.id, id))
      .returning();
    
    return this.getCafeteriaEmployeeCustody(id);
  }
  
  async deleteCafeteriaEmployeeCustody(id: number): Promise<boolean> {
    const custody = await this.getCafeteriaEmployeeCustody(id);
    if (!custody || custody.status !== "open") return false;
    
    // Reverse the stock movement - return items to stock
    const today = new Date().toISOString().split('T')[0];
    
    await db.insert(cafeteriaStockMovements).values({
      cafeteriaProductId: custody.cafeteriaProductId,
      movementDate: today,
      movementType: "custody_return",
      quantityIn: custody.quantity,
      quantityOut: 0,
      unitCost: custody.unitCost,
      referenceType: "CafeteriaEmployeeCustody",
      referenceId: id,
    });
    
    // Add quantity back to stock batch
    const existingBatches = await db.select().from(cafeteriaStockBatches)
      .where(eq(cafeteriaStockBatches.cafeteriaProductId, custody.cafeteriaProductId))
      .orderBy(desc(cafeteriaStockBatches.createdAt))
      .limit(1);
    
    if (existingBatches.length > 0) {
      await db.update(cafeteriaStockBatches)
        .set({ 
          quantityOnHand: existingBatches[0].quantityOnHand + custody.quantity,
          updatedAt: new Date()
        })
        .where(eq(cafeteriaStockBatches.id, existingBatches[0].id));
    }
    
    // Delete the custody record
    await db.delete(cafeteriaEmployeeCustodies).where(eq(cafeteriaEmployeeCustodies.id, id));
    return true;
  }
  
  // ========================================
  // CAFETERIA SALES RETURNS (مرتجعات مبيعات الكافتيريا)
  // ========================================
  
  async getCafeteriaSalesReturns(orderId?: number): Promise<CafeteriaSalesReturnWithDetails[]> {
    let returns: CafeteriaSalesReturn[];
    
    if (orderId) {
      returns = await db.select().from(cafeteriaSalesReturns)
        .where(eq(cafeteriaSalesReturns.cafeteriaSalesOrderId, orderId))
        .orderBy(desc(cafeteriaSalesReturns.createdAt));
    } else {
      returns = await db.select().from(cafeteriaSalesReturns)
        .orderBy(desc(cafeteriaSalesReturns.createdAt));
    }
    
    if (returns.length === 0) return [];
    
    const allOrders = await db.select().from(cafeteriaSalesOrders);
    const allEmployees = await db.select().from(employees);
    const allLines = await db.select().from(cafeteriaSalesReturnLines);
    const allOrderLines = await db.select().from(cafeteriaSalesOrderLines);
    const allProducts = await db.select().from(cafeteriaProducts);
    
    return returns.map(ret => ({
      ...ret,
      cafeteriaSalesOrder: allOrders.find(o => o.id === ret.cafeteriaSalesOrderId),
      processedByEmployee: ret.processedByEmployeeId ? allEmployees.find(e => e.id === ret.processedByEmployeeId) : undefined,
      lines: allLines
        .filter(l => l.returnId === ret.id)
        .map(l => ({
          ...l,
          orderLine: allOrderLines.find(ol => ol.id === l.cafeteriaSalesOrderLineId),
          cafeteriaProduct: allProducts.find(p => {
            const orderLine = allOrderLines.find(ol => ol.id === l.cafeteriaSalesOrderLineId);
            return orderLine && p.id === orderLine.cafeteriaProductId;
          }),
        })),
    }));
  }
  
  async getCafeteriaSalesReturn(id: number): Promise<CafeteriaSalesReturnWithDetails | undefined> {
    const [ret] = await db.select().from(cafeteriaSalesReturns).where(eq(cafeteriaSalesReturns.id, id));
    if (!ret) return undefined;
    
    const allOrders = await db.select().from(cafeteriaSalesOrders);
    const allEmployees = await db.select().from(employees);
    const allLines = await db.select().from(cafeteriaSalesReturnLines).where(eq(cafeteriaSalesReturnLines.returnId, id));
    const allOrderLines = await db.select().from(cafeteriaSalesOrderLines);
    const allProducts = await db.select().from(cafeteriaProducts);
    
    return {
      ...ret,
      cafeteriaSalesOrder: allOrders.find(o => o.id === ret.cafeteriaSalesOrderId),
      processedByEmployee: ret.processedByEmployeeId ? allEmployees.find(e => e.id === ret.processedByEmployeeId) : undefined,
      lines: allLines.map(l => ({
        ...l,
        orderLine: allOrderLines.find(ol => ol.id === l.cafeteriaSalesOrderLineId),
        cafeteriaProduct: allProducts.find(p => {
          const orderLine = allOrderLines.find(ol => ol.id === l.cafeteriaSalesOrderLineId);
          return orderLine && p.id === orderLine.cafeteriaProductId;
        }),
      })),
    };
  }
  
  async createCafeteriaSalesCancellationReturn(orderId: number, reason: string, processedByEmployeeId?: number): Promise<CafeteriaSalesReturnWithDetails> {
    const order = await this.getCafeteriaSalesOrder(orderId);
    if (!order) throw new Error("الطلب غير موجود");
    if (order.status === "canceled") throw new Error("الطلب ملغي بالفعل");
    
    // Generate return number (CF-RET-xxxx)
    const lastReturn = await db.select().from(cafeteriaSalesReturns).orderBy(desc(cafeteriaSalesReturns.id)).limit(1);
    const lastReturnNumber = lastReturn.length > 0 ? parseInt(lastReturn[0].returnNumber.split('-')[2] || '0') : 0;
    const returnNumber = `CF-RET-${String(lastReturnNumber + 1).padStart(4, '0')}`;
    
    // Generate credit note number (CF-CN-xxxx)
    const lastCNNumber = lastReturn.length > 0 && lastReturn[0].creditNoteNumber
      ? parseInt(lastReturn[0].creditNoteNumber.split('-')[2] || '0') 
      : 0;
    const creditNoteNumber = `CF-CN-${String(lastCNNumber + 1).padStart(4, '0')}`;
    
    // Create return header
    const [newReturn] = await db.insert(cafeteriaSalesReturns).values({
      cafeteriaSalesOrderId: orderId,
      returnNumber,
      creditNoteNumber,
      returnDate: new Date(),
      processedByEmployeeId: processedByEmployeeId || order.createdByEmployeeId || null,
      reason,
      returnType: "cancellation",
      originalInvoiceNumber: order.orderNumber,
      subtotalAmount: order.subtotalAmount,
      vatAmount: order.vatAmount,
      refundAmount: order.totalAmount,
      notes: `إشعار دائن تلقائي - إلغاء طلب كافتيريا رقم ${order.orderNumber}`,
    }).returning();
    
    // Create return lines
    for (const line of order.lines) {
      await db.insert(cafeteriaSalesReturnLines).values({
        returnId: newReturn.id,
        cafeteriaSalesOrderLineId: line.id,
        quantity: line.quantity,
        refundAmount: line.lineTotal,
        reason,
      });
      
      // Restore stock (add quantity back)
      const today = new Date().toISOString().split('T')[0];
      await db.insert(cafeteriaStockMovements).values({
        cafeteriaProductId: line.cafeteriaProductId,
        movementDate: today,
        movementType: "return_in",
        quantityIn: line.quantity,
        quantityOut: 0,
        unitCost: line.unitPrice,
        referenceType: "CafeteriaSalesReturn",
        referenceId: newReturn.id,
      });
      
      // Add quantity back to stock batch
      const existingBatch = await db.select().from(cafeteriaStockBatches)
        .where(eq(cafeteriaStockBatches.cafeteriaProductId, line.cafeteriaProductId))
        .orderBy(desc(cafeteriaStockBatches.createdAt))
        .limit(1);
      
      if (existingBatch.length > 0) {
        await db.update(cafeteriaStockBatches)
          .set({ 
            quantityOnHand: existingBatch[0].quantityOnHand + line.quantity,
            updatedAt: new Date()
          })
          .where(eq(cafeteriaStockBatches.id, existingBatch[0].id));
      }
    }
    
    // Update order status to canceled
    await db.update(cafeteriaSalesOrders)
      .set({ status: "canceled", updatedAt: new Date() })
      .where(eq(cafeteriaSalesOrders.id, orderId));
    
    return this.getCafeteriaSalesReturn(newReturn.id) as Promise<CafeteriaSalesReturnWithDetails>;
  }
  
  // ========================================
  // SYSTEM SETTINGS
  // ========================================
  
  async getSystemSetting(key: string): Promise<SystemSetting | undefined> {
    const [setting] = await db.select().from(systemSettings).where(eq(systemSettings.key, key));
    return setting;
  }
  
  async setSystemSetting(key: string, value: string, description?: string, valueType?: string): Promise<SystemSetting> {
    const existing = await this.getSystemSetting(key);
    
    if (existing) {
      const [updated] = await db.update(systemSettings)
        .set({ 
          value,
          description: description ?? existing.description,
          valueType: valueType ?? existing.valueType,
          updatedAt: new Date() 
        })
        .where(eq(systemSettings.key, key))
        .returning();
      return updated;
    } else {
      const [created] = await db.insert(systemSettings).values({
        key,
        value,
        description: description || null,
        valueType: valueType || "string",
      }).returning();
      return created;
    }
  }
  
  async getSystemSettings(): Promise<SystemSetting[]> {
    return await db.select().from(systemSettings).orderBy(asc(systemSettings.key));
  }
  
  // ========================================
  // MARKETING - PACKAGES (الحزم)
  // ========================================
  
  async getPackages(filters?: PackageFilters): Promise<PackageWithDetails[]> {
    const conditions: any[] = [];
    
    if (filters?.isActive !== undefined) {
      conditions.push(eq(packages.isActive, filters.isActive));
    }
    
    if (filters?.validToday) {
      const today = new Date().toISOString().split('T')[0];
      conditions.push(eq(packages.isActive, true));
      conditions.push(lte(packages.startDate, today));
      conditions.push(gte(packages.endDate, today));
    }
    
    let packageRows: Package[];
    if (conditions.length > 0) {
      packageRows = await db.select().from(packages).where(and(...conditions)).orderBy(desc(packages.createdAt));
    } else {
      packageRows = await db.select().from(packages).orderBy(desc(packages.createdAt));
    }
    
    const result: PackageWithDetails[] = [];
    for (const pkg of packageRows) {
      const full = await this.getPackage(pkg.id);
      if (full) result.push(full);
    }
    
    return result;
  }
  
  async getPackage(id: number): Promise<PackageWithDetails | undefined> {
    const [pkg] = await db.select().from(packages).where(eq(packages.id, id));
    if (!pkg) return undefined;
    
    // Get services with details
    const serviceRows = await db.select().from(packageServices).where(eq(packageServices.packageId, id));
    const servicesWithDetails: PackageServiceWithDetails[] = [];
    
    for (const svc of serviceRows) {
      const [service] = await db.select().from(services).where(eq(services.id, svc.serviceId));
      servicesWithDetails.push({
        ...svc,
        service,
      });
    }
    
    // Get products with details
    const productRows = await db.select().from(packageProducts).where(eq(packageProducts.packageId, id));
    const productsWithDetails: PackageProductWithDetails[] = [];
    
    for (const prod of productRows) {
      const [product] = await db.select().from(products).where(eq(products.id, prod.productId));
      productsWithDetails.push({
        ...prod,
        product,
      });
    }
    
    return {
      ...pkg,
      services: servicesWithDetails,
      products: productsWithDetails,
    };
  }
  
  async getValidPackage(id: number): Promise<PackageWithDetails | undefined> {
    const pkg = await this.getPackage(id);
    if (!pkg) return undefined;
    
    const today = new Date().toISOString().split('T')[0];
    if (!pkg.isActive || pkg.startDate > today || pkg.endDate < today) {
      return undefined;
    }
    
    return pkg;
  }
  
  async createPackage(request: CreatePackageRequest): Promise<PackageWithDetails> {
    const [newPackage] = await db.insert(packages).values({
      nameAr: request.nameAr,
      imageUrl: request.imageUrl || null,
      startDate: request.startDate,
      endDate: request.endDate,
      descriptionAr: request.descriptionAr || null,
      isActive: request.isActive ?? true,
    }).returning();
    
    // Insert services
    for (const svc of request.services) {
      await db.insert(packageServices).values({
        packageId: newPackage.id,
        serviceId: svc.serviceId,
        quantity: svc.quantity ?? 1,
        packagePrice: svc.packagePrice,
      });
    }
    
    // Insert products
    if (request.products && request.products.length > 0) {
      for (const prod of request.products) {
        await db.insert(packageProducts).values({
          packageId: newPackage.id,
          productId: prod.productId,
          quantity: prod.quantity ?? 1,
          packagePrice: prod.packagePrice,
        });
      }
    }
    
    return await this.getPackage(newPackage.id) as PackageWithDetails;
  }
  
  async updatePackage(id: number, request: Partial<CreatePackageRequest>): Promise<PackageWithDetails | undefined> {
    const existing = await this.getPackage(id);
    if (!existing) return undefined;
    
    // Update package header
    const updateData: Partial<Package> = { updatedAt: new Date() };
    if (request.nameAr !== undefined) updateData.nameAr = request.nameAr;
    if (request.imageUrl !== undefined) updateData.imageUrl = request.imageUrl;
    if (request.startDate !== undefined) updateData.startDate = request.startDate;
    if (request.endDate !== undefined) updateData.endDate = request.endDate;
    if (request.descriptionAr !== undefined) updateData.descriptionAr = request.descriptionAr;
    if (request.isActive !== undefined) updateData.isActive = request.isActive;
    
    await db.update(packages).set(updateData).where(eq(packages.id, id));
    
    // Replace services if provided
    if (request.services) {
      await db.delete(packageServices).where(eq(packageServices.packageId, id));
      for (const svc of request.services) {
        await db.insert(packageServices).values({
          packageId: id,
          serviceId: svc.serviceId,
          quantity: svc.quantity ?? 1,
          packagePrice: svc.packagePrice,
        });
      }
    }
    
    // Replace products if provided
    if (request.products) {
      await db.delete(packageProducts).where(eq(packageProducts.packageId, id));
      for (const prod of request.products) {
        await db.insert(packageProducts).values({
          packageId: id,
          productId: prod.productId,
          quantity: prod.quantity ?? 1,
          packagePrice: prod.packagePrice,
        });
      }
    }
    
    return await this.getPackage(id);
  }
  
  async deletePackage(id: number): Promise<boolean> {
    // Soft delete by setting isActive = false
    const [updated] = await db.update(packages)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(packages.id, id))
      .returning();
    return !!updated;
  }
  
  // ========================================
  // MARKETING - COUPONS (الكوبونات)
  // ========================================
  
  async getCoupons(filters?: CouponFilters): Promise<CouponWithDetails[]> {
    const conditions: any[] = [];
    
    if (filters?.isActive !== undefined) {
      conditions.push(eq(coupons.isActive, filters.isActive));
    }
    
    if (filters?.validToday) {
      const today = new Date().toISOString().split('T')[0];
      conditions.push(eq(coupons.isActive, true));
      conditions.push(lte(coupons.startDate, today));
      conditions.push(gte(coupons.endDate, today));
    }
    
    let couponRows: Coupon[];
    if (conditions.length > 0) {
      couponRows = await db.select().from(coupons).where(and(...conditions)).orderBy(desc(coupons.createdAt));
    } else {
      couponRows = await db.select().from(coupons).orderBy(desc(coupons.createdAt));
    }
    
    const result: CouponWithDetails[] = [];
    for (const cpn of couponRows) {
      const full = await this.getCoupon(cpn.id);
      if (full) result.push(full);
    }
    
    return result;
  }
  
  async getCoupon(id: number): Promise<CouponWithDetails | undefined> {
    const [cpn] = await db.select().from(coupons).where(eq(coupons.id, id));
    if (!cpn) return undefined;
    
    // Get allowed services
    const serviceRows = await db.select().from(couponServices).where(eq(couponServices.couponId, id));
    const allowedServices: CouponServiceWithDetails[] = [];
    
    for (const svc of serviceRows) {
      const [service] = await db.select().from(services).where(eq(services.id, svc.serviceId));
      allowedServices.push({
        ...svc,
        service,
      });
    }
    
    // Get allowed products
    const productRows = await db.select().from(couponProducts).where(eq(couponProducts.couponId, id));
    const allowedProducts: CouponProductWithDetails[] = [];
    
    for (const prod of productRows) {
      const [product] = await db.select().from(products).where(eq(products.id, prod.productId));
      allowedProducts.push({
        ...prod,
        product,
      });
    }
    
    return {
      ...cpn,
      allowedServices,
      allowedProducts,
    };
  }
  
  async getCouponByCode(code: string): Promise<CouponWithDetails | undefined> {
    const [cpn] = await db.select().from(coupons).where(eq(coupons.code, code));
    if (!cpn) return undefined;
    return await this.getCoupon(cpn.id);
  }
  
  async getValidCoupon(id: number): Promise<CouponWithDetails | undefined> {
    const cpn = await this.getCoupon(id);
    if (!cpn) return undefined;
    
    const today = new Date().toISOString().split('T')[0];
    if (!cpn.isActive || cpn.startDate > today || cpn.endDate < today) {
      return undefined;
    }
    
    return cpn;
  }
  
  async getValidCouponByCode(code: string): Promise<CouponWithDetails | undefined> {
    const cpn = await this.getCouponByCode(code);
    if (!cpn) return undefined;
    
    const today = new Date().toISOString().split('T')[0];
    if (!cpn.isActive || cpn.startDate > today || cpn.endDate < today) {
      return undefined;
    }
    
    return cpn;
  }
  
  async createCoupon(request: CreateCouponRequest): Promise<CouponWithDetails> {
    const [newCoupon] = await db.insert(coupons).values({
      nameAr: request.nameAr,
      code: request.code,
      descriptionAr: request.descriptionAr || null,
      discountType: request.discountType,
      discountValue: request.discountValue,
      appliesToServices: request.appliesToServices ?? true,
      appliesToProducts: request.appliesToProducts ?? false,
      startDate: request.startDate,
      endDate: request.endDate,
      isActive: request.isActive ?? true,
    }).returning();
    
    // Insert allowed services
    if (request.serviceIds && request.serviceIds.length > 0) {
      for (const serviceId of request.serviceIds) {
        await db.insert(couponServices).values({
          couponId: newCoupon.id,
          serviceId,
        });
      }
    }
    
    // Insert allowed products
    if (request.productIds && request.productIds.length > 0) {
      for (const productId of request.productIds) {
        await db.insert(couponProducts).values({
          couponId: newCoupon.id,
          productId,
        });
      }
    }
    
    return await this.getCoupon(newCoupon.id) as CouponWithDetails;
  }
  
  async updateCoupon(id: number, request: Partial<CreateCouponRequest>): Promise<CouponWithDetails | undefined> {
    const existing = await this.getCoupon(id);
    if (!existing) return undefined;
    
    // Update coupon header
    const updateData: Partial<Coupon> = { updatedAt: new Date() };
    if (request.nameAr !== undefined) updateData.nameAr = request.nameAr;
    if (request.code !== undefined) updateData.code = request.code;
    if (request.descriptionAr !== undefined) updateData.descriptionAr = request.descriptionAr;
    if (request.discountType !== undefined) updateData.discountType = request.discountType;
    if (request.discountValue !== undefined) updateData.discountValue = request.discountValue;
    if (request.appliesToServices !== undefined) updateData.appliesToServices = request.appliesToServices;
    if (request.appliesToProducts !== undefined) updateData.appliesToProducts = request.appliesToProducts;
    if (request.startDate !== undefined) updateData.startDate = request.startDate;
    if (request.endDate !== undefined) updateData.endDate = request.endDate;
    if (request.isActive !== undefined) updateData.isActive = request.isActive;
    
    await db.update(coupons).set(updateData).where(eq(coupons.id, id));
    
    // Replace service restrictions if provided
    if (request.serviceIds !== undefined) {
      await db.delete(couponServices).where(eq(couponServices.couponId, id));
      for (const serviceId of request.serviceIds) {
        await db.insert(couponServices).values({
          couponId: id,
          serviceId,
        });
      }
    }
    
    // Replace product restrictions if provided
    if (request.productIds !== undefined) {
      await db.delete(couponProducts).where(eq(couponProducts.couponId, id));
      for (const productId of request.productIds) {
        await db.insert(couponProducts).values({
          couponId: id,
          productId,
        });
      }
    }
    
    return await this.getCoupon(id);
  }
  
  async deleteCoupon(id: number): Promise<boolean> {
    // Soft delete by setting isActive = false
    const [updated] = await db.update(coupons)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(coupons.id, id))
      .returning();
    return !!updated;
  }
  
  // ========================================
  // CUSTOMER REVIEWS (تقييمات العملاء)
  // ========================================

  async getReviewReasons(filters?: { reasonType?: string; isActive?: boolean }): Promise<ReviewReason[]> {
    let conditions = [];
    if (filters?.reasonType) {
      conditions.push(eq(reviewReasons.reasonType, filters.reasonType));
    }
    if (filters?.isActive !== undefined) {
      conditions.push(eq(reviewReasons.isActive, filters.isActive));
    }
    if (conditions.length > 0) {
      return db.select().from(reviewReasons).where(and(...conditions)).orderBy(asc(reviewReasons.sortOrder));
    }
    return db.select().from(reviewReasons).orderBy(asc(reviewReasons.sortOrder));
  }

  async getReviewReason(id: number): Promise<ReviewReason | undefined> {
    const [reason] = await db.select().from(reviewReasons).where(eq(reviewReasons.id, id));
    return reason;
  }

  async createReviewReason(reason: InsertReviewReason): Promise<ReviewReason> {
    const [created] = await db.insert(reviewReasons).values(reason).returning();
    return created;
  }

  async updateReviewReason(id: number, reason: Partial<InsertReviewReason>): Promise<ReviewReason | undefined> {
    const [updated] = await db.update(reviewReasons).set(reason).where(eq(reviewReasons.id, id)).returning();
    return updated;
  }

  async deleteReviewReason(id: number): Promise<boolean> {
    const [updated] = await db.update(reviewReasons).set({ isActive: false }).where(eq(reviewReasons.id, id)).returning();
    return !!updated;
  }

  async getCustomerReviews(filters?: { customerId?: number; orderId?: number; reviewReasonId?: number; status?: string }): Promise<CustomerReviewWithDetails[]> {
    let conditions = [];
    if (filters?.customerId) {
      conditions.push(eq(customerReviews.customerId, filters.customerId));
    }
    if (filters?.orderId) {
      conditions.push(eq(customerReviews.orderId, filters.orderId));
    }
    if (filters?.reviewReasonId) {
      conditions.push(eq(customerReviews.reviewReasonId, filters.reviewReasonId));
    }
    if (filters?.status) {
      conditions.push(eq(customerReviews.status, filters.status));
    }
    
    const rows = conditions.length > 0
      ? await db.select().from(customerReviews).where(and(...conditions)).orderBy(desc(customerReviews.createdAt))
      : await db.select().from(customerReviews).orderBy(desc(customerReviews.createdAt));
    
    // Fetch related data
    const results: CustomerReviewWithDetails[] = [];
    for (const row of rows) {
      const [customer] = await db.select().from(customers).where(eq(customers.id, row.customerId));
      const [reason] = await db.select().from(reviewReasons).where(eq(reviewReasons.id, row.reviewReasonId));
      let reviewedByEmployee;
      if (row.reviewedByEmployeeId) {
        const [emp] = await db.select().from(employees).where(eq(employees.id, row.reviewedByEmployeeId));
        reviewedByEmployee = emp;
      }
      results.push({
        ...row,
        customer,
        reviewReason: reason,
        reviewedByEmployee,
      });
    }
    return results;
  }

  async getCustomerReview(id: number): Promise<CustomerReviewWithDetails | undefined> {
    const [row] = await db.select().from(customerReviews).where(eq(customerReviews.id, id));
    if (!row) return undefined;
    
    const [customer] = await db.select().from(customers).where(eq(customers.id, row.customerId));
    const [reason] = await db.select().from(reviewReasons).where(eq(reviewReasons.id, row.reviewReasonId));
    let reviewedByEmployee;
    if (row.reviewedByEmployeeId) {
      const [emp] = await db.select().from(employees).where(eq(employees.id, row.reviewedByEmployeeId));
      reviewedByEmployee = emp;
    }
    return {
      ...row,
      customer,
      reviewReason: reason,
      reviewedByEmployee,
    };
  }

  async createCustomerReview(review: InsertCustomerReview): Promise<CustomerReview> {
    const [created] = await db.insert(customerReviews).values({
      customerId: review.customerId,
      reviewReasonId: review.reviewReasonId,
      orderId: review.orderId,
      rating: review.rating,
      additionalNotes: review.additionalNotes,
      status: "pending",
    }).returning();
    return created;
  }

  async updateCustomerReviewStatus(id: number, status: string, reviewedByEmployeeId?: number, responseNotes?: string): Promise<CustomerReview | undefined> {
    const updateData: any = {
      status,
      updatedAt: new Date(),
    };
    if (reviewedByEmployeeId) {
      updateData.reviewedByEmployeeId = reviewedByEmployeeId;
      updateData.reviewedAt = new Date();
    }
    if (responseNotes !== undefined) {
      updateData.responseNotes = responseNotes;
    }
    const [updated] = await db.update(customerReviews).set(updateData).where(eq(customerReviews.id, id)).returning();
    return updated;
  }

  async deleteCustomerReview(id: number): Promise<boolean> {
    const deleted = await db.delete(customerReviews).where(eq(customerReviews.id, id));
    return true;
  }
  
  // ========================================
  // HR - JOB POSITIONS (الوظائف)
  // ========================================
  
  async getJobPositions(isActive?: boolean): Promise<JobPosition[]> {
    if (isActive !== undefined) {
      return db.select().from(jobPositions).where(eq(jobPositions.isActive, isActive)).orderBy(asc(jobPositions.nameAr));
    }
    return db.select().from(jobPositions).orderBy(asc(jobPositions.nameAr));
  }
  
  async getJobPosition(id: number): Promise<JobPosition | undefined> {
    const [position] = await db.select().from(jobPositions).where(eq(jobPositions.id, id));
    return position;
  }
  
  async createJobPosition(position: InsertJobPosition): Promise<JobPosition> {
    const [newPosition] = await db.insert(jobPositions).values(position).returning();
    return newPosition;
  }
  
  async updateJobPosition(id: number, position: Partial<InsertJobPosition>): Promise<JobPosition | undefined> {
    const [updated] = await db.update(jobPositions)
      .set({ ...position, updatedAt: new Date() })
      .where(eq(jobPositions.id, id))
      .returning();
    return updated;
  }
  
  async deleteJobPosition(id: number): Promise<boolean> {
    const [updated] = await db.update(jobPositions)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(jobPositions.id, id))
      .returning();
    return !!updated;
  }
  
  // ========================================
  // HR - NATIONALITIES (الجنسيات)
  // ========================================
  
  async getNationalities(isActive?: boolean): Promise<Nationality[]> {
    if (isActive !== undefined) {
      return db.select().from(nationalities).where(eq(nationalities.isActive, isActive)).orderBy(asc(nationalities.nameAr));
    }
    return db.select().from(nationalities).orderBy(asc(nationalities.nameAr));
  }
  
  async getNationality(id: number): Promise<Nationality | undefined> {
    const [nationality] = await db.select().from(nationalities).where(eq(nationalities.id, id));
    return nationality;
  }
  
  async createNationality(nationality: InsertNationality): Promise<Nationality> {
    const [newNationality] = await db.insert(nationalities).values(nationality).returning();
    return newNationality;
  }
  
  async updateNationality(id: number, nationality: Partial<InsertNationality>): Promise<Nationality | undefined> {
    const [updated] = await db.update(nationalities)
      .set({ ...nationality, updatedAt: new Date() })
      .where(eq(nationalities.id, id))
      .returning();
    return updated;
  }
  
  async deleteNationality(id: number): Promise<boolean> {
    const [updated] = await db.update(nationalities)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(nationalities.id, id))
      .returning();
    return !!updated;
  }
  
  // ========================================
  // HR - EMPLOYEE SUPERVISORS (المشرفين)
  // ========================================
  
  async getEmployeeSupervisors(): Promise<EmployeeSupervisorWithDetails[]> {
    const allSupervisorRelations = await db.select().from(employeeSupervisors);
    const allEmployees = await db.select().from(employees);
    
    return allSupervisorRelations.map(relation => ({
      ...relation,
      supervisor: allEmployees.find(e => e.id === relation.supervisorEmployeeId),
      employee: allEmployees.find(e => e.id === relation.employeeId),
    }));
  }
  
  async getSupervisorsWithSubordinates(): Promise<SupervisorWithSubordinates[]> {
    const allSupervisorRelations = await db.select().from(employeeSupervisors);
    const allEmployees = await db.select().from(employees).where(eq(employees.isActive, true));
    
    const supervisorIds = Array.from(new Set(allSupervisorRelations.map(r => r.supervisorEmployeeId)));
    
    return supervisorIds.map(supervisorId => {
      const supervisor = allEmployees.find(e => e.id === supervisorId);
      const subordinateCount = allSupervisorRelations.filter(r => r.supervisorEmployeeId === supervisorId).length;
      return {
        ...supervisor!,
        subordinateCount,
      };
    }).filter(s => s.id !== undefined);
  }
  
  async getSubordinates(supervisorEmployeeId: number): Promise<EmployeeSupervisorWithDetails[]> {
    const relations = await db.select().from(employeeSupervisors)
      .where(eq(employeeSupervisors.supervisorEmployeeId, supervisorEmployeeId));
    const allEmployees = await db.select().from(employees);
    
    return relations.map(relation => ({
      ...relation,
      supervisor: allEmployees.find(e => e.id === relation.supervisorEmployeeId),
      employee: allEmployees.find(e => e.id === relation.employeeId),
    }));
  }
  
  async getSupervisorFor(employeeId: number): Promise<EmployeeSupervisorWithDetails | undefined> {
    const [relation] = await db.select().from(employeeSupervisors)
      .where(eq(employeeSupervisors.employeeId, employeeId));
    if (!relation) return undefined;
    
    const allEmployees = await db.select().from(employees);
    return {
      ...relation,
      supervisor: allEmployees.find(e => e.id === relation.supervisorEmployeeId),
      employee: allEmployees.find(e => e.id === relation.employeeId),
    };
  }
  
  async assignSupervisor(data: InsertEmployeeSupervisor): Promise<EmployeeSupervisor> {
    // Remove existing supervisor assignment for this employee
    await db.delete(employeeSupervisors).where(eq(employeeSupervisors.employeeId, data.employeeId));
    
    // Create new assignment
    const [newRelation] = await db.insert(employeeSupervisors).values(data).returning();
    return newRelation;
  }
  
  async removeSupervisor(employeeId: number): Promise<boolean> {
    const result = await db.delete(employeeSupervisors)
      .where(eq(employeeSupervisors.employeeId, employeeId))
      .returning();
    return result.length > 0;
  }
  
  // ========================================
  // HR - WORK SHIFTS (الورديات)
  // ========================================
  
  async getWorkShifts(isActive?: boolean): Promise<WorkShift[]> {
    if (isActive !== undefined) {
      return db.select().from(workShifts).where(eq(workShifts.isActive, isActive)).orderBy(asc(workShifts.nameAr));
    }
    return db.select().from(workShifts).orderBy(asc(workShifts.nameAr));
  }
  
  async getWorkShift(id: number): Promise<WorkShift | undefined> {
    const [shift] = await db.select().from(workShifts).where(eq(workShifts.id, id));
    return shift;
  }
  
  async createWorkShift(shift: InsertWorkShift): Promise<WorkShift> {
    const [newShift] = await db.insert(workShifts).values(shift).returning();
    return newShift;
  }
  
  async updateWorkShift(id: number, shift: Partial<InsertWorkShift>): Promise<WorkShift | undefined> {
    const [updated] = await db.update(workShifts)
      .set({ ...shift, updatedAt: new Date() })
      .where(eq(workShifts.id, id))
      .returning();
    return updated;
  }
  
  async deleteWorkShift(id: number): Promise<boolean> {
    const [updated] = await db.update(workShifts)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(workShifts.id, id))
      .returning();
    return !!updated;
  }
  
  // ========================================
  // HR - SHIFT ASSIGNMENTS (مناوبات الموظفين)
  // ========================================
  
  async getShiftAssignments(filters?: { employeeId?: number; workShiftId?: number; date?: string }): Promise<ShiftAssignmentWithDetails[]> {
    let query = db.select().from(employeeShiftAssignments);
    const conditions = [];
    
    if (filters?.employeeId) {
      conditions.push(eq(employeeShiftAssignments.employeeId, filters.employeeId));
    }
    if (filters?.workShiftId) {
      conditions.push(eq(employeeShiftAssignments.workShiftId, filters.workShiftId));
    }
    if (filters?.date) {
      conditions.push(lte(employeeShiftAssignments.startDate, filters.date));
      conditions.push(gte(employeeShiftAssignments.endDate, filters.date));
    }
    
    const assignments = conditions.length > 0
      ? await db.select().from(employeeShiftAssignments).where(and(...conditions)).orderBy(desc(employeeShiftAssignments.startDate))
      : await db.select().from(employeeShiftAssignments).orderBy(desc(employeeShiftAssignments.startDate));
    
    const allEmployees = await db.select().from(employees);
    const allShifts = await db.select().from(workShifts);
    
    return assignments.map(a => ({
      ...a,
      employee: allEmployees.find(e => e.id === a.employeeId),
      workShift: allShifts.find(s => s.id === a.workShiftId),
    }));
  }
  
  async getShiftAssignment(id: number): Promise<ShiftAssignmentWithDetails | undefined> {
    const [assignment] = await db.select().from(employeeShiftAssignments).where(eq(employeeShiftAssignments.id, id));
    if (!assignment) return undefined;
    
    const [employee] = await db.select().from(employees).where(eq(employees.id, assignment.employeeId));
    const [workShift] = await db.select().from(workShifts).where(eq(workShifts.id, assignment.workShiftId));
    
    return {
      ...assignment,
      employee,
      workShift,
    };
  }
  
  async getActiveShiftAssignment(employeeId: number, date: string): Promise<ShiftAssignmentWithDetails | undefined> {
    const [assignment] = await db.select().from(employeeShiftAssignments)
      .where(and(
        eq(employeeShiftAssignments.employeeId, employeeId),
        lte(employeeShiftAssignments.startDate, date),
        gte(employeeShiftAssignments.endDate, date)
      ));
    
    if (!assignment) return undefined;
    
    const [employee] = await db.select().from(employees).where(eq(employees.id, assignment.employeeId));
    const [workShift] = await db.select().from(workShifts).where(eq(workShifts.id, assignment.workShiftId));
    
    return {
      ...assignment,
      employee,
      workShift,
    };
  }
  
  async createShiftAssignment(assignment: InsertShiftAssignment): Promise<ShiftAssignment> {
    const [newAssignment] = await db.insert(employeeShiftAssignments).values(assignment).returning();
    return newAssignment;
  }
  
  async updateShiftAssignment(id: number, assignment: Partial<InsertShiftAssignment>): Promise<ShiftAssignment | undefined> {
    const [updated] = await db.update(employeeShiftAssignments)
      .set({ ...assignment, updatedAt: new Date() })
      .where(eq(employeeShiftAssignments.id, id))
      .returning();
    return updated;
  }
  
  async deleteShiftAssignment(id: number): Promise<boolean> {
    const result = await db.delete(employeeShiftAssignments)
      .where(eq(employeeShiftAssignments.id, id))
      .returning();
    return result.length > 0;
  }
  
  // ========================================
  // HR - ATTENDANCE RAW (البصمات الخام)
  // ========================================
  
  async getAttendanceRaw(filters?: { employeeId?: number; startDate?: string; endDate?: string }): Promise<AttendanceRaw[]> {
    const conditions = [];
    
    if (filters?.employeeId) {
      conditions.push(eq(attendanceRaw.employeeId, filters.employeeId));
    }
    if (filters?.startDate) {
      conditions.push(gte(attendanceRaw.timestamp, new Date(filters.startDate)));
    }
    if (filters?.endDate) {
      conditions.push(lte(attendanceRaw.timestamp, new Date(filters.endDate + 'T23:59:59')));
    }
    
    if (conditions.length > 0) {
      return db.select().from(attendanceRaw).where(and(...conditions)).orderBy(desc(attendanceRaw.timestamp));
    }
    return db.select().from(attendanceRaw).orderBy(desc(attendanceRaw.timestamp));
  }
  
  async createAttendanceRaw(record: InsertAttendanceRaw): Promise<AttendanceRaw> {
    const [newRecord] = await db.insert(attendanceRaw).values({
      ...record,
      timestamp: new Date(record.timestamp),
    }).returning();
    return newRecord;
  }
  
  // ========================================
  // HR - ATTENDANCE LOGS (سجل الحضور)
  // ========================================
  
  async getAttendanceLogs(filters?: { employeeId?: number; startDate?: string; endDate?: string; status?: string }): Promise<AttendanceLogWithDetails[]> {
    const conditions = [];
    
    if (filters?.employeeId) {
      conditions.push(eq(attendanceLogs.employeeId, filters.employeeId));
    }
    if (filters?.startDate) {
      conditions.push(gte(attendanceLogs.date, filters.startDate));
    }
    if (filters?.endDate) {
      conditions.push(lte(attendanceLogs.date, filters.endDate));
    }
    if (filters?.status) {
      conditions.push(eq(attendanceLogs.status, filters.status));
    }
    
    const logs = conditions.length > 0
      ? await db.select().from(attendanceLogs).where(and(...conditions)).orderBy(desc(attendanceLogs.date))
      : await db.select().from(attendanceLogs).orderBy(desc(attendanceLogs.date));
    
    const allEmployees = await db.select().from(employees);
    const allShifts = await db.select().from(workShifts);
    
    return logs.map(log => ({
      ...log,
      employee: allEmployees.find(e => e.id === log.employeeId),
      workShift: log.workShiftId ? allShifts.find(s => s.id === log.workShiftId) : undefined,
    }));
  }
  
  async getAttendanceLog(id: number): Promise<AttendanceLogWithDetails | undefined> {
    const [log] = await db.select().from(attendanceLogs).where(eq(attendanceLogs.id, id));
    if (!log) return undefined;
    
    const [employee] = await db.select().from(employees).where(eq(employees.id, log.employeeId));
    const workShift = log.workShiftId 
      ? (await db.select().from(workShifts).where(eq(workShifts.id, log.workShiftId)))[0]
      : undefined;
    
    return {
      ...log,
      employee,
      workShift,
    };
  }
  
  async getEmployeeAttendanceForDate(employeeId: number, date: string): Promise<AttendanceLog | undefined> {
    const [log] = await db.select().from(attendanceLogs)
      .where(and(
        eq(attendanceLogs.employeeId, employeeId),
        eq(attendanceLogs.date, date)
      ));
    return log;
  }
  
  async createAttendanceLog(log: InsertAttendanceLog): Promise<AttendanceLog> {
    const [newLog] = await db.insert(attendanceLogs).values(log).returning();
    return newLog;
  }
  
  async updateAttendanceLog(id: number, log: Partial<InsertAttendanceLog>): Promise<AttendanceLog | undefined> {
    const [updated] = await db.update(attendanceLogs)
      .set({ ...log, updatedAt: new Date() })
      .where(eq(attendanceLogs.id, id))
      .returning();
    return updated;
  }
  
  async deleteAttendanceLog(id: number): Promise<boolean> {
    const result = await db.delete(attendanceLogs)
      .where(eq(attendanceLogs.id, id))
      .returning();
    return result.length > 0;
  }
  
  // ========================================
  // HR - OVERTIME (الوقت الإضافي)
  // ========================================
  
  async getOvertimeEntries(filters?: { employeeId?: number; startDate?: string; endDate?: string }): Promise<OvertimeEntryWithDetails[]> {
    const conditions = [];
    
    if (filters?.employeeId) {
      conditions.push(eq(overtimeEntries.employeeId, filters.employeeId));
    }
    if (filters?.startDate) {
      conditions.push(gte(overtimeEntries.date, filters.startDate));
    }
    if (filters?.endDate) {
      conditions.push(lte(overtimeEntries.date, filters.endDate));
    }
    
    const entries = conditions.length > 0
      ? await db.select().from(overtimeEntries).where(and(...conditions)).orderBy(desc(overtimeEntries.date))
      : await db.select().from(overtimeEntries).orderBy(desc(overtimeEntries.date));
    
    const allEmployees = await db.select().from(employees);
    
    return entries.map(entry => ({
      ...entry,
      employee: allEmployees.find(e => e.id === entry.employeeId),
      approvedByEmployee: entry.approvedByEmployeeId ? allEmployees.find(e => e.id === entry.approvedByEmployeeId) : undefined,
    }));
  }
  
  async getOvertimeEntry(id: number): Promise<OvertimeEntryWithDetails | undefined> {
    const [entry] = await db.select().from(overtimeEntries).where(eq(overtimeEntries.id, id));
    if (!entry) return undefined;
    
    const [employee] = await db.select().from(employees).where(eq(employees.id, entry.employeeId));
    const approvedByEmployee = entry.approvedByEmployeeId 
      ? (await db.select().from(employees).where(eq(employees.id, entry.approvedByEmployeeId)))[0]
      : undefined;
    
    return {
      ...entry,
      employee,
      approvedByEmployee,
    };
  }
  
  async createOvertimeEntry(entry: InsertOvertimeEntry): Promise<OvertimeEntry> {
    const [newEntry] = await db.insert(overtimeEntries).values(entry).returning();
    return newEntry;
  }
  
  async updateOvertimeEntry(id: number, entry: Partial<InsertOvertimeEntry>): Promise<OvertimeEntry | undefined> {
    const [updated] = await db.update(overtimeEntries)
      .set({ ...entry, updatedAt: new Date() })
      .where(eq(overtimeEntries.id, id))
      .returning();
    return updated;
  }
  
  async deleteOvertimeEntry(id: number): Promise<boolean> {
    const result = await db.delete(overtimeEntries)
      .where(eq(overtimeEntries.id, id))
      .returning();
    return result.length > 0;
  }
  
  // ========================================
  // HR - LEAVES (الإجازات)
  // ========================================
  
  async getLeaves(filters?: { employeeId?: number; status?: string; leaveType?: string; startDate?: string; endDate?: string }): Promise<LeaveWithDetails[]> {
    const conditions = [];
    
    if (filters?.employeeId) {
      conditions.push(eq(leaves.employeeId, filters.employeeId));
    }
    if (filters?.status) {
      conditions.push(eq(leaves.status, filters.status));
    }
    if (filters?.leaveType) {
      conditions.push(eq(leaves.leaveType, filters.leaveType));
    }
    if (filters?.startDate) {
      conditions.push(gte(leaves.startDate, filters.startDate));
    }
    if (filters?.endDate) {
      conditions.push(lte(leaves.endDate, filters.endDate));
    }
    
    const allLeaves = conditions.length > 0
      ? await db.select().from(leaves).where(and(...conditions)).orderBy(desc(leaves.startDate))
      : await db.select().from(leaves).orderBy(desc(leaves.startDate));
    
    const allEmployees = await db.select().from(employees);
    
    return allLeaves.map(leave => ({
      ...leave,
      employee: allEmployees.find(e => e.id === leave.employeeId),
      requestedByEmployee: leave.requestedByEmployeeId ? allEmployees.find(e => e.id === leave.requestedByEmployeeId) : undefined,
    }));
  }
  
  async getLeave(id: number): Promise<LeaveWithDetails | undefined> {
    const [leave] = await db.select().from(leaves).where(eq(leaves.id, id));
    if (!leave) return undefined;
    
    const [employee] = await db.select().from(employees).where(eq(employees.id, leave.employeeId));
    const requestedByEmployee = leave.requestedByEmployeeId 
      ? (await db.select().from(employees).where(eq(employees.id, leave.requestedByEmployeeId)))[0]
      : undefined;
    
    return {
      ...leave,
      employee,
      requestedByEmployee,
    };
  }
  
  async createLeave(leave: InsertLeave): Promise<Leave> {
    const [newLeave] = await db.insert(leaves).values(leave).returning();
    return newLeave;
  }
  
  async updateLeave(id: number, leave: Partial<InsertLeave>): Promise<Leave | undefined> {
    const [updated] = await db.update(leaves)
      .set({ ...leave, updatedAt: new Date() })
      .where(eq(leaves.id, id))
      .returning();
    return updated;
  }
  
  async deleteLeave(id: number): Promise<boolean> {
    const result = await db.delete(leaves)
      .where(eq(leaves.id, id))
      .returning();
    return result.length > 0;
  }
  
  async approveLeave(id: number): Promise<Leave | undefined> {
    const [updated] = await db.update(leaves)
      .set({ status: "approved", updatedAt: new Date() })
      .where(eq(leaves.id, id))
      .returning();
    return updated;
  }
  
  async rejectLeave(id: number): Promise<Leave | undefined> {
    const [updated] = await db.update(leaves)
      .set({ status: "rejected", updatedAt: new Date() })
      .where(eq(leaves.id, id))
      .returning();
    return updated;
  }

  // ========================================
  // EXPENSE TYPES (أنواع النفقات)
  // ========================================

  async getExpenseTypes(isActive?: boolean): Promise<ExpenseType[]> {
    if (isActive !== undefined) {
      return db.select().from(expenseTypes).where(eq(expenseTypes.isActive, isActive)).orderBy(expenseTypes.nameAr);
    }
    return db.select().from(expenseTypes).orderBy(expenseTypes.nameAr);
  }

  async getExpenseType(id: number): Promise<ExpenseType | undefined> {
    const [expenseType] = await db.select().from(expenseTypes).where(eq(expenseTypes.id, id));
    return expenseType;
  }

  async createExpenseType(expenseType: InsertExpenseType): Promise<ExpenseType> {
    // Generate expense account code (5xxx series for expenses)
    const existingAccounts = await db.select().from(accounts)
      .where(like(accounts.code, '5%'))
      .orderBy(desc(accounts.code));
    
    // Find max expense sub-account code under 5xxx
    let nextCode = "5301"; // Start from 5301 (after common expense accounts)
    if (existingAccounts.length > 0) {
      const maxCode = existingAccounts[0].code;
      if (maxCode && maxCode.startsWith('5')) {
        const numericPart = parseInt(maxCode.slice(1), 10);
        if (!isNaN(numericPart) && numericPart >= 300) {
          nextCode = `5${(numericPart + 1).toString().padStart(3, '0')}`;
        }
      }
    }
    
    // Create GL account for this expense type
    const [newAccount] = await db.insert(accounts).values({
      code: nextCode,
      nameAr: expenseType.nameAr,
      nameEn: expenseType.nameAr, // Use Arabic name as fallback
      type: "expense",
      parentCode: "5300", // Expenses parent account
      isActive: true,
    }).returning();
    
    // Create expense type with linked account
    const [newExpenseType] = await db.insert(expenseTypes).values({
      ...expenseType,
      accountId: newAccount.id,
    }).returning();
    return newExpenseType;
  }

  async updateExpenseType(id: number, expenseType: Partial<InsertExpenseType>): Promise<ExpenseType | undefined> {
    const [updated] = await db.update(expenseTypes)
      .set({ ...expenseType, updatedAt: new Date() })
      .where(eq(expenseTypes.id, id))
      .returning();
    return updated;
  }

  async deleteExpenseType(id: number): Promise<boolean> {
    const [updated] = await db.update(expenseTypes)
      .set({ isActive: false, updatedAt: new Date() })
      .where(eq(expenseTypes.id, id))
      .returning();
    return !!updated;
  }

  // ========================================
  // EXPENSES (النفقات)
  // ========================================

  async getExpenses(filters?: ExpenseFilters): Promise<ExpenseWithDetails[]> {
    const conditions = [];
    
    if (filters?.fromDate) {
      conditions.push(gte(expenses.paymentDate, filters.fromDate));
    }
    if (filters?.toDate) {
      conditions.push(lte(expenses.paymentDate, filters.toDate));
    }
    if (filters?.expenseTypeId) {
      conditions.push(eq(expenses.expenseTypeId, filters.expenseTypeId));
    }
    if (filters?.supplierId) {
      conditions.push(eq(expenses.supplierId, filters.supplierId));
    }
    if (filters?.paymentMethodId) {
      conditions.push(eq(expenses.paymentMethodId, filters.paymentMethodId));
    }
    
    const allExpenses = conditions.length > 0
      ? await db.select().from(expenses).where(and(...conditions)).orderBy(desc(expenses.paymentDate))
      : await db.select().from(expenses).orderBy(desc(expenses.paymentDate));
    
    const allExpenseTypes = await db.select().from(expenseTypes);
    const allSuppliers = await db.select().from(suppliers);
    const allPaymentMethods = await db.select().from(paymentMethods);
    const allEmployees = await db.select().from(employees);
    
    return allExpenses.map(expense => ({
      ...expense,
      expenseType: allExpenseTypes.find(et => et.id === expense.expenseTypeId),
      supplier: expense.supplierId ? allSuppliers.find(s => s.id === expense.supplierId) : undefined,
      paymentMethod: expense.paymentMethodId ? allPaymentMethods.find(pm => pm.id === expense.paymentMethodId) : undefined,
      createdByEmployee: expense.createdByEmployeeId ? allEmployees.find(e => e.id === expense.createdByEmployeeId) : undefined,
    }));
  }

  async getExpense(id: number): Promise<ExpenseWithDetails | undefined> {
    const [expense] = await db.select().from(expenses).where(eq(expenses.id, id));
    if (!expense) return undefined;
    
    const [expenseType] = await db.select().from(expenseTypes).where(eq(expenseTypes.id, expense.expenseTypeId));
    const supplier = expense.supplierId 
      ? (await db.select().from(suppliers).where(eq(suppliers.id, expense.supplierId)))[0]
      : undefined;
    const paymentMethod = expense.paymentMethodId
      ? (await db.select().from(paymentMethods).where(eq(paymentMethods.id, expense.paymentMethodId)))[0]
      : undefined;
    const createdByEmployee = expense.createdByEmployeeId
      ? (await db.select().from(employees).where(eq(employees.id, expense.createdByEmployeeId)))[0]
      : undefined;
    
    return {
      ...expense,
      expenseType,
      supplier,
      paymentMethod,
      createdByEmployee,
    };
  }

  async createExpense(expense: InsertExpense): Promise<Expense> {
    // Get the expense type to find its linked GL account
    const expenseType = await this.getExpenseType(expense.expenseTypeId);
    if (!expenseType) {
      throw new Error("نوع المصروف غير موجود");
    }
    
    // Get payment method for credit account
    let creditAccountCode = ACCOUNT_CODES.CASH; // Default to cash
    let creditAccountName = ACCOUNT_NAMES[ACCOUNT_CODES.CASH];
    
    if (expense.paymentMethodId) {
      const [pm] = await db.select().from(paymentMethods).where(eq(paymentMethods.id, expense.paymentMethodId));
      if (pm?.accountId) {
        const [pmAccount] = await db.select().from(accounts).where(eq(accounts.id, pm.accountId));
        if (pmAccount) {
          creditAccountCode = pmAccount.code;
          creditAccountName = pmAccount.nameAr;
        }
      } else if (pm?.type === 'bank') {
        creditAccountCode = ACCOUNT_CODES.BANK;
        creditAccountName = ACCOUNT_NAMES[ACCOUNT_CODES.BANK];
      }
    }
    
    // Determine expense account code
    let expenseAccountCode = "5300"; // General expenses
    let expenseAccountName = "مصروفات عمومية";
    if (expenseType.accountId) {
      const [expenseAccount] = await db.select().from(accounts).where(eq(accounts.id, expenseType.accountId));
      if (expenseAccount) {
        expenseAccountCode = expenseAccount.code;
        expenseAccountName = expenseAccount.nameAr;
      }
    }
    
    // Create journal entry
    const count = await db.select({ count: sql<number>`count(*)` }).from(journalEntries);
    const entryNumber = `JE-${String((count[0]?.count || 0) + 1).padStart(6, '0')}`;
    
    const [journalEntry] = await db.insert(journalEntries).values({
      entryNumber,
      date: expense.paymentDate,
      description: `مصروف: ${expenseType.nameAr}${expense.descriptionAr ? ' - ' + expense.descriptionAr : ''}`,
      referenceType: "Expense",
      referenceId: 0, // Will update after expense creation
    }).returning();
    
    const journalLinesData: InsertJournalLine[] = [
      // Debit Expense account
      {
        journalEntryId: journalEntry.id,
        accountCode: expenseAccountCode,
        accountName: expenseAccountName,
        debit: expense.amountWithoutVat,
        credit: 0,
      },
    ];
    
    // Add VAT input if applicable
    if (expense.vatAmount > 0 && expense.vatType !== 'exempt') {
      journalLinesData.push({
        journalEntryId: journalEntry.id,
        accountCode: ACCOUNT_CODES.VAT_INPUT,
        accountName: ACCOUNT_NAMES[ACCOUNT_CODES.VAT_INPUT],
        debit: expense.vatAmount,
        credit: 0,
      });
    }
    
    // Credit Cash/Bank account
    journalLinesData.push({
      journalEntryId: journalEntry.id,
      accountCode: creditAccountCode,
      accountName: creditAccountName,
      debit: 0,
      credit: expense.totalAmount,
    });
    
    await db.insert(journalLines).values(journalLinesData);
    
    // Create expense with journal entry reference
    const [newExpense] = await db.insert(expenses).values({
      ...expense,
      journalEntryId: journalEntry.id,
    }).returning();
    
    // Update journal entry with correct expense reference
    await db.update(journalEntries)
      .set({ referenceId: newExpense.id })
      .where(eq(journalEntries.id, journalEntry.id));
    
    return newExpense;
  }

  async updateExpense(id: number, expense: Partial<InsertExpense>): Promise<Expense | undefined> {
    const [updated] = await db.update(expenses)
      .set({ ...expense, updatedAt: new Date() })
      .where(eq(expenses.id, id))
      .returning();
    return updated;
  }

  async deleteExpense(id: number): Promise<boolean> {
    const result = await db.delete(expenses)
      .where(eq(expenses.id, id))
      .returning();
    return result.length > 0;
  }

  async getExpensesSummary(filters?: ExpenseFilters): Promise<ExpensesSummary> {
    const conditions = [];
    
    if (filters?.fromDate) {
      conditions.push(gte(expenses.paymentDate, filters.fromDate));
    }
    if (filters?.toDate) {
      conditions.push(lte(expenses.paymentDate, filters.toDate));
    }
    if (filters?.expenseTypeId) {
      conditions.push(eq(expenses.expenseTypeId, filters.expenseTypeId));
    }
    if (filters?.supplierId) {
      conditions.push(eq(expenses.supplierId, filters.supplierId));
    }
    
    const allExpenses = conditions.length > 0
      ? await db.select().from(expenses).where(and(...conditions))
      : await db.select().from(expenses);
    
    const allExpenseTypes = await db.select().from(expenseTypes);
    
    // Calculate totals
    const totalExpensesWithoutVat = allExpenses.reduce((sum, e) => sum + e.amountWithoutVat, 0);
    const totalVat = allExpenses.reduce((sum, e) => sum + e.vatAmount, 0);
    const totalExpensesWithVat = allExpenses.reduce((sum, e) => sum + e.totalAmount, 0);
    
    // Group by type
    const byTypeMap = new Map<number, { count: number; total: number }>();
    for (const expense of allExpenses) {
      const existing = byTypeMap.get(expense.expenseTypeId) || { count: 0, total: 0 };
      byTypeMap.set(expense.expenseTypeId, {
        count: existing.count + 1,
        total: existing.total + expense.totalAmount,
      });
    }
    
    const byType = Array.from(byTypeMap.entries()).map(([typeId, data]) => {
      const et = allExpenseTypes.find(t => t.id === typeId);
      return {
        expenseTypeId: typeId,
        expenseTypeNameAr: et?.nameAr || "",
        totalAmount: data.total,
        count: data.count,
      };
    }).sort((a, b) => b.totalAmount - a.totalAmount);
    
    // Group by month
    const byMonthMap = new Map<string, number>();
    for (const expense of allExpenses) {
      const date = new Date(expense.paymentDate);
      const key = `${date.getFullYear()}-${date.getMonth() + 1}`;
      byMonthMap.set(key, (byMonthMap.get(key) || 0) + expense.totalAmount);
    }
    
    const byMonth = Array.from(byMonthMap.entries()).map(([key, total]) => {
      const [year, month] = key.split("-").map(Number);
      return { year, month, totalAmount: total };
    }).sort((a, b) => {
      if (a.year !== b.year) return b.year - a.year;
      return b.month - a.month;
    });
    
    return {
      totalExpensesWithoutVat,
      totalVat,
      totalExpensesWithVat,
      byType,
      byMonth,
    };
  }

  // ========================================
  // PAYABLES (الديون)
  // ========================================

  async getPayablesSummary(): Promise<PayablesSummary> {
    const today = new Date().toISOString().split("T")[0];
    
    // Get all unpaid/partially paid invoices
    const allInvoices = await db.select().from(purchaseInvoices)
      .where(ne(purchaseInvoices.paymentStatus, "paid"));
    
    // Get all payments
    const allPayments = await db.select().from(purchasePayments);
    
    // Get all suppliers
    const allSuppliers = await db.select().from(suppliers);
    
    // Calculate invoice totals and payments
    const supplierData = new Map<number, {
      totalOutstanding: number;
      overdueAmount: number;
      invoiceCountUnpaid: number;
      invoiceCountOverdue: number;
    }>();
    
    let totalOutstanding = 0;
    let totalOverdue = 0;
    const agingBuckets = {
      "0_30": 0,
      "31_60": 0,
      "61_90": 0,
      "90_plus": 0,
    };
    
    for (const invoice of allInvoices) {
      // Calculate payments for this invoice
      const invoicePayments = allPayments.filter(p => p.purchaseInvoiceId === invoice.id);
      const totalPaid = invoicePayments.reduce((sum, p) => sum + p.amount, 0);
      const remaining = invoice.totalAmount - totalPaid;
      
      if (remaining <= 0) continue;
      
      // Initialize supplier data
      if (!supplierData.has(invoice.supplierId)) {
        supplierData.set(invoice.supplierId, {
          totalOutstanding: 0,
          overdueAmount: 0,
          invoiceCountUnpaid: 0,
          invoiceCountOverdue: 0,
        });
      }
      const data = supplierData.get(invoice.supplierId)!;
      
      // Update totals
      totalOutstanding += remaining;
      data.totalOutstanding += remaining;
      data.invoiceCountUnpaid += 1;
      
      // Check if overdue
      const dueDate = invoice.dueDate || invoice.invoiceDate;
      if (dueDate < today) {
        totalOverdue += remaining;
        data.overdueAmount += remaining;
        data.invoiceCountOverdue += 1;
        
        // Calculate aging bucket
        const daysDiff = Math.floor((new Date(today).getTime() - new Date(dueDate).getTime()) / (1000 * 60 * 60 * 24));
        if (daysDiff <= 30) {
          agingBuckets["0_30"] += remaining;
        } else if (daysDiff <= 60) {
          agingBuckets["31_60"] += remaining;
        } else if (daysDiff <= 90) {
          agingBuckets["61_90"] += remaining;
        } else {
          agingBuckets["90_plus"] += remaining;
        }
      }
    }
    
    // Build suppliers array
    const suppliersList = Array.from(supplierData.entries()).map(([supplierId, data]) => {
      const supplier = allSuppliers.find(s => s.id === supplierId);
      return {
        supplierId,
        supplierNameAr: supplier?.nameAr || "",
        ...data,
      };
    }).sort((a, b) => b.totalOutstanding - a.totalOutstanding);
    
    return {
      globalSummary: {
        totalOutstanding,
        totalOverdue,
        agingBuckets,
      },
      suppliers: suppliersList,
    };
  }

  async getSupplierStatement(supplierId: number, fromDate?: string, toDate?: string): Promise<SupplierStatement> {
    const supplier = await this.getSupplier(supplierId);
    if (!supplier) {
      throw new Error("المورد غير موجود");
    }
    
    // Get all invoices for this supplier
    const allInvoices = await db.select().from(purchaseInvoices)
      .where(eq(purchaseInvoices.supplierId, supplierId))
      .orderBy(asc(purchaseInvoices.invoiceDate));
    
    // Get all payments for this supplier
    const allPayments = await db.select().from(purchasePayments)
      .where(eq(purchasePayments.supplierId, supplierId))
      .orderBy(asc(purchasePayments.paymentDate));
    
    // Build transactions list
    const transactions: SupplierStatementTransaction[] = [];
    let runningBalance = 0;
    let openingBalance = 0;
    
    // Combine invoices and payments into a single list and sort by date
    type TransactionItem = {
      date: string;
      type: "invoice" | "payment";
      id: number;
      amount: number;
      notes?: string | null;
    };
    
    const allTransactions: TransactionItem[] = [
      ...allInvoices.map(inv => ({
        date: inv.invoiceDate,
        type: "invoice" as const,
        id: inv.id,
        amount: inv.totalAmount,
        notes: inv.notes,
      })),
      ...allPayments.map(pay => ({
        date: pay.paymentDate,
        type: "payment" as const,
        id: pay.id,
        amount: pay.amount,
        notes: pay.notes,
      })),
    ].sort((a, b) => {
      const dateCompare = a.date.localeCompare(b.date);
      if (dateCompare !== 0) return dateCompare;
      // Invoices before payments on the same day
      if (a.type === "invoice" && b.type === "payment") return -1;
      if (a.type === "payment" && b.type === "invoice") return 1;
      return 0;
    });
    
    // Calculate opening balance (sum of transactions before fromDate)
    if (fromDate) {
      for (const tx of allTransactions) {
        if (tx.date < fromDate) {
          if (tx.type === "invoice") {
            openingBalance += tx.amount;
          } else {
            openingBalance -= tx.amount;
          }
        }
      }
    }
    
    runningBalance = openingBalance;
    
    // Build transactions within date range
    for (const tx of allTransactions) {
      // Skip transactions before fromDate
      if (fromDate && tx.date < fromDate) continue;
      // Skip transactions after toDate
      if (toDate && tx.date > toDate) continue;
      
      if (tx.type === "invoice") {
        runningBalance += tx.amount;
        transactions.push({
          date: tx.date,
          type: "invoice",
          referenceNumber: `INV-${tx.id}`,
          descriptionAr: tx.notes || "فاتورة مشتريات",
          debitAmount: 0,
          creditAmount: tx.amount,
          balanceAfter: runningBalance,
        });
      } else {
        runningBalance -= tx.amount;
        transactions.push({
          date: tx.date,
          type: "payment",
          referenceNumber: `PAY-${tx.id}`,
          descriptionAr: tx.notes || "دفعة",
          debitAmount: tx.amount,
          creditAmount: 0,
          balanceAfter: runningBalance,
        });
      }
    }
    
    const closingDate = toDate || new Date().toISOString().split("T")[0];
    
    return {
      supplier: {
        id: supplier.id,
        nameAr: supplier.nameAr,
        agreementText: supplier.agreementText,
        paymentTermDays: supplier.paymentTermDays,
      },
      openingBalance: {
        amount: Math.abs(openingBalance),
        direction: openingBalance >= 0 ? "credit" : "debit",
      },
      transactions,
      closingBalance: {
        amount: runningBalance,
        explanation: `رصيد نهائي للمورد حتى تاريخ ${closingDate}`,
      },
    };
  }

  // ========================================
  // EMPLOYEE CUSTODIES (عهدة الموظف)
  // ========================================

  async getEmployeeCustodies(filters?: { employeeId?: number; status?: string; productId?: number }): Promise<EmployeeCustodyWithDetails[]> {
    let query = db.select().from(employeeCustodies);
    
    const conditions = [];
    if (filters?.employeeId) {
      conditions.push(eq(employeeCustodies.employeeId, filters.employeeId));
    }
    if (filters?.status) {
      conditions.push(eq(employeeCustodies.status, filters.status));
    }
    if (filters?.productId) {
      conditions.push(eq(employeeCustodies.productId, filters.productId));
    }
    
    const custodiesList = conditions.length > 0 
      ? await db.select().from(employeeCustodies).where(and(...conditions)).orderBy(desc(employeeCustodies.createdAt))
      : await db.select().from(employeeCustodies).orderBy(desc(employeeCustodies.createdAt));
    
    const allEmployees = await db.select().from(employees);
    const allProducts = await db.select().from(products);
    
    return custodiesList.map(custody => {
      const employee = allEmployees.find(e => e.id === custody.employeeId);
      const product = allProducts.find(p => p.id === custody.productId);
      const createdByEmployee = custody.createdByEmployeeId 
        ? allEmployees.find(e => e.id === custody.createdByEmployeeId) 
        : undefined;
      const closedByEmployee = custody.closedByEmployeeId 
        ? allEmployees.find(e => e.id === custody.closedByEmployeeId) 
        : undefined;
      
      return {
        ...custody,
        employee,
        product,
        createdByEmployee,
        closedByEmployee,
      };
    });
  }

  async getEmployeeCustody(id: number): Promise<EmployeeCustodyWithDetails | undefined> {
    const [custody] = await db.select().from(employeeCustodies).where(eq(employeeCustodies.id, id));
    if (!custody) return undefined;
    
    const [employee] = await db.select().from(employees).where(eq(employees.id, custody.employeeId));
    const [product] = await db.select().from(products).where(eq(products.id, custody.productId));
    const createdByEmployee = custody.createdByEmployeeId 
      ? (await db.select().from(employees).where(eq(employees.id, custody.createdByEmployeeId)))[0]
      : undefined;
    const closedByEmployee = custody.closedByEmployeeId 
      ? (await db.select().from(employees).where(eq(employees.id, custody.closedByEmployeeId)))[0]
      : undefined;
    
    return {
      ...custody,
      employee,
      product,
      createdByEmployee,
      closedByEmployee,
    };
  }

  async createEmployeeCustody(custody: InsertEmployeeCustody): Promise<EmployeeCustodyWithDetails> {
    // Validate product exists
    const [product] = await db.select().from(products).where(eq(products.id, custody.productId));
    if (!product) {
      throw new Error("المنتج غير موجود");
    }
    
    // Validate product usage type - cannot issue sale-only products as custody
    if (product.usageType === "sale") {
      throw new Error("لا يمكن إنشاء عهدة من منتج مخصص للبيع فقط. يجب أن يكون نوع الاستخدام 'للاستهلاك' أو 'للبيع والاستهلاك'");
    }
    
    // Check available stock before creating custody
    const batches = await db.select().from(stockBatches)
      .where(and(
        eq(stockBatches.productId, custody.productId),
        sql`${stockBatches.quantityOnHand} > 0`
      ));
    
    const totalAvailable = batches.reduce((sum, b) => sum + b.quantityOnHand, 0);
    if (totalAvailable < custody.quantity) {
      throw new Error(`الكمية المطلوبة (${custody.quantity}) أكبر من الكمية المتوفرة في المخزون (${totalAvailable})`);
    }
    
    // Get product average cost
    const unitCost = product.averageUnitCost || 0;
    const totalCost = custody.quantity * unitCost;
    
    // Create the custody record
    const [newCustody] = await db.insert(employeeCustodies).values({
      ...custody,
      unitCost,
      totalCost,
    }).returning();
    
    // Create stock movement
    await db.insert(stockMovements).values({
      productId: custody.productId,
      movementDate: custody.custodyDate,
      movementType: "custody_issue",
      quantityIn: 0,
      quantityOut: custody.quantity,
      unitCost,
      referenceType: "EmployeeCustody",
      referenceId: newCustody.id,
    });
    
    // Reduce stock from batches (FIFO) - reuse batches from availability check
    let remainingQty = custody.quantity;
    const sortedBatches = batches.sort((a, b) => 
      new Date(a.createdAt).getTime() - new Date(b.createdAt).getTime()
    );
    
    for (const batch of sortedBatches) {
      if (remainingQty <= 0) break;
      
      const deductQty = Math.min(remainingQty, batch.quantityOnHand);
      await db.update(stockBatches)
        .set({ 
          quantityOnHand: batch.quantityOnHand - deductQty,
          updatedAt: new Date(),
        })
        .where(eq(stockBatches.id, batch.id));
      
      remainingQty -= deductQty;
    }
    
    return this.getEmployeeCustody(newCustody.id) as Promise<EmployeeCustodyWithDetails>;
  }

  async updateEmployeeCustody(id: number, custody: Partial<InsertEmployeeCustody>): Promise<EmployeeCustody | undefined> {
    const [updated] = await db.update(employeeCustodies)
      .set({ ...custody, updatedAt: new Date() })
      .where(eq(employeeCustodies.id, id))
      .returning();
    return updated;
  }

  async deleteEmployeeCustody(id: number): Promise<boolean> {
    const [result] = await db.delete(employeeCustodies)
      .where(eq(employeeCustodies.id, id))
      .returning();
    return !!result;
  }

  async getEmployeeActiveCustodies(employeeId: number): Promise<EmployeeCustodyWithDetails[]> {
    return this.getEmployeeCustodies({ employeeId, status: "open" });
  }

  // ========================================
  // EMPLOYEE COMMISSIONS REPORT (تقرير العمولات)
  // ========================================

  async getEmployeeCommissions(employeeId: number, startDate: string, endDate: string): Promise<CommissionRecord[]> {
    // TODO: The actual calculation logic will be implemented later
    // For now, return an empty array as a placeholder
    // The user mentioned they will explain how to calculate the report later
    return [];
  }

  // ========================================
  // INVENTORY REPORTS (تقارير المخزون)
  // ========================================

  async getStockMovementSummary(fromDate: string, toDate: string, productId?: number): Promise<StockMovementSummaryItem[]> {
    // Get all products
    const allProducts = await db.select().from(products).where(eq(products.isActive, true));
    const allUnits = await db.select().from(units);
    
    const results: StockMovementSummaryItem[] = [];
    
    for (const product of allProducts) {
      if (productId && product.id !== productId) continue;
      
      // Get opening quantity (movements before fromDate)
      const openingResult = await db.select({
        totalIn: sql<number>`COALESCE(SUM(${stockMovements.quantityIn}), 0)`,
        totalOut: sql<number>`COALESCE(SUM(${stockMovements.quantityOut}), 0)`,
      })
      .from(stockMovements)
      .where(and(
        eq(stockMovements.productId, product.id),
        sql`${stockMovements.movementDate} < ${fromDate}`
      ));
      
      const openingQuantity = (openingResult[0]?.totalIn || 0) - (openingResult[0]?.totalOut || 0);
      
      // Get movements in period
      const periodResult = await db.select({
        totalIn: sql<number>`COALESCE(SUM(${stockMovements.quantityIn}), 0)`,
        totalOut: sql<number>`COALESCE(SUM(${stockMovements.quantityOut}), 0)`,
      })
      .from(stockMovements)
      .where(and(
        eq(stockMovements.productId, product.id),
        sql`${stockMovements.movementDate} >= ${fromDate}`,
        sql`${stockMovements.movementDate} <= ${toDate}`
      ));
      
      const quantityInPeriod = periodResult[0]?.totalIn || 0;
      const quantityOutPeriod = periodResult[0]?.totalOut || 0;
      const closingQuantity = openingQuantity + quantityInPeriod - quantityOutPeriod;
      
      const sellUnit = product.sellUnitId ? allUnits.find(u => u.id === product.sellUnitId) : undefined;
      
      results.push({
        productId: product.id,
        productNameAr: product.nameAr,
        sku: product.sku || undefined,
        openingQuantity,
        quantityInPeriod,
        quantityOutPeriod,
        closingQuantity,
        sellUnitNameAr: sellUnit?.nameAr,
      });
    }
    
    return results;
  }

  async getNearExpiryProducts(days: number = 30): Promise<NearExpiryProductItem[]> {
    const today = new Date();
    const targetDate = new Date();
    targetDate.setDate(targetDate.getDate() + days);
    
    const todayStr = today.toISOString().split("T")[0];
    const targetDateStr = targetDate.toISOString().split("T")[0];
    
    const batches = await db.select().from(stockBatches)
      .where(and(
        isNotNull(stockBatches.expiryDate),
        sql`${stockBatches.expiryDate} >= ${todayStr}`,
        sql`${stockBatches.expiryDate} <= ${targetDateStr}`,
        sql`${stockBatches.quantityOnHand} > 0`
      ))
      .orderBy(asc(stockBatches.expiryDate));
    
    const allProducts = await db.select().from(products);
    
    return batches.map(batch => {
      const product = allProducts.find(p => p.id === batch.productId);
      const expiryDate = batch.expiryDate as string;
      const daysToExpiry = Math.ceil((new Date(expiryDate).getTime() - today.getTime()) / (1000 * 60 * 60 * 24));
      
      return {
        productId: batch.productId,
        productNameAr: product?.nameAr || "",
        batchId: batch.id,
        expiryDate,
        quantityOnHand: batch.quantityOnHand,
        unitCost: batch.unitCost,
        totalCost: batch.quantityOnHand * batch.unitCost,
        daysToExpiry,
      };
    });
  }

  // ========================================
  // MOVING AVERAGE COST (متوسط التكلفة)
  // ========================================

  async updateProductAverageCost(productId: number, newQuantity: number, newCost: number): Promise<Product | undefined> {
    const [product] = await db.select().from(products).where(eq(products.id, productId));
    if (!product) return undefined;
    
    // Get current stock quantity
    const stockResult = await db.select({
      totalQty: sql<number>`COALESCE(SUM(${stockBatches.quantityOnHand}), 0)`,
    })
    .from(stockBatches)
    .where(eq(stockBatches.productId, productId));
    
    const currentQty = stockResult[0]?.totalQty || 0;
    const currentCost = product.averageUnitCost || 0;
    
    // Calculate new weighted average
    const totalCostPrev = currentQty * currentCost;
    const totalCostNew = newQuantity * newCost;
    const totalQty = currentQty + newQuantity;
    
    const newAverageCost = totalQty > 0 ? (totalCostPrev + totalCostNew) / totalQty : newCost;
    
    // Update product
    const [updated] = await db.update(products)
      .set({ 
        averageUnitCost: newAverageCost,
        updatedAt: new Date(),
      })
      .where(eq(products.id, productId))
      .returning();
    
    return updated;
  }

  async getProductAverageCost(productId: number): Promise<number> {
    const [product] = await db.select().from(products).where(eq(products.id, productId));
    return product?.averageUnitCost || 0;
  }

  // ========================================
  // DRIVERS (السائقون)
  // ========================================

  async getDrivers(): Promise<Driver[]> {
    return await db.select().from(drivers).orderBy(desc(drivers.createdAt));
  }

  async getDriver(id: number): Promise<Driver | undefined> {
    const [driver] = await db.select().from(drivers).where(eq(drivers.id, id));
    return driver;
  }

  async createDriver(driver: InsertDriver): Promise<Driver> {
    // Helper to convert empty/undefined to null
    const toNull = (val: any) => (val === '' || val === undefined || val === null) ? null : val;
    
    // Convert empty strings AND undefined to null for optional fields (dates and unique fields)
    const cleanedDriver = {
      ...driver,
      // Unique fields - must be null not empty string/undefined
      email: toNull(driver.email),
      code: toNull(driver.code),
      // Other optional text fields
      phone: toNull(driver.phone),
      city: toNull(driver.city),
      address: toNull(driver.address),
      nationality: toNull(driver.nationality),
      iqamaNumber: toNull(driver.iqamaNumber),
      passportNumber: toNull(driver.passportNumber),
      insuranceNumber: toNull(driver.insuranceNumber),
      insuranceCardNumber: toNull(driver.insuranceCardNumber),
      licenseNumber: toNull(driver.licenseNumber),
      vehicleType: toNull(driver.vehicleType),
      vehiclePlateNumber: toNull(driver.vehiclePlateNumber),
      notes: toNull(driver.notes),
      // Date fields
      birthDate: toNull(driver.birthDate),
      hireDate: toNull(driver.hireDate),
      iqamaExpiryDate: toNull(driver.iqamaExpiryDate),
      insuranceExpiryDate: toNull(driver.insuranceExpiryDate),
      insuranceCardExpiryDate: toNull(driver.insuranceCardExpiryDate),
      passportExpiryDate: toNull(driver.passportExpiryDate),
      licenseExpiryDate: toNull(driver.licenseExpiryDate),
    };
    const [newDriver] = await db.insert(drivers).values(cleanedDriver).returning();
    return newDriver;
  }

  async updateDriver(id: number, driver: Partial<InsertDriver>): Promise<Driver | undefined> {
    // Helper to convert empty/undefined to null (only for keys that exist in the update payload)
    const toNullIfEmpty = (val: any) => (val === '' || val === undefined) ? null : val;
    
    // Build cleaned object, only including fields that were provided
    const cleanedDriver: Record<string, any> = { updatedAt: new Date() };
    
    // Unique fields
    if ('email' in driver) cleanedDriver.email = toNullIfEmpty(driver.email);
    if ('code' in driver) cleanedDriver.code = toNullIfEmpty(driver.code);
    
    // Other optional text fields
    if ('fullName' in driver) cleanedDriver.fullName = driver.fullName;
    if ('phone' in driver) cleanedDriver.phone = toNullIfEmpty(driver.phone);
    if ('city' in driver) cleanedDriver.city = toNullIfEmpty(driver.city);
    if ('address' in driver) cleanedDriver.address = toNullIfEmpty(driver.address);
    if ('nationality' in driver) cleanedDriver.nationality = toNullIfEmpty(driver.nationality);
    if ('iqamaNumber' in driver) cleanedDriver.iqamaNumber = toNullIfEmpty(driver.iqamaNumber);
    if ('passportNumber' in driver) cleanedDriver.passportNumber = toNullIfEmpty(driver.passportNumber);
    if ('insuranceNumber' in driver) cleanedDriver.insuranceNumber = toNullIfEmpty(driver.insuranceNumber);
    if ('insuranceCardNumber' in driver) cleanedDriver.insuranceCardNumber = toNullIfEmpty(driver.insuranceCardNumber);
    if ('licenseNumber' in driver) cleanedDriver.licenseNumber = toNullIfEmpty(driver.licenseNumber);
    if ('vehicleType' in driver) cleanedDriver.vehicleType = toNullIfEmpty(driver.vehicleType);
    if ('vehiclePlateNumber' in driver) cleanedDriver.vehiclePlateNumber = toNullIfEmpty(driver.vehiclePlateNumber);
    if ('notes' in driver) cleanedDriver.notes = toNullIfEmpty(driver.notes);
    if ('isActive' in driver) cleanedDriver.isActive = driver.isActive;
    
    // Date fields
    if ('birthDate' in driver) cleanedDriver.birthDate = toNullIfEmpty(driver.birthDate);
    if ('hireDate' in driver) cleanedDriver.hireDate = toNullIfEmpty(driver.hireDate);
    if ('iqamaExpiryDate' in driver) cleanedDriver.iqamaExpiryDate = toNullIfEmpty(driver.iqamaExpiryDate);
    if ('insuranceExpiryDate' in driver) cleanedDriver.insuranceExpiryDate = toNullIfEmpty(driver.insuranceExpiryDate);
    if ('insuranceCardExpiryDate' in driver) cleanedDriver.insuranceCardExpiryDate = toNullIfEmpty(driver.insuranceCardExpiryDate);
    if ('passportExpiryDate' in driver) cleanedDriver.passportExpiryDate = toNullIfEmpty(driver.passportExpiryDate);
    if ('licenseExpiryDate' in driver) cleanedDriver.licenseExpiryDate = toNullIfEmpty(driver.licenseExpiryDate);
    
    const [updatedDriver] = await db.update(drivers)
      .set(cleanedDriver)
      .where(eq(drivers.id, id))
      .returning();
    return updatedDriver;
  }

  async deleteDriver(id: number): Promise<boolean> {
    const result = await db.delete(drivers).where(eq(drivers.id, id));
    return true;
  }

  // ========================================
  // TRIPS (الرحلات)
  // ========================================

  async getTrips(filters?: { driverId?: number; tripType?: string; status?: string }): Promise<TripWithDetails[]> {
    const allTrips = await db.select().from(trips).orderBy(desc(trips.createdAt));
    const allDrivers = await db.select().from(drivers);
    const allAppointments = await db.select().from(appointments);

    let filteredTrips = allTrips;
    if (filters?.driverId) {
      filteredTrips = filteredTrips.filter(t => t.driverId === filters.driverId);
    }
    if (filters?.tripType) {
      filteredTrips = filteredTrips.filter(t => t.tripType === filters.tripType);
    }
    if (filters?.status) {
      filteredTrips = filteredTrips.filter(t => t.status === filters.status);
    }

    return filteredTrips.map(trip => ({
      ...trip,
      driver: allDrivers.find(d => d.id === trip.driverId),
      appointment: trip.appointmentId ? allAppointments.find(a => a.id === trip.appointmentId) : undefined,
    }));
  }

  async getTrip(id: number): Promise<TripWithDetails | undefined> {
    const [trip] = await db.select().from(trips).where(eq(trips.id, id));
    if (!trip) return undefined;

    const [driver] = await db.select().from(drivers).where(eq(drivers.id, trip.driverId));
    let appointment: Appointment | undefined;
    if (trip.appointmentId) {
      const [apt] = await db.select().from(appointments).where(eq(appointments.id, trip.appointmentId));
      appointment = apt;
    }

    return {
      ...trip,
      driver,
      appointment,
    };
  }

  async createTrip(trip: InsertTrip): Promise<Trip> {
    // Convert empty strings to null for optional timestamp fields
    const cleanedTrip = {
      ...trip,
      scheduledAt: trip.scheduledAt || null,
      startedAt: trip.startedAt || null,
      completedAt: trip.completedAt || null,
    };
    const [newTrip] = await db.insert(trips).values(cleanedTrip).returning();
    return newTrip;
  }

  async updateTrip(id: number, trip: Partial<InsertTrip>): Promise<Trip | undefined> {
    // Convert empty strings to null for optional timestamp fields
    const cleanedTrip = {
      ...trip,
      scheduledAt: trip.scheduledAt === '' ? null : trip.scheduledAt,
      startedAt: trip.startedAt === '' ? null : trip.startedAt,
      completedAt: trip.completedAt === '' ? null : trip.completedAt,
      updatedAt: new Date(),
    };
    const [updatedTrip] = await db.update(trips)
      .set(cleanedTrip)
      .where(eq(trips.id, id))
      .returning();
    return updatedTrip;
  }

  async deleteTrip(id: number): Promise<boolean> {
    await db.delete(trips).where(eq(trips.id, id));
    return true;
  }

  async generateTripNumber(): Promise<string> {
    const lastTrip = await db.select({ tripNumber: trips.tripNumber })
      .from(trips)
      .orderBy(desc(trips.id))
      .limit(1);

    let nextNumber = 1;
    if (lastTrip.length > 0 && lastTrip[0].tripNumber) {
      const match = lastTrip[0].tripNumber.match(/TR-(\d+)/);
      if (match) {
        nextNumber = parseInt(match[1], 10) + 1;
      }
    }

    return `TR-${nextNumber.toString().padStart(4, "0")}`;
  }

  // ========================================
  // CASH DISBURSEMENTS (صرف نقدي)
  // ========================================

  async getCashDisbursements(filters?: { 
    disbursementType?: string; 
    employeeId?: number; 
    driverId?: number;
    fromDate?: string;
    toDate?: string;
  }): Promise<CashDisbursementWithDetails[]> {
    let allDisbursements = await db.select().from(cashDisbursements).orderBy(desc(cashDisbursements.createdAt));
    
    // Apply filters
    if (filters?.disbursementType) {
      allDisbursements = allDisbursements.filter(d => d.disbursementType === filters.disbursementType);
    }
    if (filters?.employeeId) {
      allDisbursements = allDisbursements.filter(d => d.employeeId === filters.employeeId);
    }
    if (filters?.driverId) {
      allDisbursements = allDisbursements.filter(d => d.driverId === filters.driverId);
    }
    if (filters?.fromDate) {
      allDisbursements = allDisbursements.filter(d => d.disbursementDate >= filters.fromDate!);
    }
    if (filters?.toDate) {
      allDisbursements = allDisbursements.filter(d => d.disbursementDate <= filters.toDate!);
    }
    
    // Get related data
    const allEmployees = await db.select().from(employees);
    const allDrivers = await db.select().from(drivers);
    const allPaymentMethods = await db.select().from(paymentMethods);
    const allInstallments = await db.select().from(advanceInstallments);
    
    return allDisbursements.map(d => ({
      ...d,
      employee: d.employeeId ? allEmployees.find(e => e.id === d.employeeId) : undefined,
      driver: d.driverId ? allDrivers.find(dr => dr.id === d.driverId) : undefined,
      paymentMethod: allPaymentMethods.find(pm => pm.id === d.paymentMethodId),
      installments: allInstallments.filter(i => i.cashDisbursementId === d.id),
    }));
  }

  async getCashDisbursement(id: number): Promise<CashDisbursementWithDetails | undefined> {
    const [disbursement] = await db.select().from(cashDisbursements).where(eq(cashDisbursements.id, id));
    if (!disbursement) return undefined;

    const [employee] = disbursement.employeeId 
      ? await db.select().from(employees).where(eq(employees.id, disbursement.employeeId))
      : [undefined];
    const [driver] = disbursement.driverId
      ? await db.select().from(drivers).where(eq(drivers.id, disbursement.driverId))
      : [undefined];
    const [paymentMethod] = await db.select().from(paymentMethods).where(eq(paymentMethods.id, disbursement.paymentMethodId));
    const installments = await db.select().from(advanceInstallments).where(eq(advanceInstallments.cashDisbursementId, id));

    return {
      ...disbursement,
      employee: employee,
      driver: driver,
      paymentMethod: paymentMethod,
      installments: installments,
    };
  }

  async createCashDisbursement(disbursement: InsertCashDisbursement): Promise<CashDisbursementWithDetails> {
    // Calculate installment amount if installments are enabled
    let installmentAmount: number | null = null;
    if (disbursement.isInstallment && disbursement.installmentMonths && disbursement.installmentMonths > 0) {
      installmentAmount = disbursement.amount / disbursement.installmentMonths;
    }

    const [newDisbursement] = await db.insert(cashDisbursements).values({
      ...disbursement,
      installmentAmount,
    }).returning();

    // Create installment records if this is an installment advance
    if (disbursement.isInstallment && disbursement.installmentMonths && disbursement.installmentMonths > 0) {
      const installmentsToCreate: InsertAdvanceInstallment[] = [];
      const baseDate = new Date(disbursement.disbursementDate);
      
      for (let i = 1; i <= disbursement.installmentMonths; i++) {
        const dueDate = new Date(baseDate);
        dueDate.setMonth(dueDate.getMonth() + i);
        
        installmentsToCreate.push({
          cashDisbursementId: newDisbursement.id,
          installmentNumber: i,
          amount: installmentAmount!,
          dueDate: dueDate.toISOString().split('T')[0],
          status: 'pending',
        });
      }
      
      await db.insert(advanceInstallments).values(installmentsToCreate);
    }

    return this.getCashDisbursement(newDisbursement.id) as Promise<CashDisbursementWithDetails>;
  }

  async updateCashDisbursement(id: number, disbursement: Partial<InsertCashDisbursement>): Promise<CashDisbursement | undefined> {
    const [updated] = await db.update(cashDisbursements)
      .set({ ...disbursement, updatedAt: new Date() })
      .where(eq(cashDisbursements.id, id))
      .returning();
    return updated;
  }

  async deleteCashDisbursement(id: number): Promise<boolean> {
    // Delete related installments first
    await db.delete(advanceInstallments).where(eq(advanceInstallments.cashDisbursementId, id));
    await db.delete(cashDisbursements).where(eq(cashDisbursements.id, id));
    return true;
  }

  async generateDisbursementNumber(): Promise<string> {
    const lastDisbursement = await db.select({ disbursementNumber: cashDisbursements.disbursementNumber })
      .from(cashDisbursements)
      .orderBy(desc(cashDisbursements.id))
      .limit(1);

    let nextNumber = 1;
    if (lastDisbursement.length > 0 && lastDisbursement[0].disbursementNumber) {
      const match = lastDisbursement[0].disbursementNumber.match(/CD-(\d+)/);
      if (match) {
        nextNumber = parseInt(match[1], 10) + 1;
      }
    }

    return `CD-${nextNumber.toString().padStart(4, "0")}`;
  }

  async getEmployeeDisbursements(employeeId: number): Promise<CashDisbursementWithDetails[]> {
    return this.getCashDisbursements({ employeeId });
  }

  async getDriverDisbursements(driverId: number): Promise<CashDisbursementWithDetails[]> {
    return this.getCashDisbursements({ driverId });
  }

  async getAdvanceInstallments(cashDisbursementId: number): Promise<AdvanceInstallment[]> {
    return await db.select().from(advanceInstallments)
      .where(eq(advanceInstallments.cashDisbursementId, cashDisbursementId))
      .orderBy(asc(advanceInstallments.installmentNumber));
  }

  async updateInstallmentStatus(id: number, status: string, payrollLineId?: number): Promise<AdvanceInstallment | undefined> {
    const updateData: any = {
      status,
      updatedAt: new Date(),
    };
    
    if (status === 'deducted') {
      updateData.deductedAt = new Date();
      if (payrollLineId) {
        updateData.payrollLineId = payrollLineId;
      }
    }

    const [updated] = await db.update(advanceInstallments)
      .set(updateData)
      .where(eq(advanceInstallments.id, id))
      .returning();
    return updated;
  }

  // ========================================
  // DAILY REPORT (تقرير اليومية)
  // ========================================
  
  async getSalonDailyReport(filters: DailyReportFilters): Promise<SalonDailyReport> {
    const reportDate = filters.date;
    const startOfDay = `${reportDate}T00:00:00.000Z`;
    const endOfDay = `${reportDate}T23:59:59.999Z`;

    // 1. In-Salon Services Revenue (إيرادات الخدمات داخل الصالون)
    // From orders where status != 'canceled' and source = 'pos'
    const inSalonServicesResult = await db.select({
      subtotal: sql<number>`COALESCE(SUM(${orderServices.lineSubtotal}), 0)`,
      vat: sql<number>`COALESCE(SUM(${orderServices.vatAmount}), 0)`,
      total: sql<number>`COALESCE(SUM(${orderServices.lineTotal}), 0)`,
    })
    .from(orderServices)
    .innerJoin(orders, eq(orderServices.orderId, orders.id))
    .where(and(
      gte(orders.orderDate, new Date(startOfDay)),
      lte(orders.orderDate, new Date(endOfDay)),
      ne(orders.status, 'canceled'),
      ne(orders.status, 'returned'),
      eq(orders.source, 'pos')
    ));

    const inSalonServicesRevenue = {
      label: "إيرادات الخدمات داخل الصالون",
      subtotal: Number(inSalonServicesResult[0]?.subtotal || 0),
      vat: Number(inSalonServicesResult[0]?.vat || 0),
      total: Number(inSalonServicesResult[0]?.total || 0),
    };

    // 2. Home Services Revenue (إيرادات الخدمات المنزلية)
    // From appointments with home_service type where status = 'completed'
    const homeServicesResult = await db.select({
      subtotal: sql<number>`COALESCE(SUM(${appointmentServices.lineSubtotal}), 0)`,
      vat: sql<number>`COALESCE(SUM(${appointmentServices.vatAmount}), 0)`,
      total: sql<number>`COALESCE(SUM(${appointmentServices.lineTotal}), 0)`,
    })
    .from(appointmentServices)
    .innerJoin(appointments, eq(appointmentServices.appointmentId, appointments.id))
    .where(and(
      gte(appointments.completedAt, new Date(startOfDay)),
      lte(appointments.completedAt, new Date(endOfDay)),
      eq(appointments.status, 'completed'),
      eq(appointments.appointmentType, 'home_service')
    ));

    const homeServicesRevenue = {
      label: "إيرادات الخدمات المنزلية",
      subtotal: Number(homeServicesResult[0]?.subtotal || 0),
      vat: Number(homeServicesResult[0]?.vat || 0),
      total: Number(homeServicesResult[0]?.total || 0),
    };

    // 3. Returned Services Amount (مبالغ الخدمات المسترجعة)
    // From cancelled orders - services only
    const returnedServicesResult = await db.select({
      subtotal: sql<number>`COALESCE(SUM(${orderServices.lineSubtotal}), 0)`,
      vat: sql<number>`COALESCE(SUM(${orderServices.vatAmount}), 0)`,
      total: sql<number>`COALESCE(SUM(${orderServices.lineTotal}), 0)`,
    })
    .from(orderServices)
    .innerJoin(orders, eq(orderServices.orderId, orders.id))
    .where(and(
      gte(orders.cancelledAt, new Date(startOfDay)),
      lte(orders.cancelledAt, new Date(endOfDay)),
      or(eq(orders.status, 'canceled'), eq(orders.status, 'returned'))
    ));

    const returnedServicesAmount = {
      label: "مبالغ الخدمات المسترجعة",
      subtotal: Number(returnedServicesResult[0]?.subtotal || 0),
      vat: Number(returnedServicesResult[0]?.vat || 0),
      total: Number(returnedServicesResult[0]?.total || 0),
    };

    // 4. Product Sales Revenue (إيرادات مبيعات المنتجات)
    // From orders where status != 'canceled' - products only
    const productSalesResult = await db.select({
      subtotal: sql<number>`COALESCE(SUM(${orderProducts.lineSubtotal}), 0)`,
      vat: sql<number>`COALESCE(SUM(${orderProducts.vatAmount}), 0)`,
      total: sql<number>`COALESCE(SUM(${orderProducts.lineTotal}), 0)`,
    })
    .from(orderProducts)
    .innerJoin(orders, eq(orderProducts.orderId, orders.id))
    .where(and(
      gte(orders.orderDate, new Date(startOfDay)),
      lte(orders.orderDate, new Date(endOfDay)),
      ne(orders.status, 'canceled'),
      ne(orders.status, 'returned')
    ));

    const productSalesRevenue = {
      label: "إيرادات مبيعات المنتجات",
      subtotal: Number(productSalesResult[0]?.subtotal || 0),
      vat: Number(productSalesResult[0]?.vat || 0),
      total: Number(productSalesResult[0]?.total || 0),
    };

    // 5. Returned Products Amount (مبالغ المنتجات المسترجعة)
    // From cancelled orders - products only
    const returnedProductsResult = await db.select({
      subtotal: sql<number>`COALESCE(SUM(${orderProducts.lineSubtotal}), 0)`,
      vat: sql<number>`COALESCE(SUM(${orderProducts.vatAmount}), 0)`,
      total: sql<number>`COALESCE(SUM(${orderProducts.lineTotal}), 0)`,
    })
    .from(orderProducts)
    .innerJoin(orders, eq(orderProducts.orderId, orders.id))
    .where(and(
      gte(orders.cancelledAt, new Date(startOfDay)),
      lte(orders.cancelledAt, new Date(endOfDay)),
      or(eq(orders.status, 'canceled'), eq(orders.status, 'returned'))
    ));

    const returnedProductsAmount = {
      label: "مبالغ المنتجات المسترجعة",
      subtotal: Number(returnedProductsResult[0]?.subtotal || 0),
      vat: Number(returnedProductsResult[0]?.vat || 0),
      total: Number(returnedProductsResult[0]?.total || 0),
    };

    // Calculate totals
    const totalRevenueBeforeVat = 
      inSalonServicesRevenue.subtotal + 
      homeServicesRevenue.subtotal + 
      productSalesRevenue.subtotal - 
      returnedServicesAmount.subtotal - 
      returnedProductsAmount.subtotal;

    const totalVat = 
      inSalonServicesRevenue.vat + 
      homeServicesRevenue.vat + 
      productSalesRevenue.vat - 
      returnedServicesAmount.vat - 
      returnedProductsAmount.vat;

    const grandTotal = totalRevenueBeforeVat + totalVat;

    // 6. Payment Method Breakdown (تفصيل الإيرادات حسب طرق الدفع)
    const paymentMethodsResult = await db.select({
      paymentMethodId: orderPayments.paymentMethodId,
      paymentMethodName: paymentMethods.nameAr,
      amount: sql<number>`COALESCE(SUM(${orderPayments.amount}), 0)`,
    })
    .from(orderPayments)
    .innerJoin(paymentMethods, eq(orderPayments.paymentMethodId, paymentMethods.id))
    .innerJoin(orders, eq(orderPayments.orderId, orders.id))
    .where(and(
      eq(orderPayments.paymentDate, reportDate),
      ne(orders.status, 'canceled'),
      ne(orders.status, 'returned')
    ))
    .groupBy(orderPayments.paymentMethodId, paymentMethods.nameAr);

    // Get refunds by payment method (from cancelled order payments made that day)
    const refundsByMethodResult = await db.select({
      paymentMethodId: orderPayments.paymentMethodId,
      amount: sql<number>`COALESCE(SUM(${orderPayments.amount}), 0)`,
    })
    .from(orderPayments)
    .innerJoin(orders, eq(orderPayments.orderId, orders.id))
    .where(and(
      eq(orderPayments.paymentDate, reportDate),
      or(eq(orders.status, 'canceled'), eq(orders.status, 'returned'))
    ))
    .groupBy(orderPayments.paymentMethodId);

    const refundsMap = new Map<number, number>();
    refundsByMethodResult.forEach(r => {
      refundsMap.set(r.paymentMethodId, Number(r.amount));
    });

    // Build payment method breakdown
    const paymentMethodBreakdown: DailyReportPaymentMethodLine[] = [];
    let totalPaymentsReceived = 0;
    let totalRefunds = 0;

    for (const pm of paymentMethodsResult) {
      const revenue = Number(pm.amount);
      const refunds = refundsMap.get(pm.paymentMethodId) || 0;
      totalPaymentsReceived += revenue;
      totalRefunds += refunds;
      
      paymentMethodBreakdown.push({
        paymentMethodId: pm.paymentMethodId,
        paymentMethodName: pm.paymentMethodName || "غير معرف",
        revenue,
        refunds,
        net: revenue - refunds,
      });
    }

    // Add any payment methods that only have refunds
    for (const [pmId, refundAmount] of refundsMap) {
      if (!paymentMethodBreakdown.find(p => p.paymentMethodId === pmId)) {
        const pmDetails = await db.select().from(paymentMethods).where(eq(paymentMethods.id, pmId)).limit(1);
        totalRefunds += refundAmount;
        paymentMethodBreakdown.push({
          paymentMethodId: pmId,
          paymentMethodName: pmDetails[0]?.nameAr || "غير معرف",
          revenue: 0,
          refunds: refundAmount,
          net: -refundAmount,
        });
      }
    }

    const netPayments = totalPaymentsReceived - totalRefunds;

    // 7. Loyalty Points Redemption (استرداد نقاط الولاء)
    // This would be tracked in a separate table or in orders - for now, we'll estimate from orders
    // where loyalty points were used (if there's a field for it)
    const loyaltyPointsRedemption = 0; // Placeholder - would need to add loyalty tracking to orders

    // 8. Cash Summary (صافي النقدي)
    // Get cash payments for today
    const cashPaymentMethod = await db.select().from(paymentMethods)
      .where(or(
        sql`lower(${paymentMethods.nameAr}) LIKE '%نقد%'`,
        sql`lower(${paymentMethods.nameAr}) LIKE '%كاش%'`,
        eq(paymentMethods.code, 'cash')
      ))
      .limit(1);

    let todaysCashRevenue = 0;
    if (cashPaymentMethod.length > 0) {
      const cashEntry = paymentMethodBreakdown.find(p => p.paymentMethodId === cashPaymentMethod[0].id);
      todaysCashRevenue = cashEntry?.net || 0;
    }

    // Previous cash balance would need to be tracked separately or calculated from previous days
    // For now, we'll use 0 as placeholder
    const previousBalance = 0;
    const netCash = todaysCashRevenue + previousBalance;

    const cashSummary: DailyReportCashSummary = {
      todaysRevenue: todaysCashRevenue,
      previousBalance,
      netCash,
    };

    return {
      date: reportDate,
      inSalonServicesRevenue,
      homeServicesRevenue,
      returnedServicesAmount,
      productSalesRevenue,
      returnedProductsAmount,
      totalRevenueBeforeVat,
      totalVat,
      grandTotal,
      paymentMethodBreakdown,
      totalPaymentsReceived,
      totalRefunds,
      netPayments,
      loyaltyPointsRedemption,
      cashSummary,
    };
  }

  // ========================================
  // MARKETING REPORTS (تقارير التسويق)
  // ========================================
  
  async getMarketingReport(filters?: MarketingReportFilters): Promise<MarketingReportSummary> {
    const fromDate = filters?.fromDate ? new Date(filters.fromDate + 'T00:00:00.000Z') : new Date('2000-01-01');
    const toDate = filters?.toDate ? new Date(filters.toDate + 'T23:59:59.999Z') : new Date();
    
    // ======== COUPON STATISTICS ========
    
    // Get all coupons
    const allCoupons = await db.select().from(coupons);
    const today = new Date();
    const activeCoupons = allCoupons.filter(c => 
      c.isActive && 
      new Date(c.startDate) <= today && 
      new Date(c.endDate) >= today
    );
    
    // Get coupon usage from orders
    const couponUsageResult = await db.select({
      couponId: orders.couponId,
      couponCode: orders.couponCode,
      usageCount: sql<number>`COUNT(*)`,
      totalDiscountGiven: sql<number>`COALESCE(SUM(${orders.couponDiscountAmount}), 0)`,
      totalOrdersValue: sql<number>`COALESCE(SUM(${orders.totalAmount}), 0)`,
      lastUsedDate: sql<string>`MAX(${orders.orderDate})`,
    })
    .from(orders)
    .where(and(
      isNotNull(orders.couponId),
      gte(orders.orderDate, fromDate),
      lte(orders.orderDate, toDate),
      ne(orders.status, 'canceled')
    ))
    .groupBy(orders.couponId, orders.couponCode);
    
    // Map coupon IDs to coupon details
    const couponsMap = new Map(allCoupons.map(c => [c.id, c]));
    
    const couponUsageList: CouponUsageStat[] = couponUsageResult.map(cu => {
      const coupon = couponsMap.get(cu.couponId!);
      const usageCount = Number(cu.usageCount);
      const totalOrdersValue = Number(cu.totalOrdersValue);
      return {
        couponId: cu.couponId!,
        couponCode: cu.couponCode || coupon?.code || '',
        couponNameAr: coupon?.nameAr || '',
        discountType: coupon?.discountType || '',
        discountValue: coupon?.discountValue || 0,
        usageCount,
        totalDiscountGiven: Number(cu.totalDiscountGiven),
        totalOrdersValue,
        avgOrderValue: usageCount > 0 ? totalOrdersValue / usageCount : 0,
        lastUsedDate: cu.lastUsedDate || null,
      };
    }).sort((a, b) => b.usageCount - a.usageCount);
    
    const totalCouponUsage = couponUsageList.reduce((sum, c) => sum + c.usageCount, 0);
    const totalDiscountGiven = couponUsageList.reduce((sum, c) => sum + c.totalDiscountGiven, 0);
    
    // ======== PACKAGE STATISTICS ========
    
    // Get all packages
    const allPackages = await db.select().from(packages);
    const activePackages = allPackages.filter(p => 
      p.isActive && 
      new Date(p.startDate) <= today && 
      new Date(p.endDate) >= today
    );
    
    // Get package usage from order services
    const packageServicesUsage = await db.select({
      packageId: orderServices.packageId,
      orderId: orderServices.orderId,
      totalRevenue: sql<number>`COALESCE(SUM(${orderServices.lineTotal}), 0)`,
      servicesCount: sql<number>`COUNT(*)`,
    })
    .from(orderServices)
    .innerJoin(orders, eq(orderServices.orderId, orders.id))
    .where(and(
      isNotNull(orderServices.packageId),
      gte(orders.orderDate, fromDate),
      lte(orders.orderDate, toDate),
      ne(orders.status, 'canceled')
    ))
    .groupBy(orderServices.packageId, orderServices.orderId);
    
    // Get package usage from order products
    const packageProductsUsage = await db.select({
      packageId: orderProducts.packageId,
      orderId: orderProducts.orderId,
      totalRevenue: sql<number>`COALESCE(SUM(${orderProducts.lineTotal}), 0)`,
      productsCount: sql<number>`COUNT(*)`,
    })
    .from(orderProducts)
    .innerJoin(orders, eq(orderProducts.orderId, orders.id))
    .where(and(
      isNotNull(orderProducts.packageId),
      gte(orders.orderDate, fromDate),
      lte(orders.orderDate, toDate),
      ne(orders.status, 'canceled')
    ))
    .groupBy(orderProducts.packageId, orderProducts.orderId);
    
    // Aggregate by package
    const packageStatsMap = new Map<number, {
      totalSales: number;
      totalRevenue: number;
      servicesCount: number;
      productsCount: number;
      orderIds: Set<number>;
    }>();
    
    packageServicesUsage.forEach(psu => {
      if (!psu.packageId) return;
      const existing = packageStatsMap.get(psu.packageId) || {
        totalSales: 0,
        totalRevenue: 0,
        servicesCount: 0,
        productsCount: 0,
        orderIds: new Set<number>(),
      };
      existing.totalRevenue += Number(psu.totalRevenue);
      existing.servicesCount += Number(psu.servicesCount);
      existing.orderIds.add(psu.orderId);
      packageStatsMap.set(psu.packageId, existing);
    });
    
    packageProductsUsage.forEach(ppu => {
      if (!ppu.packageId) return;
      const existing = packageStatsMap.get(ppu.packageId) || {
        totalSales: 0,
        totalRevenue: 0,
        servicesCount: 0,
        productsCount: 0,
        orderIds: new Set<number>(),
      };
      existing.totalRevenue += Number(ppu.totalRevenue);
      existing.productsCount += Number(ppu.productsCount);
      existing.orderIds.add(ppu.orderId);
      packageStatsMap.set(ppu.packageId, existing);
    });
    
    // Get last sold date for each package
    const lastSoldByPackage = await db.select({
      packageId: orderServices.packageId,
      lastSoldDate: sql<string>`MAX(${orders.orderDate})`,
    })
    .from(orderServices)
    .innerJoin(orders, eq(orderServices.orderId, orders.id))
    .where(and(
      isNotNull(orderServices.packageId),
      ne(orders.status, 'canceled')
    ))
    .groupBy(orderServices.packageId);
    
    const lastSoldMap = new Map(lastSoldByPackage.map(ls => [ls.packageId, ls.lastSoldDate]));
    
    // Build package usage list
    const packagesMap = new Map(allPackages.map(p => [p.id, p]));
    const packageUsageList: PackageUsageStat[] = [];
    
    packageStatsMap.forEach((stats, packageId) => {
      const pkg = packagesMap.get(packageId);
      stats.totalSales = stats.orderIds.size;
      packageUsageList.push({
        packageId,
        packageNameAr: pkg?.nameAr || '',
        totalSales: stats.totalSales,
        totalRevenue: stats.totalRevenue,
        avgOrderValue: stats.totalSales > 0 ? stats.totalRevenue / stats.totalSales : 0,
        servicesCount: stats.servicesCount,
        productsCount: stats.productsCount,
        lastSoldDate: lastSoldMap.get(packageId) || null,
      });
    });
    
    packageUsageList.sort((a, b) => b.totalRevenue - a.totalRevenue);
    
    const totalPackagesSold = packageUsageList.reduce((sum, p) => sum + p.totalSales, 0);
    const totalPackageRevenue = packageUsageList.reduce((sum, p) => sum + p.totalRevenue, 0);
    
    // ======== TOP SERVICES IN PACKAGES ========
    const topServicesInPackagesResult = await db.select({
      serviceId: orderServices.serviceId,
      serviceNameAr: services.nameAr,
      packageCount: sql<number>`COUNT(DISTINCT ${orderServices.packageId})`,
      totalQuantitySold: sql<number>`COALESCE(SUM(${orderServices.quantity}), 0)`,
      totalRevenue: sql<number>`COALESCE(SUM(${orderServices.lineTotal}), 0)`,
    })
    .from(orderServices)
    .innerJoin(services, eq(orderServices.serviceId, services.id))
    .innerJoin(orders, eq(orderServices.orderId, orders.id))
    .where(and(
      isNotNull(orderServices.packageId),
      gte(orders.orderDate, fromDate),
      lte(orders.orderDate, toDate),
      ne(orders.status, 'canceled')
    ))
    .groupBy(orderServices.serviceId, services.nameAr)
    .orderBy(sql`SUM(${orderServices.lineTotal}) DESC`)
    .limit(10);
    
    const topServicesInPackages: PackageTopService[] = topServicesInPackagesResult.map(s => ({
      serviceId: s.serviceId,
      serviceNameAr: s.serviceNameAr,
      packageCount: Number(s.packageCount),
      totalQuantitySold: Number(s.totalQuantitySold),
      totalRevenue: Number(s.totalRevenue),
    }));
    
    // ======== TOP PRODUCTS IN PACKAGES ========
    const topProductsInPackagesResult = await db.select({
      productId: orderProducts.productId,
      productNameAr: products.nameAr,
      packageCount: sql<number>`COUNT(DISTINCT ${orderProducts.packageId})`,
      totalQuantitySold: sql<number>`COALESCE(SUM(${orderProducts.quantity}), 0)`,
      totalRevenue: sql<number>`COALESCE(SUM(${orderProducts.lineTotal}), 0)`,
    })
    .from(orderProducts)
    .innerJoin(products, eq(orderProducts.productId, products.id))
    .innerJoin(orders, eq(orderProducts.orderId, orders.id))
    .where(and(
      isNotNull(orderProducts.packageId),
      gte(orders.orderDate, fromDate),
      lte(orders.orderDate, toDate),
      ne(orders.status, 'canceled')
    ))
    .groupBy(orderProducts.productId, products.nameAr)
    .orderBy(sql`SUM(${orderProducts.lineTotal}) DESC`)
    .limit(10);
    
    const topProductsInPackages: PackageTopProduct[] = topProductsInPackagesResult.map(p => ({
      productId: p.productId,
      productNameAr: p.productNameAr,
      packageCount: Number(p.packageCount),
      totalQuantitySold: Number(p.totalQuantitySold),
      totalRevenue: Number(p.totalRevenue),
    }));
    
    return {
      couponStats: {
        totalCoupons: allCoupons.length,
        activeCoupons: activeCoupons.length,
        totalUsageCount: totalCouponUsage,
        totalDiscountGiven,
      },
      couponUsageList,
      packageStats: {
        totalPackages: allPackages.length,
        activePackages: activePackages.length,
        totalPackagesSold,
        totalPackageRevenue,
      },
      packageUsageList,
      topServicesInPackages,
      topProductsInPackages,
    };
  }

  // ======== VAT DECLARATION REPORT (ZATCA Compliant) ========
  async getVatDeclarationReport(fromDate: string, toDate: string): Promise<{
    period: { from: string; to: string };
    sales: {
      standardRatedSales: number;
      vatOnStandardRatedSales: number;
      zeroRatedSales: number;
      exemptSales: number;
      totalSales: number;
    };
    adjustments: {
      salesReturns: number;
      vatOnSalesReturns: number;
    };
    purchases: {
      standardRatedPurchases: number;
      vatOnStandardRatedPurchases: number;
      zeroRatedPurchases: number;
      exemptPurchases: number;
      totalPurchases: number;
    };
    summary: {
      totalVatDue: number;
      totalVatRecoverable: number;
      netVatPayable: number;
    };
    breakdown: {
      servicesSales: { subtotal: number; vat: number; total: number };
      productsSales: { subtotal: number; vat: number; total: number };
      purchaseInvoices: { subtotal: number; vat: number; total: number };
    };
  }> {
    const startDate = new Date(fromDate);
    const endDate = new Date(toDate);
    endDate.setHours(23, 59, 59, 999);

    // ======== SALES DATA ========
    // Services sales (from completed orders)
    const servicesSalesResult = await db.select({
      subtotal: sql<number>`COALESCE(SUM(${orderServices.lineSubtotal}), 0)`,
      vat: sql<number>`COALESCE(SUM(${orderServices.vatAmount}), 0)`,
      total: sql<number>`COALESCE(SUM(${orderServices.lineTotal}), 0)`,
    })
    .from(orderServices)
    .innerJoin(orders, eq(orderServices.orderId, orders.id))
    .where(and(
      gte(orders.orderDate, startDate),
      lte(orders.orderDate, endDate),
      ne(orders.status, 'canceled'),
      ne(orders.status, 'returned')
    ));

    const servicesSales = {
      subtotal: Number(servicesSalesResult[0]?.subtotal || 0),
      vat: Number(servicesSalesResult[0]?.vat || 0),
      total: Number(servicesSalesResult[0]?.total || 0),
    };

    // Products sales (from completed orders)
    const productsSalesResult = await db.select({
      subtotal: sql<number>`COALESCE(SUM(${orderProducts.lineSubtotal}), 0)`,
      vat: sql<number>`COALESCE(SUM(${orderProducts.vatAmount}), 0)`,
      total: sql<number>`COALESCE(SUM(${orderProducts.lineTotal}), 0)`,
    })
    .from(orderProducts)
    .innerJoin(orders, eq(orderProducts.orderId, orders.id))
    .where(and(
      gte(orders.orderDate, startDate),
      lte(orders.orderDate, endDate),
      ne(orders.status, 'canceled'),
      ne(orders.status, 'returned')
    ));

    const productsSales = {
      subtotal: Number(productsSalesResult[0]?.subtotal || 0),
      vat: Number(productsSalesResult[0]?.vat || 0),
      total: Number(productsSalesResult[0]?.total || 0),
    };

    // Sales returns (cancelled/returned orders)
    const salesReturnsServicesResult = await db.select({
      subtotal: sql<number>`COALESCE(SUM(${orderServices.lineSubtotal}), 0)`,
      vat: sql<number>`COALESCE(SUM(${orderServices.vatAmount}), 0)`,
    })
    .from(orderServices)
    .innerJoin(orders, eq(orderServices.orderId, orders.id))
    .where(and(
      gte(orders.cancelledAt, startDate),
      lte(orders.cancelledAt, endDate),
      or(eq(orders.status, 'canceled'), eq(orders.status, 'returned'))
    ));

    const salesReturnsProductsResult = await db.select({
      subtotal: sql<number>`COALESCE(SUM(${orderProducts.lineSubtotal}), 0)`,
      vat: sql<number>`COALESCE(SUM(${orderProducts.vatAmount}), 0)`,
    })
    .from(orderProducts)
    .innerJoin(orders, eq(orderProducts.orderId, orders.id))
    .where(and(
      gte(orders.cancelledAt, startDate),
      lte(orders.cancelledAt, endDate),
      or(eq(orders.status, 'canceled'), eq(orders.status, 'returned'))
    ));

    const salesReturns = 
      Number(salesReturnsServicesResult[0]?.subtotal || 0) +
      Number(salesReturnsProductsResult[0]?.subtotal || 0);
    
    const vatOnSalesReturns = 
      Number(salesReturnsServicesResult[0]?.vat || 0) +
      Number(salesReturnsProductsResult[0]?.vat || 0);

    // ======== PURCHASES DATA ========
    // Purchase invoices VAT
    const purchasesResult = await db.select({
      subtotal: sql<number>`COALESCE(SUM(${purchaseInvoices.subtotalAmount}), 0)`,
      vat: sql<number>`COALESCE(SUM(${purchaseInvoices.vatAmount}), 0)`,
      total: sql<number>`COALESCE(SUM(${purchaseInvoices.totalAmount}), 0)`,
    })
    .from(purchaseInvoices)
    .where(and(
      gte(purchaseInvoices.invoiceDate, fromDate),
      lte(purchaseInvoices.invoiceDate, toDate)
    ));

    const purchaseInvoicesData = {
      subtotal: Number(purchasesResult[0]?.subtotal || 0),
      vat: Number(purchasesResult[0]?.vat || 0),
      total: Number(purchasesResult[0]?.total || 0),
    };

    // Calculate totals
    const totalStandardRatedSales = servicesSales.subtotal + productsSales.subtotal;
    const totalVatOnSales = servicesSales.vat + productsSales.vat;
    const totalSales = servicesSales.total + productsSales.total;

    const totalStandardRatedPurchases = purchaseInvoicesData.subtotal;
    const totalVatOnPurchases = purchaseInvoicesData.vat;
    const totalPurchases = purchaseInvoicesData.total;

    // Net VAT calculation (VAT due - VAT recoverable + adjustments)
    const totalVatDue = totalVatOnSales - vatOnSalesReturns;
    const totalVatRecoverable = totalVatOnPurchases;
    const netVatPayable = totalVatDue - totalVatRecoverable;

    return {
      period: { from: fromDate, to: toDate },
      sales: {
        standardRatedSales: totalStandardRatedSales - salesReturns,
        vatOnStandardRatedSales: totalVatDue,
        zeroRatedSales: 0, // Would need separate tracking for zero-rated items
        exemptSales: 0, // Would need separate tracking for exempt items
        totalSales: totalSales - salesReturns - vatOnSalesReturns,
      },
      adjustments: {
        salesReturns,
        vatOnSalesReturns,
      },
      purchases: {
        standardRatedPurchases: totalStandardRatedPurchases,
        vatOnStandardRatedPurchases: totalVatRecoverable,
        zeroRatedPurchases: 0, // Would need separate tracking
        exemptPurchases: 0, // Would need separate tracking
        totalPurchases,
      },
      summary: {
        totalVatDue,
        totalVatRecoverable,
        netVatPayable,
      },
      breakdown: {
        servicesSales,
        productsSales,
        purchaseInvoices: purchaseInvoicesData,
      },
    };
  }

  // ========================================
  // COMMISSION MODULE (نظام العمولات)
  // ========================================

  async getCommissionProfiles(): Promise<CommissionProfileWithSources[]> {
    const profiles = await db.select().from(commissionProfiles);
    const sources = await db.select().from(commissionTargetSources);
    const allEmployees = await db.select().from(employees);
    
    return profiles.map(profile => ({
      ...profile,
      employee: allEmployees.find(e => e.id === profile.employeeId),
      targetSources: sources.filter(s => s.profileId === profile.id),
    }));
  }

  async getCommissionProfile(id: number): Promise<CommissionProfileWithSources | undefined> {
    const profile = await db.select().from(commissionProfiles).where(eq(commissionProfiles.id, id)).limit(1);
    if (!profile[0]) return undefined;
    
    const sources = await db.select().from(commissionTargetSources).where(eq(commissionTargetSources.profileId, id));
    const employee = await db.select().from(employees).where(eq(employees.id, profile[0].employeeId)).limit(1);
    
    return {
      ...profile[0],
      employee: employee[0],
      targetSources: sources,
    };
  }

  async getCommissionProfileByEmployee(employeeId: number): Promise<CommissionProfileWithSources | undefined> {
    const profile = await db.select().from(commissionProfiles).where(eq(commissionProfiles.employeeId, employeeId)).limit(1);
    if (!profile[0]) return undefined;
    
    const sources = await db.select().from(commissionTargetSources).where(eq(commissionTargetSources.profileId, profile[0].id));
    const employee = await db.select().from(employees).where(eq(employees.id, employeeId)).limit(1);
    
    return {
      ...profile[0],
      employee: employee[0],
      targetSources: sources,
    };
  }

  async createCommissionProfile(profile: InsertCommissionProfile, targetSources?: string[]): Promise<CommissionProfileWithSources> {
    const [created] = await db.insert(commissionProfiles).values(profile).returning();
    
    // Create target sources if provided
    const createdSources: CommissionTargetSource[] = [];
    if (targetSources && targetSources.length > 0) {
      for (const sourceType of targetSources) {
        const [source] = await db.insert(commissionTargetSources).values({
          profileId: created.id,
          sourceType,
          isEnabled: true,
        }).returning();
        createdSources.push(source);
      }
    }
    
    const employee = await db.select().from(employees).where(eq(employees.id, created.employeeId)).limit(1);
    
    return {
      ...created,
      employee: employee[0],
      targetSources: createdSources,
    };
  }

  async updateCommissionProfile(id: number, profile: Partial<InsertCommissionProfile>, targetSources?: string[]): Promise<CommissionProfileWithSources | undefined> {
    const [updated] = await db.update(commissionProfiles)
      .set({ ...profile, updatedAt: new Date() })
      .where(eq(commissionProfiles.id, id))
      .returning();
    
    if (!updated) return undefined;
    
    // Update target sources if provided
    if (targetSources !== undefined) {
      // Delete existing sources
      await db.delete(commissionTargetSources).where(eq(commissionTargetSources.profileId, id));
      
      // Create new sources
      for (const sourceType of targetSources) {
        await db.insert(commissionTargetSources).values({
          profileId: id,
          sourceType,
          isEnabled: true,
        });
      }
    }
    
    const sources = await db.select().from(commissionTargetSources).where(eq(commissionTargetSources.profileId, id));
    const employee = await db.select().from(employees).where(eq(employees.id, updated.employeeId)).limit(1);
    
    return {
      ...updated,
      employee: employee[0],
      targetSources: sources,
    };
  }

  async deleteCommissionProfile(id: number): Promise<boolean> {
    // Delete target sources first
    await db.delete(commissionTargetSources).where(eq(commissionTargetSources.profileId, id));
    // Delete transactions
    await db.delete(commissionTransactions).where(eq(commissionTransactions.profileId, id));
    // Delete snapshots
    await db.delete(commissionPerformanceSnapshots).where(eq(commissionPerformanceSnapshots.profileId, id));
    // Delete profile
    const result = await db.delete(commissionProfiles).where(eq(commissionProfiles.id, id)).returning();
    return result.length > 0;
  }

  async createCommissionTransaction(transaction: InsertCommissionTransaction): Promise<CommissionTransaction> {
    const [created] = await db.insert(commissionTransactions).values(transaction).returning();
    return created;
  }

  async getCommissionTransactions(profileId: number, periodMonth?: number, periodYear?: number): Promise<CommissionTransaction[]> {
    let query = db.select().from(commissionTransactions).where(eq(commissionTransactions.profileId, profileId));
    
    if (periodMonth && periodYear) {
      query = db.select().from(commissionTransactions).where(
        and(
          eq(commissionTransactions.profileId, profileId),
          eq(commissionTransactions.periodMonth, periodMonth),
          eq(commissionTransactions.periodYear, periodYear)
        )
      );
    }
    
    return query;
  }

  async getEmployeeCommissionReport(employeeId: number, periodMonth: number, periodYear: number): Promise<EmployeeCommissionReport> {
    // Get employee
    const [employee] = await db.select().from(employees).where(eq(employees.id, employeeId)).limit(1);
    if (!employee) throw new Error("الموظف غير موجود");
    
    // Get commission profile
    const profile = await this.getCommissionProfileByEmployee(employeeId);
    
    // Calculate date range for the period (using Date objects for timestamps)
    const startDateObj = new Date(periodYear, periodMonth - 1, 1); // First day of month
    const endDateObj = new Date(periodYear, periodMonth, 0, 23, 59, 59); // Last day of month at 23:59:59
    const startDateStr = `${periodYear}-${String(periodMonth).padStart(2, '0')}-01`;
    const endDateStr = endDateObj.toISOString().split('T')[0]; // For string comparisons
    
    // Get all orders created by this employee in the period
    const allOrders = await db.select().from(orders)
      .where(and(
        eq(orders.createdByEmployeeId, employeeId),
        gte(orders.orderDate, startDateObj),
        lte(orders.orderDate, endDateObj)
      ));
    
    // Get all order services executed by this employee in the period
    const allOrderServices = await db.select().from(orderServices);
    const allServicesInfo = await db.select().from(services);
    
    // Get services this employee executed (completed status)
    const executedServices: EmployeeServiceActivity[] = [];
    
    for (const os of allOrderServices) {
      if (os.executingEmployeeId !== employeeId) continue;
      if (os.status !== 'completed') continue;
      
      // Get the order for this service
      const order = await db.select().from(orders).where(eq(orders.id, os.orderId)).limit(1);
      if (!order[0]) continue;
      
      // Check if order is within period (convert to string for comparison if needed)
      const orderDateStr = order[0].orderDate instanceof Date 
        ? order[0].orderDate.toISOString().split('T')[0] 
        : String(order[0].orderDate).split('T')[0];
      if (orderDateStr < startDateStr || orderDateStr > endDateStr) continue;
      
      const serviceInfo = allServicesInfo.find(s => s.id === os.serviceId);
      
      // Calculate inventory deduction for this service (from service products)
      let inventoryDeduction = 0;
      const serviceProducts = await db.select().from(serviceProducts).where(eq(serviceProducts.serviceId, os.serviceId));
      for (const sp of serviceProducts) {
        const product = await db.select().from(products).where(eq(products.id, sp.productId)).limit(1);
        if (product[0]) {
          // Get average cost from stock batches
          const batches = await db.select().from(stockBatches).where(eq(stockBatches.productId, sp.productId));
          const avgCost = batches.length > 0 
            ? batches.reduce((sum, b) => sum + (b.unitCost || 0), 0) / batches.length 
            : 0;
          inventoryDeduction += (sp.quantity || 1) * avgCost * os.quantity;
        }
      }
      
      executedServices.push({
        id: os.id,
        date: order[0].orderDate,
        type: 'order',
        referenceNumber: order[0].orderNumber,
        referenceId: order[0].id,
        serviceName: serviceInfo?.nameAr || 'خدمة غير معروفة',
        serviceId: os.serviceId,
        quantity: os.quantity,
        amount: os.lineTotal,
        inventoryDeduction,
        netAmount: os.lineTotal - inventoryDeduction,
        status: os.status,
        completedAt: os.completedAt ? (os.completedAt instanceof Date ? os.completedAt.toISOString() : String(os.completedAt)) : null,
      });
    }
    
    // Get appointment services executed by this employee
    const allAppointments = await db.select().from(appointments)
      .where(and(
        gte(appointments.scheduledAt, startDateObj),
        lte(appointments.scheduledAt, endDateObj)
      ));
    
    const allAppointmentServices = await db.select().from(appointmentServices);
    
    for (const appt of allAppointments) {
      const apptServices = allAppointmentServices.filter(as => as.appointmentId === appt.id);
      
      for (const as of apptServices) {
        // Check if this employee executed it
        const executingEmployeeId = (as as any).executingEmployeeId || as.preferredEmployeeId;
        if (executingEmployeeId !== employeeId) continue;
        
        const status = (as as any).status || 'completed';
        if (status !== 'completed') continue;
        
        const serviceInfo = allServicesInfo.find(s => s.id === as.serviceId);
        
        // Calculate inventory deduction
        let inventoryDeduction = 0;
        const svcProducts = await db.select().from(serviceProducts).where(eq(serviceProducts.serviceId, as.serviceId));
        for (const sp of svcProducts) {
          const product = await db.select().from(products).where(eq(products.id, sp.productId)).limit(1);
          if (product[0]) {
            const batches = await db.select().from(stockBatches).where(eq(stockBatches.productId, sp.productId));
            const avgCost = batches.length > 0 
              ? batches.reduce((sum, b) => sum + (b.unitCost || 0), 0) / batches.length 
              : 0;
            inventoryDeduction += (sp.quantity || 1) * avgCost * as.quantity;
          }
        }
        
        const apptDate = appt.scheduledAt instanceof Date ? appt.scheduledAt.toISOString().split('T')[0] : String(appt.scheduledAt).split('T')[0];
        const asCompletedAt = (as as any).completedAt;
        
        executedServices.push({
          id: as.id,
          date: apptDate,
          type: 'appointment',
          referenceNumber: appt.appointmentNumber,
          referenceId: appt.id,
          serviceName: serviceInfo?.nameAr || 'خدمة غير معروفة',
          serviceId: as.serviceId,
          quantity: as.quantity,
          amount: as.lineTotal,
          inventoryDeduction,
          netAmount: as.lineTotal - inventoryDeduction,
          status: status,
          completedAt: asCompletedAt ? (asCompletedAt instanceof Date ? asCompletedAt.toISOString() : String(asCompletedAt)) : null,
        });
      }
    }
    
    // Get order returns related to services this employee executed
    const allReturns = await db.select().from(orderReturns)
      .where(and(
        gte(orderReturns.returnDate, startDateObj),
        lte(orderReturns.returnDate, endDateObj)
      ));
    
    const allReturnLines = await db.select().from(orderReturnLines);
    
    // Calculate returns for executed services
    for (const ret of allReturns) {
      const retLines = allReturnLines.filter(rl => rl.returnId === ret.id);
      for (const line of retLines) {
        if (!line.orderServiceId) continue;
        
        // Check if this service was executed by this employee
        const os = allOrderServices.find(s => s.id === line.orderServiceId);
        if (!os || os.executingEmployeeId !== employeeId) continue;
        
        const serviceInfo = allServicesInfo.find(s => s.id === os.serviceId);
        
        const retDate = ret.returnDate instanceof Date ? ret.returnDate.toISOString().split('T')[0] : String(ret.returnDate).split('T')[0];
        const retDateFull = ret.returnDate instanceof Date ? ret.returnDate.toISOString() : String(ret.returnDate);
        
        executedServices.push({
          id: line.id,
          date: retDate,
          type: 'return',
          referenceNumber: ret.returnNumber,
          referenceId: ret.id,
          serviceName: serviceInfo?.nameAr || 'خدمة غير معروفة',
          serviceId: os.serviceId,
          quantity: -line.quantity,
          amount: -line.refundAmount,
          inventoryDeduction: 0,
          netAmount: -line.refundAmount,
          status: 'returned',
          completedAt: retDateFull,
        });
      }
    }
    
    // Get orders created by this employee
    const createdOrders: EmployeeOrderActivity[] = [];
    
    for (const order of allOrders) {
      // Calculate returns for this order
      const orderReturnsData = allReturns.filter(r => r.orderId === order.id);
      const returnsAmount = orderReturnsData.reduce((sum, r) => sum + r.refundAmount, 0);
      
      createdOrders.push({
        id: order.id,
        date: order.orderDate,
        type: 'order',
        referenceNumber: order.orderNumber,
        referenceId: order.id,
        totalAmount: order.grandTotal,
        returnsAmount,
        netAmount: order.grandTotal - returnsAmount,
        status: order.status,
      });
    }
    
    // Get appointments created by this employee
    const createdAppointments = await db.select().from(appointments)
      .where(and(
        eq(appointments.createdByEmployeeId, employeeId),
        gte(appointments.scheduledAt, startDateObj),
        lte(appointments.scheduledAt, endDateObj)
      ));
    
    for (const appt of createdAppointments) {
      const apptDateStr = appt.scheduledAt instanceof Date ? appt.scheduledAt.toISOString().split('T')[0] : String(appt.scheduledAt).split('T')[0];
      createdOrders.push({
        id: appt.id,
        date: apptDateStr,
        type: 'appointment',
        referenceNumber: appt.appointmentNumber,
        referenceId: appt.id,
        totalAmount: appt.totalAmount,
        returnsAmount: 0, // Appointments don't have returns yet
        netAmount: appt.totalAmount,
        status: appt.status,
      });
    }
    
    // Get cafeteria sales by this employee
    const cafeteriaSalesData = await db.select().from(cafeteriaSalesOrders)
      .where(and(
        eq(cafeteriaSalesOrders.createdByEmployeeId, employeeId),
        gte(cafeteriaSalesOrders.orderDate, startDateObj),
        lte(cafeteriaSalesOrders.orderDate, endDateObj)
      ));
    
    const cafeteriaSales = cafeteriaSalesData.map(order => {
      const cafDateStr = order.orderDate instanceof Date ? order.orderDate.toISOString().split('T')[0] : String(order.orderDate).split('T')[0];
      return {
        orderId: order.id,
        orderNumber: order.orderNumber,
        date: cafDateStr,
        totalAmount: order.totalAmount,
        status: order.status,
      };
    });
    
    // Get product sales from orders created by this employee
    const orderProductsData = await db.select().from(orderProducts);
    const allProductsInfo = await db.select().from(products);
    
    const productSales: EmployeeCommissionReport['productSales'] = [];
    
    for (const order of allOrders) {
      const orderProds = orderProductsData.filter(op => op.orderId === order.id);
      for (const op of orderProds) {
        const productInfo = allProductsInfo.find(p => p.id === op.productId);
        
        // Calculate returns for this product
        const orderRet = allReturns.find(r => r.orderId === order.id);
        let returnedAmount = 0;
        if (orderRet) {
          const retLines = allReturnLines.filter(rl => rl.returnId === orderRet.id && rl.orderProductId === op.id);
          returnedAmount = retLines.reduce((sum, rl) => sum + rl.refundAmount, 0);
        }
        
        productSales.push({
          orderId: order.id,
          orderNumber: order.orderNumber,
          date: order.orderDate,
          productName: productInfo?.nameAr || 'منتج غير معروف',
          quantity: op.quantity,
          amount: op.lineTotal,
          returnedAmount,
        });
      }
    }
    
    // Calculate summary totals
    const executedServicesTotal = executedServices
      .filter(s => s.type !== 'return')
      .reduce((sum, s) => sum + s.amount, 0);
    const inventoryDeductionTotal = executedServices
      .filter(s => s.type !== 'return')
      .reduce((sum, s) => sum + s.inventoryDeduction, 0);
    const executedServicesReturns = executedServices
      .filter(s => s.type === 'return')
      .reduce((sum, s) => sum + Math.abs(s.amount), 0);
    
    const createdOrdersTotal = createdOrders.reduce((sum, o) => sum + o.totalAmount, 0);
    const ordersReturnsTotal = createdOrders.reduce((sum, o) => sum + o.returnsAmount, 0);
    
    const cafeteriaSalesTotal = cafeteriaSales.reduce((sum, s) => sum + s.totalAmount, 0);
    const cafeteriaReturnsTotal = 0; // TODO: Implement cafeteria returns tracking
    
    const productSalesTotal = productSales.reduce((sum, p) => sum + p.amount, 0);
    const productReturnsTotal = productSales.reduce((sum, p) => sum + p.returnedAmount, 0);
    
    // Calculate gross and deductions based on profile target sources
    let grossTotal = 0;
    let deductionsTotal = 0;
    
    if (profile) {
      const enabledSources = profile.targetSources.filter(s => s.isEnabled).map(s => s.sourceType);
      
      if (enabledSources.includes('executed_services')) {
        grossTotal += executedServicesTotal;
        deductionsTotal += inventoryDeductionTotal + executedServicesReturns;
      }
      if (enabledSources.includes('created_orders')) {
        grossTotal += createdOrdersTotal;
        deductionsTotal += ordersReturnsTotal;
      }
      if (enabledSources.includes('cafeteria_sales')) {
        grossTotal += cafeteriaSalesTotal;
        deductionsTotal += cafeteriaReturnsTotal;
      }
      if (enabledSources.includes('product_sales')) {
        grossTotal += productSalesTotal;
        deductionsTotal += productReturnsTotal;
      }
    }
    
    const netTotal = grossTotal - deductionsTotal;
    const targetAmount = profile?.monthlyTarget || 0;
    const achievementRate = targetAmount > 0 ? (netTotal / targetAmount) * 100 : 0;
    const commissionRate = profile?.commissionRate || 0;
    const commissionAmount = achievementRate >= 100 ? (netTotal * commissionRate / 100) : 0;
    
    return {
      employeeId,
      employeeName: employee.fullNameAr || employee.nameAr || 'غير محدد',
      periodMonth,
      periodYear,
      profile,
      executedServices,
      createdOrders,
      cafeteriaSales,
      productSales,
      summary: {
        executedServicesTotal,
        inventoryDeductionTotal,
        createdOrdersTotal,
        ordersReturnsTotal,
        cafeteriaSalesTotal,
        cafeteriaReturnsTotal,
        productSalesTotal,
        productReturnsTotal,
        grossTotal,
        deductionsTotal,
        netTotal,
        targetAmount,
        achievementRate,
        commissionRate,
        commissionAmount,
      },
    };
  }

  async settleCommission(profileId: number, periodMonth: number, periodYear: number, processedByEmployeeId?: number): Promise<CommissionPerformanceSnapshot> {
    // Get the commission report
    const profile = await this.getCommissionProfile(profileId);
    if (!profile) throw new Error("ملف العمولات غير موجود");
    
    const report = await this.getEmployeeCommissionReport(profile.employeeId, periodMonth, periodYear);
    
    // Check if already settled
    const existing = await db.select().from(commissionPerformanceSnapshots)
      .where(and(
        eq(commissionPerformanceSnapshots.profileId, profileId),
        eq(commissionPerformanceSnapshots.periodMonth, periodMonth),
        eq(commissionPerformanceSnapshots.periodYear, periodYear)
      ))
      .limit(1);
    
    if (existing[0]) {
      throw new Error("تم تسوية العمولة لهذه الفترة مسبقاً");
    }
    
    // Create snapshot
    const [snapshot] = await db.insert(commissionPerformanceSnapshots).values({
      profileId,
      periodMonth,
      periodYear,
      executedServicesAmount: report.summary.executedServicesTotal,
      inventoryDeductionAmount: report.summary.inventoryDeductionTotal,
      createdOrdersAmount: report.summary.createdOrdersTotal,
      ordersReturnsAmount: report.summary.ordersReturnsTotal,
      cafeteriaSalesAmount: report.summary.cafeteriaSalesTotal,
      cafeteriaReturnsAmount: report.summary.cafeteriaReturnsTotal,
      productSalesAmount: report.summary.productSalesTotal,
      productReturnsAmount: report.summary.productReturnsTotal,
      grossAmount: report.summary.grossTotal,
      deductionsAmount: report.summary.deductionsTotal,
      netAmount: report.summary.netTotal,
      targetAmount: report.summary.targetAmount,
      achievementRate: report.summary.achievementRate,
      commissionRate: report.summary.commissionRate,
      commissionAmount: report.summary.commissionAmount,
      isPaid: false,
    }).returning();
    
    // Create accrual transaction
    if (report.summary.commissionAmount > 0) {
      await db.insert(commissionTransactions).values({
        profileId,
        transactionType: 'accrual',
        amount: report.summary.commissionAmount,
        periodMonth,
        periodYear,
        description: `استحقاق عمولة ${periodMonth}/${periodYear}`,
        processedByEmployeeId,
      });
    }
    
    return snapshot;
  }

  async payCommission(snapshotId: number, paidByEmployeeId: number): Promise<CommissionPerformanceSnapshot> {
    const [snapshot] = await db.select().from(commissionPerformanceSnapshots)
      .where(eq(commissionPerformanceSnapshots.id, snapshotId))
      .limit(1);
    
    if (!snapshot) throw new Error("سجل العمولة غير موجود");
    if (snapshot.isPaid) throw new Error("تم سداد هذه العمولة مسبقاً");
    
    // Update snapshot as paid
    const [updated] = await db.update(commissionPerformanceSnapshots)
      .set({
        isPaid: true,
        paidAt: new Date(),
        paidByEmployeeId,
        updatedAt: new Date(),
      })
      .where(eq(commissionPerformanceSnapshots.id, snapshotId))
      .returning();
    
    // Create payment transaction
    await db.insert(commissionTransactions).values({
      profileId: snapshot.profileId,
      transactionType: 'payment',
      amount: -snapshot.commissionAmount, // Negative for payment
      periodMonth: snapshot.periodMonth,
      periodYear: snapshot.periodYear,
      description: `سداد عمولة ${snapshot.periodMonth}/${snapshot.periodYear}`,
      processedByEmployeeId: paidByEmployeeId,
    });
    
    return updated;
  }

  async getCommissionSnapshots(profileId?: number): Promise<CommissionPerformanceSnapshot[]> {
    if (profileId) {
      return db.select().from(commissionPerformanceSnapshots)
        .where(eq(commissionPerformanceSnapshots.profileId, profileId))
        .orderBy(desc(commissionPerformanceSnapshots.periodYear), desc(commissionPerformanceSnapshots.periodMonth));
    }
    return db.select().from(commissionPerformanceSnapshots)
      .orderBy(desc(commissionPerformanceSnapshots.periodYear), desc(commissionPerformanceSnapshots.periodMonth));
  }

  // ========================================
  // INVENTORY WITHDRAWAL ORDERS (أوامر سحب المخزون)
  // ========================================

  async getInventoryWithdrawalOrders(): Promise<InventoryWithdrawalOrderWithDetails[]> {
    const allOrders = await db.select().from(inventoryWithdrawalOrders)
      .orderBy(desc(inventoryWithdrawalOrders.createdAt));
    
    const allEmployees = await db.select().from(employees);
    const allProducts = await db.select().from(products);
    const allLines = await db.select().from(inventoryWithdrawalLines);
    const allJournalEntries = await db.select().from(journalEntries);
    
    return allOrders.map(order => {
      const lines = allLines.filter(l => l.orderId === order.id).map(line => ({
        ...line,
        product: allProducts.find(p => p.id === line.productId),
      }));
      
      return {
        ...order,
        responsibleEmployee: allEmployees.find(e => e.id === order.responsibleEmployeeId),
        confirmedByEmployee: allEmployees.find(e => e.id === order.confirmedByEmployeeId),
        lines,
        journalEntry: allJournalEntries.find(j => j.id === order.journalEntryId),
      };
    });
  }

  async getInventoryWithdrawalOrder(id: number): Promise<InventoryWithdrawalOrderWithDetails | undefined> {
    const [order] = await db.select().from(inventoryWithdrawalOrders)
      .where(eq(inventoryWithdrawalOrders.id, id));
    
    if (!order) return undefined;
    
    const allEmployees = await db.select().from(employees);
    const allProducts = await db.select().from(products);
    const lines = await db.select().from(inventoryWithdrawalLines)
      .where(eq(inventoryWithdrawalLines.orderId, id));
    
    let journalEntry: JournalEntry | undefined;
    if (order.journalEntryId) {
      const [je] = await db.select().from(journalEntries)
        .where(eq(journalEntries.id, order.journalEntryId));
      journalEntry = je;
    }
    
    return {
      ...order,
      responsibleEmployee: allEmployees.find(e => e.id === order.responsibleEmployeeId),
      confirmedByEmployee: allEmployees.find(e => e.id === order.confirmedByEmployeeId),
      lines: lines.map(line => ({
        ...line,
        product: allProducts.find(p => p.id === line.productId),
      })),
      journalEntry,
    };
  }

  async createInventoryWithdrawalOrder(order: InsertInventoryWithdrawalOrder, lines: Omit<InsertInventoryWithdrawalLine, 'orderId'>[]): Promise<InventoryWithdrawalOrderWithDetails> {
    // Generate order number
    const orderNumber = `WD-${Date.now()}`;
    
    // Validate products and calculate costs
    let totalCost = 0;
    const linesToInsert = [];
    
    for (const line of lines) {
      const [product] = await db.select().from(products).where(eq(products.id, line.productId));
      if (!product) {
        throw new Error(`المنتج غير موجود: ${line.productId}`);
      }
      
      const unitCost = product.averageUnitCost || 0;
      const lineTotalCost = line.quantity * unitCost;
      totalCost += lineTotalCost;
      
      linesToInsert.push({
        ...line,
        unitCost,
        totalCost: lineTotalCost,
      });
    }
    
    // Create order
    const [newOrder] = await db.insert(inventoryWithdrawalOrders).values({
      ...order,
      orderNumber,
      totalCost,
      status: 'draft',
    }).returning();
    
    // Create lines
    for (const line of linesToInsert) {
      await db.insert(inventoryWithdrawalLines).values({
        ...line,
        orderId: newOrder.id,
      });
    }
    
    return this.getInventoryWithdrawalOrder(newOrder.id) as Promise<InventoryWithdrawalOrderWithDetails>;
  }

  async updateInventoryWithdrawalOrder(id: number, order: Partial<InsertInventoryWithdrawalOrder>): Promise<InventoryWithdrawalOrderWithDetails | undefined> {
    const existing = await this.getInventoryWithdrawalOrder(id);
    if (!existing) return undefined;
    
    if (existing.status !== 'draft') {
      throw new Error("لا يمكن تعديل أمر سحب مؤكد أو ملغي");
    }
    
    await db.update(inventoryWithdrawalOrders)
      .set({ ...order, updatedAt: new Date() })
      .where(eq(inventoryWithdrawalOrders.id, id));
    
    return this.getInventoryWithdrawalOrder(id);
  }

  async deleteInventoryWithdrawalOrder(id: number): Promise<boolean> {
    const existing = await this.getInventoryWithdrawalOrder(id);
    if (!existing) return false;
    
    if (existing.status !== 'draft') {
      throw new Error("لا يمكن حذف أمر سحب مؤكد أو ملغي");
    }
    
    await db.delete(inventoryWithdrawalLines).where(eq(inventoryWithdrawalLines.orderId, id));
    const result = await db.delete(inventoryWithdrawalOrders).where(eq(inventoryWithdrawalOrders.id, id)).returning();
    return result.length > 0;
  }

  async addInventoryWithdrawalLine(orderId: number, line: Omit<InsertInventoryWithdrawalLine, 'orderId'>): Promise<InventoryWithdrawalLineWithDetails> {
    const order = await this.getInventoryWithdrawalOrder(orderId);
    if (!order) throw new Error("أمر السحب غير موجود");
    if (order.status !== 'draft') throw new Error("لا يمكن إضافة سطور لأمر سحب مؤكد أو ملغي");
    
    const [product] = await db.select().from(products).where(eq(products.id, line.productId));
    if (!product) throw new Error("المنتج غير موجود");
    
    const unitCost = product.averageUnitCost || 0;
    const totalCost = line.quantity * unitCost;
    
    const [newLine] = await db.insert(inventoryWithdrawalLines).values({
      ...line,
      orderId,
      unitCost,
      totalCost,
    }).returning();
    
    // Update order total
    const newOrderTotal = order.totalCost + totalCost;
    await db.update(inventoryWithdrawalOrders)
      .set({ totalCost: newOrderTotal, updatedAt: new Date() })
      .where(eq(inventoryWithdrawalOrders.id, orderId));
    
    return { ...newLine, product };
  }

  async removeInventoryWithdrawalLine(lineId: number): Promise<boolean> {
    const [line] = await db.select().from(inventoryWithdrawalLines)
      .where(eq(inventoryWithdrawalLines.id, lineId));
    if (!line) return false;
    
    const order = await this.getInventoryWithdrawalOrder(line.orderId);
    if (!order) return false;
    if (order.status !== 'draft') throw new Error("لا يمكن حذف سطور من أمر سحب مؤكد أو ملغي");
    
    await db.delete(inventoryWithdrawalLines).where(eq(inventoryWithdrawalLines.id, lineId));
    
    // Update order total
    const newOrderTotal = order.totalCost - line.totalCost;
    await db.update(inventoryWithdrawalOrders)
      .set({ totalCost: newOrderTotal, updatedAt: new Date() })
      .where(eq(inventoryWithdrawalOrders.id, line.orderId));
    
    return true;
  }

  async confirmInventoryWithdrawalOrder(id: number, confirmedByEmployeeId?: number): Promise<InventoryWithdrawalOrderWithDetails> {
    const order = await this.getInventoryWithdrawalOrder(id);
    if (!order) throw new Error("أمر السحب غير موجود");
    if (order.status !== 'draft') throw new Error("أمر السحب مؤكد مسبقاً أو ملغي");
    if (order.lines.length === 0) throw new Error("لا يمكن تأكيد أمر سحب بدون منتجات");
    
    const today = new Date().toISOString().split('T')[0];
    
    // Create journal entry: Debit expense, Credit inventory
    // Get expense account based on reason (5200 = Cost of Goods Sold / Inventory Shrinkage)
    const expenseAccountCode = order.reason === 'damage' ? 5410 : // مصروفات التلف
                               order.reason === 'expiry' ? 5410 : // انتهاء الصلاحية
                               order.reason === 'donation' ? 5420 : // تبرعات
                               5400; // مصروفات عامة (internal_use/other)
    
    const inventoryAccountCode = 1210; // المخزون
    
    // Find accounts by code
    const [expenseAccount] = await db.select().from(accounts)
      .where(eq(accounts.code, expenseAccountCode.toString()));
    const [inventoryAccount] = await db.select().from(accounts)
      .where(eq(accounts.code, inventoryAccountCode.toString()));
    
    if (!expenseAccount || !inventoryAccount) {
      throw new Error("حسابات القيد المحاسبي غير موجودة");
    }
    
    // Get reason label for journal entry
    const reasonLabels: Record<string, string> = {
      internal_use: "استخدام داخلي",
      damage: "تلف",
      expiry: "انتهاء صلاحية",
      donation: "تبرع",
      other: "أخرى",
    };
    const reasonLabel = reasonLabels[order.reason] || order.reason;
    
    // Create journal entry
    const [journalEntry] = await db.insert(journalEntries).values({
      entryDate: order.orderDate,
      description: `سحب مخزون - ${reasonLabel} - ${order.orderNumber}`,
      referenceType: "InventoryWithdrawal",
      referenceId: order.id,
    }).returning();
    
    // Create journal lines
    await db.insert(journalLines).values([
      {
        journalEntryId: journalEntry.id,
        accountId: expenseAccount.id,
        debit: order.totalCost,
        credit: 0,
        description: `سحب مخزون - ${reasonLabel}`,
      },
      {
        journalEntryId: journalEntry.id,
        accountId: inventoryAccount.id,
        debit: 0,
        credit: order.totalCost,
        description: `سحب مخزون - ${reasonLabel}`,
      },
    ]);
    
    // Create stock movements and reduce stock
    for (const line of order.lines) {
      // Create stock movement
      await db.insert(stockMovements).values({
        productId: line.productId,
        movementDate: order.orderDate,
        movementType: order.reason === 'damage' ? 'damage' : 
                       order.reason === 'expiry' ? 'damage' :
                       'consume',
        quantityIn: 0,
        quantityOut: line.quantity,
        unitCost: line.unitCost,
        referenceType: "InventoryWithdrawal",
        referenceId: order.id,
      });
      
      // Decrease stock using FIFO
      await this.decreaseStockFIFO(line.productId, line.quantity);
    }
    
    // Update order status
    await db.update(inventoryWithdrawalOrders)
      .set({
        status: 'confirmed',
        journalEntryId: journalEntry.id,
        confirmedAt: new Date(),
        confirmedByEmployeeId,
        updatedAt: new Date(),
      })
      .where(eq(inventoryWithdrawalOrders.id, id));
    
    return this.getInventoryWithdrawalOrder(id) as Promise<InventoryWithdrawalOrderWithDetails>;
  }

  async cancelInventoryWithdrawalOrder(id: number): Promise<InventoryWithdrawalOrderWithDetails> {
    const order = await this.getInventoryWithdrawalOrder(id);
    if (!order) throw new Error("أمر السحب غير موجود");
    if (order.status === 'cancelled') throw new Error("أمر السحب ملغي مسبقاً");
    if (order.status === 'confirmed') throw new Error("لا يمكن إلغاء أمر سحب مؤكد");
    
    await db.update(inventoryWithdrawalOrders)
      .set({ status: 'cancelled', updatedAt: new Date() })
      .where(eq(inventoryWithdrawalOrders.id, id));
    
    return this.getInventoryWithdrawalOrder(id) as Promise<InventoryWithdrawalOrderWithDetails>;
  }
}

export const storage = new DatabaseStorage();
