/**
 * Mobile API Types for Flutter Application
 * ==========================================
 * هذا الملف يحتوي على جميع الأنواع والواجهات الخاصة بـ Mobile API
 */

// ========================================
// AUTH TYPES (أنواع المصادقة)
// ========================================

export type MobileUserRole = "client" | "employee" | "driver";

export interface MobileLoginRequest {
  phone: string;
  password: string;
  role: MobileUserRole;
  phoneCountryCode?: string;
}

export interface MobileLoginResponse {
  success: boolean;
  token: string;
  user: MobileUser;
  message?: string;
}

export interface MobileUser {
  id: number;
  role: MobileUserRole;
  nameAr: string;
  phone: string;
  phoneCountryCode: string;
  email?: string | null;
  profileImageUrl?: string | null;
  // Client-specific
  loyaltyPointsBalance?: number;
  // Employee-specific
  jobTitleAr?: string | null;
  branchId?: number | null;
  // Driver-specific
  licenseNumber?: string | null;
  vehiclePlateNumber?: string | null;
}

export interface MobileAuthError {
  success: false;
  error: string;
  code: "INVALID_CREDENTIALS" | "USER_NOT_FOUND" | "ACCOUNT_INACTIVE" | "SERVER_ERROR";
}

// ========================================
// CLIENT HOME TYPES (أنواع الصفحة الرئيسية)
// ========================================

export interface MobileServiceCategory {
  id: number;
  nameAr: string;
  code?: string | null;
  imageUrl?: string | null;
}

export interface MobileService {
  id: number;
  categoryId: number;
  nameAr: string;
  descriptionAr?: string | null;
  price: number;
  vatType: "inclusive" | "exclusive";
  vatRate: number;
  finalPrice: number;
  images: string[];
  isActive: boolean;
}

export interface MobilePackage {
  id: number;
  nameAr: string;
  descriptionAr?: string | null;
  price: number;
  discountPercent: number;
  originalPrice: number;
  validFrom: string;
  validTo: string;
  services: {
    id: number;
    nameAr: string;
    quantity: number;
  }[];
}

export interface MobileCoupon {
  id: number;
  code: string;
  nameAr: string;
  descriptionAr?: string | null;
  discountType: "percent" | "fixed";
  discountValue: number;
  startDate: string;
  endDate: string;
}

export interface MobileOffer {
  id: number;
  type: "package" | "coupon";
  nameAr: string;
  descriptionAr?: string | null;
  imageUrl?: string | null;
  discountPercent?: number;
  validUntil: string;
}

export interface MobileClientHomeResponse {
  categories: MobileServiceCategory[];
  featuredServices: MobileService[];
  activePackages: MobilePackage[];
  offers: MobileOffer[];
  loyaltyPoints: number;
  availableCoupons: MobileCoupon[];
  settings: {
    homeServiceDeliveryFee: number;
    currency: string;
    vatRate: number;
  };
}

// ========================================
// ORDER TYPES (أنواع الطلبات)
// ========================================

export type MobileOrderStatus = 
  | "pending" 
  | "confirmed" 
  | "in_progress" 
  | "completed" 
  | "cancelled" 
  | "postponed";

export type MobilePaymentStatus = "pending" | "partial" | "paid" | "refunded";

export interface MobileOrderItem {
  id: number;
  type: "service" | "product";
  nameAr: string;
  quantity: number;
  unitPrice: number;
  lineTotal: number;
  vatAmount: number;
  employeeId?: number | null;
  employeeNameAr?: string | null;
  status?: "pending" | "in_progress" | "completed" | "cancelled";
}

export interface MobileOrder {
  id: number;
  orderNumber: string;
  source: "pos" | "app";
  orderType: "in_salon" | "home_service";
  status: MobileOrderStatus;
  paymentStatus: MobilePaymentStatus;
  scheduledAt?: string | null;
  subtotalAmount: number;
  vatAmount: number;
  discountAmount: number;
  couponCode?: string | null;
  couponDiscountAmount: number;
  deliveryFee: number;
  totalAmount: number;
  paidAmount: number;
  remainingAmount: number;
  addressText?: string | null;
  notes?: string | null;
  items: MobileOrderItem[];
  createdAt: string;
  completedAt?: string | null;
}

export interface MobileOrderListItem {
  id: number;
  orderNumber: string;
  status: MobileOrderStatus;
  totalAmount: number;
  itemsCount: number;
  scheduledAt?: string | null;
  createdAt: string;
}

export interface MobileOrdersListResponse {
  orders: MobileOrderListItem[];
  pagination: {
    page: number;
    limit: number;
    total: number;
    totalPages: number;
  };
}

// ========================================
// CREATE ORDER TYPES (أنواع إنشاء الطلب)
// ========================================

export interface MobileCreateOrderItem {
  type: "service" | "product";
  itemId: number;
  quantity: number;
  preferredEmployeeId?: number;
}

export interface MobileCreateOrderRequest {
  orderType: "in_salon" | "home_service";
  scheduledAt?: string;
  items: MobileCreateOrderItem[];
  packageId?: number;
  couponCode?: string;
  loyaltyPointsToRedeem?: number;
  addressText?: string;
  locationLat?: number;
  locationLng?: number;
  notes?: string;
}

export interface MobileCreateOrderResponse {
  success: boolean;
  order: MobileOrder;
  message: string;
}

// ========================================
// APPOINTMENT TYPES (أنواع الحجوزات)
// ========================================

export type MobileAppointmentStatus = 
  | "pending" 
  | "confirmed" 
  | "in_progress" 
  | "completed" 
  | "cancelled" 
  | "postponed"
  | "no_show";

export interface MobileAppointment {
  id: number;
  appointmentNumber: string;
  appointmentType: "in_salon" | "home_service";
  status: MobileAppointmentStatus;
  scheduledAt: string;
  subtotalAmount: number;
  vatAmount: number;
  totalAmount: number;
  deliveryFee: number;
  couponCode?: string | null;
  couponDiscountAmount: number;
  addressText?: string | null;
  notes?: string | null;
  services: {
    id: number;
    serviceId: number;
    nameAr: string;
    quantity: number;
    lineTotal: number;
    preferredEmployeeId?: number | null;
    preferredEmployeeNameAr?: string | null;
  }[];
  relatedOrderId?: number | null;
  ratingValue?: number | null;
  ratingComment?: string | null;
  createdAt: string;
  completedAt?: string | null;
}

export interface MobileAppointmentsListResponse {
  appointments: MobileAppointment[];
  pagination: {
    page: number;
    limit: number;
    total: number;
    totalPages: number;
  };
}

// ========================================
// EMPLOYEE TYPES (أنواع الموظفين)
// ========================================

export type MobileServiceLineStatus = "pending" | "in_progress" | "completed" | "cancelled";

export interface MobileEmployeeServiceLine {
  id: number;
  orderId: number;
  orderNumber: string;
  serviceId: number;
  serviceNameAr: string;
  quantity: number;
  lineTotal: number;
  status: MobileServiceLineStatus;
  customerNameAr: string;
  customerPhone?: string;
  scheduledAt?: string | null;
  notes?: string | null;
  orderType: "in_salon" | "home_service";
  addressText?: string | null;
}

export interface MobileEmployeeTodayServicesResponse {
  date: string;
  services: MobileEmployeeServiceLine[];
  summary: {
    total: number;
    pending: number;
    inProgress: number;
    completed: number;
    cancelled: number;
    totalEarnings: number;
  };
}

export interface MobileEmployeeUpdateServiceStatusRequest {
  status: MobileServiceLineStatus;
  notes?: string;
}

export interface MobileEmployeeAttendanceSummary {
  month: string;
  year: number;
  totalWorkDays: number;
  presentDays: number;
  absentDays: number;
  lateDays: number;
  leaveDays: number;
  overtimeHours: number;
  remainingLeaveDays: number;
  attendancePercentage: number;
}

// ========================================
// DRIVER TYPES (أنواع السائقين)
// ========================================

export type MobileDeliveryStatus = 
  | "pending" 
  | "assigned" 
  | "picked_up" 
  | "on_the_way" 
  | "delivered" 
  | "cancelled";

export interface MobileDriverOrder {
  id: number;
  orderNumber?: string;
  appointmentNumber?: string;
  type: "order" | "appointment";
  status: MobileDeliveryStatus;
  customerNameAr: string;
  customerPhone?: string;
  addressText: string;
  locationLat?: number | null;
  locationLng?: number | null;
  scheduledAt?: string | null;
  deliveryFee: number;
  notes?: string | null;
  createdAt: string;
}

export interface MobileDriverOrdersResponse {
  orders: MobileDriverOrder[];
  summary: {
    total: number;
    pending: number;
    onTheWay: number;
    delivered: number;
  };
}

export interface MobileDriverUpdateStatusRequest {
  status: MobileDeliveryStatus;
  notes?: string;
  currentLocationLat?: number;
  currentLocationLng?: number;
}

// ========================================
// LOYALTY TYPES (أنواع الولاء)
// ========================================

export interface MobileLoyaltyTransaction {
  id: number;
  type: "earn" | "redeem" | "expire" | "adjust";
  points: number;
  balanceAfter: number;
  description: string;
  orderId?: number | null;
  createdAt: string;
}

export interface MobileLoyaltyResponse {
  balance: number;
  totalEarned: number;
  totalRedeemed: number;
  pointsValue: number; // Value per point in currency
  transactions: MobileLoyaltyTransaction[];
}

// ========================================
// COUPON VALIDATION TYPES (أنواع التحقق من الكوبون)
// ========================================

export interface MobileValidateCouponRequest {
  couponCode: string;
  orderItems: {
    type: "service" | "product";
    itemId: number;
    quantity: number;
    price: number;
  }[];
  orderSubtotal: number;
}

export interface MobileValidateCouponResponse {
  valid: boolean;
  coupon?: MobileCoupon;
  discountAmount?: number;
  message: string;
  errorCode?: "INVALID_CODE" | "EXPIRED" | "NOT_APPLICABLE" | "ALREADY_USED" | "MIN_ORDER_NOT_MET";
}

// ========================================
// API DOCS TYPES (أنواع توثيق الـ API)
// ========================================

export interface MobileApiEndpoint {
  method: "GET" | "POST" | "PUT" | "DELETE";
  path: string;
  description: string;
  descriptionAr: string;
  requiresAuth: boolean;
  allowedRoles: MobileUserRole[];
  requestBody?: string;
  responseExample?: string;
}

export interface MobileApiDocsResponse {
  version: string;
  baseUrl: string;
  endpoints: {
    auth: MobileApiEndpoint[];
    client: MobileApiEndpoint[];
    employee: MobileApiEndpoint[];
    driver: MobileApiEndpoint[];
    loyalty: MobileApiEndpoint[];
  };
}

// ========================================
// COMMON RESPONSE TYPES (أنواع الاستجابة المشتركة)
// ========================================

export interface MobileSuccessResponse<T = any> {
  success: true;
  data: T;
  message?: string;
}

export interface MobileErrorResponse {
  success: false;
  error: string;
  errorCode?: string;
  details?: Record<string, string>;
}

export type MobileApiResponse<T = any> = MobileSuccessResponse<T> | MobileErrorResponse;
