import { sql } from "drizzle-orm";
import { pgTable, text, varchar, integer, real, boolean, date, timestamp, serial } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

// ========================================
// 1) SUPPLIERS MODULE (الموردين)
// ========================================

export const suppliers = pgTable("suppliers", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull(),
  crNumber: text("cr_number"),
  vatNumber: text("vat_number"),
  cityAr: text("city_ar"),
  addressAr: text("address_ar"),
  supplierType: text("supplier_type").default("normal").notNull(),
  imageUrl: text("image_url"),
  representativeName: text("representative_name"),
  representativePhone: text("representative_phone"),
  agreementText: text("agreement_text"),
  agreementStatus: text("agreement_status").default("none"),
  paymentDueDays: integer("payment_due_days"),
  monthlyPaymentDay: integer("monthly_payment_day"),
  paymentTermDays: integer("payment_term_days"),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const supplierBankAccounts = pgTable("supplier_bank_accounts", {
  id: serial("id").primaryKey(),
  supplierId: integer("supplier_id").references(() => suppliers.id).notNull(),
  bankNameAr: text("bank_name_ar"),
  iban: text("iban"),
  accountNumber: text("account_number"),
  isDefault: boolean("is_default").default(false).notNull(),
});

export const insertSupplierSchema = createInsertSchema(suppliers).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertSupplierBankAccountSchema = createInsertSchema(supplierBankAccounts).omit({
  id: true,
});

export type InsertSupplier = z.infer<typeof insertSupplierSchema>;
export type Supplier = typeof suppliers.$inferSelect;
export type InsertSupplierBankAccount = z.infer<typeof insertSupplierBankAccountSchema>;
export type SupplierBankAccount = typeof supplierBankAccounts.$inferSelect;

// Extended supplier type with bank accounts
export type SupplierWithBankAccounts = Supplier & {
  bankAccounts: SupplierBankAccount[];
};

// ========================================
// 2) UNITS & PRODUCTS & INVENTORY (المخزون)
// ========================================

export const units = pgTable("units", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull(),
  code: text("code"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Brands (العلامات التجارية)
export const brands = pgTable("brands", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull().unique(),
  nameEn: text("name_en"),
  descriptionAr: text("description_ar"),
  imageUrl: text("image_url"),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertBrandSchema = createInsertSchema(brands).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  nameAr: z.string().min(1, "اسم العلامة التجارية مطلوب"),
  nameEn: z.string().optional().nullable(),
  descriptionAr: z.string().optional().nullable(),
  imageUrl: z.string().optional().nullable(),
  isActive: z.boolean().optional().default(true),
});

export const updateBrandSchema = insertBrandSchema.partial();

export type InsertBrand = z.infer<typeof insertBrandSchema>;
export type Brand = typeof brands.$inferSelect;

export const products = pgTable("products", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull(),
  sku: text("sku").unique(),
  brandId: integer("brand_id").references(() => brands.id),
  purchaseUnitId: integer("purchase_unit_id").references(() => units.id),
  sellUnitId: integer("sell_unit_id").references(() => units.id),
  conversionFactor: real("conversion_factor").default(1).notNull(),
  defaultPurchasePrice: real("default_purchase_price"),
  defaultSellPrice: real("default_sell_price"),
  hasExpiryDate: boolean("has_expiry_date").default(false).notNull(),
  usageType: text("usage_type").default("sale").notNull(), // "sale", "consume", "sale_and_consume"
  averageUnitCost: real("average_unit_cost"), // Moving weighted average cost
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const stockBatches = pgTable("stock_batches", {
  id: serial("id").primaryKey(),
  productId: integer("product_id").references(() => products.id).notNull(),
  expiryDate: date("expiry_date"),
  quantityOnHand: real("quantity_on_hand").notNull(),
  unitCost: real("unit_cost").notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const stockMovements = pgTable("stock_movements", {
  id: serial("id").primaryKey(),
  productId: integer("product_id").references(() => products.id).notNull(),
  movementDate: date("movement_date").notNull(),
  movementType: text("movement_type").notNull(), // "purchase", "sale", "consume", "damage", "adjustment", "custody_issue", "custody_return"
  quantityIn: real("quantity_in").default(0).notNull(),
  quantityOut: real("quantity_out").default(0).notNull(),
  unitCost: real("unit_cost"),
  referenceType: text("reference_type"), // "PurchaseInvoice", "ManualAdjustment", "EmployeeCustody", etc.
  referenceId: integer("reference_id"),
  batchId: integer("batch_id").references(() => stockBatches.id),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertUnitSchema = createInsertSchema(units).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const usageTypeEnum = z.enum(["sale", "consume", "sale_and_consume"]);

export const insertProductSchema = createInsertSchema(products).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  usageType: usageTypeEnum.default("sale"),
}).refine((data) => {
  if ((data.usageType === "sale" || data.usageType === "sale_and_consume") && !data.defaultSellPrice) {
    return false;
  }
  return true;
}, {
  message: "سعر البيع مطلوب للمنتجات القابلة للبيع",
  path: ["defaultSellPrice"],
});

export const insertStockBatchSchema = createInsertSchema(stockBatches).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertStockMovementSchema = createInsertSchema(stockMovements).omit({
  id: true,
  createdAt: true,
});

export type InsertUnit = z.infer<typeof insertUnitSchema>;
export type Unit = typeof units.$inferSelect;
export type InsertProduct = z.infer<typeof insertProductSchema>;
export type Product = typeof products.$inferSelect;
export type InsertStockBatch = z.infer<typeof insertStockBatchSchema>;
export type StockBatch = typeof stockBatches.$inferSelect;
export type InsertStockMovement = z.infer<typeof insertStockMovementSchema>;
export type StockMovement = typeof stockMovements.$inferSelect;

// Extended product type with units
export type ProductWithUnits = Product & {
  purchaseUnit?: Unit;
  sellUnit?: Unit;
};

// ========================================
// 3) PAYMENT METHODS (طرق الدفع)
// ========================================

export const paymentMethods = pgTable("payment_methods", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull(),
  code: text("code"),
  type: text("type"), // "cash", "bank", "card"
  accountId: integer("account_id").references(() => accounts.id), // Link to GL account for journal entries
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertPaymentMethodSchema = createInsertSchema(paymentMethods).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type InsertPaymentMethod = z.infer<typeof insertPaymentMethodSchema>;
export type PaymentMethod = typeof paymentMethods.$inferSelect;

// ========================================
// 4) PURCHASE INVOICES (فاتورة المشتريات)
// ========================================

export const purchaseInvoices = pgTable("purchase_invoices", {
  id: serial("id").primaryKey(),
  supplierId: integer("supplier_id").references(() => suppliers.id).notNull(),
  supplierRepresentativeName: text("supplier_representative_name"),
  invoiceDate: date("invoice_date").notNull(),
  dueDate: date("due_date"),
  paymentStatus: text("payment_status").default("unpaid").notNull(), // "unpaid", "partially_paid", "paid"
  subtotalAmount: real("subtotal_amount").default(0).notNull(),
  vatAmount: real("vat_amount").default(0).notNull(),
  totalAmount: real("total_amount").default(0).notNull(),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const purchaseInvoiceLines = pgTable("purchase_invoice_lines", {
  id: serial("id").primaryKey(),
  purchaseInvoiceId: integer("purchase_invoice_id").references(() => purchaseInvoices.id).notNull(),
  productId: integer("product_id").references(() => products.id).notNull(),
  purchaseUnitId: integer("purchase_unit_id").references(() => units.id).notNull(),
  sellUnitId: integer("sell_unit_id").references(() => units.id).notNull(),
  quantityPurchaseUnit: real("quantity_purchase_unit").notNull(),
  conversionFactor: real("conversion_factor").notNull(),
  quantitySellUnit: real("quantity_sell_unit").notNull(),
  purchasePricePerPurchaseUnit: real("purchase_price_per_purchase_unit").notNull(),
  purchasePricePerSellUnit: real("purchase_price_per_sell_unit").notNull(),
  vatType: text("vat_type").notNull(), // "inclusive", "exclusive", "exempt"
  vatRate: real("vat_rate").notNull(),
  vatAmount: real("vat_amount").notNull(),
  lineSubtotal: real("line_subtotal").notNull(),
  lineTotal: real("line_total").notNull(),
  expiryDate: date("expiry_date"),
});

export const purchasePayments = pgTable("purchase_payments", {
  id: serial("id").primaryKey(),
  purchaseInvoiceId: integer("purchase_invoice_id").references(() => purchaseInvoices.id),
  supplierId: integer("supplier_id").references(() => suppliers.id).notNull(),
  paymentMethodId: integer("payment_method_id").references(() => paymentMethods.id).notNull(),
  paymentDate: date("payment_date").notNull(),
  amount: real("amount").notNull(),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertPurchaseInvoiceSchema = createInsertSchema(purchaseInvoices).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertPurchaseInvoiceLineSchema = createInsertSchema(purchaseInvoiceLines).omit({
  id: true,
});

export const insertPurchasePaymentSchema = createInsertSchema(purchasePayments).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type InsertPurchaseInvoice = z.infer<typeof insertPurchaseInvoiceSchema>;
export type PurchaseInvoice = typeof purchaseInvoices.$inferSelect;
export type InsertPurchaseInvoiceLine = z.infer<typeof insertPurchaseInvoiceLineSchema>;
export type PurchaseInvoiceLine = typeof purchaseInvoiceLines.$inferSelect;
export type InsertPurchasePayment = z.infer<typeof insertPurchasePaymentSchema>;
export type PurchasePayment = typeof purchasePayments.$inferSelect;

// Extended types
export type PurchaseInvoiceLineWithProduct = PurchaseInvoiceLine & {
  product?: Product;
  purchaseUnit?: Unit;
  sellUnit?: Unit;
};

export type PurchaseInvoiceWithDetails = PurchaseInvoice & {
  supplier?: Supplier;
  lines: PurchaseInvoiceLineWithProduct[];
  payments: (PurchasePayment & { paymentMethod?: PaymentMethod })[];
};

// ========================================
// 4.3) PURCHASE DEBIT NOTES (إشعارات الخصم للمشتريات - مرتجعات للموردين)
// ========================================

export const purchaseDebitNotes = pgTable("purchase_debit_notes", {
  id: serial("id").primaryKey(),
  purchaseInvoiceId: integer("purchase_invoice_id").references(() => purchaseInvoices.id).notNull(),
  supplierId: integer("supplier_id").references(() => suppliers.id).notNull(),
  debitNoteNumber: text("debit_note_number").unique().notNull(), // رقم إشعار الخصم e.g. "DN-0001"
  debitNoteDate: date("debit_note_date").notNull(),
  returnType: text("return_type").notNull(), // "full" | "partial"
  reason: text("reason").notNull(), // سبب المرتجع
  originalInvoiceNumber: text("original_invoice_number"), // رقم الفاتورة الأصلية
  subtotalAmount: real("subtotal_amount").default(0).notNull(),
  vatAmount: real("vat_amount").default(0).notNull(),
  totalAmount: real("total_amount").default(0).notNull(), // إجمالي المبلغ المسترد
  processedByEmployeeId: integer("processed_by_employee_id").references(() => employees.id),
  notes: text("notes"),
  status: text("status").default("draft").notNull(), // "draft" | "confirmed" | "cancelled"
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const purchaseDebitNoteLines = pgTable("purchase_debit_note_lines", {
  id: serial("id").primaryKey(),
  debitNoteId: integer("debit_note_id").references(() => purchaseDebitNotes.id).notNull(),
  purchaseInvoiceLineId: integer("purchase_invoice_line_id").references(() => purchaseInvoiceLines.id).notNull(),
  productId: integer("product_id").references(() => products.id).notNull(),
  quantityReturned: real("quantity_returned").notNull(), // الكمية المرتجعة بوحدة البيع
  unitPrice: real("unit_price").notNull(), // سعر الوحدة
  vatRate: real("vat_rate").notNull(),
  vatAmount: real("vat_amount").notNull(),
  lineSubtotal: real("line_subtotal").notNull(),
  lineTotal: real("line_total").notNull(),
  reason: text("reason"), // سبب خاص بهذا الخط
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertPurchaseDebitNoteSchema = createInsertSchema(purchaseDebitNotes).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertPurchaseDebitNoteLineSchema = createInsertSchema(purchaseDebitNoteLines).omit({
  id: true,
  createdAt: true,
});

export type InsertPurchaseDebitNote = z.infer<typeof insertPurchaseDebitNoteSchema>;
export type PurchaseDebitNote = typeof purchaseDebitNotes.$inferSelect;
export type InsertPurchaseDebitNoteLine = z.infer<typeof insertPurchaseDebitNoteLineSchema>;
export type PurchaseDebitNoteLine = typeof purchaseDebitNoteLines.$inferSelect;

export type PurchaseDebitNoteLineWithProduct = PurchaseDebitNoteLine & {
  product?: Product;
};

export type PurchaseDebitNoteWithDetails = PurchaseDebitNote & {
  supplier?: Supplier;
  purchaseInvoice?: PurchaseInvoice;
  lines: PurchaseDebitNoteLineWithProduct[];
  processedByEmployee?: Employee;
};

// ========================================
// 5) JOURNAL ENTRIES (القيود المحاسبية)
// ========================================

export const journalEntries = pgTable("journal_entries", {
  id: serial("id").primaryKey(),
  entryNumber: text("entry_number").unique().notNull(),
  date: date("date").notNull(),
  description: text("description"),
  referenceType: text("reference_type"), // "PurchaseInvoice", "Payment"
  referenceId: integer("reference_id"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const journalLines = pgTable("journal_lines", {
  id: serial("id").primaryKey(),
  journalEntryId: integer("journal_entry_id").references(() => journalEntries.id).notNull(),
  accountCode: text("account_code").notNull(),
  accountName: text("account_name"),
  debit: real("debit").default(0).notNull(),
  credit: real("credit").default(0).notNull(),
  supplierId: integer("supplier_id").references(() => suppliers.id),
  customerId: integer("customer_id"), // For sales-related journal entries
  description: text("description"), // Per-line description for manual entries
});

export const insertJournalEntrySchema = createInsertSchema(journalEntries).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertJournalLineSchema = createInsertSchema(journalLines).omit({
  id: true,
});

export type InsertJournalEntry = z.infer<typeof insertJournalEntrySchema>;
export type JournalEntry = typeof journalEntries.$inferSelect;
export type InsertJournalLine = z.infer<typeof insertJournalLineSchema>;
export type JournalLine = typeof journalLines.$inferSelect;

export type JournalEntryWithLines = JournalEntry & {
  lines: JournalLine[];
};

// ========================================
// 6) CHART OF ACCOUNTS (دليل الحسابات)
// ========================================

export const accountTypes = ["asset", "liability", "equity", "revenue", "expense"] as const;
export type AccountType = typeof accountTypes[number];

export const accounts = pgTable("accounts", {
  id: serial("id").primaryKey(),
  code: text("code").unique().notNull(),
  nameAr: text("name_ar").notNull(),
  nameEn: text("name_en"),
  accountType: text("account_type").notNull(), // asset, liability, equity, revenue, expense
  parentId: integer("parent_id").references((): any => accounts.id),
  level: integer("level").default(1).notNull(), // hierarchy level (1=root, 2=child, etc.)
  isPostable: boolean("is_postable").default(true).notNull(), // can post journal entries
  isActive: boolean("is_active").default(true).notNull(),
  normalBalance: text("normal_balance").default("debit").notNull(), // "debit" or "credit"
  description: text("description"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertAccountSchema = createInsertSchema(accounts).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type InsertAccount = z.infer<typeof insertAccountSchema>;
export type Account = typeof accounts.$inferSelect;

export type AccountWithChildren = Account & {
  children?: AccountWithChildren[];
};

// ========================================
// ACCOUNT CONFIGURATION (تكوين الحسابات)
// ========================================

// Default account codes for system operations
export const ACCOUNT_CODES = {
  // Assets (1xxx)
  CASH: "1110",                  // النقدية
  BANK: "1120",                  // البنك
  INVENTORY: "1201",             // مخزون منتجات
  VAT_INPUT: "1410",             // ضريبة مدخلات (قابلة للاسترداد)
  ACCOUNTS_RECEIVABLE: "1100",   // العملاء (المدينون)
  // Liabilities (2xxx)
  SUPPLIER_CONTROL: "2100",      // الموردين
  VAT_OUTPUT: "2410",            // ضريبة مخرجات (مستحقة)
  ACCOUNTS_PAYABLE: "2100",      // الدائنون
  // Equity (3xxx)
  CAPITAL: "3100",               // رأس المال
  RETAINED_EARNINGS: "3200",     // الأرباح المحتجزة
  // Revenue (4xxx)
  SALES_REVENUE: "4100",         // إيرادات المبيعات
  SERVICE_REVENUE: "4200",       // إيرادات الخدمات
  // Expenses (5xxx)
  COST_OF_GOODS_SOLD: "5100",    // تكلفة البضاعة المباعة
  PURCHASES: "5200",             // المشتريات
} as const;

export const ACCOUNT_NAMES = {
  [ACCOUNT_CODES.CASH]: "النقدية",
  [ACCOUNT_CODES.BANK]: "البنك",
  [ACCOUNT_CODES.INVENTORY]: "مخزون البضائع",
  [ACCOUNT_CODES.VAT_INPUT]: "ضريبة القيمة المضافة - مدخلات",
  [ACCOUNT_CODES.SUPPLIER_CONTROL]: "الموردين",
  [ACCOUNT_CODES.VAT_OUTPUT]: "ضريبة القيمة المضافة - مخرجات",
  [ACCOUNT_CODES.ACCOUNTS_RECEIVABLE]: "العملاء (المدينون)",
  [ACCOUNT_CODES.CAPITAL]: "رأس المال",
  [ACCOUNT_CODES.RETAINED_EARNINGS]: "الأرباح المحتجزة",
  [ACCOUNT_CODES.SALES_REVENUE]: "إيرادات المبيعات",
  [ACCOUNT_CODES.SERVICE_REVENUE]: "إيرادات الخدمات",
  [ACCOUNT_CODES.COST_OF_GOODS_SOLD]: "تكلفة البضاعة المباعة",
  [ACCOUNT_CODES.PURCHASES]: "المشتريات",
} as const;

// Account type labels in Arabic
export const ACCOUNT_TYPE_LABELS = {
  asset: "أصول",
  liability: "خصوم",
  equity: "حقوق الملكية",
  revenue: "إيرادات",
  expense: "مصروفات",
} as const;

// ========================================
// 7) SERVICE CATEGORIES & SERVICES (الخدمات)
// ========================================

export const vatTypes = ["inclusive", "exclusive", "exempt"] as const;
export type VatType = typeof vatTypes[number];

export const serviceCategories = pgTable("service_categories", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull(),
  code: text("code").unique(),
  imageUrl: text("image_url"),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const services = pgTable("services", {
  id: serial("id").primaryKey(),
  categoryId: integer("category_id").references(() => serviceCategories.id).notNull(),
  nameAr: text("name_ar").notNull(),
  code: text("code").unique(),
  descriptionAr: text("description_ar"),
  price: real("price").notNull(),
  vatType: text("vat_type").notNull(), // "inclusive", "exclusive", "exempt"
  vatRate: real("vat_rate").notNull(), // e.g., 0.15 for 15%
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Service-Products link table for inventory consumption
export const serviceProducts = pgTable("service_products", {
  id: serial("id").primaryKey(),
  serviceId: integer("service_id").references(() => services.id).notNull(),
  productId: integer("product_id").references(() => products.id).notNull(),
  quantityPerService: real("quantity_per_service").default(1).notNull(),
  deductFromCommission: boolean("deduct_from_commission").default(false).notNull(),
  notes: text("notes"),
});

// Service Images table for multiple images per service
export const serviceImages = pgTable("service_images", {
  id: serial("id").primaryKey(),
  serviceId: integer("service_id").references(() => services.id).notNull(),
  imageUrl: text("image_url").notNull(),
  filename: text("filename").notNull(),
  displayOrder: integer("display_order").default(0).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertServiceCategorySchema = createInsertSchema(serviceCategories).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  nameAr: z.string().min(1, "اسم الفئة مطلوب"),
  code: z.string().optional().nullable(),
  imageUrl: z.string().url("رابط الصورة غير صالح").optional().nullable().or(z.literal("")),
  isActive: z.boolean().optional().default(true),
});

export const insertServiceSchema = createInsertSchema(services).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  categoryId: z.number().int().positive("فئة الخدمة مطلوبة"),
  nameAr: z.string().min(1, "اسم الخدمة مطلوب"),
  code: z.string().optional().nullable(),
  descriptionAr: z.string().optional().nullable(),
  price: z.number().positive("السعر يجب أن يكون أكبر من صفر"),
  vatType: z.enum(vatTypes, { errorMap: () => ({ message: "نوع الضريبة غير صالح" }) }),
  vatRate: z.number().min(0, "نسبة الضريبة لا تقل عن 0").max(1, "نسبة الضريبة لا تتجاوز 100%"),
  isActive: z.boolean().optional().default(true),
});

export const insertServiceProductSchema = createInsertSchema(serviceProducts).omit({
  id: true,
}).extend({
  serviceId: z.number().int().positive().optional(),
  productId: z.number().int().positive("المنتج مطلوب"),
  quantityPerService: z.number().positive("الكمية يجب أن تكون أكبر من صفر"),
  deductFromCommission: z.boolean().optional().default(false),
  notes: z.string().optional().nullable(),
});

// Partial schemas for updates
export const updateServiceCategorySchema = insertServiceCategorySchema.partial();
export const updateServiceSchema = insertServiceSchema.partial().extend({
  categoryId: z.number().int().positive("فئة الخدمة مطلوبة").optional(),
});

// Related products array validation schema
export const relatedProductsArraySchema = z.array(
  z.object({
    productId: z.number().int().positive("المنتج مطلوب"),
    quantityPerService: z.number().positive("الكمية يجب أن تكون أكبر من صفر").optional().default(1),
    deductFromCommission: z.boolean().optional().default(false),
    notes: z.string().optional().nullable(),
  })
).optional();

export const insertServiceImageSchema = createInsertSchema(serviceImages).omit({
  id: true,
  createdAt: true,
}).extend({
  serviceId: z.number().int().positive("معرف الخدمة مطلوب"),
  imageUrl: z.string().min(1, "رابط الصورة مطلوب"),
  filename: z.string().min(1, "اسم الملف مطلوب"),
  displayOrder: z.number().int().optional().default(0),
});

export type InsertServiceCategory = z.infer<typeof insertServiceCategorySchema>;
export type ServiceCategory = typeof serviceCategories.$inferSelect;
export type InsertService = z.infer<typeof insertServiceSchema>;
export type Service = typeof services.$inferSelect;
export type InsertServiceProduct = z.infer<typeof insertServiceProductSchema>;
export type ServiceProduct = typeof serviceProducts.$inferSelect;
export type InsertServiceImage = z.infer<typeof insertServiceImageSchema>;
export type ServiceImage = typeof serviceImages.$inferSelect;

// Extended types with relations
export type ServiceCategoryWithCount = ServiceCategory & {
  servicesCount: number;
};

export type ServiceCategoryWithServices = ServiceCategory & {
  services: Service[];
};

export type ServiceProductWithProduct = ServiceProduct & {
  product?: Product;
};

export type ServiceWithDetails = Service & {
  category?: ServiceCategory;
  relatedProducts?: ServiceProductWithProduct[];
  images?: ServiceImage[];
};

// ========================================
// 8) CUSTOMERS (العملاء)
// ========================================

export const customers = pgTable("customers", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull(),
  phone: text("phone"),
  phoneCountryCode: text("phone_country_code").default("966"),
  email: text("email"),
  passwordHash: text("password_hash"), // bcrypt hashed password for app login
  vatNumber: text("vat_number"),
  cityAr: text("city_ar"),
  branchId: integer("branch_id"), // future: foreign key to branches table
  address: text("address"),
  profileImageUrl: text("profile_image_url"),
  loyaltyPointsBalance: integer("loyalty_points_balance").default(0).notNull(),
  isActive: boolean("is_active").default(true).notNull(),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertCustomerSchema = createInsertSchema(customers).omit({
  id: true,
  passwordHash: true, // password hashed in server layer
  loyaltyPointsBalance: true,
  createdAt: true,
  updatedAt: true,
});

export const updateCustomerSchema = createInsertSchema(customers).omit({
  id: true,
  passwordHash: true, // password updated separately with hashing
  loyaltyPointsBalance: true,
  createdAt: true,
  updatedAt: true,
}).partial();

export type InsertCustomer = z.infer<typeof insertCustomerSchema>;
export type UpdateCustomer = z.infer<typeof updateCustomerSchema>;
export type Customer = typeof customers.$inferSelect;

// ========================================
// 8.0.1) CUSTOMER REVIEWS (تقييمات العملاء)
// ========================================

// Review Reasons (أسباب التقييم - predefined)
export const reviewReasons = pgTable("review_reasons", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull(),
  reasonType: text("reason_type").notNull(), // "positive", "negative", "neutral"
  sortOrder: integer("sort_order").default(0).notNull(),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertReviewReasonSchema = createInsertSchema(reviewReasons).omit({
  id: true,
  createdAt: true,
});

export type InsertReviewReason = z.infer<typeof insertReviewReasonSchema>;
export type ReviewReason = typeof reviewReasons.$inferSelect;

// Customer Reviews (تقييمات العملاء)
export const customerReviews = pgTable("customer_reviews", {
  id: serial("id").primaryKey(),
  customerId: integer("customer_id").references(() => customers.id).notNull(),
  orderId: integer("order_id").references(() => orders.id),
  reviewReasonId: integer("review_reason_id").references(() => reviewReasons.id).notNull(),
  rating: integer("rating"), // 1-5 stars (optional)
  additionalNotes: text("additional_notes"), // for "أخرى" or additional comments
  status: text("status").default("pending").notNull(), // "pending", "reviewed", "resolved"
  reviewedByEmployeeId: integer("reviewed_by_employee_id").references(() => employees.id),
  reviewedAt: timestamp("reviewed_at"),
  responseNotes: text("response_notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertCustomerReviewSchema = createInsertSchema(customerReviews).omit({
  id: true,
  status: true,
  reviewedByEmployeeId: true,
  reviewedAt: true,
  responseNotes: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  customerId: z.number().int().positive("العميل مطلوب"),
  reviewReasonId: z.number().int().positive("سبب التقييم مطلوب"),
  orderId: z.number().int().positive().optional().nullable(),
  rating: z.number().int().min(1).max(5).optional().nullable(),
  additionalNotes: z.string().optional().nullable(),
});

export type InsertCustomerReview = z.infer<typeof insertCustomerReviewSchema>;
export type CustomerReview = typeof customerReviews.$inferSelect;

export type CustomerReviewWithDetails = CustomerReview & {
  customer?: Customer;
  reviewReason?: ReviewReason;
  reviewedByEmployee?: Employee;
};

// ========================================
// 8.1) LOYALTY POINTS (نقاط الولاء)
// ========================================

// Loyalty settings (singleton configuration)
export const loyaltySettings = pgTable("loyalty_settings", {
  id: serial("id").primaryKey(),
  pointsPerCurrency: real("points_per_currency").default(1).notNull(), // e.g., 1 point per 1 SAR
  currencyPerPoint: real("currency_per_point").default(0.1).notNull(), // e.g., 0.1 SAR per point when redeeming
  expiryMonths: integer("expiry_months").default(12).notNull(), // points expire after X months
  minRedeemPoints: integer("min_redeem_points").default(100).notNull(), // minimum points to redeem
  isActive: boolean("is_active").default(true).notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertLoyaltySettingsSchema = createInsertSchema(loyaltySettings).omit({
  id: true,
  updatedAt: true,
});

export type InsertLoyaltySettings = z.infer<typeof insertLoyaltySettingsSchema>;
export type LoyaltySettings = typeof loyaltySettings.$inferSelect;

// Loyalty points transactions
export const customerLoyaltyPoints = pgTable("customer_loyalty_points", {
  id: serial("id").primaryKey(),
  customerId: integer("customer_id").references(() => customers.id).notNull(),
  transactionType: text("transaction_type").notNull(), // earn, redeem, expire, adjust
  points: integer("points").notNull(), // positive for earn, negative for redeem/expire
  balanceAfter: integer("balance_after").notNull(),
  referenceType: text("reference_type"), // order, manual, promo, return
  referenceId: integer("reference_id"), // orderId if from order
  notes: text("notes"),
  expiresAt: timestamp("expires_at"), // when these earned points expire
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertCustomerLoyaltyPointsSchema = createInsertSchema(customerLoyaltyPoints).omit({
  id: true,
  createdAt: true,
});

export type InsertCustomerLoyaltyPoints = z.infer<typeof insertCustomerLoyaltyPointsSchema>;
export type CustomerLoyaltyPoints = typeof customerLoyaltyPoints.$inferSelect;

// ========================================
// 8.2) SYSTEM SETTINGS (إعدادات النظام)
// ========================================

export const systemSettings = pgTable("system_settings", {
  id: serial("id").primaryKey(),
  key: text("key").unique().notNull(), // e.g., "home_service_delivery_fee"
  value: text("value").notNull(), // stored as string, parsed as needed
  description: text("description"), // وصف الإعداد
  valueType: text("value_type").default("string").notNull(), // "string", "number", "boolean", "json"
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertSystemSettingSchema = createInsertSchema(systemSettings).omit({
  id: true,
  updatedAt: true,
});

export type InsertSystemSetting = z.infer<typeof insertSystemSettingSchema>;
export type SystemSetting = typeof systemSettings.$inferSelect;

// ========================================
// API REQUEST/RESPONSE TYPES
// ========================================

// Create purchase invoice request
export type CreatePurchaseInvoiceRequest = {
  supplier: {
    id: number | null;
    nameAr?: string;
    crNumber?: string;
    vatNumber?: string;
    cityAr?: string;
    supplierType?: string;
    imageUrl?: string;
    representativeName?: string;
    representativePhone?: string;
  };
  invoiceDate: string;
  dueDate?: string;
  lines: {
    product: {
      id: number | null;
      nameAr?: string;
      purchaseUnitNameAr?: string;
      sellUnitNameAr?: string;
      conversionFactor?: number;
      hasExpiryDate?: boolean;
      defaultSellPrice?: number;
    };
    quantityPurchaseUnit: number;
    purchasePricePerPurchaseUnit: number;
    vatType: "inclusive" | "exclusive" | "exempt";
    vatRate: number;
    expiryDate?: string;
  }[];
  notes?: string;
};

// Payment request
export type CreatePaymentRequest = {
  paymentDate: string;
  paymentMethodId: number;
  amount: number;
  notes?: string;
};

// Inventory summary report item
export type InventorySummaryItem = {
  productId: number;
  productNameAr: string;
  sku?: string;
  quantityOnHand: number;
  totalCost: number;
  sellUnitNameAr?: string;
  usageType: string; // "sale", "consume", "sale_and_consume"
  brandNameAr?: string;
};

// Stock movement report item
export type StockMovementReportItem = {
  id: number;
  movementDate: string;
  movementType: string;
  quantityIn: number;
  quantityOut: number;
  unitCost?: number;
  referenceType?: string;
  referenceId?: number;
  balance?: number;
};

// Stock movement summary item (for date range report)
export type StockMovementSummaryItem = {
  productId: number;
  productNameAr: string;
  sku?: string;
  openingQuantity: number;
  quantityInPeriod: number;
  quantityOutPeriod: number;
  closingQuantity: number;
  sellUnitNameAr?: string;
};

// Near expiry product item
export type NearExpiryProductItem = {
  productId: number;
  productNameAr: string;
  batchId: number;
  expiryDate: string;
  quantityOnHand: number;
  unitCost: number;
  totalCost: number;
  daysToExpiry: number;
};

// Commission record for employee commission report
export type CommissionRecord = {
  id: number;
  date: string;
  type: "service" | "deduction";
  description: string;
  valueBeforeTax: number;
  commissionPercent: number;
  commissionAmount: number;
  orderId?: number;
};

// Supplier balance
export type SupplierBalance = {
  supplierId: number;
  supplierNameAr: string;
  totalCredits: number;
  totalDebits: number;
  balance: number; // positive = we owe supplier, negative = supplier owes us
};

// Dashboard stats
export type DashboardStats = {
  totalSuppliers: number;
  activeSuppliers: number;
  totalProducts: number;
  activeProducts: number;
  totalInventoryValue: number;
  totalUnpaidInvoices: number;
  totalUnpaidAmount: number;
  recentInvoices: PurchaseInvoiceWithDetails[];
  lowStockProducts: (Product & { quantityOnHand: number })[];
  expiringProducts: (StockBatch & { product?: Product })[];
};

// Comprehensive Salon Dashboard Summary
export type SalonDashboardFilters = {
  fromDate?: string;
  toDate?: string;
  branchId?: number;
  source?: "all" | "pos" | "app";
};

export type SalonDashboardSummary = {
  salesToday: {
    totalAmount: number;
    ordersCount: number;
    avgOrderValue: number;
  };
  bookingsToday: {
    total: number;
    cancelled: number;
    upcoming: number;
  };
  cancelledOrders: {
    count: number;
    lostAmount: number;
  };
  customers: {
    totalCustomers: number;
    newLast30Days: number;
  };
  topServices: {
    serviceId: number;
    nameAr: string;
    count: number;
    totalAmount: number;
  }[];
  topEmployees: {
    employeeId: number;
    name: string;
    servicesCount: number;
    totalSales: number;
  }[];
  inventory: {
    lowStockCount: number;
    nearExpiryCount: number;
    estimatedValue: number;
  };
  salesOverTime: {
    date: string;
    amount: number;
  }[];
  ordersStatusDistribution: {
    completed: number;
    inProgress: number;
    cancelled: number;
    scheduled: number;
  };
  latestOrders: {
    id: number;
    orderNumber: string;
    date: string;
    customer: string;
    status: string;
    total: number;
  }[];
  salesRevenue: {
    totalSales: number;
    vatCollected: number;
    totalDiscounts: number;
  };
  topCustomers: {
    customerId: number;
    name: string;
    totalSpent: number;
    ordersCount: number;
  }[];
};

// ========================================
// 9) ROLES & EMPLOYEES & HR (الموظفين والموارد البشرية)
// ========================================

// 9.1 SYSTEM MODULES (موديولات النظام)
export const systemModules = [
  { code: "dashboard", nameAr: "لوحة التحكم", icon: "LayoutDashboard" },
  { code: "orders", nameAr: "الطلبات", icon: "ShoppingCart" },
  { code: "appointments", nameAr: "المواعيد", icon: "Calendar" },
  { code: "customers", nameAr: "العملاء", icon: "Users" },
  { code: "services", nameAr: "الخدمات", icon: "Scissors" },
  { code: "products", nameAr: "المنتجات", icon: "Package" },
  { code: "inventory", nameAr: "المخزون", icon: "Warehouse" },
  { code: "suppliers", nameAr: "الموردين", icon: "Truck" },
  { code: "purchases", nameAr: "المشتريات", icon: "Receipt" },
  { code: "employees", nameAr: "الموظفين", icon: "UserCog" },
  { code: "payroll", nameAr: "الرواتب", icon: "Banknote" },
  { code: "attendance", nameAr: "الحضور والانصراف", icon: "Clock" },
  { code: "finance", nameAr: "المالية", icon: "DollarSign" },
  { code: "accounting", nameAr: "المحاسبة", icon: "Calculator" },
  { code: "reports", nameAr: "التقارير", icon: "BarChart" },
  { code: "marketing", nameAr: "التسويق", icon: "Megaphone" },
  { code: "drivers", nameAr: "السائقين", icon: "Car" },
  { code: "cafeteria", nameAr: "الكافتيريا", icon: "Coffee" },
  { code: "settings", nameAr: "الإعدادات", icon: "Settings" },
  { code: "roles", nameAr: "الأدوار والصلاحيات", icon: "Shield" },
  { code: "brands", nameAr: "العلامات التجارية", icon: "Award" },
] as const;

export type SystemModuleCode = typeof systemModules[number]["code"];

// 9.2 PERMISSION ACTIONS (أنواع الصلاحيات)
export const permissionActions = [
  { code: "view", nameAr: "عرض" },
  { code: "create", nameAr: "إضافة" },
  { code: "edit", nameAr: "تعديل" },
  { code: "delete", nameAr: "حذف" },
  { code: "export", nameAr: "تصدير" },
  { code: "print", nameAr: "طباعة" },
] as const;

export type PermissionActionCode = typeof permissionActions[number]["code"];

// 9.3 ROLES (الأدوار)
export const roles = pgTable("roles", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull().unique(),
  code: text("code").unique(),
  descriptionAr: text("description_ar"),
  isSuperAdmin: boolean("is_super_admin").default(false).notNull(),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// 9.4 ROLE PERMISSIONS (صلاحيات الأدوار)
export const rolePermissions = pgTable("role_permissions", {
  id: serial("id").primaryKey(),
  roleId: integer("role_id").references(() => roles.id, { onDelete: "cascade" }).notNull(),
  moduleCode: text("module_code").notNull(),
  canView: boolean("can_view").default(false).notNull(),
  canCreate: boolean("can_create").default(false).notNull(),
  canEdit: boolean("can_edit").default(false).notNull(),
  canDelete: boolean("can_delete").default(false).notNull(),
  canExport: boolean("can_export").default(false).notNull(),
  canPrint: boolean("can_print").default(false).notNull(),
});

export const insertRoleSchema = createInsertSchema(roles).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  nameAr: z.string().min(1, "اسم الدور مطلوب"),
  code: z.string().optional().nullable(),
  descriptionAr: z.string().optional().nullable(),
  isSuperAdmin: z.boolean().optional().default(false),
  isActive: z.boolean().optional().default(true),
});

export const updateRoleSchema = insertRoleSchema.partial();

export const insertRolePermissionSchema = createInsertSchema(rolePermissions).omit({
  id: true,
});

export type InsertRole = z.infer<typeof insertRoleSchema>;
export type Role = typeof roles.$inferSelect;
export type InsertRolePermission = z.infer<typeof insertRolePermissionSchema>;
export type RolePermission = typeof rolePermissions.$inferSelect;

// Extended role type with permissions
export type RoleWithPermissions = Role & {
  permissions: RolePermission[];
  employeeCount?: number;
};

// 9.2 EMPLOYEES (الموظفون) - Extended with full HR data
export const weeklyOffDays = ["friday", "saturday", "sunday", "friday_saturday", "none"] as const;
export type WeeklyOffDay = typeof weeklyOffDays[number];

export const employees = pgTable("employees", {
  id: serial("id").primaryKey(),
  // Legacy columns (kept for backward compatibility)
  nameAr: text("name_ar"),
  code: text("code").unique(),
  phone: text("phone"),
  role: text("role"),
  // New HR columns
  fullNameAr: text("full_name_ar"),
  phoneCountryCode: text("phone_country_code").default("966"),
  phoneNumber: text("phone_number"),
  email: text("email").unique(),
  fingerprintEmployeeNo: text("fingerprint_employee_no").unique(),
  profileImageUrl: text("profile_image_url"),
  jobTitleAr: text("job_title_ar"),
  roleId: integer("role_id").references(() => roles.id),
  cityNameAr: text("city_name_ar"),
  branchId: integer("branch_id"),
  nationalityNameAr: text("nationality_name_ar"),
  jobPositionId: integer("job_position_id"), // FK to job_positions - added later to avoid circular reference
  nationalityId: integer("nationality_id"), // FK to nationalities - added later to avoid circular reference
  // HR data
  birthDate: date("birth_date"),
  hireDate: date("hire_date"),
  iqamaNumber: text("iqama_number"),
  iqamaExpiryDate: date("iqama_expiry_date"),
  insuranceCompany: text("insurance_company"),
  insuranceNumber: text("insurance_number"),
  insuranceExpiryDate: date("insurance_expiry_date"),
  insuranceCardExpiryDate: date("insurance_card_expiry_date"),
  healthNumber: text("health_number"),
  passportNumber: text("passport_number"),
  passportExpiryDate: date("passport_expiry_date"),
  annualLeaveDays: integer("annual_leave_days").default(0),
  weeklyOffDay: text("weekly_off_day"),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertEmployeeSchema = createInsertSchema(employees).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  fullNameAr: z.string().min(1, "اسم الموظف مطلوب"),
  phoneCountryCode: z.string().optional().default("966"),
  phoneNumber: z.string().regex(/^0[0-9]{9}$/, "رقم الجوال غير صحيح - يجب أن يكون 10 أرقام ويبدأ بـ 0"),
  email: z.string().min(1, "البريد الإلكتروني مطلوب").email("البريد الإلكتروني غير صالح"),
  fingerprintEmployeeNo: z.string().optional().nullable(),
  profileImageUrl: z.string().url().optional().nullable().or(z.literal("")),
  jobTitleAr: z.string().min(1, "الوظيفة مطلوبة"),
  roleId: z.number().int().positive("الدور الوظيفي مطلوب"),
  cityNameAr: z.string().optional().nullable(),
  branchId: z.number().int().positive().optional().nullable(),
  nationalityNameAr: z.string().optional().nullable(), // Legacy, now using nationalityId
  jobPositionId: z.number().int().positive().optional().nullable(),
  nationalityId: z.number().int().positive().optional().nullable(),
  birthDate: z.string().optional().nullable(),
  hireDate: z.string().optional().nullable(),
  iqamaNumber: z.string().optional().nullable(),
  iqamaExpiryDate: z.string().optional().nullable(),
  insuranceCompany: z.string().optional().nullable(),
  insuranceNumber: z.string().optional().nullable(),
  insuranceExpiryDate: z.string().optional().nullable(),
  insuranceCardExpiryDate: z.string().optional().nullable(),
  healthNumber: z.string().optional().nullable(),
  passportNumber: z.string().optional().nullable(),
  passportExpiryDate: z.string().optional().nullable(),
  annualLeaveDays: z.number().int().min(0).optional().default(0),
  weeklyOffDay: z.enum(weeklyOffDays).optional().nullable(),
  isActive: z.boolean().optional().default(true),
});

export const updateEmployeeSchema = insertEmployeeSchema.partial();

export type InsertEmployee = z.infer<typeof insertEmployeeSchema>;
export type Employee = typeof employees.$inferSelect;

// Employee with role and other relations
export type EmployeeWithRole = Employee & {
  role?: Role;
  jobPosition?: JobPosition;
  nationality?: Nationality;
};

// 9.3 EMPLOYEE TASKS (مهام الموظفين)
export const taskStatuses = ["new", "in_progress", "completed", "canceled"] as const;
export type TaskStatus = typeof taskStatuses[number];

export const taskPriorities = ["low", "medium", "high"] as const;
export type TaskPriority = typeof taskPriorities[number];

export const employeeTasks = pgTable("employee_tasks", {
  id: serial("id").primaryKey(),
  title: text("title").notNull(),
  description: text("description"),
  assignedToEmployeeId: integer("assigned_to_employee_id").references(() => employees.id).notNull(),
  assignedByEmployeeId: integer("assigned_by_employee_id").references(() => employees.id),
  dueDate: date("due_date"),
  status: text("status").notNull(), // "new", "in_progress", "completed", "canceled"
  priority: text("priority"), // "low", "medium", "high"
  completedAt: timestamp("completed_at"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertEmployeeTaskSchema = createInsertSchema(employeeTasks).omit({
  id: true,
  completedAt: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  title: z.string().min(1, "عنوان المهمة مطلوب"),
  description: z.string().optional().nullable(),
  assignedToEmployeeId: z.number().int().positive("الموظف المكلف مطلوب"),
  assignedByEmployeeId: z.number().int().positive().optional().nullable(),
  dueDate: z.string().optional().nullable(),
  status: z.enum(taskStatuses, { errorMap: () => ({ message: "حالة المهمة غير صالحة" }) }),
  priority: z.enum(taskPriorities).optional().nullable(),
});

export const updateEmployeeTaskSchema = insertEmployeeTaskSchema.partial();

export type InsertEmployeeTask = z.infer<typeof insertEmployeeTaskSchema>;
export type EmployeeTask = typeof employeeTasks.$inferSelect;

export type EmployeeTaskWithDetails = EmployeeTask & {
  assignedToEmployee?: Employee;
  assignedByEmployee?: Employee;
};

// 9.5 EMPLOYEE CUSTODY (عهدة الموظف)
export const custodyStatuses = ["open", "returned", "consumed", "partially_returned", "lost", "damaged"] as const;
export type CustodyStatus = typeof custodyStatuses[number];

export const employeeCustodies = pgTable("employee_custodies", {
  id: serial("id").primaryKey(),
  employeeId: integer("employee_id").references(() => employees.id).notNull(),
  productId: integer("product_id").references(() => products.id).notNull(),
  quantity: real("quantity").notNull(),
  unitCost: real("unit_cost").notNull(), // Unit cost at time of custody (from moving average)
  totalCost: real("total_cost").notNull(), // = quantity * unitCost
  custodyDate: date("custody_date").notNull(),
  createdByEmployeeId: integer("created_by_employee_id").references(() => employees.id),
  status: text("status").default("open").notNull(), // "open", "returned", "consumed", "partially_returned", "lost", "damaged"
  notes: text("notes"),
  // Closure tracking fields (تتبع إغلاق العهدة)
  closedDate: date("closed_date"), // تاريخ إغلاق العهدة
  closedByEmployeeId: integer("closed_by_employee_id").references(() => employees.id), // من أغلق العهدة
  closureReason: text("closure_reason"), // سبب الإغلاق (تالفة/مفقودة/مستهلكة/مرتجعة)
  returnedQuantity: real("returned_quantity"), // الكمية المرتجعة (في حالة الإرجاع الجزئي)
  closureNotes: text("closure_notes"), // ملاحظات الإغلاق
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertEmployeeCustodySchema = createInsertSchema(employeeCustodies).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  employeeId: z.number().int().positive("الموظف مطلوب"),
  productId: z.number().int().positive("المنتج مطلوب"),
  quantity: z.number().positive("الكمية يجب أن تكون أكبر من صفر"),
  unitCost: z.number().min(0, "تكلفة الوحدة غير صالحة"),
  totalCost: z.number().min(0, "التكلفة الإجمالية غير صالحة"),
  custodyDate: z.string().min(1, "تاريخ العهدة مطلوب"),
  createdByEmployeeId: z.number().int().positive().optional().nullable(),
  status: z.enum(custodyStatuses).default("open"),
  notes: z.string().optional().nullable(),
  closedDate: z.string().optional().nullable(),
  closedByEmployeeId: z.number().int().positive().optional().nullable(),
  closureReason: z.string().optional().nullable(),
  returnedQuantity: z.number().min(0).optional().nullable(),
  closureNotes: z.string().optional().nullable(),
});

export const updateEmployeeCustodySchema = insertEmployeeCustodySchema.partial();

export type InsertEmployeeCustody = z.infer<typeof insertEmployeeCustodySchema>;
export type EmployeeCustody = typeof employeeCustodies.$inferSelect;

export type EmployeeCustodyWithDetails = EmployeeCustody & {
  employee?: Employee;
  product?: Product;
  createdByEmployee?: Employee;
  closedByEmployee?: Employee;
};

// 9.4 PAYROLL (الرواتب)

// Salary profiles
export const salaryProfiles = pgTable("salary_profiles", {
  id: serial("id").primaryKey(),
  employeeId: integer("employee_id").references(() => employees.id).notNull().unique(),
  basicSalary: real("basic_salary").default(0).notNull(),
  housingAllowance: real("housing_allowance").default(0).notNull(),
  transportAllowance: real("transport_allowance").default(0).notNull(),
  otherFixedAllowances: real("other_fixed_allowances").default(0).notNull(),
  targetAmount: real("target_amount").default(0).notNull(),
  commissionPercentServices: real("commission_percent_services").default(0).notNull(),
  commissionPercentProducts: real("commission_percent_products").default(0).notNull(),
  commissionPercentCafeteria: real("commission_percent_cafeteria").default(0).notNull(),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertSalaryProfileSchema = createInsertSchema(salaryProfiles).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  employeeId: z.number().int().positive("الموظف مطلوب"),
  basicSalary: z.number().min(0, "الراتب الأساسي لا يقل عن صفر").optional().default(0),
  housingAllowance: z.number().min(0).optional().default(0),
  transportAllowance: z.number().min(0).optional().default(0),
  otherFixedAllowances: z.number().min(0).optional().default(0),
  targetAmount: z.number().min(0).optional().default(0),
  commissionPercentServices: z.number().min(0).max(100).optional().default(0),
  commissionPercentProducts: z.number().min(0).max(100).optional().default(0),
  commissionPercentCafeteria: z.number().min(0).max(100).optional().default(0),
  isActive: z.boolean().optional().default(true),
});

export const updateSalaryProfileSchema = insertSalaryProfileSchema.partial().omit({ employeeId: true });

export type InsertSalaryProfile = z.infer<typeof insertSalaryProfileSchema>;
export type SalaryProfile = typeof salaryProfiles.$inferSelect;

export type SalaryProfileWithEmployee = SalaryProfile & {
  employee?: Employee;
};

// Payroll runs
export const payrollRunStatuses = ["draft", "approved", "paid", "canceled"] as const;
export type PayrollRunStatus = typeof payrollRunStatuses[number];

export const payrollRuns = pgTable("payroll_runs", {
  id: serial("id").primaryKey(),
  periodName: text("period_name").notNull(),
  periodStartDate: date("period_start_date").notNull(),
  periodEndDate: date("period_end_date").notNull(),
  status: text("status").notNull(), // "draft", "approved", "paid", "canceled"
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertPayrollRunSchema = createInsertSchema(payrollRuns).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  periodName: z.string().min(1, "اسم الفترة مطلوب"),
  periodStartDate: z.string().min(1, "تاريخ بداية الفترة مطلوب"),
  periodEndDate: z.string().min(1, "تاريخ نهاية الفترة مطلوب"),
  status: z.enum(payrollRunStatuses).optional().default("draft"),
  notes: z.string().optional().nullable(),
});

export const updatePayrollRunSchema = z.object({
  status: z.enum(payrollRunStatuses).optional(),
  notes: z.string().optional().nullable(),
});

export type InsertPayrollRun = z.infer<typeof insertPayrollRunSchema>;
export type PayrollRun = typeof payrollRuns.$inferSelect;

// Payroll lines
export const payrollLines = pgTable("payroll_lines", {
  id: serial("id").primaryKey(),
  payrollRunId: integer("payroll_run_id").references(() => payrollRuns.id).notNull(),
  employeeId: integer("employee_id").references(() => employees.id).notNull(),
  basicSalary: real("basic_salary").default(0).notNull(),
  fixedAllowances: real("fixed_allowances").default(0).notNull(),
  overtimeAmount: real("overtime_amount").default(0).notNull(),
  commissionAmount: real("commission_amount").default(0).notNull(),
  otherEarnings: real("other_earnings").default(0).notNull(),
  deductionLate: real("deduction_late").default(0).notNull(),
  deductionOther: real("deduction_other").default(0).notNull(),
  grossSalary: real("gross_salary").default(0).notNull(),
  totalDeductions: real("total_deductions").default(0).notNull(),
  netSalary: real("net_salary").default(0).notNull(),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertPayrollLineSchema = createInsertSchema(payrollLines).omit({
  id: true,
  grossSalary: true,
  totalDeductions: true,
  netSalary: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  payrollRunId: z.number().int().positive(),
  employeeId: z.number().int().positive(),
  basicSalary: z.number().min(0).optional().default(0),
  fixedAllowances: z.number().min(0).optional().default(0),
  overtimeAmount: z.number().min(0).optional().default(0),
  commissionAmount: z.number().min(0).optional().default(0),
  otherEarnings: z.number().min(0).optional().default(0),
  deductionLate: z.number().min(0).optional().default(0),
  deductionOther: z.number().min(0).optional().default(0),
  notes: z.string().optional().nullable(),
});

export const updatePayrollLineSchema = z.object({
  basicSalary: z.number().min(0).optional(),
  fixedAllowances: z.number().min(0).optional(),
  overtimeAmount: z.number().min(0).optional(),
  commissionAmount: z.number().min(0).optional(),
  otherEarnings: z.number().min(0).optional(),
  deductionLate: z.number().min(0).optional(),
  deductionOther: z.number().min(0).optional(),
  notes: z.string().optional().nullable(),
});

export type InsertPayrollLine = z.infer<typeof insertPayrollLineSchema>;
export type PayrollLine = typeof payrollLines.$inferSelect;

export type PayrollLineWithEmployee = PayrollLine & {
  employee?: Employee;
};

export type PayrollRunWithLines = PayrollRun & {
  lines: PayrollLineWithEmployee[];
  totalNetSalary: number;
  employeeCount: number;
};

// 9.5 JOB POSITIONS (الوظائف)
export const jobPositions = pgTable("job_positions", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull().unique(),
  code: text("code").unique(),
  descriptionAr: text("description_ar"),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertJobPositionSchema = createInsertSchema(jobPositions).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  nameAr: z.string().min(1, "اسم الوظيفة مطلوب"),
  code: z.string().optional().nullable(),
  descriptionAr: z.string().optional().nullable(),
  isActive: z.boolean().optional().default(true),
});

export const updateJobPositionSchema = insertJobPositionSchema.partial();

export type InsertJobPosition = z.infer<typeof insertJobPositionSchema>;
export type JobPosition = typeof jobPositions.$inferSelect;

// 9.6 NATIONALITIES (الجنسيات)
export const nationalities = pgTable("nationalities", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull().unique(),
  code: text("code").unique(),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertNationalitySchema = createInsertSchema(nationalities).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  nameAr: z.string().min(1, "اسم الجنسية مطلوب"),
  code: z.string().optional().nullable(),
  isActive: z.boolean().optional().default(true),
});

export const updateNationalitySchema = insertNationalitySchema.partial();

export type InsertNationality = z.infer<typeof insertNationalitySchema>;
export type Nationality = typeof nationalities.$inferSelect;

// 9.7 EMPLOYEE SUPERVISORS (المشرفين)
export const employeeSupervisors = pgTable("employee_supervisors", {
  id: serial("id").primaryKey(),
  supervisorEmployeeId: integer("supervisor_employee_id").references(() => employees.id).notNull(),
  employeeId: integer("employee_id").references(() => employees.id).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertEmployeeSupervisorSchema = createInsertSchema(employeeSupervisors).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  supervisorEmployeeId: z.number().int().positive("المشرف مطلوب"),
  employeeId: z.number().int().positive("الموظف مطلوب"),
});

export type InsertEmployeeSupervisor = z.infer<typeof insertEmployeeSupervisorSchema>;
export type EmployeeSupervisor = typeof employeeSupervisors.$inferSelect;

export type EmployeeSupervisorWithDetails = EmployeeSupervisor & {
  supervisor?: Employee;
  employee?: Employee;
};

export type SupervisorWithSubordinates = Employee & {
  subordinateCount: number;
};

// 9.8 WORK SHIFTS (الورديات)
export const shiftTypes = ["morning", "evening", "custom"] as const;
export type ShiftType = typeof shiftTypes[number];

export const workShifts = pgTable("work_shifts", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull(),
  shiftType: text("shift_type").notNull(), // "morning", "evening", "custom"
  startTime: text("start_time").notNull(), // HH:MM
  endTime: text("end_time").notNull(), // HH:MM
  breakStartTime: text("break_start_time"),
  breakEndTime: text("break_end_time"),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertWorkShiftSchema = createInsertSchema(workShifts).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  nameAr: z.string().min(1, "اسم الشفت مطلوب"),
  shiftType: z.enum(shiftTypes, { errorMap: () => ({ message: "نوع الشفت غير صالح" }) }),
  startTime: z.string().min(1, "وقت بداية الشفت مطلوب"),
  endTime: z.string().min(1, "وقت نهاية الشفت مطلوب"),
  breakStartTime: z.string().optional().nullable(),
  breakEndTime: z.string().optional().nullable(),
  isActive: z.boolean().optional().default(true),
});

export const updateWorkShiftSchema = insertWorkShiftSchema.partial();

export type InsertWorkShift = z.infer<typeof insertWorkShiftSchema>;
export type WorkShift = typeof workShifts.$inferSelect;

// 9.9 EMPLOYEE SHIFT ASSIGNMENTS (مناوبات الموظفين)
export const employeeShiftAssignments = pgTable("employee_shift_assignments", {
  id: serial("id").primaryKey(),
  employeeId: integer("employee_id").references(() => employees.id).notNull(),
  workShiftId: integer("work_shift_id").references(() => workShifts.id).notNull(),
  startDate: date("start_date").notNull(),
  endDate: date("end_date").notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertShiftAssignmentSchema = createInsertSchema(employeeShiftAssignments).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  employeeId: z.number().int().positive("الموظف مطلوب"),
  workShiftId: z.number().int().positive("الشفت مطلوب"),
  startDate: z.string().min(1, "تاريخ البداية مطلوب"),
  endDate: z.string().min(1, "تاريخ النهاية مطلوب"),
});

export const updateShiftAssignmentSchema = insertShiftAssignmentSchema.partial();

export type InsertShiftAssignment = z.infer<typeof insertShiftAssignmentSchema>;
export type ShiftAssignment = typeof employeeShiftAssignments.$inferSelect;

export type ShiftAssignmentWithDetails = ShiftAssignment & {
  employee?: Employee;
  workShift?: WorkShift;
};

// 9.10 ATTENDANCE RAW (البصمات الخام)
export const attendanceRaw = pgTable("attendance_raw", {
  id: serial("id").primaryKey(),
  employeeId: integer("employee_id").references(() => employees.id).notNull(),
  timestamp: timestamp("timestamp").notNull(),
  direction: text("direction"), // "in" or "out"
  deviceId: text("device_id"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertAttendanceRawSchema = createInsertSchema(attendanceRaw).omit({
  id: true,
  createdAt: true,
}).extend({
  employeeId: z.number().int().positive("الموظف مطلوب"),
  timestamp: z.string().min(1, "وقت البصمة مطلوب"),
  direction: z.enum(["in", "out"]).optional().nullable(),
  deviceId: z.string().optional().nullable(),
});

export type InsertAttendanceRaw = z.infer<typeof insertAttendanceRawSchema>;
export type AttendanceRaw = typeof attendanceRaw.$inferSelect;

// 9.11 ATTENDANCE LOGS (سجل الحضور اليومي)
export const attendanceStatuses = ["present", "absent", "late", "on_leave", "off"] as const;
export type AttendanceStatus = typeof attendanceStatuses[number];

export const attendanceLogs = pgTable("attendance_logs", {
  id: serial("id").primaryKey(),
  employeeId: integer("employee_id").references(() => employees.id).notNull(),
  date: date("date").notNull(),
  workShiftId: integer("work_shift_id").references(() => workShifts.id),
  scheduledStartTime: text("scheduled_start_time"),
  scheduledEndTime: text("scheduled_end_time"),
  actualCheckIn: text("actual_check_in"),
  actualCheckOut: text("actual_check_out"),
  lateMinutes: integer("late_minutes").default(0).notNull(),
  overtimeMinutes: integer("overtime_minutes").default(0).notNull(),
  totalWorkMinutes: integer("total_work_minutes").default(0).notNull(),
  status: text("status").notNull(), // "present", "absent", "late", "on_leave", "off"
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertAttendanceLogSchema = createInsertSchema(attendanceLogs).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  employeeId: z.number().int().positive("الموظف مطلوب"),
  date: z.string().min(1, "التاريخ مطلوب"),
  workShiftId: z.number().int().positive().optional().nullable(),
  scheduledStartTime: z.string().optional().nullable(),
  scheduledEndTime: z.string().optional().nullable(),
  actualCheckIn: z.string().optional().nullable(),
  actualCheckOut: z.string().optional().nullable(),
  lateMinutes: z.number().int().min(0).optional().default(0),
  overtimeMinutes: z.number().int().min(0).optional().default(0),
  totalWorkMinutes: z.number().int().min(0).optional().default(0),
  status: z.enum(attendanceStatuses, { errorMap: () => ({ message: "حالة الحضور غير صالحة" }) }),
  notes: z.string().optional().nullable(),
});

export const updateAttendanceLogSchema = insertAttendanceLogSchema.partial();

export type InsertAttendanceLog = z.infer<typeof insertAttendanceLogSchema>;
export type AttendanceLog = typeof attendanceLogs.$inferSelect;

export type AttendanceLogWithDetails = AttendanceLog & {
  employee?: Employee;
  workShift?: WorkShift;
};

// 9.12 OVERTIME ENTRIES (الوقت الإضافي)
export const overtimeEntries = pgTable("overtime_entries", {
  id: serial("id").primaryKey(),
  employeeId: integer("employee_id").references(() => employees.id).notNull(),
  date: date("date").notNull(),
  startTime: text("start_time").notNull(),
  endTime: text("end_time").notNull(),
  totalMinutes: integer("total_minutes").notNull(),
  reason: text("reason"),
  approvedByEmployeeId: integer("approved_by_employee_id").references(() => employees.id),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertOvertimeEntrySchema = createInsertSchema(overtimeEntries).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  employeeId: z.number().int().positive("الموظف مطلوب"),
  date: z.string().min(1, "التاريخ مطلوب"),
  startTime: z.string().min(1, "وقت البداية مطلوب"),
  endTime: z.string().min(1, "وقت النهاية مطلوب"),
  totalMinutes: z.number().int().min(1, "مدة الوقت الإضافي مطلوبة"),
  reason: z.string().optional().nullable(),
  approvedByEmployeeId: z.number().int().positive().optional().nullable(),
});

export const updateOvertimeEntrySchema = insertOvertimeEntrySchema.partial();

export type InsertOvertimeEntry = z.infer<typeof insertOvertimeEntrySchema>;
export type OvertimeEntry = typeof overtimeEntries.$inferSelect;

export type OvertimeEntryWithDetails = OvertimeEntry & {
  employee?: Employee;
  approvedByEmployee?: Employee;
};

// 9.13 LEAVES (الإجازات)
export const leaveTypes = ["annual", "sick", "unpaid", "public_holiday", "paid", "other", "hourly"] as const;
export type LeaveType = typeof leaveTypes[number];

export const leaveStatuses = ["pending", "approved", "rejected", "canceled"] as const;
export type LeaveStatus = typeof leaveStatuses[number];

export const leaves = pgTable("leaves", {
  id: serial("id").primaryKey(),
  employeeId: integer("employee_id").references(() => employees.id).notNull(),
  requestedByEmployeeId: integer("requested_by_employee_id").references(() => employees.id),
  leaveType: text("leave_type").notNull(), // "annual", "sick", "unpaid", "public_holiday", "paid", "other", "hourly"
  startDate: date("start_date").notNull(),
  endDate: date("end_date").notNull(),
  startTime: text("start_time"), // for hourly leave
  endTime: text("end_time"), // for hourly leave
  durationDays: real("duration_days").default(0).notNull(),
  durationHours: real("duration_hours").default(0).notNull(),
  status: text("status").notNull(), // "pending", "approved", "rejected", "canceled"
  attachmentUrl: text("attachment_url"),
  reason: text("reason"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertLeaveSchema = createInsertSchema(leaves).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  employeeId: z.number().int().positive("الموظف مطلوب"),
  requestedByEmployeeId: z.number().int().positive().optional().nullable(),
  leaveType: z.enum(leaveTypes, { errorMap: () => ({ message: "نوع الإجازة غير صالح" }) }),
  startDate: z.string().min(1, "تاريخ البداية مطلوب"),
  endDate: z.string().min(1, "تاريخ النهاية مطلوب"),
  startTime: z.string().optional().nullable(),
  endTime: z.string().optional().nullable(),
  durationDays: z.number().min(0).optional().default(0),
  durationHours: z.number().min(0).optional().default(0),
  status: z.enum(leaveStatuses).optional().default("pending"),
  attachmentUrl: z.string().url().optional().nullable().or(z.literal("")),
  reason: z.string().optional().nullable(),
});

export const updateLeaveSchema = insertLeaveSchema.partial();

export type InsertLeave = z.infer<typeof insertLeaveSchema>;
export type Leave = typeof leaves.$inferSelect;

export type LeaveWithDetails = Leave & {
  employee?: Employee;
  requestedByEmployee?: Employee;
};

// ========================================
// 10) SALON ORDERS (طلبات الصالون)
// ========================================

export const orderStatuses = ["new", "in_progress", "completed", "canceled", "scheduled", "postponed", "returned"] as const;
export type OrderStatus = typeof orderStatuses[number];

export const orderServiceStatuses = ["pending", "in_progress", "completed", "deferred", "cancelled"] as const;
export type OrderServiceStatus = typeof orderServiceStatuses[number];

export const orderSources = ["pos", "app", "phone", "online"] as const;
export type OrderSource = typeof orderSources[number];

export const orderTypes = ["normal", "gift"] as const;
export type OrderType = typeof orderTypes[number];

// Gift sender types
export const giftFromTypes = ["customer", "supervisor"] as const;
export type GiftFromType = typeof giftFromTypes[number];

// Main orders table
export const orders = pgTable("orders", {
  id: serial("id").primaryKey(),
  orderNumber: text("order_number").unique().notNull(), // e.g. "SO-0001"
  orderDate: timestamp("order_date").notNull(),
  source: text("source").notNull(), // "pos", "app", "phone", "online"
  orderType: text("order_type").notNull(), // "normal", "gift"
  clientId: integer("client_id").references(() => customers.id), // nullable for walk-in
  createdByEmployeeId: integer("created_by_employee_id").references(() => employees.id).notNull(),
  branchId: integer("branch_id"), // future: foreign key to branches table
  packageId: integer("package_id"), // Marketing: linked package
  couponId: integer("coupon_id"), // Marketing: linked coupon
  couponCode: text("coupon_code"),
  couponDiscountAmount: real("coupon_discount_amount").default(0).notNull(),
  status: text("status").notNull(), // "new", "in_progress", "completed", "canceled", "scheduled", "postponed", "returned"
  scheduledAt: timestamp("scheduled_at"), // تاريخ ووقت الموعد المجدول
  notes: text("notes"),
  subtotalAmount: real("subtotal_amount").default(0).notNull(), // مجموع الخدمات/المنتجات بدون ضريبة
  vatAmount: real("vat_amount").default(0).notNull(), // مجموع الضريبة
  totalAmount: real("total_amount").default(0).notNull(), // الإجمالي = subtotal + vat - coupon_discount
  paidAmount: real("paid_amount").default(0).notNull(), // المبلغ المدفوع
  paymentStatus: text("payment_status").default("unpaid").notNull(), // "unpaid", "partially_paid", "paid"
  // Gift order fields
  giftFromType: text("gift_from_type"), // "customer" or "supervisor"
  giftFromCustomerId: integer("gift_from_customer_id").references(() => customers.id), // المُهدي إذا كان عميل
  giftFromSupervisorId: integer("gift_from_supervisor_id").references(() => employees.id), // المُهدي إذا كان مشرف
  giftToCustomerId: integer("gift_to_customer_id").references(() => customers.id), // المُهدى له
  giftExpiryDate: date("gift_expiry_date"), // تاريخ انتهاء صلاحية الهدية
  // Rating fields (for app ratings)
  ratingValue: integer("rating_value"), // 1-5
  ratingComment: text("rating_comment"),
  // Accounting linkage
  journalEntryId: integer("journal_entry_id"), // FK to journal_entries.id (for tracking)
  cogsJournalEntryId: integer("cogs_journal_entry_id"), // FK to COGS journal entry
  // Order lifecycle
  completedAt: timestamp("completed_at"), // تاريخ الإكمال
  deferredAt: timestamp("deferred_at"), // تاريخ التأجيل
  deferredReason: text("deferred_reason"), // سبب التأجيل
  cancelledAt: timestamp("cancelled_at"), // تاريخ الإلغاء
  cancelledReason: text("cancelled_reason"), // سبب الإلغاء
  returnedAt: timestamp("returned_at"), // تاريخ المرتجع
  returnedReason: text("returned_reason"), // سبب المرتجع
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Order services line items
export const orderServices = pgTable("order_services", {
  id: serial("id").primaryKey(),
  orderId: integer("order_id").references(() => orders.id).notNull(),
  serviceId: integer("service_id").references(() => services.id).notNull(),
  packageId: integer("package_id").references(() => packages.id), // null = standalone service, not null = from package
  quantity: real("quantity").default(1).notNull(), // عدد مرات تنفيذ الخدمة
  basePrice: real("base_price").notNull(), // سعر الخدمة المفترض (من service.price)
  vatType: text("vat_type").notNull(), // "inclusive", "exclusive", "exempt"
  vatRate: real("vat_rate").notNull(), // نسبة الضريبة
  lineSubtotal: real("line_subtotal").notNull(), // المبلغ بدون ضريبة
  vatAmount: real("vat_amount").notNull(), // قيمة الضريبة
  lineTotal: real("line_total").notNull(), // المبلغ مع الضريبة
  scheduledAt: timestamp("scheduled_at"), // تاريخ ووقت تنفيذ الخدمة (اختياري)
  executingEmployeeId: integer("executing_employee_id").references(() => employees.id),
  status: text("status").default("pending").notNull(), // "pending", "in_progress", "completed", "deferred", "cancelled"
  deferredAt: timestamp("deferred_at"), // تاريخ التأجيل
  deferredReason: text("deferred_reason"), // سبب التأجيل
  completedAt: timestamp("completed_at"), // تاريخ الاكتمال
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Order products line items (additional products sold)
export const orderProducts = pgTable("order_products", {
  id: serial("id").primaryKey(),
  orderId: integer("order_id").references(() => orders.id).notNull(),
  productId: integer("product_id").references(() => products.id).notNull(),
  packageId: integer("package_id").references(() => packages.id), // null for standalone, number for package products
  quantity: real("quantity").notNull(), // بوحدة البيع للمنتج
  unitPrice: real("unit_price").notNull(), // سعر بيع الوحدة
  vatType: text("vat_type").notNull(), // "inclusive", "exclusive", "exempt"
  vatRate: real("vat_rate").notNull(),
  lineSubtotal: real("line_subtotal").notNull(), // بدون الضريبة
  vatAmount: real("vat_amount").notNull(),
  lineTotal: real("line_total").notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Order packages junction table (for multi-package orders)
export const orderPackages = pgTable("order_packages", {
  id: serial("id").primaryKey(),
  orderId: integer("order_id").references(() => orders.id).notNull(),
  packageId: integer("package_id").references(() => packages.id).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertOrderPackageSchema = createInsertSchema(orderPackages).omit({
  id: true,
  createdAt: true,
});

export type InsertOrderPackage = z.infer<typeof insertOrderPackageSchema>;
export type OrderPackage = typeof orderPackages.$inferSelect;

// Order ratings
export const orderRatings = pgTable("order_ratings", {
  id: serial("id").primaryKey(),
  orderId: integer("order_id").references(() => orders.id).notNull(),
  clientId: integer("client_id").references(() => customers.id),
  rating: integer("rating").notNull(), // 1-5
  comment: text("comment"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

// Order returns (مرتجعات الطلبات)
export const orderReturns = pgTable("order_returns", {
  id: serial("id").primaryKey(),
  orderId: integer("order_id").references(() => orders.id).notNull(),
  returnNumber: text("return_number").unique().notNull(), // e.g. "RET-0001"
  creditNoteNumber: text("credit_note_number").unique().notNull(), // رقم إشعار الدائن e.g. "CN-0001"
  returnDate: timestamp("return_date").notNull(),
  processedByEmployeeId: integer("processed_by_employee_id").references(() => employees.id),
  refundPaymentMethodId: integer("refund_payment_method_id").references(() => paymentMethods.id), // طريقة استرداد المبلغ
  reason: text("reason").notNull(), // سبب المرتجع
  returnType: text("return_type").notNull(), // "full" | "partial" | "cancellation"
  originalInvoiceNumber: text("original_invoice_number"), // رقم الفاتورة الأصلية
  subtotalAmount: real("subtotal_amount").default(0).notNull(), // المبلغ قبل الضريبة
  vatAmount: real("vat_amount").default(0).notNull(), // قيمة الضريبة
  refundAmount: real("refund_amount").default(0).notNull(), // قيمة المبلغ المسترد (الإجمالي شامل الضريبة)
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Order return lines (خطوط المرتجعات)
export const orderReturnLines = pgTable("order_return_lines", {
  id: serial("id").primaryKey(),
  returnId: integer("return_id").references(() => orderReturns.id).notNull(),
  orderServiceId: integer("order_service_id").references(() => orderServices.id), // if returning a service
  orderProductId: integer("order_product_id").references(() => orderProducts.id), // if returning a product
  quantity: real("quantity").notNull(), // الكمية المرتجعة
  refundAmount: real("refund_amount").notNull(), // قيمة المبلغ المسترد لهذا الخط
  reason: text("reason"), // سبب خاص بهذا الخط
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

// Order payments (مدفوعات الطلبات)
export const orderPayments = pgTable("order_payments", {
  id: serial("id").primaryKey(),
  orderId: integer("order_id").references(() => orders.id).notNull(),
  paymentMethodId: integer("payment_method_id").references(() => paymentMethods.id).notNull(),
  paymentDate: date("payment_date").notNull(),
  amount: real("amount").notNull(),
  notes: text("notes"),
  journalEntryId: integer("journal_entry_id"), // FK to journal_entries.id (for tracking)
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertOrderPaymentSchema = createInsertSchema(orderPayments).omit({
  id: true,
  journalEntryId: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  orderId: z.number().int().positive("الطلب مطلوب"),
  paymentMethodId: z.number().int().positive("طريقة الدفع مطلوبة"),
  paymentDate: z.string({ required_error: "تاريخ الدفع مطلوب" }),
  amount: z.number().positive("المبلغ يجب أن يكون أكبر من صفر"),
  notes: z.string().optional().nullable(),
});

export type InsertOrderPayment = z.infer<typeof insertOrderPaymentSchema>;
export type OrderPayment = typeof orderPayments.$inferSelect;

// Insert schemas with Arabic validation messages
export const insertOrderSchema = createInsertSchema(orders).omit({
  id: true,
  orderNumber: true,
  subtotalAmount: true,
  vatAmount: true,
  totalAmount: true,
  journalEntryId: true,
  cogsJournalEntryId: true,
  completedAt: true,
  deferredAt: true,
  deferredReason: true,
  cancelledAt: true,
  cancelledReason: true,
  returnedAt: true,
  returnedReason: true,
  ratingValue: true,
  ratingComment: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  orderDate: z.coerce.date({ errorMap: () => ({ message: "تاريخ الطلب مطلوب" }) }),
  source: z.enum(orderSources, { errorMap: () => ({ message: "مصدر الطلب غير صالح" }) }),
  orderType: z.enum(orderTypes, { errorMap: () => ({ message: "نوع الطلب غير صالح" }) }).default("normal"),
  clientId: z.number().int().positive().optional().nullable(),
  createdByEmployeeId: z.number().int().positive("الموظف المنشئ مطلوب"),
  branchId: z.number().int().positive().optional().nullable(),
  packageId: z.number().int().positive().optional().nullable(),
  couponId: z.number().int().positive().optional().nullable(),
  couponCode: z.string().optional().nullable(),
  couponDiscountAmount: z.number().min(0, "قيمة الخصم لا تقل عن صفر").optional().default(0),
  status: z.enum(orderStatuses, { errorMap: () => ({ message: "حالة الطلب غير صالحة" }) }),
  scheduledAt: z.coerce.date().optional().nullable(),
  notes: z.string().optional().nullable(),
  // Gift fields
  giftFromType: z.enum(giftFromTypes, { errorMap: () => ({ message: "نوع المُهدي غير صالح" }) }).optional().nullable(),
  giftFromCustomerId: z.number().int().positive().optional().nullable(),
  giftFromSupervisorId: z.number().int().positive().optional().nullable(),
  giftToCustomerId: z.number().int().positive().optional().nullable(),
  giftExpiryDate: z.string().optional().nullable(), // date as string
}).refine(
  (data) => {
    // If order type is gift, validate gift fields
    if (data.orderType === "gift") {
      if (!data.giftFromType) return false;
      if (!data.giftToCustomerId) return false;
      if (!data.giftExpiryDate) return false;
      if (data.giftFromType === "customer" && !data.giftFromCustomerId) return false;
      if (data.giftFromType === "supervisor" && !data.giftFromSupervisorId) return false;
    }
    return true;
  },
  { message: "للطلبات من نوع هدية، يجب تحديد المُهدي والمُهدى له وتاريخ انتهاء الهدية" }
);

export const insertOrderServiceSchema = createInsertSchema(orderServices).omit({
  id: true,
  lineSubtotal: true,
  vatAmount: true,
  lineTotal: true,
  status: true,
  deferredAt: true,
  deferredReason: true,
  completedAt: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  orderId: z.number().int().positive().optional(),
  serviceId: z.number().int().positive("الخدمة مطلوبة"),
  quantity: z.number().positive("الكمية يجب أن تكون أكبر من صفر").optional().default(1),
  basePrice: z.number().min(0, "السعر لا يقل عن صفر").optional(),
  vatType: z.enum(vatTypes, { errorMap: () => ({ message: "نوع الضريبة غير صالح" }) }).optional(),
  vatRate: z.number().min(0).max(1).optional(),
  scheduledAt: z.coerce.date().optional().nullable(),
  executingEmployeeId: z.number().int().positive().optional().nullable(),
});

export const insertOrderProductSchema = createInsertSchema(orderProducts).omit({
  id: true,
  lineSubtotal: true,
  vatAmount: true,
  lineTotal: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  orderId: z.number().int().positive().optional(),
  productId: z.number().int().positive("المنتج مطلوب"),
  quantity: z.number().positive("الكمية يجب أن تكون أكبر من صفر"),
  unitPrice: z.number().min(0, "السعر لا يقل عن صفر").optional(),
  vatType: z.enum(vatTypes, { errorMap: () => ({ message: "نوع الضريبة غير صالح" }) }).optional(),
  vatRate: z.number().min(0).max(1).optional(),
});

export const insertOrderRatingSchema = createInsertSchema(orderRatings).omit({
  id: true,
  createdAt: true,
}).extend({
  orderId: z.number().int().positive(),
  clientId: z.number().int().positive().optional().nullable(),
  rating: z.number().int().min(1, "التقييم يجب أن يكون 1 على الأقل").max(5, "التقييم لا يتجاوز 5"),
  comment: z.string().optional().nullable(),
});

export const updateOrderSchema = z.object({
  status: z.enum(orderStatuses, { errorMap: () => ({ message: "حالة الطلب غير صالحة" }) }).optional(),
  notes: z.string().optional().nullable(),
  couponCode: z.string().optional().nullable(),
  couponDiscountAmount: z.number().min(0).optional(),
  scheduledAt: z.coerce.date().optional().nullable(),
  completedAt: z.coerce.date().optional().nullable(),
  deferredAt: z.coerce.date().optional().nullable(),
  deferredReason: z.string().optional().nullable(),
  cancelledAt: z.coerce.date().optional().nullable(),
  cancelledReason: z.string().optional().nullable(),
  returnedAt: z.coerce.date().optional().nullable(),
  returnedReason: z.string().optional().nullable(),
  journalEntryId: z.number().int().positive().optional().nullable(),
  cogsJournalEntryId: z.number().int().positive().optional().nullable(),
});

// Order rating schema (for submitting ratings from app)
export const submitOrderRatingSchema = z.object({
  ratingValue: z.number().int().min(1, "التقييم يجب أن يكون 1 على الأقل").max(5, "التقييم لا يتجاوز 5"),
  ratingComment: z.string().optional().nullable(),
});

// Update order service schema (for per-service status and employee assignment)
export const updateOrderServiceSchema = z.object({
  executingEmployeeId: z.number().int().positive().optional().nullable(),
  status: z.enum(orderServiceStatuses, { errorMap: () => ({ message: "حالة الخدمة غير صالحة" }) }).optional(),
  deferredAt: z.coerce.date().optional().nullable(),
  deferredReason: z.string().optional().nullable(),
  completedAt: z.coerce.date().optional().nullable(),
  scheduledAt: z.coerce.date().optional().nullable(),
});

// Insert order return schema
export const insertOrderReturnSchema = createInsertSchema(orderReturns).omit({
  id: true,
  returnNumber: true,
  creditNoteNumber: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  orderId: z.number().int().positive("الطلب مطلوب"),
  returnDate: z.coerce.date({ errorMap: () => ({ message: "تاريخ المرتجع مطلوب" }) }),
  processedByEmployeeId: z.number().int().positive().optional().nullable(),
  reason: z.string().min(1, "سبب المرتجع مطلوب"),
  returnType: z.enum(["full", "partial", "cancellation"], { errorMap: () => ({ message: "نوع المرتجع غير صالح" }) }),
  originalInvoiceNumber: z.string().optional().nullable(),
  subtotalAmount: z.number().min(0).optional().default(0),
  vatAmount: z.number().min(0).optional().default(0),
  refundAmount: z.number().min(0, "قيمة الاسترداد لا تقل عن صفر").optional().default(0),
  notes: z.string().optional().nullable(),
});

// Insert order return line schema
export const insertOrderReturnLineSchema = createInsertSchema(orderReturnLines).omit({
  id: true,
  createdAt: true,
}).extend({
  returnId: z.number().int().positive().optional(),
  orderServiceId: z.number().int().positive().optional().nullable(),
  orderProductId: z.number().int().positive().optional().nullable(),
  quantity: z.number().positive("الكمية يجب أن تكون أكبر من صفر"),
  refundAmount: z.number().min(0, "قيمة الاسترداد لا تقل عن صفر"),
  reason: z.string().optional().nullable(),
});

// Types
export type InsertOrder = z.infer<typeof insertOrderSchema>;
export type Order = typeof orders.$inferSelect;
export type InsertOrderService = z.infer<typeof insertOrderServiceSchema>;
export type OrderService = typeof orderServices.$inferSelect;
export type InsertOrderProduct = z.infer<typeof insertOrderProductSchema>;
export type OrderProduct = typeof orderProducts.$inferSelect;
export type InsertOrderRating = z.infer<typeof insertOrderRatingSchema>;
export type OrderRating = typeof orderRatings.$inferSelect;
export type InsertOrderReturn = z.infer<typeof insertOrderReturnSchema>;
export type OrderReturn = typeof orderReturns.$inferSelect;
export type InsertOrderReturnLine = z.infer<typeof insertOrderReturnLineSchema>;
export type OrderReturnLine = typeof orderReturnLines.$inferSelect;

// Extended types with relations
export type OrderServiceWithDetails = OrderService & {
  service?: Service & { category?: ServiceCategory };
  executingEmployee?: Employee;
};

export type OrderProductWithDetails = OrderProduct & {
  product?: Product & { sellUnit?: Unit };
};

export type OrderReturnLineWithDetails = OrderReturnLine & {
  orderService?: OrderServiceWithDetails;
  orderProduct?: OrderProductWithDetails;
};

export type OrderReturnWithDetails = OrderReturn & {
  order?: Order;
  processedByEmployee?: Employee;
  lines: OrderReturnLineWithDetails[];
};

export type OrderPackageWithDetails = OrderPackage & {
  package?: Package;
};

export type OrderWithDetails = Order & {
  client?: Customer;
  createdByEmployee?: Employee;
  services: OrderServiceWithDetails[];
  products: OrderProductWithDetails[];
  packages?: OrderPackageWithDetails[];
  rating?: OrderRating;
  returns?: OrderReturnWithDetails[];
};

// Create order request type
export type CreateOrderRequest = {
  orderType: OrderType;
  source: OrderSource;
  clientId?: number | null;
  createdByEmployeeId: number;
  branchId?: number | null;
  packageId?: number | null; // Legacy single package support
  packageIds?: number[]; // New multi-package support
  couponId?: number | null;
  couponCode?: string | null;
  couponDiscountAmount?: number;
  services: {
    serviceId: number;
    quantity?: number;
    scheduledAt?: string | null;
    executingEmployeeId?: number | null;
    packageId?: number | null; // null for standalone, number for package services
    packagePrice?: number | null; // VAT-inclusive package price (overrides service price when in package)
  }[];
  products?: {
    productId: number;
    quantity: number;
    packageId?: number | null; // null for standalone, number for package products
    packagePrice?: number | null; // VAT-inclusive package price (overrides product price when in package)
  }[];
  notes?: string | null;
  // Gift order fields
  giftFromType?: string | null;
  giftFromCustomerId?: number | null;
  giftFromSupervisorId?: number | null;
  giftToCustomerId?: number | null;
  giftExpiryDate?: string | null;
  // Payment fields (optional - if provided, creates payments on order creation)
  // Supports multiple partial payments with different methods
  payments?: OrderPaymentEntry[];
  // Legacy single payment support (deprecated, use payments array instead)
  paymentMethodId?: number | null;
  paymentAmount?: number | null;
  // Loyalty points redemption (optional)
  redeemLoyaltyPoints?: number | null; // Number of points to redeem
  loyaltyPointsValue?: number | null; // Value in SAR of redeemed points
};

// Payment entry for order creation (partial payments support)
export type OrderPaymentEntry = {
  paymentMethodId: number;
  amount: number;
  notes?: string | null;
};

// Order list filters
export type OrderFilters = {
  date?: string; // YYYY-MM-DD format for "today" filter
  status?: OrderStatus;
  source?: OrderSource;
  orderType?: OrderType;
  clientId?: number;
  branchId?: number;
  hasRating?: boolean; // filter orders with ratings
  isScheduled?: boolean; // filter scheduled orders
};

// ========================================
// SALES ACCOUNTING CONFIGURATION
// ========================================
// Account codes for sales journal entries
export const SALES_ACCOUNT_CODES = {
  SALON_REVENUE: "4100",     // إيرادات الصالون
  VAT_OUTPUT: "2411",        // ضريبة القيمة المضافة المحصلة
  COGS: "5100",              // تكلفة البضاعة المباعة
  INVENTORY: "1201",         // المخزون
  CASH: "1110",              // النقدية
  BANK: "1120",              // البنك
  CARD: "1130",              // الدفع بالبطاقة
  ACCOUNTS_RECEIVABLE: "1100", // العملاء (المدينون)
} as const;

// ========================================
// 11) APPOINTMENTS MODULE (حجوزات التطبيق)
// ========================================

// Appointment status types
export const appointmentStatusValues = ["pending", "confirmed", "in_progress", "completed", "canceled", "no_show", "postponed"] as const;
export type AppointmentStatus = typeof appointmentStatusValues[number];

// Appointment type (in salon or home service)
export const appointmentTypeValues = ["in_salon", "home_service"] as const;
export type AppointmentType = typeof appointmentTypeValues[number];

// Appointment source
export const appointmentSourceValues = ["app", "phone", "walk_in"] as const;
export type AppointmentSource = typeof appointmentSourceValues[number];

// Appointments table (header)
export const appointments = pgTable("appointments", {
  id: serial("id").primaryKey(),
  appointmentNumber: text("appointment_number").unique().notNull(), // e.g. "AP-0001"
  source: text("source").notNull(), // "app", "phone", "walk_in"
  clientId: integer("client_id").references(() => customers.id).notNull(),
  createdByEmployeeId: integer("created_by_employee_id").references(() => employees.id), // الموظف المنشئ
  branchId: integer("branch_id"), // future: foreign key to branches table
  appointmentType: text("appointment_type").notNull(), // "in_salon", "home_service"
  scheduledAt: timestamp("scheduled_at").notNull(), // تاريخ ووقت الحجز المطلوب
  status: text("status").notNull(), // "pending", "confirmed", "in_progress", "completed", "canceled", "no_show", "postponed"
  packageId: integer("package_id"), // Marketing: linked package
  couponId: integer("coupon_id"), // Marketing: linked coupon
  couponCode: text("coupon_code"), // كود الكوبون
  couponDiscountAmount: real("coupon_discount_amount").default(0).notNull(), // خصم الكوبون
  deliveryFee: real("delivery_fee").default(0).notNull(), // رسوم التوصيل للخدمة المنزلية - يُحمّل من الإعدادات
  locationLat: real("location_lat"), // إحداثيات الموقع - للخدمة المنزلية فقط
  locationLng: real("location_lng"),
  addressText: text("address_text"), // عنوان نصي (الحي، الشارع، الخ)
  notes: text("notes"),
  subtotalAmount: real("subtotal_amount").default(0).notNull(), // مجموع الخدمات بدون ضريبة
  vatAmount: real("vat_amount").default(0).notNull(), // مجموع الضريبة على الخدمات
  totalAmount: real("total_amount").default(0).notNull(), // الإجمالي مع الضريبة + رسوم التوصيل - خصم الكوبون
  relatedOrderId: integer("related_order_id").references(() => orders.id), // ربط اختياري مع طلب صالون
  // Rating fields (for app ratings)
  ratingValue: integer("rating_value"), // 1-5
  ratingComment: text("rating_comment"),
  // Accounting linkage
  journalEntryId: integer("journal_entry_id"), // FK to journal_entries.id (for tracking)
  // Lifecycle fields
  completedAt: timestamp("completed_at"), // تاريخ الإكمال
  cancelledAt: timestamp("cancelled_at"), // تاريخ الإلغاء
  cancelledReason: text("cancelled_reason"), // سبب الإلغاء
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Appointment services table
export const appointmentServices = pgTable("appointment_services", {
  id: serial("id").primaryKey(),
  appointmentId: integer("appointment_id").references(() => appointments.id).notNull(),
  serviceId: integer("service_id").references(() => services.id).notNull(),
  quantity: real("quantity").default(1).notNull(), // عدد مرات تنفيذ الخدمة في الحجز
  basePrice: real("base_price").notNull(), // من service.price في وقت إنشاء الحجز
  vatType: text("vat_type").notNull(), // "inclusive", "exclusive", "exempt"
  vatRate: real("vat_rate").notNull(), // نسبة الضريبة
  lineSubtotal: real("line_subtotal").notNull(), // بدون ضريبة
  vatAmount: real("vat_amount").notNull(), // قيمة الضريبة
  lineTotal: real("line_total").notNull(), // مع الضريبة
  preferredEmployeeId: integer("preferred_employee_id").references(() => employees.id), // موظف مفضّل
  executingEmployeeId: integer("executing_employee_id").references(() => employees.id), // الموظف المنفذ الفعلي
  status: text("status").default("pending").notNull(), // "pending", "in_progress", "completed", "cancelled"
  completedAt: timestamp("completed_at"), // تاريخ الاكتمال
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Insert schemas for appointments
export const insertAppointmentSchema = createInsertSchema(appointments).omit({
  id: true,
  appointmentNumber: true,
  deliveryFee: true, // loaded from settings for home_service
  subtotalAmount: true,
  vatAmount: true,
  totalAmount: true,
  journalEntryId: true,
  completedAt: true,
  cancelledAt: true,
  ratingValue: true,
  ratingComment: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  clientId: z.number().int().positive("العميل مطلوب"),
  createdByEmployeeId: z.number().int().positive().optional().nullable(),
  appointmentType: z.enum(appointmentTypeValues, { errorMap: () => ({ message: "نوع الحجز غير صالح" }) }),
  scheduledAt: z.coerce.date({ errorMap: () => ({ message: "تاريخ ووقت الحجز مطلوب" }) }),
  status: z.enum(appointmentStatusValues, { errorMap: () => ({ message: "حالة الحجز غير صالحة" }) }),
  packageId: z.number().int().positive().optional().nullable(),
  couponId: z.number().int().positive().optional().nullable(),
  couponCode: z.string().optional().nullable(),
  couponDiscountAmount: z.number().min(0).optional().default(0),
  notes: z.string().optional().nullable(),
  cancelledReason: z.string().optional().nullable(),
});

// Update appointment schema
export const updateAppointmentSchema = z.object({
  status: z.enum(appointmentStatusValues, { errorMap: () => ({ message: "حالة الحجز غير صالحة" }) }).optional(),
  scheduledAt: z.coerce.date().optional(),
  notes: z.string().optional().nullable(),
  completedAt: z.coerce.date().optional().nullable(),
  cancelledAt: z.coerce.date().optional().nullable(),
  cancelledReason: z.string().optional().nullable(),
  journalEntryId: z.number().int().positive().optional().nullable(),
});

// Submit appointment rating schema
export const submitAppointmentRatingSchema = z.object({
  ratingValue: z.number().int().min(1, "التقييم يجب أن يكون 1 على الأقل").max(5, "التقييم لا يتجاوز 5"),
  ratingComment: z.string().optional().nullable(),
});

export const insertAppointmentServiceSchema = createInsertSchema(appointmentServices).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

// Types for appointments
export type InsertAppointment = z.infer<typeof insertAppointmentSchema>;
export type Appointment = typeof appointments.$inferSelect;
export type InsertAppointmentService = z.infer<typeof insertAppointmentServiceSchema>;
export type AppointmentService = typeof appointmentServices.$inferSelect;

// Extended types with relations
export type AppointmentServiceWithDetails = AppointmentService & {
  service?: Service;
  preferredEmployee?: Employee;
};

export type AppointmentWithDetails = Appointment & {
  client?: Customer;
  services: AppointmentServiceWithDetails[];
  relatedOrder?: Order;
};

// Create appointment request type (matches order creation flow)
export type CreateAppointmentRequest = {
  clientId: number;
  branchId?: number | null;
  appointmentType: AppointmentType;
  scheduledAt: string; // ISO datetime
  // For home_service only:
  locationLat?: number | null;
  locationLng?: number | null;
  addressText?: string | null;
  // deliveryFee is NOT passed by API - loaded from settings automatically
  services: {
    serviceId: number;
    quantity?: number;
    preferredEmployeeId?: number | null;
    scheduledAt?: string | null; // optional per-service scheduled time
  }[];
  notes?: string | null;
  source?: AppointmentSource; // default: "app"
};

// Appointment list filters
export type AppointmentFilters = {
  date?: string; // YYYY-MM-DD format
  status?: AppointmentStatus;
  appointmentType?: AppointmentType;
  source?: AppointmentSource;
  clientId?: number;
  branchId?: number;
  hasRating?: boolean; // filter appointments with ratings
};

// ========================================
// 12) MARKETING MODULE (التسويق: الحزم + الكوبونات)
// ========================================

// Discount type for coupons
export const discountTypeValues = ["percent", "amount"] as const;
export type DiscountType = typeof discountTypeValues[number];

// Packages (الحزم / البكجات)
export const packages = pgTable("packages", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull(),
  imageUrl: text("image_url"),
  startDate: date("start_date").notNull(),
  endDate: date("end_date").notNull(),
  descriptionAr: text("description_ar"),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Package Services (خدمات ضمن الحزمة)
export const packageServices = pgTable("package_services", {
  id: serial("id").primaryKey(),
  packageId: integer("package_id").references(() => packages.id).notNull(),
  serviceId: integer("service_id").references(() => services.id).notNull(),
  quantity: real("quantity").default(1).notNull(),
  packagePrice: real("package_price").notNull(),
});

// Package Products (منتجات ضمن الحزمة)
export const packageProducts = pgTable("package_products", {
  id: serial("id").primaryKey(),
  packageId: integer("package_id").references(() => packages.id).notNull(),
  productId: integer("product_id").references(() => products.id).notNull(),
  quantity: real("quantity").default(1).notNull(),
  packagePrice: real("package_price").notNull(),
});

// Coupons (الكوبونات)
export const coupons = pgTable("coupons", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull(),
  code: text("code").unique().notNull(),
  descriptionAr: text("description_ar"),
  discountType: text("discount_type").notNull(), // "percent" or "amount"
  discountValue: real("discount_value").notNull(),
  appliesToServices: boolean("applies_to_services").default(true).notNull(),
  appliesToProducts: boolean("applies_to_products").default(false).notNull(),
  startDate: date("start_date").notNull(),
  endDate: date("end_date").notNull(),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Coupon Services (خدمات محددة للكوبون - إذا فارغة يطبق على الكل)
export const couponServices = pgTable("coupon_services", {
  id: serial("id").primaryKey(),
  couponId: integer("coupon_id").references(() => coupons.id).notNull(),
  serviceId: integer("service_id").references(() => services.id).notNull(),
});

// Coupon Products (منتجات محددة للكوبون - إذا فارغة يطبق على الكل)
export const couponProducts = pgTable("coupon_products", {
  id: serial("id").primaryKey(),
  couponId: integer("coupon_id").references(() => coupons.id).notNull(),
  productId: integer("product_id").references(() => products.id).notNull(),
});

// Insert schemas for Marketing
export const insertPackageSchema = createInsertSchema(packages).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertPackageServiceSchema = createInsertSchema(packageServices).omit({
  id: true,
});

export const insertPackageProductSchema = createInsertSchema(packageProducts).omit({
  id: true,
});

export const insertCouponSchema = createInsertSchema(coupons).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertCouponServiceSchema = createInsertSchema(couponServices).omit({
  id: true,
});

export const insertCouponProductSchema = createInsertSchema(couponProducts).omit({
  id: true,
});

// Types for Marketing
export type InsertPackage = z.infer<typeof insertPackageSchema>;
export type Package = typeof packages.$inferSelect;
export type InsertPackageService = z.infer<typeof insertPackageServiceSchema>;
export type PackageService = typeof packageServices.$inferSelect;
export type InsertPackageProduct = z.infer<typeof insertPackageProductSchema>;
export type PackageProduct = typeof packageProducts.$inferSelect;

export type InsertCoupon = z.infer<typeof insertCouponSchema>;
export type Coupon = typeof coupons.$inferSelect;
export type InsertCouponService = z.infer<typeof insertCouponServiceSchema>;
export type CouponService = typeof couponServices.$inferSelect;
export type InsertCouponProduct = z.infer<typeof insertCouponProductSchema>;
export type CouponProduct = typeof couponProducts.$inferSelect;

// Extended types with relations
export type PackageServiceWithDetails = PackageService & {
  service?: Service;
};

export type PackageProductWithDetails = PackageProduct & {
  product?: Product;
};

export type PackageWithDetails = Package & {
  services: PackageServiceWithDetails[];
  products: PackageProductWithDetails[];
};

export type CouponServiceWithDetails = CouponService & {
  service?: Service;
};

export type CouponProductWithDetails = CouponProduct & {
  product?: Product;
};

export type CouponWithDetails = Coupon & {
  allowedServices: CouponServiceWithDetails[];
  allowedProducts: CouponProductWithDetails[];
};

// Create package request type
export type CreatePackageRequest = {
  nameAr: string;
  imageUrl?: string | null;
  startDate: string; // YYYY-MM-DD
  endDate: string; // YYYY-MM-DD
  descriptionAr?: string | null;
  isActive?: boolean;
  services: {
    serviceId: number;
    quantity?: number;
    packagePrice: number;
  }[];
  products?: {
    productId: number;
    quantity?: number;
    packagePrice: number;
  }[];
};

// Create coupon request type
export type CreateCouponRequest = {
  nameAr: string;
  code: string;
  descriptionAr?: string | null;
  discountType: DiscountType;
  discountValue: number;
  appliesToServices?: boolean;
  appliesToProducts?: boolean;
  startDate: string; // YYYY-MM-DD
  endDate: string; // YYYY-MM-DD
  isActive?: boolean;
  serviceIds?: number[];
  productIds?: number[];
};

// Package list filters
export type PackageFilters = {
  isActive?: boolean;
  validToday?: boolean;
};

// Coupon list filters
export type CouponFilters = {
  isActive?: boolean;
  validToday?: boolean;
};

// ========================================
// 13) CAFETERIA MODULE (الكافتيريا)
// ========================================

// Cafeteria Products (منتجات الكافتيريا)
export const cafeteriaProducts = pgTable("cafeteria_products", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull(),
  sku: text("sku").unique(),
  unitId: integer("unit_id").references(() => units.id),
  defaultPurchasePrice: real("default_purchase_price"),
  defaultSellPrice: real("default_sell_price"),
  hasExpiryDate: boolean("has_expiry_date").default(false).notNull(),
  productType: text("product_type").default("both").notNull(), // "consume" (صرف), "sell" (بيع), "both" (كلاهما)
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Cafeteria Stock Batches
export const cafeteriaStockBatches = pgTable("cafeteria_stock_batches", {
  id: serial("id").primaryKey(),
  cafeteriaProductId: integer("cafeteria_product_id").references(() => cafeteriaProducts.id).notNull(),
  expiryDate: date("expiry_date"),
  quantityOnHand: real("quantity_on_hand").notNull(),
  unitCost: real("unit_cost").notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Cafeteria Stock Movements
export const cafeteriaStockMovements = pgTable("cafeteria_stock_movements", {
  id: serial("id").primaryKey(),
  cafeteriaProductId: integer("cafeteria_product_id").references(() => cafeteriaProducts.id).notNull(),
  movementDate: date("movement_date").notNull(),
  movementType: text("movement_type").notNull(), // "purchase", "sale", "consume", "damage", "adjustment"
  quantityIn: real("quantity_in").default(0).notNull(),
  quantityOut: real("quantity_out").default(0).notNull(),
  unitCost: real("unit_cost"),
  referenceType: text("reference_type"), // "CafeteriaPurchaseInvoice", "CafeteriaSaleOrder", "CafeteriaManualAdjustment"
  referenceId: integer("reference_id"),
  batchId: integer("batch_id").references(() => cafeteriaStockBatches.id),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

// Cafeteria Purchase Invoices
export const cafeteriaPurchaseInvoices = pgTable("cafeteria_purchase_invoices", {
  id: serial("id").primaryKey(),
  supplierId: integer("supplier_id").references(() => suppliers.id).notNull(),
  invoiceDate: date("invoice_date").notNull(),
  dueDate: date("due_date"),
  paymentStatus: text("payment_status").default("unpaid").notNull(), // "unpaid", "partially_paid", "paid"
  subtotalAmount: real("subtotal_amount").default(0).notNull(),
  vatAmount: real("vat_amount").default(0).notNull(),
  totalAmount: real("total_amount").default(0).notNull(),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Cafeteria Purchase Invoice Lines
export const cafeteriaPurchaseInvoiceLines = pgTable("cafeteria_purchase_invoice_lines", {
  id: serial("id").primaryKey(),
  cafeteriaPurchaseInvoiceId: integer("cafeteria_purchase_invoice_id").references(() => cafeteriaPurchaseInvoices.id).notNull(),
  cafeteriaProductId: integer("cafeteria_product_id").references(() => cafeteriaProducts.id).notNull(),
  quantity: real("quantity").notNull(),
  purchasePricePerUnit: real("purchase_price_per_unit").notNull(),
  vatType: text("vat_type").notNull(), // "inclusive", "exclusive", "exempt"
  vatRate: real("vat_rate").notNull(),
  vatAmount: real("vat_amount").notNull(),
  lineSubtotal: real("line_subtotal").notNull(),
  lineTotal: real("line_total").notNull(),
  expiryDate: date("expiry_date"),
});

// Cafeteria Purchase Payments
export const cafeteriaPurchasePayments = pgTable("cafeteria_purchase_payments", {
  id: serial("id").primaryKey(),
  cafeteriaPurchaseInvoiceId: integer("cafeteria_purchase_invoice_id").references(() => cafeteriaPurchaseInvoices.id),
  supplierId: integer("supplier_id").references(() => suppliers.id).notNull(),
  paymentMethodId: integer("payment_method_id").references(() => paymentMethods.id).notNull(),
  paymentDate: date("payment_date").notNull(),
  amount: real("amount").notNull(),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Cafeteria Sales Order Statuses
export const cafeteriaOrderStatuses = ["new", "completed", "canceled"] as const;
export type CafeteriaOrderStatus = typeof cafeteriaOrderStatuses[number];

// Cafeteria Sales Orders
export const cafeteriaSalesOrders = pgTable("cafeteria_sales_orders", {
  id: serial("id").primaryKey(),
  orderNumber: text("order_number").unique().notNull(),
  orderDate: timestamp("order_date").notNull(),
  source: text("source").default("pos").notNull(),
  clientId: integer("client_id").references(() => customers.id),
  createdByEmployeeId: integer("created_by_employee_id").references(() => employees.id).notNull(),
  branchId: integer("branch_id"),
  status: text("status").default("new").notNull(),
  notes: text("notes"),
  subtotalAmount: real("subtotal_amount").default(0).notNull(),
  vatAmount: real("vat_amount").default(0).notNull(),
  totalAmount: real("total_amount").default(0).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Cafeteria Sales Order Lines
export const cafeteriaSalesOrderLines = pgTable("cafeteria_sales_order_lines", {
  id: serial("id").primaryKey(),
  cafeteriaSalesOrderId: integer("cafeteria_sales_order_id").references(() => cafeteriaSalesOrders.id).notNull(),
  cafeteriaProductId: integer("cafeteria_product_id").references(() => cafeteriaProducts.id).notNull(),
  quantity: real("quantity").notNull(),
  unitPrice: real("unit_price").notNull(),
  vatType: text("vat_type").notNull(),
  vatRate: real("vat_rate").notNull(),
  lineSubtotal: real("line_subtotal").notNull(),
  vatAmount: real("vat_amount").notNull(),
  lineTotal: real("line_total").notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Insert schemas for Cafeteria
export const insertCafeteriaProductSchema = createInsertSchema(cafeteriaProducts).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  nameAr: z.string().min(1, "اسم المنتج مطلوب"),
  sku: z.string().optional().nullable(),
  unitId: z.number().int().positive("وحدة القياس مطلوبة").optional().nullable(),
  defaultPurchasePrice: z.number().min(0).optional().nullable(),
  defaultSellPrice: z.number().min(0).optional().nullable(),
  hasExpiryDate: z.boolean().optional().default(false),
  productType: z.enum(["consume", "sell", "both"]).optional().default("both"), // صرف، بيع، كلاهما
  isActive: z.boolean().optional().default(true),
});

export const insertCafeteriaStockBatchSchema = createInsertSchema(cafeteriaStockBatches).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertCafeteriaStockMovementSchema = createInsertSchema(cafeteriaStockMovements).omit({
  id: true,
  createdAt: true,
});

export const insertCafeteriaPurchaseInvoiceSchema = createInsertSchema(cafeteriaPurchaseInvoices).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  supplierId: z.number().int().positive("المورد مطلوب"),
  invoiceDate: z.string().min(1, "تاريخ الفاتورة مطلوب"),
  dueDate: z.string().optional().nullable(),
  notes: z.string().optional().nullable(),
});

export const insertCafeteriaPurchaseInvoiceLineSchema = createInsertSchema(cafeteriaPurchaseInvoiceLines).omit({
  id: true,
}).extend({
  cafeteriaPurchaseInvoiceId: z.number().int().positive().optional(),
  cafeteriaProductId: z.number().int().positive("المنتج مطلوب"),
  quantity: z.number().positive("الكمية يجب أن تكون أكبر من صفر"),
  purchasePricePerUnit: z.number().min(0, "سعر الشراء لا يقل عن صفر"),
  vatType: z.enum(vatTypes, { errorMap: () => ({ message: "نوع الضريبة غير صالح" }) }),
  vatRate: z.number().min(0).max(1),
  expiryDate: z.string().optional().nullable(),
});

export const insertCafeteriaPurchasePaymentSchema = createInsertSchema(cafeteriaPurchasePayments).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  cafeteriaPurchaseInvoiceId: z.number().int().positive().optional().nullable(),
  supplierId: z.number().int().positive("المورد مطلوب"),
  paymentMethodId: z.number().int().positive("طريقة الدفع مطلوبة"),
  paymentDate: z.string().min(1, "تاريخ الدفع مطلوب"),
  amount: z.number().positive("المبلغ يجب أن يكون أكبر من صفر"),
  notes: z.string().optional().nullable(),
});

export const insertCafeteriaSalesOrderSchema = createInsertSchema(cafeteriaSalesOrders).omit({
  id: true,
  orderNumber: true,
  subtotalAmount: true,
  vatAmount: true,
  totalAmount: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  orderDate: z.coerce.date({ errorMap: () => ({ message: "تاريخ الطلب مطلوب" }) }).optional(),
  source: z.string().default("pos"),
  clientId: z.number().int().positive().optional().nullable(),
  createdByEmployeeId: z.number().int().positive("الموظف المنشئ مطلوب"),
  branchId: z.number().int().positive().optional().nullable(),
  status: z.enum(cafeteriaOrderStatuses).optional().default("new"),
  notes: z.string().optional().nullable(),
});

export const insertCafeteriaSalesOrderLineSchema = createInsertSchema(cafeteriaSalesOrderLines).omit({
  id: true,
  lineSubtotal: true,
  vatAmount: true,
  lineTotal: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  cafeteriaSalesOrderId: z.number().int().positive().optional(),
  cafeteriaProductId: z.number().int().positive("المنتج مطلوب"),
  quantity: z.number().positive("الكمية يجب أن تكون أكبر من صفر"),
  unitPrice: z.number().min(0, "السعر لا يقل عن صفر").optional(),
  vatType: z.enum(vatTypes).optional(),
  vatRate: z.number().min(0).max(1).optional(),
});

export const updateCafeteriaSalesOrderSchema = z.object({
  status: z.enum(cafeteriaOrderStatuses, { errorMap: () => ({ message: "حالة الطلب غير صالحة" }) }).optional(),
  notes: z.string().optional().nullable(),
});

// Types for Cafeteria
export type InsertCafeteriaProduct = z.infer<typeof insertCafeteriaProductSchema>;
export type CafeteriaProduct = typeof cafeteriaProducts.$inferSelect;
export type InsertCafeteriaStockBatch = z.infer<typeof insertCafeteriaStockBatchSchema>;
export type CafeteriaStockBatch = typeof cafeteriaStockBatches.$inferSelect;
export type InsertCafeteriaStockMovement = z.infer<typeof insertCafeteriaStockMovementSchema>;
export type CafeteriaStockMovement = typeof cafeteriaStockMovements.$inferSelect;
export type InsertCafeteriaPurchaseInvoice = z.infer<typeof insertCafeteriaPurchaseInvoiceSchema>;
export type CafeteriaPurchaseInvoice = typeof cafeteriaPurchaseInvoices.$inferSelect;
export type InsertCafeteriaPurchaseInvoiceLine = z.infer<typeof insertCafeteriaPurchaseInvoiceLineSchema>;
export type CafeteriaPurchaseInvoiceLine = typeof cafeteriaPurchaseInvoiceLines.$inferSelect;
export type InsertCafeteriaPurchasePayment = z.infer<typeof insertCafeteriaPurchasePaymentSchema>;
export type CafeteriaPurchasePayment = typeof cafeteriaPurchasePayments.$inferSelect;
export type InsertCafeteriaSalesOrder = z.infer<typeof insertCafeteriaSalesOrderSchema>;
export type CafeteriaSalesOrder = typeof cafeteriaSalesOrders.$inferSelect;
export type InsertCafeteriaSalesOrderLine = z.infer<typeof insertCafeteriaSalesOrderLineSchema>;
export type CafeteriaSalesOrderLine = typeof cafeteriaSalesOrderLines.$inferSelect;

// Extended types for Cafeteria with relations
export type CafeteriaProductWithUnit = CafeteriaProduct & {
  unit?: Unit;
};

export type CafeteriaPurchaseInvoiceLineWithProduct = CafeteriaPurchaseInvoiceLine & {
  cafeteriaProduct?: CafeteriaProduct;
};

export type CafeteriaPurchaseInvoiceWithDetails = CafeteriaPurchaseInvoice & {
  supplier?: Supplier;
  lines: CafeteriaPurchaseInvoiceLineWithProduct[];
  payments: (CafeteriaPurchasePayment & { paymentMethod?: PaymentMethod })[];
};

export type CafeteriaSalesOrderLineWithProduct = CafeteriaSalesOrderLine & {
  cafeteriaProduct?: CafeteriaProduct;
};

export type CafeteriaSalesOrderWithDetails = CafeteriaSalesOrder & {
  client?: Customer;
  createdByEmployee?: Employee;
  lines: CafeteriaSalesOrderLineWithProduct[];
};

// Cafeteria Inventory Summary Item
export type CafeteriaInventorySummaryItem = {
  cafeteriaProductId: number;
  productNameAr: string;
  sku?: string;
  quantityOnHand: number;
  totalCost: number;
  unitNameAr?: string;
};

// Cafeteria Stock Movement Report Item
export type CafeteriaStockMovementReportItem = {
  id: number;
  movementDate: string;
  movementType: string;
  quantityIn: number;
  quantityOut: number;
  unitCost?: number;
  referenceType?: string;
  referenceId?: number;
  balance?: number;
};

// Create Cafeteria Purchase Invoice Request
export type CreateCafeteriaPurchaseInvoiceRequest = {
  supplierId: number;
  invoiceDate: string;
  dueDate?: string;
  lines: {
    cafeteriaProductId: number;
    quantity: number;
    purchasePricePerUnit: number;
    vatType: "inclusive" | "exclusive" | "exempt";
    vatRate: number;
    expiryDate?: string;
  }[];
  notes?: string;
};

// Create Cafeteria Sales Order Request
export type CreateCafeteriaSalesOrderRequest = {
  clientId?: number | null;
  createdByEmployeeId: number;
  branchId?: number | null;
  notes?: string | null;
  lines: {
    cafeteriaProductId: number;
    quantity: number;
  }[];
};

// Cafeteria Order Filters
export type CafeteriaOrderFilters = {
  date?: string;
  status?: CafeteriaOrderStatus;
  branchId?: number;
  createdByEmployeeId?: number;
};

// Cafeteria Employee Custodies (عهد الموظفين - الكافتيريا)
export const cafeteriaCustodyStatuses = ["open", "returned", "consumed", "partially_returned", "lost", "damaged"] as const;
export type CafeteriaCustodyStatus = typeof cafeteriaCustodyStatuses[number];

export const cafeteriaEmployeeCustodies = pgTable("cafeteria_employee_custodies", {
  id: serial("id").primaryKey(),
  employeeId: integer("employee_id").references(() => employees.id).notNull(),
  cafeteriaProductId: integer("cafeteria_product_id").references(() => cafeteriaProducts.id).notNull(),
  quantity: real("quantity").notNull(),
  unitCost: real("unit_cost").notNull(),
  totalCost: real("total_cost").notNull(),
  custodyDate: date("custody_date").notNull(),
  createdByEmployeeId: integer("created_by_employee_id").references(() => employees.id),
  status: text("status").default("open").notNull(),
  notes: text("notes"),
  closedDate: date("closed_date"),
  closedByEmployeeId: integer("closed_by_employee_id").references(() => employees.id),
  closureReason: text("closure_reason"),
  returnedQuantity: real("returned_quantity"),
  closureNotes: text("closure_notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertCafeteriaEmployeeCustodySchema = createInsertSchema(cafeteriaEmployeeCustodies).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  employeeId: z.number().int().positive("الموظف مطلوب"),
  cafeteriaProductId: z.number().int().positive("المنتج مطلوب"),
  quantity: z.number().positive("الكمية يجب أن تكون أكبر من صفر"),
  unitCost: z.number().min(0, "تكلفة الوحدة لا تقل عن صفر"),
  totalCost: z.number().min(0, "إجمالي التكلفة لا يقل عن صفر"),
  custodyDate: z.string().min(1, "تاريخ العهدة مطلوب"),
  createdByEmployeeId: z.number().int().positive().optional().nullable(),
  status: z.enum(cafeteriaCustodyStatuses).default("open"),
  notes: z.string().optional().nullable(),
  closedDate: z.string().optional().nullable(),
  closedByEmployeeId: z.number().int().positive().optional().nullable(),
  closureReason: z.string().optional().nullable(),
  returnedQuantity: z.number().min(0).optional().nullable(),
  closureNotes: z.string().optional().nullable(),
});

export const updateCafeteriaEmployeeCustodySchema = insertCafeteriaEmployeeCustodySchema.partial();

export type InsertCafeteriaEmployeeCustody = z.infer<typeof insertCafeteriaEmployeeCustodySchema>;
export type CafeteriaEmployeeCustody = typeof cafeteriaEmployeeCustodies.$inferSelect;

export type CafeteriaEmployeeCustodyWithDetails = CafeteriaEmployeeCustody & {
  employee?: Employee;
  cafeteriaProduct?: CafeteriaProduct;
  createdByEmployee?: Employee;
  closedByEmployee?: Employee;
};

// Cafeteria Sales Returns (مرتجعات مبيعات الكافتيريا - إشعار دائن ZATCA)
export const cafeteriaSalesReturns = pgTable("cafeteria_sales_returns", {
  id: serial("id").primaryKey(),
  cafeteriaSalesOrderId: integer("cafeteria_sales_order_id").references(() => cafeteriaSalesOrders.id).notNull(),
  returnNumber: text("return_number").unique().notNull(), // e.g. "CF-RET-0001"
  creditNoteNumber: text("credit_note_number").unique().notNull(), // رقم إشعار الدائن e.g. "CF-CN-0001"
  returnDate: timestamp("return_date").notNull(),
  processedByEmployeeId: integer("processed_by_employee_id").references(() => employees.id),
  reason: text("reason").notNull(), // سبب المرتجع
  returnType: text("return_type").notNull(), // "full" | "partial" | "cancellation"
  originalInvoiceNumber: text("original_invoice_number"), // رقم الفاتورة الأصلية
  subtotalAmount: real("subtotal_amount").default(0).notNull(),
  vatAmount: real("vat_amount").default(0).notNull(),
  refundAmount: real("refund_amount").default(0).notNull(),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Cafeteria Sales Return Lines (خطوط مرتجعات الكافتيريا)
export const cafeteriaSalesReturnLines = pgTable("cafeteria_sales_return_lines", {
  id: serial("id").primaryKey(),
  returnId: integer("return_id").references(() => cafeteriaSalesReturns.id).notNull(),
  cafeteriaSalesOrderLineId: integer("cafeteria_sales_order_line_id").references(() => cafeteriaSalesOrderLines.id).notNull(),
  quantity: real("quantity").notNull(),
  refundAmount: real("refund_amount").notNull(),
  reason: text("reason"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertCafeteriaSalesReturnSchema = createInsertSchema(cafeteriaSalesReturns).omit({
  id: true,
  returnNumber: true,
  creditNoteNumber: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  cafeteriaSalesOrderId: z.number().int().positive("الطلب مطلوب"),
  returnDate: z.coerce.date({ errorMap: () => ({ message: "تاريخ المرتجع مطلوب" }) }),
  processedByEmployeeId: z.number().int().positive().optional().nullable(),
  reason: z.string().min(1, "سبب المرتجع مطلوب"),
  returnType: z.enum(["full", "partial", "cancellation"], { errorMap: () => ({ message: "نوع المرتجع غير صالح" }) }),
  originalInvoiceNumber: z.string().optional().nullable(),
  subtotalAmount: z.number().min(0).optional().default(0),
  vatAmount: z.number().min(0).optional().default(0),
  refundAmount: z.number().min(0).optional().default(0),
  notes: z.string().optional().nullable(),
});

export const insertCafeteriaSalesReturnLineSchema = createInsertSchema(cafeteriaSalesReturnLines).omit({
  id: true,
  createdAt: true,
}).extend({
  returnId: z.number().int().positive().optional(),
  cafeteriaSalesOrderLineId: z.number().int().positive("خط الطلب مطلوب"),
  quantity: z.number().positive("الكمية يجب أن تكون أكبر من صفر"),
  refundAmount: z.number().min(0, "قيمة الاسترداد لا تقل عن صفر"),
  reason: z.string().optional().nullable(),
});

export type InsertCafeteriaSalesReturn = z.infer<typeof insertCafeteriaSalesReturnSchema>;
export type CafeteriaSalesReturn = typeof cafeteriaSalesReturns.$inferSelect;
export type InsertCafeteriaSalesReturnLine = z.infer<typeof insertCafeteriaSalesReturnLineSchema>;
export type CafeteriaSalesReturnLine = typeof cafeteriaSalesReturnLines.$inferSelect;

export type CafeteriaSalesReturnWithDetails = CafeteriaSalesReturn & {
  cafeteriaSalesOrder?: CafeteriaSalesOrder;
  processedByEmployee?: Employee;
  lines?: (CafeteriaSalesReturnLine & {
    orderLine?: CafeteriaSalesOrderLine;
    cafeteriaProduct?: CafeteriaProduct;
  })[];
};

// ========================================
// 14) EXPENSES MODULE (النفقات)
// ========================================

// Expense Types (أنواع النفقات)
export const expenseTypes = pgTable("expense_types", {
  id: serial("id").primaryKey(),
  nameAr: text("name_ar").notNull().unique(),
  descriptionAr: text("description_ar"),
  isFixedAmount: boolean("is_fixed_amount").default(false).notNull(),
  fixedAmount: real("fixed_amount"),
  accountId: integer("account_id").references(() => accounts.id), // رابط حساب دفتر الأستاذ للمصروفات
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Expenses (النفقات)
export const expenses = pgTable("expenses", {
  id: serial("id").primaryKey(),
  expenseTypeId: integer("expense_type_id").references(() => expenseTypes.id).notNull(),
  supplierId: integer("supplier_id").references(() => suppliers.id),
  invoiceReference: text("invoice_reference"),
  descriptionAr: text("description_ar"),
  amountWithoutVat: real("amount_without_vat").notNull(),
  vatType: text("vat_type").notNull(), // "inclusive", "exclusive", "exempt"
  vatRate: real("vat_rate").notNull(),
  vatAmount: real("vat_amount").default(0).notNull(),
  totalAmount: real("total_amount").notNull(),
  paymentMethodId: integer("payment_method_id").references(() => paymentMethods.id),
  paymentDate: date("payment_date").notNull(),
  isPaid: boolean("is_paid").default(true).notNull(),
  attachmentUrl: text("attachment_url"),
  createdByEmployeeId: integer("created_by_employee_id").references(() => employees.id),
  journalEntryId: integer("journal_entry_id"), // رابط القيد المحاسبي المُنشأ تلقائياً
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Insert schemas
export const insertExpenseTypeSchema = createInsertSchema(expenseTypes).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertExpenseSchema = createInsertSchema(expenses).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

// Types
export type InsertExpenseType = z.infer<typeof insertExpenseTypeSchema>;
export type ExpenseType = typeof expenseTypes.$inferSelect;
export type InsertExpense = z.infer<typeof insertExpenseSchema>;
export type Expense = typeof expenses.$inferSelect;

// Extended types
export type ExpenseWithDetails = Expense & {
  expenseType?: ExpenseType;
  supplier?: Supplier;
  paymentMethod?: PaymentMethod;
  createdByEmployee?: Employee;
};

// Expense Filters
export type ExpenseFilters = {
  fromDate?: string;
  toDate?: string;
  expenseTypeId?: number;
  supplierId?: number;
  paymentMethodId?: number;
};

// Expense Summary Response
export type ExpensesSummary = {
  totalExpensesWithoutVat: number;
  totalVat: number;
  totalExpensesWithVat: number;
  byType: {
    expenseTypeId: number;
    expenseTypeNameAr: string;
    totalAmount: number;
    count: number;
  }[];
  byMonth: {
    year: number;
    month: number;
    totalAmount: number;
  }[];
};

// Payables Summary Response
export type PayablesSummary = {
  globalSummary: {
    totalOutstanding: number;
    totalOverdue: number;
    agingBuckets: {
      "0_30": number;
      "31_60": number;
      "61_90": number;
      "90_plus": number;
    };
  };
  suppliers: {
    supplierId: number;
    supplierNameAr: string;
    totalOutstanding: number;
    overdueAmount: number;
    invoiceCountUnpaid: number;
    invoiceCountOverdue: number;
  }[];
};

// ========================================
// DRIVERS MODULE (السائقون)
// ========================================

export const drivers = pgTable("drivers", {
  id: serial("id").primaryKey(),
  // Basic Info
  nameAr: text("name_ar"),
  code: text("code").unique(),
  phone: text("phone"),
  // HR columns similar to employees
  fullNameAr: text("full_name_ar"),
  phoneCountryCode: text("phone_country_code").default("966"),
  phoneNumber: text("phone_number"),
  email: text("email").unique(),
  fingerprintEmployeeNo: text("fingerprint_employee_no"),
  profileImageUrl: text("profile_image_url"),
  jobTitleAr: text("job_title_ar"),
  cityNameAr: text("city_name_ar"),
  branchId: integer("branch_id"),
  nationalityNameAr: text("nationality_name_ar"),
  nationalityId: integer("nationality_id"),
  // HR data
  birthDate: date("birth_date"),
  hireDate: date("hire_date"),
  iqamaNumber: text("iqama_number"),
  iqamaExpiryDate: date("iqama_expiry_date"),
  insuranceCompany: text("insurance_company"),
  insuranceNumber: text("insurance_number"),
  insuranceExpiryDate: date("insurance_expiry_date"),
  insuranceCardExpiryDate: date("insurance_card_expiry_date"),
  healthNumber: text("health_number"),
  passportNumber: text("passport_number"),
  passportExpiryDate: date("passport_expiry_date"),
  // Driver-specific fields
  licenseNumber: text("license_number"),
  licenseExpiryDate: date("license_expiry_date"),
  vehiclePlateNumber: text("vehicle_plate_number"),
  vehicleType: text("vehicle_type"),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const trips = pgTable("trips", {
  id: serial("id").primaryKey(),
  tripNumber: text("trip_number").unique().notNull(), // e.g. "TR-0001"
  driverId: integer("driver_id").references(() => drivers.id).notNull(),
  tripType: text("trip_type").notNull(), // "appointment" or "general"
  appointmentId: integer("appointment_id").references(() => appointments.id), // Only when tripType = "appointment"
  // Location fields for general trips
  startLocationLat: real("start_location_lat"),
  startLocationLng: real("start_location_lng"),
  startAddressText: text("start_address_text"),
  endLocationLat: real("end_location_lat"),
  endLocationLng: real("end_location_lng"),
  endAddressText: text("end_address_text"),
  // Trip info
  status: text("status").notNull().default("pending"), // "pending", "in_progress", "completed", "cancelled"
  notes: text("notes"),
  scheduledAt: timestamp("scheduled_at"),
  startedAt: timestamp("started_at"),
  completedAt: timestamp("completed_at"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertDriverSchema = createInsertSchema(drivers).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertTripSchema = createInsertSchema(trips).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type InsertDriver = z.infer<typeof insertDriverSchema>;
export type Driver = typeof drivers.$inferSelect;
export type InsertTrip = z.infer<typeof insertTripSchema>;
export type Trip = typeof trips.$inferSelect;

// Extended Trip type with relations
export type TripWithDetails = Trip & {
  driver?: Driver;
  appointment?: Appointment;
};

// Supplier Statement Response
export type SupplierStatementTransaction = {
  date: string;
  type: "invoice" | "payment" | "adjustment";
  referenceNumber: string;
  descriptionAr: string;
  debitAmount: number;
  creditAmount: number;
  balanceAfter: number;
};

export type SupplierStatement = {
  supplier: {
    id: number;
    nameAr: string;
    agreementText?: string | null;
    paymentTermDays?: number | null;
  };
  openingBalance: {
    amount: number;
    direction: "debit" | "credit";
  };
  transactions: SupplierStatementTransaction[];
  closingBalance: {
    amount: number;
    explanation: string;
  };
};

// ========================================
// CASH DISBURSEMENTS MODULE (صرف نقدي)
// ========================================

export const cashDisbursementTypes = ["advance", "gift", "deduction", "owner_withdrawal"] as const;
export type CashDisbursementType = typeof cashDisbursementTypes[number];

export const installmentStatuses = ["pending", "deducted", "cancelled"] as const;
export type InstallmentStatus = typeof installmentStatuses[number];

export const cashDisbursements = pgTable("cash_disbursements", {
  id: serial("id").primaryKey(),
  disbursementNumber: text("disbursement_number").unique().notNull(), // e.g. "CD-0001"
  disbursementType: text("disbursement_type").notNull(), // "advance", "gift", "deduction", "owner_withdrawal"
  // Employee or Driver (for advance, gift, deduction types only)
  recipientType: text("recipient_type"), // "employee" or "driver" (null for owner_withdrawal)
  employeeId: integer("employee_id").references(() => employees.id),
  driverId: integer("driver_id").references(() => drivers.id),
  // Amount and reason
  amount: real("amount").notNull(),
  reason: text("reason"),
  // Payment method
  paymentMethodId: integer("payment_method_id").references(() => paymentMethods.id).notNull(),
  disbursementDate: date("disbursement_date").notNull(),
  // Installment info (for advances only)
  isInstallment: boolean("is_installment").default(false).notNull(),
  installmentMonths: integer("installment_months"),
  installmentAmount: real("installment_amount"), // amount / installmentMonths
  // Journal entry reference
  journalEntryId: integer("journal_entry_id"),
  // Metadata
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Track installments for advances with payment schedules
export const advanceInstallments = pgTable("advance_installments", {
  id: serial("id").primaryKey(),
  cashDisbursementId: integer("cash_disbursement_id").references(() => cashDisbursements.id).notNull(),
  installmentNumber: integer("installment_number").notNull(),
  amount: real("amount").notNull(),
  dueDate: date("due_date").notNull(),
  status: text("status").default("pending").notNull(), // "pending", "deducted", "cancelled"
  deductedAt: timestamp("deducted_at"), // When it was deducted from salary
  payrollLineId: integer("payroll_line_id"), // Link to payroll line when deducted
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertCashDisbursementSchema = createInsertSchema(cashDisbursements).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  disbursementType: z.enum(cashDisbursementTypes),
  recipientType: z.enum(["employee", "driver"]).optional().nullable(),
  amount: z.number().positive("المبلغ يجب أن يكون أكبر من صفر"),
  paymentMethodId: z.number().int().positive("طريقة الدفع مطلوبة"),
  disbursementDate: z.string().min(1, "تاريخ الصرف مطلوب"),
  isInstallment: z.boolean().optional().default(false),
  installmentMonths: z.number().int().min(1).optional().nullable(),
});

export const insertAdvanceInstallmentSchema = createInsertSchema(advanceInstallments).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type InsertCashDisbursement = z.infer<typeof insertCashDisbursementSchema>;
export type CashDisbursement = typeof cashDisbursements.$inferSelect;
export type InsertAdvanceInstallment = z.infer<typeof insertAdvanceInstallmentSchema>;
export type AdvanceInstallment = typeof advanceInstallments.$inferSelect;

// Extended types with relations
export type CashDisbursementWithDetails = CashDisbursement & {
  employee?: Employee;
  driver?: Driver;
  paymentMethod?: PaymentMethod;
  installments?: AdvanceInstallment[];
};

export type AdvanceInstallmentWithDetails = AdvanceInstallment & {
  cashDisbursement?: CashDisbursement;
};

// ========================================
// 18) DAILY REPORT (تقرير اليومية)
// ========================================

// Revenue line item (for each category)
export type DailyReportRevenueLine = {
  label: string;
  subtotal: number; // قبل الضريبة
  vat: number; // الضريبة
  total: number; // الإجمالي
};

// Payment method breakdown
export type DailyReportPaymentMethodLine = {
  paymentMethodId: number;
  paymentMethodName: string;
  revenue: number; // إيرادات
  refunds: number; // مسترجعات
  net: number; // صافي
};

// Cash summary
export type DailyReportCashSummary = {
  todaysRevenue: number;
  previousBalance: number;
  netCash: number;
};

// Daily Report Summary
export type SalonDailyReport = {
  date: string; // YYYY-MM-DD
  
  // Revenue section (قسم الإيرادات)
  inSalonServicesRevenue: DailyReportRevenueLine; // إيرادات الخدمات داخل الصالون
  homeServicesRevenue: DailyReportRevenueLine; // إيرادات الخدمات المنزلية
  returnedServicesAmount: DailyReportRevenueLine; // مبالغ الخدمات المسترجعة
  productSalesRevenue: DailyReportRevenueLine; // إيرادات مبيعات المنتجات
  returnedProductsAmount: DailyReportRevenueLine; // مبالغ المنتجات المسترجعة
  
  // Revenue totals (إجماليات الإيرادات)
  totalRevenueBeforeVat: number; // المجموع النهائي للإيرادات قبل الضريبة
  totalVat: number; // المجموع النهائي للضريبة
  grandTotal: number; // الإجمالي النهائي
  
  // Payment method breakdown (تفصيل الإيرادات حسب طرق الدفع)
  paymentMethodBreakdown: DailyReportPaymentMethodLine[];
  totalPaymentsReceived: number; // إجمالي الإيرادات المدفوعة
  totalRefunds: number; // المبالغ المسترجعة
  netPayments: number; // الصافي النهائي
  
  // Loyalty points
  loyaltyPointsRedemption: number; // إيرادات نقاط الولاء
  
  // Cash summary (صافي النقدي)
  cashSummary: DailyReportCashSummary;
};

// Request filters for daily report
export type DailyReportFilters = {
  date: string; // YYYY-MM-DD
  branchId?: number;
};

// ========================================
// 19) MARKETING REPORTS (تقارير التسويق)
// ========================================

// Coupon usage statistics
export type CouponUsageStat = {
  couponId: number;
  couponCode: string;
  couponNameAr: string;
  discountType: string;
  discountValue: number;
  usageCount: number;
  totalDiscountGiven: number;
  totalOrdersValue: number;
  avgOrderValue: number;
  lastUsedDate: string | null;
};

// Package usage statistics
export type PackageUsageStat = {
  packageId: number;
  packageNameAr: string;
  totalSales: number;
  totalRevenue: number;
  avgOrderValue: number;
  servicesCount: number;
  productsCount: number;
  lastSoldDate: string | null;
};

// Top services in packages
export type PackageTopService = {
  serviceId: number;
  serviceNameAr: string;
  packageCount: number;
  totalQuantitySold: number;
  totalRevenue: number;
};

// Top products in packages
export type PackageTopProduct = {
  productId: number;
  productNameAr: string;
  packageCount: number;
  totalQuantitySold: number;
  totalRevenue: number;
};

// Marketing report summary
export type MarketingReportSummary = {
  // Coupon summary
  couponStats: {
    totalCoupons: number;
    activeCoupons: number;
    totalUsageCount: number;
    totalDiscountGiven: number;
  };
  couponUsageList: CouponUsageStat[];
  
  // Package summary
  packageStats: {
    totalPackages: number;
    activePackages: number;
    totalPackagesSold: number;
    totalPackageRevenue: number;
  };
  packageUsageList: PackageUsageStat[];
  topServicesInPackages: PackageTopService[];
  topProductsInPackages: PackageTopProduct[];
};

// Marketing report filters
export type MarketingReportFilters = {
  fromDate?: string; // YYYY-MM-DD
  toDate?: string; // YYYY-MM-DD
};

// ========================================
// 20) COMMISSION MODULE (نظام العمولات)
// ========================================

// Commission Target Source Types
export const commissionTargetSourceTypes = [
  "executed_services",     // الخدمات المنفذة
  "created_orders",        // طلبات الصالون المُنشأة
  "cafeteria_sales",       // مبيعات الكافتريا
  "product_sales",         // مبيعات المنتجات
] as const;
export type CommissionTargetSourceType = typeof commissionTargetSourceTypes[number];

// Commission Transaction Types
export const commissionTransactionTypes = [
  "accrual",    // استحقاق
  "payment",    // سداد
  "adjustment", // تعديل
] as const;
export type CommissionTransactionType = typeof commissionTransactionTypes[number];

// Commission Profile (ملف عمولات الموظف)
export const commissionProfiles = pgTable("commission_profiles", {
  id: serial("id").primaryKey(),
  employeeId: integer("employee_id").references(() => employees.id).notNull().unique(),
  monthlyTarget: real("monthly_target").default(0).notNull(), // التارجت الشهري
  commissionRate: real("commission_rate").default(0).notNull(), // نسبة العمولة (0-100)
  isActive: boolean("is_active").default(true).notNull(),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Commission Target Sources (مصادر التارجت لكل موظف)
export const commissionTargetSources = pgTable("commission_target_sources", {
  id: serial("id").primaryKey(),
  profileId: integer("profile_id").references(() => commissionProfiles.id).notNull(),
  sourceType: text("source_type").notNull(), // from commissionTargetSourceTypes
  isEnabled: boolean("is_enabled").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

// Commission Transactions (معاملات العمولات)
export const commissionTransactions = pgTable("commission_transactions", {
  id: serial("id").primaryKey(),
  profileId: integer("profile_id").references(() => commissionProfiles.id).notNull(),
  transactionType: text("transaction_type").notNull(), // accrual, payment, adjustment
  amount: real("amount").notNull(), // positive for accrual, negative for payment
  periodMonth: integer("period_month").notNull(), // 1-12
  periodYear: integer("period_year").notNull(),
  description: text("description"),
  referenceType: text("reference_type"), // order, appointment, cafeteria, etc.
  referenceId: integer("reference_id"),
  processedByEmployeeId: integer("processed_by_employee_id").references(() => employees.id),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

// Commission Performance Snapshots (لقطات الأداء الشهري)
export const commissionPerformanceSnapshots = pgTable("commission_performance_snapshots", {
  id: serial("id").primaryKey(),
  profileId: integer("profile_id").references(() => commissionProfiles.id).notNull(),
  periodMonth: integer("period_month").notNull(), // 1-12
  periodYear: integer("period_year").notNull(),
  // Achievement breakdown
  executedServicesAmount: real("executed_services_amount").default(0).notNull(),
  inventoryDeductionAmount: real("inventory_deduction_amount").default(0).notNull(), // صرف مخزني
  createdOrdersAmount: real("created_orders_amount").default(0).notNull(),
  ordersReturnsAmount: real("orders_returns_amount").default(0).notNull(),
  cafeteriaSalesAmount: real("cafeteria_sales_amount").default(0).notNull(),
  cafeteriaReturnsAmount: real("cafeteria_returns_amount").default(0).notNull(),
  productSalesAmount: real("product_sales_amount").default(0).notNull(),
  productReturnsAmount: real("product_returns_amount").default(0).notNull(),
  // Totals
  grossAmount: real("gross_amount").default(0).notNull(), // إجمالي قبل الخصومات
  deductionsAmount: real("deductions_amount").default(0).notNull(), // إجمالي الخصومات
  netAmount: real("net_amount").default(0).notNull(), // صافي التارجت المحقق
  targetAmount: real("target_amount").default(0).notNull(), // التارجت الشهري المطلوب
  achievementRate: real("achievement_rate").default(0).notNull(), // نسبة الإنجاز
  commissionRate: real("commission_rate").default(0).notNull(), // نسبة العمولة
  commissionAmount: real("commission_amount").default(0).notNull(), // قيمة العمولة المستحقة
  isPaid: boolean("is_paid").default(false).notNull(),
  paidAt: timestamp("paid_at"),
  paidByEmployeeId: integer("paid_by_employee_id").references(() => employees.id),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

// Insert schemas for commission
export const insertCommissionProfileSchema = createInsertSchema(commissionProfiles).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  employeeId: z.number().int().positive("الموظف مطلوب"),
  monthlyTarget: z.number().min(0, "التارجت يجب أن يكون صفر أو أكثر"),
  commissionRate: z.number().min(0, "نسبة العمولة يجب أن تكون صفر أو أكثر").max(100, "نسبة العمولة لا تتجاوز 100%"),
  notes: z.string().optional().nullable(),
});

export const insertCommissionTargetSourceSchema = createInsertSchema(commissionTargetSources).omit({
  id: true,
  createdAt: true,
}).extend({
  profileId: z.number().int().positive("ملف العمولات مطلوب"),
  sourceType: z.enum(commissionTargetSourceTypes, { errorMap: () => ({ message: "نوع المصدر غير صالح" }) }),
  isEnabled: z.boolean().optional().default(true),
});

export const insertCommissionTransactionSchema = createInsertSchema(commissionTransactions).omit({
  id: true,
  createdAt: true,
}).extend({
  profileId: z.number().int().positive("ملف العمولات مطلوب"),
  transactionType: z.enum(commissionTransactionTypes, { errorMap: () => ({ message: "نوع المعاملة غير صالح" }) }),
  amount: z.number(),
  periodMonth: z.number().int().min(1).max(12),
  periodYear: z.number().int().min(2020),
  description: z.string().optional().nullable(),
  referenceType: z.string().optional().nullable(),
  referenceId: z.number().int().positive().optional().nullable(),
  processedByEmployeeId: z.number().int().positive().optional().nullable(),
});

// Types for commission
export type InsertCommissionProfile = z.infer<typeof insertCommissionProfileSchema>;
export type CommissionProfile = typeof commissionProfiles.$inferSelect;
export type InsertCommissionTargetSource = z.infer<typeof insertCommissionTargetSourceSchema>;
export type CommissionTargetSource = typeof commissionTargetSources.$inferSelect;
export type InsertCommissionTransaction = z.infer<typeof insertCommissionTransactionSchema>;
export type CommissionTransaction = typeof commissionTransactions.$inferSelect;
export type CommissionPerformanceSnapshot = typeof commissionPerformanceSnapshots.$inferSelect;

// Extended types
export type CommissionProfileWithSources = CommissionProfile & {
  employee?: Employee;
  targetSources: CommissionTargetSource[];
};

// Employee Services/Orders Activity Report
export type EmployeeServiceActivity = {
  id: number;
  date: string;
  type: "order" | "appointment" | "return";
  referenceNumber: string;
  referenceId: number;
  serviceName: string;
  serviceId: number;
  quantity: number;
  amount: number; // lineTotal for services
  inventoryDeduction: number; // صرف مخزني للخدمة
  netAmount: number; // amount - inventoryDeduction
  status: string;
  completedAt: string | null;
};

export type EmployeeOrderActivity = {
  id: number;
  date: string;
  type: "order" | "appointment";
  referenceNumber: string;
  referenceId: number;
  totalAmount: number;
  returnsAmount: number;
  netAmount: number;
  status: string;
};

export type EmployeeCommissionReport = {
  employeeId: number;
  employeeName: string;
  periodMonth: number;
  periodYear: number;
  profile: CommissionProfileWithSources | null;
  // Activity details
  executedServices: EmployeeServiceActivity[];
  createdOrders: EmployeeOrderActivity[];
  cafeteriaSales: {
    orderId: number;
    orderNumber: string;
    date: string;
    totalAmount: number;
    status: string;
  }[];
  productSales: {
    orderId: number;
    orderNumber: string;
    date: string;
    productName: string;
    quantity: number;
    amount: number;
    returnedAmount: number;
  }[];
  // Totals
  summary: {
    executedServicesTotal: number;
    inventoryDeductionTotal: number;
    createdOrdersTotal: number;
    ordersReturnsTotal: number;
    cafeteriaSalesTotal: number;
    cafeteriaReturnsTotal: number;
    productSalesTotal: number;
    productReturnsTotal: number;
    grossTotal: number;
    deductionsTotal: number;
    netTotal: number;
    targetAmount: number;
    achievementRate: number;
    commissionRate: number;
    commissionAmount: number;
  };
};

// ========================================
// INVENTORY WITHDRAWAL ORDERS (أوامر سحب المخزون)
// ========================================

export const inventoryWithdrawalReasons = [
  "internal_use",      // استخدام داخلي
  "damage",            // تلف
  "expiry",            // انتهاء صلاحية
  "donation",          // تبرع
  "other",             // أخرى
] as const;

export const inventoryWithdrawalStatuses = [
  "draft",      // مسودة
  "confirmed",  // مؤكد
  "cancelled",  // ملغي
] as const;

export const inventoryWithdrawalOrders = pgTable("inventory_withdrawal_orders", {
  id: serial("id").primaryKey(),
  orderNumber: text("order_number").unique(),
  orderDate: date("order_date").notNull(),
  reason: text("reason").notNull(), // from inventoryWithdrawalReasons
  status: text("status").default("draft").notNull(), // from inventoryWithdrawalStatuses
  responsibleEmployeeId: integer("responsible_employee_id").references(() => employees.id),
  notes: text("notes"),
  journalEntryId: integer("journal_entry_id").references(() => journalEntries.id),
  totalCost: real("total_cost").default(0).notNull(),
  confirmedAt: timestamp("confirmed_at"),
  confirmedByEmployeeId: integer("confirmed_by_employee_id").references(() => employees.id),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const inventoryWithdrawalLines = pgTable("inventory_withdrawal_lines", {
  id: serial("id").primaryKey(),
  orderId: integer("order_id").references(() => inventoryWithdrawalOrders.id).notNull(),
  productId: integer("product_id").references(() => products.id).notNull(),
  quantity: real("quantity").notNull(),
  unitCost: real("unit_cost").default(0).notNull(), // from product average cost
  totalCost: real("total_cost").default(0).notNull(),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertInventoryWithdrawalOrderSchema = createInsertSchema(inventoryWithdrawalOrders).omit({
  id: true,
  orderNumber: true,
  totalCost: true,
  confirmedAt: true,
  confirmedByEmployeeId: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  orderDate: z.string().min(1, "تاريخ الأمر مطلوب"),
  reason: z.enum(inventoryWithdrawalReasons, { errorMap: () => ({ message: "سبب السحب مطلوب" }) }),
  responsibleEmployeeId: z.number().int().positive("الموظف المسؤول مطلوب").optional().nullable(),
  notes: z.string().optional().nullable(),
});

export const insertInventoryWithdrawalLineSchema = createInsertSchema(inventoryWithdrawalLines).omit({
  id: true,
  unitCost: true,
  totalCost: true,
  createdAt: true,
}).extend({
  orderId: z.number().int().positive("أمر السحب مطلوب"),
  productId: z.number().int().positive("المنتج مطلوب"),
  quantity: z.number().positive("الكمية يجب أن تكون أكبر من صفر"),
  notes: z.string().optional().nullable(),
});

export type InsertInventoryWithdrawalOrder = z.infer<typeof insertInventoryWithdrawalOrderSchema>;
export type InventoryWithdrawalOrder = typeof inventoryWithdrawalOrders.$inferSelect;
export type InsertInventoryWithdrawalLine = z.infer<typeof insertInventoryWithdrawalLineSchema>;
export type InventoryWithdrawalLine = typeof inventoryWithdrawalLines.$inferSelect;

export type InventoryWithdrawalLineWithDetails = InventoryWithdrawalLine & {
  product?: Product;
};

export type InventoryWithdrawalOrderWithDetails = InventoryWithdrawalOrder & {
  responsibleEmployee?: Employee;
  confirmedByEmployee?: Employee;
  lines: InventoryWithdrawalLineWithDetails[];
  journalEntry?: JournalEntry;
};
